// (C) Copyright 2011-2012 Hewlett-Packard Development Company, L.P.
/**
 * @constructor
 * @type {AssociationAddView}
 */
define(['hp/core/Localizer',
    'text!hpPages/core/association_add.html',
    'text!hpPages/core/dialog_overlay.html',
    'jquery',
    'lib/jquery.dataTables',
    'hp/lib/jquery.hpEllipsis'],
function (localizer, associationAddHtml, dialogOverlayHtml) {
    "use strict";

    var AssociationAddView = (function () {
      
        var CONTAINER = '.hp-dialog-container';
        var CONTROLS = '.hp-controls';
        var TABLE = '.hp-add-choices';
        var ROW = TABLE + ' tbody tr';
        var SELECTED_ROW = ROW + '.hp-selected';
        var SEARCH = '.hp-search';
        var ADD = '.hp-add';
        var ADD_AGAIN = '.hp-add-again';
        var CANCEL = '.hp-cancel';
        var MESSAGE = '.hp-control-message';
        var SPINNER = '#hp-associations-spinner';
        var ACTIVE = 'hp-active';
        var SELECTED = 'hp-selected';
        var ERROR = 'hp-error';
        var PRIMARY = 'hp-primary';
        var ENTER = 13;
        var ESCAPE = 27;
        var A = 65; // for Ctrl-A

        /**
         * Constructor
         * Show a dialog to add associations
         * @param {args} Object with properties as follows:
         *
         *    title: title of the dialog
         *    dataTableOptions: jquery.dataTables options.
         *        Recommend at least providing aoColumns with sName and sWidth.
         *    getResults: the function called when results are needed.
         *        function(query, handlers)
         *        The function should call one of the handlers as follows
         *          handlers.success({count: N, members:[...]});
         *          handlers.error(errorMessage);
         *    apply: the function called when the user clicks ok or apply.
         *        function(resultsMembers)
         *        The resultsMembers are the same members provided in the
         *        getResults() success case.
         */
        function AssociationAddView(args) {
          
            var self;
            var overlay;
            var component;
            var dataTable;
            // remember the last search to avoid redoing
            var lastSearch = null;
            // preserve the table width across search changes,
            // keeps it from being jumpy
            var tableWidth = null;
            var busyCount = 0;
            // Forward reference so Sonar won't complain
            var onKeyDown;
            
            function calculateTableHeight() {
                var windowHeight = $(window).height();
                // use a percentage of the window, if possible
                var percentage = windowHeight * 0.5;
                var available = percentage;
                if (tableWidth) {
                    // otherwise, only use what we have remaining space for
                    available = windowHeight -
                        ($('.dataTables_scrollBody', component).offset().top +
                         $('footer', component).height() + 35);
                }
                    
                return Math.min(available, percentage);
            }
            
            function onResize() {
                var tableHeight = calculateTableHeight();
                var oSettings = dataTable.fnSettings();
                var spinnerWidth = 0;
                oSettings.oScroll.sY = tableHeight;
                if (tableWidth) {
                    oSettings.oScroll.sX = tableWidth;
                    $('.dataTables_scrollHead', component).
                        css({'width': tableWidth});
                    $('.dataTables_scrollBody', component).
                        css({'height': tableHeight, 'width': tableWidth});
                } else {
                    $('.dataTables_scrollBody', component).
                        css({'height': tableHeight});
                }
                
                if (busyCount > 0) {
                    spinnerWidth = $(SPINNER, component).outerWidth();
                }
                // adjust message to take all of the footer except the controls
                $(MESSAGE, component).
                    css({'margin-right': $(CONTROLS, component).outerWidth(),
                        'margin-left': spinnerWidth});
            }
            
            function updateSelectedMessage() {
                $(MESSAGE, component).
                    text(localizer.getString('core.associationAdd.selected',
                        [$(SELECTED_ROW, component).length]));
            }
            
            function onClickRow(event) {
                var row = $(this);
                if (event.ctrlKey || event.metaKey) {
                    // Ctrl, toggle selection of this row only
                    row.toggleClass(SELECTED);
                } else if (event.shiftKey) {
                    // user wants a range, calculate beginning and end
                    // based on nearest earlier selection, or top of table
                    // if nothing is selected.
                    var startIndex = 0;
                    var endIndex = row.index();
                    
                    $(SELECTED_ROW, component).each(function () {
                        var index = $(this).index();
                        // select last up to endIndex
                        if (index < endIndex) {
                            startIndex = index;
                        // select first after endIndex
                        } else if (0 === startIndex && index >= endIndex) {
                            startIndex = endIndex;
                            endIndex = index;
                            return false;
                        }
                    });
                    
                    $(ROW, component).each(function (index, row2) {
                        if (index >= startIndex && index <= endIndex) {
                          $(row2).addClass(SELECTED);
                        }
                    });
                } else {
                    // de-select everything else
                    $(ROW, component).removeClass(SELECTED);
                    // select just this row
                    row.addClass(SELECTED);
                }
                updateSelectedMessage();
            }
            
            function onResultsSuccess(results) {
                busyCount -= 1;
                if (busyCount <= 0) {
                    $(SPINNER, overlay).hide();
                    busyCount = 0;
                }
                dataTable.fnClearTable(results.count === 0);
                dataTable.fnAddData(results.members);
                
                // on the first time through with some data, use
                // the width of the table from now on
                if (results.count > 0 && ! tableWidth) {
                    tableWidth = $(TABLE, component).width();
                }
                $(SEARCH).focus();
                onResize();
                
                $(ROW, component).click(onClickRow);
                $(MESSAGE, component).removeClass(ERROR);
            }
            
            function onResultsError(message) {
                busyCount -= 1;
                if (busyCount <= 0) {
                    $(SPINNER).hide();
                    busyCount = 0;
                }
                dataTable.fnClearTable(true);
                $(SEARCH).focus();
                onResize();
                $(MESSAGE, component).text(message);
                $(MESSAGE, component).addClass(ERROR);
            }
            
            function getResults() {
                var search = $(SEARCH, component).val();
                if (args.getResults && search !== lastSearch) {
                    // ask caller for results
                    busyCount += 1;
                    $(SPINNER).show();
                    $(MESSAGE, component).text('');
                    args.getResults(search, {
                        success: onResultsSuccess,
                        error: onResultsError
                    });
                    onResize();
                }
                lastSearch = search;
            }
            
            function onCancel() {
                overlay.remove();
                $(CONTAINER, overlay).removeClass(ACTIVE);
                overlay.removeClass(ACTIVE);
                $(window).unbind('resize', onResize);
                $(window).unbind('keydown', onKeyDown);
            }
            
            function onAddAgain(event) {
                if (event) {
                    // user clicked Add +, make that the primary button
                    $(ADD, component).removeClass(PRIMARY);
                    $(ADD_AGAIN, component).addClass(PRIMARY);
                }
                var rows = $(SELECTED_ROW, component);
                var results = $.map(rows, function(row) {
                    return dataTable.fnGetData(row);
                });
                if (results.length > 0) {
                    // tell caller we have something to apply
                    if (args.apply) {
                        // allow caller to return false to prevent
                        // us from displaying a message indicating we've
                        // performed the apply
                        if (false !== args.apply(results)) {
                            $(MESSAGE, component).text(
                                localizer.getString('core.associationAdd.added',
                                    [results.length]));
                        }
                    }
                    rows.removeClass(SELECTED);
                    $(SEARCH, component).val('');
                    lastSearch = null;
                    getResults();
                }
            }
            
            function onAdd(event) {
                if (event) {
                    // user clicked Add, make that the primary button
                    $(ADD_AGAIN, component).removeClass(PRIMARY);
                    $(ADD, component).addClass(PRIMARY);
                }
                onAddAgain();
                onCancel();
            }
            
            function onKeyUp(event) {
                getResults();
            }
            
            function onKeyDown(event) {
                var keyCode = (event.which ? event.which : event.keyCode);
                if (keyCode == ESCAPE) {
                    onCancel();
                    event.preventDefault();
                } else if (keyCode == ENTER) {
                    if ($(ADD_AGAIN).hasClass(PRIMARY)) {
                        onAddAgain();
                    } else {
                        onAdd();
                    }
                    event.preventDefault();
                // Ctrl-A outside of the search control means select everything
                } else if (keyCode == A && (event.ctrlKey || event.metaKey) &&
                    ! $(SEARCH, component).is(':focus')) {
                    $(ROW, component).addClass(SELECTED);
                    updateSelectedMessage();
                    event.preventDefault();
                }
            }
            
            function setLanguage(options) {
            	if (! options.oLanguage) {
            		options.oLanguage = localizer.getString('core.dataTables.oLanguage');
            	}
            }
            
            /**
             * @public
             */
            function initialize() {
                component = $(associationAddHtml);
                localizer.localizeDom(component);
                overlay = $(dialogOverlayHtml);
                localizer.localizeDom(overlay);
                $(CONTAINER, overlay).append(component).addClass(ACTIVE);
                $('h1', component).text(args.title);
                
                var options = {
                    bPaginate : false,
                    bFilter : false,
                    bInfo : false,
                    bDeferRender : false,
                    bAutoWidth : false,
                    sScrollY : calculateTableHeight(),
                    bSort : false,
                    aaData : args.results ? args.results : []
                };

                $.extend(options, args.dataTableOptions);
                // Set the oLanguage property if not set.
                setLanguage(options);
                dataTable = $(TABLE, component).dataTable(options);

                overlay.addClass(ACTIVE);
                $('#hp-body-div').append(overlay);
                $('> header h1', component).hpEllipsis();
                                
                $(ADD, component).click(onAdd);
                $(ADD_AGAIN, component).click(onAddAgain);
                $(CANCEL, component).click(onCancel);
                $(SEARCH, component).focus().keyup(onKeyUp);
                $(document).bind('keydown', onKeyDown);
                $(window).bind('resize', onResize);
                onResize();

                // get initial content
                getResults();
            }
            
            self = this;
            initialize();
            
            this.cancel = function () {
                onCancel();
            };
        }

        return AssociationAddView;
    }());

    return AssociationAddView;
});
