// (C) Copyright 2012 Hewlett-Packard Development Company, L.P.
/**
 * @type (BaselinePresenter)
 * This is presenter class for Baseline resource.
 * Provides functions for adding a new baseline item into IndexService and Deleting an existing baseline item from IndexService
 */
define(['hpsum/model/baseline/BaselineResource',
	'hp/core/LinkTargetBuilder',
	'hp/services/IndexService',
	'hp/core/Router',
	'hp/core/UrlFragment',
	'hp/core/Notifications',
	'hp/core/EventDispatcher',
    'hp/services/Log',
	'hp/services/REST', 
    'hp/model/Session',
    'hp/core/Localizer'],
function(resource, linkTargetBuilder, indexService, router, urlFragment, notifications, EventDispatcher, log, restapi , session, localizer) {"use strict";

    var BaselinePresenter = ( function() {

        // consts/statics can go here.
        //

        /**
         * @constructor
         * @type {BaselinePresenter}
         */
        function BaselinePresenter() {

            var dispatcher = new EventDispatcher();
	    var selectedName = '';
            var trackUri = null;
            var baselines = [], iterator;
			var lastLocation = '',
				lastAction = '';
                        
            function getItem() {
                var uris = resource.getSelection().uris;
                selectedName = '';
                if (uris && uris.length > 0) {
                    resource.getItem(uris/*resource.getSelection().uris[0]*/, {
                        success: function (item) {
                            selectedName = item.hapi.baseline_name;
                            dispatcher.fire("itemChange", item);
                        },
                        error: function (errorMessage) {
                            dispatcher.fire("itemError", errorMessage);
                        }
                    });
                }
            }
            
            function getAssociations() {
                var uris = resource.getSelection().uris;
                if (uris && uris.length > 0) {
                    indexService.getParentAndChildrenAssociations({
                        uri: uris[0],
                        parentDepth: 1,
                        childDepth: 1,
                        handlers: {
                            success: function (tree) {
                                dispatcher.fire("associationsChange", tree);
                            },
                            error: function (errorMessage) {
                                dispatcher.fire("associationsError", errorMessage);
                            }}
                    });
                }
            }
	    
	    	function selectionChange(selection) {
                dispatcher.fire("selectionChange", selection);
                if (selection.uris) {
                    if (! selection.multiSelectId) {
                        // get the selected item
                        getItem();
                        //getAssociations();
                    } else {
                        // generate aggregate information
                        var aggregate = {count: selection.uris.length,
                            status:{ok: 0, warning: 0, error: 0, disabled: 0, unknown: 0}};
                        var indexResult;
                        
                        $.each(selection.uris, function(index, uri) {
                            indexResult = resource.getIndexResultForUri(uri);
                            aggregate.status[indexResult.status] += 1;
                        });
                        selectedName = aggregate.count + " servers";
                        
                        dispatcher.fire("aggregateChange", aggregate);
                    }
                }
            }
	    	
	    	function successAddBaseline(data, handlers) {
	    		
	    		//Adding a new baseline is success
				if (0 == data.hapi.hcode) {
					//add a notification about successful operation
	                notifications.add({
	                	summary: localizer.getString('hpsum.baseline.add.locationAdded'),
	                    sourceName: data.hapi.command.output_data.uri,
	                    status: 'ok'
	            	});
	            	
	            	//Perform inventory on location
					var uri = "/Session/" + session.getToken()  + "/Baseline/location/" + data.hapi.command.output_data.location_id + "/inventory";
					restapi.getURI(uri, {
						success : function(data) {
							//Started inventory on location successfully.
							//We will wait for a while to complete the location inentory.
							var timer = setInterval(function() {
							    	restapi.getURI(uri.replace("inventory", "getstatus"), {
							    		success : function(data) {
							    			if (data.hapi.location_state == "inventory_completed") {
							    				resource.getIndexResults();
												//Location inventory completed, will go back
												clearInterval(timer);
							    				restapi.getURI(uri.replace("inventory", "index"), {
							    					success : function(data) {
							    						data = data.hapi.output_data;
							    					 if (data.hasOwnProperty("baselines")) {
							    						var baselines = [],
							    						    baselineURI;
							    						baselines = data.baselines.baseline;
							    						var baselineInfo = baselines[0];
							    						if (baselineInfo != undefined && baselineInfo.hasOwnProperty("baseline_id")) {
							    							baselineURI = baselineInfo.baseline_uri + '/getBaselineData';
							    							resource.setSelectedUris([baselineURI]);
							    							
							    						}
							    						resource.getIndexResults();
							    						dispatcher.fire('itemChange', baselineInfo);
							    						handlers.success({baseline_uri: baselineURI});
							    					} else {
							    						resource.getIndexResults();
							    						handlers.success(data);
							    					}
							    											    						
							    					},
							    					error : function(data) {
							    						clearInterval(timer);
							    						handlers.error(data.hapi.hmessage);
							    					}
							    				});
							    			}
							    		},
							    		error : function(data) {
							    			clearInterval(timer);
							    			handlers.error(data.hapi.hmessage);
							    		}
							    	});
							    	
							    }, 1000);	
						},
						error : function(data) {
							handlers.error(data.hapi.hmessage);
						}
						
					});
				} else {
					//Failed to add a new baseline
					handlers.error(data.hapi.hmessage);
				}
			}
			
	    	/**
             * Handles the success of the addBaseline command attempt by saving the
             * baseline information into indexservice.
             *
             * @param {{Baseline Details}}
             *            baselineDetails: The DTO from the server with
             *            the baseline ID and other information.
             * @param {{success:function():void,
             *            error:function(string):void}} handlers The
             *            caller's success and error handler functions.
             */
            function onaddBaselineSuccess(data, handlers) {
				
				if(data.hapi.hcode === 0)
				{
					//Start inventory for the location
					var uri = "/Session/" + session.getToken()  + "/Baseline/location/" + data.hapi.command.output_data.location_id + "/inventory";
					
					var inventoryHandlers = {
	                    						success : function(data, handlers) {
	                    							//update the master pane immediately
													resource.getIndexResults();
	                        						onInventorySuccess(data, handlers);
	                    		 				},
	                    						error : function(errorInfo, handlers) {
	                        						onInventoryError(errorInfo, handlers);
	                    						}
	                					};
					
					restapi.getURI(uri, inventoryHandlers);
					
					handlers.success(data);
	                
	                //add a notification about successful operation
	                notifications.add({
	                	summary: localizer.getString('hpsum.baseline.add.locationAdded'),
	                    sourceName: data.hapi.command.output_data.uri,
	                    status: 'ok'
	            	});
				}
				else
				{
					handlers.error(data.hapi.hmessage);
				}
			}
            
            function onInventorySuccess(data, handlers) 
            {
            	//handlers.success(data);
            }
            
            function onInventoryError(error, handlers) 
            {
            	//handlers.error(error);
            }
            
            function ongetBaselineError(error, handlers)
            {
            	handlers.error(error);
            }
            
            function ongetBaselineSuccess(data, handlers) {
            	
            	
	        	var i;
	        	//Create resources out of the data received as response
	        	for(i = 0; i < data.hapi.output_data.baselines.baseline.length; i += 1)
	        	{
	        		var item = {
	                    category: 'baseline',
	                    baselineId: data.hapi.output_data.baselines.baseline[i].baseline_id,
	                    name: data.hapi.output_data.baselines.baseline[i].baseline_name,
	                    location: data.hapi.output_data.baselines.baseline[i].baseline_url,
	                    uri: data.hapi.output_data.baselines.baseline[i].baseline_uri, //'/' + 'baseline' + '/item',
	                    status: data.hapi.output_data.baselines.baseline[i].baseline_state, //'ok',
	                    changing: 1,
	                    updatedAt: data.hapi.output_data.baselines.baseline[i].baseline_updated_at,
	                    stateDetails: '',
	                    index: baselines.length,
	                    attributes: {version: data.hapi.output_data.baselines.baseline[i].baseline_version, size: data.hapi.output_data.baselines.baseline[i].baseline_size}
	               	};
	        		
	        		baselines = baselines.concat(item);
	        	}
	        	
	        	iterator -= 1;
	        	
	        	if(iterator == 0)
	        	{
	        		handlers.success(baselines);
	        	}
            }

            /**
             * Handle an error coming back from the addBaseline request.
             *
             * @param {Object}
             *            errorInfo Error information from the server
             * @param {success:function():void,
             *            error:function(string):void
             *            }} handlers The
             *            caller's success and error handler functions.
             */
            function onaddBaselineError(errorInfo, handlers) {
                handlers.error(errorInfo);
            }
            
            function startInventorySuccess(data, handlers) 
            {
            	//if(data.hapi.message == "COMPLETE")
            	//{
            	//}
            	//handlers.success(data);
            }
            
            function startInventoryError(errorinfo, handlers)
            {
            	//handlers.error(errorinfo);
            }
            
            function indexLocations(handlers) {
            	
            	var baselineHandlers = {
	                    			success : function(data) {
	                        			ongetBaselineSuccess(data, handlers);
	                    		 	},
	                    			error : function(errorInfo) {
	                        			ongetBaselineError(errorInfo, handlers);
	                    			}
                					}
                    
                return {
                    success: function (data) {
                    	var i, uri;
                    	iterator = 0;
                    	baselines = [];
                    	//iterate through locations, call REST API for each location
                    	for(i = 0; i < data.hapi.output_data.baselines.baseline.length; i += 1)
                    	{
                    		var str = data.hapi.output_data.baselines.baseline[i].location_uri;
                    		str = str.replace(new RegExp(/\\/g),"/");
                    		var item = {
                    			location_id: data.hapi.output_data.baselines.baseline[i].locationid,
                    			location_uri: str,
                    			location_name: data.hapi.output_data.baselines.baseline[i].location_name
                    		};
                    		
                    		uri = '/session/'+ session.getToken() +'/Baseline/location/' + item.location_id + '/index';
                
                			iterator += 1;
               	 			restapi.getURI(uri, baselineHandlers);                    		
                    	}
                    	
                    	if(data.hapi.output_data.baselines.baseline.length == 0) {
                    		//return empty list if baselines list is empty
                    		handlers.success(baselines);
                    	}
                    },
                    error: handlers.error};
            }
            
            /**
             * @public
             */
            this.init = function() {
            	resource.setCompareResourcesFunction(indexResultsCompare);
				resource.on("selectionChange", selectionChange);
			};
            this.resource = function () {
                return resource;
            }
            
            this.getSelection = function() {
                return resource.getSelection();
            };
            
            this.selectionLabel = function () {
                return selectedName;
            };
            
            /**
             * @public
             * Add a listener for a specified event.
             * @param {string} eventName The name of the event.
             * @param {function(...)}
             */
            this.on = function(eventName, callback) {
            	var uris;
            	
                dispatcher.on(eventName, callback);
                
                // fire initial state
                switch (eventName) {
                    case 'selectionChange':
                        callback(resource.getSelection());
                        break;
                    case 'itemChange':
                    
                    	uris = resource.getSelection().uris;
                        if (uris && uris.length === 1) {
                        	resource.getItem(uris[0], {success: callback});
                        }
                        break;
                    case 'itemError':
                        uris = resource.getSelection().uris;
                        if (uris && uris.length === 1) {
                            resource.getItem(uris[0], {error: callback});
                        }
                        break;
                    case 'associationsChange':
                        getAssociations();
                        break;
                }
            };

            /**
             * @public
             * Remove a listener for a specified event.
             * @param {string} eventName The name of the event.
             * @param {function(...)} The function that was registered with "on"
             */
            this.off = function (eventName, callback) {
                dispatcher.off(eventName, callback);
            };
            
            this.getAddLink = function () {
                var result = null;
                if (session.canPerformAction('BASELINE', 'ADD')) {
                    result = linkTargetBuilder.makeLink(localizer.getString('hpsum.baseline.add.title'), null, 'add', 'baseline', 'ADD');
                }
                return result;
            };
            
            this.getAllBaselines = function (handlers) {
                
                iterator = 0;
                baselines = [];
                
                var localHandlers = {
	                    			success : function(data) {
	                        			ongetLocationSuccess(data, handlers);
	                    		 	},
	                    			error : function(errorInfo) {
	                        			ongetLocationError(errorInfo, handlers);
	                    			}
                			}
                			
               	restapi.getURI('/Session/' + session.getToken() + '/Baseline/location/index', indexLocations(handlers));
               
            };
            
			this.getBaselines = function(handlers) {
					var uri = '/Session/' + session.getToken() + '/baseline/location/index?indexChilds=Yes';
					restapi.getURI(uri, {
						success : function(data) {
							var locations = [];
							baselines = [];
							locations = data.hapi.output_data.locations.location;
							
							for (var idx = 0; idx < locations.length; idx++) {
								//get list of baselines which are inventory completed and dont include baselines that are still in inventory
								var baselines_invcompl = $.grep(locations[idx].baselines.baseline, function(baseline, index) {
						                            return (
						                            	(baseline.baseline_state == "inventory_completed") 
						                            	);
								});
								if(baselines_invcompl && baselines_invcompl.length > 0)
									baselines = baselines.concat(baselines_invcompl);
							}
							handlers.success(baselines);
						},
						error : function(msg) {
							handlers.error(msg);
						}
					});
			};
			
			this.getBaselinesAll = function(handlers) {
					var uri = '/Session/' + session.getToken() + '/baseline/location/index?indexChilds=Yes';
					restapi.getURI(uri, {
						success : function(data) {
							var locations = [];
							baselines = [];
							locations = data.hapi.output_data.locations.location;
							
							for (var idx = 0; idx < locations.length; idx++) {
								baselines = baselines.concat(locations[idx].baselines.baseline);
							}
							handlers.success(baselines);
						},
						error : function(msg) {
							handlers.error(msg);
						}
					});
			};
                
            /**
             * @public
             * Add a new baseline item into IndexService
             * @param {item} item that need to be added into indexservice
             * @param {handlers(...)} The function that was registered with "on"
             */
            this.addItem = function (item, handlers) {
		var workingLoc;
		// In case of local baseline the working location will
		// be set automatically by the engine. only in case of
		// FTP we need to set the working location	
                if(item.locationType == "local")
                {
                    item.uripath = item.location;
                }
                if(item.locationType == "FTP")
                {
                    item.uripath = 'PFW';
		    workingLoc = item.location;
                }
                
                var proxySettings;
                if (item.proxySettingsType == "generalProxy") {                	
                	proxySettings = {proxy_username : item.pusername,
                					 proxy_password : item.ppassword,
                					 proxy_url : item.proxyurl,
                					 proxy_ftp_port : item.port,                					 
                					};                					
                }
                if (item.proxySettingsType == "scriptedProxy") {
                	proxySettings = {
                						 use_proxy_script: true,
                						 proxy_script : item.proxyscript
                				};
                }
                

                var input = {
                                 hapi: { 
                                            uri: item.uripath,
                                            username:item.username,
                                            password:item.password,
                                            working_location:workingLoc,
                                            locationtype: item.locationType ,
                                            schedule_information : { 
                                                                date : item.scheduleDate , 
                                                                time : item.scheduleTime , 
                                                                scheduletype : item.scheduleType , 
                                                                recurs_every: item.scheduleRecurring 
                                                           },
                                            proxy_settings: proxySettings 
                                            	
                                        }
                            }
				restapi.postObject("Session/" + session.getToken() + "/Baseline/add", 
								    input,  "", 
            		         {
                    			success : function(baselineDetails) {
                        			//onaddBaselineSuccess(baselineDetails, handlers);
                        			successAddBaseline(baselineDetails, handlers);
                    		 	},
                    			error : function(errorInfo) {
                        			onaddBaselineError(errorInfo, handlers);
                    			}
                			}, {});
					
				//Mockup data to simulate adding new baseline into IndexService		 
				/*
                item.status = 'unknown';
				indexService.addItem('baseline', item);
                handlers.success("");
                notifications.add({
                	summary: localizer.getString('hpsum.baseline.add.baselineAdded'),
                    sourceName: item.name,
                    status: 'ok'
            	});
				*/
            };
            
            /**
             * @public
             * Craete a new custom baseline
             * @param {item} item that need to be used to create custom baseline
             * @param {handlers(...)} The function that was registered with "on"
             */
            this.createCustomBaseline = function (item, handlers) {
			
				restapi.postObject("Session/" + session.getToken() + "/Baseline/createbundle", 
								item,  "", 
            		         {
                    			success : function(baselineDetails) {
                        			onaddBaselineSuccess(baselineDetails, handlers);
                    		 	},
                    			error : function(errorInfo) {
                        			onaddBaselineError(errorInfo, handlers);
                    			}
                			}, {});
			};
			
			function indexResultsCompare(last, latest){
				//Check if path has changed from last time.
                var loc = router.location().split('/');
                if((loc.length < 6) && (loc[2] == 'show'))
                    return false;
                var location = loc[1];
                var action = loc[2];
                                    
                if(lastLocation != location || lastAction != action) {
                    lastLocation = location;
                    lastAction = action;
                    return false;
                }
                
				if(last.location != latest.location)
					return false;
				if(last.id != latest.id)
					return false;
                if((last.state == latest.state))
           			return true;
            	return false;
            }
			
			/**
             * @public
             * Get custom baseline creation status
             * @param {item} This is response of createbundle command. This response has resource_id for the newly creating custom baseline
             * @param {handlers(...)} The function that was registered with "on"
             */
			this.getCustomBaselineStatus = function (item, handlers) {
				
            	var uri = "Session/" + session.getToken() + "/baseline/" + item.hapi.command.output_data.resource_id + "/getstatus";
            	var thishandlers =	{
                    			success : function(data) {
                        			handlers.success(data);
                    		 	},
                    			error : function(errorInfo) {
                        			handlers.error(errorInfo);
                    			}
                			};
            	restapi.getURI(uri, thishandlers);
            };
            
            this.startInventory = function (item, handlers) {
            	var uri = item.uri.replace('getBaselineData','inventory');
            	
            	restapi.getURI(uri, {
                    			success : function(data) {
                    				if (handlers) {
                        				handlers.success(data);
                        			}
                    		 	},
                    			error : function(errorInfo) {
                    				if (handlers) {                   
                        				handlers.error(errorInfo);
                        		}
                    			}
                    	});
            };
            
            // Sends the inventory command for the PFW bundles
            // with the schedule information.
            this.startPFWInventory = function (item, handlers) {
            	var uri;
            	uri = item.uri + '/inventory';
            	var isscheduled = "yes";
            	if (item.scheduled == 'Off')
            		isscheduled = "no";
            	var input = {
                                 hapi: {                                          
                                         scheduled: isscheduled ,
                                         schedule_information : { 
                                                                date : item.scheduleDate , 
                                                                time : item.scheduleTime , 
                                                                scheduletype : item.scheduleType , 
                                                                recurs_every: item.scheduleRecurs 
                                                            }
                                        }
                            }
                            
                restapi.postObject(uri, 
								input,  "", 
            		         {
                    			success : function(data) {
                        			startInventorySuccess(data, handlers);
                    		 	},
                    			error : function(errorInfo) {
                        			startInventoryError(errorInfo, handlers);
                    			}
                			}, {});            	
            };
            
            this.getComponentData = function (item, handlers) {
            	var uri = "Session/" + session.getToken() + "/Baseline/location/" + item.location_id + "/bundle/" + item.baseline_id + "/getComponentData";
            	var thishandlers =	{
                    			success : function(data) {
                        			handlers.success(data);
                    		 	},
                    			error : function(errorInfo) {
                        			handlers.error(errorInfo);
                    			}
                			};
            	//restapi.getURI(uri, handlers);
            	
            	var payload = { hapi: { component_id: item.component_id,
					filter: item.filter
					 }
				};
				if(item.filter == undefined)
					delete payload['filter'];
            	
            	restapi.postObject(uri, payload
								,  "", thishandlers, {});

            };
            
            this.getBaselineStatus = function (item, handlers) {
            	var uri = "Session/" + session.getToken() + "/Baseline/location/" + item.location_id + "/bundle/" + item.baseline_id + "/getstatus";
            	var thishandlers =	{
                    			success : function(data) {
                        			handlers.success(data);
                    		 	},
                    			error : function(errorInfo) {
                        			handlers.error(errorInfo);
                    			}
                			};
            	restapi.getURI(uri, thishandlers);
            };
            
            this.getBundleIndexData = function (item, handlers) {
            	var uri = "Session/" + session.getToken() + "/Baseline/location/" + item.location_id + "/bundle/" + item.baseline_id + "/index";
            	var thishandlers =	{
                    			success : function(data) {
                        			handlers.success(data);
                    		 	},
                    			error : function(errorInfo) {
                        			handlers.error(errorInfo);
                    			}
                			};
            	restapi.getURI(uri, thishandlers);
            };
			this.getCompConfDetail = function (item, handlers) {
            	var uri = "Session/" + session.getToken() + "/Baseline/location/" + item.location_id + "/bundle/" + item.baseline_id + "/getcomponentconfigdata";
            	var thishandlers =	{
                    			success : function(data) {
                        			handlers.success(data);
                    		 	},
                    			error : function(errorInfo) {
                        			handlers.error(errorInfo);
                    			}
                			};
            	//restapi.getURI(uri, handlers);
            
            	restapi.postObject(uri, 
								{
								 hapi: { component_id: item.component_id }
								},  "", thishandlers, {});
            };
			this.saveComponentConfig = function (item, compdata, handlers) {
				var uri = "Session/" + session.getToken() + "/Baseline/location/" + item.location_id + "/bundle/" + item.baseline_id + "/savecomponentconfigdata";
				var thishandlers =	{
                    			success : function(data) {
									handlers.success(data);
                    		 	},
                    			error : function(errorInfo) {
                        			handlers.error(errorInfo);
                    			}
							};
				restapi.postObject(uri, compdata, "", thishandlers, {});
			};
			
            this.removeBaseline = function (item, handlers) {
                var uri = "Session/" + session.getToken() + "/Baseline/location/" + item.location_id + "/bundle/" + item.baseline_id + "/remove";
                var thishandlers = {
                    success : function(data) {
                        if(data.hapi.hcode == 0) {
                            resource.clearAllSelectedUris();
                            handlers.success("");
                            notifications.add({
                                summary: "Deleted baseline",
                                sourceName: item.baseline_url,
                                status: 'ok'
                            });
                            resource.getIndexResults();
                            router.go(urlFragment.replaceUris(router.location(), []), 'hpsum deleted baseline');
                            dispatcher.fire("selectionChange", []);
                            handlers.success(data);
                        }
                        else if(data.hapi.hcode == 3008){
                            //Baseline is associated with a node, confirm the user wants to delete. If yes, send remove with force option
                            var text = data.hapi.hmessage.replace(/,/g, '<br>');
                            text = text.replace(/:/g, '<br><br>');
                            text = text + "<li>Assign a new baseline to the node</li><li>Delete the node(s)</li>";
                            $("<div></div>").dialog( {
				 open : function(event,ui) {
						$(this).css({'max-height':200, 'overflow-y':'scroll'});
			},
                                buttons: { "Ok": function () {                                                    
                                                    $(this).dialog("close");
                                                    handlers.success(data); 
                                                }                                           
                                         },
                                    close: function (event, ui) { $(this).remove(); handlers.success(data);},
                                    resizable: false,
                                    title: "Delete Baseline",
                                    modal: true
                                }).html(text);
                        }
                    },
                    error : function(errorInfo) {
                        handlers.error(errorInfo);
                    }
                };
                restapi.postObject(uri,"", "", thishandlers, {});
            }
            
            /**
             * @public
             * Get components with their supported OS list
             * @param {item} This is baseline uri
             * @param {handlers(...)} The function that was registered with "on"
             */
			this.getosname = function (baseline_uri, handlers) {
				
            	var uri = baseline_uri + "/getosname";
            	var thishandlers =	{
                    			success : function(data) {
                        			handlers.success(data);
                    		 	},
                    			error : function(errorInfo) {
                        			handlers.error(errorInfo);
                    			}
                			};
            	restapi.getURI(uri, thishandlers);
            };
            
            /**
             * @public
             * Get components with their supported device list
             * @param {item} This is baseline uri
             * @param {handlers(...)} The function that was registered with "on"
             */
			this.getsupporteddevice = function (baseline_uri, handlers) {
				
            	var uri = "Session/" + session.getToken() + baseline_uri + "/getsupporteddevice";
            	var thishandlers =	{
                    			success : function(data) {
                        			handlers.success(data);
                    		 	},
                    			error : function(errorInfo) {
                        			handlers.error(errorInfo);
                    			}
                			};
            	restapi.getURI(uri, thishandlers);
            };
            
           this.getComponents = function (baseline_uri, handlers) {		
            	var uri = baseline_uri + "/getComponents";
            	var thishandlers =	{
                    			success : function(data) {
                        			handlers.success(data);
                    		 	},
                    			error : function(errorInfo) {
                        			handlers.error(errorInfo);
                    			}
                			};
            	restapi.getURI(uri, thishandlers);
            };
            
            this.getEnhancements = function (baseline_uri, handlers) {		
            	var uri = baseline_uri + "/getEnhancements";
            	var thishandlers =	{
                    			success : function(data) {
                        			handlers.success(data);
                    		 	},
                    			error : function(errorInfo) {
                        			handlers.error(errorInfo);
                    			}
                			};
            	restapi.getURI(uri, thishandlers);
            };
            
            this.getOsInfo = function (baseline_uri, handlers) {		
            	var uri = baseline_uri + "/getOsInfo";
            	var thishandlers =	{
                    			success : function(data) {
                        			handlers.success(data);
                    		 	},
                    			error : function(errorInfo) {
                        			handlers.error(errorInfo);
                    			}
                			};
            	restapi.getURI(uri, thishandlers);
            };
        }

        return BaselinePresenter;
    }());

    return new BaselinePresenter();
});
