// (C) Copyright 2011-2012 Hewlett-Packard Development Company, L.P.
define(['hp/core/Sidebar',
    'hp/model/DevelopmentSettings',
    'hp/core/LinkTargetBuilder',
    'hp/core/Localizer',
    'hp/core/Style',
    'jquery',
    'modernizr',
    'hp/lib/date',
    'hp/lib/jquery.hpStatus',
    'hp/lib/jquery.hpSafeClone',
    'hp/lib/jquery.hpTooltip'],
function (sidebar, settings, linkTargetBuilder, localizer, style) {
"use strict";

    var Notifications = (function () {

        var BRIEF = '#hp-activity-notification';
        var BRIEF_STATUS = BRIEF + ' .hp-status';
        var BRIEF_MESSAGE = BRIEF + ' .hp-message';
        var CONTROL = '#hp-activity-control';
        var CONTROL_COUNT = '#hp-activity-control-new-count';
        var NEW_COUNT = '#hp-activity-new-count';
        var FLYOUT = '#hp-activity-flyout';
        var CONTENTS = FLYOUT + ' .hp-flyout-contents';
        var LIST = '#hp-flyout-activities';
        var TEMPLATE = '#hp-flyout-activity-template';
        var BODY = '#hp-body-div';
        var ACTIVE = 'hp-active';
        var NEW = 'hp-new';
        var SELECTED = 'hp-selected';
        var DELAY = '5000'; // milliseconds
        var TAB = 9;
        var ESCAPE = 27;
        var UP_ARROW = 38;
        var DOWN_ARROW = 40;

        /**
         * @constructor
         * @type {Notifications}
         */
        function Notifications() {

            var flashNotifications = [];
            var notifications = [];
            var running = false;
            var template = null;
            var selecting = false;
            
            function onMouseMove() {
                selecting = true;
            }
            
            function onMouseUp() {
                $(BODY).off('mousemove', onMouseMove);
                $(BODY).off('mouseup', onMouseUp);
            }
            
            function onMouseDown() {
                $(BODY).on('mousemove', onMouseMove);
                $(BODY).on('mouseup', onMouseUp);
            }
            
            function updateCount() {
                var count = $(LIST + ' > li.' + NEW).length;
                if (count > 0) {
                    $(CONTROL_COUNT).addClass(ACTIVE);
                } else {
                    $(CONTROL_COUNT).removeClass(ACTIVE);
                }
                $(CONTROL_COUNT).text(count);
                $(NEW_COUNT).text(count);
            }
            
            var deselectActivity = function() {}; // forward reference for Sonar
            var selectActivity = function() {}; // forward reference for Sonar
            
            function nextActivity() {
                var activities = $(FLYOUT + ' .hp-activity');
                activities.each(function (index, elem) {
                    if ($(elem).hasClass(SELECTED) &&
                        activities.length > (index + 1)) {
                        deselectActivity();
                        selectActivity($(activities[index+1]));
                        return false;
                    }
                });
            }

            function previousActivity() {
                var activities = $(FLYOUT + ' .hp-activity');
                activities.each(function (index, elem) {
                    if ($(elem).hasClass(SELECTED) && index > 0) {
                        deselectActivity();
                        selectActivity($(activities[index-1]));
                        return false;
                    }
                });
            }
            
            function onKeypress(ev) {
                var keyCode = (ev.which ? ev.which : ev.keyCode);
                if (keyCode == ESCAPE || keyCode == TAB) {
                    deselectActivity();
                } else if (keyCode == DOWN_ARROW) {
                    nextActivity();
                    ev.preventDefault();
                } else if (keyCode == UP_ARROW) {
                    previousActivity();
                    ev.preventDefault();
                }
            }
            
            function onClickBody(event) {
                if (! selecting) {
                    deselectActivity();
                } else {
                    selecting = false;
                }
            }
            
            deselectActivity = function () {
                $(FLYOUT + ' .hp-activity.hp-selected').off('mousedown', onMouseDown);
                $(FLYOUT + ' .hp-activity').removeClass(SELECTED);
                $(BODY).off('click', onClickBody);
                $(document).off('keydown', onKeypress);
            };
            
            selectActivity = function (activity) {
                activity.addClass(SELECTED);
                $(activity).on('mousedown', onMouseDown);
                // position next to activity, unless it would be off the bottom
                if (activity.position().top + $('.hp-full', activity).outerHeight() >
                    $(CONTENTS).position().top + $(CONTENTS).height() + 10) {
                    $('.hp-full', activity).css('top',
                        Math.max(0, $(CONTENTS).position().top +
                            $(CONTENTS).height() -
                            $('.hp-full', activity).outerHeight()));
                } else {
                    $('.hp-full', activity).css('top', activity.position().top);
                }
                $(document).on('keydown', onKeypress);
                // delay to avoid flickering
                setTimeout(function () {$(BODY).on('click', onClickBody);}, 50);
            };
            
            function onClickActivity(event) {
                var activity = $(this).closest('.hp-activity');
                var add = ! activity.hasClass('hp-selected');
                if (! selecting) {
                    deselectActivity();
                    if (add) {
                        selectActivity(activity);
                    }
                } else if (! event || $(event.target).parents(CONTENTS).length === 0) {
                    // let body click handler turn this off
                    selecting = false;
                }
            }
            
            function onScroll() {
                deselectActivity();
            }
            
            /**
             * Completely hide flyout.
             */
            function hideSidebar(remove) {
                if ($(FLYOUT).hasClass(ACTIVE)) {
                    deselectActivity();
                    $(FLYOUT).off('click', '.hp-activity', onClickActivity);
                    $(CONTENTS).off('scroll', onScroll);
                    if (remove) {
                        sidebar.remove($(FLYOUT));
                    }
                    $(FLYOUT).removeClass(ACTIVE);
                    $(CONTROL).removeClass(SELECTED);
                }
            }
            
            /**
             * Put the flyout into the sidebar.
             */
            function showSidebar() {
                if (! $(FLYOUT).hasClass(ACTIVE)) {
                    $(FLYOUT).addClass(ACTIVE);
                    $(CONTROL).addClass(SELECTED);
                    sidebar.add($(FLYOUT));
                    $(FLYOUT).on('click', '.hp-activity', onClickActivity);
                    $(CONTENTS).on('scroll', onScroll);
                    $(LIST + ' > li.' + NEW).removeClass(NEW);
                    updateCount();
                    selecting = false;
                }
            }
            
            function onSidebarRemove(element) {
                if (element.attr('id') === $(FLYOUT).attr('id')) {
                    hideSidebar();
                }
            }
            
            function onSidebarAdd(element) {
                if (element.attr('id') === $(FLYOUT).attr('id')) {
                    showSidebar();
                }
            }
            
            /**
             * Forward reference to the start function, so that Sonar won't
             * complain about calling it before it's defined.
             */
            var start = function() {
            };

            function done() {
                $(BRIEF).css('right','');
            	$(BRIEF).removeClass(ACTIVE);
                running = false;
                start();
            }

            start = function() {
                var notif;
                var bannerIconsWidth = $('#hp-main-banner .hp-header-secondary').outerWidth();
                if (! running && flashNotifications.length > 0) {
                    running = true;
                    notif = flashNotifications.shift();
                    if (! notif.status || 'unknown' != notif.status) {
                        $(BRIEF_STATUS).attr('class',
                            'hp-active hp-status hp-status-' + notif.status);
                    } else {
                        $(BRIEF_STATUS).removeClass('hp-active');
                    }
                    $(BRIEF_MESSAGE).text(notif.summary +
                        (notif.sourceName ? (' ' + notif.sourceName) : ''));
                    $(BRIEF).css('right',bannerIconsWidth);
                    $(BRIEF).addClass(ACTIVE);
                    if (Modernizr.audio && ! settings.disableAudio()) {
                        setTimeout(function () {
                            try {
                                var audio = new Audio();
                                audio.src = Modernizr.audio.mp3 ? 'media/notify.mp3' :
                                            'media/notify.ogg';
                                audio.play();
                            } catch(err) {
                                // do nothing
                            }
                        }, 1000);
                    }
                    setTimeout(done, DELAY);
                }
            };
            
            function setupItem(item, notif) {
                if (notif.status) {
                    $('.hp-status', item).hpStatus(notif.status);
                }
                $('.hp-activity-message', item).text(notif.summary);
                if (notif.timestamp) {
                    $('.hp-timestamp', item).show().
                        text(style.beautifyTimestamp(notif.timestamp));
                } else {
                    $('.hp-timestamp', item).text('').hide();
                }
                if (notif.sourceName) {
                    $('.hp-activity-source', item).text(notif.sourceName);
                }
                if (notif.sourceUri) {
                    var link = $(linkTargetBuilder.makeLink(notif.sourceName, notif.sourceUri)).
                        addClass('hp-activity-source');
                    $('.hp-full .hp-activity-source', item).replaceWith(link);
                }
                if (notif.details) {
                    $('.hp-details', item).html(notif.details);
                } else {
                    $('.hp-details', item).empty().addClass('hp-unset');
                }
                $('.hp-actions', item).empty();
                if (notif.actions) {
                    $.each(notif.actions, function (index, action) {
                        $('.hp-actions', item).
                            append($(action).wrap('<li></li>').parent());
                    });
                }
            }
            
            function sortByTimestamp(a, b) {
                return ((a.timestamp > b.timestamp) ? -1 :
                    ((a.timestamp < b.timestamp) ? 1 : 0));
            }

            /**
             * @private
             * Add an error message to the list.
             * @param {Object} msg The message should contain a "text" property
             *                     which is the (short) text to add.
             *                     It should also contain a "type" property
             *                     whose value is either "warning" or "error".
             *                     This will determine the icon to display.
             */
            function addToList(notif)
            {
                var item = null;
                
                if (! notif.hasOwnProperty('timestamp')) {
                    notif.timestamp = (new Date()).toISOString();
                }
                
                // replace if we already have it
                var existingIndex = -1;
                $.each(notifications, function (index, notif2) {
                    if (notif.uri && notif.uri === notif2.uri) {
                        existingIndex = index;
                        return false;
                    }
                });
                if (existingIndex >= 0) {
                    notifications[existingIndex] = notif;
                } else {
                    notifications.push(notif);
                }
                notifications.sort(sortByTimestamp);
                
                if (existingIndex < 0) {
                    // create a new one
                    item = template.hpSafeClone();
                    if (notif.uri) {
                        item.attr('data-uri', notif.uri);
                    }
                } else if (notif.uri) {
                    item = $(LIST + ' > li[data-uri="' + notif.uri + '"]');
                    if (item.length > 0) {
                        // move to top of list
                        item.remove();
                    }
                }
                
                if (item.length > 0) {
                    item.addClass(ACTIVE);
                    setupItem(item, notif);
                    if (! $(FLYOUT).hasClass(ACTIVE)) {
                        item.addClass(NEW);
                    }
                    $(LIST).prepend(item);
                }
                
                updateCount();
                
                $(FLYOUT).addClass('hp-notable');
            }
            
            /**
             * @public
             * Add a notification message.
             * @param {data} Object with properties as follows:
             *
             *    summary: minimal text summary, should start with an action if possible
             *    uri: identifier for this alert, task, etc.
             *          non-server-side notifications can use anything they want,
             *          this is used to allow updating progress of an existing
             *          notification.
             *    timestamp: timestamp in ISO 8601 string format
             *    sourceName: user visible name for the source
             *    sourceUri: uri for the source
             *    status: one of 'ok', 'error', 'warning', 'info'
             *    changing: true|false, whether the status should be anotated as changing
             *    progress: 0-100, a number indication what % of a task has been completed
             *    step: a short text description of the current progress step
             *    details: longer description that can have line breaks and links,
             *              will show up in a tooltip/flyout
             *
             *    The only property that is required is the summary.
             *
             * @param flash Boolean flash in banner    
             */
            this.add = function (notif, flash) {
                addToList(notif);
                if (flash) {
                    flashNotifications.push(notif);
                    if (! running) {
                        start();
                    }
                }
            };
            
            // Deprecated !!! restoring just for Fusion beta
            this.show = function (text, status) {
                this.add({summary: text, status: status}, true);
            };

            this.init = function(banner, placement) {
                template = $(TEMPLATE, banner);
                template.remove().attr('id', '').removeClass('hp-template');
                
                if (! placement || placement === 'sidebar') {
                    $(BRIEF, banner).removeClass(ACTIVE);
                    $(CONTROL, banner).attr('tooltip',
                        localizer.getString('core.notify.control.tooltip')).
                        hpTooltip();
                    $(CONTROL, banner).click(function (ev) {
                        if (!$(CONTROL).hasClass(SELECTED)) {
                            showSidebar();
                        } else {
                            hideSidebar(true);
                        }
                    });
                    sidebar.on('sidebarAdd', onSidebarAdd);
                    sidebar.on('sidebarRemove', onSidebarRemove);
                } else if (placement === 'inline') {
                    $(CONTROL, banner).parent().hide();
                }
            };
        }

        return new Notifications();
    }());

    return Notifications;
});
