// (C) Copyright 2011-2013 Hewlett-Packard Development Company, L.P.
define(['hp/core/UrlFragment', 'jquery'], function (urlFragment) {
"use strict";

    var IndexFilter = (function () {

        var PROP_MAP = {
            // query string property name : REST argument name
            'f_suri' : 'startObjUri',
            'f_euri' : 'endObjUri',
            'f_an' : 'associationName',
            'f_q' : 'userQuery',
            'f_category' : 'category',
            'f_name' : 'name',
            'f_start' : 'start',
            'f_count' : 'count',
            'f_sort' : 'sort'
        };

        // When hasFilterResetFlag flag is set, a call to updateLocation() will add 
        // a flag FILTER_RESET to the param list to signal that the user
        // has selected one of the "reset all" filter. The existence of this 
        // flag in location prevents the reuse of the old filter stored prior.
        var FILTER_RESET = 'freset=true';
        var hasFilterResetFlag = false;

        /**
         * arg is either a location or an IndexFilter to start from
         */
        function IndexFilter(arg) {

            var data = {};
            var defaults = {};
            
            // trim white space and remove surrounding quotes from all entries in the array
            function trimEntries(terms) {
                if (terms) {
                    $.each(terms, function(index,term) {
                        terms[index] = $.trim(terms[index]);
                        terms[index] = terms[index].replace(/^"|"$/g, '');
                        terms[index] = terms[index].replace(/:"/g, ':');
                        terms[index] = terms[index].replace(/^'|'$/g, '');
                        terms[index] = terms[index].replace(/:'/g, ':');
                    });
                }
            }

            // extract all the terms from query. Expect query to be a non null string.
            // 'property terms' are terms separated by ":"; they are returned in "properties"
            // e.g. roles:"system administrator"
            //      firmware:"need update"
            //      status:ok
            // 'user terms' are terms that are not property; they are returned in "userTerms"
            // e.g. "John Doe"
            //      fc_switch
            //      rack
            function extractQueryTerms(query) {

                // extract properties e.g. roles:"System Administrator" category:switch
                // doesn't handle escaped quotes in value
                var propertyRegexp = /[\w\-\.]+:(("[^"]+")|([\S]+))/g;
                var properties = query.match(propertyRegexp);
                if (properties) {
                    trimEntries(properties);
                    query = query.replace(propertyRegexp, '');
                }

                // extract all double quoted terms, e.g. "rack 3".
                // allow escaping quotes inside, e.g. "rack\"s 3"
                var userTermRegexp = /"(?:[^"\\]|\\.)+"/g;
                var userTerms = query.match(userTermRegexp);
                if (userTerms) {
                    trimEntries(userTerms);
                    query = query.replace(userTermRegexp, '');
                }
                // extract all single quoted terms, e.g. 'rack 3'.
                userTermRegexp = /'(?:[^'\\]|\\.)+'/g;
                var userTerms2 = query.match(userTermRegexp);
                if (userTerms2) {
                    trimEntries(userTerms2);
                    query = query.replace(userTermRegexp, '');
                    if (userTerms) {
                        userTerms = userTerms.concat(userTerms2);
                    } else {
                        userTerms = userTerms2;
                    }
                }

                // extract the rest of the non-quoted terms
                if ($.trim(query).length > 0) {
                    var tmp = query.match(/[\S]+/g);
                    if (userTerms) {
                        $.merge(userTerms, tmp);
                    } else {
                        userTerms = tmp;
                    }
                }

                return {properties: properties, userTerms: userTerms ? userTerms : []}; 
            }

            // parse user search terms from the query string
            function parseQueryTerms() {
                if (data.userQuery) {
                    data.terms = extractQueryTerms(data.userQuery).userTerms;
                }
            }

            // formalize any parameters in the query
            function parseQuery() {
                if (!data.userQuery) {
                    return;
                }
                var queryTerms = extractQueryTerms(data.userQuery);
                if (queryTerms.properties) {
                    $.each(queryTerms.properties, function (index, param) {
                        var fields = param.split(':');
                        var name = fields[0];
                        var value = fields.slice(1).join(':');
                        if (! data.properties) {
                            data.properties = {};
                        }
                        if (data.properties[name]) {
                            // already have, make an array and add
                            if ('array' !== $.type(data.properties[name])) {
                                data.properties[name] = [data.properties[name]];
                            }
                            data.properties[name].push(value);
                        } else {
                            data.properties[name] = value;
                        }
                    });
                }

                data.terms = queryTerms.userTerms;
            }

            // add double quotes to val if it contains white space or an embedded quote
            function addQuote(val) {
                return (/\s|"/g.test(val)) ? '"' + val + '"' : val;
            }

            // generate query string from formalized parameters
            function generateQuery() {
                var terms = [];

                if (data.terms) {
                    $.each(data.terms, function (idx, value) {
                        terms.push(addQuote(value));
                    });
                }

                if (data.properties) {
                    $.each(data.properties, function (name, value) {
                        if ($.isArray(value)) {
                            $.each(value, function (index, val) {
                                terms.push(name + ':' + addQuote(val));
                            });
                        } else {
                            terms.push(name + ':' + addQuote(value));
                        }
                    });
                }

                if (terms.length > 0) {
                    data.userQuery = terms.join(' ');
                    // 2000 limit based on:
                    // http://stackoverflow.com/questions/417142/what-is-the-maximum-length-of-a-url
                    // Even though the overall query would be even longer, this is to catch
                    // problems with the just the query portion.
                    if (data.userQuery.length > 2000) {
                        throw ("query too long");
                    }
                } else {
                    delete data.userQuery;
                }
            }

            function hasResetParam(paramList) {
                var resetFlag = false;
                $.each(paramList, function (index, param) {
                    if (index === 'freset') {
                        resetFlag = true;
                        return false;
                    } 
                });
                return resetFlag;
            }

            function loadUrlParameters(urlParameters) {
                hasFilterResetFlag = hasResetParam(urlParameters);
                $.each(PROP_MAP, function (queryParam, dataProp) {
                    if (urlParameters.hasOwnProperty(queryParam)) {
                        data[dataProp] = urlParameters[queryParam];
                    }
                });

                if (data.hasOwnProperty('associationName')) {
                    data.hasARelationship = true;
                }

                // check for variable properties
                $.each(urlParameters, function (name, value) {
                    var match = name.match(/^fp_(.*)/);
                    if (match) {
                        if (! data.properties) {
                            data.properties = {};
                        }
                        data.properties[match[1]] = value;
                    }
                });

                // If don't have formal properties, use whatever's in the query string
                if (data.userQuery && ! data.properties) {
                    parseQuery();
                } else {
                    // otherwise, get the non-formal search terms and generate the
                    // a fresh query with the formal properties
                    parseQueryTerms();
                }
                generateQuery();
            }

            function arePropertiesDifferent(props1, props2) {
                var result = false;

                // NOTE: It's hard to improve coverage here because
                // we tend to catch differences comparing the user query
                // before we get here

                $.each(props1, function (n1, v1) {
                    if (! props2.hasOwnProperty(n1)) {
                        result = true;
                        return false;
                    } else {
                        var v2 = props2[n1];
                        if ($.isArray(v2) && $.isArray(v1)) {
                            if ($(v1).not(v2).length !== 0 ||
                                $(v2).not(v1).length !== 0) {
                                result = true;
                                return false;
                            }
                        } else if (v1 !== v2) {
                            result = true;
                            return false;
                        }
                    }
                });

                if (! result) {
                    // just to check for missing properties the other way
                    $.each(props2, function (n2, v2) {
                        if (! props1.hasOwnProperty(n2)) {
                            result = true;
                            return false;
                        }
                    });
                }

                return result;
            }

            function initialize() {
                if ('string' === typeof arg) {
                    var urlParameters = urlFragment.getParameters(arg);
                    loadUrlParameters(urlParameters);
                } else if (arg && arg.hasOwnProperty('data')) {
                    $.extend(data, arg.data);
                    delete data.properties;
                    if (arg.data.properties) {
                        data.properties = {};
                        $.extend(data.properties, arg.data.properties);
                    }
                    $.extend(defaults, arg.defaults);
                }
            }

            this.updateLocation = function (location) {
                var oldParameters = urlFragment.getParameters(location);
                var newParameters = {};

                // add non-filter parameters
                $.each(oldParameters, function (name, value) {
                    if (! name.match(/^f_/) && ! name.match(/^fp_/)) {
                        // not a filter, keep it
                        newParameters[name] = value;
                    }
                });

                // add formal properties
                $.each(PROP_MAP, function (queryParam, dataProp) {
                    if (data.hasOwnProperty(dataProp) &&
                        data[dataProp].length > 0) {
                        if ('category' !== dataProp &&
                            (! defaults.hasOwnProperty(dataProp) ||
                            defaults[dataProp] !== data[dataProp])) {
                            newParameters[queryParam] = data[dataProp];
                        }
                    }
                });

                // add variable properties
                if (data.properties) {
                    $.each(data.properties, function (name, value) {
                        if (value.length > 0) {
                            newParameters['fp_' + name] = value;
                        }
                    });
                }
                
                if (hasFilterResetFlag) {
                    newParameters['freset'] = 'true';
                } else {
                    delete newParameters['freset'];
                }

                return urlFragment.replaceParameters(location, newParameters);
            };


            function sameOrDefault(data, data2, defaults,dataProp) {
                return (
                    (data.hasOwnProperty(dataProp) &&
                        data2.hasOwnProperty(dataProp) &&
                        data[dataProp] === data2[dataProp]) ||      // Same
                    (!data.hasOwnProperty(dataProp) &&
                        !data2.hasOwnProperty(dataProp)) ||         // Same
                    (defaults.hasOwnProperty(dataProp) &&
                        defaults[dataProp] === data[dataProp] &&
                        !data2.hasOwnProperty(dataProp))            // Default
                );
            }

            this.isDifferent = function (filter2) {
                var result = (filter2 === null || typeof filter2 === 'undefined');
                var data2;

                if (result && ! data.properties) {
                    result = false;
                    // filter2 doesn't have anything, see if all this has is defaults
                    $.each(PROP_MAP, function (queryParam, dataProp) {
                        if (data.hasOwnProperty(dataProp) &&
                            (!defaults.hasOwnProperty(dataProp) ||
                            defaults[dataProp] !== data[dataProp])) {
                            result = true;
                            return true;
                        }
                    });
                }

                if (! result && filter2) {
                    data2 = filter2.data;
                    // check formal filter properties
                    $.each(PROP_MAP, function (queryParam, dataProp) {
                        if (!sameOrDefault(data, data2, defaults, dataProp)) {
                            result = true;
                            return false;
                        }
                    });
                }

                if (! result && data2) {
                    // check variable filter properties
                    if (! data.properties && ! data2.properties) {
                        // same
                    } else if (data.properties && data2.properties) {
                        // check all properties
                        if (arePropertiesDifferent(data.properties,
                            data2.properties)) {
                            result = true;
                        }
                    } else {
                        // one has, one doesn't
                        result = true;
                    }
                }

                return result;
            };

            this.isCustom = function () {
                return (data.hasOwnProperty('userQuery') ||
                    data.hasOwnProperty('associationName') ||
                    data.hasOwnProperty('properties'));
            };

            this.getUserQuery = function () {
                return data.userQuery;
            };

            this.setUserQuery = function (query) {
                data.userQuery = query;
                parseQuery();
                generateQuery();
            };

            this.getReferrerUri = function () {
                if (data.hasOwnProperty('startObjUri')) {
                    return data.startObjUri;
                } else if (data.hasOwnProperty('endObjUri')) {
                    return data.endObjUri;
                } else {
                    return null;
                }
            };

            this.setProperty = function (name, value) {
                var result = true;
                if (! data.properties) {
                    data.properties = {};
                }
                if (data.properties[name] === value) {
                    result = false;
                } else {
                    data.properties[name] = value;
                    generateQuery();
                }
                return result;
            };

            this.unsetProperty = function (name) {
                var result = false;
                if (data.properties) {
                    if (data.properties.hasOwnProperty(name)) {
                        delete data.properties[name];
                        result = true;
                        generateQuery();
                    }
                }
                return result;
            };

            this.getProperty = function (name) {
                if (data.properties) {
                    return data.properties[name];
                } else {
                    return undefined;
                }
            };

            this.setProperties = function (properties) {
                data.properties = properties;
                generateQuery();
            };

            this.setSort = function (propertyName, direction) {
                if (propertyName) {
                    // allow for string args or single object
                    if ('string' === typeof propertyName) {
                        data.sort = propertyName + ':' + direction;
                    } else {
                        data.sort = propertyName.name + ':' +
                            propertyName.direction;
                    }
                } else {
                    delete data.sort;
                }
            };

            this.getSort = function () {
                if (data.sort) {
                    var parts = data.sort.split(':');
                    return {name: parts[0], direction: parts[1]};
                } else {
                    return null;
                }
            };

            this.bumpCount = function () {
                if(data.defaultCount) {
                    data.count += data.defaultCount;
                } else {
                    data.count += defaults.count;
                }
            };

            this.resetCount = function () {
                if (defaults.hasOwnProperty('count') && data.count !== defaults.count) {
                    data.count = defaults.count;
                    return true;
                }
                return false;
            };

            this.useCount = function (filter2) {
                if (filter2 && (! this.data.count ||
                    (filter2.data.count > this.data.count))) {
                    this.data.count = filter2.data.count;
                }
                if (filter2 && filter2.defaults &&
                    (! this.defaults.hasOwnProperty('count'))) {
                    this.defaults.count = filter2.defaults.count;
                }
            };

            // @private, for IndexService and Resource

            this.data = data;
            this.defaults = defaults;

            this.ensureDefaults = function (category, start, count) {
                defaults.category = category;
                defaults.start = start;
                defaults.count = count;

                if (data.properties && data.properties.hasOwnProperty('category')) {
                    data.category = data.properties.category;
                }
                if (! data.hasOwnProperty('category')) {
                    data.category = category;
                }
                if (! data.hasOwnProperty('start')) {
                    data.start = start;
                }
                if (! data.hasOwnProperty('count')) {
                    data.count = count;
                }
            };

            this.hasResetFlag = function () {
                return hasFilterResetFlag;
            };
 
            this.unsetResetFlag = function () {
                hasFilterResetFlag = false;
            };
 
            this.setResetFlag = function () {
                hasFilterResetFlag = true;
            };

            // Reset this filter using the info in location and userQuery to
            // prepare for a local search. See SearchBoxPresenter.
            this.updateLocalSearchQuery = function (location, userQuery) {
                var parameters = urlFragment.getParameters(location);
                if (userQuery && userQuery.length > 0) {
                    parameters.f_q = userQuery;
                    // clear uris on filter change
                    location = urlFragment.replaceUris(location, []);
                } else {
                    delete parameters.f_q;
                }

                if (!userQuery || userQuery.length === 0) {
                    parameters = [FILTER_RESET];
                } else {
                    // delete any formalized filter parameters
                    $.each(parameters, function (name, value) {
                        if (name.match('^fp_')) {
                            delete parameters[name];
                        }
                    });
                }

                // reset this filter
                data={};
                defaults={};
                location = urlFragment.replaceParameters(location, parameters);
                loadUrlParameters(urlFragment.getParameters(location));
            };

            this.toString = function () {
                var string = '';
                // add formal properties
                $.each(PROP_MAP, function (queryParam, dataProp) {
                    if (data.hasOwnProperty(dataProp)) {
                        if ($.isArray(data[dataProp])) {
                            string += queryParam + "[]=" + data[dataProp].join(',') + ' ';
                        } else {
                            string += queryParam + '=' + data[dataProp] + ' ';
                        }
                    }
                });
                // add variable properties
                if (data.properties) {
                    $.each(data.properties, function (name, value) {
                        if ($.isArray(value)) {
                            string += 'fp_' + name + "[]=" + value.join(',') + ' ';
                        } else {
                            string += 'fp_' + name + '=' + value + ' ';
                        }
                    });
                }
                // defaults
                $.each(defaults, function (n, v) {
                    string += ' D:' + n + '=' + v;
                });
                return string;
            };

            initialize();
        }

        return IndexFilter;
    }());

    return IndexFilter;
});
