/*******************************************************************************
 * The information contained in this file is confidential and proprietary to
 * QLogic Corporation.  No part of this file may be reproduced or
 * distributed, in any form or by any means for any purpose, without the
 * express written permission of QLogic Corporation.
 *
 * (c) COPYRIGHT 2015 QLogic Corporation, ALL RIGHTS RESERVED.
 *******************************************************************************/
/* **********************************************************
 * Copyright 2015 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * qfle3_dcbx.c --
 *
 *      Kernel module implementation for native qfle3 driver.
 */

#include "qfle3.h"
#include "qfle3_mgmt_api.h"

void
qfle3_dcbx_start(struct qfle3_adapter *adapter)
{
    u32 dcbx_lldp_params_offset = SHMEM_LLDP_DCBX_PARAMS_NONE;

    if (!adapter->dcbx_sb.enabled)
       return;

    /* only PMF can send ADMIN msg to MFW in old MFW versions */
    if ((!adapter->port.pmf) && (!(adapter->flags & BC_SUPPORTS_DCBX_MSG_NON_PMF)))
       return;

    QFLE3_DBG(QFLE3_DBG_DCB, "pmf: %d\n", adapter->port.pmf);

    if (SHMEM2_HAS(adapter, dcbx_lldp_params_offset)) {
       dcbx_lldp_params_offset =
               SHMEM2_RD(adapter, dcbx_lldp_params_offset);

       QFLE3_DBG(QFLE3_DBG_DCB, "dcbx_lldp_params_offset 0x%x\n",
                                                 dcbx_lldp_params_offset);

       qfle3_update_drv_flags(adapter, 1 << DRV_FLAGS_DCB_CONFIGURED, 0);

       if (SHMEM_LLDP_DCBX_PARAMS_NONE != dcbx_lldp_params_offset) {
          qfle3_acquire_hw_lock(adapter, HW_LOCK_RESOURCE_DCBX_ADMIN_MIB);

          /* Trigger DCBX negotiation in adapter. */
          qfle3_mfw_command(adapter, DRV_MSG_CODE_DCBX_ADMIN_PMF_MSG, 0);

          /* Release HW lock only after MFW acks that it finished
           * reading values from shmem
           */
          qfle3_release_hw_lock(adapter, HW_LOCK_RESOURCE_DCBX_ADMIN_MIB);
          
          QFLE3_DBG(QFLE3_DBG_DCB, "DCBX started in FW\n");
       }
       }
    }

void
qfle3_read_data(struct qfle3_adapter *adapter, u32 *buff,
                                        u32 addr, u32 len)
{
   int i;
   for (i = 0; i < len; i += 4, buff++)
      *buff = REG_RD(adapter, addr + i);
}

static void
qfle3_dcbx_get_pfc_feature(struct qfle3_adapter *adapter)
{
   struct dcbx_pfc_feature *pfc = &adapter->dcbx_sb.local_feat.pfc;
   u32 error = adapter->dcbx_sb.dcbx_error;

   if (GET_FLAGS(error, DCBX_LOCAL_PFC_ERROR))
      QFLE3_DBG(QFLE3_DBG_DCB, "DCBX_LOCAL_PFC_ERROR\n");

   if (GET_FLAGS(error, DCBX_REMOTE_PFC_TLV_NOT_FOUND))
      QFLE3_DBG(QFLE3_DBG_DCB, "DCBX_REMOTE_PFC_TLV_NOT_FOUND\n");

   if (adapter->dcbx_sb.port_params.app.enabled && pfc->enabled &&
         !GET_FLAGS(error, DCBX_LOCAL_PFC_ERROR | DCBX_LOCAL_PFC_MISMATCH |
            DCBX_REMOTE_PFC_TLV_NOT_FOUND)) {
      QFLE3_DBG(QFLE3_DBG_DCB, "DCBX_LOCAL_PFC_ENABLED\n");
      adapter->dcbx_sb.port_params.pfc.enabled = VMK_TRUE;
      adapter->dcbx_sb.port_params.pfc.priority_non_pauseable_mask =
         ~(pfc->pri_en_bitmap);
   } else {
      QFLE3_DBG(QFLE3_DBG_DCB, "DCBX_LOCAL_PFC_DISABLED\n");
      adapter->dcbx_sb.port_params.pfc.enabled = VMK_FALSE;
      adapter->dcbx_sb.port_params.pfc.priority_non_pauseable_mask = 0;
   }
}

void
qfle3_dcbx_get_ets_pri_pg_tbl(struct qfle3_adapter *adapter,
                                u32 *set_configuration_ets_pg,
                                u32 *pri_pg_tbl)
{
   int i;

   for (i = 0; i < DCBX_MAX_NUM_PRI_PG_ENTRIES; i++) {
      set_configuration_ets_pg[i] = DCBX_PRI_PG_GET(pri_pg_tbl, i);

      QFLE3_DBG(QFLE3_DBG_DCB, "set_configuration_ets_pg[%d] = 0x%x\n",
            i, set_configuration_ets_pg[i]);
   }
}

static void
qfle3_dcbx_get_num_pg_traf_type(struct qfle3_adapter *adapter,
                                u32 *pg_pri_orginal_spread,
                                struct pg_help_data *help_data)
{
   vmk_Bool pg_found  = VMK_FALSE;
   u32 i, traf_type, add_traf_type, add_pg;
   u32 *ttp = adapter->dcbx_sb.port_params.app.traffic_type_priority;
   struct pg_entry_help_data *data = help_data->data; /*shortcut*/

   /* Set to invalid */
   for (i = 0; i < LLFC_DRIVER_TRAFFIC_TYPE_MAX; i++)
      data[i].pg = DCBX_ILLEGAL_PG;

   for (add_traf_type = 0;
         add_traf_type < LLFC_DRIVER_TRAFFIC_TYPE_MAX; add_traf_type++) {
      pg_found = VMK_FALSE;
      if (ttp[add_traf_type] < MAX_PFC_PRIORITIES) {
         add_pg = (u8)pg_pri_orginal_spread[ttp[add_traf_type]];
         for (traf_type = 0;
               traf_type < LLFC_DRIVER_TRAFFIC_TYPE_MAX;
               traf_type++) {
            if (data[traf_type].pg == add_pg) {
               if (!(data[traf_type].pg_priority &
                        (1 << ttp[add_traf_type])))
                  data[traf_type].
                     num_of_dif_pri++;
               data[traf_type].pg_priority |=
                  (1 << ttp[add_traf_type]);
               pg_found = VMK_TRUE;
               break;
            }
         }
         if (VMK_FALSE== pg_found) {
            data[help_data->num_of_pg].pg = add_pg;
            data[help_data->num_of_pg].pg_priority =
               (1 << ttp[add_traf_type]);
            data[help_data->num_of_pg].num_of_dif_pri = 1;
            help_data->num_of_pg++;
         }
      }
      QFLE3_DBG(QFLE3_DBG_DCB,
            "add_traf_type %d pg_found %s num_of_pg %d\n",
            add_traf_type, (VMK_FALSE == pg_found) ? "NO" : "YES",
            help_data->num_of_pg);
   }
}

static int qfle3_dcbx_join_pgs(
                        struct qfle3_adapter *adapter,
                        struct dcbx_ets_feature *ets,
                        struct pg_help_data     *pg_help_data,
                        u8                      required_num_of_pg)
{
   u8 entry_joined    = pg_help_data->num_of_pg - 1;
   u8 entry_removed   = entry_joined + 1;
   u8 pg_joined       = 0;

   if (required_num_of_pg == 0 || ARRAY_SIZE(pg_help_data->data)
         <= pg_help_data->num_of_pg) {

      QFLE3_ERR("required_num_of_pg can't be zero\n");
      return VMK_FAILURE;
   }

   while (required_num_of_pg < pg_help_data->num_of_pg) {
      entry_joined = pg_help_data->num_of_pg - 2;
      entry_removed = entry_joined + 1;
      /* protect index */
      entry_removed %= ARRAY_SIZE(pg_help_data->data);

      pg_help_data->data[entry_joined].pg_priority |=
         pg_help_data->data[entry_removed].pg_priority;

      pg_help_data->data[entry_joined].num_of_dif_pri +=
         pg_help_data->data[entry_removed].num_of_dif_pri;

      if (pg_help_data->data[entry_joined].pg == DCBX_STRICT_PRI_PG ||
            pg_help_data->data[entry_removed].pg == DCBX_STRICT_PRI_PG)
         /* Entries joined strict priority rules */
         pg_help_data->data[entry_joined].pg =
            DCBX_STRICT_PRI_PG;
      else {
         /* Entries can be joined join BW */
         pg_joined = DCBX_PG_BW_GET(ets->pg_bw_tbl,
               pg_help_data->data[entry_joined].pg) +
            DCBX_PG_BW_GET(ets->pg_bw_tbl,
                  pg_help_data->data[entry_removed].pg);

         DCBX_PG_BW_SET(ets->pg_bw_tbl,
               pg_help_data->data[entry_joined].pg, pg_joined);
      }
      /* Joined the entries */
      pg_help_data->num_of_pg--;
   }

   return 0;
}

static void qfle3_dcbx_ets_disabled_entry_data(struct qfle3_adapter *adapter,
                                                struct cos_help_data *cos_data,
                                                u32 pri_join_mask)
{
   /* Only one priority than only one COS */
   cos_data->data[0].pausable =
      IS_DCBX_PFC_PRI_ONLY_PAUSE(adapter, pri_join_mask);
   cos_data->data[0].pri_join_mask = pri_join_mask;
   cos_data->data[0].cos_bw = 100;
   cos_data->num_of_cos = 1;
}

static int
qfle3_dcbx_spread_strict_pri(struct qfle3_adapter *adapter,
                                struct cos_help_data *cos_data,
                                u8 entry,
                                u8 num_spread_of_entries,
                                u8 strict_app_pris)
{
   u8 strict_pri = QFLE3_DCBX_STRICT_COS_HIGHEST;
   u8 num_of_app_pri = MAX_PFC_PRIORITIES;
   u8 app_pri_bit = 0;

   while (num_spread_of_entries && num_of_app_pri > 0) {
      app_pri_bit = 1 << (num_of_app_pri - 1);
      if (app_pri_bit & strict_app_pris) {
         struct cos_entry_help_data *data = &cos_data->
            data[entry];
         num_spread_of_entries--;
         if (num_spread_of_entries == 0) {
            /* last entry needed put all the entries left */
            data->cos_bw = DCBX_INVALID_COS_BW;
            data->strict = strict_pri;
            data->pri_join_mask = strict_app_pris;
            data->pausable = DCBX_IS_PFC_PRI_SOME_PAUSE(adapter,
                  data->pri_join_mask);
         } else {
            strict_app_pris &= ~app_pri_bit;

            data->cos_bw = DCBX_INVALID_COS_BW;
            data->strict = strict_pri;
            data->pri_join_mask = app_pri_bit;
            data->pausable = DCBX_IS_PFC_PRI_SOME_PAUSE(adapter,
                  data->pri_join_mask);
         }

         strict_pri =
            QFLE3_DCBX_STRICT_COS_NEXT_LOWER_PRI(strict_pri);
         entry++;
      }

      num_of_app_pri--;
   }

   if (num_spread_of_entries) {
      QFLE3_ERR("Didn't succeed to spread strict priorities\n");
      return VMK_FAILURE;
   }

   return 0;
}


static u8
qfle3_dcbx_cee_fill_strict_pri(struct qfle3_adapter *adapter,
                                struct cos_help_data *cos_data,
                                u8 entry,
                                u8 num_spread_of_entries,
                                u8 strict_app_pris)
{
   if (qfle3_dcbx_spread_strict_pri(adapter, cos_data, entry,
            num_spread_of_entries,
            strict_app_pris)) {
      struct cos_entry_help_data *data = &cos_data->
         data[entry];
      /* Fill BW entry */
      data->cos_bw = DCBX_INVALID_COS_BW;
      data->strict = QFLE3_DCBX_STRICT_COS_HIGHEST;
      data->pri_join_mask = strict_app_pris;
      data->pausable = DCBX_IS_PFC_PRI_SOME_PAUSE(adapter,
            data->pri_join_mask);
      return 1;
   }

   return num_spread_of_entries;
}

static void qfle3_dcbx_cee_fill_cos_params(struct qfle3_adapter *adapter,
                                        struct pg_help_data *help_data,
                                        struct dcbx_ets_feature *ets,
                                        struct cos_help_data *cos_data,
                                        u32 pri_join_mask)
{
   u8 need_num_of_entries = 0;
   u8 i = 0;
   u8 entry = 0;

   /*
    * if the number of requested PG-s in CEE is greater than 3
    * then the results are not determined since this is a violation
    * of the standard.
    */
   if (help_data->num_of_pg > DCBX_COS_MAX_NUM_E3B0) {
      if (qfle3_dcbx_join_pgs(adapter, ets, help_data,
               DCBX_COS_MAX_NUM_E3B0)) {
         QFLE3_ERR("Unable to reduce the number of PGs - we will disables ETS\n");
         qfle3_dcbx_ets_disabled_entry_data(adapter, cos_data,
               pri_join_mask);
         return;
      }
   }

   for (i = 0 ; i < help_data->num_of_pg; i++) {
      struct pg_entry_help_data *pg =  &help_data->data[i];
      if (pg->pg < DCBX_MAX_NUM_PG_BW_ENTRIES) {
         struct cos_entry_help_data *data = &cos_data->
            data[entry];
         /* Fill BW entry */
         data->cos_bw = DCBX_PG_BW_GET(ets->pg_bw_tbl, pg->pg);
         data->strict = QFLE3_DCBX_STRICT_INVALID;
         data->pri_join_mask = pg->pg_priority;
         data->pausable = DCBX_IS_PFC_PRI_SOME_PAUSE(adapter,
               data->pri_join_mask);

         entry++;
      } else {
         need_num_of_entries = MIN_OF(
               (u8)pg->num_of_dif_pri,
               (u8)DCBX_COS_MAX_NUM_E3B0 -
               help_data->num_of_pg + 1);
         /*
          * If there are still VOQ-s which have no associated PG,
          * then associate these VOQ-s to PG15. These PG-s will
          * be used for SP between priorities on PG15.
          */
         entry += qfle3_dcbx_cee_fill_strict_pri(adapter, cos_data,
               entry, need_num_of_entries, pg->pg_priority);
      }
   }

   /* the entry will represent the number of COSes used */
   cos_data->num_of_cos = entry;
}

static void
qfle3_dcbx_2cos_limit_cee_single_pg_to_cos_params(struct qfle3_adapter *adapter,
                                        struct pg_help_data *pg_help_data,
                                        struct cos_help_data *cos_data,
                                        u32 pri_join_mask,
                                        u8 num_of_dif_pri)
{
   u8 i = 0;
   u32 pri_tested = 0;
   u32 pri_mask_without_pri = 0;
   u32 *ttp = adapter->dcbx_sb.port_params.app.traffic_type_priority;
   /*debug*/
   if (num_of_dif_pri == 1) {
      qfle3_dcbx_ets_disabled_entry_data(adapter, cos_data, pri_join_mask);
      return;
   }
   /* single priority group */
   if (pg_help_data->data[0].pg < DCBX_MAX_NUM_PG_BW_ENTRIES) {
      /* If there are both pauseable and non-pauseable priorities,
       * the pauseable priorities go to the first queue and
       * the non-pauseable priorities go to the second queue.
       */
      if (IS_DCBX_PFC_PRI_MIX_PAUSE(adapter, pri_join_mask)) {
         /* Pauseable */
         cos_data->data[0].pausable = VMK_TRUE;
         /* Non pauseable.*/
         cos_data->data[1].pausable = VMK_FALSE;

         if (2 == num_of_dif_pri) {
            cos_data->data[0].cos_bw = 50;
            cos_data->data[1].cos_bw = 50;
         }

         if (3 == num_of_dif_pri) {
            if (POWER_OF_2(DCBX_PFC_PRI_GET_PAUSE(adapter,
                        pri_join_mask))) {
               cos_data->data[0].cos_bw = 33;
               cos_data->data[1].cos_bw = 67;
            } else {
               cos_data->data[0].cos_bw = 67;
               cos_data->data[1].cos_bw = 33;
            }
         }

      } else if (IS_DCBX_PFC_PRI_ONLY_PAUSE(adapter, pri_join_mask)) {
         /* If there are only pauseable priorities,
          * then one/two priorities go to the first queue
          * and one priority goes to the second queue.
          */
         if (2 == num_of_dif_pri) {
            cos_data->data[0].cos_bw = 50;
            cos_data->data[1].cos_bw = 50;
         } else {
            cos_data->data[0].cos_bw = 67;
            cos_data->data[1].cos_bw = 33;
         }
         cos_data->data[1].pausable = VMK_TRUE;
         cos_data->data[0].pausable = VMK_TRUE;
         /* All priorities except FCOE */
         cos_data->data[0].pri_join_mask = (pri_join_mask &
               ((u8)~(1 << ttp[LLFC_TRAFFIC_TYPE_FCOE])));
         /* Only FCOE priority.*/
         cos_data->data[1].pri_join_mask =
            (1 << ttp[LLFC_TRAFFIC_TYPE_FCOE]);
      } else
         /* If there are only non-pauseable priorities,
          * they will all go to the same queue.
          */
         qfle3_dcbx_ets_disabled_entry_data(adapter,
               cos_data, pri_join_mask);
   } else {
      /* priority group which is not BW limited (PG#15):*/
      if (IS_DCBX_PFC_PRI_MIX_PAUSE(adapter, pri_join_mask)) {
         /* If there are both pauseable and non-pauseable
          * priorities, the pauseable priorities go to the first
          * queue and the non-pauseable priorities
          * go to the second queue.
          */
         if (DCBX_PFC_PRI_GET_PAUSE(adapter, pri_join_mask) >
               DCBX_PFC_PRI_GET_NON_PAUSE(adapter, pri_join_mask)) {
            cos_data->data[0].strict =
               QFLE3_DCBX_STRICT_COS_HIGHEST;
            cos_data->data[1].strict =
               QFLE3_DCBX_STRICT_COS_NEXT_LOWER_PRI(
                     QFLE3_DCBX_STRICT_COS_HIGHEST);
         } else {
            cos_data->data[0].strict =
               QFLE3_DCBX_STRICT_COS_NEXT_LOWER_PRI(
                     QFLE3_DCBX_STRICT_COS_HIGHEST);
            cos_data->data[1].strict =
               QFLE3_DCBX_STRICT_COS_HIGHEST;
         }
         /* Pauseable */
         cos_data->data[0].pausable = VMK_TRUE;
         /* Non pause-able.*/
         cos_data->data[1].pausable = VMK_FALSE;
      } else {
         /* If there are only pauseable priorities or
          * only non-pauseable,* the lower priorities go
          * to the first queue and the higher priorities go
          * to the second queue.
          */
         cos_data->data[0].pausable =
            cos_data->data[1].pausable =
            IS_DCBX_PFC_PRI_ONLY_PAUSE(adapter, pri_join_mask);

         for (i = 0 ; i < LLFC_DRIVER_TRAFFIC_TYPE_MAX; i++) {
            pri_tested = 1 << adapter->dcbx_sb.port_params.
               app.traffic_type_priority[i];
            /* Remove priority tested */
            pri_mask_without_pri =
               (pri_join_mask & ((u8)(~pri_tested)));
            if (pri_mask_without_pri < pri_tested)
               break;
         }

         if (i == LLFC_DRIVER_TRAFFIC_TYPE_MAX)
            QFLE3_ERR("Invalid value for pri_join_mask - could not find a priority\n");

         cos_data->data[0].pri_join_mask = pri_mask_without_pri;
         cos_data->data[1].pri_join_mask = pri_tested;
         /* Both queues are strict priority,
          * and that with the highest priority
          * gets the highest strict priority in the arbiter.
          */
         cos_data->data[0].strict =
            QFLE3_DCBX_STRICT_COS_NEXT_LOWER_PRI(
                  QFLE3_DCBX_STRICT_COS_HIGHEST);
         cos_data->data[1].strict =
            QFLE3_DCBX_STRICT_COS_HIGHEST;
      }
   }
}

static inline void qfle3_dcbx_add_to_cos_bw(struct qfle3_adapter *adapter,
                                        struct cos_entry_help_data *data,
                                        u8 pg_bw)
{
   if (data->cos_bw == DCBX_INVALID_COS_BW)
      data->cos_bw = pg_bw;
   else
      data->cos_bw += pg_bw;
}


static void
qfle3_dcbx_separate_pauseable_from_non(struct qfle3_adapter *adapter,
                                struct cos_help_data *cos_data,
                                u32 *pg_pri_orginal_spread,
                                struct dcbx_ets_feature *ets)
{
   u32	pri_tested	= 0;
   u8	i		= 0;
   u8	entry		= 0;
   u8	pg_entry	= 0;
   u8	num_of_pri	= LLFC_DRIVER_TRAFFIC_TYPE_MAX;

   cos_data->data[0].pausable = VMK_TRUE;
   cos_data->data[1].pausable = VMK_FALSE;
   cos_data->data[0].pri_join_mask = cos_data->data[1].pri_join_mask = 0;

   for (i = 0 ; i < num_of_pri ; i++) {
      pri_tested = 1 << adapter->dcbx_sb.port_params.
         app.traffic_type_priority[i];

      if (pri_tested & DCBX_PFC_PRI_NON_PAUSE_MASK(adapter)) {
         cos_data->data[1].pri_join_mask |= pri_tested;
         entry = 1;
      } else {
         cos_data->data[0].pri_join_mask |= pri_tested;
         entry = 0;
      }
      pg_entry = (u8)pg_pri_orginal_spread[adapter->dcbx_sb.port_params.
         app.traffic_type_priority[i]];
      /* There can be only one strict pg */
      if (pg_entry < DCBX_MAX_NUM_PG_BW_ENTRIES)
         qfle3_dcbx_add_to_cos_bw(adapter, &cos_data->data[entry],
               DCBX_PG_BW_GET(ets->pg_bw_tbl, pg_entry));
      else
         /* If we join a group and one is strict
          * then the bw rules
          */
         cos_data->data[entry].strict =
            QFLE3_DCBX_STRICT_COS_HIGHEST;
   }
   if ((0 == cos_data->data[0].pri_join_mask) &&
         (0 == cos_data->data[1].pri_join_mask))
      QFLE3_ERR("dcbx error: Both groups must have priorities\n");
}

static void
qfle3_dcbx_2cos_limit_cee_two_pg_to_cos_params(
                    struct qfle3_adapter *adapter,
                    struct  pg_help_data	*pg_help_data,
                    struct dcbx_ets_feature	*ets,
                    struct cos_help_data	*cos_data,
                    u32			*pg_pri_orginal_spread,
                    u32				pri_join_mask,
                    u8				num_of_dif_pri)
{
   u8 i = 0;
   u8 pg[DCBX_COS_MAX_NUM_E2] = { 0 };

   /* If there are both pauseable and non-pauseable priorities,
    * the pauseable priorities go to the first queue and
    * the non-pauseable priorities go to the second queue.
    */
   if (IS_DCBX_PFC_PRI_MIX_PAUSE(adapter, pri_join_mask)) {
      if (IS_DCBX_PFC_PRI_MIX_PAUSE(adapter,
               pg_help_data->data[0].pg_priority) ||
            IS_DCBX_PFC_PRI_MIX_PAUSE(adapter,
               pg_help_data->data[1].pg_priority)) {
         /* If one PG contains both pauseable and
          * non-pauseable priorities then ETS is disabled.
          */
         qfle3_dcbx_separate_pauseable_from_non(adapter, cos_data,
               pg_pri_orginal_spread, ets);
         adapter->dcbx_sb.port_params.ets.enabled = VMK_FALSE;
         return;
      }

      /* Pauseable */
      cos_data->data[0].pausable = VMK_TRUE;
      /* Non pauseable. */
      cos_data->data[1].pausable = VMK_FALSE;
      if (IS_DCBX_PFC_PRI_ONLY_PAUSE(adapter,
               pg_help_data->data[0].pg_priority)) {
         /* 0 is pauseable */
         cos_data->data[0].pri_join_mask =
            pg_help_data->data[0].pg_priority;
         pg[0] = pg_help_data->data[0].pg;
         cos_data->data[1].pri_join_mask =
            pg_help_data->data[1].pg_priority;
         pg[1] = pg_help_data->data[1].pg;
      } else {/* 1 is pauseable */
         cos_data->data[0].pri_join_mask =
            pg_help_data->data[1].pg_priority;
         pg[0] = pg_help_data->data[1].pg;
         cos_data->data[1].pri_join_mask =
            pg_help_data->data[0].pg_priority;
         pg[1] = pg_help_data->data[0].pg;
      }
   } else {
      /* If there are only pauseable priorities or
       * only non-pauseable, each PG goes to a queue.
       */
      cos_data->data[0].pausable = cos_data->data[1].pausable =
         IS_DCBX_PFC_PRI_ONLY_PAUSE(adapter, pri_join_mask);
      cos_data->data[0].pri_join_mask =
         pg_help_data->data[0].pg_priority;
      pg[0] = pg_help_data->data[0].pg;
      cos_data->data[1].pri_join_mask =
         pg_help_data->data[1].pg_priority;
      pg[1] = pg_help_data->data[1].pg;
   }

   /* There can be only one strict pg */
   for (i = 0 ; i < ARRAY_SIZE(pg); i++) {
      if (pg[i] < DCBX_MAX_NUM_PG_BW_ENTRIES)
         cos_data->data[i].cos_bw =
            DCBX_PG_BW_GET(ets->pg_bw_tbl, pg[i]);
      else
         cos_data->data[i].strict =
            QFLE3_DCBX_STRICT_COS_HIGHEST;
   }
}

static void
qfle3_dcbx_2cos_limit_cee_three_pg_to_cos_params(
            struct qfle3_adapter *adapter,
            struct pg_help_data	*pg_help_data,
            struct dcbx_ets_feature	*ets,
            struct cos_help_data	*cos_data,
            u32			*pg_pri_orginal_spread,
            u32			pri_join_mask,
            u8			num_of_dif_pri)
{
   u8 i = 0;
   u32 pri_tested = 0;
   u8 entry = 0;
   u8 pg_entry = 0;
   vmk_Bool b_found_strict = VMK_FALSE;
   u8 num_of_pri = LLFC_DRIVER_TRAFFIC_TYPE_MAX;

   cos_data->data[0].pri_join_mask = cos_data->data[1].pri_join_mask = 0;
   /* If there are both pauseable and non-pauseable priorities,
    * the pauseable priorities go to the first queue and the
    * non-pauseable priorities go to the second queue.
    */
   if (IS_DCBX_PFC_PRI_MIX_PAUSE(adapter, pri_join_mask))
      qfle3_dcbx_separate_pauseable_from_non(adapter,
            cos_data, pg_pri_orginal_spread, ets);
   else {
      /* If two BW-limited PG-s were combined to one queue,
       * the BW is their sum.
       *
       * If there are only pauseable priorities or only non-pauseable,
       * and there are both BW-limited and non-BW-limited PG-s,
       * the BW-limited PG/s go to one queue and the non-BW-limited
       * PG/s go to the second queue.
       *
       * If there are only pauseable priorities or only non-pauseable
       * and all are BW limited, then	two priorities go to the first
       * queue and one priority goes to the second queue.
       *
       * We will join this two cases:
       * if one is BW limited it will go to the second queue
       * otherwise the last priority will get it
       */

      cos_data->data[0].pausable = cos_data->data[1].pausable =
         IS_DCBX_PFC_PRI_ONLY_PAUSE(adapter, pri_join_mask);

      for (i = 0 ; i < num_of_pri; i++) {
         pri_tested = 1 << adapter->dcbx_sb.port_params.
            app.traffic_type_priority[i];
         pg_entry = (u8)pg_pri_orginal_spread[adapter->
            dcbx_sb.port_params.app.traffic_type_priority[i]];

         if (pg_entry < DCBX_MAX_NUM_PG_BW_ENTRIES) {
            entry = 0;

            if (i == (num_of_pri-1) &&
                  VMK_FALSE == b_found_strict)
               /* last entry will be handled separately
                * If no priority is strict than last
                * enty goes to last queue.*/
               entry = 1;
            cos_data->data[entry].pri_join_mask |=
               pri_tested;
            qfle3_dcbx_add_to_cos_bw(adapter,
                  &cos_data->data[entry],
                  DCBX_PG_BW_GET(ets->pg_bw_tbl,
                     pg_entry));
         } else {
            b_found_strict = VMK_TRUE;
            cos_data->data[1].pri_join_mask |= pri_tested;
            /* If we join a group and one is strict
             * then the bw rules
             */
            cos_data->data[1].strict =
               QFLE3_DCBX_STRICT_COS_HIGHEST;
         }
      }
   }
}


static void
qfle3_dcbx_2cos_limit_cee_fill_cos_params(struct qfle3_adapter *adapter,
				       struct pg_help_data *help_data,
				       struct dcbx_ets_feature *ets,
				       struct cos_help_data *cos_data,
				       u32 *pg_pri_orginal_spread,
				       u32 pri_join_mask,
				       u8 num_of_dif_pri)
{
   /* default E2 settings */
   cos_data->num_of_cos = DCBX_COS_MAX_NUM_E2;

   switch (help_data->num_of_pg) {
      case 1:
         qfle3_dcbx_2cos_limit_cee_single_pg_to_cos_params(
               adapter,
               help_data,
               cos_data,
               pri_join_mask,
               num_of_dif_pri);
         break;
      case 2:
         qfle3_dcbx_2cos_limit_cee_two_pg_to_cos_params(
               adapter,
               help_data,
               ets,
               cos_data,
               pg_pri_orginal_spread,
               pri_join_mask,
               num_of_dif_pri);
         break;

      case 3:
         qfle3_dcbx_2cos_limit_cee_three_pg_to_cos_params(
               adapter,
               help_data,
               ets,
               cos_data,
               pg_pri_orginal_spread,
               pri_join_mask,
               num_of_dif_pri);
         break;
      default:
         QFLE3_ERR("Wrong pg_help_data.num_of_pg\n");
         qfle3_dcbx_ets_disabled_entry_data(adapter,
               cos_data, pri_join_mask);
   }
}

static void
qfle3_dcbx_fill_cos_params(struct qfle3_adapter *adapter,
                    struct pg_help_data *help_data,
                    struct dcbx_ets_feature *ets,
                    u32 *pg_pri_orginal_spread)
{
   struct cos_help_data         cos_data;
   u8                    i                           = 0;
   u32                   pri_join_mask               = 0;
   u8                    num_of_dif_pri              = 0;

   vmk_Memset(&cos_data, 0, sizeof(cos_data));

   /* Validate the pg value */
   for (i = 0; i < help_data->num_of_pg ; i++) {
      if (DCBX_STRICT_PRIORITY != help_data->data[i].pg &&
            DCBX_MAX_NUM_PG_BW_ENTRIES <= help_data->data[i].pg)
         QFLE3_ERR("Invalid pg[%d] data %x\n", i,
               help_data->data[i].pg);
      pri_join_mask   |=  help_data->data[i].pg_priority;
      num_of_dif_pri  += help_data->data[i].num_of_dif_pri;
   }

   /* defaults */
   cos_data.num_of_cos = 1;
   for (i = 0; i < ARRAY_SIZE(cos_data.data); i++) {
      cos_data.data[i].pri_join_mask = 0;
      cos_data.data[i].pausable = VMK_FALSE;
      cos_data.data[i].strict = QFLE3_DCBX_STRICT_INVALID;
      cos_data.data[i].cos_bw = DCBX_INVALID_COS_BW;
   }

   if (CHIP_IS_E3B0(adapter))
      qfle3_dcbx_cee_fill_cos_params(adapter, help_data, ets,
            &cos_data, pri_join_mask);
   else /* E2 + E3A0 */
      qfle3_dcbx_2cos_limit_cee_fill_cos_params(adapter,
            help_data, ets,
            &cos_data,
            pg_pri_orginal_spread,
            pri_join_mask,
            num_of_dif_pri);

   for (i = 0; i < cos_data.num_of_cos ; i++) {
      struct qfle3_dcbx_cos_params *p =
         &adapter->dcbx_sb.port_params.ets.cos_params[i];

      p->strict = cos_data.data[i].strict;
      p->bw_tbl = cos_data.data[i].cos_bw;
      p->pri_bitmask = cos_data.data[i].pri_join_mask;
      p->pauseable = cos_data.data[i].pausable;

      /* sanity */
      if (p->bw_tbl != DCBX_INVALID_COS_BW ||
            p->strict != QFLE3_DCBX_STRICT_INVALID) {
         if (p->pri_bitmask == 0)
            QFLE3_ERR("Invalid pri_bitmask for %d\n", i);

         if (CHIP_IS_E2(adapter) || CHIP_IS_E3A0(adapter)) {

            if (p->pauseable &&
                  DCBX_PFC_PRI_GET_NON_PAUSE(adapter,
                     p->pri_bitmask) != 0)
               QFLE3_ERR("Inconsistent config for pausable COS %d\n",
                     i);

            if (!p->pauseable &&
                  DCBX_PFC_PRI_GET_PAUSE(adapter,
                     p->pri_bitmask) != 0)
               QFLE3_ERR("Inconsistent config for nonpausable COS %d\n",
                     i);
         }
      }

      if (p->pauseable)
         QFLE3_DBG(QFLE3_DBG_DCB, "COS %d PAUSABLE prijoinmask 0x%x\n",
               i, cos_data.data[i].pri_join_mask);
      else
         QFLE3_DBG(QFLE3_DBG_DCB,
               "COS %d NONPAUSABLE prijoinmask 0x%x\n",
               i, cos_data.data[i].pri_join_mask);
   }

   adapter->dcbx_sb.port_params.ets.num_of_cos = cos_data.num_of_cos ;
}


static void
qfle3_dcbx_get_ets_feature(struct qfle3_adapter *adapter)
{
   int i = 0;
   struct dcbx_ets_feature *ets = &adapter->dcbx_sb.local_feat.ets;
   u32 error = adapter->dcbx_sb.dcbx_error;
   u32 pg_pri_orginal_spread[DCBX_MAX_NUM_PG_BW_ENTRIES] = {0};
   struct pg_help_data pg_help_data;
   struct qfle3_dcbx_cos_params *cos_params =
      adapter->dcbx_sb.port_params.ets.cos_params;
   struct qfle3_dcbx_port_params *port_params =
      &adapter->dcbx_sb.port_params;

   vmk_Memset(&pg_help_data, 0, sizeof(struct pg_help_data));

   if (GET_FLAGS(error, DCBX_LOCAL_ETS_ERROR))
      QFLE3_DBG(QFLE3_DBG_DCB, "DCBX_LOCAL_ETS_ERROR\n");

   if (GET_FLAGS(error, DCBX_REMOTE_ETS_TLV_NOT_FOUND))
      QFLE3_DBG(QFLE3_DBG_DCB, "DCBX_REMOTE_ETS_TLV_NOT_FOUND\n");

   /* Clean up old settings of ets on COS */
   for (i = 0; i < ARRAY_SIZE(port_params->ets.cos_params) ; i++) {
      cos_params[i].pauseable = 0;
      cos_params[i].strict = QFLE3_DCBX_STRICT_INVALID;
      cos_params[i].bw_tbl = DCBX_INVALID_COS_BW;
      cos_params[i].pri_bitmask = 0;
   }

   if (port_params->app.enabled && ets->enabled &&
         !GET_FLAGS(error,
            DCBX_LOCAL_ETS_ERROR | DCBX_REMOTE_ETS_TLV_NOT_FOUND)) {
      QFLE3_DBG(QFLE3_DBG_DCB, "DCBX_LOCAL_ETS_ENABLE\n");
      port_params->ets.enabled = 1;

      qfle3_dcbx_get_ets_pri_pg_tbl(adapter,
            pg_pri_orginal_spread,
            ets->pri_pg_tbl);

      qfle3_dcbx_get_num_pg_traf_type(adapter,
            pg_pri_orginal_spread,
            &pg_help_data);

      qfle3_dcbx_fill_cos_params(adapter, &pg_help_data,
            ets, pg_pri_orginal_spread);

   } else {
      QFLE3_DBG(QFLE3_DBG_DCB, "DCBX_LOCAL_ETS_DISABLED\n");
      port_params->ets.enabled = 0;
      ets->pri_pg_tbl[0] = 0;

      for (i = 0; i < DCBX_MAX_NUM_PRI_PG_ENTRIES ; i++)
         DCBX_PG_BW_SET(ets->pg_bw_tbl, i, 1);
   }
}

/* maps unmapped priorities to to the same COS as L2 */
static void
qfle3_dcbx_map_nw(struct qfle3_adapter *adapter)
{
   int i;
   u32 unmapped = (1 << MAX_PFC_PRIORITIES) - 1; /* all ones */
   u32 *ttp = adapter->dcbx_sb.port_params.app.traffic_type_priority;
   u32 nw_prio = 1 << ttp[LLFC_TRAFFIC_TYPE_NW];
   struct qfle3_dcbx_cos_params *cos_params =
      adapter->dcbx_sb.port_params.ets.cos_params;

   /* get unmapped priorities by clearing mapped bits */
   for (i = 0; i < LLFC_DRIVER_TRAFFIC_TYPE_MAX; i++)
      unmapped &= ~(1 << ttp[i]);

   /* find cos for nw prio and extend it with unmapped */
   for (i = 0; i < ARRAY_SIZE(adapter->dcbx_sb.port_params.ets.cos_params); i++) {
      if (cos_params[i].pri_bitmask & nw_prio) {
         /* extend the bitmask with unmapped */
         QFLE3_DBG(QFLE3_DBG_DCB,
               "cos %d extended with 0x%08x\n", i, unmapped);
         cos_params[i].pri_bitmask |= unmapped;
         break;
      }
   }
}

static void
qfle3_dcbx_get_ap_priority(struct qfle3_adapter *adapter,
      u8 pri_bitmap,
      u8 llfc_traf_type)
{
   u32 pri = MAX_PFC_PRIORITIES;
   u32 index = MAX_PFC_PRIORITIES - 1;
   u32 pri_mask;
   u32 *ttp = adapter->dcbx_sb.port_params.app.traffic_type_priority;

   /* Choose the highest priority */
   while ((MAX_PFC_PRIORITIES == pri) && (0 != index)) {
      pri_mask = 1 << index;
      if (GET_FLAGS(pri_bitmap, pri_mask))
         pri = index ;
      index--;
   }

   if (pri < MAX_PFC_PRIORITIES)
      ttp[llfc_traf_type] = MAX_OF(ttp[llfc_traf_type], pri);
}

static void
qfle3_dcbx_get_ap_feature(struct qfle3_adapter *adapter) {
   u8 index;
   struct dcbx_app_priority_feature *app = &adapter->dcbx_sb.local_feat.app;
   u32 error = adapter->dcbx_sb.dcbx_error;
   u32 *ttp = adapter->dcbx_sb.port_params.app.traffic_type_priority;
   u8 iscsi_pri_found = 0, fcoe_pri_found = 0;

   if (GET_FLAGS(error, DCBX_LOCAL_APP_ERROR))
      QFLE3_DBG(QFLE3_DBG_DCB, "DCBX_LOCAL_APP_ERROR\n");

   if (GET_FLAGS(error, DCBX_LOCAL_APP_MISMATCH))
      QFLE3_DBG(QFLE3_DBG_DCB, "DCBX_LOCAL_APP_MISMATCH\n");

   if (GET_FLAGS(error, DCBX_REMOTE_APP_TLV_NOT_FOUND))
      QFLE3_DBG(QFLE3_DBG_DCB, "DCBX_REMOTE_APP_TLV_NOT_FOUND\n");

   if (app->enabled &&
         !GET_FLAGS(error, DCBX_LOCAL_APP_ERROR | DCBX_LOCAL_APP_MISMATCH |
            DCBX_REMOTE_APP_TLV_NOT_FOUND)) {

      adapter->dcbx_sb.port_params.app.enabled = 1;

      /* Use 0 as the default application priority for all. */
      for (index = 0 ; index < LLFC_DRIVER_TRAFFIC_TYPE_MAX; index++)
         ttp[index] = 0;

      for (index = 0 ; index < DCBX_MAX_APP_PROTOCOL; index++) {
         struct dcbx_app_priority_entry *entry =
            app->app_pri_tbl;

         if (GET_FLAGS(entry[index].appBitfield,
                  DCBX_APP_SF_DEFAULT) &&
               GET_FLAGS(entry[index].appBitfield,
                  DCBX_APP_SF_ETH_TYPE)) {
            qfle3_dcbx_get_ap_priority(adapter,
                  entry[index].pri_bitmap,
                  LLFC_TRAFFIC_TYPE_NW);

         } else if (GET_FLAGS(entry[index].appBitfield,
                  DCBX_APP_SF_PORT) &&
               TCP_PORT_ISCSI == entry[index].app_id) {
            qfle3_dcbx_get_ap_priority(adapter,
                  entry[index].pri_bitmap,
                  LLFC_TRAFFIC_TYPE_ISCSI);
            iscsi_pri_found = 1;

         } else if (GET_FLAGS(entry[index].appBitfield,
                  DCBX_APP_SF_ETH_TYPE) &&
               VMK_ETH_TYPE_FCOE == entry[index].app_id) {
            qfle3_dcbx_get_ap_priority(adapter,
                  entry[index].pri_bitmap,
                  LLFC_TRAFFIC_TYPE_FCOE);
            fcoe_pri_found = 1;
         }
      }

      /* If we have received a non-zero default application
       * priority, then use that for applications which are
       * not configured with any priority.
       */
      if (ttp[LLFC_TRAFFIC_TYPE_NW] != 0) {
         if (!iscsi_pri_found) {
            ttp[LLFC_TRAFFIC_TYPE_ISCSI] =
               ttp[LLFC_TRAFFIC_TYPE_NW];
            QFLE3_DBG(QFLE3_DBG_DCB,
                  "ISCSI is using default priority.\n");
         }
         if (!fcoe_pri_found) {
            ttp[LLFC_TRAFFIC_TYPE_FCOE] =
               ttp[LLFC_TRAFFIC_TYPE_NW];
            QFLE3_DBG(QFLE3_DBG_DCB,
                  "FCoE is using default priority.\n");
         }
      }
   } else {
      QFLE3_DBG(QFLE3_DBG_DCB, "DCBX_LOCAL_APP_DISABLED\n");
      adapter->dcbx_sb.port_params.app.enabled = 0;
      for (index = 0 ; index < LLFC_DRIVER_TRAFFIC_TYPE_MAX; index++)
         ttp[index] = INVALID_TRAFFIC_TYPE_PRIORITY;
   }
}

static inline void qfle3_dcbx_update_tc_mapping(struct qfle3_adapter *adapter)
{
   u8 prio, cos;
   for (cos = 0; cos < adapter->dcbx_sb.port_params.ets.num_of_cos; cos++) {
      for (prio = 0; prio < QFLE3_MAX_PRIORITY; prio++) {
         if (adapter->dcbx_sb.port_params.ets.cos_params[cos].pri_bitmask
               & (1 << prio)) {
            adapter->dcbx_sb.prio_to_cos[prio] = cos;
            QFLE3_DBG(QFLE3_DBG_DCB,
                  "tx_mapping %d --> %d\n", prio, cos);
         }
      }
   }

   /* setup tc must be called under rtnl lock, but we can't take it here
    * as we are handling an attention on a work queue which must be
    * flushed at some rtnl-locked contexts (e.g. if down)
    */
   /*TODO: setup TC mappings for upper layer. */
}

   static void
qfle3_get_dcbx_drv_param(struct qfle3_adapter *adapter)
{
   qfle3_dcbx_get_ap_feature(adapter);

   qfle3_dcbx_get_pfc_feature(adapter);

   qfle3_dcbx_get_ets_feature(adapter);

   qfle3_dcbx_map_nw(adapter);
}

int qfle3_dcbx_stop_hw_tx(struct qfle3_adapter *adapter)
{
   struct ecore_func_state_params func_params = {NULL};
   int rc;

   func_params.f_obj = &adapter->func_obj;
   func_params.cmd = ECORE_F_CMD_TX_STOP;

   ECORE_SET_BIT(RAMROD_COMP_WAIT, &func_params.ramrod_flags);
   ECORE_SET_BIT(RAMROD_RETRY, &func_params.ramrod_flags);

   QFLE3_DBG(QFLE3_DBG_DCB, "DCBX TX STOP\n");

   rc = ecore_func_state_change(adapter, &func_params);
   if (rc) {
      QFLE3_ERR("Unable to hold traffic for HW configuration\n");
   }

   return rc;
}

   static void
qfle3_dcbx_print_cos_params(struct qfle3_adapter *adapter,
      struct ecore_func_tx_start_params *pfc_fw_cfg)
{
   u8 pri = 0;
   u8 cos = 0;

   QFLE3_DBG(QFLE3_DBG_DCB,
         "pfc_fw_cfg->dcb_version %x\n", pfc_fw_cfg->dcb_version);
   QFLE3_DBG(QFLE3_DBG_DCB,
         "pdev->params.dcbx_port_params.pfc.priority_non_pauseable_mask %x\n",
         adapter->dcbx_sb.port_params.pfc.priority_non_pauseable_mask);

   for (cos = 0 ; cos < adapter->dcbx_sb.port_params.ets.num_of_cos ; cos++) {
      QFLE3_DBG(QFLE3_DBG_DCB,
            "pdev->params.dcbx_port_params.ets.cos_params[%d].pri_bitmask %x\n",
            cos, adapter->dcbx_sb.port_params.ets.cos_params[cos].pri_bitmask);

      QFLE3_DBG(QFLE3_DBG_DCB,
            "pdev->params.dcbx_port_params.ets.cos_params[%d].bw_tbl %x\n",
            cos, adapter->dcbx_sb.port_params.ets.cos_params[cos].bw_tbl);

      QFLE3_DBG(QFLE3_DBG_DCB,
            "pdev->params.dcbx_port_params.ets.cos_params[%d].strict %x\n",
            cos, adapter->dcbx_sb.port_params.ets.cos_params[cos].strict);

      QFLE3_DBG(QFLE3_DBG_DCB,
            "pdev->params.dcbx_port_params.ets.cos_params[%d].pauseable %x\n",
            cos, adapter->dcbx_sb.port_params.ets.cos_params[cos].pauseable);
   }

   for (pri = 0; pri < LLFC_DRIVER_TRAFFIC_TYPE_MAX; pri++) {
      QFLE3_DBG(QFLE3_DBG_DCB,
            "pfc_fw_cfg->traffic_type_to_priority_cos[%d].priority %x\n",
            pri, pfc_fw_cfg->traffic_type_to_priority_cos[pri].priority);

      QFLE3_DBG(QFLE3_DBG_DCB,
            "pfc_fw_cfg->traffic_type_to_priority_cos[%d].cos %x\n",
            pri, pfc_fw_cfg->traffic_type_to_priority_cos[pri].cos);
   }
}


static void qfle3_dcbx_init_tx_params(struct qfle3_adapter *adapter,
				 struct ecore_func_tx_start_params *tx_start)
{
   u16 pri_bit = 0;
   u8 cos = 0, pri = 0;
   struct priority_cos *tt2cos;
   u32 *ttp = adapter->dcbx_sb.port_params.app.traffic_type_priority;
   int mfw_configured = SHMEM2_HAS(adapter, drv_flags) &&
      GET_FLAGS(SHMEM2_RD(adapter, drv_flags),
            1 << DRV_FLAGS_DCB_MFW_CONFIGURED);

   vmk_Memset(tx_start, 0, sizeof(*tx_start));

   /* to disable DCB - the structure must be zeroed */
   if ((adapter->dcbx_sb.dcbx_error & DCBX_REMOTE_MIB_ERROR) && !mfw_configured)
      return;

   /*shortcut*/
   tt2cos = tx_start->traffic_type_to_priority_cos;

   /* Fw version should be incremented each update */
   tx_start->dcb_version = ++adapter->dcbx_sb.dcb_version;
   tx_start->dcb_enabled = 1;

   /* Fill priority parameters */
   for (pri = 0; pri < LLFC_DRIVER_TRAFFIC_TYPE_MAX; pri++) {
      tt2cos[pri].priority = ttp[pri];
      pri_bit = 1 << tt2cos[pri].priority;

      /* Fill COS parameters based on COS calculated to
       * make it more general for future use */
      for (cos = 0; cos < adapter->dcbx_sb.port_params.ets.num_of_cos; cos++)
         if (adapter->dcbx_sb.port_params.ets.cos_params[cos].
               pri_bitmask & pri_bit)
            tt2cos[pri].cos = cos;

      /* TODO - this retains the current logic; But who exactly
       * requested anything different?
       */
      tx_start->dcb_outer_pri[pri]  = ttp[pri];
   }

   /* we never want the FW to add a 0 vlan tag */
   tx_start->dont_add_pri_0_en = 1;

   qfle3_dcbx_print_cos_params(adapter, tx_start);
}


int qfle3_dcbx_resume_hw_tx(struct qfle3_adapter *adapter)
{
   struct ecore_func_state_params func_params = {NULL};
   struct ecore_func_tx_start_params *tx_params =
      &func_params.params.tx_start;
   int rc;

   func_params.f_obj = &adapter->func_obj;
   func_params.cmd = ECORE_F_CMD_TX_START;

   ECORE_SET_BIT(RAMROD_COMP_WAIT, &func_params.ramrod_flags);
   ECORE_SET_BIT(RAMROD_RETRY, &func_params.ramrod_flags);

   qfle3_dcbx_init_tx_params(adapter, tx_params);

   QFLE3_DBG(QFLE3_DBG_DCB, "DCBX START TX\n");

   rc = ecore_func_state_change(adapter, &func_params);
   if (rc) {
      QFLE3_ERR("Unable to resume traffic after HW configuration\n");
   }

   return rc;
}

void qfle3_dcbx_restart_tx(void *adap)
{
   struct qfle3_adapter *adapter = (struct qfle3_adapter *)adap;

   qfle3_dcbx_stop_hw_tx(adapter);
   qfle3_dcbx_resume_hw_tx(adapter);
}

static void qfle3_dcbx_pfc_set(struct qfle3_adapter *adapter)
{
   struct elink_nig_brb_pfc_port_params pfc_params = {0};
   u32 pri_bit, val = 0;
   int i;

   pfc_params.num_of_rx_cos_priority_mask =
      adapter->dcbx_sb.port_params.ets.num_of_cos;

   /* Tx COS configuration */
   for (i = 0; i < adapter->dcbx_sb.port_params.ets.num_of_cos; i++)
      /*
       * We configure only the pauseable bits (non pauseable aren't
       * configured at all) it's done to avoid false pauses from
       * network
       */
      pfc_params.rx_cos_priority_mask[i] =
         adapter->dcbx_sb.port_params.ets.cos_params[i].pri_bitmask
         & DCBX_PFC_PRI_PAUSE_MASK(adapter);

   /*
    * Rx COS configuration
    * Changing PFC RX configuration .
    * In RX COS0 will always be configured to lossless and COS1 to lossy
    */
   for (i = 0 ; i < MAX_PFC_PRIORITIES ; i++) {
      pri_bit = 1 << i;

      if (!(pri_bit & DCBX_PFC_PRI_PAUSE_MASK(adapter)))
         val |= 1 << (i * 4);
   }

   pfc_params.pkt_priority_to_cos = val;

   /* RX COS0 */
   pfc_params.llfc_low_priority_classes = DCBX_PFC_PRI_PAUSE_MASK(adapter);
   /* RX COS1 */
   
   pfc_params.llfc_high_priority_classes = 0;
   QFLE3_DBG(QFLE3_DBG_DCB, "DCBX PFC SET:\n");
   QFLE3_DBG(QFLE3_DBG_DCB, "pause_enable %d, llfc_out_en %d, llfc_enable %d, "
      "pkt_priority_to_cos 0x%x, num_of_rx_cos_priority_mask %x",
      pfc_params.pause_enable,pfc_params.llfc_out_en,pfc_params.llfc_enable,pfc_params.pkt_priority_to_cos,pfc_params.num_of_rx_cos_priority_mask);
   QFLE3_DBG(QFLE3_DBG_DCB, "rx cos[0] %u, [1] %u [2] %u [3] %u [4] %u [5] %u ", 
      pfc_params.rx_cos_priority_mask[0],pfc_params.rx_cos_priority_mask[1],pfc_params.rx_cos_priority_mask[2],
      pfc_params.rx_cos_priority_mask[3],pfc_params.rx_cos_priority_mask[4],pfc_params.rx_cos_priority_mask[5]);
   QFLE3_DBG(QFLE3_DBG_DCB, " llfc high priority classes 0x%x, low priority classes 0x%x\n", 
      pfc_params.llfc_high_priority_classes,pfc_params.llfc_low_priority_classes);

   qfle3_acquire_phy_lock(adapter);
   adapter->link_params.feature_config_flags |=
                                       ELINK_FEATURE_CONFIG_PFC_ENABLED;
   elink_update_pfc(&adapter->link_params, &adapter->link_vars, &pfc_params);
   qfle3_release_phy_lock(adapter);
}

static void
 qfle3_dcbx_pfc_clear(struct qfle3_adapter *adapter)
{
   struct elink_nig_brb_pfc_port_params nig_params = {0};
   nig_params.pause_enable = 1;
   
   QFLE3_DBG(QFLE3_DBG_DCB, "DCBX PFC clear:\n");
   qfle3_acquire_phy_lock(adapter);
   adapter->link_params.feature_config_flags &=
                                        ~ELINK_FEATURE_CONFIG_PFC_ENABLED;
   elink_update_pfc(&adapter->link_params, &adapter->link_vars, &nig_params);
   qfle3_release_phy_lock(adapter);
}

static void
qfle3_dcbx_pfc_set_pfc(struct qfle3_adapter *adapter)
{
   int mfw_configured = SHMEM2_HAS(adapter, drv_flags) &&
      GET_FLAGS(SHMEM2_RD(adapter, drv_flags),
            1 << DRV_FLAGS_DCB_MFW_CONFIGURED);

   if (adapter->dcbx_sb.port_params.pfc.enabled &&
     (!(adapter->dcbx_sb.dcbx_error & DCBX_REMOTE_MIB_ERROR) || mfw_configured))
      /*
       * 1. Fills up common PFC structures if required
       * 2. Configure NIG, MAC and BRB via the elink
       */
      qfle3_dcbx_pfc_set(adapter);
   else
      qfle3_dcbx_pfc_clear(adapter);
}

static void
qfle3_dcbx_update_ets_config(struct qfle3_adapter *adapter)
{
   struct qfle3_dcbx_pg_params *ets = &(adapter->dcbx_sb.port_params.ets);
   struct elink_ets_params ets_params = { 0 };
   u8 i;

   ets_params.num_of_cos = ets->num_of_cos;

   for (i = 0; i < ets->num_of_cos; i++) {
      /* COS is SP */
      if (ets->cos_params[i].strict != QFLE3_DCBX_STRICT_INVALID) {
         if (ets->cos_params[i].bw_tbl != DCBX_INVALID_COS_BW) {
            QFLE3_ERR("COS can't be not BW and not SP\n");
            return;
         }

         ets_params.cos[i].state = elink_cos_state_strict;
         ets_params.cos[i].params.sp_params.pri =
            ets->cos_params[i].strict;
      } else { /* COS is BW */
         if (ets->cos_params[i].bw_tbl == DCBX_INVALID_COS_BW) {
            QFLE3_ERR("COS can't be not BW and not SP\n");
            return;
         }
         ets_params.cos[i].state = elink_cos_state_bw;
         ets_params.cos[i].params.bw_params.bw =
            (u8)ets->cos_params[i].bw_tbl;
      }
   }

   /* Configure the ETS in HW */
   if (elink_ets_e3b0_config(&adapter->link_params, &adapter->link_vars,
            &ets_params)) {
      QFLE3_ERR("elink_ets_e3b0_config failed\n");
      elink_ets_disabled(&adapter->link_params, &adapter->link_vars);
   }
}

static void
qfle3_dcbx_2cos_limit_update_ets_config(struct qfle3_adapter *adapter)
{
   struct qfle3_dcbx_pg_params *ets = &(adapter->dcbx_sb.port_params.ets);
   int rc = 0;

   if (ets->num_of_cos == 0 || ets->num_of_cos > DCBX_COS_MAX_NUM_E2) {
      QFLE3_ERR("Illegal number of COSes %d\n", ets->num_of_cos);
      return;
   }

   /* valid COS entries */
   if (ets->num_of_cos == 1)   /* no ETS */
      return;

   /* sanity */
   if (((QFLE3_DCBX_STRICT_INVALID == ets->cos_params[0].strict) &&
            (DCBX_INVALID_COS_BW == ets->cos_params[0].bw_tbl)) ||
         ((QFLE3_DCBX_STRICT_INVALID == ets->cos_params[1].strict) &&
          (DCBX_INVALID_COS_BW == ets->cos_params[1].bw_tbl))) {
      QFLE3_ERR("all COS should have at least bw_limit or strict"
            "ets->cos_params[0].strict= %x"
            "ets->cos_params[0].bw_tbl= %x"
            "ets->cos_params[1].strict= %x"
            "ets->cos_params[1].bw_tbl= %x",
            ets->cos_params[0].strict,
            ets->cos_params[0].bw_tbl,
            ets->cos_params[1].strict,
            ets->cos_params[1].bw_tbl);
      return;
   }
   /* If we join a group and there is bw_tbl and strict then bw rules */
   if ((DCBX_INVALID_COS_BW != ets->cos_params[0].bw_tbl) &&
         (DCBX_INVALID_COS_BW != ets->cos_params[1].bw_tbl)) {
      u32 bw_tbl_0 = ets->cos_params[0].bw_tbl;
      u32 bw_tbl_1 = ets->cos_params[1].bw_tbl;
      /* Do not allow 0-100 configuration
       * since PBF does not support it
       * force 1-99 instead
       */
      if (bw_tbl_0 == 0) {
         bw_tbl_0 = 1;
         bw_tbl_1 = 99;
      } else if (bw_tbl_1 == 0) {
         bw_tbl_1 = 1;
         bw_tbl_0 = 99;
      }

      elink_ets_bw_limit(&adapter->link_params, bw_tbl_0, bw_tbl_1);
   } else {
      if (ets->cos_params[0].strict == QFLE3_DCBX_STRICT_COS_HIGHEST)
         rc = elink_ets_strict(&adapter->link_params, 0);
      else if (ets->cos_params[1].strict
            == QFLE3_DCBX_STRICT_COS_HIGHEST)
         rc = elink_ets_strict(&adapter->link_params, 1);
      if (rc)
         QFLE3_ERR("update_ets_params failed\n");
   }
}


static void
qfle3_dcbx_update_ets_params(struct qfle3_adapter *adapter)
{
   int mfw_configured = SHMEM2_HAS(adapter, drv_flags) &&
      GET_FLAGS(SHMEM2_RD(adapter, drv_flags),
            1 << DRV_FLAGS_DCB_MFW_CONFIGURED);

   elink_ets_disabled(&adapter->link_params, &adapter->link_vars);

   if (!adapter->dcbx_sb.port_params.ets.enabled ||
     ((adapter->dcbx_sb.dcbx_error & DCBX_REMOTE_MIB_ERROR) && !mfw_configured))
      return;

   if (CHIP_IS_E3B0(adapter))
      qfle3_dcbx_update_ets_config(adapter);
   else
      qfle3_dcbx_2cos_limit_update_ets_config(adapter);
}

static void
qfle3_dump_dcbx_drv_param(struct qfle3_adapter *adapter)
{
   u8 i = 0;
   struct dcbx_features *features = &adapter->dcbx_sb.local_feat;

   QFLE3_DBG(QFLE3_DBG_DCB, "local_mib.error %x\n", adapter->dcbx_sb.dcbx_error);

   /* PG */
   QFLE3_DBG(QFLE3_DBG_DCB,
         "local_mib.features.ets.enabled %x\n", features->ets.enabled);

   for (i = 0; i < DCBX_MAX_NUM_PG_BW_ENTRIES; i++)
      QFLE3_DBG(QFLE3_DBG_DCB,
            "local_mib.features.ets.pg_bw_tbl[%d] %d\n", i,
            DCBX_PG_BW_GET(features->ets.pg_bw_tbl, i));

   for (i = 0; i < DCBX_MAX_NUM_PRI_PG_ENTRIES; i++)
      QFLE3_DBG(QFLE3_DBG_DCB,
            "local_mib.features.ets.pri_pg_tbl[%d] %d\n", i,
            DCBX_PRI_PG_GET(features->ets.pri_pg_tbl, i));

   /* pfc */
   QFLE3_DBG(QFLE3_DBG_DCB, "dcbx_features.pfc.pri_en_bitmap %x\n",
         features->pfc.pri_en_bitmap);
   QFLE3_DBG(QFLE3_DBG_DCB, "dcbx_features.pfc.pfc_caps %x\n",
         features->pfc.pfc_caps);
   QFLE3_DBG(QFLE3_DBG_DCB, "dcbx_features.pfc.enabled %x\n",
         features->pfc.enabled);
   QFLE3_DBG(QFLE3_DBG_DCB, "dcbx_features.app.tc_supported %x\n",
         features->app.tc_supported);
   QFLE3_DBG(QFLE3_DBG_DCB, "dcbx_features.app.enabled %x\n",
         features->app.enabled);

   for (i = 0; i < DCBX_MAX_APP_PROTOCOL; i++) {
      QFLE3_DBG(QFLE3_DBG_DCB,
            "dcbx_features.app.app_pri_tbl[%x].app_id %x\n",
            i, features->app.app_pri_tbl[i].app_id);

      QFLE3_DBG(QFLE3_DBG_DCB,
            "dcbx_features.app.app_pri_tbl[%x].pri_bitmap %x\n",
            i, features->app.app_pri_tbl[i].pri_bitmap);

      QFLE3_DBG(QFLE3_DBG_DCB,
            "dcbx_features.app.app_pri_tbl[%x].appBitfield %x\n",
            i, features->app.app_pri_tbl[i].appBitfield);
   }
}

int qfle3_dcbx_read_mib(struct qfle3_adapter *adapter,
      u32 *base_mib_addr,
      u32 offset,
      int read_mib_type)
{
   int max_try_read = 0;
   u32 mib_size, prefix_seq_num, suffix_seq_num;
   struct lldp_remote_mib *remote_mib ;
   struct lldp_local_mib  *local_mib;

   switch (read_mib_type) {
      case DCBX_READ_LOCAL_MIB:
         mib_size = sizeof(struct lldp_local_mib);
         break;
      case DCBX_READ_REMOTE_MIB:
         mib_size = sizeof(struct lldp_remote_mib);
         break;
      default:
         return 1;
   }

   offset += QFLE3_PORT(adapter) * mib_size;

   do {
      qfle3_read_data(adapter, base_mib_addr, offset, mib_size);

      max_try_read++;

      switch (read_mib_type) {
         case DCBX_READ_LOCAL_MIB:
            local_mib = (struct lldp_local_mib *) base_mib_addr;
            prefix_seq_num = local_mib->prefix_seq_num;
            suffix_seq_num = local_mib->suffix_seq_num;
            break;
         case DCBX_READ_REMOTE_MIB:
            remote_mib = (struct lldp_remote_mib *) base_mib_addr;
            prefix_seq_num = remote_mib->prefix_seq_num;
            suffix_seq_num = remote_mib->suffix_seq_num;
            break;
         default:
            return 1;
      }
   } while ((prefix_seq_num != suffix_seq_num) &&
         (max_try_read < DCBX_LOCAL_MIB_MAX_TRY_READ));

   if (max_try_read >= DCBX_LOCAL_MIB_MAX_TRY_READ) {
      QFLE3_ERR("MIB could not be read\n");
      return 1;
   }

   return 0;
}

static int
qfle3_dcbx_read_shmem_neg_results(struct qfle3_adapter *adapter)
{
   struct lldp_local_mib local_mib = {0};
   u32 dcbx_neg_res_offset = SHMEM2_RD(adapter, dcbx_neg_res_offset);
   int rc;

   QFLE3_DBG(QFLE3_DBG_DCB, "dcbx_neg_res_offset 0x%x\n", dcbx_neg_res_offset);

   if (SHMEM_DCBX_NEG_RES_NONE == dcbx_neg_res_offset) {
      QFLE3_ERR("FW doesn't support dcbx_neg_res_offset\n");
      return VMK_FAILURE;
   }

   rc = qfle3_dcbx_read_mib(adapter, (u32 *)&local_mib, dcbx_neg_res_offset,
         DCBX_READ_LOCAL_MIB);

   if (rc) {
      QFLE3_ERR("Failed to read local mib from FW\n");
      return rc;
   }

   /* save features and error */
   adapter->dcbx_sb.local_feat = local_mib.features;
   adapter->dcbx_sb.dcbx_error = local_mib.error;
   return 0;
}
static int
qfle3_dump_dcbx_remote_param(struct qfle3_adapter *adapter)
{
   struct lldp_remote_mib remote_mib = {0};
   u32 dcbx_remote_offset = SHMEM2_RD(adapter, dcbx_remote_mib_offset);
   int rc;
   u8 i = 0;
   struct dcbx_features *features;
   
   QFLE3_DBG(QFLE3_DBG_DCB, "dcbx_remote_mib_offset 0x%x\n", dcbx_remote_offset);

   if (SHMEM_DCBX_REMOTE_MIB_NONE == dcbx_remote_offset) {
      QFLE3_ERR("Shmem does not contain remove mib offset\n");
      return VMK_FAILURE;
   }

   rc = qfle3_dcbx_read_mib(adapter, (u32 *)&remote_mib, dcbx_remote_offset,
         DCBX_READ_REMOTE_MIB);

   if (rc) {
      QFLE3_ERR("Failed to read remote mib from FW\n");
      return rc;
   }
   features = &remote_mib.features;

   QFLE3_DBG(QFLE3_DBG_DCB, "remote_mib.flags %x\n", remote_mib.flags);

   /* PG */
   QFLE3_DBG(QFLE3_DBG_DCB,
         "remote_mib.features.ets.enabled %x\n", features->ets.enabled);

   for (i = 0; i < DCBX_MAX_NUM_PG_BW_ENTRIES; i++)
      QFLE3_DBG(QFLE3_DBG_DCB,
            "remote_mib.features.ets.pg_bw_tbl[%d] %d\n", i,
            DCBX_PG_BW_GET(features->ets.pg_bw_tbl, i));

   for (i = 0; i < DCBX_MAX_NUM_PRI_PG_ENTRIES; i++)
      QFLE3_DBG(QFLE3_DBG_DCB,
            "remote_mib.features.ets.pri_pg_tbl[%d] %d\n", i,
            DCBX_PRI_PG_GET(features->ets.pri_pg_tbl, i));

   /* pfc */
   QFLE3_DBG(QFLE3_DBG_DCB, "remote_mib.pfc.pri_en_bitmap %x\n",
         features->pfc.pri_en_bitmap);
   QFLE3_DBG(QFLE3_DBG_DCB, "remote_mib.pfc.pfc_caps %x\n",
         features->pfc.pfc_caps);
   QFLE3_DBG(QFLE3_DBG_DCB, "remote_mib.pfc.enabled %x\n",
         features->pfc.enabled);
   QFLE3_DBG(QFLE3_DBG_DCB, "remote_mib.app.tc_supported %x\n",
         features->app.tc_supported);
   QFLE3_DBG(QFLE3_DBG_DCB, "remote_mib.app.enabled %x\n",
         features->app.enabled);

   for (i = 0; i < DCBX_MAX_APP_PROTOCOL; i++) {
      QFLE3_DBG(QFLE3_DBG_DCB,
            "remote_mib.app.app_pri_tbl[%x].app_id %x\n",
            i, features->app.app_pri_tbl[i].app_id);

      QFLE3_DBG(QFLE3_DBG_DCB,
            "remote_mib.app.app_pri_tbl[%x].pri_bitmap %x\n",
            i, features->app.app_pri_tbl[i].pri_bitmap);

      QFLE3_DBG(QFLE3_DBG_DCB,
            "remote_mib.app.app_pri_tbl[%x].appBitfield %x\n",
            i, features->app.app_pri_tbl[i].appBitfield);
   }

   return 0;
}

void
qfle3_dcbx_set_params(struct qfle3_adapter *adapter, u32 state)
{
   if (adapter->dcbx_sb.enabled == 0) {
      QFLE3_DBG(QFLE3_DBG_DCB, "DCBX disbaled state:%x\n", state);
      return;
   }
   if (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_RESETTING)){
      QFLE3_ERR("Adapter in Reset, abandon DCBX operation\n");
      return;
   }

   adapter->dcbx_sb.state = state;
   switch (state) {
      case QFLE3_DCBX_STATE_NEG_RECEIVED:
         {
            QFLE3_DBG(QFLE3_DBG_DCB, "Received DCBX negotiation results.\n");

            /* Read neg results if dcbx is in the FW */
            if (qfle3_dcbx_read_shmem_neg_results(adapter))
               return;

            qfle3_dump_dcbx_drv_param(adapter);

            qfle3_dump_dcbx_remote_param(adapter);
            qfle3_get_dcbx_drv_param(adapter);

            /* mark DCBX result for PMF migration */
            qfle3_update_drv_flags(adapter, 1 << DRV_FLAGS_DCB_CONFIGURED, 1);

            /* TODO: reconfigure the uplink with the results of the new
             * dcbx negotiation.
             */
            qfle3_dcbx_update_tc_mapping(adapter);

            /* allow other functions to update their netdevices
             * accordingly
             */
            if (IS_MF(adapter))
               qfle3_link_sync_notify(adapter);

            qfle3_schedule_helper(adapter, adapter->recovery_helper,
                                  qfle3_dcbx_restart_tx, 0);
            return;
         }
      case QFLE3_DCBX_STATE_TX_PAUSED:
         {
            QFLE3_DBG(QFLE3_DBG_DCB, "QFLE3_DCBX_STATE_TX_PAUSED\n");

            qfle3_dcbx_pfc_set_pfc(adapter);

            qfle3_dcbx_update_ets_params(adapter);

            /* ets may affect cmng configuration: reinit it in hw */
            qfle3_set_local_cmng(adapter);
            return;
         }
      case QFLE3_DCBX_STATE_TX_RELEASED:
         {
            QFLE3_DBG(QFLE3_DBG_DCB, "QFLE3_DCBX_STATE_TX_RELEASED\n");
            qfle3_mfw_command(adapter, DRV_MSG_CODE_DCBX_PMF_DRV_OK, 0);
            return;
         }
      default:
         QFLE3_ERR("Unknown DCBX_STATE\n");
   }
}

void qfle3_dcbx_pmf_update(struct qfle3_adapter *adapter)
{
   if (adapter->dcbx_sb.enabled == 0) {
      QFLE3_DBG(QFLE3_DBG_DCB, "DCBX is disbaled\n");
      return;
   }

   /* if we need to synchronize DCBX result from prev PMF
    * read it from shmem and update adapter and uplink accordingly
    */
   if (SHMEM2_HAS(adapter, drv_flags) &&
         GET_FLAGS(SHMEM2_RD(adapter, drv_flags), 1 << DRV_FLAGS_DCB_CONFIGURED)) {

      /* Read neg results if dcbx is in the FW */
      if (qfle3_dcbx_read_shmem_neg_results(adapter))
         return;

      qfle3_dump_dcbx_drv_param(adapter);

      qfle3_get_dcbx_drv_param(adapter);

      /*
       * reconfigure the uplink with the results of the new
       * dcbx negotiation.
       */
      qfle3_dcbx_update_tc_mapping(adapter);
   }
}

