/******************************************************************************
* The information contained in this file is confidential and proprietary to
* QLogic Corporation. No part of this file may be reproduced or
* distributed, in any form or by any means for any purpose, without the
* express written permission of QLogic Corporation.
*
*              Copyright (c) 2015 QLogic Corporation
*                        All rights reserved
*******************************************************************************/
#ifndef _QFLE3_MGMT_API_H_
#define _QFLE3_MGMT_API_H_

#include <vmkapi.h>

// VMKernel Management Callback Ids
#define QFLE3_DO_CALLBACK              (VMK_MGMT_RESERVED_CALLBACKS + 1)
#define QFLE3_NPAR_CALLBACK            (VMK_MGMT_RESERVED_CALLBACKS + 2)
#define QFLE3_SRIOV_CALLBACK           (VMK_MGMT_RESERVED_CALLBACKS + 3)
#define QFLE3_MODULE_DATA              (VMK_MGMT_RESERVED_CALLBACKS + 4)
#define QFLE3_INSTANCE_INFO            (VMK_MGMT_RESERVED_CALLBACKS + 5)
#define QFLE3_STATISTICS               (VMK_MGMT_RESERVED_CALLBACKS + 6)
#define QEDENTV_DIAG_CALLBACK				(VMK_MGMT_RESERVED_CALLBACKS + 7)
#define QEDENTV_NVM_CALLBACK				(VMK_MGMT_RESERVED_CALLBACKS + 8)
#define QFLE3_PHY_CALLBACK                             (VMK_MGMT_RESERVED_CALLBACKS + 9)

#define QFLE3_NUM_CALLBACKS 9

#define QFLE3_INTERFACE_NAME   "QFLE3"

#define QFLE3_INTERFACE_VENDOR "QLogic"

#define QFLE3_INTERFACE_VERSION  VMK_REVISION_FROM_NUMBERS(1, 0, 0, 1)        // This version needs to increase with each change.
// Major and Minor mismatch will cause a problem.

// This is the return value used by many callbacks.
typedef enum {
   QFLE3_CMD_RT_SUCCESS = 0,
   QFLE3_CMD_RT_INVALID_PARAM,
   QFLE3_CMD_RT_ERROR,
   QFLE3_CMD_RT_MAX
} QFLE3_RETURN;

typedef struct {
   vmk_uint32 phy_id;
   vmk_uint16 reg_num;
   vmk_uint16 val_in;
   vmk_uint16 val_out;
   vmk_uint16 magic;
   vmk_uint32 rv;               // return value from management interface
} __attribute__ ((__packed__)) qfle3_phy_data_t;

typedef struct {
   enum {
      QFLE3_CMD_PHY_REG_READ,
      QFLE3_CMD_PHY_REG_WRITE,
   } qfle3_phy_cmd;
} __attribute__ ((__packed__)) qfle3_phy_cmd_t;
/*******************************************QFLE3_DO_CALLBACK***********************************/

// Cmd codes for QFLE3_DO_CALLBACK
// The callback requires a QFLE3_mgmt_data_t as both input and output parameter
// Depends on the command, different fields of the structure is filled out
typedef enum {
   QFLE3_CMD_NONE = 0,        // should not be used

   QFLE3_CMD_PCI_CFG_READ,    // Read and write PCI config registers
      /*************************************************
       * INPUT:
       *    cmd = QFLE3_CMD_PCI_CFG_READ
       *    off = register offset
       *    size = size of register in bytes (1,2,4)
       * OUTPUT:
       *    size = size of data in bytes (4)
       *    u[] = read data
       *    rv = return value.  None zero indicates error
       *************************************************/
   QFLE3_CMD_PCI_CFG_WRITE,   // sizes 1, 2, or 4 bytes
       /*************************************************
       * INPUT:
       *    cmd = QFLE3_CMD_PCI_CFG_WRITE
       *    off = register offset
       *    size = size of data in bytes(1,2,4)
       *    u[] = data to write
       * OUTPUT:
       *    rv = return value.  None zero indicates error
       *************************************************/
   QFLE3_CMD_REG_READ,        // Read a register
       /*************************************************
       * INPUT:
       *    cmd = QFLE3_CMD_REG_READ
       *    off = register offset (4 byte aligned)
       *    size = size of register in bytes (4)
       * OUTPUT:
       *    size = size of data in bytes (4)
       *    u[] = read data
       *    rv = return value.  None zero indicates error
       *************************************************/
   QFLE3_CMD_REG_WRITE,       // Write to a register
       /*************************************************
       * INPUT:
       *    cmd = QFLE3_CMD_REG_WRITE
       *    off = register offset
       *    size = size of data in bytes(4)
       *    u[] = data to write
       * OUTPUT:
       *    rv = return value.  None zero indicates error
       *************************************************/
   QFLE3_CMD_TRCBUF_INFO,     // get trace buffer INFO
       /*************************************************
       * INPUT:
       *    cmd = QFLE3_CMD_TRCBUF_INFO
       * OUTPUT:
       *    size = size of QFLE3_trc_buff_into_t
       *    u[] = QFLE3_trc_buff_into_t
       *    rv = return value.  None zero indicates error
       *************************************************/
   QFLE3_CMD_TRCBUF_READ,     // read a section of the trace buffer
       /*************************************************
       * INPUT:
       *    cmd = QFLE3_CMD_TRCBUF_READ
       *    off = trace buffer line offset
       *    size = number of lines to read
       *    ptr[] = memory >= size * lineSize (returned in QFLE3_trc_buff_info_t)
       * OUTPUT:
       *    size = number of lines returned (<= number requested)
       *    ptr[] = $size number of trace buffer lines
       *    rv = return value.  None zero indicates error
       *************************************************/
   QFLE3_CMD_MINIDUMP_SIZE,   // get minidump size
       /*************************************************
       * INPUT:
       *    cmd = QFLE3_CMD_MINIDUMP_SIZE
       * OUTPUT:
       *    size = size of returned data in u[]
       *    u[] = minidump size
       *    rv = return value.  None zero indicates error
       *************************************************/
   QFLE3_CMD_MINIDUMP_READ,   // read a section of the minidump
       /*************************************************
       * INPUT:
       *    cmd = QFLE3_CMD_MINIDUMP_READ
       *    off = offset in minidump buffer
       *    size = number of bytes to read from the offset
       * OUTPUT:
       *    size = size of returned data in ptr[]
       *    ptr[] = minidump buffer
       *    rv = return value.  None zero indicates error
       *************************************************/
   QFLE3_CMD_DIAG_TEST,       // do diagnostic test, test cases are in the payload.
       /*************************************************
       * INPUT:
       *    cmd = QFLE3_CMD_DIAG_TEST
       *    u[] = test case as a string of 32 chars
       * OUTPUT:
       *    rv = return value.  None zero indicates error
       *************************************************/
   QFLE3_CMD_DCBX_VALUES_SIZE,        // get the size(length) of the DCBX values
       /*************************************************
       * INPUT:
       *    cmd = QFLE3_CMD_DCBX_VALUES_SIZE
       * OUTPUT:
       *    size = size of returned data in u[]
       *    u[] = dcbx size
       *    rv = return value.  None zero indicates error
       *************************************************/
   QFLE3_CMD_DCBX_VALUES_READ,        // read all of the DCBX values
       /*************************************************
       * INPUT:
       *    cmd = QFLE3_CMD_DCBX_VALUES_READ
       *    size = dcbx size
       * OUTPUT:
       *    ptr[] = dcbx buffer
       *    rv = return value.  None zero indicates error
       *************************************************/
   QFLE3_CMD_MOD_PARAMS_INFO, // get the number of module parameters
       /*************************************************
       * INPUT:
       *    cmd = QFLE3_CMD_MOD_PARAMS_INFO
       * OUTPUT:
       *    size = size of QFLE3_mod_param_into_t
       *    u[] = QFLE3_mod_param_into_t
       *    rv = return value.  None zero indicates error
       *************************************************/
   QFLE3_CMD_MOD_PARAMS_READ, // read module parameters
       /*************************************************
       * INPUT:
       *    cmd = QFLE3_CMD_MOD_PARAMS_READ
       *    off = mod parameter offset
       *    size = number of mod parameters to read
       *    ptr[] = memory >= size * mod parameter size (returned in QFLE3_mod_param_into_t)
       * OUTPUT:
       *    size = number of mod parameters returned (<= number requested)
       *    ptr[] = $size number of mod parameters
       *    rv = return value.  None zero indicates error
       *************************************************/
   QFLE3_CMD_WOL_GET,         // read WOL parameters
       /*************************************************
       * INPUT:
       *    cmd = QFLE3_CMD_WOL_GET
       * OUTPUT:
       *    size = size of QFLE3_wol_param_t
       *    u[] = QFLE3_wol_param_t
       *    rv = return value.	None zero indicates error
       *************************************************/
   QFLE3_CMD_WOL_SET,         // change WOL parameters
       /*************************************************
       * INPUT:
       *	cmd = QFLE3_CMD_WOL_SET
       *	size = size of QFLE3_wol_param_t
       *	u[] = QFLE3_wol_param_t
       * OUTPUT:
       *	rv = return value.	None zero indicates error
       *************************************************/
   QFLE3_PAUSE_GET,           // read pause parameters
       /*************************************************
       * INPUT:
       *    cmd = QFLE3_PAUSE_GET
       * OUTPUT:
       *    size = size of QFLE3_pause_param_t
       *    u[] = QFLE3_pause_param_t
       *    rv = return value.	None zero indicates error
       *************************************************/
   QFLE3_PAUSE_SET,           // change pause parameters
       /*************************************************
       * INPUT:
       *    cmd = QFLE3_PAUSE_SET
       *    size = size of QFLE3_pause_param_t
       *    u[] = QFLE3_pause_param_t
       * OUTPUT:
       *    rv = return value.	None zero indicates error
       *************************************************/
   QFLE3_COALESCE_GET,        // read coalesce parameters
       /*************************************************
       * INPUT:
       *    cmd = QFLE3_COALESCE_GET
       * OUTPUT:
       *    size = size of QFLE3_coalesce_param_t
       *    u[] = QFLE3_coalesce_param_t
       *    rv = return value.	None zero indicates error
       *************************************************/
   QFLE3_COALESCE_SET,        // change coalesce parameters
       /*************************************************
       * INPUT:
       *    cmd = QFLE3_COALESCE_SET
       *    size = size of QFLE3_coalesce_param_t
       *    u[] = QFLE3_coalesce_param_t
       * OUTPUT:
       *    rv = return value.	None zero indicates error
       *************************************************/
   QFLE3_PHYSICAL_ID,         // blink LED to physically identify port
       /*************************************************
       * INPUT:
       *    cmd = QFLE3_PHYSICAL_ID
       *    size = 4
       *    u[] = 1 - on, 0 - off;
       * OUTPUT:
       *    rv = return value.	None zero indicates error
       *************************************************/
   QFLE3_OFFLOAD_GET,         // read offload parameters
       /*************************************************
       * INPUT:
       *    cmd = QFLE3_OFFLOAD_GET
       * OUTPUT:
       *    size = size of QFLE3_offload_param_t
       *    u[] = QFLE3_offload_param_t
       *    rv = return value.	None zero indicates error
       *************************************************/
   QFLE3_OFFLOAD_SET,         // change offload parameters
       /*************************************************
       * INPUT:
       *    cmd = QFLE3_OFFLOAD_SET
       *    size = size of QFLE3_offload_param_t
       *    u[] = QFLE3_offload_param_t
       * OUTPUT:
       *    rv = return value.	None zero indicates error
       *************************************************/
   QFLE3_RESTART_NEG,         // restart negotiation
       /*************************************************
       * INPUT:
       *    cmd = QFLE3_RESTART_NEG
       * OUTPUT:
       *    rv = return value.	None zero indicates error
       *************************************************/
   QFLE3_RING_GET,            // read ring parameters
       /*************************************************
       * INPUT:
       *    cmd = QFLE3_RING_GET
       * OUTPUT:
       *    size = size of QFLE3_ring_param_t
       *    u[] = QFLE3_ring_param_t
       *    rv = return value.	None zero indicates error
       *************************************************/
   QFLE3_RING_SET,            // change ring parameters
       /*************************************************
       * INPUT:
       *    cmd = QFLE3_RING_SET
       *    size = size of QFLE3_ring_param_t
       *    u[] = QFLE3_ring_param_t
       * OUTPUT:
       *    rv = return value.	None zero indicates error
       *************************************************/

   QFLE3_CMD_IDLECHK,
   QFLE3_CMD_IDLECHK_SIZE,

   QFLE3_CMD_MAX
} QFLE3_MGMT_CMD;

/* Self test commands to management interface. Any
 * changes to below enum must also update the corresponding
 * function table - 'qedrntv_selftest_ecore'.
 */
typedef enum  {
   QFLE3_SELFTEST_MEMORY,
   QFLE3_SELFTEST_INTERRUPT,
   QFLE3_SELFTEST_REGISTER,
   QFLE3_SELFTEST_CLOCK,
   QFLE3_SELFTEST_NVRAM,
   QFLE3_SELFTEST_LOOPBACK,
   QFLE3_SELFTEST_MAX
} qfle3_selftest_cmd_t;

// used by QFLE3_DO_CALLBACK
typedef struct {
   vmk_uint32 cmd;     // input;          CMD opcode from QFLE3_MGMT_CMD
   vmk_uint32 hwfn;    // input/output    hwfn number or number of hwfns
   vmk_uint64 off;     // input;          offset
   vmk_uint32 size;    // input/output;   size of the value in u[] or pointed to by ptr
   vmk_uint32 rv;      // output;         return value, QFLE3_RETURN
   char u[64];         // input/output;   data value passed in/ passed out that is less than 64 bytes.
   vmk_uint64 ptr;     // output;         pointer to memory for passing information in and out larger than 64 bytes
} __attribute__ ((__packed__)) qfle3_mgmt_data_t;

// a line in the trace buffer consists of a trace_t structure.
typedef struct trace_s {
   vmk_uint64 fn_ptr;
   vmk_uint64 ts;
   vmk_uint32 cpu;
   vmk_uint32 line;
   vmk_uint64 val;
} __attribute__ ((__packed__)) qfle3_trace_t;

typedef struct {
   vmk_uint16 lineSize;
   vmk_uint16 lineNum;
   vmk_uint16 startLine;
   vmk_uint16 endLine;
} __attribute__ ((__packed__)) qfle3_trc_buff_info_t;

typedef struct {
   vmk_uint16 modParamNum;      // number of module parameters
   vmk_uint16 modParamSize;     // size of each module paramter structure.
} __attribute__ ((__packed__)) qfle3_mod_param_info_t;

#define QFLE3_MOD_PARAM_NAME_LEN            64
#define QFLE3_MOD_PARAM_DESC_LEN            256
#define QFLE3_MOD_PARAM_MAX_VALID_VALUES    32
#define QFLE3_MOD_PARAM_VALID_VALS_TERM     (-1)

enum qfle3_module_param_type {
   param_bool = 0,
   param_int_dis,
   param_int
};

typedef struct qfle3_module_param_s {
   char name[QFLE3_MOD_PARAM_NAME_LEN];
   char description[QFLE3_MOD_PARAM_DESC_LEN];
   int type;                    /* 0:bool, 1:int-discrete, 2:int */
   int default_value;
   int min_value;
   int max_value;
   /* QFLE3_MOD_PARAM_VALID_VALS_TERM. terminated array. */
   int valid_values[QFLE3_MOD_PARAM_MAX_VALID_VALUES];
   int rsvd1;
   int rsvd2;
} qfle3_module_param_t;

//enum qfle3_diag_test {
//   QFLE3_REGISTER_TEST,
//   QFLE3_HARDWARE_TEST,
//   QFLE3_LINK_TEST,
//   QFLE3_INTERRUPT_TEST,
//   QFLE3_LED_TEST,
//};

typedef struct {
   vmk_Bool csum;
   vmk_Bool tso;
   vmk_Bool vxlan;
} qfle3_offload_param_t;

typedef enum {
   /** Wake on directed frames */
   WAKE_ON_PHY         =       0x01,

   /** Wake on unicast frame */
   WAKE_ON_UCAST       =       0x02,

   /** Wake on multicat frame */
   WAKE_ON_MCAST       =       0x04,

   /** Wake on broadcast frame */
   WAKE_ON_BCAST       =       0x08,

   /** Wake on arp */
   WAKE_ON_ARP         =       0x10,

   /** Wake up magic frame */
   WAKE_ON_MAGIC       =       0x20,

   /** Wake on magic frame */
   WAKE_ON_MAGICSECURE =       0x40

} WolCaps;


/**
 * \brief Structure describing the wake-on-lan features and state of an
 *        uplink
 */
typedef struct {

   /** Uplink supported wake-on-lan features */
   WolCaps supported;

   /** Uplink enabled wake-on-lan features */
   WolCaps enabled;
#define WOL_STRING_MAX       16
   /** Wake-On-LAN secure on password */
   char secureONPassword[WOL_STRING_MAX];

} qfle3_wol_param_t;

typedef struct {
   vmk_uint32             rxUsecs;
   vmk_uint32             rxMaxFrames;
   vmk_uint32             txUsecs;
   vmk_uint32             txMaxFrames;
   vmk_Bool               useAdaptiveRx;
   vmk_Bool               useAdaptiveTx;
   vmk_uint32             rateSampleInterval;
   vmk_uint32             pktRateLowWatermark;
   vmk_uint32             pktRateHighWatermark;
   vmk_uint32             rxUsecsLow;
   vmk_uint32             rxFramesLow;
   vmk_uint32             txUsecsLow;
   vmk_uint32             txFramesLow;
   vmk_uint32             rxUsecsHigh;
   vmk_uint32             rxFramesHigh;
   vmk_uint32             txUsecsHigh;
   vmk_uint32             txFramesHigh;
} qfle3_coalesce_param_t;

typedef struct {
   vmk_uint32  rxMaxPending;
   vmk_uint32  rxMiniMaxPending;
   vmk_uint32  rxJumboMaxPending;
   vmk_uint32  txMaxPending;
   vmk_uint32  rxPending;
   vmk_uint32  rxMiniPending;
   vmk_uint32  rxJumboPending;
   vmk_uint32  txPending;
} qfle3_ring_param_t;

typedef enum {
   /** PAUSE bit, device supports symmetric PAUSE ability when set */
   FLOW_CTRL_PAUSE       = (1 << 0),

   /** ASM_DIR bit, device supports asymmetric PAUSE ability when set */
   FLOW_CTRL_ASYM_PAUSE  = (1 << 1),
} FlowCtrlAbility;
 
typedef struct {
   vmk_Bool autoNegotiate;
   vmk_Bool rxPauseEnabled;
   vmk_Bool txPauseEnabled;
   FlowCtrlAbility localDeviceAdvertise;
   FlowCtrlAbility linkPartnerAdvertise;
} qfle3_pause_params_t;


#define DCBX_CONFIG_MAX_APP_PROTOCOL 4
#define DCBX_MAX_APP_PROTOCOL         16

struct qfle3_admin_priority_app_table {
                vmk_uint32 valid;
                vmk_uint32 priority;
#define INVALID_TRAFFIC_TYPE_PRIORITY   (0xFFFFFFFF)
                vmk_uint32 traffic_type;
#define TRAFFIC_TYPE_ETH                0
#define TRAFFIC_TYPE_PORT               1
                vmk_uint32 app_id;
};

struct qfle3_config_dcbx_params {
    vmk_uint32 overwrite_settings;
    vmk_uint32 admin_dcbx_version;
    vmk_uint32 admin_ets_enable;
    vmk_uint32 admin_pfc_enable;
    vmk_uint32 admin_tc_supported_tx_enable;
    vmk_uint32 admin_ets_configuration_tx_enable;
    vmk_uint32 admin_ets_recommendation_tx_enable;
    vmk_uint32 admin_pfc_tx_enable;
    vmk_uint32 admin_application_priority_tx_enable;
    vmk_uint32 admin_ets_willing;
    vmk_uint32 admin_ets_reco_valid;
    vmk_uint32 admin_pfc_willing;
    vmk_uint32 admin_app_priority_willing;
    vmk_uint32 admin_configuration_bw_percentage[8];
    vmk_uint32 admin_configuration_ets_pg[8];
    vmk_uint32 admin_recommendation_bw_percentage[8];
    vmk_uint32 admin_recommendation_ets_pg[8];
    vmk_uint32 admin_pfc_bitmap;
    struct qfle3_admin_priority_app_table
        admin_priority_app_table[DCBX_CONFIG_MAX_APP_PROTOCOL];
    vmk_uint32 admin_default_priority;
};

struct qfle3_dcbx_params_get {
    vmk_uint32 ver_num;
    vmk_uint32 dcb_state;
    vmk_uint32 dcbx_enabled;
    struct qfle3_config_dcbx_params config_dcbx_params;
    /* The reserved field should follow in case the struct above will increase*/
    vmk_uint32 _reserved[19];

    vmk_uint32 dcb_current_state;
#define QFLE3_DCBX_CURRENT_STATE_IS_SYNC            (1 << 0)
#define QFLE3_PFC_IS_CURRENTLY_OPERATIONAL          (1 << 1)
#define QFLE3_ETS_IS_CURRENTLY_OPERATIONAL          (1 << 2)
#define QFLE3_PRIORITY_TAGGING_IS_CURRENTLY_OPERATIONAL     (1 << 3)

    vmk_uint32 local_tc_supported;
    vmk_uint32 local_pfc_caps;
    vmk_uint32 remote_tc_supported;
    vmk_uint32 remote_pfc_cap;
    vmk_uint32 remote_ets_willing;
    vmk_uint32 remote_ets_reco_valid;
    vmk_uint32 remote_pfc_willing;
    vmk_uint32 remote_app_priority_willing;
    vmk_uint32 remote_configuration_bw_percentage[8];
    vmk_uint32 remote_configuration_ets_pg[8];
    vmk_uint32 remote_recommendation_bw_percentage[8];
    vmk_uint32 remote_recommendation_ets_pg[8];
    vmk_uint32 remote_pfc_bitmap;
    struct qfle3_admin_priority_app_table
            remote_priority_app_table[DCBX_MAX_APP_PROTOCOL];
    vmk_uint32 local_ets_enable;
    vmk_uint32 local_pfc_enable;
    vmk_uint32 local_configuration_bw_percentage[8];
    vmk_uint32 local_configuration_ets_pg[8];
    vmk_uint32 local_pfc_bitmap;
    struct qfle3_admin_priority_app_table
            local_priority_app_table[DCBX_MAX_APP_PROTOCOL];
    vmk_uint32 pfc_mismatch;
    vmk_uint32 priority_app_mismatch;
    vmk_uint32 dcbx_frames_sent;
    vmk_uint32 dcbx_frames_received;
    vmk_uint32 pfc_frames_sent[2];
    vmk_uint32 pfc_frames_received[2];
};

 /*******************************************QFLE3_SRIOV_CALLBACK***********************************/
typedef struct {
   enum {
      QFLE3_NPAR_GET,         // Get operation for NPAR
      QFLE3_NPAR_SET,         // Set operation for NPAR
      QFLE3_SRIOV_GET,        // Get operation for SRIOV
      QFLE3_SRIOV_SET,        // Set operation for SRIOV
      QFLE3_PF_BW_GET,        //get PF bandwidth
      QFLE3_PF_BW_SET,        //Set PF Bandwidth
   } npar_sriov_cmd;
} __attribute__ ((__packed__)) qfle3_npar_sriov_cmd_t;

typedef struct {
   vmk_uint32 pfid;             // PF function id.  Must be provided for all GET/SET operation
   vmk_uint32 min_bw;           // rw, minimum bandwidth allocated
   vmk_uint32 max_bw;           // rw, maximum bandwidth allocated
   vmk_uint8 mac_addr[6];       // ro, mac address of the PF.
   vmk_uint8 max_num_vf;        // rw, maximum number of VFs allowed on this PF
   vmk_uint8 enabled;           // ro, This partition is enabled
   vmk_uint8 sriov_enabled;     // rw, enable sriov on a per PF basis
   vmk_uint8 type;              // ro, NIC, FCoE, iSCSI
} __attribute__ ((__packed__)) qfle3_pf_data_t;

typedef struct {
   vmk_uint16 start_vlan_id;    // starting vlan id.
   vmk_uint16 num_vlanid;       // number of vlan ids in the block, (<=1)
} __attribute__ ((__packed__)) qfle3_vlan_blocks;

typedef struct {                // one per vf.
   vmk_uint32 vfid;             // VFID.  Must be provide for all GET/SET
   vmk_uint32 pfid;             // ro, The associated physical function.
   vmk_uint32 min_bw;           // rw, Minimum bandwidth guaranteed
   vmk_uint32 max_bw;           // rw, Maximum bandwidth allowed
   vmk_uint32 max_tx_queues;    // ro, maximum number of TX queues allowed.
   vmk_uint32 max_rx_queues;    // ro, maximum number of RX queues allowed.
   vmk_uint32 max_filters_allowed;      // ro, Maximum number of filters allowed.
   vmk_uint16 pvid;             // ro, Port VLAN ID.
   vmk_int8 vf_status;          // ro, -1 is disabled, 0 is enabled ( assigned but VM is OFF or VF driver is unloaded ), 1 is assigned with VM up, 2 VF Uplinked
   vmk_uint8 spoofing_check_enabled;    // ro, allow spoofing check.
   vmk_uint32 num_vlan_blocks;  // ro, number of vlan blocks in the array below;
   qfle3_vlan_blocks vlan_blocks[32]; // ro, QFLE3_vlan_blocks vlan_blocks[] Array size should be preferably less than or equal to SRIOV_GVLAN_ARR_SIZE
   vmk_uint64 rsvd;
} __attribute__ ((__packed__)) qfle3_sriov_vf_data_t;

/*******************************************QFLE3_MODULE_DATA***********************************/

typedef struct {
   vmk_Name drv_name;
   vmk_Name drv_version;
   vmk_Name fw_version;
   vmk_Name pci_bus_info;
   vmk_uint32 nvm_size;
   vmk_uint32 vmkapi;
} __attribute__ ((__packed__)) qfle3_module_data_t;

/*******************************************QFLE3_INSTANCE_INFO***********************************/
typedef enum CableType {
   /** Other non supported cable type */
   CABLE_TYPE_OTHER = 0,

   /** Twisted Pair */
   CABLE_TYPE_TP = 0x01,

   /** Attachment Unit Interface */
   CABLE_TYPE_AUI = 0x02,

   /** Media Independent Interface */
   CABLE_TYPE_MII = 0x04,

   /** Fibre */
   CABLE_TYPE_FIBRE = 0x08,

   /** Bayonet Neill-Concelman */
   CABLE_TYPE_BNC = 0x10,

   /** Direct Attach Copper */
   CABLE_TYPE_DA = 0x20,
} CableType_T;
#define VMK_ETH_ADDR_LENGTH           6
typedef vmk_uint8 vmk_EthAddress[VMK_ETH_ADDR_LENGTH];

typedef struct {
   vmk_EthAddress MAC;
   vmk_uint32 MTU;
   vmk_uint16 link_speed;
   vmk_uint8 link_status;
   vmk_uint8 duplexity;
   vmk_uint8 port_type;
   vmk_uint32 supportedCableTypes;
   vmk_uint8 phy_addr;
   vmk_uint8 transceiver_type;
   vmk_Bool autonegotiationSupported;
   vmk_uint8 autonegotiation;
   vmk_uint32 supportedModesArray[64];
   vmk_uint8 supportedModesArraySize;
   vmk_uint8 npar_mode;         // ro, npar mode: NPAR_DISABLED, NPAR_SWITCH_INDEPENDENT, NPAR_SWITCH_DEPENDENT,etc
} __attribute__ ((__packed__)) qfle3_instance_info_t;

/*******************************************QFLE3_STATISTICS***********************************/
#define QFLE3_MAX_NO_QUEUES                           12
#define QFLE3_MAX_NO_TC                               8

typedef struct {
   vmk_uint64 xmitencapcalled;
   vmk_uint64 tso_bytes;
   vmk_uint64 tso_pkts;
   vmk_uint64 encap_tso;
   vmk_uint64 inr_ipv4;
   vmk_uint64 inr_ipv6;
   vmk_uint64 outer_ipv4;
   vmk_uint64 outer_ipv6;
   vmk_uint64 inr_tcp;
   vmk_uint64 inr_udp;
   vmk_uint64 outer_udp;
   vmk_uint64 outer_tcp;
   vmk_uint64 drop_malformed;
   vmk_uint64 no_offload;
} __attribute__ ((__packed__)) qfle3_tx_encap_stats_t;

typedef struct {
   vmk_uint64 q_state;
   vmk_uint64 tx_packets;
   vmk_uint64 tx_bytes;
   vmk_uint64 tx_drops;
   vmk_uint64 tx_compl;
   vmk_uint64 frag_get_error;
   vmk_uint64 frag_cpy_error;
   vmk_uint64 too_many_frags;
   vmk_uint64 ring_error;
   vmk_uint64 sg_elem_error;
   vmk_uint64 dma_error;
   vmk_uint64 drop_bad_q;
   vmk_uint64 q_stopped;
   vmk_uint64 ring_full;
   vmk_uint64 cso_pkt;
   vmk_uint64 cso_bytes;
   vmk_uint64 tso_pkts;
   vmk_uint64 tso_bytes;
   vmk_uint64 drop_hdr_inspect_err;
   qfle3_tx_encap_stats_t encap_stats;
   vmk_uint64 unsupported_proto;
   vmk_uint64 no_csum;
   vmk_uint64 IPv4_csum;
   vmk_uint64 IPv4_tso_pkt;
   vmk_uint64 IPv6_csum;
   vmk_uint64 IPv6_tso_pkt;
   vmk_uint64 tcp_pkt;
   vmk_uint64 udp_pkt;
   vmk_uint64 pkt_larger_than_mtu;
   vmk_uint64 vlan_insertion;
   vmk_uint64 tso_error;
   vmk_uint64 dma_map_error;
   vmk_uint64 map_length_gt_hlength;
} __attribute__ ((__packed__)) qfle3_per_txq_stats_t;

typedef struct {
   vmk_uint64 rx_bytes;
   vmk_uint64 rx_pkts;
   vmk_uint64 lro_bytes;
   vmk_uint64 lro_pkts;
   vmk_uint64 outer_l4_csummed;
   vmk_uint64 outer_l3_csummed;
   vmk_uint64 inner_l4_csummed;
   vmk_uint64 inner_l3_csummed;
   vmk_uint64 outer_l4_csum_err;
   vmk_uint64 outer_l3_csum_err;
   vmk_uint64 inner_l4_csum_err;
   vmk_uint64 inner_l3_csum_err;
} __attribute__ ((__packed__)) qfle3_rx_encap_stats_t;

typedef struct {
   vmk_uint64 q_state;
   vmk_uint64 rx_packets;
   vmk_uint64 rx_bytes;
   vmk_uint64 rx_drops;         /* FIX ME: Populate or remove */
   vmk_uint64 drop_err;         /* FIX ME: Populate or remove */
   vmk_uint64 drop_fcs;         /* FIX ME: Populate or remove */
   vmk_uint64 rx_Buf_Alloc_fail;        /* FIX ME: Populate or remove */
   vmk_uint64 b_type_h;         /* FIX ME: Populate or remove */
   vmk_uint64 b_type_b;         /* FIX ME: Populate or remove */
   vmk_uint64 rx_hw_errors;
   vmk_uint64 rx_alloc_errors;
   vmk_uint64 netpoll_rx_errors;
   vmk_uint64 cso_pkt;
   vmk_uint64 lro_pkts;
   vmk_uint64 lro_bytes;
   qfle3_rx_encap_stats_t encap_stats;
   vmk_uint64 null_pkt;
   vmk_uint64 rx_frame_len_errors;
   vmk_uint64 rx_offset_errors;
   vmk_uint64 rx_dev_not_rdy_errors;
   vmk_uint64 cqe_type_slow_path;
   vmk_uint64 cqe_type_tpa_start;
   vmk_uint64 cqe_type_tpa_cont;
   vmk_uint64 cqe_type_tpa_end;
   vmk_uint64 rx_vlan_pkt;
} __attribute__ ((__packed__)) qfle3_per_rxq_stats_t;

typedef struct {
   qfle3_per_rxq_stats_t qfle3_rxq_stats_t;
   qfle3_per_txq_stats_t qfle3_txq_stats_t[QFLE3_MAX_NO_TC];
} __attribute__ ((__packed__)) qfle3_per_q_stats_t;

typedef struct {

           /** The number of rx packets received by the device */
   vmk_uint64 rxPkts;

           /** The number of tx packets sent by the device */
   vmk_uint64 txPkts;

           /** The number of rx bytes by the device */
   vmk_ByteCount rxBytes;

           /** The number of tx bytes by the device */
   vmk_ByteCount txBytes;

           /** The number of rx packets with errors */
   vmk_uint64 rxErrors;

           /** The number of tx packets with errors */
   vmk_uint64 txErrors;

           /** The number of rx packets dropped */
   vmk_uint64 rxDrops;

           /** The number of tx packets dropped */
   vmk_uint64 txDrops;

           /** The number of rx multicast packets */
   vmk_uint64 rxMulticastPkts;

           /** The number of rx broadcast packets */
   vmk_uint64 rxBroadcastPkts;

           /** The number of tx multicast packets */
   vmk_uint64 txMulticastPkts;

           /** The number of tx broadcast packets */
   vmk_uint64 txBroadcastPkts;

           /** The number of collisions */
   vmk_uint64 collisions;

           /** The number of rx length errors */
   vmk_uint64 rxLengthErrors;

           /** The number of rx ring buffer overflow */
   vmk_uint64 rxOverflowErrors;

           /** The number of rx packets with crc errors */
   vmk_uint64 rxCRCErrors;

           /** The number of rx packets with frame alignment error */
   vmk_uint64 rxFrameAlignErrors;

           /** The number of rx fifo overrun */
   vmk_uint64 rxFifoErrors;

           /** The number of rx packets missed */
   vmk_uint64 rxMissErrors;

           /** The number of tx aborted errors */
   vmk_uint64 txAbortedErrors;

           /** The number of tx carriers errors */
   vmk_uint64 txCarrierErrors;

           /** The number of tx fifo errors */
   vmk_uint64 txFifoErrors;

           /** The number of tx heartbeat errors */
   vmk_uint64 txHeartbeatErrors;

           /** The number of tx windows errors */
   vmk_uint64 txWindowErrors;

   /* Number of Fast Path Queues */
   vmk_uint64 num_fp_queues;

   /* Number of tc Per Fast Path Queue */
   vmk_uint64 num_tc;

   /* Per queue Counters */
   qfle3_per_q_stats_t qfle3_q_stats_t[QFLE3_MAX_NO_QUEUES];

} __attribute__ ((__packed__)) qfle3_statistics_t;

typedef struct {
//   vmk_uint32 cmd;              // input;               CMD opcode from QEDENTV_DIAG_CMD
//   vmk_uint32 size;             // input/output
//   vmk_uint32 flags;
//   vmk_uint32 off;
//   vmk_uint32 hwfn;             // hwfn number
//   vmk_uint32 rv;               // output, return value
//   vmk_uint64 v;
//   vmk_uint64 ptr;              // input/output
//   vmk_uint64 ptr2;             // input
//   vmk_uint32 size2;            // input
//   vmk_uint32 reserved;
} __attribute__ ((__packed__)) qfle3_diag_data_t;

typedef struct {
   vmk_uint32 cmd;              // mcp command
   vmk_uint32 param_offset;     // parameter
   vmk_uint32 param_size;
   vmk_uint32 param;
   vmk_uint32 mcp_resp;         // mcp response code
   vmk_uint32 mcp_param;        // return parameter from mcp
   vmk_uint32 txn_size;         // size of buf below in 32 bit words
   vmk_uint32 rv;               // return value from management interface
   vmk_uint32 unused;
   vmk_uint64 buf;              // 32 bit aligned buffer
} __attribute__ ((__packed__)) qfle3_nvm_data_t;


#define DRV_MSG_CODE_NVM_READ_NVRAM		0x00050000      /* Read nvram data. Param_offset is the offset, param_size  Len in Bytes */
#define DRV_MSG_CODE_NVM_WRITE_NVRAM		0x00060000      /* Writes to nvram. Param_offset is the offset, param_size  Len in Bytes. In case this address is in the range of secured file in secured mode, the operation will fail */

/********************************************************************************************/

extern vmk_MgmtCallbackInfo qfle3_mgmt_callbacks[QFLE3_NUM_CALLBACKS];
extern vmk_MgmtApiSignature qfle3_api_sig;
typedef vmk_uint8 u8;
typedef vmk_uint16 u16;
typedef vmk_uint32 u32;
typedef vmk_int32 s32;


#ifdef VMKERNEL
extern vmk_MgmtHandle vmkmgmt_api_handler;

void qfle3_callback(vmk_MgmtCookies * cookie, vmk_MgmtEnvelope * envelope,
                      qfle3_mgmt_data_t *);
void qfle3_npar_callback(vmk_MgmtCookies * cookie,
                           vmk_MgmtEnvelope * envelope,
                           qfle3_npar_sriov_cmd_t *, qfle3_pf_data_t *);
void qfle3_sriov_callback(vmk_MgmtCookies * cookie,
                            vmk_MgmtEnvelope * envelope,
                            qfle3_npar_sriov_cmd_t *,
                            qfle3_sriov_vf_data_t *);
void qfle3_module_data(vmk_MgmtCookies * cookie, vmk_MgmtEnvelope * envelope,
                         qfle3_module_data_t *);
void qfle3_instance_info(vmk_MgmtCookies * cookie,
                           vmk_MgmtEnvelope * envelope,
                           qfle3_instance_info_t *);
void qfle3_statistics(vmk_MgmtCookies * cookie, vmk_MgmtEnvelope * envelope,
                        qfle3_statistics_t * data);
void qfle3_diag_callback(vmk_MgmtCookies * cookie,
                           vmk_MgmtEnvelope * envelope, qfle3_diag_data_t *);
void qfle3_nvm_callback(vmk_MgmtCookies * cookie, vmk_MgmtEnvelope * envelope,
                          qfle3_nvm_data_t *);
void qfle3_phy_reg_callback(vmk_MgmtCookies * cookie,
                           vmk_MgmtEnvelope * envelope,
                           qfle3_phy_cmd_t *, qfle3_phy_data_t *);
#else
#define qfle3_callback NULL
#define qfle3_npar_callback NULL
#define qfle3_sriov_callback NULL
#define qfle3_module_data NULL
#define qfle3_instance_info NULL
#define qfle3_statistics NULL
#define qfle3_diag_callback NULL
#define qfle3_nvm_callback NULL
#define qfle3_phy_reg_callback NULL

#endif

#endif
