/*******************************************************************************
 * The information contained in this file is confidential and proprietary to
 * QLogic Corporation.  No part of this file may be reproduced or
 * distributed, in any form or by any means for any purpose, without the
 * express written permission of QLogic Corporation.
 *
 * (c) COPYRIGHT 2015 QLogic Corporation, ALL RIGHTS RESERVED.
 *******************************************************************************/
/* **********************************************************
  * Copyright 2015 VMware, Inc.  All rights reserved. -- VMware Confidential
  * **********************************************************/


/*
 * qfle3_mod.c --
 *
 *      Kernel module implementation for native qfle3 driver.
 */

#include "qfle3.h"
#include "qfle3_mgmt_api.h"
/*
* Module Parameters
*/

/* Debug */
vmk_uint32 qfle3_debugMask = 0x0;
VMK_MODPARAM_NAMED(debug_mask, qfle3_debugMask, uint, "Enabled debug mask (default: 0)");

/* Tunable device values... */
vmk_uint32 load_mtu = 1500;
VMK_MODPARAM_NAMED(mtu, load_mtu, uint, "MTU when the driver is loaded. Range: 0-9000. (default: 1500)");

/* Interrupt Mode: 0 (Auto), 1 (IRQ), 2 (MSI), and 3 (MSI-X) */
vmk_uint32 qfle3_intr_mode = QFLE3_IT_AUTO;
//VMK_MODPARAM_NAMED(intr_mode, qfle3_intr_mode, uint,
//             " Interrupt Mode: 0 (AUTO), 1 (IRQ), 2 (MSI), and 3 (MSI-X). Default: Auto");

vmk_uint32 qfle3_txqueue_count = QFLE3_DEVICE_MAX_TX_QUEUES;
vmk_uint32 qfle3_rxqueue_count = QFLE3_DEVICE_MAX_RX_QUEUES;
vmk_uint32 DRSS_argc = 0;
vmk_uint32 RSS_argc = 0;
vmk_uint32 DRSS[QFLE3_MAX_NUM_OF_PFS] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
vmk_uint32 RSS[QFLE3_MAX_NUM_OF_PFS] = {1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1};

VMK_MODPARAM_NAMED(txqueue_nr, qfle3_txqueue_count, uint,
             "Number of Tx Queues: 0 (Auto) or 1 to "VMK_STRINGIFY(QFLE3_DEVICE_MAX_TX_QUEUES)" (fixed queue number). Default: Auto");
VMK_MODPARAM_NAMED(rxqueue_nr, qfle3_rxqueue_count, uint,
             "Number of Rx Queues: 0 (Auto) or 1 to "VMK_STRINGIFY(QFLE3_DEVICE_MAX_RX_QUEUES)" (fixed queue number). Default: Auto");
VMK_MODPARAM_ARRAY_NAMED(RSS, RSS, uint, &RSS_argc,
             "Number of RSS Queues: 0 (Auto) or 1 to 4 (fixed queue number). Default: 1");
VMK_MODPARAM_ARRAY_NAMED(DRSS, DRSS, uint, &DRSS_argc,
       "Number of RSS Queues associated with default queue: 0 (Disabled), 2 (min), 4 (max). Default: Disabled");
#if (ESX_DDK_VERSION >= 2017)
vmk_uint32 rss_engine_nr_argc = 0;
vmk_uint32 rss_engine_nr[QFLE3_MAX_NUM_OF_PFS] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
VMK_MODPARAM_ARRAY_NAMED(rss_engine_nr, rss_engine_nr, uint, &rss_engine_nr_argc,
       "Number of RSS Engines: 0 (Disabled) or 1 to 4 (fixed NUmber of RSS Engines). Default: 0");
#endif
vmk_uint8 hw_vlan_en = 1;
VMK_MODPARAM_NAMED(hw_vlan, hw_vlan_en, uint,
	"Enable/Disable VLAN removal/insertion by hardware. 0: Disabled and 1: Enabled. Default: 1");

/* Offload flags: 1 (cso) 2(tso) 4(vxlan offload) 8(Geneve offload)*/
vmk_uint32 qfle3_offload_en = 15;
VMK_MODPARAM_NAMED(offload_flags, qfle3_offload_en, uint,
             "Offload flags: 1 (cso) 2(tso) 4(vxlan offload) 8(Geneve offload). Default: 15");

vmk_int32 qfle3_multi_rx_filters = -1;
VMK_MODPARAM_NAMED(rx_filters, qfle3_multi_rx_filters, int,"Define the number of RX filters per NetQueue"
                                      "-1: use the default number of RX filters based on availability:"
                                      "0: Disable use of multiple RX filters; "
                                      "1,2,3,...: Will force the number of RX filters to use for NetQueue; "
									  "Default: -1");
vmk_uint32 enable_tpa = 1;
VMK_MODPARAM_NAMED(enable_lro, enable_tpa, uint, "Enable/Disable the TPA (LRO) feature"
                                        "Set to 0 to disable TPA, "
                                        "Set to 1 to enable TPA [Default]");

vmk_uint32 enable_vxlan_filters = 0;
VMK_MODPARAM_NAMED(enable_vxlan_filters, enable_vxlan_filters, uint, "Enable/Disable the VXLAN Rx filters."
                                        "Set to 0 to disable VXLAN Rx filters"
                                        "Set to 1 to enable VXLAN Rx filters"
					"Default: 0");
vmk_uint32 enable_fw_dump = 1;
VMK_MODPARAM_NAMED(enable_fwdump, enable_fw_dump, uint, "Enable/Disable the firmware dump file. "
                                        "Set to 1 to enable firmware dump, "
                                        "Set to 0 to disable firmware dump [Default]");
vmk_uint32 rx_bd_rings = 4096;
VMK_MODPARAM_NAMED(rxring_bd_nr, rx_bd_rings, uint, "Number of RX BD Buffers: 4096(min) 16384(max), must be power of two. Default: 4096");

vmk_uint32 tx_bd_rings = 4096;
VMK_MODPARAM_NAMED(txring_bd_nr, tx_bd_rings, uint, "Number of TX BD Buffers: 4096(min) 16384(max), must be power of two. Default: 4096");

vmk_uint32 enable_fcoe_queue = 0;
//VMK_MODPARAM_NAMED(enable_fcoe_queue, enable_fcoe_queue, uint, "Enable FCoE queue. 0: Disable (Default) 1: Enable");

vmk_uint32 disable_cnic = 0;
//VMK_MODPARAM_NAMED(disable_cnic, disable_cnic, uint, "Disable Offloaded CNIC functionality if supported. 0 (default)");
vmk_uint32 netpoll_count=0;
vmk_uint32 enable_dcbx = 1;
VMK_MODPARAM_NAMED(enable_dcbx, enable_dcbx, uint, "Enable DCBX. 0: Disable 1: Enable (Default)");

vmk_uint32 dropless_fc = 0;
VMK_MODPARAM_NAMED(dropless_fc, dropless_fc, uint, "Pause on exhausted host ring. 0: Disable (Default) 1: Disable");

vmk_uint32 sw_fcoe_pri = 3;// fcoe priority, negotiated through dcbx. could make into a module parameter in the future
//vmk_uint32 fcoe_ol_rx_bd_rings = 4096;
//vmk_uint32 fcoe_ol_tx_bd_rings = 4096;
vmk_uint32 iscsi_ol_rx_bd_rings = 4096;
vmk_uint32 iscsi_ol_tx_bd_rings = 4096;
vmk_uint32 iscsi_ol_mtu = 9000;


#ifdef QFLE3_SRIOV

int relative_pf_num = 0;

vmk_uint32 max_vfs_argc = 0;
vmk_uint32 max_vfs[QFLE3_MAX_NUM_OF_PFS] = {0};
VMK_MODPARAM_ARRAY_NAMED(max_vfs, max_vfs, uint, &max_vfs_argc, "Number of VFs to be enabled "
                        "for each pci function. Default:0, Valid values: "
                        "0:Disable, 1 to " VMK_STRINGIFY(QFLE3_MAX_NUM_OF_VFS));

#endif //QFLE3_SRIOV

vmk_uint32 fairness_threshold = 0;
VMK_MODPARAM_NAMED(fairness_threshold, fairness_threshold, uint, "When set to 1 will enable the fairness threshold; 0 by Default");


#if 0
vmk_int8 mp_pause = -1;
VMK_MODPARAM_NAMED(pause, mp_pause, uint, "Pause Parameters:  1(tx), 2(rx), 3(both), 4(none).");
#endif

vmk_uint32 tx_to_delay = 5;
VMK_MODPARAM_NAMED(tx_to_delay, tx_to_delay, uint, "Time interval (in seconds) to wait for before considering a Tx queue stuck and triggering a tx timeout. Default:5, Disable:0 (use netdev tx timeout), Minimum:5");
vmk_uint32 auto_recovery = 1;
vmk_uint32 psod_on_panic = 0;
//VMK_MODPARAM_NAMED(auto_recovery,auto_recovery, uint, "Automatically recover interface after detecting/inducing hardware error. Enable: 1, Disable:0, Default:1");
vmk_uint32 enable_live_grcdump = GRC_DMP_PARITY_ENABLED | GRC_DMP_ERROR_ENABLED;
//VMK_MODPARAM_NAMED(enable_live_grcdump, enable_live_grcdump, uint, "Enable live GRC dump "
//				      "0x0: Disable live GRC dump, "
//				      "0x1: Enable Parity error GRC dump"
//				      " [Enabled by default], "
//				      "0x2: Enable Any error GRC dump"
//				      " [Enabled by default], "
//				      "0x4: Enable Debug mode GRC dump.");

extern vmk_MgmtHandle vmkmgmt_api_handler;

qfle3_mod_info_t qfle3_mod_info = {
   .heapID = VMK_INVALID_HEAP_ID,
   .memPoolID = VMK_MEMPOOL_INVALID,
   .logID = VMK_INVALID_LOG_HANDLE,
   .logThrottledID = VMK_INVALID_LOG_HANDLE,
   .lockDomain = VMK_LOCKDOMAIN_INVALID,
};

static void
qfle3_mod_info_cleanup(void)
{

   vmk_MgmtDestroy(vmkmgmt_api_handler);

   while (!vmk_ListIsEmpty(&qfle3_mod_info.ecore_lock_list)) {
      vmk_ListLinks *p_entry;
      struct qfle3_lock *q_lock;

      p_entry = vmk_ListFirst(&qfle3_mod_info.ecore_lock_list);
      q_lock = VMK_LIST_ENTRY(p_entry, struct qfle3_lock, link);

      if (q_lock->type == QFLE3_LOCK_TYPE_SPINLOCK) {
         vmk_SpinlockDestroy(q_lock->spin_lock);
      } else if (q_lock->type == QFLE3_LOCK_TYPE_SEMAPHORE) {
         vmk_SemaDestroy(&q_lock->sema);
      } else if (q_lock->type == QFLE3_LOCK_TYPE_RW_SEMAPHORE) {
         vmk_RWSemaDestroy(&q_lock->rw_sema);
      }

      vmk_ListRemove(&q_lock->link);
      qfle3_heap_free(q_lock);
   }

   while (!vmk_ListIsEmpty(&qfle3_mod_info.prev_list)) {
      vmk_ListLinks *p_entry;
	  struct qfle3_prev_path_list *tmp_list;

      p_entry = vmk_ListFirst(&qfle3_mod_info.prev_list);
      tmp_list = VMK_LIST_ENTRY(p_entry, struct qfle3_prev_path_list, link);

      vmk_ListRemove(&tmp_list->link);
      qfle3_heap_free(tmp_list);
   }

   if (qfle3_mod_info.prev_mutex) {
	   vmk_SemaDestroy(&qfle3_mod_info.prev_mutex);
   }

   if (qfle3_mod_info.drv_lock) {
	   qfle3_destroy_spinlock(qfle3_mod_info.drv_lock);
   }

   if (qfle3_mod_info.driverID != NULL) {
      vmk_DriverUnregister(qfle3_mod_info.driverID);
   }
   if (qfle3_mod_info.dumpFile != NULL) {
      vmk_DumpDeleteFileCallback(qfle3_mod_info.dumpFile);
   }
   
   if (qfle3_mod_info.recoveryDumpFile != NULL) {
      vmk_DumpDeleteFileCallback(qfle3_mod_info.recoveryDumpFile);
   }
   if (qfle3_mod_info.lockDomain != VMK_LOCKDOMAIN_INVALID) {
      vmk_LockDomainDestroy(qfle3_mod_info.lockDomain);
   }
   if (qfle3_mod_info.memPoolID != VMK_MEMPOOL_INVALID) {
      vmk_MemPoolDestroy(qfle3_mod_info.memPoolID);
   }
   if (qfle3_mod_info.logThrottledID != VMK_INVALID_LOG_HANDLE) {
      vmk_LogUnregister(qfle3_mod_info.logThrottledID);
   }
   if (qfle3_mod_info.logID != VMK_INVALID_LOG_HANDLE) {
      vmk_LogUnregister(qfle3_mod_info.logID);
   }
   
   if (qfle3_mod_info.timer_queue != VMK_INVALID_TIMER_QUEUE) {
      vmk_TimerQueueDestroy(qfle3_mod_info.timer_queue);
   }
   if (qfle3_mod_info.heapID != VMK_INVALID_HEAP_ID) {
      vmk_HeapDestroy(qfle3_mod_info.heapID);
   }
   
}

#define QFLE3_HEAP_EST       (2 * VMK_MEGABYTE)
#define QFLE3_MAX_ADAPTERS   16

#define DRV_DUMP_TRACE_BUFFER_SIZE               (0x800)
#define DRV_DUMP_VFC_DATA_SIZE                   (0x10000)
#define DRV_DUMP_IGU_DATA_SIZE                   (0x10000)
/* DRV_DUMP_TRACE_BUFFER_SIZE should be same as DBG_DMP_TRACE_BUFFER_SIZE in hw_dump.c*/
#define DRV_DUMP_PRELIMINARY_DATA_SIZE           DRV_DUMP_TRACE_BUFFER_SIZE

#define DRV_DUMP_SPLIT_REGISTERS_SIZE_DEFUALT    (0x2000)
#define DRV_DUMP_SPLIT_REGISTERS_SIZE_E1  DRV_DUMP_SPLIT_REGISTERS_SIZE_DEFUALT
#define DRV_DUMP_SPLIT_REGISTERS_SIZE_E1H DRV_DUMP_SPLIT_REGISTERS_SIZE_DEFUALT
#define DRV_DUMP_SPLIT_REGISTERS_SIZE_E2  DRV_DUMP_SPLIT_REGISTERS_SIZE_DEFUALT
#define DRV_DUMP_SPLIT_REGISTERS_SIZE_E3A0 DRV_DUMP_SPLIT_REGISTERS_SIZE_DEFUALT
#define DRV_DUMP_SPLIT_REGISTERS_SIZE_E3B0 DRV_DUMP_SPLIT_REGISTERS_SIZE_DEFUALT

#define DRV_DUMP_EXTRA_BLOCKS_SIZE         (DRV_DUMP_VFC_DATA_SIZE + \
                                            DRV_DUMP_IGU_DATA_SIZE)
#define DRV_DUMP_CRASH_DMP_BUF_SIZE_E1     (0xB0000 + \
                                            DRV_DUMP_PRELIMINARY_DATA_SIZE + \
                                            DRV_DUMP_EXTRA_BLOCKS_SIZE + \
                                            DRV_DUMP_SPLIT_REGISTERS_SIZE_E1)
#define DRV_DUMP_CRASH_DMP_BUF_SIZE_E1H    (0xE0000 + \
                                            DRV_DUMP_PRELIMINARY_DATA_SIZE + \
                                            DRV_DUMP_EXTRA_BLOCKS_SIZE + \
                                            DRV_DUMP_SPLIT_REGISTERS_SIZE_E1H)
#define DRV_DUMP_CRASH_DMP_BUF_SIZE_E2     (0x100000 + \
                                            DRV_DUMP_PRELIMINARY_DATA_SIZE + \
                                            DRV_DUMP_EXTRA_BLOCKS_SIZE + \
                                            DRV_DUMP_SPLIT_REGISTERS_SIZE_E2)
#define DRV_DUMP_CRASH_DMP_BUF_SIZE_E3A0   (0x140000 + \
                                            DRV_DUMP_PRELIMINARY_DATA_SIZE + \
                                            DRV_DUMP_EXTRA_BLOCKS_SIZE + \
                                            DRV_DUMP_SPLIT_REGISTERS_SIZE_E3A0)
#define DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0   (0x140000 + \
                                            DRV_DUMP_PRELIMINARY_DATA_SIZE + \
                                            DRV_DUMP_EXTRA_BLOCKS_SIZE + \
                                            DRV_DUMP_SPLIT_REGISTERS_SIZE_E3B0)



static VMK_ReturnStatus
qfle3_mod_info_init(void)
{
   VMK_ReturnStatus status;
   vmk_HeapCreateProps heapProps;
   vmk_HeapAllocationDescriptor allocDesc[] = {
      /*
       * size, alignment, count 
       */
      {QFLE3_HEAP_EST, 1, 1},
      {vmk_LogHeapAllocSize(), 1, 1},
      {vmk_LockDomainAllocSize(), 1, 1},
      {sizeof(qfle3_adapter), 1, QFLE3_MAX_ADAPTERS},
      {(sizeof(qfle3_txbuf_info) * (TX_BD_TOTAL_PER_PAGE * TX_BD_MAX_PAGES)), 1,
       ((QFLE3_DEVICE_MAX_TX_QUEUES + CNIC_MAX_QUEUES) * QFLE3_MAX_ADAPTERS)},
      {(sizeof(qfle3_rxbuf_info) * (RX_BD_TOTAL_PER_PAGE * RX_BD_MAX_PAGES)), 1,
       ((QFLE3_DEVICE_MAX_RX_QUEUES + (QFLE3_DEVICE_MAX_RSS_QUEUES - 1) + CNIC_MAX_QUEUES) * QFLE3_MAX_ADAPTERS)},
      {vmk_SpinlockAllocSize(VMK_SPINLOCK), 1,
       ((2 + QFLE3_DEVICE_MAX_QUEUES) * QFLE3_MAX_ADAPTERS)},
      {(vmk_BitVectorSize(QFLE3_STATE_MAX) +
        vmk_BitVectorSize(QFLE3_DEVICE_MAX_TX_QUEUES + QFLE3_DEVICE_MAX_RX_QUEUES)),
       1, QFLE3_MAX_ADAPTERS},

      {DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0, 1, 2},
   };
   vmk_ByteCount byteCount;
   vmk_LogProperties logProps;
   vmk_LogThrottleProperties logThrottledProps;
   vmk_MemPoolProps memPoolProps;
   vmk_DriverProps driverProps;
   vmk_MgmtProps mgmt_props;
   vmk_TimerQueueProps timerqueue_props;

   vmk_ListInit(&qfle3_mod_info.ecore_lock_list);

   vmk_ListInit(&qfle3_mod_info.prev_list);

   vmk_NameInitialize(&qfle3_mod_info.driverName, QFLE3_DRIVER_NAME);

//	   /* calculate the real_tx_bd_pages and real_rx_bd_pages */
//	   real_tx_bd_pages = 2;
//	   while (real_tx_bd_pages < tx_bd_ring_pages)
//	   	 real_tx_bd_pages << 1;

   /*
    * Heap 
    */
   status = vmk_HeapDetermineMaxSize(allocDesc,
                                     sizeof(allocDesc) / sizeof(allocDesc[0]),
                                     &byteCount);
   if (status != VMK_OK) {
      vmk_WarningMessage("Failed to determine heap max size (%x)", status);
      goto err;
   }
   heapProps.type = VMK_HEAP_TYPE_SIMPLE;
   vmk_NameCopy(&heapProps.name, &qfle3_mod_info.driverName);
   heapProps.module = vmk_ModuleCurrentID;
   heapProps.initial = 0;
   heapProps.max = byteCount;
   heapProps.creationTimeoutMS = VMK_TIMEOUT_UNLIMITED_MS;
   status = vmk_HeapCreate(&heapProps, &qfle3_mod_info.heapID);
   if (status != VMK_OK) {
      vmk_WarningMessage("Failed to create heap (%x)", status);
      goto err;
   }
   /*
    * The module heap is used by some internal vmkernel
    * interfaces to allocate memory on behalf of the module
    */
   vmk_ModuleSetHeapID(vmk_ModuleCurrentID, qfle3_mod_info.heapID);

   /*
    * Log 
    */
   vmk_NameCopy(&logProps.name, &qfle3_mod_info.driverName);
   logProps.module = vmk_ModuleCurrentID;
   logProps.heap = qfle3_mod_info.heapID;
   logProps.defaultLevel = 0;
   logProps.throttle = NULL;
   status = vmk_LogRegister(&logProps, &qfle3_mod_info.logID);
   if (status != VMK_OK) {
      vmk_WarningMessage("Failed to register log component (%x)", status);
      goto err;
   }

   logThrottledProps.type = VMK_LOG_THROTTLE_COUNT;
   logProps.throttle = &logThrottledProps;
   vmk_NameInitialize(&logProps.name, QFLE3_DRIVER_NAME "_throttled");
   status = vmk_LogRegister(&logProps, &qfle3_mod_info.logThrottledID);
   if (status != VMK_OK) {
      vmk_WarningMessage("Failed to register throttled log component (%x)", status);
      goto err;
   }

   /*
    * Mempool 
    */
   vmk_NameCopy(&memPoolProps.name, &qfle3_mod_info.driverName);
   memPoolProps.module = vmk_ModuleCurrentID;
   memPoolProps.parentMemPool = VMK_MEMPOOL_INVALID;
   memPoolProps.memPoolType = VMK_MEM_POOL_LEAF;
   memPoolProps.resourceProps.reservation = 0;
   memPoolProps.resourceProps.limit = VMK_MEMPOOL_NO_LIMIT;
   status = vmk_MemPoolCreate(&memPoolProps, &qfle3_mod_info.memPoolID);
   if (status != VMK_OK) {
      QFLE3_ERROR("Failed to create mempool (%x)", status);
      goto err;
   }

   /*
    * Lock Domain 
    */
   status = vmk_LockDomainCreate(vmk_ModuleCurrentID,
                                 qfle3_mod_info.heapID,
                                 &qfle3_mod_info.driverName, &qfle3_mod_info.lockDomain);
   if (status != VMK_OK) {
      QFLE3_ERROR("Failed to create lock domain (%x)", status);
      goto err;
   }

   /*
    * Create a global driver lock to access global data
    */
   status = qfle3_create_spinlock("drv_lock", NULL,
                                  QFLE3_LOCK_RANK_HIGH, &qfle3_mod_info.drv_lock);
   if (status != VMK_OK) {

      goto err;
   }
#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)
   status = vmk_SemaCreate(&qfle3_mod_info.prev_mutex,
                           qfle3_mod_info.heapID, "qfle3_prev_mutex", 1);
#else
   status = vmk_SemaCreate(&qfle3_mod_info.prev_mutex,
                           vmk_ModuleCurrentID, "qfle3_prev_mutex", 1);
#endif
   if (status != VMK_OK) {

      goto err;
   }
   /*
    * Adapter List 
    */
   vmk_ListInit(&qfle3_mod_info.adapterList);

   /*
    * Register Driver with device layer 
    */
   driverProps.moduleID = vmk_ModuleCurrentID;
   vmk_NameCopy(&driverProps.name, &qfle3_mod_info.driverName);
   driverProps.ops = &qfle3_drv_ops;
   driverProps.privateData = (vmk_AddrCookie) NULL;
   status = vmk_DriverRegister(&driverProps, &qfle3_mod_info.driverID);
   if (status != VMK_OK) {
      QFLE3_ERROR("Failed to register driver (%x)", status);
      goto err;
   }

   if (enable_fw_dump) {

      status = vmk_DumpAddFileCallback(vmk_ModuleCurrentID,
                                    qfle3_mod_info.heapID,
                                    "qfle3_fwdmp",
                                    qfle3_fw_dump_handler,
                                    NULL,
                                    "qfle3_fwdmp",
                                    &qfle3_mod_info.dumpFile);

      if (status != VMK_OK) {
         QFLE3_ERROR("Failed to add dump file callback.");
      }
   }
   /*
   * Register Management Interface
   */

   vmk_Memset(&mgmt_props, 0, sizeof(mgmt_props));
   mgmt_props.modId = vmk_ModuleCurrentID;
   mgmt_props.heapId = qfle3_mod_info.heapID;
   mgmt_props.sig = &qfle3_api_sig;

   status = vmk_MgmtInit(&mgmt_props, &vmkmgmt_api_handler);

   if (status == VMK_OK) {
     vmk_LogMessage("Management interface initialized successfully\n");
   } else {
     vmk_LogMessage("Management interface initalization FAILED\n");
     goto err;
   }
   qfle3_mod_info.reboot_handler = NULL;
//   qfle3_mod_info.reboot_adapter_cnt = 0;

   /*
    * Timer Queue
    */
   vmk_NameInitialize(&timerqueue_props.name, "qfle3_timerq");
   timerqueue_props.moduleID = vmk_ModuleCurrentID;
   timerqueue_props.heapID = qfle3_mod_info.heapID;
   timerqueue_props.attribs = VMK_TIMER_QUEUE_ATTR_NONE;

   status = vmk_TimerQueueCreate(&timerqueue_props,&qfle3_mod_info.timer_queue);
   if (status != VMK_OK) {
      QFLE3_ERROR("Failed to create qfle3 module timer queue.");
   }

  err:
   if (status != VMK_OK) {
      qfle3_mod_info_cleanup();
   }
   return status;
}

int
init_module(void)
{
   return qfle3_mod_info_init();
}

void
cleanup_module(void)
{
   qfle3_mod_info_cleanup();
}
