/* SPDX-License-Identifier: BSD-3-Clause-Clear */
/*
 * Copyright (c) 2018-2021 The Linux Foundation. All rights reserved.
 * Copyright (c) 2021-2022 Qualcomm Innovation Center, Inc. All rights reserved.
 */

#ifndef ATH12K_WMI_H
#define ATH12K_WMI_H

#include <net/mac80211.h>
#include "htc.h"

/* Naming conventions for structures:
 *
 * _cmd means that this is a firmware command sent from host to firmware.
 *
 * _event means that this is a firmware event sent from firmware to host
 *
 * _params is a structure which is embedded either into _cmd or _event (or
 * both), it is not sent individually.
 *
 * _arg is used inside the host, the firmware does not see that at all.
 */

struct ath12k_base;
struct ath12k;

/* There is no signed version of __le32, so for a temporary solution come
 * up with our own version. The idea is from fs/ntfs/endian.h.
 *
 * Use a_ prefix so that it doesn't conflict if we get proper support to
 * linux/types.h.
 */
typedef __s32 __bitwise a_sle32;

static inline a_sle32 a_cpu_to_sle32(s32 val)
{
	return (__force a_sle32)cpu_to_le32(val);
}

static inline s32 a_sle32_to_cpu(a_sle32 val)
{
	return le32_to_cpu((__force __le32)val);
}

/* defines to set Packet extension values which can be 0 us, 8 usec or 16 usec */
#define MAX_HE_NSS               8
#define MAX_HE_MODULATION        8
#define MAX_HE_RU                4
#define HE_MODULATION_NONE       7
#define HE_PET_0_USEC            0
#define HE_PET_8_USEC            1
#define HE_PET_16_USEC           2

#define WMI_MAX_CHAINS		 8

#define WMI_MAX_NUM_SS                    MAX_HE_NSS
#define WMI_MAX_NUM_RU                    MAX_HE_RU

#define WMI_TLV_CMD(grp_id) (((grp_id) << 12) | 0x1)
#define WMI_TLV_EV(grp_id) (((grp_id) << 12) | 0x1)
#define WMI_TLV_CMD_UNSUPPORTED 0
#define WMI_TLV_PDEV_PARAM_UNSUPPORTED 0
#define WMI_TLV_VDEV_PARAM_UNSUPPORTED 0

struct wmi_cmd_hdr {
	__le32 cmd_id;
} __packed;

struct wmi_tlv {
	__le32 header;
	u8 value[];
} __packed;

#define WMI_TLV_LEN	GENMASK(15, 0)
#define WMI_TLV_TAG	GENMASK(31, 16)
#define TLV_HDR_SIZE	sizeof_field(struct wmi_tlv, header)

#define WMI_CMD_HDR_CMD_ID      GENMASK(23, 0)
#define WMI_MAX_MEM_REQS        32
#define ATH12K_MAX_HW_LISTEN_INTERVAL 5

#define WMI_HOST_RC_DS_FLAG			0x01
#define WMI_HOST_RC_CW40_FLAG			0x02
#define WMI_HOST_RC_SGI_FLAG			0x04
#define WMI_HOST_RC_HT_FLAG			0x08
#define WMI_HOST_RC_RTSCTS_FLAG			0x10
#define WMI_HOST_RC_TX_STBC_FLAG		0x20
#define WMI_HOST_RC_RX_STBC_FLAG		0xC0
#define WMI_HOST_RC_RX_STBC_FLAG_S		6
#define WMI_HOST_RC_WEP_TKIP_FLAG		0x100
#define WMI_HOST_RC_TS_FLAG			0x200
#define WMI_HOST_RC_UAPSD_FLAG			0x400

#define WMI_HT_CAP_ENABLED			0x0001
#define WMI_HT_CAP_HT20_SGI			0x0002
#define WMI_HT_CAP_DYNAMIC_SMPS			0x0004
#define WMI_HT_CAP_TX_STBC			0x0008
#define WMI_HT_CAP_TX_STBC_MASK_SHIFT		3
#define WMI_HT_CAP_RX_STBC			0x0030
#define WMI_HT_CAP_RX_STBC_MASK_SHIFT		4
#define WMI_HT_CAP_LDPC				0x0040
#define WMI_HT_CAP_L_SIG_TXOP_PROT		0x0080
#define WMI_HT_CAP_MPDU_DENSITY			0x0700
#define WMI_HT_CAP_MPDU_DENSITY_MASK_SHIFT	8
#define WMI_HT_CAP_HT40_SGI			0x0800
#define WMI_HT_CAP_RX_LDPC			0x1000
#define WMI_HT_CAP_TX_LDPC			0x2000
#define WMI_HT_CAP_IBF_BFER			0x4000

/* These macros should be used when we wish to advertise STBC support for
 * only 1SS or 2SS or 3SS.
 */
#define WMI_HT_CAP_RX_STBC_1SS			0x0010
#define WMI_HT_CAP_RX_STBC_2SS			0x0020
#define WMI_HT_CAP_RX_STBC_3SS			0x0030

#define WMI_HT_CAP_DEFAULT_ALL (WMI_HT_CAP_ENABLED    | \
				WMI_HT_CAP_HT20_SGI   | \
				WMI_HT_CAP_HT40_SGI   | \
				WMI_HT_CAP_TX_STBC    | \
				WMI_HT_CAP_RX_STBC    | \
				WMI_HT_CAP_LDPC)

#define WMI_VHT_CAP_MAX_MPDU_LEN_MASK		0x00000003
#define WMI_VHT_CAP_RX_LDPC			0x00000010
#define WMI_VHT_CAP_SGI_80MHZ			0x00000020
#define WMI_VHT_CAP_SGI_160MHZ			0x00000040
#define WMI_VHT_CAP_TX_STBC			0x00000080
#define WMI_VHT_CAP_RX_STBC_MASK		0x00000300
#define WMI_VHT_CAP_RX_STBC_MASK_SHIFT		8
#define WMI_VHT_CAP_SU_BFER			0x00000800
#define WMI_VHT_CAP_SU_BFEE			0x00001000
#define WMI_VHT_CAP_MAX_CS_ANT_MASK		0x0000E000
#define WMI_VHT_CAP_MAX_CS_ANT_MASK_SHIFT	13
#define WMI_VHT_CAP_MAX_SND_DIM_MASK		0x00070000
#define WMI_VHT_CAP_MAX_SND_DIM_MASK_SHIFT	16
#define WMI_VHT_CAP_MU_BFER			0x00080000
#define WMI_VHT_CAP_MU_BFEE			0x00100000
#define WMI_VHT_CAP_MAX_AMPDU_LEN_EXP		0x03800000
#define WMI_VHT_CAP_MAX_AMPDU_LEN_EXP_SHIT	23
#define WMI_VHT_CAP_RX_FIXED_ANT		0x10000000
#define WMI_VHT_CAP_TX_FIXED_ANT		0x20000000

#define WMI_VHT_CAP_MAX_MPDU_LEN_11454		0x00000002

/* These macros should be used when we wish to advertise STBC support for
 * only 1SS or 2SS or 3SS.
 */
#define WMI_VHT_CAP_RX_STBC_1SS			0x00000100
#define WMI_VHT_CAP_RX_STBC_2SS			0x00000200
#define WMI_VHT_CAP_RX_STBC_3SS			0x00000300

#define WMI_VHT_CAP_DEFAULT_ALL (WMI_VHT_CAP_MAX_MPDU_LEN_11454  | \
				 WMI_VHT_CAP_SGI_80MHZ      |       \
				 WMI_VHT_CAP_TX_STBC        |       \
				 WMI_VHT_CAP_RX_STBC_MASK   |       \
				 WMI_VHT_CAP_RX_LDPC        |       \
				 WMI_VHT_CAP_MAX_AMPDU_LEN_EXP   |  \
				 WMI_VHT_CAP_RX_FIXED_ANT   |       \
				 WMI_VHT_CAP_TX_FIXED_ANT)

#define WLAN_SCAN_MAX_HINT_S_SSID        10
#define WLAN_SCAN_MAX_HINT_BSSID         10
#define MAX_RNR_BSS                    5

#define WLAN_SCAN_MAX_HINT_S_SSID        10
#define WLAN_SCAN_MAX_HINT_BSSID         10
#define MAX_RNR_BSS                    5

#define WLAN_SCAN_PARAMS_MAX_SSID    16
#define WLAN_SCAN_PARAMS_MAX_BSSID   4
#define WLAN_SCAN_PARAMS_MAX_IE_LEN  256

#define WMI_APPEND_TO_EXISTING_CHAN_LIST_FLAG 1

#define WMI_BA_MODE_BUFFER_SIZE_256  3

/* HW mode config type replicated from FW header
 * @WMI_HOST_HW_MODE_SINGLE: Only one PHY is active.
 * @WMI_HOST_HW_MODE_DBS: Both PHYs are active in different bands,
 *                        one in 2G and another in 5G.
 * @WMI_HOST_HW_MODE_SBS_PASSIVE: Both PHYs are in passive mode (only rx) in
 *                        same band; no tx allowed.
 * @WMI_HOST_HW_MODE_SBS: Both PHYs are active in the same band.
 *                        Support for both PHYs within one band is planned
 *                        for 5G only(as indicated in WMI_MAC_PHY_CAPABILITIES),
 *                        but could be extended to other bands in the future.
 *                        The separation of the band between the two PHYs needs
 *                        to be communicated separately.
 * @WMI_HOST_HW_MODE_DBS_SBS: 3 PHYs, with 2 on the same band doing SBS
 *                           as in WMI_HW_MODE_SBS, and 3rd on the other band
 * @WMI_HOST_HW_MODE_DBS_OR_SBS: Two PHY with one PHY capabale of both 2G and
 *                        5G. It can support SBS (5G + 5G) OR DBS (5G + 2G).
 * @WMI_HOST_HW_MODE_MAX: Max hw_mode_id. Used to indicate invalid mode.
 */
enum wmi_host_hw_mode_config_type {
	WMI_HOST_HW_MODE_SINGLE       = 0,
	WMI_HOST_HW_MODE_DBS          = 1,
	WMI_HOST_HW_MODE_SBS_PASSIVE  = 2,
	WMI_HOST_HW_MODE_SBS          = 3,
	WMI_HOST_HW_MODE_DBS_SBS      = 4,
	WMI_HOST_HW_MODE_DBS_OR_SBS   = 5,

	/* keep last */
	WMI_HOST_HW_MODE_MAX
};

/* HW mode priority values used to detect the preferred HW mode
 * on the available modes.
 */
enum wmi_host_hw_mode_priority {
	WMI_HOST_HW_MODE_DBS_SBS_PRI,
	WMI_HOST_HW_MODE_DBS_PRI,
	WMI_HOST_HW_MODE_DBS_OR_SBS_PRI,
	WMI_HOST_HW_MODE_SBS_PRI,
	WMI_HOST_HW_MODE_SBS_PASSIVE_PRI,
	WMI_HOST_HW_MODE_SINGLE_PRI,

	/* keep last the lowest priority */
	WMI_HOST_HW_MODE_MAX_PRI
};

enum WMI_HOST_WLAN_BAND {
	WMI_HOST_WLAN_2G_CAP	= 1,
	WMI_HOST_WLAN_5G_CAP	= 2,
	WMI_HOST_WLAN_2G_5G_CAP	= 3,
};

enum wmi_cmd_group {
	/* 0 to 2 are reserved */
	WMI_GRP_START = 0x3,
	WMI_GRP_SCAN = WMI_GRP_START,
	WMI_GRP_PDEV		= 0x4,
	WMI_GRP_VDEV           = 0x5,
	WMI_GRP_PEER           = 0x6,
	WMI_GRP_MGMT           = 0x7,
	WMI_GRP_BA_NEG         = 0x8,
	WMI_GRP_STA_PS         = 0x9,
	WMI_GRP_DFS            = 0xa,
	WMI_GRP_ROAM           = 0xb,
	WMI_GRP_OFL_SCAN       = 0xc,
	WMI_GRP_P2P            = 0xd,
	WMI_GRP_AP_PS          = 0xe,
	WMI_GRP_RATE_CTRL      = 0xf,
	WMI_GRP_PROFILE        = 0x10,
	WMI_GRP_SUSPEND        = 0x11,
	WMI_GRP_BCN_FILTER     = 0x12,
	WMI_GRP_WOW            = 0x13,
	WMI_GRP_RTT            = 0x14,
	WMI_GRP_SPECTRAL       = 0x15,
	WMI_GRP_STATS          = 0x16,
	WMI_GRP_ARP_NS_OFL     = 0x17,
	WMI_GRP_NLO_OFL        = 0x18,
	WMI_GRP_GTK_OFL        = 0x19,
	WMI_GRP_CSA_OFL        = 0x1a,
	WMI_GRP_CHATTER        = 0x1b,
	WMI_GRP_TID_ADDBA      = 0x1c,
	WMI_GRP_MISC           = 0x1d,
	WMI_GRP_GPIO           = 0x1e,
	WMI_GRP_FWTEST         = 0x1f,
	WMI_GRP_TDLS           = 0x20,
	WMI_GRP_RESMGR         = 0x21,
	WMI_GRP_STA_SMPS       = 0x22,
	WMI_GRP_WLAN_HB        = 0x23,
	WMI_GRP_RMC            = 0x24,
	WMI_GRP_MHF_OFL        = 0x25,
	WMI_GRP_LOCATION_SCAN  = 0x26,
	WMI_GRP_OEM            = 0x27,
	WMI_GRP_NAN            = 0x28,
	WMI_GRP_COEX           = 0x29,
	WMI_GRP_OBSS_OFL       = 0x2a,
	WMI_GRP_LPI            = 0x2b,
	WMI_GRP_EXTSCAN        = 0x2c,
	WMI_GRP_DHCP_OFL       = 0x2d,
	WMI_GRP_IPA            = 0x2e,
	WMI_GRP_MDNS_OFL       = 0x2f,
	WMI_GRP_SAP_OFL        = 0x30,
	WMI_GRP_OCB            = 0x31,
	WMI_GRP_SOC            = 0x32,
	WMI_GRP_PKT_FILTER     = 0x33,
	WMI_GRP_MAWC           = 0x34,
	WMI_GRP_PMF_OFFLOAD    = 0x35,
	WMI_GRP_BPF_OFFLOAD    = 0x36,
	WMI_GRP_NAN_DATA       = 0x37,
	WMI_GRP_PROTOTYPE      = 0x38,
	WMI_GRP_MONITOR        = 0x39,
	WMI_GRP_REGULATORY     = 0x3a,
	WMI_GRP_HW_DATA_FILTER = 0x3b,
	WMI_GRP_WLM            = 0x3c,
	WMI_GRP_11K_OFFLOAD    = 0x3d,
	WMI_GRP_TWT            = 0x3e,
	WMI_GRP_MOTION_DET     = 0x3f,
	WMI_GRP_SPATIAL_REUSE  = 0x40,
};

#define WMI_CMD_GRP(grp_id) (((grp_id) << 12) | 0x1)
#define WMI_EVT_GRP_START_ID(grp_id) (((grp_id) << 12) | 0x1)

enum wmi_tlv_cmd_id {
	WMI_CMD_UNSUPPORTED = 0,
	WMI_INIT_CMDID = 0x1,
	WMI_START_SCAN_CMDID = WMI_TLV_CMD(WMI_GRP_SCAN),
	WMI_STOP_SCAN_CMDID,
	WMI_SCAN_CHAN_LIST_CMDID,
	WMI_SCAN_SCH_PRIO_TBL_CMDID,
	WMI_SCAN_UPDATE_REQUEST_CMDID,
	WMI_SCAN_PROB_REQ_OUI_CMDID,
	WMI_SCAN_ADAPTIVE_DWELL_CONFIG_CMDID,
	WMI_PDEV_SET_REGDOMAIN_CMDID = WMI_TLV_CMD(WMI_GRP_PDEV),
	WMI_PDEV_SET_CHANNEL_CMDID,
	WMI_PDEV_SET_PARAM_CMDID,
	WMI_PDEV_PKTLOG_ENABLE_CMDID,
	WMI_PDEV_PKTLOG_DISABLE_CMDID,
	WMI_PDEV_SET_WMM_PARAMS_CMDID,
	WMI_PDEV_SET_HT_CAP_IE_CMDID,
	WMI_PDEV_SET_VHT_CAP_IE_CMDID,
	WMI_PDEV_SET_DSCP_TID_MAP_CMDID,
	WMI_PDEV_SET_QUIET_MODE_CMDID,
	WMI_PDEV_GREEN_AP_PS_ENABLE_CMDID,
	WMI_PDEV_GET_TPC_CONFIG_CMDID,
	WMI_PDEV_SET_BASE_MACADDR_CMDID,
	WMI_PDEV_DUMP_CMDID,
	WMI_PDEV_SET_LED_CONFIG_CMDID,
	WMI_PDEV_GET_TEMPERATURE_CMDID,
	WMI_PDEV_SET_LED_FLASHING_CMDID,
	WMI_PDEV_SMART_ANT_ENABLE_CMDID,
	WMI_PDEV_SMART_ANT_SET_RX_ANTENNA_CMDID,
	WMI_PDEV_SET_ANTENNA_SWITCH_TABLE_CMDID,
	WMI_PDEV_SET_CTL_TABLE_CMDID,
	WMI_PDEV_SET_MIMOGAIN_TABLE_CMDID,
	WMI_PDEV_FIPS_CMDID,
	WMI_PDEV_GET_ANI_CCK_CONFIG_CMDID,
	WMI_PDEV_GET_ANI_OFDM_CONFIG_CMDID,
	WMI_PDEV_GET_NFCAL_POWER_CMDID,
	WMI_PDEV_GET_TPC_CMDID,
	WMI_MIB_STATS_ENABLE_CMDID,
	WMI_PDEV_SET_PCL_CMDID,
	WMI_PDEV_SET_HW_MODE_CMDID,
	WMI_PDEV_SET_MAC_CONFIG_CMDID,
	WMI_PDEV_SET_ANTENNA_MODE_CMDID,
	WMI_SET_PERIODIC_CHANNEL_STATS_CONFIG_CMDID,
	WMI_PDEV_WAL_POWER_DEBUG_CMDID,
	WMI_PDEV_SET_REORDER_TIMEOUT_VAL_CMDID,
	WMI_PDEV_SET_WAKEUP_CONFIG_CMDID,
	WMI_PDEV_GET_ANTDIV_STATUS_CMDID,
	WMI_PDEV_GET_CHIP_POWER_STATS_CMDID,
	WMI_PDEV_SET_STATS_THRESHOLD_CMDID,
	WMI_PDEV_MULTIPLE_VDEV_RESTART_REQUEST_CMDID,
	WMI_PDEV_UPDATE_PKT_ROUTING_CMDID,
	WMI_PDEV_CHECK_CAL_VERSION_CMDID,
	WMI_PDEV_SET_DIVERSITY_GAIN_CMDID,
	WMI_PDEV_DIV_GET_RSSI_ANTID_CMDID,
	WMI_PDEV_BSS_CHAN_INFO_REQUEST_CMDID,
	WMI_PDEV_UPDATE_PMK_CACHE_CMDID,
	WMI_PDEV_UPDATE_FILS_HLP_PKT_CMDID,
	WMI_PDEV_UPDATE_CTLTABLE_REQUEST_CMDID,
	WMI_PDEV_CONFIG_VENDOR_OUI_ACTION_CMDID,
	WMI_PDEV_SET_AC_TX_QUEUE_OPTIMIZED_CMDID,
	WMI_PDEV_SET_RX_FILTER_PROMISCUOUS_CMDID,
	WMI_PDEV_DMA_RING_CFG_REQ_CMDID,
	WMI_PDEV_HE_TB_ACTION_FRM_CMDID,
	WMI_PDEV_PKTLOG_FILTER_CMDID,
	WMI_VDEV_CREATE_CMDID = WMI_TLV_CMD(WMI_GRP_VDEV),
	WMI_VDEV_DELETE_CMDID,
	WMI_VDEV_START_REQUEST_CMDID,
	WMI_VDEV_RESTART_REQUEST_CMDID,
	WMI_VDEV_UP_CMDID,
	WMI_VDEV_STOP_CMDID,
	WMI_VDEV_DOWN_CMDID,
	WMI_VDEV_SET_PARAM_CMDID,
	WMI_VDEV_INSTALL_KEY_CMDID,
	WMI_VDEV_WNM_SLEEPMODE_CMDID,
	WMI_VDEV_WMM_ADDTS_CMDID,
	WMI_VDEV_WMM_DELTS_CMDID,
	WMI_VDEV_SET_WMM_PARAMS_CMDID,
	WMI_VDEV_SET_GTX_PARAMS_CMDID,
	WMI_VDEV_IPSEC_NATKEEPALIVE_FILTER_CMDID,
	WMI_VDEV_PLMREQ_START_CMDID,
	WMI_VDEV_PLMREQ_STOP_CMDID,
	WMI_VDEV_TSF_TSTAMP_ACTION_CMDID,
	WMI_VDEV_SET_IE_CMDID,
	WMI_VDEV_RATEMASK_CMDID,
	WMI_VDEV_ATF_REQUEST_CMDID,
	WMI_VDEV_SET_DSCP_TID_MAP_CMDID,
	WMI_VDEV_FILTER_NEIGHBOR_RX_PACKETS_CMDID,
	WMI_VDEV_SET_QUIET_MODE_CMDID,
	WMI_VDEV_SET_CUSTOM_AGGR_SIZE_CMDID,
	WMI_VDEV_ENCRYPT_DECRYPT_DATA_REQ_CMDID,
	WMI_VDEV_ADD_MAC_ADDR_TO_RX_FILTER_CMDID,
	WMI_PEER_CREATE_CMDID = WMI_TLV_CMD(WMI_GRP_PEER),
	WMI_PEER_DELETE_CMDID,
	WMI_PEER_FLUSH_TIDS_CMDID,
	WMI_PEER_SET_PARAM_CMDID,
	WMI_PEER_ASSOC_CMDID,
	WMI_PEER_ADD_WDS_ENTRY_CMDID,
	WMI_PEER_REMOVE_WDS_ENTRY_CMDID,
	WMI_PEER_MCAST_GROUP_CMDID,
	WMI_PEER_INFO_REQ_CMDID,
	WMI_PEER_GET_ESTIMATED_LINKSPEED_CMDID,
	WMI_PEER_SET_RATE_REPORT_CONDITION_CMDID,
	WMI_PEER_UPDATE_WDS_ENTRY_CMDID,
	WMI_PEER_ADD_PROXY_STA_ENTRY_CMDID,
	WMI_PEER_SMART_ANT_SET_TX_ANTENNA_CMDID,
	WMI_PEER_SMART_ANT_SET_TRAIN_INFO_CMDID,
	WMI_PEER_SMART_ANT_SET_NODE_CONFIG_OPS_CMDID,
	WMI_PEER_ATF_REQUEST_CMDID,
	WMI_PEER_BWF_REQUEST_CMDID,
	WMI_PEER_REORDER_QUEUE_SETUP_CMDID,
	WMI_PEER_REORDER_QUEUE_REMOVE_CMDID,
	WMI_PEER_SET_RX_BLOCKSIZE_CMDID,
	WMI_PEER_ANTDIV_INFO_REQ_CMDID,
	WMI_BCN_TX_CMDID = WMI_TLV_CMD(WMI_GRP_MGMT),
	WMI_PDEV_SEND_BCN_CMDID,
	WMI_BCN_TMPL_CMDID,
	WMI_BCN_FILTER_RX_CMDID,
	WMI_PRB_REQ_FILTER_RX_CMDID,
	WMI_MGMT_TX_CMDID,
	WMI_PRB_TMPL_CMDID,
	WMI_MGMT_TX_SEND_CMDID,
	WMI_OFFCHAN_DATA_TX_SEND_CMDID,
	WMI_PDEV_SEND_FD_CMDID,
	WMI_BCN_OFFLOAD_CTRL_CMDID,
	WMI_BSS_COLOR_CHANGE_ENABLE_CMDID,
	WMI_VDEV_BCN_OFFLOAD_QUIET_CONFIG_CMDID,
	WMI_FILS_DISCOVERY_TMPL_CMDID,
	WMI_ADDBA_CLEAR_RESP_CMDID = WMI_TLV_CMD(WMI_GRP_BA_NEG),
	WMI_ADDBA_SEND_CMDID,
	WMI_ADDBA_STATUS_CMDID,
	WMI_DELBA_SEND_CMDID,
	WMI_ADDBA_SET_RESP_CMDID,
	WMI_SEND_SINGLEAMSDU_CMDID,
	WMI_STA_POWERSAVE_MODE_CMDID = WMI_TLV_CMD(WMI_GRP_STA_PS),
	WMI_STA_POWERSAVE_PARAM_CMDID,
	WMI_STA_MIMO_PS_MODE_CMDID,
	WMI_PDEV_DFS_ENABLE_CMDID = WMI_TLV_CMD(WMI_GRP_DFS),
	WMI_PDEV_DFS_DISABLE_CMDID,
	WMI_DFS_PHYERR_FILTER_ENA_CMDID,
	WMI_DFS_PHYERR_FILTER_DIS_CMDID,
	WMI_PDEV_DFS_PHYERR_OFFLOAD_ENABLE_CMDID,
	WMI_PDEV_DFS_PHYERR_OFFLOAD_DISABLE_CMDID,
	WMI_VDEV_ADFS_CH_CFG_CMDID,
	WMI_VDEV_ADFS_OCAC_ABORT_CMDID,
	WMI_ROAM_SCAN_MODE = WMI_TLV_CMD(WMI_GRP_ROAM),
	WMI_ROAM_SCAN_RSSI_THRESHOLD,
	WMI_ROAM_SCAN_PERIOD,
	WMI_ROAM_SCAN_RSSI_CHANGE_THRESHOLD,
	WMI_ROAM_AP_PROFILE,
	WMI_ROAM_CHAN_LIST,
	WMI_ROAM_SCAN_CMD,
	WMI_ROAM_SYNCH_COMPLETE,
	WMI_ROAM_SET_RIC_REQUEST_CMDID,
	WMI_ROAM_INVOKE_CMDID,
	WMI_ROAM_FILTER_CMDID,
	WMI_ROAM_SUBNET_CHANGE_CONFIG_CMDID,
	WMI_ROAM_CONFIGURE_MAWC_CMDID,
	WMI_ROAM_SET_MBO_PARAM_CMDID,
	WMI_ROAM_PER_CONFIG_CMDID,
	WMI_ROAM_BTM_CONFIG_CMDID,
	WMI_ENABLE_FILS_CMDID,
	WMI_OFL_SCAN_ADD_AP_PROFILE = WMI_TLV_CMD(WMI_GRP_OFL_SCAN),
	WMI_OFL_SCAN_REMOVE_AP_PROFILE,
	WMI_OFL_SCAN_PERIOD,
	WMI_P2P_DEV_SET_DEVICE_INFO = WMI_TLV_CMD(WMI_GRP_P2P),
	WMI_P2P_DEV_SET_DISCOVERABILITY,
	WMI_P2P_GO_SET_BEACON_IE,
	WMI_P2P_GO_SET_PROBE_RESP_IE,
	WMI_P2P_SET_VENDOR_IE_DATA_CMDID,
	WMI_P2P_DISC_OFFLOAD_CONFIG_CMDID,
	WMI_P2P_DISC_OFFLOAD_APPIE_CMDID,
	WMI_P2P_DISC_OFFLOAD_PATTERN_CMDID,
	WMI_P2P_SET_OPPPS_PARAM_CMDID,
	WMI_P2P_LISTEN_OFFLOAD_START_CMDID,
	WMI_P2P_LISTEN_OFFLOAD_STOP_CMDID,
	WMI_AP_PS_PEER_PARAM_CMDID = WMI_TLV_CMD(WMI_GRP_AP_PS),
	WMI_AP_PS_PEER_UAPSD_COEX_CMDID,
	WMI_AP_PS_EGAP_PARAM_CMDID,
	WMI_PEER_RATE_RETRY_SCHED_CMDID = WMI_TLV_CMD(WMI_GRP_RATE_CTRL),
	WMI_WLAN_PROFILE_TRIGGER_CMDID = WMI_TLV_CMD(WMI_GRP_PROFILE),
	WMI_WLAN_PROFILE_SET_HIST_INTVL_CMDID,
	WMI_WLAN_PROFILE_GET_PROFILE_DATA_CMDID,
	WMI_WLAN_PROFILE_ENABLE_PROFILE_ID_CMDID,
	WMI_WLAN_PROFILE_LIST_PROFILE_ID_CMDID,
	WMI_PDEV_SUSPEND_CMDID = WMI_TLV_CMD(WMI_GRP_SUSPEND),
	WMI_PDEV_RESUME_CMDID,
	WMI_ADD_BCN_FILTER_CMDID = WMI_TLV_CMD(WMI_GRP_BCN_FILTER),
	WMI_RMV_BCN_FILTER_CMDID,
	WMI_WOW_ADD_WAKE_PATTERN_CMDID = WMI_TLV_CMD(WMI_GRP_WOW),
	WMI_WOW_DEL_WAKE_PATTERN_CMDID,
	WMI_WOW_ENABLE_DISABLE_WAKE_EVENT_CMDID,
	WMI_WOW_ENABLE_CMDID,
	WMI_WOW_HOSTWAKEUP_FROM_SLEEP_CMDID,
	WMI_WOW_IOAC_ADD_KEEPALIVE_CMDID,
	WMI_WOW_IOAC_DEL_KEEPALIVE_CMDID,
	WMI_WOW_IOAC_ADD_WAKE_PATTERN_CMDID,
	WMI_WOW_IOAC_DEL_WAKE_PATTERN_CMDID,
	WMI_D0_WOW_ENABLE_DISABLE_CMDID,
	WMI_EXTWOW_ENABLE_CMDID,
	WMI_EXTWOW_SET_APP_TYPE1_PARAMS_CMDID,
	WMI_EXTWOW_SET_APP_TYPE2_PARAMS_CMDID,
	WMI_WOW_ENABLE_ICMPV6_NA_FLT_CMDID,
	WMI_WOW_UDP_SVC_OFLD_CMDID,
	WMI_WOW_HOSTWAKEUP_GPIO_PIN_PATTERN_CONFIG_CMDID,
	WMI_WOW_SET_ACTION_WAKE_UP_CMDID,
	WMI_RTT_MEASREQ_CMDID = WMI_TLV_CMD(WMI_GRP_RTT),
	WMI_RTT_TSF_CMDID,
	WMI_VDEV_SPECTRAL_SCAN_CONFIGURE_CMDID = WMI_TLV_CMD(WMI_GRP_SPECTRAL),
	WMI_VDEV_SPECTRAL_SCAN_ENABLE_CMDID,
	WMI_REQUEST_STATS_CMDID = WMI_TLV_CMD(WMI_GRP_STATS),
	WMI_MCC_SCHED_TRAFFIC_STATS_CMDID,
	WMI_REQUEST_STATS_EXT_CMDID,
	WMI_REQUEST_LINK_STATS_CMDID,
	WMI_START_LINK_STATS_CMDID,
	WMI_CLEAR_LINK_STATS_CMDID,
	WMI_GET_FW_MEM_DUMP_CMDID,
	WMI_DEBUG_MESG_FLUSH_CMDID,
	WMI_DIAG_EVENT_LOG_CONFIG_CMDID,
	WMI_REQUEST_WLAN_STATS_CMDID,
	WMI_REQUEST_RCPI_CMDID,
	WMI_REQUEST_PEER_STATS_INFO_CMDID,
	WMI_REQUEST_RADIO_CHAN_STATS_CMDID,
	WMI_SET_ARP_NS_OFFLOAD_CMDID = WMI_TLV_CMD(WMI_GRP_ARP_NS_OFL),
	WMI_ADD_PROACTIVE_ARP_RSP_PATTERN_CMDID,
	WMI_DEL_PROACTIVE_ARP_RSP_PATTERN_CMDID,
	WMI_NETWORK_LIST_OFFLOAD_CONFIG_CMDID = WMI_TLV_CMD(WMI_GRP_NLO_OFL),
	WMI_APFIND_CMDID,
	WMI_PASSPOINT_LIST_CONFIG_CMDID,
	WMI_NLO_CONFIGURE_MAWC_CMDID,
	WMI_GTK_OFFLOAD_CMDID = WMI_TLV_CMD(WMI_GRP_GTK_OFL),
	WMI_CSA_OFFLOAD_ENABLE_CMDID = WMI_TLV_CMD(WMI_GRP_CSA_OFL),
	WMI_CSA_OFFLOAD_CHANSWITCH_CMDID,
	WMI_CHATTER_SET_MODE_CMDID = WMI_TLV_CMD(WMI_GRP_CHATTER),
	WMI_CHATTER_ADD_COALESCING_FILTER_CMDID,
	WMI_CHATTER_DELETE_COALESCING_FILTER_CMDID,
	WMI_CHATTER_COALESCING_QUERY_CMDID,
	WMI_PEER_TID_ADDBA_CMDID = WMI_TLV_CMD(WMI_GRP_TID_ADDBA),
	WMI_PEER_TID_DELBA_CMDID,
	WMI_STA_DTIM_PS_METHOD_CMDID,
	WMI_STA_UAPSD_AUTO_TRIG_CMDID,
	WMI_STA_KEEPALIVE_CMDID,
	WMI_BA_REQ_SSN_CMDID,
	WMI_ECHO_CMDID = WMI_TLV_CMD(WMI_GRP_MISC),
	WMI_PDEV_UTF_CMDID,
	WMI_DBGLOG_CFG_CMDID,
	WMI_PDEV_QVIT_CMDID,
	WMI_PDEV_FTM_INTG_CMDID,
	WMI_VDEV_SET_KEEPALIVE_CMDID,
	WMI_VDEV_GET_KEEPALIVE_CMDID,
	WMI_FORCE_FW_HANG_CMDID,
	WMI_SET_MCASTBCAST_FILTER_CMDID,
	WMI_THERMAL_MGMT_CMDID,
	WMI_HOST_AUTO_SHUTDOWN_CFG_CMDID,
	WMI_TPC_CHAINMASK_CONFIG_CMDID,
	WMI_SET_ANTENNA_DIVERSITY_CMDID,
	WMI_OCB_SET_SCHED_CMDID,
	WMI_RSSI_BREACH_MONITOR_CONFIG_CMDID,
	WMI_LRO_CONFIG_CMDID,
	WMI_TRANSFER_DATA_TO_FLASH_CMDID,
	WMI_CONFIG_ENHANCED_MCAST_FILTER_CMDID,
	WMI_VDEV_WISA_CMDID,
	WMI_DBGLOG_TIME_STAMP_SYNC_CMDID,
	WMI_SET_MULTIPLE_MCAST_FILTER_CMDID,
	WMI_READ_DATA_FROM_FLASH_CMDID,
	WMI_THERM_THROT_SET_CONF_CMDID,
	WMI_RUNTIME_DPD_RECAL_CMDID,
	WMI_GET_TPC_POWER_CMDID,
	WMI_IDLE_TRIGGER_MONITOR_CMDID,
	WMI_GPIO_CONFIG_CMDID = WMI_TLV_CMD(WMI_GRP_GPIO),
	WMI_GPIO_OUTPUT_CMDID,
	WMI_TXBF_CMDID,
	WMI_FWTEST_VDEV_MCC_SET_TBTT_MODE_CMDID = WMI_TLV_CMD(WMI_GRP_FWTEST),
	WMI_FWTEST_P2P_SET_NOA_PARAM_CMDID,
	WMI_UNIT_TEST_CMDID,
	WMI_FWTEST_CMDID,
	WMI_QBOOST_CFG_CMDID,
	WMI_TDLS_SET_STATE_CMDID = WMI_TLV_CMD(WMI_GRP_TDLS),
	WMI_TDLS_PEER_UPDATE_CMDID,
	WMI_TDLS_SET_OFFCHAN_MODE_CMDID,
	WMI_RESMGR_ADAPTIVE_OCS_EN_DIS_CMDID = WMI_TLV_CMD(WMI_GRP_RESMGR),
	WMI_RESMGR_SET_CHAN_TIME_QUOTA_CMDID,
	WMI_RESMGR_SET_CHAN_LATENCY_CMDID,
	WMI_STA_SMPS_FORCE_MODE_CMDID = WMI_TLV_CMD(WMI_GRP_STA_SMPS),
	WMI_STA_SMPS_PARAM_CMDID,
	WMI_HB_SET_ENABLE_CMDID = WMI_TLV_CMD(WMI_GRP_WLAN_HB),
	WMI_HB_SET_TCP_PARAMS_CMDID,
	WMI_HB_SET_TCP_PKT_FILTER_CMDID,
	WMI_HB_SET_UDP_PARAMS_CMDID,
	WMI_HB_SET_UDP_PKT_FILTER_CMDID,
	WMI_RMC_SET_MODE_CMDID = WMI_TLV_CMD(WMI_GRP_RMC),
	WMI_RMC_SET_ACTION_PERIOD_CMDID,
	WMI_RMC_CONFIG_CMDID,
	WMI_RMC_SET_MANUAL_LEADER_CMDID,
	WMI_MHF_OFFLOAD_SET_MODE_CMDID = WMI_TLV_CMD(WMI_GRP_MHF_OFL),
	WMI_MHF_OFFLOAD_PLUMB_ROUTING_TBL_CMDID,
	WMI_BATCH_SCAN_ENABLE_CMDID = WMI_TLV_CMD(WMI_GRP_LOCATION_SCAN),
	WMI_BATCH_SCAN_DISABLE_CMDID,
	WMI_BATCH_SCAN_TRIGGER_RESULT_CMDID,
	WMI_OEM_REQ_CMDID = WMI_TLV_CMD(WMI_GRP_OEM),
	WMI_OEM_REQUEST_CMDID,
	WMI_LPI_OEM_REQ_CMDID,
	WMI_NAN_CMDID = WMI_TLV_CMD(WMI_GRP_NAN),
	WMI_MODEM_POWER_STATE_CMDID = WMI_TLV_CMD(WMI_GRP_COEX),
	WMI_CHAN_AVOID_UPDATE_CMDID,
	WMI_COEX_CONFIG_CMDID,
	WMI_CHAN_AVOID_RPT_ALLOW_CMDID,
	WMI_COEX_GET_ANTENNA_ISOLATION_CMDID,
	WMI_SAR_LIMITS_CMDID,
	WMI_OBSS_SCAN_ENABLE_CMDID = WMI_TLV_CMD(WMI_GRP_OBSS_OFL),
	WMI_OBSS_SCAN_DISABLE_CMDID,
	WMI_OBSS_COLOR_COLLISION_DET_CONFIG_CMDID,
	WMI_LPI_MGMT_SNOOPING_CONFIG_CMDID = WMI_TLV_CMD(WMI_GRP_LPI),
	WMI_LPI_START_SCAN_CMDID,
	WMI_LPI_STOP_SCAN_CMDID,
	WMI_EXTSCAN_START_CMDID = WMI_TLV_CMD(WMI_GRP_EXTSCAN),
	WMI_EXTSCAN_STOP_CMDID,
	WMI_EXTSCAN_CONFIGURE_WLAN_CHANGE_MONITOR_CMDID,
	WMI_EXTSCAN_CONFIGURE_HOTLIST_MONITOR_CMDID,
	WMI_EXTSCAN_GET_CACHED_RESULTS_CMDID,
	WMI_EXTSCAN_GET_WLAN_CHANGE_RESULTS_CMDID,
	WMI_EXTSCAN_SET_CAPABILITIES_CMDID,
	WMI_EXTSCAN_GET_CAPABILITIES_CMDID,
	WMI_EXTSCAN_CONFIGURE_HOTLIST_SSID_MONITOR_CMDID,
	WMI_EXTSCAN_CONFIGURE_MAWC_CMDID,
	WMI_SET_DHCP_SERVER_OFFLOAD_CMDID = WMI_TLV_CMD(WMI_GRP_DHCP_OFL),
	WMI_IPA_OFFLOAD_ENABLE_DISABLE_CMDID = WMI_TLV_CMD(WMI_GRP_IPA),
	WMI_MDNS_OFFLOAD_ENABLE_CMDID = WMI_TLV_CMD(WMI_GRP_MDNS_OFL),
	WMI_MDNS_SET_FQDN_CMDID,
	WMI_MDNS_SET_RESPONSE_CMDID,
	WMI_MDNS_GET_STATS_CMDID,
	WMI_SAP_OFL_ENABLE_CMDID = WMI_TLV_CMD(WMI_GRP_SAP_OFL),
	WMI_SAP_SET_BLACKLIST_PARAM_CMDID,
	WMI_OCB_SET_CONFIG_CMDID = WMI_TLV_CMD(WMI_GRP_OCB),
	WMI_OCB_SET_UTC_TIME_CMDID,
	WMI_OCB_START_TIMING_ADVERT_CMDID,
	WMI_OCB_STOP_TIMING_ADVERT_CMDID,
	WMI_OCB_GET_TSF_TIMER_CMDID,
	WMI_DCC_GET_STATS_CMDID,
	WMI_DCC_CLEAR_STATS_CMDID,
	WMI_DCC_UPDATE_NDL_CMDID,
	WMI_SOC_SET_PCL_CMDID = WMI_TLV_CMD(WMI_GRP_SOC),
	WMI_SOC_SET_HW_MODE_CMDID,
	WMI_SOC_SET_DUAL_MAC_CONFIG_CMDID,
	WMI_SOC_SET_ANTENNA_MODE_CMDID,
	WMI_PACKET_FILTER_CONFIG_CMDID = WMI_TLV_CMD(WMI_GRP_PKT_FILTER),
	WMI_PACKET_FILTER_ENABLE_CMDID,
	WMI_MAWC_SENSOR_REPORT_IND_CMDID = WMI_TLV_CMD(WMI_GRP_MAWC),
	WMI_PMF_OFFLOAD_SET_SA_QUERY_CMDID = WMI_TLV_CMD(WMI_GRP_PMF_OFFLOAD),
	WMI_BPF_GET_CAPABILITY_CMDID = WMI_TLV_CMD(WMI_GRP_BPF_OFFLOAD),
	WMI_BPF_GET_VDEV_STATS_CMDID,
	WMI_BPF_SET_VDEV_INSTRUCTIONS_CMDID,
	WMI_BPF_DEL_VDEV_INSTRUCTIONS_CMDID,
	WMI_BPF_SET_VDEV_ACTIVE_MODE_CMDID,
	WMI_MNT_FILTER_CMDID = WMI_TLV_CMD(WMI_GRP_MONITOR),
	WMI_SET_CURRENT_COUNTRY_CMDID = WMI_TLV_CMD(WMI_GRP_REGULATORY),
	WMI_11D_SCAN_START_CMDID,
	WMI_11D_SCAN_STOP_CMDID,
	WMI_SET_INIT_COUNTRY_CMDID,
	WMI_NDI_GET_CAP_REQ_CMDID = WMI_TLV_CMD(WMI_GRP_PROTOTYPE),
	WMI_NDP_INITIATOR_REQ_CMDID,
	WMI_NDP_RESPONDER_REQ_CMDID,
	WMI_NDP_END_REQ_CMDID,
	WMI_HW_DATA_FILTER_CMDID = WMI_TLV_CMD(WMI_GRP_HW_DATA_FILTER),
	WMI_TWT_ENABLE_CMDID = WMI_TLV_CMD(WMI_GRP_TWT),
	WMI_TWT_DISABLE_CMDID,
	WMI_TWT_ADD_DIALOG_CMDID,
	WMI_TWT_DEL_DIALOG_CMDID,
	WMI_TWT_PAUSE_DIALOG_CMDID,
	WMI_TWT_RESUME_DIALOG_CMDID,
	WMI_PDEV_OBSS_PD_SPATIAL_REUSE_CMDID =
				WMI_TLV_CMD(WMI_GRP_SPATIAL_REUSE),
	WMI_PDEV_OBSS_PD_SPATIAL_REUSE_SET_DEF_OBSS_THRESH_CMDID,
};

enum wmi_tlv_event_id {
	WMI_SERVICE_READY_EVENTID = 0x1,
	WMI_READY_EVENTID,
	WMI_SERVICE_AVAILABLE_EVENTID,
	WMI_SCAN_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_SCAN),
	WMI_PDEV_TPC_CONFIG_EVENTID = WMI_TLV_CMD(WMI_GRP_PDEV),
	WMI_CHAN_INFO_EVENTID,
	WMI_PHYERR_EVENTID,
	WMI_PDEV_DUMP_EVENTID,
	WMI_TX_PAUSE_EVENTID,
	WMI_DFS_RADAR_EVENTID,
	WMI_PDEV_L1SS_TRACK_EVENTID,
	WMI_PDEV_TEMPERATURE_EVENTID,
	WMI_SERVICE_READY_EXT_EVENTID,
	WMI_PDEV_FIPS_EVENTID,
	WMI_PDEV_CHANNEL_HOPPING_EVENTID,
	WMI_PDEV_ANI_CCK_LEVEL_EVENTID,
	WMI_PDEV_ANI_OFDM_LEVEL_EVENTID,
	WMI_PDEV_TPC_EVENTID,
	WMI_PDEV_NFCAL_POWER_ALL_CHANNELS_EVENTID,
	WMI_PDEV_SET_HW_MODE_RESP_EVENTID,
	WMI_PDEV_HW_MODE_TRANSITION_EVENTID,
	WMI_PDEV_SET_MAC_CONFIG_RESP_EVENTID,
	WMI_PDEV_ANTDIV_STATUS_EVENTID,
	WMI_PDEV_CHIP_POWER_STATS_EVENTID,
	WMI_PDEV_CHIP_POWER_SAVE_FAILURE_DETECTED_EVENTID,
	WMI_PDEV_CSA_SWITCH_COUNT_STATUS_EVENTID,
	WMI_PDEV_CHECK_CAL_VERSION_EVENTID,
	WMI_PDEV_DIV_RSSI_ANTID_EVENTID,
	WMI_PDEV_BSS_CHAN_INFO_EVENTID,
	WMI_PDEV_UPDATE_CTLTABLE_EVENTID,
	WMI_PDEV_DMA_RING_CFG_RSP_EVENTID,
	WMI_PDEV_DMA_RING_BUF_RELEASE_EVENTID,
	WMI_PDEV_CTL_FAILSAFE_CHECK_EVENTID,
	WMI_PDEV_CSC_SWITCH_COUNT_STATUS_EVENTID,
	WMI_PDEV_COLD_BOOT_CAL_DATA_EVENTID,
	WMI_PDEV_RAP_INFO_EVENTID,
	WMI_CHAN_RF_CHARACTERIZATION_INFO_EVENTID,
	WMI_SERVICE_READY_EXT2_EVENTID,
	WMI_VDEV_START_RESP_EVENTID = WMI_TLV_CMD(WMI_GRP_VDEV),
	WMI_VDEV_STOPPED_EVENTID,
	WMI_VDEV_INSTALL_KEY_COMPLETE_EVENTID,
	WMI_VDEV_MCC_BCN_INTERVAL_CHANGE_REQ_EVENTID,
	WMI_VDEV_TSF_REPORT_EVENTID,
	WMI_VDEV_DELETE_RESP_EVENTID,
	WMI_VDEV_ENCRYPT_DECRYPT_DATA_RESP_EVENTID,
	WMI_VDEV_ADD_MAC_ADDR_TO_RX_FILTER_STATUS_EVENTID,
	WMI_PEER_STA_KICKOUT_EVENTID = WMI_TLV_CMD(WMI_GRP_PEER),
	WMI_PEER_INFO_EVENTID,
	WMI_PEER_TX_FAIL_CNT_THR_EVENTID,
	WMI_PEER_ESTIMATED_LINKSPEED_EVENTID,
	WMI_PEER_STATE_EVENTID,
	WMI_PEER_ASSOC_CONF_EVENTID,
	WMI_PEER_DELETE_RESP_EVENTID,
	WMI_PEER_RATECODE_LIST_EVENTID,
	WMI_WDS_PEER_EVENTID,
	WMI_PEER_STA_PS_STATECHG_EVENTID,
	WMI_PEER_ANTDIV_INFO_EVENTID,
	WMI_PEER_RESERVED0_EVENTID,
	WMI_PEER_RESERVED1_EVENTID,
	WMI_PEER_RESERVED2_EVENTID,
	WMI_PEER_RESERVED3_EVENTID,
	WMI_PEER_RESERVED4_EVENTID,
	WMI_PEER_RESERVED5_EVENTID,
	WMI_PEER_RESERVED6_EVENTID,
	WMI_PEER_RESERVED7_EVENTID,
	WMI_PEER_RESERVED8_EVENTID,
	WMI_PEER_RESERVED9_EVENTID,
	WMI_PEER_RESERVED10_EVENTID,
	WMI_PEER_OPER_MODE_CHANGE_EVENTID,
	WMI_MGMT_RX_EVENTID = WMI_TLV_CMD(WMI_GRP_MGMT),
	WMI_HOST_SWBA_EVENTID,
	WMI_TBTTOFFSET_UPDATE_EVENTID,
	WMI_OFFLOAD_BCN_TX_STATUS_EVENTID,
	WMI_OFFLOAD_PROB_RESP_TX_STATUS_EVENTID,
	WMI_MGMT_TX_COMPLETION_EVENTID,
	WMI_MGMT_TX_BUNDLE_COMPLETION_EVENTID,
	WMI_TBTTOFFSET_EXT_UPDATE_EVENTID,
	WMI_OFFCHAN_DATA_TX_COMPLETION_EVENTID,
	WMI_HOST_FILS_DISCOVERY_EVENTID,
	WMI_TX_DELBA_COMPLETE_EVENTID = WMI_TLV_CMD(WMI_GRP_BA_NEG),
	WMI_TX_ADDBA_COMPLETE_EVENTID,
	WMI_BA_RSP_SSN_EVENTID,
	WMI_AGGR_STATE_TRIG_EVENTID,
	WMI_ROAM_EVENTID = WMI_TLV_CMD(WMI_GRP_ROAM),
	WMI_PROFILE_MATCH,
	WMI_ROAM_SYNCH_EVENTID,
	WMI_P2P_DISC_EVENTID = WMI_TLV_CMD(WMI_GRP_P2P),
	WMI_P2P_NOA_EVENTID,
	WMI_P2P_LISTEN_OFFLOAD_STOPPED_EVENTID,
	WMI_AP_PS_EGAP_INFO_EVENTID = WMI_TLV_CMD(WMI_GRP_AP_PS),
	WMI_PDEV_RESUME_EVENTID = WMI_TLV_CMD(WMI_GRP_SUSPEND),
	WMI_WOW_WAKEUP_HOST_EVENTID = WMI_TLV_CMD(WMI_GRP_WOW),
	WMI_D0_WOW_DISABLE_ACK_EVENTID,
	WMI_WOW_INITIAL_WAKEUP_EVENTID,
	WMI_RTT_MEASUREMENT_REPORT_EVENTID = WMI_TLV_CMD(WMI_GRP_RTT),
	WMI_TSF_MEASUREMENT_REPORT_EVENTID,
	WMI_RTT_ERROR_REPORT_EVENTID,
	WMI_STATS_EXT_EVENTID = WMI_TLV_CMD(WMI_GRP_STATS),
	WMI_IFACE_LINK_STATS_EVENTID,
	WMI_PEER_LINK_STATS_EVENTID,
	WMI_RADIO_LINK_STATS_EVENTID,
	WMI_UPDATE_FW_MEM_DUMP_EVENTID,
	WMI_DIAG_EVENT_LOG_SUPPORTED_EVENTID,
	WMI_INST_RSSI_STATS_EVENTID,
	WMI_RADIO_TX_POWER_LEVEL_STATS_EVENTID,
	WMI_REPORT_STATS_EVENTID,
	WMI_UPDATE_RCPI_EVENTID,
	WMI_PEER_STATS_INFO_EVENTID,
	WMI_RADIO_CHAN_STATS_EVENTID,
	WMI_NLO_MATCH_EVENTID = WMI_TLV_CMD(WMI_GRP_NLO_OFL),
	WMI_NLO_SCAN_COMPLETE_EVENTID,
	WMI_APFIND_EVENTID,
	WMI_PASSPOINT_MATCH_EVENTID,
	WMI_GTK_OFFLOAD_STATUS_EVENTID = WMI_TLV_CMD(WMI_GRP_GTK_OFL),
	WMI_GTK_REKEY_FAIL_EVENTID,
	WMI_CSA_HANDLING_EVENTID = WMI_TLV_CMD(WMI_GRP_CSA_OFL),
	WMI_CHATTER_PC_QUERY_EVENTID = WMI_TLV_CMD(WMI_GRP_CHATTER),
	WMI_PDEV_DFS_RADAR_DETECTION_EVENTID = WMI_TLV_CMD(WMI_GRP_DFS),
	WMI_VDEV_DFS_CAC_COMPLETE_EVENTID,
	WMI_VDEV_ADFS_OCAC_COMPLETE_EVENTID,
	WMI_ECHO_EVENTID = WMI_TLV_CMD(WMI_GRP_MISC),
	WMI_PDEV_UTF_EVENTID,
	WMI_DEBUG_MESG_EVENTID,
	WMI_UPDATE_STATS_EVENTID,
	WMI_DEBUG_PRINT_EVENTID,
	WMI_DCS_INTERFERENCE_EVENTID,
	WMI_PDEV_QVIT_EVENTID,
	WMI_WLAN_PROFILE_DATA_EVENTID,
	WMI_PDEV_FTM_INTG_EVENTID,
	WMI_WLAN_FREQ_AVOID_EVENTID,
	WMI_VDEV_GET_KEEPALIVE_EVENTID,
	WMI_THERMAL_MGMT_EVENTID,
	WMI_DIAG_DATA_CONTAINER_EVENTID,
	WMI_HOST_AUTO_SHUTDOWN_EVENTID,
	WMI_UPDATE_WHAL_MIB_STATS_EVENTID,
	WMI_UPDATE_VDEV_RATE_STATS_EVENTID,
	WMI_DIAG_EVENTID,
	WMI_OCB_SET_SCHED_EVENTID,
	WMI_DEBUG_MESG_FLUSH_COMPLETE_EVENTID,
	WMI_RSSI_BREACH_EVENTID,
	WMI_TRANSFER_DATA_TO_FLASH_COMPLETE_EVENTID,
	WMI_PDEV_UTF_SCPC_EVENTID,
	WMI_READ_DATA_FROM_FLASH_EVENTID,
	WMI_REPORT_RX_AGGR_FAILURE_EVENTID,
	WMI_PKGID_EVENTID,
	WMI_GPIO_INPUT_EVENTID = WMI_TLV_CMD(WMI_GRP_GPIO),
	WMI_UPLOADH_EVENTID,
	WMI_CAPTUREH_EVENTID,
	WMI_RFKILL_STATE_CHANGE_EVENTID,
	WMI_TDLS_PEER_EVENTID = WMI_TLV_CMD(WMI_GRP_TDLS),
	WMI_STA_SMPS_FORCE_MODE_COMPL_EVENTID = WMI_TLV_CMD(WMI_GRP_STA_SMPS),
	WMI_BATCH_SCAN_ENABLED_EVENTID = WMI_TLV_CMD(WMI_GRP_LOCATION_SCAN),
	WMI_BATCH_SCAN_RESULT_EVENTID,
	WMI_OEM_CAPABILITY_EVENTID = WMI_TLV_CMD(WMI_GRP_OEM),
	WMI_OEM_MEASUREMENT_REPORT_EVENTID,
	WMI_OEM_ERROR_REPORT_EVENTID,
	WMI_OEM_RESPONSE_EVENTID,
	WMI_NAN_EVENTID = WMI_TLV_CMD(WMI_GRP_NAN),
	WMI_NAN_DISC_IFACE_CREATED_EVENTID,
	WMI_NAN_DISC_IFACE_DELETED_EVENTID,
	WMI_NAN_STARTED_CLUSTER_EVENTID,
	WMI_NAN_JOINED_CLUSTER_EVENTID,
	WMI_COEX_REPORT_ANTENNA_ISOLATION_EVENTID = WMI_TLV_CMD(WMI_GRP_COEX),
	WMI_LPI_RESULT_EVENTID = WMI_TLV_CMD(WMI_GRP_LPI),
	WMI_LPI_STATUS_EVENTID,
	WMI_LPI_HANDOFF_EVENTID,
	WMI_EXTSCAN_START_STOP_EVENTID = WMI_TLV_CMD(WMI_GRP_EXTSCAN),
	WMI_EXTSCAN_OPERATION_EVENTID,
	WMI_EXTSCAN_TABLE_USAGE_EVENTID,
	WMI_EXTSCAN_CACHED_RESULTS_EVENTID,
	WMI_EXTSCAN_WLAN_CHANGE_RESULTS_EVENTID,
	WMI_EXTSCAN_HOTLIST_MATCH_EVENTID,
	WMI_EXTSCAN_CAPABILITIES_EVENTID,
	WMI_EXTSCAN_HOTLIST_SSID_MATCH_EVENTID,
	WMI_MDNS_STATS_EVENTID = WMI_TLV_CMD(WMI_GRP_MDNS_OFL),
	WMI_SAP_OFL_ADD_STA_EVENTID = WMI_TLV_CMD(WMI_GRP_SAP_OFL),
	WMI_SAP_OFL_DEL_STA_EVENTID,
	WMI_OCB_SET_CONFIG_RESP_EVENTID = WMI_TLV_CMD(WMI_GRP_OCB),
	WMI_OCB_GET_TSF_TIMER_RESP_EVENTID,
	WMI_DCC_GET_STATS_RESP_EVENTID,
	WMI_DCC_UPDATE_NDL_RESP_EVENTID,
	WMI_DCC_STATS_EVENTID,
	WMI_SOC_SET_HW_MODE_RESP_EVENTID = WMI_TLV_CMD(WMI_GRP_SOC),
	WMI_SOC_HW_MODE_TRANSITION_EVENTID,
	WMI_SOC_SET_DUAL_MAC_CONFIG_RESP_EVENTID,
	WMI_MAWC_ENABLE_SENSOR_EVENTID = WMI_TLV_CMD(WMI_GRP_MAWC),
	WMI_BPF_CAPABILIY_INFO_EVENTID = WMI_TLV_CMD(WMI_GRP_BPF_OFFLOAD),
	WMI_BPF_VDEV_STATS_INFO_EVENTID,
	WMI_RMC_NEW_LEADER_EVENTID = WMI_TLV_CMD(WMI_GRP_RMC),
	WMI_REG_CHAN_LIST_CC_EVENTID = WMI_TLV_CMD(WMI_GRP_REGULATORY),
	WMI_11D_NEW_COUNTRY_EVENTID,
	WMI_REG_CHAN_LIST_CC_EXT_EVENTID,
	WMI_NDI_CAP_RSP_EVENTID = WMI_TLV_CMD(WMI_GRP_PROTOTYPE),
	WMI_NDP_INITIATOR_RSP_EVENTID,
	WMI_NDP_RESPONDER_RSP_EVENTID,
	WMI_NDP_END_RSP_EVENTID,
	WMI_NDP_INDICATION_EVENTID,
	WMI_NDP_CONFIRM_EVENTID,
	WMI_NDP_END_INDICATION_EVENTID,

	WMI_TWT_ENABLE_EVENTID = WMI_TLV_CMD(WMI_GRP_TWT),
	WMI_TWT_DISABLE_EVENTID,
	WMI_TWT_ADD_DIALOG_EVENTID,
	WMI_TWT_DEL_DIALOG_EVENTID,
	WMI_TWT_PAUSE_DIALOG_EVENTID,
	WMI_TWT_RESUME_DIALOG_EVENTID,
};

enum wmi_tlv_pdev_param {
	WMI_PDEV_PARAM_TX_CHAIN_MASK = 0x1,
	WMI_PDEV_PARAM_RX_CHAIN_MASK,
	WMI_PDEV_PARAM_TXPOWER_LIMIT2G,
	WMI_PDEV_PARAM_TXPOWER_LIMIT5G,
	WMI_PDEV_PARAM_TXPOWER_SCALE,
	WMI_PDEV_PARAM_BEACON_GEN_MODE,
	WMI_PDEV_PARAM_BEACON_TX_MODE,
	WMI_PDEV_PARAM_RESMGR_OFFCHAN_MODE,
	WMI_PDEV_PARAM_PROTECTION_MODE,
	WMI_PDEV_PARAM_DYNAMIC_BW,
	WMI_PDEV_PARAM_NON_AGG_SW_RETRY_TH,
	WMI_PDEV_PARAM_AGG_SW_RETRY_TH,
	WMI_PDEV_PARAM_STA_KICKOUT_TH,
	WMI_PDEV_PARAM_AC_AGGRSIZE_SCALING,
	WMI_PDEV_PARAM_LTR_ENABLE,
	WMI_PDEV_PARAM_LTR_AC_LATENCY_BE,
	WMI_PDEV_PARAM_LTR_AC_LATENCY_BK,
	WMI_PDEV_PARAM_LTR_AC_LATENCY_VI,
	WMI_PDEV_PARAM_LTR_AC_LATENCY_VO,
	WMI_PDEV_PARAM_LTR_AC_LATENCY_TIMEOUT,
	WMI_PDEV_PARAM_LTR_SLEEP_OVERRIDE,
	WMI_PDEV_PARAM_LTR_RX_OVERRIDE,
	WMI_PDEV_PARAM_LTR_TX_ACTIVITY_TIMEOUT,
	WMI_PDEV_PARAM_L1SS_ENABLE,
	WMI_PDEV_PARAM_DSLEEP_ENABLE,
	WMI_PDEV_PARAM_PCIELP_TXBUF_FLUSH,
	WMI_PDEV_PARAM_PCIELP_TXBUF_WATERMARK,
	WMI_PDEV_PARAM_PCIELP_TXBUF_TMO_EN,
	WMI_PDEV_PARAM_PCIELP_TXBUF_TMO_VALUE,
	WMI_PDEV_PARAM_PDEV_STATS_UPDATE_PERIOD,
	WMI_PDEV_PARAM_VDEV_STATS_UPDATE_PERIOD,
	WMI_PDEV_PARAM_PEER_STATS_UPDATE_PERIOD,
	WMI_PDEV_PARAM_BCNFLT_STATS_UPDATE_PERIOD,
	WMI_PDEV_PARAM_PMF_QOS,
	WMI_PDEV_PARAM_ARP_AC_OVERRIDE,
	WMI_PDEV_PARAM_DCS,
	WMI_PDEV_PARAM_ANI_ENABLE,
	WMI_PDEV_PARAM_ANI_POLL_PERIOD,
	WMI_PDEV_PARAM_ANI_LISTEN_PERIOD,
	WMI_PDEV_PARAM_ANI_OFDM_LEVEL,
	WMI_PDEV_PARAM_ANI_CCK_LEVEL,
	WMI_PDEV_PARAM_DYNTXCHAIN,
	WMI_PDEV_PARAM_PROXY_STA,
	WMI_PDEV_PARAM_IDLE_PS_CONFIG,
	WMI_PDEV_PARAM_POWER_GATING_SLEEP,
	WMI_PDEV_PARAM_RFKILL_ENABLE,
	WMI_PDEV_PARAM_BURST_DUR,
	WMI_PDEV_PARAM_BURST_ENABLE,
	WMI_PDEV_PARAM_HW_RFKILL_CONFIG,
	WMI_PDEV_PARAM_LOW_POWER_RF_ENABLE,
	WMI_PDEV_PARAM_L1SS_TRACK,
	WMI_PDEV_PARAM_HYST_EN,
	WMI_PDEV_PARAM_POWER_COLLAPSE_ENABLE,
	WMI_PDEV_PARAM_LED_SYS_STATE,
	WMI_PDEV_PARAM_LED_ENABLE,
	WMI_PDEV_PARAM_AUDIO_OVER_WLAN_LATENCY,
	WMI_PDEV_PARAM_AUDIO_OVER_WLAN_ENABLE,
	WMI_PDEV_PARAM_WHAL_MIB_STATS_UPDATE_ENABLE,
	WMI_PDEV_PARAM_VDEV_RATE_STATS_UPDATE_PERIOD,
	WMI_PDEV_PARAM_CTS_CBW,
	WMI_PDEV_PARAM_WNTS_CONFIG,
	WMI_PDEV_PARAM_ADAPTIVE_EARLY_RX_ENABLE,
	WMI_PDEV_PARAM_ADAPTIVE_EARLY_RX_MIN_SLEEP_SLOP,
	WMI_PDEV_PARAM_ADAPTIVE_EARLY_RX_INC_DEC_STEP,
	WMI_PDEV_PARAM_EARLY_RX_FIX_SLEEP_SLOP,
	WMI_PDEV_PARAM_BMISS_BASED_ADAPTIVE_BTO_ENABLE,
	WMI_PDEV_PARAM_BMISS_BTO_MIN_BCN_TIMEOUT,
	WMI_PDEV_PARAM_BMISS_BTO_INC_DEC_STEP,
	WMI_PDEV_PARAM_BTO_FIX_BCN_TIMEOUT,
	WMI_PDEV_PARAM_CE_BASED_ADAPTIVE_BTO_ENABLE,
	WMI_PDEV_PARAM_CE_BTO_COMBO_CE_VALUE,
	WMI_PDEV_PARAM_TX_CHAIN_MASK_2G,
	WMI_PDEV_PARAM_RX_CHAIN_MASK_2G,
	WMI_PDEV_PARAM_TX_CHAIN_MASK_5G,
	WMI_PDEV_PARAM_RX_CHAIN_MASK_5G,
	WMI_PDEV_PARAM_TX_CHAIN_MASK_CCK,
	WMI_PDEV_PARAM_TX_CHAIN_MASK_1SS,
	WMI_PDEV_PARAM_CTS2SELF_FOR_P2P_GO_CONFIG,
	WMI_PDEV_PARAM_TXPOWER_DECR_DB,
	WMI_PDEV_PARAM_AGGR_BURST,
	WMI_PDEV_PARAM_RX_DECAP_MODE,
	WMI_PDEV_PARAM_FAST_CHANNEL_RESET,
	WMI_PDEV_PARAM_SMART_ANTENNA_DEFAULT_ANTENNA,
	WMI_PDEV_PARAM_ANTENNA_GAIN,
	WMI_PDEV_PARAM_RX_FILTER,
	WMI_PDEV_SET_MCAST_TO_UCAST_TID,
	WMI_PDEV_PARAM_PROXY_STA_MODE,
	WMI_PDEV_PARAM_SET_MCAST2UCAST_MODE,
	WMI_PDEV_PARAM_SET_MCAST2UCAST_BUFFER,
	WMI_PDEV_PARAM_REMOVE_MCAST2UCAST_BUFFER,
	WMI_PDEV_PEER_STA_PS_STATECHG_ENABLE,
	WMI_PDEV_PARAM_IGMPMLD_AC_OVERRIDE,
	WMI_PDEV_PARAM_BLOCK_INTERBSS,
	WMI_PDEV_PARAM_SET_DISABLE_RESET_CMDID,
	WMI_PDEV_PARAM_SET_MSDU_TTL_CMDID,
	WMI_PDEV_PARAM_SET_PPDU_DURATION_CMDID,
	WMI_PDEV_PARAM_TXBF_SOUND_PERIOD_CMDID,
	WMI_PDEV_PARAM_SET_PROMISC_MODE_CMDID,
	WMI_PDEV_PARAM_SET_BURST_MODE_CMDID,
	WMI_PDEV_PARAM_EN_STATS,
	WMI_PDEV_PARAM_MU_GROUP_POLICY,
	WMI_PDEV_PARAM_NOISE_DETECTION,
	WMI_PDEV_PARAM_NOISE_THRESHOLD,
	WMI_PDEV_PARAM_DPD_ENABLE,
	WMI_PDEV_PARAM_SET_MCAST_BCAST_ECHO,
	WMI_PDEV_PARAM_ATF_STRICT_SCH,
	WMI_PDEV_PARAM_ATF_SCHED_DURATION,
	WMI_PDEV_PARAM_ANT_PLZN,
	WMI_PDEV_PARAM_MGMT_RETRY_LIMIT,
	WMI_PDEV_PARAM_SENSITIVITY_LEVEL,
	WMI_PDEV_PARAM_SIGNED_TXPOWER_2G,
	WMI_PDEV_PARAM_SIGNED_TXPOWER_5G,
	WMI_PDEV_PARAM_ENABLE_PER_TID_AMSDU,
	WMI_PDEV_PARAM_ENABLE_PER_TID_AMPDU,
	WMI_PDEV_PARAM_CCA_THRESHOLD,
	WMI_PDEV_PARAM_RTS_FIXED_RATE,
	WMI_PDEV_PARAM_PDEV_RESET,
	WMI_PDEV_PARAM_WAPI_MBSSID_OFFSET,
	WMI_PDEV_PARAM_ARP_DBG_SRCADDR,
	WMI_PDEV_PARAM_ARP_DBG_DSTADDR,
	WMI_PDEV_PARAM_ATF_OBSS_NOISE_SCH,
	WMI_PDEV_PARAM_ATF_OBSS_NOISE_SCALING_FACTOR,
	WMI_PDEV_PARAM_CUST_TXPOWER_SCALE,
	WMI_PDEV_PARAM_ATF_DYNAMIC_ENABLE,
	WMI_PDEV_PARAM_CTRL_RETRY_LIMIT,
	WMI_PDEV_PARAM_PROPAGATION_DELAY,
	WMI_PDEV_PARAM_ENA_ANT_DIV,
	WMI_PDEV_PARAM_FORCE_CHAIN_ANT,
	WMI_PDEV_PARAM_ANT_DIV_SELFTEST,
	WMI_PDEV_PARAM_ANT_DIV_SELFTEST_INTVL,
	WMI_PDEV_PARAM_STATS_OBSERVATION_PERIOD,
	WMI_PDEV_PARAM_TX_PPDU_DELAY_BIN_SIZE_MS,
	WMI_PDEV_PARAM_TX_PPDU_DELAY_ARRAY_LEN,
	WMI_PDEV_PARAM_TX_MPDU_AGGR_ARRAY_LEN,
	WMI_PDEV_PARAM_RX_MPDU_AGGR_ARRAY_LEN,
	WMI_PDEV_PARAM_TX_SCH_DELAY,
	WMI_PDEV_PARAM_ENABLE_RTS_SIFS_BURSTING,
	WMI_PDEV_PARAM_MAX_MPDUS_IN_AMPDU,
	WMI_PDEV_PARAM_PEER_STATS_INFO_ENABLE,
	WMI_PDEV_PARAM_FAST_PWR_TRANSITION,
	WMI_PDEV_PARAM_RADIO_CHAN_STATS_ENABLE,
	WMI_PDEV_PARAM_RADIO_DIAGNOSIS_ENABLE,
	WMI_PDEV_PARAM_MESH_MCAST_ENABLE,
};

enum wmi_tlv_vdev_param {
	WMI_VDEV_PARAM_RTS_THRESHOLD = 0x1,
	WMI_VDEV_PARAM_FRAGMENTATION_THRESHOLD,
	WMI_VDEV_PARAM_BEACON_INTERVAL,
	WMI_VDEV_PARAM_LISTEN_INTERVAL,
	WMI_VDEV_PARAM_MULTICAST_RATE,
	WMI_VDEV_PARAM_MGMT_TX_RATE,
	WMI_VDEV_PARAM_SLOT_TIME,
	WMI_VDEV_PARAM_PREAMBLE,
	WMI_VDEV_PARAM_SWBA_TIME,
	WMI_VDEV_STATS_UPDATE_PERIOD,
	WMI_VDEV_PWRSAVE_AGEOUT_TIME,
	WMI_VDEV_HOST_SWBA_INTERVAL,
	WMI_VDEV_PARAM_DTIM_PERIOD,
	WMI_VDEV_OC_SCHEDULER_AIR_TIME_LIMIT,
	WMI_VDEV_PARAM_WDS,
	WMI_VDEV_PARAM_ATIM_WINDOW,
	WMI_VDEV_PARAM_BMISS_COUNT_MAX,
	WMI_VDEV_PARAM_BMISS_FIRST_BCNT,
	WMI_VDEV_PARAM_BMISS_FINAL_BCNT,
	WMI_VDEV_PARAM_FEATURE_WMM,
	WMI_VDEV_PARAM_CHWIDTH,
	WMI_VDEV_PARAM_CHEXTOFFSET,
	WMI_VDEV_PARAM_DISABLE_HTPROTECTION,
	WMI_VDEV_PARAM_STA_QUICKKICKOUT,
	WMI_VDEV_PARAM_MGMT_RATE,
	WMI_VDEV_PARAM_PROTECTION_MODE,
	WMI_VDEV_PARAM_FIXED_RATE,
	WMI_VDEV_PARAM_SGI,
	WMI_VDEV_PARAM_LDPC,
	WMI_VDEV_PARAM_TX_STBC,
	WMI_VDEV_PARAM_RX_STBC,
	WMI_VDEV_PARAM_INTRA_BSS_FWD,
	WMI_VDEV_PARAM_DEF_KEYID,
	WMI_VDEV_PARAM_NSS,
	WMI_VDEV_PARAM_BCAST_DATA_RATE,
	WMI_VDEV_PARAM_MCAST_DATA_RATE,
	WMI_VDEV_PARAM_MCAST_INDICATE,
	WMI_VDEV_PARAM_DHCP_INDICATE,
	WMI_VDEV_PARAM_UNKNOWN_DEST_INDICATE,
	WMI_VDEV_PARAM_AP_KEEPALIVE_MIN_IDLE_INACTIVE_TIME_SECS,
	WMI_VDEV_PARAM_AP_KEEPALIVE_MAX_IDLE_INACTIVE_TIME_SECS,
	WMI_VDEV_PARAM_AP_KEEPALIVE_MAX_UNRESPONSIVE_TIME_SECS,
	WMI_VDEV_PARAM_AP_ENABLE_NAWDS,
	WMI_VDEV_PARAM_ENABLE_RTSCTS,
	WMI_VDEV_PARAM_TXBF,
	WMI_VDEV_PARAM_PACKET_POWERSAVE,
	WMI_VDEV_PARAM_DROP_UNENCRY,
	WMI_VDEV_PARAM_TX_ENCAP_TYPE,
	WMI_VDEV_PARAM_AP_DETECT_OUT_OF_SYNC_SLEEPING_STA_TIME_SECS,
	WMI_VDEV_PARAM_EARLY_RX_ADJUST_ENABLE,
	WMI_VDEV_PARAM_EARLY_RX_TGT_BMISS_NUM,
	WMI_VDEV_PARAM_EARLY_RX_BMISS_SAMPLE_CYCLE,
	WMI_VDEV_PARAM_EARLY_RX_SLOP_STEP,
	WMI_VDEV_PARAM_EARLY_RX_INIT_SLOP,
	WMI_VDEV_PARAM_EARLY_RX_ADJUST_PAUSE,
	WMI_VDEV_PARAM_TX_PWRLIMIT,
	WMI_VDEV_PARAM_SNR_NUM_FOR_CAL,
	WMI_VDEV_PARAM_ROAM_FW_OFFLOAD,
	WMI_VDEV_PARAM_ENABLE_RMC,
	WMI_VDEV_PARAM_IBSS_MAX_BCN_LOST_MS,
	WMI_VDEV_PARAM_MAX_RATE,
	WMI_VDEV_PARAM_EARLY_RX_DRIFT_SAMPLE,
	WMI_VDEV_PARAM_SET_IBSS_TX_FAIL_CNT_THR,
	WMI_VDEV_PARAM_EBT_RESYNC_TIMEOUT,
	WMI_VDEV_PARAM_AGGR_TRIG_EVENT_ENABLE,
	WMI_VDEV_PARAM_IS_IBSS_POWER_SAVE_ALLOWED,
	WMI_VDEV_PARAM_IS_POWER_COLLAPSE_ALLOWED,
	WMI_VDEV_PARAM_IS_AWAKE_ON_TXRX_ENABLED,
	WMI_VDEV_PARAM_INACTIVITY_CNT,
	WMI_VDEV_PARAM_TXSP_END_INACTIVITY_TIME_MS,
	WMI_VDEV_PARAM_DTIM_POLICY,
	WMI_VDEV_PARAM_IBSS_PS_WARMUP_TIME_SECS,
	WMI_VDEV_PARAM_IBSS_PS_1RX_CHAIN_IN_ATIM_WINDOW_ENABLE,
	WMI_VDEV_PARAM_RX_LEAK_WINDOW,
	WMI_VDEV_PARAM_STATS_AVG_FACTOR,
	WMI_VDEV_PARAM_DISCONNECT_TH,
	WMI_VDEV_PARAM_RTSCTS_RATE,
	WMI_VDEV_PARAM_MCC_RTSCTS_PROTECTION_ENABLE,
	WMI_VDEV_PARAM_MCC_BROADCAST_PROBE_ENABLE,
	WMI_VDEV_PARAM_TXPOWER_SCALE,
	WMI_VDEV_PARAM_TXPOWER_SCALE_DECR_DB,
	WMI_VDEV_PARAM_MCAST2UCAST_SET,
	WMI_VDEV_PARAM_RC_NUM_RETRIES,
	WMI_VDEV_PARAM_CABQ_MAXDUR,
	WMI_VDEV_PARAM_MFPTEST_SET,
	WMI_VDEV_PARAM_RTS_FIXED_RATE,
	WMI_VDEV_PARAM_VHT_SGIMASK,
	WMI_VDEV_PARAM_VHT80_RATEMASK,
	WMI_VDEV_PARAM_PROXY_STA,
	WMI_VDEV_PARAM_VIRTUAL_CELL_MODE,
	WMI_VDEV_PARAM_RX_DECAP_TYPE,
	WMI_VDEV_PARAM_BW_NSS_RATEMASK,
	WMI_VDEV_PARAM_SENSOR_AP,
	WMI_VDEV_PARAM_BEACON_RATE,
	WMI_VDEV_PARAM_DTIM_ENABLE_CTS,
	WMI_VDEV_PARAM_STA_KICKOUT,
	WMI_VDEV_PARAM_CAPABILITIES,
	WMI_VDEV_PARAM_TSF_INCREMENT,
	WMI_VDEV_PARAM_AMPDU_PER_AC,
	WMI_VDEV_PARAM_RX_FILTER,
	WMI_VDEV_PARAM_MGMT_TX_POWER,
	WMI_VDEV_PARAM_NON_AGG_SW_RETRY_TH,
	WMI_VDEV_PARAM_AGG_SW_RETRY_TH,
	WMI_VDEV_PARAM_DISABLE_DYN_BW_RTS,
	WMI_VDEV_PARAM_ATF_SSID_SCHED_POLICY,
	WMI_VDEV_PARAM_HE_DCM,
	WMI_VDEV_PARAM_HE_RANGE_EXT,
	WMI_VDEV_PARAM_ENABLE_BCAST_PROBE_RESPONSE,
	WMI_VDEV_PARAM_FILS_MAX_CHANNEL_GUARD_TIME,
	WMI_VDEV_PARAM_BA_MODE = 0x7e,
	WMI_VDEV_PARAM_SET_HE_SOUNDING_MODE = 0x87,
	WMI_VDEV_PARAM_6GHZ_PARAMS = 0x99,
	WMI_VDEV_PARAM_PROTOTYPE = 0x8000,
	WMI_VDEV_PARAM_BSS_COLOR,
	WMI_VDEV_PARAM_SET_HEMU_MODE,
	WMI_VDEV_PARAM_HEOPS_0_31 = 0x8003,
};

enum wmi_tlv_peer_flags {
	WMI_TLV_PEER_AUTH = 0x00000001,
	WMI_TLV_PEER_QOS = 0x00000002,
	WMI_TLV_PEER_NEED_PTK_4_WAY = 0x00000004,
	WMI_TLV_PEER_NEED_GTK_2_WAY = 0x00000010,
	WMI_TLV_PEER_APSD = 0x00000800,
	WMI_TLV_PEER_HT = 0x00001000,
	WMI_TLV_PEER_40MHZ = 0x00002000,
	WMI_TLV_PEER_STBC = 0x00008000,
	WMI_TLV_PEER_LDPC = 0x00010000,
	WMI_TLV_PEER_DYN_MIMOPS = 0x00020000,
	WMI_TLV_PEER_STATIC_MIMOPS = 0x00040000,
	WMI_TLV_PEER_SPATIAL_MUX = 0x00200000,
	WMI_TLV_PEER_VHT = 0x02000000,
	WMI_TLV_PEER_80MHZ = 0x04000000,
	WMI_TLV_PEER_PMF = 0x08000000,
	WMI_PEER_IS_P2P_CAPABLE = 0x20000000,
	WMI_PEER_160MHZ         = 0x40000000,
	WMI_PEER_SAFEMODE_EN    = 0x80000000,

};

enum wmi_tlv_peer_flags_ext {
	WMI_PEER_EXT_EHT = BIT(0),
	WMI_PEER_EXT_320MHZ = BIT(1),
};

/** Enum list of TLV Tags for each parameter structure type. */
enum wmi_tlv_tag {
	WMI_TAG_LAST_RESERVED = 15,
	WMI_TAG_FIRST_ARRAY_ENUM,
	WMI_TAG_ARRAY_UINT32 = WMI_TAG_FIRST_ARRAY_ENUM,
	WMI_TAG_ARRAY_BYTE,
	WMI_TAG_ARRAY_STRUCT,
	WMI_TAG_ARRAY_FIXED_STRUCT,
	WMI_TAG_LAST_ARRAY_ENUM = 31,
	WMI_TAG_SERVICE_READY_EVENT,
	WMI_TAG_HAL_REG_CAPABILITIES,
	WMI_TAG_WLAN_HOST_MEM_REQ,
	WMI_TAG_READY_EVENT,
	WMI_TAG_SCAN_EVENT,
	WMI_TAG_PDEV_TPC_CONFIG_EVENT,
	WMI_TAG_CHAN_INFO_EVENT,
	WMI_TAG_COMB_PHYERR_RX_HDR,
	WMI_TAG_VDEV_START_RESPONSE_EVENT,
	WMI_TAG_VDEV_STOPPED_EVENT,
	WMI_TAG_VDEV_INSTALL_KEY_COMPLETE_EVENT,
	WMI_TAG_PEER_STA_KICKOUT_EVENT,
	WMI_TAG_MGMT_RX_HDR,
	WMI_TAG_TBTT_OFFSET_EVENT,
	WMI_TAG_TX_DELBA_COMPLETE_EVENT,
	WMI_TAG_TX_ADDBA_COMPLETE_EVENT,
	WMI_TAG_ROAM_EVENT,
	WMI_TAG_WOW_EVENT_INFO,
	WMI_TAG_WOW_EVENT_INFO_SECTION_BITMAP,
	WMI_TAG_RTT_EVENT_HEADER,
	WMI_TAG_RTT_ERROR_REPORT_EVENT,
	WMI_TAG_RTT_MEAS_EVENT,
	WMI_TAG_ECHO_EVENT,
	WMI_TAG_FTM_INTG_EVENT,
	WMI_TAG_VDEV_GET_KEEPALIVE_EVENT,
	WMI_TAG_GPIO_INPUT_EVENT,
	WMI_TAG_CSA_EVENT,
	WMI_TAG_GTK_OFFLOAD_STATUS_EVENT,
	WMI_TAG_IGTK_INFO,
	WMI_TAG_DCS_INTERFERENCE_EVENT,
	WMI_TAG_ATH_DCS_CW_INT,
	WMI_TAG_WLAN_DCS_CW_INT = /* ALIAS */
		WMI_TAG_ATH_DCS_CW_INT,
	WMI_TAG_ATH_DCS_WLAN_INT_STAT,
	WMI_TAG_WLAN_DCS_IM_TGT_STATS_T = /* ALIAS */
		WMI_TAG_ATH_DCS_WLAN_INT_STAT,
	WMI_TAG_WLAN_PROFILE_CTX_T,
	WMI_TAG_WLAN_PROFILE_T,
	WMI_TAG_PDEV_QVIT_EVENT,
	WMI_TAG_HOST_SWBA_EVENT,
	WMI_TAG_TIM_INFO,
	WMI_TAG_P2P_NOA_INFO,
	WMI_TAG_STATS_EVENT,
	WMI_TAG_AVOID_FREQ_RANGES_EVENT,
	WMI_TAG_AVOID_FREQ_RANGE_DESC,
	WMI_TAG_GTK_REKEY_FAIL_EVENT,
	WMI_TAG_INIT_CMD,
	WMI_TAG_RESOURCE_CONFIG,
	WMI_TAG_WLAN_HOST_MEMORY_CHUNK,
	WMI_TAG_START_SCAN_CMD,
	WMI_TAG_STOP_SCAN_CMD,
	WMI_TAG_SCAN_CHAN_LIST_CMD,
	WMI_TAG_CHANNEL,
	WMI_TAG_PDEV_SET_REGDOMAIN_CMD,
	WMI_TAG_PDEV_SET_PARAM_CMD,
	WMI_TAG_PDEV_SET_WMM_PARAMS_CMD,
	WMI_TAG_WMM_PARAMS,
	WMI_TAG_PDEV_SET_QUIET_CMD,
	WMI_TAG_VDEV_CREATE_CMD,
	WMI_TAG_VDEV_DELETE_CMD,
	WMI_TAG_VDEV_START_REQUEST_CMD,
	WMI_TAG_P2P_NOA_DESCRIPTOR,
	WMI_TAG_P2P_GO_SET_BEACON_IE,
	WMI_TAG_GTK_OFFLOAD_CMD,
	WMI_TAG_VDEV_UP_CMD,
	WMI_TAG_VDEV_STOP_CMD,
	WMI_TAG_VDEV_DOWN_CMD,
	WMI_TAG_VDEV_SET_PARAM_CMD,
	WMI_TAG_VDEV_INSTALL_KEY_CMD,
	WMI_TAG_PEER_CREATE_CMD,
	WMI_TAG_PEER_DELETE_CMD,
	WMI_TAG_PEER_FLUSH_TIDS_CMD,
	WMI_TAG_PEER_SET_PARAM_CMD,
	WMI_TAG_PEER_ASSOC_COMPLETE_CMD,
	WMI_TAG_VHT_RATE_SET,
	WMI_TAG_BCN_TMPL_CMD,
	WMI_TAG_PRB_TMPL_CMD,
	WMI_TAG_BCN_PRB_INFO,
	WMI_TAG_PEER_TID_ADDBA_CMD,
	WMI_TAG_PEER_TID_DELBA_CMD,
	WMI_TAG_STA_POWERSAVE_MODE_CMD,
	WMI_TAG_STA_POWERSAVE_PARAM_CMD,
	WMI_TAG_STA_DTIM_PS_METHOD_CMD,
	WMI_TAG_ROAM_SCAN_MODE,
	WMI_TAG_ROAM_SCAN_RSSI_THRESHOLD,
	WMI_TAG_ROAM_SCAN_PERIOD,
	WMI_TAG_ROAM_SCAN_RSSI_CHANGE_THRESHOLD,
	WMI_TAG_PDEV_SUSPEND_CMD,
	WMI_TAG_PDEV_RESUME_CMD,
	WMI_TAG_ADD_BCN_FILTER_CMD,
	WMI_TAG_RMV_BCN_FILTER_CMD,
	WMI_TAG_WOW_ENABLE_CMD,
	WMI_TAG_WOW_HOSTWAKEUP_FROM_SLEEP_CMD,
	WMI_TAG_STA_UAPSD_AUTO_TRIG_CMD,
	WMI_TAG_STA_UAPSD_AUTO_TRIG_PARAM,
	WMI_TAG_SET_ARP_NS_OFFLOAD_CMD,
	WMI_TAG_ARP_OFFLOAD_TUPLE,
	WMI_TAG_NS_OFFLOAD_TUPLE,
	WMI_TAG_FTM_INTG_CMD,
	WMI_TAG_STA_KEEPALIVE_CMD,
	WMI_TAG_STA_KEEPALVE_ARP_RESPONSE,
	WMI_TAG_P2P_SET_VENDOR_IE_DATA_CMD,
	WMI_TAG_AP_PS_PEER_CMD,
	WMI_TAG_PEER_RATE_RETRY_SCHED_CMD,
	WMI_TAG_WLAN_PROFILE_TRIGGER_CMD,
	WMI_TAG_WLAN_PROFILE_SET_HIST_INTVL_CMD,
	WMI_TAG_WLAN_PROFILE_GET_PROF_DATA_CMD,
	WMI_TAG_WLAN_PROFILE_ENABLE_PROFILE_ID_CMD,
	WMI_TAG_WOW_DEL_PATTERN_CMD,
	WMI_TAG_WOW_ADD_DEL_EVT_CMD,
	WMI_TAG_RTT_MEASREQ_HEAD,
	WMI_TAG_RTT_MEASREQ_BODY,
	WMI_TAG_RTT_TSF_CMD,
	WMI_TAG_VDEV_SPECTRAL_CONFIGURE_CMD,
	WMI_TAG_VDEV_SPECTRAL_ENABLE_CMD,
	WMI_TAG_REQUEST_STATS_CMD,
	WMI_TAG_NLO_CONFIG_CMD,
	WMI_TAG_NLO_CONFIGURED_PARAMETERS,
	WMI_TAG_CSA_OFFLOAD_ENABLE_CMD,
	WMI_TAG_CSA_OFFLOAD_CHANSWITCH_CMD,
	WMI_TAG_CHATTER_SET_MODE_CMD,
	WMI_TAG_ECHO_CMD,
	WMI_TAG_VDEV_SET_KEEPALIVE_CMD,
	WMI_TAG_VDEV_GET_KEEPALIVE_CMD,
	WMI_TAG_FORCE_FW_HANG_CMD,
	WMI_TAG_GPIO_CONFIG_CMD,
	WMI_TAG_GPIO_OUTPUT_CMD,
	WMI_TAG_PEER_ADD_WDS_ENTRY_CMD,
	WMI_TAG_PEER_REMOVE_WDS_ENTRY_CMD,
	WMI_TAG_BCN_TX_HDR,
	WMI_TAG_BCN_SEND_FROM_HOST_CMD,
	WMI_TAG_MGMT_TX_HDR,
	WMI_TAG_ADDBA_CLEAR_RESP_CMD,
	WMI_TAG_ADDBA_SEND_CMD,
	WMI_TAG_DELBA_SEND_CMD,
	WMI_TAG_ADDBA_SETRESPONSE_CMD,
	WMI_TAG_SEND_SINGLEAMSDU_CMD,
	WMI_TAG_PDEV_PKTLOG_ENABLE_CMD,
	WMI_TAG_PDEV_PKTLOG_DISABLE_CMD,
	WMI_TAG_PDEV_SET_HT_IE_CMD,
	WMI_TAG_PDEV_SET_VHT_IE_CMD,
	WMI_TAG_PDEV_SET_DSCP_TID_MAP_CMD,
	WMI_TAG_PDEV_GREEN_AP_PS_ENABLE_CMD,
	WMI_TAG_PDEV_GET_TPC_CONFIG_CMD,
	WMI_TAG_PDEV_SET_BASE_MACADDR_CMD,
	WMI_TAG_PEER_MCAST_GROUP_CMD,
	WMI_TAG_ROAM_AP_PROFILE,
	WMI_TAG_AP_PROFILE,
	WMI_TAG_SCAN_SCH_PRIORITY_TABLE_CMD,
	WMI_TAG_PDEV_DFS_ENABLE_CMD,
	WMI_TAG_PDEV_DFS_DISABLE_CMD,
	WMI_TAG_WOW_ADD_PATTERN_CMD,
	WMI_TAG_WOW_BITMAP_PATTERN_T,
	WMI_TAG_WOW_IPV4_SYNC_PATTERN_T,
	WMI_TAG_WOW_IPV6_SYNC_PATTERN_T,
	WMI_TAG_WOW_MAGIC_PATTERN_CMD,
	WMI_TAG_SCAN_UPDATE_REQUEST_CMD,
	WMI_TAG_CHATTER_PKT_COALESCING_FILTER,
	WMI_TAG_CHATTER_COALESCING_ADD_FILTER_CMD,
	WMI_TAG_CHATTER_COALESCING_DELETE_FILTER_CMD,
	WMI_TAG_CHATTER_COALESCING_QUERY_CMD,
	WMI_TAG_TXBF_CMD,
	WMI_TAG_DEBUG_LOG_CONFIG_CMD,
	WMI_TAG_NLO_EVENT,
	WMI_TAG_CHATTER_QUERY_REPLY_EVENT,
	WMI_TAG_UPLOAD_H_HDR,
	WMI_TAG_CAPTURE_H_EVENT_HDR,
	WMI_TAG_VDEV_WNM_SLEEPMODE_CMD,
	WMI_TAG_VDEV_IPSEC_NATKEEPALIVE_FILTER_CMD,
	WMI_TAG_VDEV_WMM_ADDTS_CMD,
	WMI_TAG_VDEV_WMM_DELTS_CMD,
	WMI_TAG_VDEV_SET_WMM_PARAMS_CMD,
	WMI_TAG_TDLS_SET_STATE_CMD,
	WMI_TAG_TDLS_PEER_UPDATE_CMD,
	WMI_TAG_TDLS_PEER_EVENT,
	WMI_TAG_TDLS_PEER_CAPABILITIES,
	WMI_TAG_VDEV_MCC_SET_TBTT_MODE_CMD,
	WMI_TAG_ROAM_CHAN_LIST,
	WMI_TAG_VDEV_MCC_BCN_INTVL_CHANGE_EVENT,
	WMI_TAG_RESMGR_ADAPTIVE_OCS_ENABLE_DISABLE_CMD,
	WMI_TAG_RESMGR_SET_CHAN_TIME_QUOTA_CMD,
	WMI_TAG_RESMGR_SET_CHAN_LATENCY_CMD,
	WMI_TAG_BA_REQ_SSN_CMD,
	WMI_TAG_BA_RSP_SSN_EVENT,
	WMI_TAG_STA_SMPS_FORCE_MODE_CMD,
	WMI_TAG_SET_MCASTBCAST_FILTER_CMD,
	WMI_TAG_P2P_SET_OPPPS_CMD,
	WMI_TAG_P2P_SET_NOA_CMD,
	WMI_TAG_BA_REQ_SSN_CMD_SUB_STRUCT_PARAM,
	WMI_TAG_BA_REQ_SSN_EVENT_SUB_STRUCT_PARAM,
	WMI_TAG_STA_SMPS_PARAM_CMD,
	WMI_TAG_VDEV_SET_GTX_PARAMS_CMD,
	WMI_TAG_MCC_SCHED_TRAFFIC_STATS_CMD,
	WMI_TAG_MCC_SCHED_STA_TRAFFIC_STATS,
	WMI_TAG_OFFLOAD_BCN_TX_STATUS_EVENT,
	WMI_TAG_P2P_NOA_EVENT,
	WMI_TAG_HB_SET_ENABLE_CMD,
	WMI_TAG_HB_SET_TCP_PARAMS_CMD,
	WMI_TAG_HB_SET_TCP_PKT_FILTER_CMD,
	WMI_TAG_HB_SET_UDP_PARAMS_CMD,
	WMI_TAG_HB_SET_UDP_PKT_FILTER_CMD,
	WMI_TAG_HB_IND_EVENT,
	WMI_TAG_TX_PAUSE_EVENT,
	WMI_TAG_RFKILL_EVENT,
	WMI_TAG_DFS_RADAR_EVENT,
	WMI_TAG_DFS_PHYERR_FILTER_ENA_CMD,
	WMI_TAG_DFS_PHYERR_FILTER_DIS_CMD,
	WMI_TAG_BATCH_SCAN_RESULT_SCAN_LIST,
	WMI_TAG_BATCH_SCAN_RESULT_NETWORK_INFO,
	WMI_TAG_BATCH_SCAN_ENABLE_CMD,
	WMI_TAG_BATCH_SCAN_DISABLE_CMD,
	WMI_TAG_BATCH_SCAN_TRIGGER_RESULT_CMD,
	WMI_TAG_BATCH_SCAN_ENABLED_EVENT,
	WMI_TAG_BATCH_SCAN_RESULT_EVENT,
	WMI_TAG_VDEV_PLMREQ_START_CMD,
	WMI_TAG_VDEV_PLMREQ_STOP_CMD,
	WMI_TAG_THERMAL_MGMT_CMD,
	WMI_TAG_THERMAL_MGMT_EVENT,
	WMI_TAG_PEER_INFO_REQ_CMD,
	WMI_TAG_PEER_INFO_EVENT,
	WMI_TAG_PEER_INFO,
	WMI_TAG_PEER_TX_FAIL_CNT_THR_EVENT,
	WMI_TAG_RMC_SET_MODE_CMD,
	WMI_TAG_RMC_SET_ACTION_PERIOD_CMD,
	WMI_TAG_RMC_CONFIG_CMD,
	WMI_TAG_MHF_OFFLOAD_SET_MODE_CMD,
	WMI_TAG_MHF_OFFLOAD_PLUMB_ROUTING_TABLE_CMD,
	WMI_TAG_ADD_PROACTIVE_ARP_RSP_PATTERN_CMD,
	WMI_TAG_DEL_PROACTIVE_ARP_RSP_PATTERN_CMD,
	WMI_TAG_NAN_CMD_PARAM,
	WMI_TAG_NAN_EVENT_HDR,
	WMI_TAG_PDEV_L1SS_TRACK_EVENT,
	WMI_TAG_DIAG_DATA_CONTAINER_EVENT,
	WMI_TAG_MODEM_POWER_STATE_CMD_PARAM,
	WMI_TAG_PEER_GET_ESTIMATED_LINKSPEED_CMD,
	WMI_TAG_PEER_ESTIMATED_LINKSPEED_EVENT,
	WMI_TAG_AGGR_STATE_TRIG_EVENT,
	WMI_TAG_MHF_OFFLOAD_ROUTING_TABLE_ENTRY,
	WMI_TAG_ROAM_SCAN_CMD,
	WMI_TAG_REQ_STATS_EXT_CMD,
	WMI_TAG_STATS_EXT_EVENT,
	WMI_TAG_OBSS_SCAN_ENABLE_CMD,
	WMI_TAG_OBSS_SCAN_DISABLE_CMD,
	WMI_TAG_OFFLOAD_PRB_RSP_TX_STATUS_EVENT,
	WMI_TAG_PDEV_SET_LED_CONFIG_CMD,
	WMI_TAG_HOST_AUTO_SHUTDOWN_CFG_CMD,
	WMI_TAG_HOST_AUTO_SHUTDOWN_EVENT,
	WMI_TAG_UPDATE_WHAL_MIB_STATS_EVENT,
	WMI_TAG_CHAN_AVOID_UPDATE_CMD_PARAM,
	WMI_TAG_WOW_IOAC_PKT_PATTERN_T,
	WMI_TAG_WOW_IOAC_TMR_PATTERN_T,
	WMI_TAG_WOW_IOAC_ADD_KEEPALIVE_CMD,
	WMI_TAG_WOW_IOAC_DEL_KEEPALIVE_CMD,
	WMI_TAG_WOW_IOAC_KEEPALIVE_T,
	WMI_TAG_WOW_IOAC_ADD_PATTERN_CMD,
	WMI_TAG_WOW_IOAC_DEL_PATTERN_CMD,
	WMI_TAG_START_LINK_STATS_CMD,
	WMI_TAG_CLEAR_LINK_STATS_CMD,
	WMI_TAG_REQUEST_LINK_STATS_CMD,
	WMI_TAG_IFACE_LINK_STATS_EVENT,
	WMI_TAG_RADIO_LINK_STATS_EVENT,
	WMI_TAG_PEER_STATS_EVENT,
	WMI_TAG_CHANNEL_STATS,
	WMI_TAG_RADIO_LINK_STATS,
	WMI_TAG_RATE_STATS,
	WMI_TAG_PEER_LINK_STATS,
	WMI_TAG_WMM_AC_STATS,
	WMI_TAG_IFACE_LINK_STATS,
	WMI_TAG_LPI_MGMT_SNOOPING_CONFIG_CMD,
	WMI_TAG_LPI_START_SCAN_CMD,
	WMI_TAG_LPI_STOP_SCAN_CMD,
	WMI_TAG_LPI_RESULT_EVENT,
	WMI_TAG_PEER_STATE_EVENT,
	WMI_TAG_EXTSCAN_BUCKET_CMD,
	WMI_TAG_EXTSCAN_BUCKET_CHANNEL_EVENT,
	WMI_TAG_EXTSCAN_START_CMD,
	WMI_TAG_EXTSCAN_STOP_CMD,
	WMI_TAG_EXTSCAN_CONFIGURE_WLAN_CHANGE_MONITOR_CMD,
	WMI_TAG_EXTSCAN_WLAN_CHANGE_BSSID_PARAM_CMD,
	WMI_TAG_EXTSCAN_CONFIGURE_HOTLIST_MONITOR_CMD,
	WMI_TAG_EXTSCAN_GET_CACHED_RESULTS_CMD,
	WMI_TAG_EXTSCAN_GET_WLAN_CHANGE_RESULTS_CMD,
	WMI_TAG_EXTSCAN_SET_CAPABILITIES_CMD,
	WMI_TAG_EXTSCAN_GET_CAPABILITIES_CMD,
	WMI_TAG_EXTSCAN_OPERATION_EVENT,
	WMI_TAG_EXTSCAN_START_STOP_EVENT,
	WMI_TAG_EXTSCAN_TABLE_USAGE_EVENT,
	WMI_TAG_EXTSCAN_WLAN_DESCRIPTOR_EVENT,
	WMI_TAG_EXTSCAN_RSSI_INFO_EVENT,
	WMI_TAG_EXTSCAN_CACHED_RESULTS_EVENT,
	WMI_TAG_EXTSCAN_WLAN_CHANGE_RESULTS_EVENT,
	WMI_TAG_EXTSCAN_WLAN_CHANGE_RESULT_BSSID_EVENT,
	WMI_TAG_EXTSCAN_HOTLIST_MATCH_EVENT,
	WMI_TAG_EXTSCAN_CAPABILITIES_EVENT,
	WMI_TAG_EXTSCAN_CACHE_CAPABILITIES_EVENT,
	WMI_TAG_EXTSCAN_WLAN_CHANGE_MONITOR_CAPABILITIES_EVENT,
	WMI_TAG_EXTSCAN_HOTLIST_MONITOR_CAPABILITIES_EVENT,
	WMI_TAG_D0_WOW_ENABLE_DISABLE_CMD,
	WMI_TAG_D0_WOW_DISABLE_ACK_EVENT,
	WMI_TAG_UNIT_TEST_CMD,
	WMI_TAG_ROAM_OFFLOAD_TLV_PARAM,
	WMI_TAG_ROAM_11I_OFFLOAD_TLV_PARAM,
	WMI_TAG_ROAM_11R_OFFLOAD_TLV_PARAM,
	WMI_TAG_ROAM_ESE_OFFLOAD_TLV_PARAM,
	WMI_TAG_ROAM_SYNCH_EVENT,
	WMI_TAG_ROAM_SYNCH_COMPLETE,
	WMI_TAG_EXTWOW_ENABLE_CMD,
	WMI_TAG_EXTWOW_SET_APP_TYPE1_PARAMS_CMD,
	WMI_TAG_EXTWOW_SET_APP_TYPE2_PARAMS_CMD,
	WMI_TAG_LPI_STATUS_EVENT,
	WMI_TAG_LPI_HANDOFF_EVENT,
	WMI_TAG_VDEV_RATE_STATS_EVENT,
	WMI_TAG_VDEV_RATE_HT_INFO,
	WMI_TAG_RIC_REQUEST,
	WMI_TAG_PDEV_GET_TEMPERATURE_CMD,
	WMI_TAG_PDEV_TEMPERATURE_EVENT,
	WMI_TAG_SET_DHCP_SERVER_OFFLOAD_CMD,
	WMI_TAG_TPC_CHAINMASK_CONFIG_CMD,
	WMI_TAG_RIC_TSPEC,
	WMI_TAG_TPC_CHAINMASK_CONFIG,
	WMI_TAG_IPA_OFFLOAD_ENABLE_DISABLE_CMD,
	WMI_TAG_SCAN_PROB_REQ_OUI_CMD,
	WMI_TAG_KEY_MATERIAL,
	WMI_TAG_TDLS_SET_OFFCHAN_MODE_CMD,
	WMI_TAG_SET_LED_FLASHING_CMD,
	WMI_TAG_MDNS_OFFLOAD_CMD,
	WMI_TAG_MDNS_SET_FQDN_CMD,
	WMI_TAG_MDNS_SET_RESP_CMD,
	WMI_TAG_MDNS_GET_STATS_CMD,
	WMI_TAG_MDNS_STATS_EVENT,
	WMI_TAG_ROAM_INVOKE_CMD,
	WMI_TAG_PDEV_RESUME_EVENT,
	WMI_TAG_PDEV_SET_ANTENNA_DIVERSITY_CMD,
	WMI_TAG_SAP_OFL_ENABLE_CMD,
	WMI_TAG_SAP_OFL_ADD_STA_EVENT,
	WMI_TAG_SAP_OFL_DEL_STA_EVENT,
	WMI_TAG_APFIND_CMD_PARAM,
	WMI_TAG_APFIND_EVENT_HDR,
	WMI_TAG_OCB_SET_SCHED_CMD,
	WMI_TAG_OCB_SET_SCHED_EVENT,
	WMI_TAG_OCB_SET_CONFIG_CMD,
	WMI_TAG_OCB_SET_CONFIG_RESP_EVENT,
	WMI_TAG_OCB_SET_UTC_TIME_CMD,
	WMI_TAG_OCB_START_TIMING_ADVERT_CMD,
	WMI_TAG_OCB_STOP_TIMING_ADVERT_CMD,
	WMI_TAG_OCB_GET_TSF_TIMER_CMD,
	WMI_TAG_OCB_GET_TSF_TIMER_RESP_EVENT,
	WMI_TAG_DCC_GET_STATS_CMD,
	WMI_TAG_DCC_CHANNEL_STATS_REQUEST,
	WMI_TAG_DCC_GET_STATS_RESP_EVENT,
	WMI_TAG_DCC_CLEAR_STATS_CMD,
	WMI_TAG_DCC_UPDATE_NDL_CMD,
	WMI_TAG_DCC_UPDATE_NDL_RESP_EVENT,
	WMI_TAG_DCC_STATS_EVENT,
	WMI_TAG_OCB_CHANNEL,
	WMI_TAG_OCB_SCHEDULE_ELEMENT,
	WMI_TAG_DCC_NDL_STATS_PER_CHANNEL,
	WMI_TAG_DCC_NDL_CHAN,
	WMI_TAG_QOS_PARAMETER,
	WMI_TAG_DCC_NDL_ACTIVE_STATE_CONFIG,
	WMI_TAG_ROAM_SCAN_EXTENDED_THRESHOLD_PARAM,
	WMI_TAG_ROAM_FILTER,
	WMI_TAG_PASSPOINT_CONFIG_CMD,
	WMI_TAG_PASSPOINT_EVENT_HDR,
	WMI_TAG_EXTSCAN_CONFIGURE_HOTLIST_SSID_MONITOR_CMD,
	WMI_TAG_EXTSCAN_HOTLIST_SSID_MATCH_EVENT,
	WMI_TAG_VDEV_TSF_TSTAMP_ACTION_CMD,
	WMI_TAG_VDEV_TSF_REPORT_EVENT,
	WMI_TAG_GET_FW_MEM_DUMP,
	WMI_TAG_UPDATE_FW_MEM_DUMP,
	WMI_TAG_FW_MEM_DUMP_PARAMS,
	WMI_TAG_DEBUG_MESG_FLUSH,
	WMI_TAG_DEBUG_MESG_FLUSH_COMPLETE,
	WMI_TAG_PEER_SET_RATE_REPORT_CONDITION,
	WMI_TAG_ROAM_SUBNET_CHANGE_CONFIG,
	WMI_TAG_VDEV_SET_IE_CMD,
	WMI_TAG_RSSI_BREACH_MONITOR_CONFIG,
	WMI_TAG_RSSI_BREACH_EVENT,
	WMI_TAG_WOW_EVENT_INITIAL_WAKEUP,
	WMI_TAG_SOC_SET_PCL_CMD,
	WMI_TAG_SOC_SET_HW_MODE_CMD,
	WMI_TAG_SOC_SET_HW_MODE_RESPONSE_EVENT,
	WMI_TAG_SOC_HW_MODE_TRANSITION_EVENT,
	WMI_TAG_VDEV_TXRX_STREAMS,
	WMI_TAG_SOC_SET_HW_MODE_RESPONSE_VDEV_MAC_ENTRY,
	WMI_TAG_SOC_SET_DUAL_MAC_CONFIG_CMD,
	WMI_TAG_SOC_SET_DUAL_MAC_CONFIG_RESPONSE_EVENT,
	WMI_TAG_WOW_IOAC_SOCK_PATTERN_T,
	WMI_TAG_WOW_ENABLE_ICMPV6_NA_FLT_CMD,
	WMI_TAG_DIAG_EVENT_LOG_CONFIG,
	WMI_TAG_DIAG_EVENT_LOG_SUPPORTED_EVENT_FIXED_PARAMS,
	WMI_TAG_PACKET_FILTER_CONFIG,
	WMI_TAG_PACKET_FILTER_ENABLE,
	WMI_TAG_SAP_SET_BLACKLIST_PARAM_CMD,
	WMI_TAG_MGMT_TX_SEND_CMD,
	WMI_TAG_MGMT_TX_COMPL_EVENT,
	WMI_TAG_SOC_SET_ANTENNA_MODE_CMD,
	WMI_TAG_WOW_UDP_SVC_OFLD_CMD,
	WMI_TAG_LRO_INFO_CMD,
	WMI_TAG_ROAM_EARLYSTOP_RSSI_THRES_PARAM,
	WMI_TAG_SERVICE_READY_EXT_EVENT,
	WMI_TAG_MAWC_SENSOR_REPORT_IND_CMD,
	WMI_TAG_MAWC_ENABLE_SENSOR_EVENT,
	WMI_TAG_ROAM_CONFIGURE_MAWC_CMD,
	WMI_TAG_NLO_CONFIGURE_MAWC_CMD,
	WMI_TAG_EXTSCAN_CONFIGURE_MAWC_CMD,
	WMI_TAG_PEER_ASSOC_CONF_EVENT,
	WMI_TAG_WOW_HOSTWAKEUP_GPIO_PIN_PATTERN_CONFIG_CMD,
	WMI_TAG_AP_PS_EGAP_PARAM_CMD,
	WMI_TAG_AP_PS_EGAP_INFO_EVENT,
	WMI_TAG_PMF_OFFLOAD_SET_SA_QUERY_CMD,
	WMI_TAG_TRANSFER_DATA_TO_FLASH_CMD,
	WMI_TAG_TRANSFER_DATA_TO_FLASH_COMPLETE_EVENT,
	WMI_TAG_SCPC_EVENT,
	WMI_TAG_AP_PS_EGAP_INFO_CHAINMASK_LIST,
	WMI_TAG_STA_SMPS_FORCE_MODE_COMPLETE_EVENT,
	WMI_TAG_BPF_GET_CAPABILITY_CMD,
	WMI_TAG_BPF_CAPABILITY_INFO_EVT,
	WMI_TAG_BPF_GET_VDEV_STATS_CMD,
	WMI_TAG_BPF_VDEV_STATS_INFO_EVT,
	WMI_TAG_BPF_SET_VDEV_INSTRUCTIONS_CMD,
	WMI_TAG_BPF_DEL_VDEV_INSTRUCTIONS_CMD,
	WMI_TAG_VDEV_DELETE_RESP_EVENT,
	WMI_TAG_PEER_DELETE_RESP_EVENT,
	WMI_TAG_ROAM_DENSE_THRES_PARAM,
	WMI_TAG_ENLO_CANDIDATE_SCORE_PARAM,
	WMI_TAG_PEER_UPDATE_WDS_ENTRY_CMD,
	WMI_TAG_VDEV_CONFIG_RATEMASK,
	WMI_TAG_PDEV_FIPS_CMD,
	WMI_TAG_PDEV_SMART_ANT_ENABLE_CMD,
	WMI_TAG_PDEV_SMART_ANT_SET_RX_ANTENNA_CMD,
	WMI_TAG_PEER_SMART_ANT_SET_TX_ANTENNA_CMD,
	WMI_TAG_PEER_SMART_ANT_SET_TRAIN_ANTENNA_CMD,
	WMI_TAG_PEER_SMART_ANT_SET_NODE_CONFIG_OPS_CMD,
	WMI_TAG_PDEV_SET_ANT_SWITCH_TBL_CMD,
	WMI_TAG_PDEV_SET_CTL_TABLE_CMD,
	WMI_TAG_PDEV_SET_MIMOGAIN_TABLE_CMD,
	WMI_TAG_FWTEST_SET_PARAM_CMD,
	WMI_TAG_PEER_ATF_REQUEST,
	WMI_TAG_VDEV_ATF_REQUEST,
	WMI_TAG_PDEV_GET_ANI_CCK_CONFIG_CMD,
	WMI_TAG_PDEV_GET_ANI_OFDM_CONFIG_CMD,
	WMI_TAG_INST_RSSI_STATS_RESP,
	WMI_TAG_MED_UTIL_REPORT_EVENT,
	WMI_TAG_PEER_STA_PS_STATECHANGE_EVENT,
	WMI_TAG_WDS_ADDR_EVENT,
	WMI_TAG_PEER_RATECODE_LIST_EVENT,
	WMI_TAG_PDEV_NFCAL_POWER_ALL_CHANNELS_EVENT,
	WMI_TAG_PDEV_TPC_EVENT,
	WMI_TAG_ANI_OFDM_EVENT,
	WMI_TAG_ANI_CCK_EVENT,
	WMI_TAG_PDEV_CHANNEL_HOPPING_EVENT,
	WMI_TAG_PDEV_FIPS_EVENT,
	WMI_TAG_ATF_PEER_INFO,
	WMI_TAG_PDEV_GET_TPC_CMD,
	WMI_TAG_VDEV_FILTER_NRP_CONFIG_CMD,
	WMI_TAG_QBOOST_CFG_CMD,
	WMI_TAG_PDEV_SMART_ANT_GPIO_HANDLE,
	WMI_TAG_PEER_SMART_ANT_SET_TX_ANTENNA_SERIES,
	WMI_TAG_PEER_SMART_ANT_SET_TRAIN_ANTENNA_PARAM,
	WMI_TAG_PDEV_SET_ANT_CTRL_CHAIN,
	WMI_TAG_PEER_CCK_OFDM_RATE_INFO,
	WMI_TAG_PEER_MCS_RATE_INFO,
	WMI_TAG_PDEV_NFCAL_POWER_ALL_CHANNELS_NFDBR,
	WMI_TAG_PDEV_NFCAL_POWER_ALL_CHANNELS_NFDBM,
	WMI_TAG_PDEV_NFCAL_POWER_ALL_CHANNELS_FREQNUM,
	WMI_TAG_MU_REPORT_TOTAL_MU,
	WMI_TAG_VDEV_SET_DSCP_TID_MAP_CMD,
	WMI_TAG_ROAM_SET_MBO,
	WMI_TAG_MIB_STATS_ENABLE_CMD,
	WMI_TAG_NAN_DISC_IFACE_CREATED_EVENT,
	WMI_TAG_NAN_DISC_IFACE_DELETED_EVENT,
	WMI_TAG_NAN_STARTED_CLUSTER_EVENT,
	WMI_TAG_NAN_JOINED_CLUSTER_EVENT,
	WMI_TAG_NDI_GET_CAP_REQ,
	WMI_TAG_NDP_INITIATOR_REQ,
	WMI_TAG_NDP_RESPONDER_REQ,
	WMI_TAG_NDP_END_REQ,
	WMI_TAG_NDI_CAP_RSP_EVENT,
	WMI_TAG_NDP_INITIATOR_RSP_EVENT,
	WMI_TAG_NDP_RESPONDER_RSP_EVENT,
	WMI_TAG_NDP_END_RSP_EVENT,
	WMI_TAG_NDP_INDICATION_EVENT,
	WMI_TAG_NDP_CONFIRM_EVENT,
	WMI_TAG_NDP_END_INDICATION_EVENT,
	WMI_TAG_VDEV_SET_QUIET_CMD,
	WMI_TAG_PDEV_SET_PCL_CMD,
	WMI_TAG_PDEV_SET_HW_MODE_CMD,
	WMI_TAG_PDEV_SET_MAC_CONFIG_CMD,
	WMI_TAG_PDEV_SET_ANTENNA_MODE_CMD,
	WMI_TAG_PDEV_SET_HW_MODE_RESPONSE_EVENT,
	WMI_TAG_PDEV_HW_MODE_TRANSITION_EVENT,
	WMI_TAG_PDEV_SET_HW_MODE_RESPONSE_VDEV_MAC_ENTRY,
	WMI_TAG_PDEV_SET_MAC_CONFIG_RESPONSE_EVENT,
	WMI_TAG_COEX_CONFIG_CMD,
	WMI_TAG_CONFIG_ENHANCED_MCAST_FILTER,
	WMI_TAG_CHAN_AVOID_RPT_ALLOW_CMD,
	WMI_TAG_SET_PERIODIC_CHANNEL_STATS_CONFIG,
	WMI_TAG_VDEV_SET_CUSTOM_AGGR_SIZE_CMD,
	WMI_TAG_PDEV_WAL_POWER_DEBUG_CMD,
	WMI_TAG_MAC_PHY_CAPABILITIES,
	WMI_TAG_HW_MODE_CAPABILITIES,
	WMI_TAG_SOC_MAC_PHY_HW_MODE_CAPS,
	WMI_TAG_HAL_REG_CAPABILITIES_EXT,
	WMI_TAG_SOC_HAL_REG_CAPABILITIES,
	WMI_TAG_VDEV_WISA_CMD,
	WMI_TAG_TX_POWER_LEVEL_STATS_EVT,
	WMI_TAG_SCAN_ADAPTIVE_DWELL_PARAMETERS_TLV,
	WMI_TAG_SCAN_ADAPTIVE_DWELL_CONFIG,
	WMI_TAG_WOW_SET_ACTION_WAKE_UP_CMD,
	WMI_TAG_NDP_END_RSP_PER_NDI,
	WMI_TAG_PEER_BWF_REQUEST,
	WMI_TAG_BWF_PEER_INFO,
	WMI_TAG_DBGLOG_TIME_STAMP_SYNC_CMD,
	WMI_TAG_RMC_SET_LEADER_CMD,
	WMI_TAG_RMC_MANUAL_LEADER_EVENT,
	WMI_TAG_PER_CHAIN_RSSI_STATS,
	WMI_TAG_RSSI_STATS,
	WMI_TAG_P2P_LO_START_CMD,
	WMI_TAG_P2P_LO_STOP_CMD,
	WMI_TAG_P2P_LO_STOPPED_EVENT,
	WMI_TAG_REORDER_QUEUE_SETUP_CMD,
	WMI_TAG_REORDER_QUEUE_REMOVE_CMD,
	WMI_TAG_SET_MULTIPLE_MCAST_FILTER_CMD,
	WMI_TAG_MGMT_TX_COMPL_BUNDLE_EVENT,
	WMI_TAG_READ_DATA_FROM_FLASH_CMD,
	WMI_TAG_READ_DATA_FROM_FLASH_EVENT,
	WMI_TAG_PDEV_SET_REORDER_TIMEOUT_VAL_CMD,
	WMI_TAG_PEER_SET_RX_BLOCKSIZE_CMD,
	WMI_TAG_PDEV_SET_WAKEUP_CONFIG_CMDID,
	WMI_TAG_TLV_BUF_LEN_PARAM,
	WMI_TAG_SERVICE_AVAILABLE_EVENT,
	WMI_TAG_PEER_ANTDIV_INFO_REQ_CMD,
	WMI_TAG_PEER_ANTDIV_INFO_EVENT,
	WMI_TAG_PEER_ANTDIV_INFO,
	WMI_TAG_PDEV_GET_ANTDIV_STATUS_CMD,
	WMI_TAG_PDEV_ANTDIV_STATUS_EVENT,
	WMI_TAG_MNT_FILTER_CMD,
	WMI_TAG_GET_CHIP_POWER_STATS_CMD,
	WMI_TAG_PDEV_CHIP_POWER_STATS_EVENT,
	WMI_TAG_COEX_GET_ANTENNA_ISOLATION_CMD,
	WMI_TAG_COEX_REPORT_ISOLATION_EVENT,
	WMI_TAG_CHAN_CCA_STATS,
	WMI_TAG_PEER_SIGNAL_STATS,
	WMI_TAG_TX_STATS,
	WMI_TAG_PEER_AC_TX_STATS,
	WMI_TAG_RX_STATS,
	WMI_TAG_PEER_AC_RX_STATS,
	WMI_TAG_REPORT_STATS_EVENT,
	WMI_TAG_CHAN_CCA_STATS_THRESH,
	WMI_TAG_PEER_SIGNAL_STATS_THRESH,
	WMI_TAG_TX_STATS_THRESH,
	WMI_TAG_RX_STATS_THRESH,
	WMI_TAG_PDEV_SET_STATS_THRESHOLD_CMD,
	WMI_TAG_REQUEST_WLAN_STATS_CMD,
	WMI_TAG_RX_AGGR_FAILURE_EVENT,
	WMI_TAG_RX_AGGR_FAILURE_INFO,
	WMI_TAG_VDEV_ENCRYPT_DECRYPT_DATA_REQ_CMD,
	WMI_TAG_VDEV_ENCRYPT_DECRYPT_DATA_RESP_EVENT,
	WMI_TAG_PDEV_BAND_TO_MAC,
	WMI_TAG_TBTT_OFFSET_INFO,
	WMI_TAG_TBTT_OFFSET_EXT_EVENT,
	WMI_TAG_SAR_LIMITS_CMD,
	WMI_TAG_SAR_LIMIT_CMD_ROW,
	WMI_TAG_PDEV_DFS_PHYERR_OFFLOAD_ENABLE_CMD,
	WMI_TAG_PDEV_DFS_PHYERR_OFFLOAD_DISABLE_CMD,
	WMI_TAG_VDEV_ADFS_CH_CFG_CMD,
	WMI_TAG_VDEV_ADFS_OCAC_ABORT_CMD,
	WMI_TAG_PDEV_DFS_RADAR_DETECTION_EVENT,
	WMI_TAG_VDEV_ADFS_OCAC_COMPLETE_EVENT,
	WMI_TAG_VDEV_DFS_CAC_COMPLETE_EVENT,
	WMI_TAG_VENDOR_OUI,
	WMI_TAG_REQUEST_RCPI_CMD,
	WMI_TAG_UPDATE_RCPI_EVENT,
	WMI_TAG_REQUEST_PEER_STATS_INFO_CMD,
	WMI_TAG_PEER_STATS_INFO,
	WMI_TAG_PEER_STATS_INFO_EVENT,
	WMI_TAG_PKGID_EVENT,
	WMI_TAG_CONNECTED_NLO_RSSI_PARAMS,
	WMI_TAG_SET_CURRENT_COUNTRY_CMD,
	WMI_TAG_REGULATORY_RULE_STRUCT,
	WMI_TAG_REG_CHAN_LIST_CC_EVENT,
	WMI_TAG_11D_SCAN_START_CMD,
	WMI_TAG_11D_SCAN_STOP_CMD,
	WMI_TAG_11D_NEW_COUNTRY_EVENT,
	WMI_TAG_REQUEST_RADIO_CHAN_STATS_CMD,
	WMI_TAG_RADIO_CHAN_STATS,
	WMI_TAG_RADIO_CHAN_STATS_EVENT,
	WMI_TAG_ROAM_PER_CONFIG,
	WMI_TAG_VDEV_ADD_MAC_ADDR_TO_RX_FILTER_CMD,
	WMI_TAG_VDEV_ADD_MAC_ADDR_TO_RX_FILTER_STATUS_EVENT,
	WMI_TAG_BPF_SET_VDEV_ACTIVE_MODE_CMD,
	WMI_TAG_HW_DATA_FILTER_CMD,
	WMI_TAG_CONNECTED_NLO_BSS_BAND_RSSI_PREF,
	WMI_TAG_PEER_OPER_MODE_CHANGE_EVENT,
	WMI_TAG_CHIP_POWER_SAVE_FAILURE_DETECTED,
	WMI_TAG_PDEV_MULTIPLE_VDEV_RESTART_REQUEST_CMD,
	WMI_TAG_PDEV_CSA_SWITCH_COUNT_STATUS_EVENT,
	WMI_TAG_PDEV_UPDATE_PKT_ROUTING_CMD,
	WMI_TAG_PDEV_CHECK_CAL_VERSION_CMD,
	WMI_TAG_PDEV_CHECK_CAL_VERSION_EVENT,
	WMI_TAG_PDEV_SET_DIVERSITY_GAIN_CMD,
	WMI_TAG_MAC_PHY_CHAINMASK_COMBO,
	WMI_TAG_MAC_PHY_CHAINMASK_CAPABILITY,
	WMI_TAG_VDEV_SET_ARP_STATS_CMD,
	WMI_TAG_VDEV_GET_ARP_STATS_CMD,
	WMI_TAG_VDEV_GET_ARP_STATS_EVENT,
	WMI_TAG_IFACE_OFFLOAD_STATS,
	WMI_TAG_REQUEST_STATS_CMD_SUB_STRUCT_PARAM,
	WMI_TAG_RSSI_CTL_EXT,
	WMI_TAG_SINGLE_PHYERR_EXT_RX_HDR,
	WMI_TAG_COEX_BT_ACTIVITY_EVENT,
	WMI_TAG_VDEV_GET_TX_POWER_CMD,
	WMI_TAG_VDEV_TX_POWER_EVENT,
	WMI_TAG_OFFCHAN_DATA_TX_COMPL_EVENT,
	WMI_TAG_OFFCHAN_DATA_TX_SEND_CMD,
	WMI_TAG_TX_SEND_PARAMS,
	WMI_TAG_HE_RATE_SET,
	WMI_TAG_CONGESTION_STATS,
	WMI_TAG_SET_INIT_COUNTRY_CMD,
	WMI_TAG_SCAN_DBS_DUTY_CYCLE,
	WMI_TAG_SCAN_DBS_DUTY_CYCLE_PARAM_TLV,
	WMI_TAG_PDEV_DIV_GET_RSSI_ANTID,
	WMI_TAG_THERM_THROT_CONFIG_REQUEST,
	WMI_TAG_THERM_THROT_LEVEL_CONFIG_INFO,
	WMI_TAG_THERM_THROT_STATS_EVENT,
	WMI_TAG_THERM_THROT_LEVEL_STATS_INFO,
	WMI_TAG_PDEV_DIV_RSSI_ANTID_EVENT,
	WMI_TAG_OEM_DMA_RING_CAPABILITIES,
	WMI_TAG_OEM_DMA_RING_CFG_REQ,
	WMI_TAG_OEM_DMA_RING_CFG_RSP,
	WMI_TAG_OEM_INDIRECT_DATA,
	WMI_TAG_OEM_DMA_BUF_RELEASE,
	WMI_TAG_OEM_DMA_BUF_RELEASE_ENTRY,
	WMI_TAG_PDEV_BSS_CHAN_INFO_REQUEST,
	WMI_TAG_PDEV_BSS_CHAN_INFO_EVENT,
	WMI_TAG_ROAM_LCA_DISALLOW_CONFIG,
	WMI_TAG_VDEV_LIMIT_OFFCHAN_CMD,
	WMI_TAG_ROAM_RSSI_REJECTION_OCE_CONFIG,
	WMI_TAG_UNIT_TEST_EVENT,
	WMI_TAG_ROAM_FILS_OFFLOAD,
	WMI_TAG_PDEV_UPDATE_PMK_CACHE_CMD,
	WMI_TAG_PMK_CACHE,
	WMI_TAG_PDEV_UPDATE_FILS_HLP_PKT_CMD,
	WMI_TAG_ROAM_FILS_SYNCH,
	WMI_TAG_GTK_OFFLOAD_EXTENDED,
	WMI_TAG_ROAM_BG_SCAN_ROAMING,
	WMI_TAG_OIC_PING_OFFLOAD_PARAMS_CMD,
	WMI_TAG_OIC_PING_OFFLOAD_SET_ENABLE_CMD,
	WMI_TAG_OIC_PING_HANDOFF_EVENT,
	WMI_TAG_DHCP_LEASE_RENEW_OFFLOAD_CMD,
	WMI_TAG_DHCP_LEASE_RENEW_EVENT,
	WMI_TAG_BTM_CONFIG,
	WMI_TAG_DEBUG_MESG_FW_DATA_STALL,
	WMI_TAG_WLM_CONFIG_CMD,
	WMI_TAG_PDEV_UPDATE_CTLTABLE_REQUEST,
	WMI_TAG_PDEV_UPDATE_CTLTABLE_EVENT,
	WMI_TAG_ROAM_CND_SCORING_PARAM,
	WMI_TAG_PDEV_CONFIG_VENDOR_OUI_ACTION,
	WMI_TAG_VENDOR_OUI_EXT,
	WMI_TAG_ROAM_SYNCH_FRAME_EVENT,
	WMI_TAG_FD_SEND_FROM_HOST_CMD,
	WMI_TAG_ENABLE_FILS_CMD,
	WMI_TAG_HOST_SWFDA_EVENT,
	WMI_TAG_BCN_OFFLOAD_CTRL_CMD,
	WMI_TAG_PDEV_SET_AC_TX_QUEUE_OPTIMIZED_CMD,
	WMI_TAG_STATS_PERIOD,
	WMI_TAG_NDL_SCHEDULE_UPDATE,
	WMI_TAG_PEER_TID_MSDUQ_QDEPTH_THRESH_UPDATE_CMD,
	WMI_TAG_MSDUQ_QDEPTH_THRESH_UPDATE,
	WMI_TAG_PDEV_SET_RX_FILTER_PROMISCUOUS_CMD,
	WMI_TAG_SAR2_RESULT_EVENT,
	WMI_TAG_SAR_CAPABILITIES,
	WMI_TAG_SAP_OBSS_DETECTION_CFG_CMD,
	WMI_TAG_SAP_OBSS_DETECTION_INFO_EVT,
	WMI_TAG_DMA_RING_CAPABILITIES,
	WMI_TAG_DMA_RING_CFG_REQ,
	WMI_TAG_DMA_RING_CFG_RSP,
	WMI_TAG_DMA_BUF_RELEASE,
	WMI_TAG_DMA_BUF_RELEASE_ENTRY,
	WMI_TAG_SAR_GET_LIMITS_CMD,
	WMI_TAG_SAR_GET_LIMITS_EVENT,
	WMI_TAG_SAR_GET_LIMITS_EVENT_ROW,
	WMI_TAG_OFFLOAD_11K_REPORT,
	WMI_TAG_INVOKE_NEIGHBOR_REPORT,
	WMI_TAG_NEIGHBOR_REPORT_OFFLOAD,
	WMI_TAG_VDEV_SET_CONNECTIVITY_CHECK_STATS,
	WMI_TAG_VDEV_GET_CONNECTIVITY_CHECK_STATS,
	WMI_TAG_BPF_SET_VDEV_ENABLE_CMD,
	WMI_TAG_BPF_SET_VDEV_WORK_MEMORY_CMD,
	WMI_TAG_BPF_GET_VDEV_WORK_MEMORY_CMD,
	WMI_TAG_BPF_GET_VDEV_WORK_MEMORY_RESP_EVT,
	WMI_TAG_PDEV_GET_NFCAL_POWER,
	WMI_TAG_BSS_COLOR_CHANGE_ENABLE,
	WMI_TAG_OBSS_COLOR_COLLISION_DET_CONFIG,
	WMI_TAG_OBSS_COLOR_COLLISION_EVT,
	WMI_TAG_RUNTIME_DPD_RECAL_CMD,
	WMI_TAG_TWT_ENABLE_CMD,
	WMI_TAG_TWT_DISABLE_CMD,
	WMI_TAG_TWT_ADD_DIALOG_CMD,
	WMI_TAG_TWT_DEL_DIALOG_CMD,
	WMI_TAG_TWT_PAUSE_DIALOG_CMD,
	WMI_TAG_TWT_RESUME_DIALOG_CMD,
	WMI_TAG_TWT_ENABLE_COMPLETE_EVENT,
	WMI_TAG_TWT_DISABLE_COMPLETE_EVENT,
	WMI_TAG_TWT_ADD_DIALOG_COMPLETE_EVENT,
	WMI_TAG_TWT_DEL_DIALOG_COMPLETE_EVENT,
	WMI_TAG_TWT_PAUSE_DIALOG_COMPLETE_EVENT,
	WMI_TAG_TWT_RESUME_DIALOG_COMPLETE_EVENT,
	WMI_TAG_REQUEST_ROAM_SCAN_STATS_CMD,
	WMI_TAG_ROAM_SCAN_STATS_EVENT,
	WMI_TAG_PEER_TID_CONFIGURATIONS_CMD,
	WMI_TAG_VDEV_SET_CUSTOM_SW_RETRY_TH_CMD,
	WMI_TAG_GET_TPC_POWER_CMD,
	WMI_TAG_GET_TPC_POWER_EVENT,
	WMI_TAG_DMA_BUF_RELEASE_SPECTRAL_META_DATA,
	WMI_TAG_MOTION_DET_CONFIG_PARAMS_CMD,
	WMI_TAG_MOTION_DET_BASE_LINE_CONFIG_PARAMS_CMD,
	WMI_TAG_MOTION_DET_START_STOP_CMD,
	WMI_TAG_MOTION_DET_BASE_LINE_START_STOP_CMD,
	WMI_TAG_MOTION_DET_EVENT,
	WMI_TAG_MOTION_DET_BASE_LINE_EVENT,
	WMI_TAG_NDP_TRANSPORT_IP,
	WMI_TAG_OBSS_SPATIAL_REUSE_SET_CMD,
	WMI_TAG_ESP_ESTIMATE_EVENT,
	WMI_TAG_NAN_HOST_CONFIG,
	WMI_TAG_SPECTRAL_BIN_SCALING_PARAMS,
	WMI_TAG_PEER_CFR_CAPTURE_CMD,
	WMI_TAG_PEER_CHAN_WIDTH_SWITCH_CMD,
	WMI_TAG_CHAN_WIDTH_PEER_LIST,
	WMI_TAG_OBSS_SPATIAL_REUSE_SET_DEF_OBSS_THRESH_CMD,
	WMI_TAG_PDEV_HE_TB_ACTION_FRM_CMD,
	WMI_TAG_PEER_EXTD2_STATS,
	WMI_TAG_HPCS_PULSE_START_CMD,
	WMI_TAG_PDEV_CTL_FAILSAFE_CHECK_EVENT,
	WMI_TAG_VDEV_CHAINMASK_CONFIG_CMD,
	WMI_TAG_VDEV_BCN_OFFLOAD_QUIET_CONFIG_CMD,
	WMI_TAG_NAN_EVENT_INFO,
	WMI_TAG_NDP_CHANNEL_INFO,
	WMI_TAG_NDP_CMD,
	WMI_TAG_NDP_EVENT,
	/* TODO add all the missing cmds */
	WMI_TAG_PDEV_PEER_PKTLOG_FILTER_CMD = 0x301,
	WMI_TAG_PDEV_PEER_PKTLOG_FILTER_INFO,
	WMI_TAG_SERVICE_READY_EXT2_EVENT = 0x334,
	WMI_TAG_FILS_DISCOVERY_TMPL_CMD = 0x344,
	WMI_TAG_MAC_PHY_CAPABILITIES_EXT = 0x36F,
	WMI_TAG_REGULATORY_RULE_EXT_STRUCT = 0x3A9,
	WMI_TAG_REG_CHAN_LIST_CC_EXT_EVENT,
	WMI_TAG_EHT_RATE_SET = 0x3C4,
	WMI_TAG_MAX
};

enum wmi_tlv_service {
	WMI_TLV_SERVICE_BEACON_OFFLOAD = 0,
	WMI_TLV_SERVICE_SCAN_OFFLOAD = 1,
	WMI_TLV_SERVICE_ROAM_SCAN_OFFLOAD = 2,
	WMI_TLV_SERVICE_BCN_MISS_OFFLOAD = 3,
	WMI_TLV_SERVICE_STA_PWRSAVE = 4,
	WMI_TLV_SERVICE_STA_ADVANCED_PWRSAVE = 5,
	WMI_TLV_SERVICE_AP_UAPSD = 6,
	WMI_TLV_SERVICE_AP_DFS = 7,
	WMI_TLV_SERVICE_11AC = 8,
	WMI_TLV_SERVICE_BLOCKACK = 9,
	WMI_TLV_SERVICE_PHYERR = 10,
	WMI_TLV_SERVICE_BCN_FILTER = 11,
	WMI_TLV_SERVICE_RTT = 12,
	WMI_TLV_SERVICE_WOW = 13,
	WMI_TLV_SERVICE_RATECTRL_CACHE = 14,
	WMI_TLV_SERVICE_IRAM_TIDS = 15,
	WMI_TLV_SERVICE_ARPNS_OFFLOAD = 16,
	WMI_TLV_SERVICE_NLO = 17,
	WMI_TLV_SERVICE_GTK_OFFLOAD = 18,
	WMI_TLV_SERVICE_SCAN_SCH = 19,
	WMI_TLV_SERVICE_CSA_OFFLOAD = 20,
	WMI_TLV_SERVICE_CHATTER = 21,
	WMI_TLV_SERVICE_COEX_FREQAVOID = 22,
	WMI_TLV_SERVICE_PACKET_POWER_SAVE = 23,
	WMI_TLV_SERVICE_FORCE_FW_HANG = 24,
	WMI_TLV_SERVICE_GPIO = 25,
	WMI_TLV_SERVICE_STA_DTIM_PS_MODULATED_DTIM = 26,
	WMI_STA_UAPSD_BASIC_AUTO_TRIG = 27,
	WMI_STA_UAPSD_VAR_AUTO_TRIG = 28,
	WMI_TLV_SERVICE_STA_KEEP_ALIVE = 29,
	WMI_TLV_SERVICE_TX_ENCAP = 30,
	WMI_TLV_SERVICE_AP_PS_DETECT_OUT_OF_SYNC = 31,
	WMI_TLV_SERVICE_EARLY_RX = 32,
	WMI_TLV_SERVICE_STA_SMPS = 33,
	WMI_TLV_SERVICE_FWTEST = 34,
	WMI_TLV_SERVICE_STA_WMMAC = 35,
	WMI_TLV_SERVICE_TDLS = 36,
	WMI_TLV_SERVICE_BURST = 37,
	WMI_TLV_SERVICE_MCC_BCN_INTERVAL_CHANGE = 38,
	WMI_TLV_SERVICE_ADAPTIVE_OCS = 39,
	WMI_TLV_SERVICE_BA_SSN_SUPPORT = 40,
	WMI_TLV_SERVICE_FILTER_IPSEC_NATKEEPALIVE = 41,
	WMI_TLV_SERVICE_WLAN_HB = 42,
	WMI_TLV_SERVICE_LTE_ANT_SHARE_SUPPORT = 43,
	WMI_TLV_SERVICE_BATCH_SCAN = 44,
	WMI_TLV_SERVICE_QPOWER = 45,
	WMI_TLV_SERVICE_PLMREQ = 46,
	WMI_TLV_SERVICE_THERMAL_MGMT = 47,
	WMI_TLV_SERVICE_RMC = 48,
	WMI_TLV_SERVICE_MHF_OFFLOAD = 49,
	WMI_TLV_SERVICE_COEX_SAR = 50,
	WMI_TLV_SERVICE_BCN_TXRATE_OVERRIDE = 51,
	WMI_TLV_SERVICE_NAN = 52,
	WMI_TLV_SERVICE_L1SS_STAT = 53,
	WMI_TLV_SERVICE_ESTIMATE_LINKSPEED = 54,
	WMI_TLV_SERVICE_OBSS_SCAN = 55,
	WMI_TLV_SERVICE_TDLS_OFFCHAN = 56,
	WMI_TLV_SERVICE_TDLS_UAPSD_BUFFER_STA = 57,
	WMI_TLV_SERVICE_TDLS_UAPSD_SLEEP_STA = 58,
	WMI_TLV_SERVICE_IBSS_PWRSAVE = 59,
	WMI_TLV_SERVICE_LPASS = 60,
	WMI_TLV_SERVICE_EXTSCAN = 61,
	WMI_TLV_SERVICE_D0WOW = 62,
	WMI_TLV_SERVICE_HSOFFLOAD = 63,
	WMI_TLV_SERVICE_ROAM_HO_OFFLOAD = 64,
	WMI_TLV_SERVICE_RX_FULL_REORDER = 65,
	WMI_TLV_SERVICE_DHCP_OFFLOAD = 66,
	WMI_TLV_SERVICE_STA_RX_IPA_OFFLOAD_SUPPORT = 67,
	WMI_TLV_SERVICE_MDNS_OFFLOAD = 68,
	WMI_TLV_SERVICE_SAP_AUTH_OFFLOAD = 69,
	WMI_TLV_SERVICE_DUAL_BAND_SIMULTANEOUS_SUPPORT = 70,
	WMI_TLV_SERVICE_OCB = 71,
	WMI_TLV_SERVICE_AP_ARPNS_OFFLOAD = 72,
	WMI_TLV_SERVICE_PER_BAND_CHAINMASK_SUPPORT = 73,
	WMI_TLV_SERVICE_PACKET_FILTER_OFFLOAD = 74,
	WMI_TLV_SERVICE_MGMT_TX_HTT = 75,
	WMI_TLV_SERVICE_MGMT_TX_WMI = 76,
	WMI_TLV_SERVICE_EXT_MSG = 77,
	WMI_TLV_SERVICE_MAWC = 78,
	WMI_TLV_SERVICE_PEER_ASSOC_CONF = 79,
	WMI_TLV_SERVICE_EGAP = 80,
	WMI_TLV_SERVICE_STA_PMF_OFFLOAD = 81,
	WMI_TLV_SERVICE_UNIFIED_WOW_CAPABILITY = 82,
	WMI_TLV_SERVICE_ENHANCED_PROXY_STA = 83,
	WMI_TLV_SERVICE_ATF = 84,
	WMI_TLV_SERVICE_COEX_GPIO = 85,
	WMI_TLV_SERVICE_AUX_SPECTRAL_INTF = 86,
	WMI_TLV_SERVICE_AUX_CHAN_LOAD_INTF = 87,
	WMI_TLV_SERVICE_BSS_CHANNEL_INFO_64 = 88,
	WMI_TLV_SERVICE_ENTERPRISE_MESH = 89,
	WMI_TLV_SERVICE_RESTRT_CHNL_SUPPORT = 90,
	WMI_TLV_SERVICE_BPF_OFFLOAD = 91,
	WMI_TLV_SERVICE_SYNC_DELETE_CMDS = 92,
	WMI_TLV_SERVICE_SMART_ANTENNA_SW_SUPPORT = 93,
	WMI_TLV_SERVICE_SMART_ANTENNA_HW_SUPPORT = 94,
	WMI_TLV_SERVICE_RATECTRL_LIMIT_MAX_MIN_RATES = 95,
	WMI_TLV_SERVICE_NAN_DATA = 96,
	WMI_TLV_SERVICE_NAN_RTT = 97,
	WMI_TLV_SERVICE_11AX = 98,
	WMI_TLV_SERVICE_DEPRECATED_REPLACE = 99,
	WMI_TLV_SERVICE_TDLS_CONN_TRACKER_IN_HOST_MODE = 100,
	WMI_TLV_SERVICE_ENHANCED_MCAST_FILTER = 101,
	WMI_TLV_SERVICE_PERIODIC_CHAN_STAT_SUPPORT = 102,
	WMI_TLV_SERVICE_MESH_11S = 103,
	WMI_TLV_SERVICE_HALF_RATE_QUARTER_RATE_SUPPORT = 104,
	WMI_TLV_SERVICE_VDEV_RX_FILTER = 105,
	WMI_TLV_SERVICE_P2P_LISTEN_OFFLOAD_SUPPORT = 106,
	WMI_TLV_SERVICE_MARK_FIRST_WAKEUP_PACKET = 107,
	WMI_TLV_SERVICE_MULTIPLE_MCAST_FILTER_SET = 108,
	WMI_TLV_SERVICE_HOST_MANAGED_RX_REORDER = 109,
	WMI_TLV_SERVICE_FLASH_RDWR_SUPPORT = 110,
	WMI_TLV_SERVICE_WLAN_STATS_REPORT = 111,
	WMI_TLV_SERVICE_TX_MSDU_ID_NEW_PARTITION_SUPPORT = 112,
	WMI_TLV_SERVICE_DFS_PHYERR_OFFLOAD = 113,
	WMI_TLV_SERVICE_RCPI_SUPPORT = 114,
	WMI_TLV_SERVICE_FW_MEM_DUMP_SUPPORT = 115,
	WMI_TLV_SERVICE_PEER_STATS_INFO = 116,
	WMI_TLV_SERVICE_REGULATORY_DB = 117,
	WMI_TLV_SERVICE_11D_OFFLOAD = 118,
	WMI_TLV_SERVICE_HW_DATA_FILTERING = 119,
	WMI_TLV_SERVICE_MULTIPLE_VDEV_RESTART = 120,
	WMI_TLV_SERVICE_PKT_ROUTING = 121,
	WMI_TLV_SERVICE_CHECK_CAL_VERSION = 122,
	WMI_TLV_SERVICE_OFFCHAN_TX_WMI = 123,
	WMI_TLV_SERVICE_8SS_TX_BFEE  = 124,
	WMI_TLV_SERVICE_EXTENDED_NSS_SUPPORT = 125,
	WMI_TLV_SERVICE_ACK_TIMEOUT = 126,
	WMI_TLV_SERVICE_PDEV_BSS_CHANNEL_INFO_64 = 127,

	WMI_MAX_SERVICE = 128,

	WMI_TLV_SERVICE_CHAN_LOAD_INFO = 128,
	WMI_TLV_SERVICE_TX_PPDU_INFO_STATS_SUPPORT = 129,
	WMI_TLV_SERVICE_VDEV_LIMIT_OFFCHAN_SUPPORT = 130,
	WMI_TLV_SERVICE_FILS_SUPPORT = 131,
	WMI_TLV_SERVICE_WLAN_OIC_PING_OFFLOAD = 132,
	WMI_TLV_SERVICE_WLAN_DHCP_RENEW = 133,
	WMI_TLV_SERVICE_MAWC_SUPPORT = 134,
	WMI_TLV_SERVICE_VDEV_LATENCY_CONFIG = 135,
	WMI_TLV_SERVICE_PDEV_UPDATE_CTLTABLE_SUPPORT = 136,
	WMI_TLV_SERVICE_PKTLOG_SUPPORT_OVER_HTT = 137,
	WMI_TLV_SERVICE_VDEV_MULTI_GROUP_KEY_SUPPORT = 138,
	WMI_TLV_SERVICE_SCAN_PHYMODE_SUPPORT = 139,
	WMI_TLV_SERVICE_THERM_THROT = 140,
	WMI_TLV_SERVICE_BCN_OFFLOAD_START_STOP_SUPPORT = 141,
	WMI_TLV_SERVICE_WOW_WAKEUP_BY_TIMER_PATTERN = 142,
	WMI_TLV_SERVICE_PEER_MAP_UNMAP_V2_SUPPORT = 143,
	WMI_TLV_SERVICE_OFFCHAN_DATA_TID_SUPPORT = 144,
	WMI_TLV_SERVICE_RX_PROMISC_ENABLE_SUPPORT = 145,
	WMI_TLV_SERVICE_SUPPORT_DIRECT_DMA = 146,
	WMI_TLV_SERVICE_AP_OBSS_DETECTION_OFFLOAD = 147,
	WMI_TLV_SERVICE_11K_NEIGHBOUR_REPORT_SUPPORT = 148,
	WMI_TLV_SERVICE_LISTEN_INTERVAL_OFFLOAD_SUPPORT = 149,
	WMI_TLV_SERVICE_BSS_COLOR_OFFLOAD = 150,
	WMI_TLV_SERVICE_RUNTIME_DPD_RECAL = 151,
	WMI_TLV_SERVICE_STA_TWT = 152,
	WMI_TLV_SERVICE_AP_TWT = 153,
	WMI_TLV_SERVICE_GMAC_OFFLOAD_SUPPORT = 154,
	WMI_TLV_SERVICE_SPOOF_MAC_SUPPORT = 155,
	WMI_TLV_SERVICE_PEER_TID_CONFIGS_SUPPORT = 156,
	WMI_TLV_SERVICE_VDEV_SWRETRY_PER_AC_CONFIG_SUPPORT = 157,
	WMI_TLV_SERVICE_DUAL_BEACON_ON_SINGLE_MAC_SCC_SUPPORT = 158,
	WMI_TLV_SERVICE_DUAL_BEACON_ON_SINGLE_MAC_MCC_SUPPORT = 159,
	WMI_TLV_SERVICE_MOTION_DET = 160,
	WMI_TLV_SERVICE_INFRA_MBSSID = 161,
	WMI_TLV_SERVICE_OBSS_SPATIAL_REUSE = 162,
	WMI_TLV_SERVICE_VDEV_DIFFERENT_BEACON_INTERVAL_SUPPORT = 163,
	WMI_TLV_SERVICE_NAN_DBS_SUPPORT = 164,
	WMI_TLV_SERVICE_NDI_DBS_SUPPORT = 165,
	WMI_TLV_SERVICE_NAN_SAP_SUPPORT = 166,
	WMI_TLV_SERVICE_NDI_SAP_SUPPORT = 167,
	WMI_TLV_SERVICE_CFR_CAPTURE_SUPPORT = 168,
	WMI_TLV_SERVICE_CFR_CAPTURE_IND_MSG_TYPE_1 = 169,
	WMI_TLV_SERVICE_ESP_SUPPORT = 170,
	WMI_TLV_SERVICE_PEER_CHWIDTH_CHANGE = 171,
	WMI_TLV_SERVICE_WLAN_HPCS_PULSE = 172,
	WMI_TLV_SERVICE_PER_VDEV_CHAINMASK_CONFIG_SUPPORT = 173,
	WMI_TLV_SERVICE_TX_DATA_MGMT_ACK_RSSI = 174,
	WMI_TLV_SERVICE_NAN_DISABLE_SUPPORT = 175,
	WMI_TLV_SERVICE_HTT_H2T_NO_HTC_HDR_LEN_IN_MSG_LEN = 176,
	WMI_TLV_SERVICE_COEX_SUPPORT_UNEQUAL_ISOLATION = 177,
	WMI_TLV_SERVICE_HW_DB2DBM_CONVERSION_SUPPORT = 178,
	WMI_TLV_SERVICE_SUPPORT_EXTEND_ADDRESS = 179,
	WMI_TLV_SERVICE_BEACON_RECEPTION_STATS = 180,
	WMI_TLV_SERVICE_FETCH_TX_PN = 181,
	WMI_TLV_SERVICE_PEER_UNMAP_RESPONSE_SUPPORT = 182,
	WMI_TLV_SERVICE_TX_PER_PEER_AMPDU_SIZE = 183,
	WMI_TLV_SERVICE_BSS_COLOR_SWITCH_COUNT = 184,
	WMI_TLV_SERVICE_HTT_PEER_STATS_SUPPORT = 185,
	WMI_TLV_SERVICE_UL_RU26_ALLOWED = 186,
	WMI_TLV_SERVICE_GET_MWS_COEX_STATE = 187,
	WMI_TLV_SERVICE_GET_MWS_DPWB_STATE = 188,
	WMI_TLV_SERVICE_GET_MWS_TDM_STATE = 189,
	WMI_TLV_SERVICE_GET_MWS_IDRX_STATE = 190,
	WMI_TLV_SERVICE_GET_MWS_ANTENNA_SHARING_STATE = 191,
	WMI_TLV_SERVICE_ENHANCED_TPC_CONFIG_EVENT = 192,
	WMI_TLV_SERVICE_WLM_STATS_REQUEST = 193,
	WMI_TLV_SERVICE_EXT_PEER_TID_CONFIGS_SUPPORT = 194,
	WMI_TLV_SERVICE_WPA3_FT_SAE_SUPPORT = 195,
	WMI_TLV_SERVICE_WPA3_FT_SUITE_B_SUPPORT = 196,
	WMI_TLV_SERVICE_VOW_ENABLE = 197,
	WMI_TLV_SERVICE_CFR_CAPTURE_IND_EVT_TYPE_1 = 198,
	WMI_TLV_SERVICE_BROADCAST_TWT = 199,
	WMI_TLV_SERVICE_RAP_DETECTION_SUPPORT = 200,
	WMI_TLV_SERVICE_PS_TDCC = 201,
	WMI_TLV_SERVICE_THREE_WAY_COEX_CONFIG_LEGACY   = 202,
	WMI_TLV_SERVICE_THREE_WAY_COEX_CONFIG_OVERRIDE = 203,
	WMI_TLV_SERVICE_TX_PWR_PER_PEER = 204,
	WMI_TLV_SERVICE_STA_PLUS_STA_SUPPORT = 205,
	WMI_TLV_SERVICE_WPA3_FT_FILS = 206,
	WMI_TLV_SERVICE_ADAPTIVE_11R_ROAM = 207,
	WMI_TLV_SERVICE_CHAN_RF_CHARACTERIZATION_INFO = 208,
	WMI_TLV_SERVICE_FW_IFACE_COMBINATION_SUPPORT = 209,
	WMI_TLV_SERVICE_TX_COMPL_TSF64 = 210,
	WMI_TLV_SERVICE_DSM_ROAM_FILTER = 211,
	WMI_TLV_SERVICE_PACKET_CAPTURE_SUPPORT = 212,
	WMI_TLV_SERVICE_PER_PEER_HTT_STATS_RESET = 213,
	WMI_TLV_SERVICE_FREQINFO_IN_METADATA = 219,
	WMI_TLV_SERVICE_EXT2_MSG = 220,

	WMI_MAX_EXT_SERVICE = 256,

	WMI_TLV_SERVICE_REG_CC_EXT_EVENT_SUPPORT = 281,
	WMI_MAX_EXT2_SERVICE,
};

enum {
	WMI_SMPS_FORCED_MODE_NONE = 0,
	WMI_SMPS_FORCED_MODE_DISABLED,
	WMI_SMPS_FORCED_MODE_STATIC,
	WMI_SMPS_FORCED_MODE_DYNAMIC
};

enum wmi_tpc_chainmask {
	WMI_TPC_CHAINMASK_CONFIG_BAND_2G = 0,
	WMI_TPC_CHAINMASK_CONFIG_BAND_5G = 1,
	WMI_NUM_SUPPORTED_BAND_MAX = 2,
};

enum wmi_peer_param {
	WMI_PEER_MIMO_PS_STATE = 1,
	WMI_PEER_AMPDU = 2,
	WMI_PEER_AUTHORIZE = 3,
	WMI_PEER_CHWIDTH = 4,
	WMI_PEER_NSS = 5,
	WMI_PEER_USE_4ADDR = 6,
	WMI_PEER_MEMBERSHIP = 7,
	WMI_PEER_USERPOS = 8,
	WMI_PEER_CRIT_PROTO_HINT_ENABLED = 9,
	WMI_PEER_TX_FAIL_CNT_THR = 10,
	WMI_PEER_SET_HW_RETRY_CTS2S = 11,
	WMI_PEER_IBSS_ATIM_WINDOW_LENGTH = 12,
	WMI_PEER_PHYMODE = 13,
	WMI_PEER_USE_FIXED_PWR = 14,
	WMI_PEER_PARAM_FIXED_RATE = 15,
	WMI_PEER_SET_MU_WHITELIST = 16,
	WMI_PEER_SET_MAX_TX_RATE = 17,
	WMI_PEER_SET_MIN_TX_RATE = 18,
	WMI_PEER_SET_DEFAULT_ROUTING = 19,
};

enum wmi_slot_time {
	WMI_VDEV_SLOT_TIME_LONG = 1,
	WMI_VDEV_SLOT_TIME_SHORT = 2,
};

enum wmi_preamble {
	WMI_VDEV_PREAMBLE_LONG = 1,
	WMI_VDEV_PREAMBLE_SHORT = 2,
};

enum wmi_peer_smps_state {
	WMI_PEER_SMPS_PS_NONE =	0,
	WMI_PEER_SMPS_STATIC  = 1,
	WMI_PEER_SMPS_DYNAMIC = 2
};

enum wmi_peer_chwidth {
	WMI_PEER_CHWIDTH_20MHZ = 0,
	WMI_PEER_CHWIDTH_40MHZ = 1,
	WMI_PEER_CHWIDTH_80MHZ = 2,
	WMI_PEER_CHWIDTH_160MHZ = 3,
};

enum wmi_beacon_gen_mode {
	WMI_BEACON_STAGGERED_MODE = 0,
	WMI_BEACON_BURST_MODE = 1
};

enum wmi_direct_buffer_module {
	WMI_DIRECT_BUF_SPECTRAL = 0,
	WMI_DIRECT_BUF_CFR = 1,

	/* keep it last */
	WMI_DIRECT_BUF_MAX
};

struct ath12k_wmi_pdev_band_arg {
	u32 pdev_id;
	u32 start_freq;
	u32 end_freq;
};

struct ath12k_wmi_ppe_threshold_arg {
	u32 numss_m1;
	u32 ru_bit_mask;
	u32 ppet16_ppet8_ru3_ru0[WMI_MAX_NUM_SS];
};

#define PSOC_HOST_MAX_PHY_SIZE (3)
#define ATH12K_11B_SUPPORT                 BIT(0)
#define ATH12K_11G_SUPPORT                 BIT(1)
#define ATH12K_11A_SUPPORT                 BIT(2)
#define ATH12K_11N_SUPPORT                 BIT(3)
#define ATH12K_11AC_SUPPORT                BIT(4)
#define ATH12K_11AX_SUPPORT                BIT(5)

struct ath12k_wmi_hal_reg_capabilities_ext_arg {
	u32 phy_id;
	u32 eeprom_reg_domain;
	u32 eeprom_reg_domain_ext;
	u32 regcap1;
	u32 regcap2;
	u32 wireless_modes;
	u32 low_2ghz_chan;
	u32 high_2ghz_chan;
	u32 low_5ghz_chan;
	u32 high_5ghz_chan;
};

#define WMI_HOST_MAX_PDEV 3

struct ath12k_wmi_host_mem_chunk_params {
	__le32 tlv_header;
	__le32 req_id;
	__le32 ptr;
	__le32 size;
} __packed;

struct ath12k_wmi_host_mem_chunk_arg {
	void *vaddr;
	dma_addr_t paddr;
	u32 len;
	u32 req_id;
};

struct ath12k_wmi_resource_config_arg {
	u32 num_vdevs;
	u32 num_peers;
	u32 num_active_peers;
	u32 num_offload_peers;
	u32 num_offload_reorder_buffs;
	u32 num_peer_keys;
	u32 num_tids;
	u32 ast_skid_limit;
	u32 tx_chain_mask;
	u32 rx_chain_mask;
	u32 rx_timeout_pri[4];
	u32 rx_decap_mode;
	u32 scan_max_pending_req;
	u32 bmiss_offload_max_vdev;
	u32 roam_offload_max_vdev;
	u32 roam_offload_max_ap_profiles;
	u32 num_mcast_groups;
	u32 num_mcast_table_elems;
	u32 mcast2ucast_mode;
	u32 tx_dbg_log_size;
	u32 num_wds_entries;
	u32 dma_burst_size;
	u32 mac_aggr_delim;
	u32 rx_skip_defrag_timeout_dup_detection_check;
	u32 vow_config;
	u32 gtk_offload_max_vdev;
	u32 num_msdu_desc;
	u32 max_frag_entries;
	u32 max_peer_ext_stats;
	u32 smart_ant_cap;
	u32 bk_minfree;
	u32 be_minfree;
	u32 vi_minfree;
	u32 vo_minfree;
	u32 rx_batchmode;
	u32 tt_support;
	u32 atf_config;
	u32 iphdr_pad_config;
	u32 qwrap_config:16,
	    alloc_frag_desc_for_data_pkt:16;
	u32 num_tdls_vdevs;
	u32 num_tdls_conn_table_entries;
	u32 beacon_tx_offload_max_vdev;
	u32 num_multicast_filter_entries;
	u32 num_wow_filters;
	u32 num_keep_alive_pattern;
	u32 keep_alive_pattern_size;
	u32 max_tdls_concurrent_sleep_sta;
	u32 max_tdls_concurrent_buffer_sta;
	u32 wmi_send_separate;
	u32 num_ocb_vdevs;
	u32 num_ocb_channels;
	u32 num_ocb_schedules;
	u32 num_ns_ext_tuples_cfg;
	u32 bpf_instruction_size;
	u32 max_bssid_rx_filters;
	u32 use_pdev_id;
	u32 peer_map_unmap_version;
	u32 sched_params;
	u32 twt_ap_pdev_count;
	u32 twt_ap_sta_count;
	bool is_reg_cc_ext_event_supported;
};

struct ath12k_wmi_init_cmd_arg {
	struct ath12k_wmi_resource_config_arg res_cfg;
	u8 num_mem_chunks;
	struct ath12k_wmi_host_mem_chunk_arg *mem_chunks;
	u32 hw_mode_id;
	u32 num_band_to_mac;
	struct ath12k_wmi_pdev_band_arg band_to_mac[WMI_HOST_MAX_PDEV];
};

struct ath12k_wmi_pdev_band_to_mac_params {
	__le32 tlv_header;
	__le32 pdev_id;
	__le32 start_freq;
	__le32 end_freq;
} __packed;

/* This is both individual command WMI_PDEV_SET_HW_MODE_CMDID and also part
 * of WMI_TAG_INIT_CMD.
 */
struct ath12k_wmi_pdev_set_hw_mode_cmd {
	__le32 tlv_header;
	__le32 pdev_id;
	__le32 hw_mode_index;
	__le32 num_band_to_mac;
} __packed;

struct ath12k_wmi_ppe_threshold_params {
	__le32 numss_m1; /** NSS - 1*/
	__le32 ru_info;
	__le32 ppet16_ppet8_ru3_ru0[WMI_MAX_NUM_SS];
} __packed;

#define HW_BD_INFO_SIZE       5

struct ath12k_wmi_abi_version_params {
	__le32 abi_version_0;
	__le32 abi_version_1;
	__le32 abi_version_ns_0;
	__le32 abi_version_ns_1;
	__le32 abi_version_ns_2;
	__le32 abi_version_ns_3;
} __packed;

struct wmi_init_cmd {
	__le32 tlv_header;
	struct ath12k_wmi_abi_version_params host_abi_vers;
	__le32 num_host_mem_chunks;
} __packed;

#define WMI_RSRC_CFG_HOST_SVC_FLAG_REG_CC_EXT_SUPPORT_BIT 4

struct ath12k_wmi_resource_config_params {
	__le32 tlv_header;
	__le32 num_vdevs;
	__le32 num_peers;
	__le32 num_offload_peers;
	__le32 num_offload_reorder_buffs;
	__le32 num_peer_keys;
	__le32 num_tids;
	__le32 ast_skid_limit;
	__le32 tx_chain_mask;
	__le32 rx_chain_mask;
	__le32 rx_timeout_pri[4];
	__le32 rx_decap_mode;
	__le32 scan_max_pending_req;
	__le32 bmiss_offload_max_vdev;
	__le32 roam_offload_max_vdev;
	__le32 roam_offload_max_ap_profiles;
	__le32 num_mcast_groups;
	__le32 num_mcast_table_elems;
	__le32 mcast2ucast_mode;
	__le32 tx_dbg_log_size;
	__le32 num_wds_entries;
	__le32 dma_burst_size;
	__le32 mac_aggr_delim;
	__le32 rx_skip_defrag_timeout_dup_detection_check;
	__le32 vow_config;
	__le32 gtk_offload_max_vdev;
	__le32 num_msdu_desc;
	__le32 max_frag_entries;
	__le32 num_tdls_vdevs;
	__le32 num_tdls_conn_table_entries;
	__le32 beacon_tx_offload_max_vdev;
	__le32 num_multicast_filter_entries;
	__le32 num_wow_filters;
	__le32 num_keep_alive_pattern;
	__le32 keep_alive_pattern_size;
	__le32 max_tdls_concurrent_sleep_sta;
	__le32 max_tdls_concurrent_buffer_sta;
	__le32 wmi_send_separate;
	__le32 num_ocb_vdevs;
	__le32 num_ocb_channels;
	__le32 num_ocb_schedules;
	__le32 flag1;
	__le32 smart_ant_cap;
	__le32 bk_minfree;
	__le32 be_minfree;
	__le32 vi_minfree;
	__le32 vo_minfree;
	__le32 alloc_frag_desc_for_data_pkt;
	__le32 num_ns_ext_tuples_cfg;
	__le32 bpf_instruction_size;
	__le32 max_bssid_rx_filters;
	__le32 use_pdev_id;
	__le32 max_num_dbs_scan_duty_cycle;
	__le32 max_num_group_keys;
	__le32 peer_map_unmap_version;
	__le32 sched_params;
	__le32 twt_ap_pdev_count;
	__le32 twt_ap_sta_count;
	__le32 max_nlo_ssids;
	__le32 num_pkt_filters;
	__le32 num_max_sta_vdevs;
	__le32 max_bssid_indicator;
	__le32 ul_resp_config;
	__le32 msdu_flow_override_config0;
	__le32 msdu_flow_override_config1;
	__le32 flags2;
	__le32 host_service_flags;
	__le32 max_rnr_neighbours;
	__le32 ema_max_vap_cnt;
	__le32 ema_max_profile_period;
} __packed;

struct wmi_service_ready_event {
	__le32 fw_build_vers;
	struct ath12k_wmi_abi_version_params fw_abi_vers;
	__le32 phy_capability;
	__le32 max_frag_entry;
	__le32 num_rf_chains;
	__le32 ht_cap_info;
	__le32 vht_cap_info;
	__le32 vht_supp_mcs;
	__le32 hw_min_tx_power;
	__le32 hw_max_tx_power;
	__le32 sys_cap_info;
	__le32 min_pkt_size_enable;
	__le32 max_bcn_ie_size;
	__le32 num_mem_reqs;
	__le32 max_num_scan_channels;
	__le32 hw_bd_id;
	__le32 hw_bd_info[HW_BD_INFO_SIZE];
	__le32 max_supported_macs;
	__le32 wmi_fw_sub_feat_caps;
	__le32 num_dbs_hw_modes;
	/* txrx_chainmask
	 *    [7:0]   - 2G band tx chain mask
	 *    [15:8]  - 2G band rx chain mask
	 *    [23:16] - 5G band tx chain mask
	 *    [31:24] - 5G band rx chain mask
	 */
	__le32 txrx_chainmask;
	__le32 default_dbs_hw_mode_index;
	__le32 num_msdu_desc;
} __packed;

#define WMI_SERVICE_BM_SIZE	((WMI_MAX_SERVICE + sizeof(u32) - 1) / sizeof(u32))

#define WMI_SERVICE_SEGMENT_BM_SIZE32 4 /* 4x u32 = 128 bits */
#define WMI_SERVICE_EXT_BM_SIZE (WMI_SERVICE_SEGMENT_BM_SIZE32 * sizeof(u32))
#define WMI_AVAIL_SERVICE_BITS_IN_SIZE32 32
#define WMI_SERVICE_BITS_IN_SIZE32 4

struct wmi_service_ready_ext_event {
	__le32 default_conc_scan_config_bits;
	__le32 default_fw_config_bits;
	struct ath12k_wmi_ppe_threshold_params ppet;
	__le32 he_cap_info;
	__le32 mpdu_density;
	__le32 max_bssid_rx_filters;
	__le32 fw_build_vers_ext;
	__le32 max_nlo_ssids;
	__le32 max_bssid_indicator;
	__le32 he_cap_info_ext;
} __packed;

struct ath12k_wmi_soc_mac_phy_hw_mode_caps_params {
	__le32 num_hw_modes;
	__le32 num_chainmask_tables;
} __packed;

struct ath12k_wmi_hw_mode_cap_params {
	__le32 tlv_header;
	__le32 hw_mode_id;
	__le32 phy_id_map;
	__le32 hw_mode_config_type;
} __packed;

#define WMI_MAX_HECAP_PHY_SIZE                 (3)

struct ath12k_wmi_mac_phy_caps_params {
	__le32 hw_mode_id;
	__le32 pdev_id;
	__le32 phy_id;
	__le32 supported_flags;
	__le32 supported_bands;
	__le32 ampdu_density;
	__le32 max_bw_supported_2g;
	__le32 ht_cap_info_2g;
	__le32 vht_cap_info_2g;
	__le32 vht_supp_mcs_2g;
	__le32 he_cap_info_2g;
	__le32 he_supp_mcs_2g;
	__le32 tx_chain_mask_2g;
	__le32 rx_chain_mask_2g;
	__le32 max_bw_supported_5g;
	__le32 ht_cap_info_5g;
	__le32 vht_cap_info_5g;
	__le32 vht_supp_mcs_5g;
	__le32 he_cap_info_5g;
	__le32 he_supp_mcs_5g;
	__le32 tx_chain_mask_5g;
	__le32 rx_chain_mask_5g;
	__le32 he_cap_phy_info_2g[WMI_MAX_HECAP_PHY_SIZE];
	__le32 he_cap_phy_info_5g[WMI_MAX_HECAP_PHY_SIZE];
	struct ath12k_wmi_ppe_threshold_params he_ppet2g;
	struct ath12k_wmi_ppe_threshold_params he_ppet5g;
	__le32 chainmask_table_id;
	__le32 lmac_id;
	__le32 he_cap_info_2g_ext;
	__le32 he_cap_info_5g_ext;
	__le32 he_cap_info_internal;
} __packed;

struct ath12k_wmi_hal_reg_caps_ext_params {
	__le32 tlv_header;
	__le32 phy_id;
	__le32 eeprom_reg_domain;
	__le32 eeprom_reg_domain_ext;
	__le32 regcap1;
	__le32 regcap2;
	__le32 wireless_modes;
	__le32 low_2ghz_chan;
	__le32 high_2ghz_chan;
	__le32 low_5ghz_chan;
	__le32 high_5ghz_chan;
} __packed;

struct ath12k_wmi_soc_hal_reg_caps_params {
	__le32 num_phy;
} __packed;

#define WMI_MAX_EHTCAP_MAC_SIZE  2
#define WMI_MAX_EHTCAP_PHY_SIZE  3
#define WMI_MAX_EHTCAP_RATE_SET  3

/* Used for EHT MCS-NSS array. Data at each array index follows the format given
 * in IEEE P802.11be/D2.0, May 20229.4.2.313.4.
 *
 * Index interpretation:
 * 0 - 20 MHz only sta, all 4 bytes valid
 * 1 - index for bandwidths <= 80 MHz except 20 MHz-only, first 3 bytes valid
 * 2 - index for 160 MHz, first 3 bytes valid
 * 3 - index for 320 MHz, first 3 bytes valid
 */
#define WMI_MAX_EHT_SUPP_MCS_2G_SIZE  2
#define WMI_MAX_EHT_SUPP_MCS_5G_SIZE  4

#define WMI_EHTCAP_TXRX_MCS_NSS_IDX_80    0
#define WMI_EHTCAP_TXRX_MCS_NSS_IDX_160   1
#define WMI_EHTCAP_TXRX_MCS_NSS_IDX_320   2

#define WMI_EHT_MCS_NSS_0_7    GENMASK(3, 0)
#define WMI_EHT_MCS_NSS_8_9    GENMASK(7, 4)
#define WMI_EHT_MCS_NSS_10_11  GENMASK(11, 8)
#define WMI_EHT_MCS_NSS_12_13  GENMASK(15, 12)

struct wmi_service_ready_ext2_event {
	__le32 reg_db_version;
	__le32 hw_min_max_tx_power_2ghz;
	__le32 hw_min_max_tx_power_5ghz;
	__le32 chwidth_num_peer_caps;
	__le32 preamble_puncture_bw;
	__le32 max_user_per_ppdu_ofdma;
	__le32 max_user_per_ppdu_mumimo;
	__le32 target_cap_flags;
	__le32 eht_cap_mac_info[WMI_MAX_EHTCAP_MAC_SIZE];
	__le32 max_num_linkview_peers;
	__le32 max_num_msduq_supported_per_tid;
	__le32 default_num_msduq_supported_per_tid;
} __packed;

struct ath12k_wmi_caps_ext_params {
	__le32 hw_mode_id;
	union {
		struct {
			__le16 pdev_id;
			__le16 hw_link_id;
		} __packed ath12k_wmi_pdev_to_link_map;
		__le32 pdev_id;
	};
	__le32 phy_id;
	__le32 wireless_modes_ext;
	__le32 eht_cap_mac_info_2ghz[WMI_MAX_EHTCAP_MAC_SIZE];
	__le32 eht_cap_mac_info_5ghz[WMI_MAX_EHTCAP_MAC_SIZE];
	__le32 rsvd0[2];
	__le32 eht_cap_phy_info_2ghz[WMI_MAX_EHTCAP_PHY_SIZE];
	__le32 eht_cap_phy_info_5ghz[WMI_MAX_EHTCAP_PHY_SIZE];
	struct ath12k_wmi_ppe_threshold_params eht_ppet_2ghz;
	struct ath12k_wmi_ppe_threshold_params eht_ppet_5ghz;
	__le32 eht_cap_info_internal;
	__le32 eht_supp_mcs_ext_2ghz[WMI_MAX_EHT_SUPP_MCS_2G_SIZE];
	__le32 eht_supp_mcs_ext_5ghz[WMI_MAX_EHT_SUPP_MCS_5G_SIZE];
} __packed;

/* 2 word representation of MAC addr */
struct ath12k_wmi_mac_addr_params {
	u8 addr[ETH_ALEN];
	u8 padding[2];
} __packed;

struct ath12k_wmi_dma_ring_caps_params {
	__le32 tlv_header;
	__le32 pdev_id;
	__le32 module_id;
	__le32 min_elem;
	__le32 min_buf_sz;
	__le32 min_buf_align;
} __packed;

struct ath12k_wmi_ready_event_min_params {
	struct ath12k_wmi_abi_version_params fw_abi_vers;
	struct ath12k_wmi_mac_addr_params mac_addr;
	__le32 status;
	__le32 num_dscp_table;
	__le32 num_extra_mac_addr;
	__le32 num_total_peers;
	__le32 num_extra_peers;
} __packed;

struct wmi_ready_event {
	struct ath12k_wmi_ready_event_min_params ready_event_min;
	__le32 max_ast_index;
	__le32 pktlog_defs_checksum;
} __packed;

struct wmi_service_available_event {
	__le32 wmi_service_segment_offset;
	__le32 wmi_service_segment_bitmap[WMI_SERVICE_SEGMENT_BM_SIZE32];
} __packed;

struct ath12k_wmi_vdev_create_arg {
	u8 if_id;
	u32 type;
	u32 subtype;
	struct {
		u8 tx;
		u8 rx;
	} chains[NUM_NL80211_BANDS];
	u32 pdev_id;
	u8 if_stats_id;
};

#define ATH12K_MAX_VDEV_STATS_ID	0x30
#define ATH12K_INVAL_VDEV_STATS_ID	0xFF

struct wmi_vdev_create_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	__le32 vdev_type;
	__le32 vdev_subtype;
	struct ath12k_wmi_mac_addr_params vdev_macaddr;
	__le32 num_cfg_txrx_streams;
	__le32 pdev_id;
	__le32 vdev_stats_id;
} __packed;

struct ath12k_wmi_vdev_txrx_streams_params {
	__le32 tlv_header;
	u32 band;
	u32 supported_tx_streams;
	u32 supported_rx_streams;
} __packed;

struct wmi_vdev_delete_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
} __packed;

struct wmi_vdev_up_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	__le32 vdev_assoc_id;
	struct ath12k_wmi_mac_addr_params vdev_bssid;
	struct ath12k_wmi_mac_addr_params trans_bssid;
	__le32 profile_idx;
	__le32 profile_num;
} __packed;

struct wmi_vdev_stop_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
} __packed;

struct wmi_vdev_down_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
} __packed;

#define WMI_VDEV_START_HIDDEN_SSID  BIT(0)
#define WMI_VDEV_START_PMF_ENABLED  BIT(1)
#define WMI_VDEV_START_LDPC_RX_ENABLED BIT(3)

#define ATH12K_WMI_SSID_LEN 32

struct ath12k_wmi_ssid_params {
	__le32 ssid_len;
	u8 ssid[ATH12K_WMI_SSID_LEN];
} __packed;

#define ATH12K_VDEV_SETUP_TIMEOUT_HZ (5 * HZ)

struct wmi_vdev_start_request_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	__le32 requestor_id;
	__le32 beacon_interval;
	__le32 dtim_period;
	__le32 flags;
	struct ath12k_wmi_ssid_params ssid;
	__le32 bcn_tx_rate;
	__le32 bcn_txpower;
	__le32 num_noa_descriptors;
	__le32 disable_hw_ack;
	__le32 preferred_tx_streams;
	__le32 preferred_rx_streams;
	__le32 he_ops;
	__le32 cac_duration_ms;
	__le32 regdomain;
	__le32 min_data_rate;
	__le32 mbssid_flags;
	__le32 mbssid_tx_vdev_id;
	__le32 eht_ops;
	__le32 punct_bitmap;
} __packed;

#define MGMT_TX_DL_FRM_LEN		     64

struct ath12k_wmi_channel_arg {
	u8 chan_id;
	u8 pwr;
	u32 mhz;
	u32 half_rate:1,
	    quarter_rate:1,
	    dfs_set:1,
	    dfs_set_cfreq2:1,
	    is_chan_passive:1,
	    allow_ht:1,
	    allow_vht:1,
	    allow_he:1,
	    set_agile:1,
	    psc_channel:1;
	u32 phy_mode;
	u32 cfreq1;
	u32 cfreq2;
	char   maxpower;
	char   minpower;
	char   maxregpower;
	u8  antennamax;
	u8  reg_class_id;
};

enum wmi_phy_mode {
	MODE_11A        = 0,
	MODE_11G        = 1,   /* 11b/g Mode */
	MODE_11B        = 2,   /* 11b Mode */
	MODE_11GONLY    = 3,   /* 11g only Mode */
	MODE_11NA_HT20   = 4,
	MODE_11NG_HT20   = 5,
	MODE_11NA_HT40   = 6,
	MODE_11NG_HT40   = 7,
	MODE_11AC_VHT20 = 8,
	MODE_11AC_VHT40 = 9,
	MODE_11AC_VHT80 = 10,
	MODE_11AC_VHT20_2G = 11,
	MODE_11AC_VHT40_2G = 12,
	MODE_11AC_VHT80_2G = 13,
	MODE_11AC_VHT80_80 = 14,
	MODE_11AC_VHT160 = 15,
	MODE_11AX_HE20 = 16,
	MODE_11AX_HE40 = 17,
	MODE_11AX_HE80 = 18,
	MODE_11AX_HE80_80 = 19,
	MODE_11AX_HE160 = 20,
	MODE_11AX_HE20_2G = 21,
	MODE_11AX_HE40_2G = 22,
	MODE_11AX_HE80_2G = 23,
	MODE_11BE_EHT20 = 24,
	MODE_11BE_EHT40 = 25,
	MODE_11BE_EHT80 = 26,
	MODE_11BE_EHT80_80 = 27,
	MODE_11BE_EHT160 = 28,
	MODE_11BE_EHT160_160 = 29,
	MODE_11BE_EHT320 = 30,
	MODE_11BE_EHT20_2G = 31,
	MODE_11BE_EHT40_2G = 32,
	MODE_UNKNOWN = 33,
	MODE_MAX = 33,
};

struct wmi_vdev_start_req_arg {
	u32 vdev_id;
	u32 freq;
	u32 band_center_freq1;
	u32 band_center_freq2;
	bool passive;
	bool allow_ibss;
	bool allow_ht;
	bool allow_vht;
	bool ht40plus;
	bool chan_radar;
	bool freq2_radar;
	bool allow_he;
	u32 min_power;
	u32 max_power;
	u32 max_reg_power;
	u32 max_antenna_gain;
	enum wmi_phy_mode mode;
	u32 bcn_intval;
	u32 dtim_period;
	u8 *ssid;
	u32 ssid_len;
	u32 bcn_tx_rate;
	u32 bcn_tx_power;
	bool disable_hw_ack;
	bool hidden_ssid;
	bool pmf_enabled;
	u32 he_ops;
	u32 cac_duration_ms;
	u32 regdomain;
	u32 pref_rx_streams;
	u32 pref_tx_streams;
	u32 num_noa_descriptors;
	u32 min_data_rate;
	u32 mbssid_flags;
	u32 mbssid_tx_vdev_id;
	u32 punct_bitmap;
};

struct ath12k_wmi_peer_create_arg {
	const u8 *peer_addr;
	u32 peer_type;
	u32 vdev_id;
};

struct ath12k_wmi_pdev_set_regdomain_arg {
	u16 current_rd_in_use;
	u16 current_rd_2g;
	u16 current_rd_5g;
	u32 ctl_2g;
	u32 ctl_5g;
	u8 dfs_domain;
	u32 pdev_id;
};

struct ath12k_wmi_rx_reorder_queue_remove_arg {
	u8 *peer_macaddr;
	u16 vdev_id;
	u32 peer_tid_bitmap;
};

#define WMI_HOST_PDEV_ID_SOC 0xFF
#define WMI_HOST_PDEV_ID_0   0
#define WMI_HOST_PDEV_ID_1   1
#define WMI_HOST_PDEV_ID_2   2

#define WMI_PDEV_ID_SOC         0
#define WMI_PDEV_ID_1ST         1
#define WMI_PDEV_ID_2ND         2
#define WMI_PDEV_ID_3RD         3

/* Freq units in MHz */
#define REG_RULE_START_FREQ			0x0000ffff
#define REG_RULE_END_FREQ			0xffff0000
#define REG_RULE_FLAGS				0x0000ffff
#define REG_RULE_MAX_BW				0x0000ffff
#define REG_RULE_REG_PWR			0x00ff0000
#define REG_RULE_ANT_GAIN			0xff000000
#define REG_RULE_PSD_INFO			BIT(2)
#define REG_RULE_PSD_EIRP			0xffff0000

#define WMI_VDEV_PARAM_TXBF_SU_TX_BFEE BIT(0)
#define WMI_VDEV_PARAM_TXBF_MU_TX_BFEE BIT(1)
#define WMI_VDEV_PARAM_TXBF_SU_TX_BFER BIT(2)
#define WMI_VDEV_PARAM_TXBF_MU_TX_BFER BIT(3)

#define HECAP_PHYDWORD_0	0
#define HECAP_PHYDWORD_1	1
#define HECAP_PHYDWORD_2	2

#define HECAP_PHY_SU_BFER		BIT(31)
#define HECAP_PHY_SU_BFEE		BIT(0)
#define HECAP_PHY_MU_BFER		BIT(1)
#define HECAP_PHY_UL_MUMIMO		BIT(22)
#define HECAP_PHY_UL_MUOFDMA		BIT(23)

#define HECAP_PHY_SUBFMR_GET(hecap_phy) \
	u32_get_bits(hecap_phy[HECAP_PHYDWORD_0], HECAP_PHY_SU_BFER)

#define HECAP_PHY_SUBFME_GET(hecap_phy) \
	u32_get_bits(hecap_phy[HECAP_PHYDWORD_1], HECAP_PHY_SU_BFEE)

#define HECAP_PHY_MUBFMR_GET(hecap_phy) \
	u32_get_bits(hecap_phy[HECAP_PHYDWORD_1], HECAP_PHY_MU_BFER)

#define HECAP_PHY_ULMUMIMO_GET(hecap_phy) \
	u32_get_bits(hecap_phy[HECAP_PHYDWORD_0], HECAP_PHY_UL_MUMIMO)

#define HECAP_PHY_ULOFDMA_GET(hecap_phy) \
	u32_get_bits(hecap_phy[HECAP_PHYDWORD_0], HECAP_PHY_UL_MUOFDMA)

#define HE_MODE_SU_TX_BFEE	BIT(0)
#define HE_MODE_SU_TX_BFER	BIT(1)
#define HE_MODE_MU_TX_BFEE	BIT(2)
#define HE_MODE_MU_TX_BFER	BIT(3)
#define HE_MODE_DL_OFDMA	BIT(4)
#define HE_MODE_UL_OFDMA	BIT(5)
#define HE_MODE_UL_MUMIMO	BIT(6)

#define HE_DL_MUOFDMA_ENABLE	1
#define HE_UL_MUOFDMA_ENABLE	1
#define HE_DL_MUMIMO_ENABLE	1
#define HE_MU_BFEE_ENABLE	1
#define HE_SU_BFEE_ENABLE	1

#define HE_VHT_SOUNDING_MODE_ENABLE		1
#define HE_SU_MU_SOUNDING_MODE_ENABLE		1
#define HE_TRIG_NONTRIG_SOUNDING_MODE_ENABLE	1

/* HE or VHT Sounding */
#define HE_VHT_SOUNDING_MODE		BIT(0)
/* SU or MU Sounding */
#define HE_SU_MU_SOUNDING_MODE		BIT(2)
/* Trig or Non-Trig Sounding */
#define HE_TRIG_NONTRIG_SOUNDING_MODE	BIT(3)

#define WMI_TXBF_STS_CAP_OFFSET_LSB	4
#define WMI_TXBF_STS_CAP_OFFSET_MASK	0x70
#define WMI_BF_SOUND_DIM_OFFSET_LSB	8
#define WMI_BF_SOUND_DIM_OFFSET_MASK	0x700

enum wmi_peer_type {
	WMI_PEER_TYPE_DEFAULT = 0,
	WMI_PEER_TYPE_BSS = 1,
	WMI_PEER_TYPE_TDLS = 2,
};

struct wmi_peer_create_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	struct ath12k_wmi_mac_addr_params peer_macaddr;
	__le32 peer_type;
} __packed;

struct wmi_peer_delete_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	struct ath12k_wmi_mac_addr_params peer_macaddr;
} __packed;

struct wmi_peer_reorder_queue_setup_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	struct ath12k_wmi_mac_addr_params peer_macaddr;
	__le32 tid;
	__le32 queue_ptr_lo;
	__le32 queue_ptr_hi;
	__le32 queue_no;
	__le32 ba_window_size_valid;
	__le32 ba_window_size;
} __packed;

struct wmi_peer_reorder_queue_remove_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	struct ath12k_wmi_mac_addr_params peer_macaddr;
	__le32 tid_mask;
} __packed;

enum wmi_bss_chan_info_req_type {
	WMI_BSS_SURVEY_REQ_TYPE_READ = 1,
	WMI_BSS_SURVEY_REQ_TYPE_READ_CLEAR,
};

struct wmi_pdev_set_param_cmd {
	__le32 tlv_header;
	__le32 pdev_id;
	__le32 param_id;
	__le32 param_value;
} __packed;

struct wmi_pdev_set_ps_mode_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	__le32 sta_ps_mode;
} __packed;

struct wmi_pdev_suspend_cmd {
	__le32 tlv_header;
	__le32 pdev_id;
	__le32 suspend_opt;
} __packed;

struct wmi_pdev_resume_cmd {
	__le32 tlv_header;
	__le32 pdev_id;
} __packed;

struct wmi_pdev_bss_chan_info_req_cmd {
	__le32 tlv_header;
	/* ref wmi_bss_chan_info_req_type */
	__le32 req_type;
} __packed;

struct wmi_ap_ps_peer_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	struct ath12k_wmi_mac_addr_params peer_macaddr;
	__le32 param;
	__le32 value;
} __packed;

struct wmi_sta_powersave_param_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	__le32 param;
	__le32 value;
} __packed;

struct wmi_pdev_set_regdomain_cmd {
	__le32 tlv_header;
	__le32 pdev_id;
	__le32 reg_domain;
	__le32 reg_domain_2g;
	__le32 reg_domain_5g;
	__le32 conformance_test_limit_2g;
	__le32 conformance_test_limit_5g;
	__le32 dfs_domain;
} __packed;

struct wmi_peer_set_param_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	struct ath12k_wmi_mac_addr_params peer_macaddr;
	__le32 param_id;
	__le32 param_value;
} __packed;

struct wmi_peer_flush_tids_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	struct ath12k_wmi_mac_addr_params peer_macaddr;
	__le32 peer_tid_bitmap;
} __packed;

struct wmi_dfs_phyerr_offload_cmd {
	__le32 tlv_header;
	__le32 pdev_id;
} __packed;

struct wmi_bcn_offload_ctrl_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	__le32 bcn_ctrl_op;
} __packed;

enum scan_dwelltime_adaptive_mode {
	SCAN_DWELL_MODE_DEFAULT = 0,
	SCAN_DWELL_MODE_CONSERVATIVE = 1,
	SCAN_DWELL_MODE_MODERATE = 2,
	SCAN_DWELL_MODE_AGGRESSIVE = 3,
	SCAN_DWELL_MODE_STATIC = 4
};

#define WLAN_SCAN_MAX_NUM_SSID          10
#define WLAN_SCAN_MAX_NUM_BSSID         10

struct ath12k_wmi_element_info_arg {
	u32 len;
	u8 *ptr;
};

#define WMI_IE_BITMAP_SIZE             8

#define WMI_SCAN_MAX_NUM_SSID                0x0A
/* prefix used by scan requestor ids on the host */
#define WMI_HOST_SCAN_REQUESTOR_ID_PREFIX 0xA000

/* prefix used by scan request ids generated on the host */
/* host cycles through the lower 12 bits to generate ids */
#define WMI_HOST_SCAN_REQ_ID_PREFIX 0xA000

#define WLAN_SCAN_PARAMS_MAX_SSID    16
#define WLAN_SCAN_PARAMS_MAX_BSSID   4
#define WLAN_SCAN_PARAMS_MAX_IE_LEN  256

/* Values lower than this may be refused by some firmware revisions with a scan
 * completion with a timedout reason.
 */
#define WMI_SCAN_CHAN_MIN_TIME_MSEC 40

/* Scan priority numbers must be sequential, starting with 0 */
enum wmi_scan_priority {
	WMI_SCAN_PRIORITY_VERY_LOW = 0,
	WMI_SCAN_PRIORITY_LOW,
	WMI_SCAN_PRIORITY_MEDIUM,
	WMI_SCAN_PRIORITY_HIGH,
	WMI_SCAN_PRIORITY_VERY_HIGH,
	WMI_SCAN_PRIORITY_COUNT   /* number of priorities supported */
};

enum wmi_scan_event_type {
	WMI_SCAN_EVENT_STARTED              = BIT(0),
	WMI_SCAN_EVENT_COMPLETED            = BIT(1),
	WMI_SCAN_EVENT_BSS_CHANNEL          = BIT(2),
	WMI_SCAN_EVENT_FOREIGN_CHAN         = BIT(3),
	WMI_SCAN_EVENT_DEQUEUED             = BIT(4),
	/* possibly by high-prio scan */
	WMI_SCAN_EVENT_PREEMPTED            = BIT(5),
	WMI_SCAN_EVENT_START_FAILED         = BIT(6),
	WMI_SCAN_EVENT_RESTARTED            = BIT(7),
	WMI_SCAN_EVENT_FOREIGN_CHAN_EXIT    = BIT(8),
	WMI_SCAN_EVENT_SUSPENDED            = BIT(9),
	WMI_SCAN_EVENT_RESUMED              = BIT(10),
	WMI_SCAN_EVENT_MAX                  = BIT(15),
};

enum wmi_scan_completion_reason {
	WMI_SCAN_REASON_COMPLETED,
	WMI_SCAN_REASON_CANCELLED,
	WMI_SCAN_REASON_PREEMPTED,
	WMI_SCAN_REASON_TIMEDOUT,
	WMI_SCAN_REASON_INTERNAL_FAILURE,
	WMI_SCAN_REASON_MAX,
};

struct  wmi_start_scan_cmd {
	__le32 tlv_header;
	__le32 scan_id;
	__le32 scan_req_id;
	__le32 vdev_id;
	__le32 scan_priority;
	__le32 notify_scan_events;
	__le32 dwell_time_active;
	__le32 dwell_time_passive;
	__le32 min_rest_time;
	__le32 max_rest_time;
	__le32 repeat_probe_time;
	__le32 probe_spacing_time;
	__le32 idle_time;
	__le32 max_scan_time;
	__le32 probe_delay;
	__le32 scan_ctrl_flags;
	__le32 burst_duration;
	__le32 num_chan;
	__le32 num_bssid;
	__le32 num_ssids;
	__le32 ie_len;
	__le32 n_probes;
	struct ath12k_wmi_mac_addr_params mac_addr;
	struct ath12k_wmi_mac_addr_params mac_mask;
	u32 ie_bitmap[WMI_IE_BITMAP_SIZE];
	__le32 num_vendor_oui;
	__le32 scan_ctrl_flags_ext;
	__le32 dwell_time_active_2g;
	__le32 dwell_time_active_6g;
	__le32 dwell_time_passive_6g;
	__le32 scan_start_offset;
} __packed;

#define WMI_SCAN_FLAG_PASSIVE        0x1
#define WMI_SCAN_ADD_BCAST_PROBE_REQ 0x2
#define WMI_SCAN_ADD_CCK_RATES       0x4
#define WMI_SCAN_ADD_OFDM_RATES      0x8
#define WMI_SCAN_CHAN_STAT_EVENT     0x10
#define WMI_SCAN_FILTER_PROBE_REQ    0x20
#define WMI_SCAN_BYPASS_DFS_CHN      0x40
#define WMI_SCAN_CONTINUE_ON_ERROR   0x80
#define WMI_SCAN_FILTER_PROMISCUOS   0x100
#define WMI_SCAN_FLAG_FORCE_ACTIVE_ON_DFS 0x200
#define WMI_SCAN_ADD_TPC_IE_IN_PROBE_REQ  0x400
#define WMI_SCAN_ADD_DS_IE_IN_PROBE_REQ   0x800
#define WMI_SCAN_ADD_SPOOF_MAC_IN_PROBE_REQ   0x1000
#define WMI_SCAN_OFFCHAN_MGMT_TX    0x2000
#define WMI_SCAN_OFFCHAN_DATA_TX    0x4000
#define WMI_SCAN_CAPTURE_PHY_ERROR  0x8000
#define WMI_SCAN_FLAG_STRICT_PASSIVE_ON_PCHN 0x10000
#define WMI_SCAN_FLAG_HALF_RATE_SUPPORT      0x20000
#define WMI_SCAN_FLAG_QUARTER_RATE_SUPPORT   0x40000
#define WMI_SCAN_RANDOM_SEQ_NO_IN_PROBE_REQ 0x80000
#define WMI_SCAN_ENABLE_IE_WHTELIST_IN_PROBE_REQ 0x100000

#define WMI_SCAN_DWELL_MODE_MASK GENMASK(23, 21)

enum {
	WMI_SCAN_DWELL_MODE_DEFAULT      = 0,
	WMI_SCAN_DWELL_MODE_CONSERVATIVE = 1,
	WMI_SCAN_DWELL_MODE_MODERATE     = 2,
	WMI_SCAN_DWELL_MODE_AGGRESSIVE   = 3,
	WMI_SCAN_DWELL_MODE_STATIC       = 4,
};

struct ath12k_wmi_hint_short_ssid_arg {
	u32 freq_flags;
	u32 short_ssid;
};

struct ath12k_wmi_hint_bssid_arg {
	u32 freq_flags;
	struct ath12k_wmi_mac_addr_params bssid;
};

struct ath12k_wmi_scan_req_arg {
	u32 scan_id;
	u32 scan_req_id;
	u32 vdev_id;
	u32 pdev_id;
	enum wmi_scan_priority scan_priority;
	union {
		struct {
			u32 scan_ev_started:1,
			    scan_ev_completed:1,
			    scan_ev_bss_chan:1,
			    scan_ev_foreign_chan:1,
			    scan_ev_dequeued:1,
			    scan_ev_preempted:1,
			    scan_ev_start_failed:1,
			    scan_ev_restarted:1,
			    scan_ev_foreign_chn_exit:1,
			    scan_ev_invalid:1,
			    scan_ev_gpio_timeout:1,
			    scan_ev_suspended:1,
			    scan_ev_resumed:1;
		};
		u32 scan_events;
	};
	u32 dwell_time_active;
	u32 dwell_time_active_2g;
	u32 dwell_time_passive;
	u32 dwell_time_active_6g;
	u32 dwell_time_passive_6g;
	u32 min_rest_time;
	u32 max_rest_time;
	u32 repeat_probe_time;
	u32 probe_spacing_time;
	u32 idle_time;
	u32 max_scan_time;
	u32 probe_delay;
	union {
		struct {
			u32 scan_f_passive:1,
			    scan_f_bcast_probe:1,
			    scan_f_cck_rates:1,
			    scan_f_ofdm_rates:1,
			    scan_f_chan_stat_evnt:1,
			    scan_f_filter_prb_req:1,
			    scan_f_bypass_dfs_chn:1,
			    scan_f_continue_on_err:1,
			    scan_f_offchan_mgmt_tx:1,
			    scan_f_offchan_data_tx:1,
			    scan_f_promisc_mode:1,
			    scan_f_capture_phy_err:1,
			    scan_f_strict_passive_pch:1,
			    scan_f_half_rate:1,
			    scan_f_quarter_rate:1,
			    scan_f_force_active_dfs_chn:1,
			    scan_f_add_tpc_ie_in_probe:1,
			    scan_f_add_ds_ie_in_probe:1,
			    scan_f_add_spoofed_mac_in_probe:1,
			    scan_f_add_rand_seq_in_probe:1,
			    scan_f_en_ie_whitelist_in_probe:1,
			    scan_f_forced:1,
			    scan_f_2ghz:1,
			    scan_f_5ghz:1,
			    scan_f_80mhz:1;
		};
		u32 scan_flags;
	};
	enum scan_dwelltime_adaptive_mode adaptive_dwell_time_mode;
	u32 burst_duration;
	u32 num_chan;
	u32 num_bssid;
	u32 num_ssids;
	u32 n_probes;
	u32 *chan_list;
	u32 notify_scan_events;
	struct cfg80211_ssid ssid[WLAN_SCAN_MAX_NUM_SSID];
	struct ath12k_wmi_mac_addr_params bssid_list[WLAN_SCAN_MAX_NUM_BSSID];
	struct ath12k_wmi_element_info_arg extraie;
	u32 num_hint_s_ssid;
	u32 num_hint_bssid;
	struct ath12k_wmi_hint_short_ssid_arg hint_s_ssid[WLAN_SCAN_MAX_HINT_S_SSID];
	struct ath12k_wmi_hint_bssid_arg hint_bssid[WLAN_SCAN_MAX_HINT_BSSID];
};

struct wmi_ssid_arg {
	int len;
	const u8 *ssid;
};

struct wmi_bssid_arg {
	const u8 *bssid;
};

struct wmi_start_scan_arg {
	u32 scan_id;
	u32 scan_req_id;
	u32 vdev_id;
	u32 scan_priority;
	u32 notify_scan_events;
	u32 dwell_time_active;
	u32 dwell_time_passive;
	u32 min_rest_time;
	u32 max_rest_time;
	u32 repeat_probe_time;
	u32 probe_spacing_time;
	u32 idle_time;
	u32 max_scan_time;
	u32 probe_delay;
	u32 scan_ctrl_flags;

	u32 ie_len;
	u32 n_channels;
	u32 n_ssids;
	u32 n_bssids;

	u8 ie[WLAN_SCAN_PARAMS_MAX_IE_LEN];
	u32 channels[64];
	struct wmi_ssid_arg ssids[WLAN_SCAN_PARAMS_MAX_SSID];
	struct wmi_bssid_arg bssids[WLAN_SCAN_PARAMS_MAX_BSSID];
};

#define WMI_SCAN_STOP_ONE       0x00000000
#define WMI_SCAN_STOP_VAP_ALL   0x01000000
#define WMI_SCAN_STOP_ALL       0x04000000

/* Prefix 0xA000 indicates that the scan request
 * is trigger by HOST
 */
#define ATH12K_SCAN_ID          0xA000

enum scan_cancel_req_type {
	WLAN_SCAN_CANCEL_SINGLE = 1,
	WLAN_SCAN_CANCEL_VDEV_ALL,
	WLAN_SCAN_CANCEL_PDEV_ALL,
};

struct ath12k_wmi_scan_cancel_arg {
	u32 requester;
	u32 scan_id;
	enum scan_cancel_req_type req_type;
	u32 vdev_id;
	u32 pdev_id;
};

struct wmi_bcn_send_from_host_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	__le32 data_len;
	union {
		__le32 frag_ptr;
		__le32 frag_ptr_lo;
	};
	__le32 frame_ctrl;
	__le32 dtim_flag;
	__le32 bcn_antenna;
	__le32 frag_ptr_hi;
};

#define WMI_CHAN_INFO_MODE		GENMASK(5, 0)
#define WMI_CHAN_INFO_HT40_PLUS		BIT(6)
#define WMI_CHAN_INFO_PASSIVE		BIT(7)
#define WMI_CHAN_INFO_ADHOC_ALLOWED	BIT(8)
#define WMI_CHAN_INFO_AP_DISABLED	BIT(9)
#define WMI_CHAN_INFO_DFS		BIT(10)
#define WMI_CHAN_INFO_ALLOW_HT		BIT(11)
#define WMI_CHAN_INFO_ALLOW_VHT		BIT(12)
#define WMI_CHAN_INFO_CHAN_CHANGE_CAUSE_CSA	BIT(13)
#define WMI_CHAN_INFO_HALF_RATE		BIT(14)
#define WMI_CHAN_INFO_QUARTER_RATE	BIT(15)
#define WMI_CHAN_INFO_DFS_FREQ2		BIT(16)
#define WMI_CHAN_INFO_ALLOW_HE		BIT(17)
#define WMI_CHAN_INFO_PSC		BIT(18)

#define WMI_CHAN_REG_INFO1_MIN_PWR	GENMASK(7, 0)
#define WMI_CHAN_REG_INFO1_MAX_PWR	GENMASK(15, 8)
#define WMI_CHAN_REG_INFO1_MAX_REG_PWR	GENMASK(23, 16)
#define WMI_CHAN_REG_INFO1_REG_CLS	GENMASK(31, 24)

#define WMI_CHAN_REG_INFO2_ANT_MAX	GENMASK(7, 0)
#define WMI_CHAN_REG_INFO2_MAX_TX_PWR	GENMASK(15, 8)

struct ath12k_wmi_channel_params {
	__le32 tlv_header;
	__le32 mhz;
	__le32 band_center_freq1;
	__le32 band_center_freq2;
	__le32 info;
	__le32 reg_info_1;
	__le32 reg_info_2;
} __packed;

enum wmi_sta_ps_mode {
	WMI_STA_PS_MODE_DISABLED = 0,
	WMI_STA_PS_MODE_ENABLED = 1,
};

#define WMI_SMPS_MASK_LOWER_16BITS 0xFF
#define WMI_SMPS_MASK_UPPER_3BITS 0x7
#define WMI_SMPS_PARAM_VALUE_SHIFT 29

#define ATH12K_WMI_FW_HANG_ASSERT_TYPE 1
#define ATH12K_WMI_FW_HANG_DELAY 0

/* type, 0:unused 1: ASSERT 2: not respond detect command
 * delay_time_ms, the simulate will delay time
 */

struct wmi_force_fw_hang_cmd {
	__le32 tlv_header;
	__le32 type;
	__le32 delay_time_ms;
} __packed;

struct wmi_vdev_set_param_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	__le32 param_id;
	__le32 param_value;
} __packed;

struct wmi_get_pdev_temperature_cmd {
	__le32 tlv_header;
	__le32 param;
	__le32 pdev_id;
} __packed;

#define WMI_BEACON_TX_BUFFER_SIZE	512

struct wmi_bcn_tmpl_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	__le32 tim_ie_offset;
	__le32 buf_len;
	__le32 csa_switch_count_offset;
	__le32 ext_csa_switch_count_offset;
	__le32 csa_event_bitmap;
	__le32 mbssid_ie_offset;
	__le32 esp_ie_offset;
} __packed;

struct wmi_vdev_install_key_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	struct ath12k_wmi_mac_addr_params peer_macaddr;
	__le32 key_idx;
	__le32 key_flags;
	__le32 key_cipher;
	__le64 key_rsc_counter;
	__le64 key_global_rsc_counter;
	__le64 key_tsc_counter;
	u8 wpi_key_rsc_counter[16];
	u8 wpi_key_tsc_counter[16];
	__le32 key_len;
	__le32 key_txmic_len;
	__le32 key_rxmic_len;
	__le32 is_group_key_id_valid;
	__le32 group_key_id;

	/* Followed by key_data containing key followed by
	 * tx mic and then rx mic
	 */
} __packed;

struct wmi_vdev_install_key_arg {
	u32 vdev_id;
	const u8 *macaddr;
	u32 key_idx;
	u32 key_flags;
	u32 key_cipher;
	u32 key_len;
	u32 key_txmic_len;
	u32 key_rxmic_len;
	u64 key_rsc_counter;
	const void *key_data;
};

#define WMI_MAX_SUPPORTED_RATES			128
#define WMI_HOST_MAX_HECAP_PHY_SIZE		3
#define WMI_HOST_MAX_HE_RATE_SET		3
#define WMI_HECAP_TXRX_MCS_NSS_IDX_80		0
#define WMI_HECAP_TXRX_MCS_NSS_IDX_160		1
#define WMI_HECAP_TXRX_MCS_NSS_IDX_80_80	2

struct wmi_rate_set_arg {
	u32 num_rates;
	u8 rates[WMI_MAX_SUPPORTED_RATES];
};

struct ath12k_wmi_peer_assoc_arg {
	u32 vdev_id;
	u32 peer_new_assoc;
	u32 peer_associd;
	u32 peer_flags;
	u32 peer_caps;
	u32 peer_listen_intval;
	u32 peer_ht_caps;
	u32 peer_max_mpdu;
	u32 peer_mpdu_density;
	u32 peer_rate_caps;
	u32 peer_nss;
	u32 peer_vht_caps;
	u32 peer_phymode;
	u32 peer_ht_info[2];
	struct wmi_rate_set_arg peer_legacy_rates;
	struct wmi_rate_set_arg peer_ht_rates;
	u32 rx_max_rate;
	u32 rx_mcs_set;
	u32 tx_max_rate;
	u32 tx_mcs_set;
	u8 vht_capable;
	u8 min_data_rate;
	u32 tx_max_mcs_nss;
	u32 peer_bw_rxnss_override;
	bool is_pmf_enabled;
	bool is_wme_set;
	bool qos_flag;
	bool apsd_flag;
	bool ht_flag;
	bool bw_40;
	bool bw_80;
	bool bw_160;
	bool bw_320;
	bool stbc_flag;
	bool ldpc_flag;
	bool static_mimops_flag;
	bool dynamic_mimops_flag;
	bool spatial_mux_flag;
	bool vht_flag;
	bool vht_ng_flag;
	bool need_ptk_4_way;
	bool need_gtk_2_way;
	bool auth_flag;
	bool safe_mode_enabled;
	bool amsdu_disable;
	/* Use common structure */
	u8 peer_mac[ETH_ALEN];

	bool he_flag;
	u32 peer_he_cap_macinfo[2];
	u32 peer_he_cap_macinfo_internal;
	u32 peer_he_caps_6ghz;
	u32 peer_he_ops;
	u32 peer_he_cap_phyinfo[WMI_HOST_MAX_HECAP_PHY_SIZE];
	u32 peer_he_mcs_count;
	u32 peer_he_rx_mcs_set[WMI_HOST_MAX_HE_RATE_SET];
	u32 peer_he_tx_mcs_set[WMI_HOST_MAX_HE_RATE_SET];
	bool twt_responder;
	bool twt_requester;
	struct ath12k_wmi_ppe_threshold_arg peer_ppet;
	bool eht_flag;
	u32 peer_eht_cap_mac[WMI_MAX_EHTCAP_MAC_SIZE];
	u32 peer_eht_cap_phy[WMI_MAX_EHTCAP_PHY_SIZE];
	u32 peer_eht_mcs_count;
	u32 peer_eht_rx_mcs_set[WMI_MAX_EHTCAP_RATE_SET];
	u32 peer_eht_tx_mcs_set[WMI_MAX_EHTCAP_RATE_SET];
	struct ath12k_wmi_ppe_threshold_arg peer_eht_ppet;
	u32 punct_bitmap;
};

struct wmi_peer_assoc_complete_cmd {
	__le32 tlv_header;
	struct ath12k_wmi_mac_addr_params peer_macaddr;
	__le32 vdev_id;
	__le32 peer_new_assoc;
	__le32 peer_associd;
	__le32 peer_flags;
	__le32 peer_caps;
	__le32 peer_listen_intval;
	__le32 peer_ht_caps;
	__le32 peer_max_mpdu;
	__le32 peer_mpdu_density;
	__le32 peer_rate_caps;
	__le32 peer_nss;
	__le32 peer_vht_caps;
	__le32 peer_phymode;
	__le32 peer_ht_info[2];
	__le32 num_peer_legacy_rates;
	__le32 num_peer_ht_rates;
	__le32 peer_bw_rxnss_override;
	struct ath12k_wmi_ppe_threshold_params peer_ppet;
	__le32 peer_he_cap_info;
	__le32 peer_he_ops;
	__le32 peer_he_cap_phy[WMI_MAX_HECAP_PHY_SIZE];
	__le32 peer_he_mcs;
	__le32 peer_he_cap_info_ext;
	__le32 peer_he_cap_info_internal;
	__le32 min_data_rate;
	__le32 peer_he_caps_6ghz;
	__le32 sta_type;
	__le32 bss_max_idle_option;
	__le32 auth_mode;
	__le32 peer_flags_ext;
	__le32 punct_bitmap;
	__le32 peer_eht_cap_mac[WMI_MAX_EHTCAP_MAC_SIZE];
	__le32 peer_eht_cap_phy[WMI_MAX_EHTCAP_PHY_SIZE];
	__le32 peer_eht_ops;
	struct ath12k_wmi_ppe_threshold_params peer_eht_ppet;
} __packed;

struct wmi_stop_scan_cmd {
	__le32 tlv_header;
	__le32 requestor;
	__le32 scan_id;
	__le32 req_type;
	__le32 vdev_id;
	__le32 pdev_id;
} __packed;

struct ath12k_wmi_scan_chan_list_arg {
	u32 pdev_id;
	u16 nallchans;
	struct ath12k_wmi_channel_arg channel[];
};

struct wmi_scan_chan_list_cmd {
	__le32 tlv_header;
	__le32 num_scan_chans;
	__le32 flags;
	__le32 pdev_id;
} __packed;

#define WMI_MGMT_SEND_DOWNLD_LEN	64

#define WMI_TX_PARAMS_DWORD0_POWER		GENMASK(7, 0)
#define WMI_TX_PARAMS_DWORD0_MCS_MASK		GENMASK(19, 8)
#define WMI_TX_PARAMS_DWORD0_NSS_MASK		GENMASK(27, 20)
#define WMI_TX_PARAMS_DWORD0_RETRY_LIMIT	GENMASK(31, 28)

#define WMI_TX_PARAMS_DWORD1_CHAIN_MASK		GENMASK(7, 0)
#define WMI_TX_PARAMS_DWORD1_BW_MASK		GENMASK(14, 8)
#define WMI_TX_PARAMS_DWORD1_PREAMBLE_TYPE	GENMASK(19, 15)
#define WMI_TX_PARAMS_DWORD1_FRAME_TYPE		BIT(20)
#define WMI_TX_PARAMS_DWORD1_RSVD		GENMASK(31, 21)

struct wmi_mgmt_send_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	__le32 desc_id;
	__le32 chanfreq;
	__le32 paddr_lo;
	__le32 paddr_hi;
	__le32 frame_len;
	__le32 buf_len;
	__le32 tx_params_valid;

	/* This TLV is followed by struct wmi_mgmt_frame */

	/* Followed by struct wmi_mgmt_send_params */
} __packed;

struct wmi_sta_powersave_mode_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	__le32 sta_ps_mode;
} __packed;

struct wmi_sta_smps_force_mode_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	__le32 forced_mode;
} __packed;

struct wmi_sta_smps_param_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	__le32 param;
	__le32 value;
} __packed;

struct ath12k_wmi_bcn_prb_info_params {
	__le32 tlv_header;
	__le32 caps;
	__le32 erp;
} __packed;

enum {
	WMI_PDEV_SUSPEND,
	WMI_PDEV_SUSPEND_AND_DISABLE_INTR,
};

struct wmi_pdev_green_ap_ps_enable_cmd_param {
	__le32 tlv_header;
	__le32 pdev_id;
	__le32 enable;
} __packed;

struct ath12k_wmi_ap_ps_arg {
	u32 vdev_id;
	u32 param;
	u32 value;
};

enum set_init_cc_type {
	WMI_COUNTRY_INFO_TYPE_ALPHA,
	WMI_COUNTRY_INFO_TYPE_COUNTRY_CODE,
	WMI_COUNTRY_INFO_TYPE_REGDOMAIN,
};

enum set_init_cc_flags {
	INVALID_CC,
	CC_IS_SET,
	REGDMN_IS_SET,
	ALPHA_IS_SET,
};

struct ath12k_wmi_init_country_arg {
	union {
		u16 country_code;
		u16 regdom_id;
		u8 alpha2[3];
	} cc_info;
	enum set_init_cc_flags flags;
};

struct wmi_init_country_cmd {
	__le32 tlv_header;
	__le32 pdev_id;
	__le32 init_cc_type;
	union {
		__le32 country_code;
		__le32 regdom_id;
		__le32 alpha2;
	} cc_info;
} __packed;

struct wmi_delba_send_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	struct ath12k_wmi_mac_addr_params peer_macaddr;
	__le32 tid;
	__le32 initiator;
	__le32 reasoncode;
} __packed;

struct wmi_addba_setresponse_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	struct ath12k_wmi_mac_addr_params peer_macaddr;
	__le32 tid;
	__le32 statuscode;
} __packed;

struct wmi_addba_send_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	struct ath12k_wmi_mac_addr_params peer_macaddr;
	__le32 tid;
	__le32 buffersize;
} __packed;

struct wmi_addba_clear_resp_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	struct ath12k_wmi_mac_addr_params peer_macaddr;
} __packed;

#define DFS_PHYERR_UNIT_TEST_CMD 0
#define DFS_UNIT_TEST_MODULE	0x2b
#define DFS_UNIT_TEST_TOKEN	0xAA

enum dfs_test_args_idx {
	DFS_TEST_CMDID = 0,
	DFS_TEST_PDEV_ID,
	DFS_TEST_RADAR_PARAM,
	DFS_MAX_TEST_ARGS,
};

struct wmi_dfs_unit_test_arg {
	u32 cmd_id;
	u32 pdev_id;
	u32 radar_param;
};

struct wmi_unit_test_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	__le32 module_id;
	__le32 num_args;
	__le32 diag_token;
	/* Followed by test args*/
} __packed;

#define MAX_SUPPORTED_RATES 128

#define WMI_PEER_AUTH		0x00000001
#define WMI_PEER_QOS		0x00000002
#define WMI_PEER_NEED_PTK_4_WAY	0x00000004
#define WMI_PEER_NEED_GTK_2_WAY	0x00000010
#define WMI_PEER_HE		0x00000400
#define WMI_PEER_APSD		0x00000800
#define WMI_PEER_HT		0x00001000
#define WMI_PEER_40MHZ		0x00002000
#define WMI_PEER_STBC		0x00008000
#define WMI_PEER_LDPC		0x00010000
#define WMI_PEER_DYN_MIMOPS	0x00020000
#define WMI_PEER_STATIC_MIMOPS	0x00040000
#define WMI_PEER_SPATIAL_MUX	0x00200000
#define WMI_PEER_TWT_REQ	0x00400000
#define WMI_PEER_TWT_RESP	0x00800000
#define WMI_PEER_VHT		0x02000000
#define WMI_PEER_80MHZ		0x04000000
#define WMI_PEER_PMF		0x08000000
/* TODO: Place holder for WLAN_PEER_F_PS_PRESEND_REQUIRED = 0x10000000.
 * Need to be cleaned up
 */
#define WMI_PEER_IS_P2P_CAPABLE	0x20000000
#define WMI_PEER_160MHZ		0x40000000
#define WMI_PEER_SAFEMODE_EN	0x80000000

struct ath12k_wmi_vht_rate_set_params {
	__le32 tlv_header;
	__le32 rx_max_rate;
	__le32 rx_mcs_set;
	__le32 tx_max_rate;
	__le32 tx_mcs_set;
	__le32 tx_max_mcs_nss;
} __packed;

struct ath12k_wmi_he_rate_set_params {
	__le32 tlv_header;
	__le32 rx_mcs_set;
	__le32 tx_mcs_set;
} __packed;

struct ath12k_wmi_eht_rate_set_params {
	__le32 tlv_header;
	__le32 rx_mcs_set;
	__le32 tx_mcs_set;
} __packed;

#define MAX_REG_RULES 10
#define REG_ALPHA2_LEN 2
#define MAX_6G_REG_RULES 5
#define REG_US_5G_NUM_REG_RULES 4

enum wmi_start_event_param {
	WMI_VDEV_START_RESP_EVENT = 0,
	WMI_VDEV_RESTART_RESP_EVENT,
};

struct wmi_vdev_start_resp_event {
	__le32 vdev_id;
	__le32 requestor_id;
	/* enum wmi_start_event_param */
	__le32 resp_type;
	__le32 status;
	__le32 chain_mask;
	__le32 smps_mode;
	union {
		__le32 mac_id;
		__le32 pdev_id;
	};
	__le32 cfgd_tx_streams;
	__le32 cfgd_rx_streams;
} __packed;

/* VDEV start response status codes */
enum wmi_vdev_start_resp_status_code {
	WMI_VDEV_START_RESPONSE_STATUS_SUCCESS = 0,
	WMI_VDEV_START_RESPONSE_INVALID_VDEVID = 1,
	WMI_VDEV_START_RESPONSE_NOT_SUPPORTED = 2,
	WMI_VDEV_START_RESPONSE_DFS_VIOLATION = 3,
	WMI_VDEV_START_RESPONSE_INVALID_REGDOMAIN = 4,
};

enum wmi_reg_6g_ap_type {
	WMI_REG_INDOOR_AP = 0,
	WMI_REG_STD_POWER_AP = 1,
	WMI_REG_VLP_AP = 2,
	WMI_REG_CURRENT_MAX_AP_TYPE,
	WMI_REG_MAX_SUPP_AP_TYPE = WMI_REG_VLP_AP,
	WMI_REG_MAX_AP_TYPE = 7,
};

enum wmi_reg_6g_client_type {
	WMI_REG_DEFAULT_CLIENT = 0,
	WMI_REG_SUBORDINATE_CLIENT = 1,
	WMI_REG_MAX_CLIENT_TYPE = 2,
};

/* Regulatory Rule Flags Passed by FW */
#define REGULATORY_CHAN_DISABLED     BIT(0)
#define REGULATORY_CHAN_NO_IR        BIT(1)
#define REGULATORY_CHAN_RADAR        BIT(3)
#define REGULATORY_CHAN_NO_OFDM      BIT(6)
#define REGULATORY_CHAN_INDOOR_ONLY  BIT(9)

#define REGULATORY_CHAN_NO_HT40      BIT(4)
#define REGULATORY_CHAN_NO_80MHZ     BIT(7)
#define REGULATORY_CHAN_NO_160MHZ    BIT(8)
#define REGULATORY_CHAN_NO_20MHZ     BIT(11)
#define REGULATORY_CHAN_NO_10MHZ     BIT(12)

enum {
	WMI_REG_SET_CC_STATUS_PASS = 0,
	WMI_REG_CURRENT_ALPHA2_NOT_FOUND = 1,
	WMI_REG_INIT_ALPHA2_NOT_FOUND = 2,
	WMI_REG_SET_CC_CHANGE_NOT_ALLOWED = 3,
	WMI_REG_SET_CC_STATUS_NO_MEMORY = 4,
	WMI_REG_SET_CC_STATUS_FAIL = 5,
};

#define WMI_REG_CLIENT_MAX 4

struct wmi_reg_chan_list_cc_ext_event {
	__le32 status_code;
	__le32 phy_id;
	__le32 alpha2;
	__le32 num_phy;
	__le32 country_id;
	__le32 domain_code;
	__le32 dfs_region;
	__le32 phybitmap;
	__le32 min_bw_2g;
	__le32 max_bw_2g;
	__le32 min_bw_5g;
	__le32 max_bw_5g;
	__le32 num_2g_reg_rules;
	__le32 num_5g_reg_rules;
	__le32 client_type;
	__le32 rnr_tpe_usable;
	__le32 unspecified_ap_usable;
	__le32 domain_code_6g_ap_lpi;
	__le32 domain_code_6g_ap_sp;
	__le32 domain_code_6g_ap_vlp;
	__le32 domain_code_6g_client_lpi[WMI_REG_CLIENT_MAX];
	__le32 domain_code_6g_client_sp[WMI_REG_CLIENT_MAX];
	__le32 domain_code_6g_client_vlp[WMI_REG_CLIENT_MAX];
	__le32 domain_code_6g_super_id;
	__le32 min_bw_6g_ap_sp;
	__le32 max_bw_6g_ap_sp;
	__le32 min_bw_6g_ap_lpi;
	__le32 max_bw_6g_ap_lpi;
	__le32 min_bw_6g_ap_vlp;
	__le32 max_bw_6g_ap_vlp;
	__le32 min_bw_6g_client_sp[WMI_REG_CLIENT_MAX];
	__le32 max_bw_6g_client_sp[WMI_REG_CLIENT_MAX];
	__le32 min_bw_6g_client_lpi[WMI_REG_CLIENT_MAX];
	__le32 max_bw_6g_client_lpi[WMI_REG_CLIENT_MAX];
	__le32 min_bw_6g_client_vlp[WMI_REG_CLIENT_MAX];
	__le32 max_bw_6g_client_vlp[WMI_REG_CLIENT_MAX];
	__le32 num_6g_reg_rules_ap_sp;
	__le32 num_6g_reg_rules_ap_lpi;
	__le32 num_6g_reg_rules_ap_vlp;
	__le32 num_6g_reg_rules_cl_sp[WMI_REG_CLIENT_MAX];
	__le32 num_6g_reg_rules_cl_lpi[WMI_REG_CLIENT_MAX];
	__le32 num_6g_reg_rules_cl_vlp[WMI_REG_CLIENT_MAX];
} __packed;

struct ath12k_wmi_reg_rule_ext_params {
	__le32 tlv_header;
	__le32 freq_info;
	__le32 bw_pwr_info;
	__le32 flag_info;
	__le32 psd_power_info;
} __packed;

struct wmi_vdev_delete_resp_event {
	__le32 vdev_id;
} __packed;

struct wmi_peer_delete_resp_event {
	__le32 vdev_id;
	struct ath12k_wmi_mac_addr_params peer_macaddr;
} __packed;

struct wmi_bcn_tx_status_event {
	__le32 vdev_id;
	__le32 tx_status;
} __packed;

struct wmi_vdev_stopped_event {
	__le32 vdev_id;
} __packed;

struct wmi_pdev_bss_chan_info_event {
	__le32 pdev_id;
	__le32 freq;	/* Units in MHz */
	__le32 noise_floor;	/* units are dBm */
	/* rx clear - how often the channel was unused */
	__le32 rx_clear_count_low;
	__le32 rx_clear_count_high;
	/* cycle count - elapsed time during measured period, in clock ticks */
	__le32 cycle_count_low;
	__le32 cycle_count_high;
	/* tx cycle count - elapsed time spent in tx, in clock ticks */
	__le32 tx_cycle_count_low;
	__le32 tx_cycle_count_high;
	/* rx cycle count - elapsed time spent in rx, in clock ticks */
	__le32 rx_cycle_count_low;
	__le32 rx_cycle_count_high;
	/*rx_cycle cnt for my bss in 64bits format */
	__le32 rx_bss_cycle_count_low;
	__le32 rx_bss_cycle_count_high;
} __packed;

#define WMI_VDEV_INSTALL_KEY_COMPL_STATUS_SUCCESS 0

struct wmi_vdev_install_key_compl_event {
	__le32 vdev_id;
	struct ath12k_wmi_mac_addr_params peer_macaddr;
	__le32 key_idx;
	__le32 key_flags;
	__le32 status;
} __packed;

struct wmi_vdev_install_key_complete_arg {
	u32 vdev_id;
	const u8 *macaddr;
	u32 key_idx;
	u32 key_flags;
	u32 status;
};

struct wmi_peer_assoc_conf_event {
	__le32 vdev_id;
	struct ath12k_wmi_mac_addr_params peer_macaddr;
} __packed;

struct wmi_peer_assoc_conf_arg {
	u32 vdev_id;
	const u8 *macaddr;
};

struct wmi_fils_discovery_event {
	__le32 vdev_id;
	__le32 fils_tt;
	__le32 tbtt;
} __packed;

struct wmi_probe_resp_tx_status_event {
	__le32 vdev_id;
	__le32 tx_status;
} __packed;

struct wmi_pdev_ctl_failsafe_chk_event {
	__le32 pdev_id;
	__le32 ctl_failsafe_status;
} __packed;

struct ath12k_wmi_pdev_csa_event {
	__le32 pdev_id;
	__le32 current_switch_count;
	__le32 num_vdevs;
} __packed;

struct ath12k_wmi_pdev_radar_event {
	__le32 pdev_id;
	__le32 detection_mode;
	__le32 chan_freq;
	__le32 chan_width;
	__le32 detector_id;
	__le32 segment_id;
	__le32 timestamp;
	__le32 is_chirp;
	a_sle32 freq_offset;
	a_sle32 sidx;
} __packed;

struct wmi_pdev_temperature_event {
	/* temperature value in Celsius degree */
	a_sle32 temp;
	__le32 pdev_id;
} __packed;

#define WMI_RX_STATUS_OK			0x00
#define WMI_RX_STATUS_ERR_CRC			0x01
#define WMI_RX_STATUS_ERR_DECRYPT		0x08
#define WMI_RX_STATUS_ERR_MIC			0x10
#define WMI_RX_STATUS_ERR_KEY_CACHE_MISS	0x20

#define WLAN_MGMT_TXRX_HOST_MAX_ANTENNA 4

struct ath12k_wmi_mgmt_rx_arg {
	u32 chan_freq;
	u32 channel;
	u32 snr;
	u8 rssi_ctl[WLAN_MGMT_TXRX_HOST_MAX_ANTENNA];
	u32 rate;
	enum wmi_phy_mode phy_mode;
	u32 buf_len;
	int status;
	u32 flags;
	int rssi;
	u32 tsf_delta;
	u8 pdev_id;
};

#define ATH_MAX_ANTENNA 4

struct ath12k_wmi_mgmt_rx_params {
	__le32 channel;
	__le32 snr;
	__le32 rate;
	__le32 phy_mode;
	__le32 buf_len;
	__le32 status;
	__le32 rssi_ctl[ATH_MAX_ANTENNA];
	__le32 flags;
	a_sle32 rssi;
	__le32 tsf_delta;
	__le32 rx_tsf_l32;
	__le32 rx_tsf_u32;
	__le32 pdev_id;
	__le32 chan_freq;
} __packed;

#define MAX_ANTENNA_EIGHT 8

struct wmi_mgmt_tx_compl_event {
	__le32 desc_id;
	__le32 status;
	__le32 pdev_id;
} __packed;

struct wmi_scan_event {
	__le32 event_type; /* %WMI_SCAN_EVENT_ */
	__le32 reason; /* %WMI_SCAN_REASON_ */
	__le32 channel_freq; /* only valid for WMI_SCAN_EVENT_FOREIGN_CHANNEL */
	__le32 scan_req_id;
	__le32 scan_id;
	__le32 vdev_id;
	/* TSF Timestamp when the scan event (%WMI_SCAN_EVENT_) is completed
	 * In case of AP it is TSF of the AP vdev
	 * In case of STA connected state, this is the TSF of the AP
	 * In case of STA not connected, it will be the free running HW timer
	 */
	__le32 tsf_timestamp;
} __packed;

struct wmi_peer_sta_kickout_arg {
	const u8 *mac_addr;
};

struct wmi_peer_sta_kickout_event {
	struct ath12k_wmi_mac_addr_params peer_macaddr;
} __packed;

enum wmi_roam_reason {
	WMI_ROAM_REASON_BETTER_AP = 1,
	WMI_ROAM_REASON_BEACON_MISS = 2,
	WMI_ROAM_REASON_LOW_RSSI = 3,
	WMI_ROAM_REASON_SUITABLE_AP_FOUND = 4,
	WMI_ROAM_REASON_HO_FAILED = 5,

	/* keep last */
	WMI_ROAM_REASON_MAX,
};

struct wmi_roam_event {
	__le32 vdev_id;
	__le32 reason;
	__le32 rssi;
} __packed;

#define WMI_CHAN_INFO_START_RESP 0
#define WMI_CHAN_INFO_END_RESP 1

struct wmi_chan_info_event {
	__le32 err_code;
	__le32 freq;
	__le32 cmd_flags;
	__le32 noise_floor;
	__le32 rx_clear_count;
	__le32 cycle_count;
	__le32 chan_tx_pwr_range;
	__le32 chan_tx_pwr_tp;
	__le32 rx_frame_count;
	__le32 my_bss_rx_cycle_count;
	__le32 rx_11b_mode_data_duration;
	__le32 tx_frame_cnt;
	__le32 mac_clk_mhz;
	__le32 vdev_id;
} __packed;

struct ath12k_wmi_target_cap_arg {
	u32 phy_capability;
	u32 max_frag_entry;
	u32 num_rf_chains;
	u32 ht_cap_info;
	u32 vht_cap_info;
	u32 vht_supp_mcs;
	u32 hw_min_tx_power;
	u32 hw_max_tx_power;
	u32 sys_cap_info;
	u32 min_pkt_size_enable;
	u32 max_bcn_ie_size;
	u32 max_num_scan_channels;
	u32 max_supported_macs;
	u32 wmi_fw_sub_feat_caps;
	u32 txrx_chainmask;
	u32 default_dbs_hw_mode_index;
	u32 num_msdu_desc;
};

enum wmi_vdev_type {
	WMI_VDEV_TYPE_AP      = 1,
	WMI_VDEV_TYPE_STA     = 2,
	WMI_VDEV_TYPE_IBSS    = 3,
	WMI_VDEV_TYPE_MONITOR = 4,
};

enum wmi_vdev_subtype {
	WMI_VDEV_SUBTYPE_NONE,
	WMI_VDEV_SUBTYPE_P2P_DEVICE,
	WMI_VDEV_SUBTYPE_P2P_CLIENT,
	WMI_VDEV_SUBTYPE_P2P_GO,
	WMI_VDEV_SUBTYPE_PROXY_STA,
	WMI_VDEV_SUBTYPE_MESH_NON_11S,
	WMI_VDEV_SUBTYPE_MESH_11S,
};

enum wmi_sta_powersave_param {
	WMI_STA_PS_PARAM_RX_WAKE_POLICY = 0,
	WMI_STA_PS_PARAM_TX_WAKE_THRESHOLD = 1,
	WMI_STA_PS_PARAM_PSPOLL_COUNT = 2,
	WMI_STA_PS_PARAM_INACTIVITY_TIME = 3,
	WMI_STA_PS_PARAM_UAPSD = 4,
};

enum wmi_sta_ps_param_uapsd {
	WMI_STA_PS_UAPSD_AC0_DELIVERY_EN = (1 << 0),
	WMI_STA_PS_UAPSD_AC0_TRIGGER_EN  = (1 << 1),
	WMI_STA_PS_UAPSD_AC1_DELIVERY_EN = (1 << 2),
	WMI_STA_PS_UAPSD_AC1_TRIGGER_EN  = (1 << 3),
	WMI_STA_PS_UAPSD_AC2_DELIVERY_EN = (1 << 4),
	WMI_STA_PS_UAPSD_AC2_TRIGGER_EN  = (1 << 5),
	WMI_STA_PS_UAPSD_AC3_DELIVERY_EN = (1 << 6),
	WMI_STA_PS_UAPSD_AC3_TRIGGER_EN  = (1 << 7),
};

enum wmi_sta_ps_param_tx_wake_threshold {
	WMI_STA_PS_TX_WAKE_THRESHOLD_NEVER = 0,
	WMI_STA_PS_TX_WAKE_THRESHOLD_ALWAYS = 1,

	/* Values greater than one indicate that many TX attempts per beacon
	 * interval before the STA will wake up
	 */
};

/* The maximum number of PS-Poll frames the FW will send in response to
 * traffic advertised in TIM before waking up (by sending a null frame with PS
 * = 0). Value 0 has a special meaning: there is no maximum count and the FW
 * will send as many PS-Poll as are necessary to retrieve buffered BU. This
 * parameter is used when the RX wake policy is
 * WMI_STA_PS_RX_WAKE_POLICY_POLL_UAPSD and ignored when the RX wake
 * policy is WMI_STA_PS_RX_WAKE_POLICY_WAKE.
 */
enum wmi_sta_ps_param_pspoll_count {
	WMI_STA_PS_PSPOLL_COUNT_NO_MAX = 0,
	/* Values greater than 0 indicate the maximum number of PS-Poll frames
	 * FW will send before waking up.
	 */
};

/* U-APSD configuration of peer station from (re)assoc request and TSPECs */
enum wmi_ap_ps_param_uapsd {
	WMI_AP_PS_UAPSD_AC0_DELIVERY_EN = (1 << 0),
	WMI_AP_PS_UAPSD_AC0_TRIGGER_EN  = (1 << 1),
	WMI_AP_PS_UAPSD_AC1_DELIVERY_EN = (1 << 2),
	WMI_AP_PS_UAPSD_AC1_TRIGGER_EN  = (1 << 3),
	WMI_AP_PS_UAPSD_AC2_DELIVERY_EN = (1 << 4),
	WMI_AP_PS_UAPSD_AC2_TRIGGER_EN  = (1 << 5),
	WMI_AP_PS_UAPSD_AC3_DELIVERY_EN = (1 << 6),
	WMI_AP_PS_UAPSD_AC3_TRIGGER_EN  = (1 << 7),
};

/* U-APSD maximum service period of peer station */
enum wmi_ap_ps_peer_param_max_sp {
	WMI_AP_PS_PEER_PARAM_MAX_SP_UNLIMITED = 0,
	WMI_AP_PS_PEER_PARAM_MAX_SP_2 = 1,
	WMI_AP_PS_PEER_PARAM_MAX_SP_4 = 2,
	WMI_AP_PS_PEER_PARAM_MAX_SP_6 = 3,
	MAX_WMI_AP_PS_PEER_PARAM_MAX_SP,
};

enum wmi_ap_ps_peer_param {
	/** Set uapsd configuration for a given peer.
	 *
	 * This include the delivery and trigger enabled state for each AC.
	 * The host MLME needs to set this based on AP capability and stations
	 * request Set in the association request  received from the station.
	 *
	 * Lower 8 bits of the value specify the UAPSD configuration.
	 *
	 * (see enum wmi_ap_ps_param_uapsd)
	 * The default value is 0.
	 */
	WMI_AP_PS_PEER_PARAM_UAPSD = 0,

	/**
	 * Set the service period for a UAPSD capable station
	 *
	 * The service period from wme ie in the (re)assoc request frame.
	 *
	 * (see enum wmi_ap_ps_peer_param_max_sp)
	 */
	WMI_AP_PS_PEER_PARAM_MAX_SP = 1,

	/** Time in seconds for aging out buffered frames
	 * for STA in power save
	 */
	WMI_AP_PS_PEER_PARAM_AGEOUT_TIME = 2,

	/** Specify frame types that are considered SIFS
	 * RESP trigger frame
	 */
	WMI_AP_PS_PEER_PARAM_SIFS_RESP_FRMTYPE = 3,

	/** Specifies the trigger state of TID.
	 * Valid only for UAPSD frame type
	 */
	WMI_AP_PS_PEER_PARAM_SIFS_RESP_UAPSD = 4,

	/* Specifies the WNM sleep state of a STA */
	WMI_AP_PS_PEER_PARAM_WNM_SLEEP = 5,
};

#define DISABLE_SIFS_RESPONSE_TRIGGER 0

#define WMI_MAX_KEY_INDEX   3
#define WMI_MAX_KEY_LEN     32

enum wmi_key_type {
	WMI_KEY_PAIRWISE = 0,
	WMI_KEY_GROUP = 1,
};

enum wmi_cipher_type {
	WMI_CIPHER_NONE = 0, /* clear key */
	WMI_CIPHER_WEP = 1,
	WMI_CIPHER_TKIP = 2,
	WMI_CIPHER_AES_OCB = 3,
	WMI_CIPHER_AES_CCM = 4,
	WMI_CIPHER_WAPI = 5,
	WMI_CIPHER_CKIP = 6,
	WMI_CIPHER_AES_CMAC = 7,
	WMI_CIPHER_ANY = 8,
	WMI_CIPHER_AES_GCM = 9,
	WMI_CIPHER_AES_GMAC = 10,
};

/* Value to disable fixed rate setting */
#define WMI_FIXED_RATE_NONE	(0xffff)

#define ATH12K_RC_VERSION_OFFSET	28
#define ATH12K_RC_PREAMBLE_OFFSET	8
#define ATH12K_RC_NSS_OFFSET		5

#define ATH12K_HW_RATE_CODE(rate, nss, preamble)	\
	((1 << ATH12K_RC_VERSION_OFFSET) |		\
	 ((nss) << ATH12K_RC_NSS_OFFSET) |		\
	 ((preamble) << ATH12K_RC_PREAMBLE_OFFSET) |	\
	 (rate))

/* Preamble types to be used with VDEV fixed rate configuration */
enum wmi_rate_preamble {
	WMI_RATE_PREAMBLE_OFDM,
	WMI_RATE_PREAMBLE_CCK,
	WMI_RATE_PREAMBLE_HT,
	WMI_RATE_PREAMBLE_VHT,
	WMI_RATE_PREAMBLE_HE,
};

/**
 * enum wmi_rtscts_prot_mode - Enable/Disable RTS/CTS and CTS2Self Protection.
 * @WMI_RTS_CTS_DISABLED: RTS/CTS protection is disabled.
 * @WMI_USE_RTS_CTS: RTS/CTS Enabled.
 * @WMI_USE_CTS2SELF: CTS to self protection Enabled.
 */
enum wmi_rtscts_prot_mode {
	WMI_RTS_CTS_DISABLED = 0,
	WMI_USE_RTS_CTS = 1,
	WMI_USE_CTS2SELF = 2,
};

/**
 * enum wmi_rtscts_profile - Selection of RTS CTS profile along with enabling
 *                           protection mode.
 * @WMI_RTSCTS_FOR_NO_RATESERIES: Neither of rate-series should use RTS-CTS
 * @WMI_RTSCTS_FOR_SECOND_RATESERIES: Only second rate-series will use RTS-CTS
 * @WMI_RTSCTS_ACROSS_SW_RETRIES: Only the second rate-series will use RTS-CTS,
 *                                but if there's a sw retry, both the rate
 *                                series will use RTS-CTS.
 * @WMI_RTSCTS_ERP: RTS/CTS used for ERP protection for every PPDU.
 * @WMI_RTSCTS_FOR_ALL_RATESERIES: Enable RTS-CTS for all rate series.
 */
enum wmi_rtscts_profile {
	WMI_RTSCTS_FOR_NO_RATESERIES = 0,
	WMI_RTSCTS_FOR_SECOND_RATESERIES = 1,
	WMI_RTSCTS_ACROSS_SW_RETRIES = 2,
	WMI_RTSCTS_ERP = 3,
	WMI_RTSCTS_FOR_ALL_RATESERIES = 4,
};

#define WMI_SKB_HEADROOM sizeof(struct wmi_cmd_hdr)

enum wmi_sta_ps_param_rx_wake_policy {
	WMI_STA_PS_RX_WAKE_POLICY_WAKE = 0,
	WMI_STA_PS_RX_WAKE_POLICY_POLL_UAPSD = 1,
};

/* Do not change existing values! Used by ath12k_frame_mode parameter
 * module parameter.
 */
enum ath12k_hw_txrx_mode {
	ATH12K_HW_TXRX_RAW = 0,
	ATH12K_HW_TXRX_NATIVE_WIFI = 1,
	ATH12K_HW_TXRX_ETHERNET = 2,
};

struct wmi_wmm_params {
	__le32 tlv_header;
	__le32 cwmin;
	__le32 cwmax;
	__le32 aifs;
	__le32 txoplimit;
	__le32 acm;
	__le32 no_ack;
} __packed;

struct wmi_wmm_params_arg {
	u8 acm;
	u8 aifs;
	u16 cwmin;
	u16 cwmax;
	u16 txop;
	u8 no_ack;
};

struct wmi_vdev_set_wmm_params_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	struct wmi_wmm_params wmm_params[4];
	__le32 wmm_param_type;
} __packed;

struct wmi_wmm_params_all_arg {
	struct wmi_wmm_params_arg ac_be;
	struct wmi_wmm_params_arg ac_bk;
	struct wmi_wmm_params_arg ac_vi;
	struct wmi_wmm_params_arg ac_vo;
};

#define ATH12K_TWT_DEF_STA_CONG_TIMER_MS		5000
#define ATH12K_TWT_DEF_DEFAULT_SLOT_SIZE		10
#define ATH12K_TWT_DEF_CONGESTION_THRESH_SETUP		50
#define ATH12K_TWT_DEF_CONGESTION_THRESH_TEARDOWN	20
#define ATH12K_TWT_DEF_CONGESTION_THRESH_CRITICAL	100
#define ATH12K_TWT_DEF_INTERFERENCE_THRESH_TEARDOWN	80
#define ATH12K_TWT_DEF_INTERFERENCE_THRESH_SETUP	50
#define ATH12K_TWT_DEF_MIN_NO_STA_SETUP			10
#define ATH12K_TWT_DEF_MIN_NO_STA_TEARDOWN		2
#define ATH12K_TWT_DEF_NO_OF_BCAST_MCAST_SLOTS		2
#define ATH12K_TWT_DEF_MIN_NO_TWT_SLOTS			2
#define ATH12K_TWT_DEF_MAX_NO_STA_TWT			500
#define ATH12K_TWT_DEF_MODE_CHECK_INTERVAL		10000
#define ATH12K_TWT_DEF_ADD_STA_SLOT_INTERVAL		1000
#define ATH12K_TWT_DEF_REMOVE_STA_SLOT_INTERVAL		5000

struct wmi_twt_enable_params_cmd {
	__le32 tlv_header;
	__le32 pdev_id;
	__le32 sta_cong_timer_ms;
	__le32 mbss_support;
	__le32 default_slot_size;
	__le32 congestion_thresh_setup;
	__le32 congestion_thresh_teardown;
	__le32 congestion_thresh_critical;
	__le32 interference_thresh_teardown;
	__le32 interference_thresh_setup;
	__le32 min_no_sta_setup;
	__le32 min_no_sta_teardown;
	__le32 no_of_bcast_mcast_slots;
	__le32 min_no_twt_slots;
	__le32 max_no_sta_twt;
	__le32 mode_check_interval;
	__le32 add_sta_slot_interval;
	__le32 remove_sta_slot_interval;
} __packed;

struct wmi_twt_disable_params_cmd {
	__le32 tlv_header;
	__le32 pdev_id;
} __packed;

struct wmi_obss_spatial_reuse_params_cmd {
	__le32 tlv_header;
	__le32 pdev_id;
	__le32 enable;
	a_sle32 obss_min;
	a_sle32 obss_max;
	__le32 vdev_id;
} __packed;

#define ATH12K_BSS_COLOR_COLLISION_SCAN_PERIOD_MS		200
#define ATH12K_OBSS_COLOR_COLLISION_DETECTION_DISABLE		0
#define ATH12K_OBSS_COLOR_COLLISION_DETECTION			1

#define ATH12K_BSS_COLOR_STA_PERIODS				10000
#define ATH12K_BSS_COLOR_AP_PERIODS				5000

struct wmi_obss_color_collision_cfg_params_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	__le32 flags;
	__le32 evt_type;
	__le32 current_bss_color;
	__le32 detection_period_ms;
	__le32 scan_period_ms;
	__le32 free_slot_expiry_time_ms;
} __packed;

struct wmi_bss_color_change_enable_params_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	__le32 enable;
} __packed;

#define ATH12K_IPV4_TH_SEED_SIZE 5
#define ATH12K_IPV6_TH_SEED_SIZE 11

struct ath12k_wmi_pdev_lro_config_cmd {
	__le32 tlv_header;
	__le32 lro_enable;
	__le32 res;
	u32 th_4[ATH12K_IPV4_TH_SEED_SIZE];
	u32 th_6[ATH12K_IPV6_TH_SEED_SIZE];
	__le32 pdev_id;
} __packed;

#define ATH12K_WMI_SPECTRAL_COUNT_DEFAULT                 0
#define ATH12K_WMI_SPECTRAL_PERIOD_DEFAULT              224
#define ATH12K_WMI_SPECTRAL_PRIORITY_DEFAULT              1
#define ATH12K_WMI_SPECTRAL_FFT_SIZE_DEFAULT              7
#define ATH12K_WMI_SPECTRAL_GC_ENA_DEFAULT                1
#define ATH12K_WMI_SPECTRAL_RESTART_ENA_DEFAULT           0
#define ATH12K_WMI_SPECTRAL_NOISE_FLOOR_REF_DEFAULT     -96
#define ATH12K_WMI_SPECTRAL_INIT_DELAY_DEFAULT           80
#define ATH12K_WMI_SPECTRAL_NB_TONE_THR_DEFAULT          12
#define ATH12K_WMI_SPECTRAL_STR_BIN_THR_DEFAULT           8
#define ATH12K_WMI_SPECTRAL_WB_RPT_MODE_DEFAULT           0
#define ATH12K_WMI_SPECTRAL_RSSI_RPT_MODE_DEFAULT         0
#define ATH12K_WMI_SPECTRAL_RSSI_THR_DEFAULT           0xf0
#define ATH12K_WMI_SPECTRAL_PWR_FORMAT_DEFAULT            0
#define ATH12K_WMI_SPECTRAL_RPT_MODE_DEFAULT              2
#define ATH12K_WMI_SPECTRAL_BIN_SCALE_DEFAULT             1
#define ATH12K_WMI_SPECTRAL_DBM_ADJ_DEFAULT               1
#define ATH12K_WMI_SPECTRAL_CHN_MASK_DEFAULT              1

struct ath12k_wmi_vdev_spectral_conf_arg {
	u32 vdev_id;
	u32 scan_count;
	u32 scan_period;
	u32 scan_priority;
	u32 scan_fft_size;
	u32 scan_gc_ena;
	u32 scan_restart_ena;
	u32 scan_noise_floor_ref;
	u32 scan_init_delay;
	u32 scan_nb_tone_thr;
	u32 scan_str_bin_thr;
	u32 scan_wb_rpt_mode;
	u32 scan_rssi_rpt_mode;
	u32 scan_rssi_thr;
	u32 scan_pwr_format;
	u32 scan_rpt_mode;
	u32 scan_bin_scale;
	u32 scan_dbm_adj;
	u32 scan_chn_mask;
};

struct ath12k_wmi_vdev_spectral_conf_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	__le32 scan_count;
	__le32 scan_period;
	__le32 scan_priority;
	__le32 scan_fft_size;
	__le32 scan_gc_ena;
	__le32 scan_restart_ena;
	__le32 scan_noise_floor_ref;
	__le32 scan_init_delay;
	__le32 scan_nb_tone_thr;
	__le32 scan_str_bin_thr;
	__le32 scan_wb_rpt_mode;
	__le32 scan_rssi_rpt_mode;
	__le32 scan_rssi_thr;
	__le32 scan_pwr_format;
	__le32 scan_rpt_mode;
	__le32 scan_bin_scale;
	__le32 scan_dbm_adj;
	__le32 scan_chn_mask;
} __packed;

#define ATH12K_WMI_SPECTRAL_TRIGGER_CMD_TRIGGER  1
#define ATH12K_WMI_SPECTRAL_TRIGGER_CMD_CLEAR    2
#define ATH12K_WMI_SPECTRAL_ENABLE_CMD_ENABLE    1
#define ATH12K_WMI_SPECTRAL_ENABLE_CMD_DISABLE   2

struct ath12k_wmi_vdev_spectral_enable_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	__le32 trigger_cmd;
	__le32 enable_cmd;
} __packed;

struct ath12k_wmi_pdev_dma_ring_cfg_arg {
	u32 tlv_header;
	u32 pdev_id;
	u32 module_id;
	u32 base_paddr_lo;
	u32 base_paddr_hi;
	u32 head_idx_paddr_lo;
	u32 head_idx_paddr_hi;
	u32 tail_idx_paddr_lo;
	u32 tail_idx_paddr_hi;
	u32 num_elems;
	u32 buf_size;
	u32 num_resp_per_event;
	u32 event_timeout_ms;
};

struct ath12k_wmi_pdev_dma_ring_cfg_req_cmd {
	__le32 tlv_header;
	__le32 pdev_id;
	__le32 module_id;		/* see enum wmi_direct_buffer_module */
	__le32 base_paddr_lo;
	__le32 base_paddr_hi;
	__le32 head_idx_paddr_lo;
	__le32 head_idx_paddr_hi;
	__le32 tail_idx_paddr_lo;
	__le32 tail_idx_paddr_hi;
	__le32 num_elems;		/* Number of elems in the ring */
	__le32 buf_size;		/* size of allocated buffer in bytes */

	/* Number of wmi_dma_buf_release_entry packed together */
	__le32 num_resp_per_event;

	/* Target should timeout and send whatever resp
	 * it has if this time expires, units in milliseconds
	 */
	__le32 event_timeout_ms;
} __packed;

struct ath12k_wmi_dma_buf_release_fixed_params {
	__le32 pdev_id;
	__le32 module_id;
	__le32 num_buf_release_entry;
	__le32 num_meta_data_entry;
} __packed;

struct ath12k_wmi_dma_buf_release_entry_params {
	__le32 tlv_header;
	__le32 paddr_lo;

	/* Bits 11:0:   address of data
	 * Bits 31:12:  host context data
	 */
	__le32 paddr_hi;
} __packed;

#define WMI_SPECTRAL_META_INFO1_FREQ1		GENMASK(15, 0)
#define WMI_SPECTRAL_META_INFO1_FREQ2		GENMASK(31, 16)

#define WMI_SPECTRAL_META_INFO2_CHN_WIDTH	GENMASK(7, 0)

struct ath12k_wmi_dma_buf_release_meta_data_params {
	__le32 tlv_header;
	a_sle32 noise_floor[WMI_MAX_CHAINS];
	__le32 reset_delay;
	__le32 freq1;
	__le32 freq2;
	__le32 ch_width;
} __packed;

enum wmi_fils_discovery_cmd_type {
	WMI_FILS_DISCOVERY_CMD,
	WMI_UNSOL_BCAST_PROBE_RESP,
};

struct wmi_fils_discovery_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	__le32 interval;
	__le32 config; /* enum wmi_fils_discovery_cmd_type */
} __packed;

struct wmi_fils_discovery_tmpl_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	__le32 buf_len;
} __packed;

struct wmi_probe_tmpl_cmd {
	__le32 tlv_header;
	__le32 vdev_id;
	__le32 buf_len;
} __packed;

#define WMI_MAX_MEM_REQS 32

#define MAX_RADIOS 3

#define WMI_SERVICE_READY_TIMEOUT_HZ (5 * HZ)
#define WMI_SEND_TIMEOUT_HZ (3 * HZ)

struct ath12k_wmi_pdev {
	struct ath12k_wmi_base *wmi_ab;
	enum ath12k_htc_ep_id eid;
	const struct wmi_peer_flags_map *peer_flags;
	u32 rx_decap_mode;
};

struct ath12k_wmi_base {
	struct ath12k_base *ab;
	struct ath12k_wmi_pdev wmi[MAX_RADIOS];
	enum ath12k_htc_ep_id wmi_endpoint_id[MAX_RADIOS];
	u32 max_msg_len[MAX_RADIOS];

	struct completion service_ready;
	struct completion unified_ready;
	DECLARE_BITMAP(svc_map, WMI_MAX_EXT2_SERVICE);
	wait_queue_head_t tx_credits_wq;
	const struct wmi_peer_flags_map *peer_flags;
	u32 num_mem_chunks;
	u32 rx_decap_mode;
	struct ath12k_wmi_host_mem_chunk_arg mem_chunks[WMI_MAX_MEM_REQS];

	enum wmi_host_hw_mode_config_type preferred_hw_mode;

	struct ath12k_wmi_target_cap_arg *targ_cap;
};

#define ATH12K_FW_STATS_BUF_SIZE (1024 * 1024)

void ath12k_wmi_init_qcn9274(struct ath12k_base *ab,
			     struct ath12k_wmi_resource_config_arg *config);
void ath12k_wmi_init_wcn7850(struct ath12k_base *ab,
			     struct ath12k_wmi_resource_config_arg *config);
int ath12k_wmi_cmd_send(struct ath12k_wmi_pdev *wmi, struct sk_buff *skb,
			u32 cmd_id);
struct sk_buff *ath12k_wmi_alloc_skb(struct ath12k_wmi_base *wmi_sc, u32 len);
int ath12k_wmi_mgmt_send(struct ath12k *ar, u32 vdev_id, u32 buf_id,
			 struct sk_buff *frame);
int ath12k_wmi_bcn_tmpl(struct ath12k *ar, u32 vdev_id,
			struct ieee80211_mutable_offsets *offs,
			struct sk_buff *bcn);
int ath12k_wmi_vdev_down(struct ath12k *ar, u8 vdev_id);
int ath12k_wmi_vdev_up(struct ath12k *ar, u32 vdev_id, u32 aid,
		       const u8 *bssid);
int ath12k_wmi_vdev_stop(struct ath12k *ar, u8 vdev_id);
int ath12k_wmi_vdev_start(struct ath12k *ar, struct wmi_vdev_start_req_arg *arg,
			  bool restart);
int ath12k_wmi_set_peer_param(struct ath12k *ar, const u8 *peer_addr,
			      u32 vdev_id, u32 param_id, u32 param_val);
int ath12k_wmi_pdev_set_param(struct ath12k *ar, u32 param_id,
			      u32 param_value, u8 pdev_id);
int ath12k_wmi_pdev_set_ps_mode(struct ath12k *ar, int vdev_id, u32 enable);
int ath12k_wmi_wait_for_unified_ready(struct ath12k_base *ab);
int ath12k_wmi_cmd_init(struct ath12k_base *ab);
int ath12k_wmi_wait_for_service_ready(struct ath12k_base *ab);
int ath12k_wmi_connect(struct ath12k_base *ab);
int ath12k_wmi_pdev_attach(struct ath12k_base *ab,
			   u8 pdev_id);
int ath12k_wmi_attach(struct ath12k_base *ab);
void ath12k_wmi_detach(struct ath12k_base *ab);
int ath12k_wmi_vdev_create(struct ath12k *ar, u8 *macaddr,
			   struct ath12k_wmi_vdev_create_arg *arg);
int ath12k_wmi_send_peer_create_cmd(struct ath12k *ar,
				    struct ath12k_wmi_peer_create_arg *arg);
int ath12k_wmi_vdev_set_param_cmd(struct ath12k *ar, u32 vdev_id,
				  u32 param_id, u32 param_value);

int ath12k_wmi_set_sta_ps_param(struct ath12k *ar, u32 vdev_id,
				u32 param, u32 param_value);
int ath12k_wmi_force_fw_hang_cmd(struct ath12k *ar, u32 type, u32 delay_time_ms);
int ath12k_wmi_send_peer_delete_cmd(struct ath12k *ar,
				    const u8 *peer_addr, u8 vdev_id);
int ath12k_wmi_vdev_delete(struct ath12k *ar, u8 vdev_id);
void ath12k_wmi_start_scan_init(struct ath12k *ar,
				struct ath12k_wmi_scan_req_arg *arg);
int ath12k_wmi_send_scan_start_cmd(struct ath12k *ar,
				   struct ath12k_wmi_scan_req_arg *arg);
int ath12k_wmi_send_scan_stop_cmd(struct ath12k *ar,
				  struct ath12k_wmi_scan_cancel_arg *arg);
int ath12k_wmi_send_wmm_update_cmd(struct ath12k *ar, u32 vdev_id,
				   struct wmi_wmm_params_all_arg *param);
int ath12k_wmi_pdev_suspend(struct ath12k *ar, u32 suspend_opt,
			    u32 pdev_id);
int ath12k_wmi_pdev_resume(struct ath12k *ar, u32 pdev_id);

int ath12k_wmi_send_peer_assoc_cmd(struct ath12k *ar,
				   struct ath12k_wmi_peer_assoc_arg *arg);
int ath12k_wmi_vdev_install_key(struct ath12k *ar,
				struct wmi_vdev_install_key_arg *arg);
int ath12k_wmi_pdev_bss_chan_info_request(struct ath12k *ar,
					  enum wmi_bss_chan_info_req_type type);
int ath12k_wmi_send_pdev_temperature_cmd(struct ath12k *ar);
int ath12k_wmi_send_peer_flush_tids_cmd(struct ath12k *ar,
					u8 peer_addr[ETH_ALEN],
					u32 peer_tid_bitmap,
					u8 vdev_id);
int ath12k_wmi_send_set_ap_ps_param_cmd(struct ath12k *ar, u8 *peer_addr,
					struct ath12k_wmi_ap_ps_arg *arg);
int ath12k_wmi_send_scan_chan_list_cmd(struct ath12k *ar,
				       struct ath12k_wmi_scan_chan_list_arg *arg);
int ath12k_wmi_send_dfs_phyerr_offload_enable_cmd(struct ath12k *ar,
						  u32 pdev_id);
int ath12k_wmi_addba_clear_resp(struct ath12k *ar, u32 vdev_id, const u8 *mac);
int ath12k_wmi_addba_send(struct ath12k *ar, u32 vdev_id, const u8 *mac,
			  u32 tid, u32 buf_size);
int ath12k_wmi_addba_set_resp(struct ath12k *ar, u32 vdev_id, const u8 *mac,
			      u32 tid, u32 status);
int ath12k_wmi_delba_send(struct ath12k *ar, u32 vdev_id, const u8 *mac,
			  u32 tid, u32 initiator, u32 reason);
int ath12k_wmi_send_bcn_offload_control_cmd(struct ath12k *ar,
					    u32 vdev_id, u32 bcn_ctrl_op);
int ath12k_wmi_send_init_country_cmd(struct ath12k *ar,
				     struct ath12k_wmi_init_country_arg *arg);
int ath12k_wmi_peer_rx_reorder_queue_setup(struct ath12k *ar,
					   int vdev_id, const u8 *addr,
					   dma_addr_t paddr, u8 tid,
					   u8 ba_window_size_valid,
					   u32 ba_window_size);
int
ath12k_wmi_rx_reord_queue_remove(struct ath12k *ar,
				 struct ath12k_wmi_rx_reorder_queue_remove_arg *arg);
int ath12k_wmi_send_pdev_set_regdomain(struct ath12k *ar,
				       struct ath12k_wmi_pdev_set_regdomain_arg *arg);
int ath12k_wmi_simulate_radar(struct ath12k *ar);
int ath12k_wmi_send_twt_enable_cmd(struct ath12k *ar, u32 pdev_id);
int ath12k_wmi_send_twt_disable_cmd(struct ath12k *ar, u32 pdev_id);
int ath12k_wmi_send_obss_spr_cmd(struct ath12k *ar, u32 vdev_id,
				 struct ieee80211_he_obss_pd *he_obss_pd);
int ath12k_wmi_obss_color_cfg_cmd(struct ath12k *ar, u32 vdev_id,
				  u8 bss_color, u32 period,
				  bool enable);
int ath12k_wmi_send_bss_color_change_enable_cmd(struct ath12k *ar, u32 vdev_id,
						bool enable);
int ath12k_wmi_pdev_lro_cfg(struct ath12k *ar, int pdev_id);
int ath12k_wmi_pdev_dma_ring_cfg(struct ath12k *ar,
				 struct ath12k_wmi_pdev_dma_ring_cfg_arg *arg);
int ath12k_wmi_vdev_spectral_enable(struct ath12k *ar, u32 vdev_id,
				    u32 trigger, u32 enable);
int ath12k_wmi_vdev_spectral_conf(struct ath12k *ar,
				  struct ath12k_wmi_vdev_spectral_conf_arg *arg);
int ath12k_wmi_fils_discovery_tmpl(struct ath12k *ar, u32 vdev_id,
				   struct sk_buff *tmpl);
int ath12k_wmi_fils_discovery(struct ath12k *ar, u32 vdev_id, u32 interval,
			      bool unsol_bcast_probe_resp_enabled);
int ath12k_wmi_probe_resp_tmpl(struct ath12k *ar, u32 vdev_id,
			       struct sk_buff *tmpl);
int ath12k_wmi_set_hw_mode(struct ath12k_base *ab,
			   enum wmi_host_hw_mode_config_type mode);

#endif
