/*
   ** Copyright 1996 Thorsten Kukuk <kukuk@uni-paderborn.de>
   **
   ** This program is free software; you can redistribute it and/or modify
   ** it under the terms of the GNU General Public License as published by
   ** the Free Software Foundation; either version 2 of the License, or
   ** (at your option) any later version.
   **
   ** This program is distributed in the hope that it will be useful,
   ** but WITHOUT ANY WARRANTY; without even the implied warranty of
   ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   ** GNU General Public License for more details.
   **
   ** You should have received a copy of the GNU General Public License
   ** along with this program; if not, write to the Free Software
   ** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <pwd.h>
#include <stdio.h>
#include <ctype.h>
#include <malloc.h>
#include <string.h>
#include <unistd.h>
#include <sys/signal.h>
#include <sys/stat.h>
#include <sys/resource.h>

#include "pwdutils.h"

/*
   ** pwd_init - allow unlimited resources and ignore signals.
   ** The changing of the passwd shouldn't fail on silly, limited
   ** resources
 */
void 
pwd_init ()
{
  struct rlimit rlim;

  /* 
   * We need unlimited resources and a core file could
   * be a security leak
   */
  rlim.rlim_cur = rlim.rlim_max = RLIM_INFINITY;
  setrlimit (RLIMIT_CPU, &rlim);
  setrlimit (RLIMIT_FSIZE, &rlim);
  setrlimit (RLIMIT_STACK, &rlim);
  setrlimit (RLIMIT_DATA, &rlim);
  setrlimit (RLIMIT_RSS, &rlim);
  rlim.rlim_cur = rlim.rlim_max = 0;
  setrlimit (RLIMIT_CORE, &rlim);

  /* 
   * Turn signals off. 
   */
  signal (SIGALRM, SIG_IGN);
  signal (SIGHUP, SIG_IGN);
  signal (SIGINT, SIG_IGN);
  signal (SIGPIPE, SIG_IGN);
  signal (SIGQUIT, SIG_IGN);
  signal (SIGTERM, SIG_IGN);
  signal (SIGTSTP, SIG_IGN);
  signal (SIGTTOU, SIG_IGN);

  umask (0);
}

/*
   ** pwd_error - print error message und remove tmp files befor 
   ** exiting the program.
 */
void 
pwd_error (const char *msg, const char *info)
{
  fprintf (stderr, "%s\n%s *NOT* changed. Try again later.\n", msg, info);
  unlink ("/etc/passwd.tmp");

  exit (1);
}

/*
   **  prompt - ask the user for a given field and return it,
   **  allows the user to use a default value
 */
char *
prompt (char *question, char *default_value)
{
  int len;
  char *ans;
  static char buf[FILENAME_MAX];

  if (!default_value)
    default_value = "";
  printf ("%s [%s]: ", question, default_value);
  *buf = 0;
  if (fgets (buf, sizeof (buf), stdin) == NULL)
    {
      printf ("\nAborted.\n");
      exit (1);
    }
  /* remove the newline at the end of buf. */
  ans = buf;
  while (isspace (*ans))
    ans++;
  len = strlen (ans);
  while (len > 0 && isspace (ans[len - 1]))
    len--;
  if (len <= 0)
    return strdup (default_value);
  ans[len] = 0;
  return strdup (buf);
}

/*
   ** set_changed_finger_data - incorporate the new data into the 
   ** old passwd entry.
 */
int 
set_changed_finger_data (struct passwd *pwd, struct npwd *newf)
{
  char *gecos;
  int len;
  char changed = 0;

  if (newf->full_name)
    changed = 1;
  else if (!newf->old_full_name)
    newf->full_name = "";
  else
    newf->full_name = newf->old_full_name;

  if (newf->office)
    changed = 1;
  else if (!newf->old_office)
    newf->office = "";
  else
    newf->office = newf->old_office;

  if (newf->office_phone)
    changed = 1;
  else if (!newf->old_office_phone)
    newf->office_phone = "";
  else
    newf->office_phone = newf->old_office_phone;

  if (newf->home_phone)
    changed = 1;
  else if (newf->old_home_phone)
    newf->home_phone = newf->old_home_phone;
  else
    newf->home_phone = "";

  if (!newf->other)
    newf->other = "";

  len = (strlen (newf->full_name) + strlen (newf->office) +
	 strlen (newf->office_phone) + strlen (newf->home_phone) +
	 strlen (newf->other) + 4);
  gecos = (char *) xmalloc (len + 1);
  sprintf (gecos, "%s,%s,%s,%s,%s", newf->full_name, newf->office,
	   newf->office_phone, newf->home_phone, newf->other);

  /* write the new struct passwd to the passwd file. */
  pwd->pw_gecos = gecos;

  return changed;
}

/*
   ** parse_passwd - take a struct password and fill in the fields 
   ** of the struct npwd.
 */
void 
parse_passwd (struct passwd *pwd, struct npwd *newf)
{
  char *cptr;

  if (pwd)
    {
      newf->old_username = pwd->pw_name;
      cptr = pwd->pw_gecos;
      newf->old_full_name = cptr;
      cptr = strchr (cptr, ',');
      if (cptr)
	{
	  *cptr = 0;
	  cptr++;
	}
      else
	return;
      newf->old_office = cptr;
      cptr = strchr (cptr, ',');
      if (cptr)
	{
	  *cptr = 0;
	  cptr++;
	}
      else
	return;
      newf->old_office_phone = cptr;
      cptr = strchr (cptr, ',');
      if (cptr)
	{
	  *cptr = 0;
	  cptr++;
	}
      else
	return;
      newf->old_home_phone = cptr;
      cptr = strchr (cptr, ',');
      if (cptr)
	{
	  *cptr = 0, cptr++;
	}
      else
	return;
      newf->other = cptr;
    }
}

/*
   **  xmalloc - malloc, that quit the program with an error message
   ** if it fails.
 */
void *
xmalloc (int bytes)
{
  void *vptr;

  vptr = malloc (bytes);
  if (!vptr && bytes > 0)
    {
      perror ("malloc failed");
      exit (1);
    }
  return vptr;
}
