/* Copyright (C) 2002, 2003 Thorsten Kukuk
   Copyright (C) 1998, 1999, 2000 Luke Howard.
   Most of this code is from the pam_ldap-148 package written by Luke Howard.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License version 2 as
   published by the Free Software Foundation.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef USE_LDAP

#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/param.h>
#include <unistd.h>
#include <syslog.h>
#include <netdb.h>
#include <errno.h>

#ifdef HAVE_LBER_H
#include <lber.h>
#endif
#ifdef HAVE_LDAP_H
#include <ldap.h>
#endif
#ifdef HAVE_LDAP_SSL_H
#include <ldap_ssl.h>
#endif

#define _INCLUDED_FROM_LDAPCFN_C_
#include "ldapfcn.h"

#ifndef HAVE_LDAP_MEMFREE
#define ldap_memfree(x)	free(x)
#endif

#ifdef __GNUC__
#define __UNUSED__ __attribute__ ((unused))
#else
#define __UNUSED__
#endif

#if LDAP_SET_REBIND_PROC_ARGS < 3
static ldap_session_t *global_session = 0;
#endif

/* TLS routines */
#if defined HAVE_LDAP_START_TLS_S || (defined(HAVE_LDAP_SET_OPTION) && defined(LDAP_OPT_X_TLS))
static int _set_ssl_default_options (ldap_session_t *);
static int _set_ssl_options (ldap_session_t *);
#endif

static int _connect_anonymously (ldap_session_t *session);
#if defined(LDAP_API_FEATURE_X_OPENLDAP) && (LDAP_API_VERSION > 2000)
#if LDAP_SET_REBIND_PROC_ARGS == 3
static int _rebind_proc (LDAP * ld, LDAP_CONST char *url, ber_tag_t request,
			 ber_int_t msgid, void *arg);
#else
static int _rebind_proc (LDAP * ld, LDAP_CONST char *url, int request,
			 ber_int_t msgid);
#endif
#else
#if LDAP_SET_REBIND_PROC_ARGS == 3
static int _rebind_proc (LDAP * ld,
			 char **whop, char **credp, int *methodp, int freeit,
			 void *arg);
#else
static int _rebind_proc (LDAP * ld, char **whop, char **credp, int *methodp,
			 int freeit);
#endif
#endif


static int _connect_as_user (ldap_session_t *session,
			     const char *password);
static int _get_integer_value (LDAP * ld, LDAPMessage * e, const char *attr,
			       int *ptr);
static int _get_long_integer_value (LDAP * ld, LDAPMessage * e,
				    const char *attr, long int *ptr);
static int _get_string_value (LDAP * ld, LDAPMessage * e, const char *attr,
			      char **ptr);
static int _get_string_values (LDAP * ld, LDAPMessage * e, const char *attr,
			       char ***ptr);
static int _escape_string (const char *str, char *buf, size_t buflen);
static int _reopen (ldap_session_t *session);

#ifdef HAVE_LIBPTHREAD
#include <dlfcn.h>

/*
 * on Linux at least, the pthread library registers an atexit
 * handler in it's constructor.  Since we are in a library and linking with
 * libpthread, if the client program is not linked with libpthread, it
 * segfaults on exit. So we open an extra reference to the library.
 *
 * If there is a better way of doing this, let us know.
 */
#ifdef __GNUC__
void nasty_pthread_hack (void) __attribute__ ((constructor));
#else
# ifdef __SUNPRO_C
#  pragma init(nasty_pthread_hack)
# endif				/* __SUNPRO_C */
#endif /* __GNUC__ */

void
nasty_pthread_hack (void)
{
  (void) dlopen ("libpthread.so", RTLD_LAZY);
}
#endif /* HAVE_LIBPTHREAD */

#ifndef HAVE_LDAP_GET_LDERRNO
static int
ldap_get_lderrno (LDAP *ld, char **m, char **s)
{
#ifdef HAVE_LDAP_GET_OPTION
  int rc;
#endif
  int lderrno;

#if defined(HAVE_LDAP_GET_OPTION) && defined(LDAP_OPT_ERROR_NUMBER)
  /* is this needed? */
  rc = ldap_get_option (ld, LDAP_OPT_ERROR_NUMBER, &lderrno);
  if (rc != LDAP_SUCCESS)
    return rc;
#else
  lderrno = ld->ld_errno;
#endif

  if (s != NULL)
    {
#if defined(HAVE_LDAP_GET_OPTION) && defined(LDAP_OPT_ERROR_STRING)
      rc = ldap_get_option (ld, LDAP_OPT_ERROR_STRING, s);
      if (rc != LDAP_SUCCESS)
	return rc;
#else
      *s = ld->ld_error;
#endif
    }

  if (s != NULL)
    {
#if defined(HAVE_LDAP_GET_OPTION) && defined(LDAP_OPT_MATCHED_DN)
      rc = ldap_get_option (ld, LDAP_OPT_MATCHED_DN, m);
      if (rc != LDAP_SUCCESS)
	return rc;
#else
      *m = ld->ld_matched;
#endif
    }

  return lderrno;
}
#endif

void
free_ldap_config (ldap_config_t **pconfig)
{
  ldap_config_t *c;

  c = *pconfig;
  if (c == NULL)
    return;

  if (c->host != NULL)
    free (c->host);

  if (c->base != NULL)
    free (c->base);

  if (c->binddn != NULL)
    free (c->binddn);

  if (c->bindpw != NULL)
    {
      _pam_overwrite (c->bindpw);
      _pam_drop (c->bindpw);
    }

  if (c->rootbinddn != NULL)
    free (c->rootbinddn);

  if (c->rootbindpw != NULL)
    {
      _pam_overwrite (c->rootbindpw);
      _pam_drop (c->rootbindpw);
    }

  if (c->sslpath != NULL)
    {
      free (c->sslpath);
    }

  if (c->userattr != NULL)
    {
      free (c->userattr);
    }

  if (c->tmplattr != NULL)
    {
      free (c->tmplattr);
    }

  if (c->tmpluser != NULL)
    {
      free (c->tmpluser);
    }

  if (c->groupattr != NULL)
    {
      free (c->groupattr);
    }

  if (c->groupdn != NULL)
    {
      free (c->groupdn);
    }

  if (c->filter != NULL)
    {
      free (c->filter);
    }

  memset (c, 0, sizeof (*c));
  free (c);
  *pconfig = NULL;

  return;
}

static void
_release_user_info (ldap_user_info_t ** info)
{
  if (*info == NULL)
    return;

  if ((*info)->userdn != NULL)
    {
      ldap_memfree ((void *) (*info)->userdn);
    }

  /*
   * Clobber the password.
   */
  _pam_overwrite ((*info)->userpw);
  _pam_drop ((*info)->userpw);

  if ((*info)->hosts_allow != NULL)
    {
      ldap_value_free ((*info)->hosts_allow);
    }

  if ((*info)->tmpluser != NULL)
    {
      free ((void *) (*info)->tmpluser);
    }

  free ((void *) (*info)->username);
  free (*info);

  *info = NULL;
  return;
}

static ldap_config_t *
alloc_ldap_config (void)
{
  ldap_config_t *result = (ldap_config_t *) calloc (1, sizeof (*result));

  if (result == NULL)
    {
      errno = ENOMEM;
      return NULL;
    }

  result->scope = LDAP_SCOPE_SUBTREE;
  result->deref = LDAP_DEREF_NEVER;
  result->host = NULL;
  result->base = NULL;
  result->port = 0;
  result->binddn = NULL;
  result->bindpw = NULL;
  result->rootbinddn = NULL;
  result->rootbindpw = NULL;
  result->ssl_on = SSL_OFF;
  result->sslpath = NULL;
  result->filter = NULL;
  result->userattr = NULL;
  result->groupattr = NULL;
  result->groupdn = NULL;
  result->getpolicy = 0;
  result->checkhostattr = 0;
#ifdef LDAP_VERSION3
  result->version = LDAP_VERSION3;
#else
  result->version = LDAP_VERSION2;
#endif /* LDAP_VERSION2 */
  result->timelimit = LDAP_NO_LIMIT;
  result->bind_timelimit = 10;
  result->referrals = 1;
  result->restart = 1;
  result->password_type = PASSWORD_CLEAR;
  result->min_uid = 0;
  result->max_uid = 0;
  result->tmplattr = NULL;
  result->tmpluser = NULL;
  result->tls_checkpeer = 0;
  result->tls_cacertfile = NULL;
  result->tls_cacertdir = NULL;
  result->tls_ciphers = NULL;
  result->tls_cert = NULL;
  result->tls_key = NULL;
  return result;
}


#define CHECKPOINTER(ptr) do { if ((ptr) == NULL) { \
    fclose(fp); \
    free_ldap_config (&result); \
    return NULL; \
} \
} while (0)

ldap_config_t *
read_ldap_config (const char *configFile)
{
  /* this is the same configuration file as nss_ldap */
  FILE *fp;
  char b[BUFSIZ];
  char *defaultBase, *passwdBase, *defaultFilter, *passwdFilter;
  int defaultScope, passwdScope;
  ldap_config_t *result;

  if ((result = alloc_ldap_config ()) == NULL)
    return NULL;

  /* configuration file location is configurable; default /etc/ldap.conf */
  if (configFile == NULL)
    configFile = LDAP_PATH_CONF;

  fp = fopen (configFile, "r");

  if (fp == NULL)
    {
      if (isatty (fileno(stderr)))
	fprintf (stderr, "missing file \"%s\"\n", configFile);
      else
	syslog (LOG_ERR, "missing file \"%s\"", configFile);
      return NULL;
    }

  defaultBase = NULL;
  defaultFilter = NULL;
  defaultScope = LDAP_SCOPE_SUBTREE;

  passwdBase = NULL;
  passwdFilter = NULL;
  passwdScope = -1;

  while (fgets (b, sizeof (b), fp) != NULL)
    {
      char *k, *v;
      int len;

      if (*b == '\n' || *b == '#')
	continue;

      k = b;
      v = k;
      while (*v != '\0' && *v != ' ' && *v != '\t')
	v++;

      if (*v == '\0')
	continue;

      *(v++) = '\0';

      /* skip all whitespaces between keyword and value */
      /* Lars Oergel <lars.oergel@innominate.de>, 05.10.2000 */
      while (*v == ' ' || *v == '\t')
	v++;

      /* kick off all whitespaces and newline at the end of value */
      /* Bob Guo <bob@mail.ied.ac.cn>, 08.10.2001 */
      len = strlen (v) - 1;
      while (v[len] == ' ' || v[len] == '\t' || v[len] == '\n')
	--len;
      v[len + 1] = '\0';

      if (!strcasecmp (k, "host"))
	{
	  CHECKPOINTER (result->host = strdup (v));
	}
      else if (!strcasecmp (k, "uri"))
	{
	  CHECKPOINTER (result->uri = strdup (v));
	}
      else if (!strcasecmp (k, "base"))
	{
	  CHECKPOINTER (defaultBase = strdup (v));
	}
      else if (!strcasecmp (k, "binddn"))
	{
	  CHECKPOINTER (result->binddn = strdup (v));
	}
      else if (!strcasecmp (k, "bindpw"))
	{
	  CHECKPOINTER (result->bindpw = strdup (v));
	}
      else if (!strcasecmp (k, "rootbinddn"))
	{
	  CHECKPOINTER (result->rootbinddn = strdup (v));
	}
      else if (!strcasecmp (k, "scope"))
	{
	  if (!strncasecmp (v, "sub", 3))
	    result->scope = LDAP_SCOPE_SUBTREE;
	  else if (!strncasecmp (v, "one", 3))
	    result->scope = LDAP_SCOPE_ONELEVEL;
	  else if (!strncasecmp (v, "base", 4))
	    result->scope = LDAP_SCOPE_BASE;
	}
      else if (!strcasecmp (k, "deref"))
	{
	  if (!strcasecmp (v, "never"))
	    result->deref = LDAP_DEREF_NEVER;
	  else if (!strcasecmp (v, "searching"))
	    result->deref = LDAP_DEREF_SEARCHING;
	  else if (!strcasecmp (v, "finding"))
	    result->deref = LDAP_DEREF_FINDING;
	  else if (!strcasecmp (v, "always"))
	    result->deref = LDAP_DEREF_ALWAYS;
	}
      else if (!strcasecmp (k, "pam_password"))
	{
	  if (!strcasecmp (v, "clear"))
	    result->password_type = PASSWORD_CLEAR;
	  else if (!strcasecmp (v, "crypt"))
	    result->password_type = PASSWORD_CRYPT;
	  else if (!strcasecmp (v, "md5"))
	    result->password_type = PASSWORD_MD5;
	  else if (!strcasecmp (v, "nds"))
	    result->password_type = PASSWORD_NDS;
	  else if (!strcasecmp (v, "ad"))
	    result->password_type = PASSWORD_AD;
	  else if (!strcasecmp (v, "exop"))
	    result->password_type = PASSWORD_EXOP;
	}
      else if (!strcasecmp (k, "pam_crypt"))
	{
	  /*
	   * we still support this even though it is
	   * deprecated, as it could be a security
	   * hole to change this behaviour on
	   * unsuspecting users of pam_ldap.
	   */
	  if (!strcasecmp (v, "local"))
	    result->password_type = PASSWORD_CRYPT;
	  else
	    result->password_type = PASSWORD_CLEAR;
	}
      else if (!strcasecmp (k, "port"))
	{
	  result->port = atoi (v);
	}
      else if (!strcasecmp (k, "timelimit"))
	{
	  result->timelimit = atoi (v);
	}
      else if (!strcasecmp (k, "bind_timelimit"))
	{
	  result->bind_timelimit = atoi (v);
	}
      else if (!strcasecmp (k, "nss_base_passwd"))
	{
	  char *s;

	  /* this doesn't do any escaping. XXX. */
	  CHECKPOINTER (passwdBase = strdup (v));
	  s = strchr (passwdBase, '?');
	  if (s != NULL)
	    {
	      *s = '\0';
	      s++;
	      if (!strcasecmp (s, "sub"))
		passwdScope = LDAP_SCOPE_SUBTREE;
	      else if (!strcasecmp (s, "one"))
		passwdScope = LDAP_SCOPE_ONELEVEL;
	      else if (!strcasecmp (s, "base"))
		passwdScope = LDAP_SCOPE_BASE;
	      s = strchr (s, '?');
	      if (s != NULL)
		{
		  *s = '\0';
		  s++;
		  CHECKPOINTER (passwdFilter = strdup (s));
		}
	    }
	}
      else if (!strcasecmp (k, "ldap_version"))
	{
	  result->version = atoi (v);
	}
      else if (!strcasecmp (k, "sslpath"))
	{
	  CHECKPOINTER (result->sslpath = strdup (v));
	}
      else if (!strcasecmp (k, "ssl"))
	{
	  if (!strcasecmp (v, "on") || !strcasecmp (v, "yes")
	      || !strcasecmp (v, "true"))
	    {
	      result->ssl_on = SSL_LDAPS;
	    }
	  else if (!strcasecmp (v, "start_tls"))
	    {
	      result->ssl_on = SSL_START_TLS;
	    }
	}
      else if (!strcasecmp (k, "referrals"))
	{
	  result->referrals = (!strcasecmp (v, "on") || !strcasecmp (v, "yes")
			       || !strcasecmp (v, "true"));
	}
      else if (!strcasecmp (k, "restart"))
	{
	  result->restart = (!strcasecmp (v, "on") || !strcasecmp (v, "yes")
			     || !strcasecmp (v, "true"));
	}
      else if (!strcasecmp (k, "pam_filter"))
	{
	  CHECKPOINTER (defaultFilter = strdup (v));
	}
      else if (!strcasecmp (k, "pam_login_attribute"))
	{
	  CHECKPOINTER (result->userattr = strdup (v));
	}
      else if (!strcasecmp (k, "pam_template_login_attribute"))
	{
	  CHECKPOINTER (result->tmplattr = strdup (v));
	}
      else if (!strcasecmp (k, "pam_template_login"))
	{
	  CHECKPOINTER (result->tmpluser = strdup (v));
	}
      else if (!strcasecmp (k, "pam_lookup_policy"))
	{
	  result->getpolicy = !strcasecmp (v, "yes");
	}
      else if (!strcasecmp (k, "pam_check_host_attr"))
	{
	  result->checkhostattr = !strcasecmp (v, "yes");
	}
      else if (!strcasecmp (k, "pam_groupdn"))
	{
	  CHECKPOINTER (result->groupdn = strdup (v));
	}
      else if (!strcasecmp (k, "pam_member_attribute"))
	{
	  CHECKPOINTER (result->groupattr = strdup (v));
	}
      else if (!strcasecmp (k, "pam_min_uid"))
	{
	  result->min_uid = (uid_t) atol (v);
	}
      else if (!strcasecmp (k, "pam_max_uid"))
	{
	  result->max_uid = (uid_t) atol (v);
	}
      else if (!strcasecmp (k, "tls_checkpeer"))
	{
	  if (!strcasecmp (v, "on") || !strcasecmp (v, "yes")
	      || !strcasecmp (v, "true"))
	    {
	      result->tls_checkpeer = 1;
	    }
	  else if (!strcasecmp (v, "off") || !strcasecmp (v, "no")
		   || !strcasecmp (v, "false"))
	    {
	      result->tls_checkpeer = 0;
	    }
	}
      else if (!strcasecmp (k, "tls_cacertfile"))
	{
	  CHECKPOINTER (result->tls_cacertfile = strdup (v));
	}
      else if (!strcasecmp (k, "tls_cacertdir"))
	{
	  CHECKPOINTER (result->tls_cacertdir = strdup (v));
	}
      else if (!strcasecmp (k, "tls_ciphers"))
	{
	  CHECKPOINTER (result->tls_ciphers = strdup (v));
	}
      else if (!strcasecmp (k, "tls_cert"))
	{
	  CHECKPOINTER (result->tls_cert = strdup (v));
	}
      else if (!strcasecmp (k, "tls_key"))
	{
	  CHECKPOINTER (result->tls_key = strdup (v));
	}
    }

  if (passwdBase != NULL)
    {
      if (defaultBase != NULL)
	{
	  size_t len = strlen (passwdBase);

	  if (passwdBase[len - 1] == ',')
	    {
	      char *p;

	      p = (char *) malloc (len + strlen (defaultBase) + 1);
	      if (p == NULL)
		{
		  fclose (fp);
		  free (defaultBase);	/* leak the rest... */
		  free_ldap_config (&result);
		  return NULL;
		}

	      strcpy (p, passwdBase);
	      strcpy (&p[len], defaultBase);
	      free (passwdBase);
	      passwdBase = p;
	    }
	  free (defaultBase);
	}
      result->base = passwdBase;
    }
  else
    {
      result->base = defaultBase;
    }

  if (passwdFilter != NULL)
    {
      result->filter = passwdFilter;
      if (defaultFilter != NULL)
	free (defaultFilter);
    }
  else
    {
      result->filter = defaultFilter;
    }

  if (passwdScope != -1)
    {
      result->scope = passwdScope;
    }
  else
    {
      result->scope = defaultScope;
    }

  if (result->host == NULL
#ifdef HAVE_LDAP_INITIALIZE
      && result->uri == NULL
#endif
      )
    {
      if (isatty (fileno(stderr)))
	fprintf (stderr, "missing \"host\" in file \"ldap.conf\"\n");
      else
	syslog (LOG_ERR, "missing \"host\" in file \"ldap.conf\"");
      return NULL;
    }

  if (result->userattr == NULL)
    {
      CHECKPOINTER (result->userattr = strdup ("uid"));
    }

  if (result->groupattr == NULL)
    {
      CHECKPOINTER (result->groupattr = strdup ("uniquemember"));
    }

  if (result->port == 0)
    {
#if defined(HAVE_LDAP_START_TLS_S)
      if (result->ssl_on == SSL_LDAPS)
	{
	  result->port = LDAPS_PORT;
	}
      else
#endif
	result->port = LDAP_PORT;
    }

  fclose (fp);

  if ((result->rootbinddn != NULL) && (geteuid () == 0))
    {
      fp = fopen (LDAP_PATH_ROOTPASSWD, "r");
      if (fp != NULL)
	{
	  if (fgets (b, sizeof (b), fp) != NULL)
	    {
	      int len;
	      len = strlen (b);
	      if (len > 0 && b[len - 1] == '\n')
		len--;

	      b[len] = '\0';
	      result->rootbindpw = strdup (b);
	    }
	  fclose (fp);
	}
      else
	{
	  int save_err = errno;

	  _pam_drop (result->rootbinddn);
	  if (isatty (fileno(stderr)))
	    fprintf (stderr,
		     "could not open secret file %s (%s)",
		     LDAP_PATH_ROOTPASSWD, strerror (save_err));
	  else
	    syslog (LOG_WARNING,
		    "could not open secret file %s (%s)",
		    LDAP_PATH_ROOTPASSWD, strerror (save_err));
	}
    }

  /* can't use _pam_overwrite because it only goes to end of string,
   * not the buffer
   */
  memset (b, 0, BUFSIZ);
  return result;
}

int
open_ldap_session (ldap_session_t *session)
{
#ifdef LDAP_X_OPT_CONNECT_TIMEOUT
  int timeout;
#endif
#ifdef LDAP_OPT_NETWORK_TIMEOUT
  struct timeval tv;
#endif

#if defined(HAVE_LDAP_SET_OPTION) && defined(LDAP_OPT_X_TLS)
  /* set defaults for global TLS-related options */
  _set_ssl_default_options (session);
#endif
#ifdef HAVE_LDAP_INITIALIZE
  if (session->conf->uri != NULL)
    {
      int rc = ldap_initialize (&session->ld, session->conf->uri);
      if (rc != LDAP_SUCCESS)
	{
	  if (isatty (fileno (stderr)))
	    fprintf (stderr, "ldap_initialize %s",
		     ldap_err2string (rc));
	  else
	    syslog (LOG_ERR, "ldap_initialize %s",
		    ldap_err2string (rc));
	  return rc;
	}
    }
  else
    {
#endif /* HAVE_LDAP_INTITIALIZE */
#ifdef HAVE_LDAP_INIT
      session->ld = ldap_init (session->conf->host, session->conf->port);
#else
      session->ld = ldap_open (session->conf->host, session->conf->port);
#endif /* HAVE_LDAP_INIT */
    }

  if (session->ld == NULL)
    return 1;

#if defined(HAVE_LDAP_SET_OPTION) && defined(LDAP_OPT_X_TLS)
  if (session->conf->ssl_on == SSL_LDAPS)
    {
      int tls = LDAP_OPT_X_TLS_HARD;
      int rc = ldap_set_option (session->ld, LDAP_OPT_X_TLS, &tls);
      if (rc != LDAP_SUCCESS)
	{
	  if (isatty (fileno (stderr)))
	    fprintf (stderr, "ldap_set_option(LDAP_OPT_X_TLS) %s",
		     ldap_err2string (rc));
	  else
	    syslog (LOG_ERR, "ldap_set_option(LDAP_OPT_X_TLS) %s",
		    ldap_err2string (rc));
	  return rc;
	}

      /* set up SSL per-context settings */
      _set_ssl_options (session);
    }
#endif /* LDAP_OPT_X_TLS */

#if defined(HAVE_LDAP_SET_OPTION) && defined(LDAP_OPT_PROTOCOL_VERSION)
  ldap_set_option (session->ld, LDAP_OPT_PROTOCOL_VERSION,
		   &session->conf->version);
#else
  session->ld->ld_version = session->conf->version;
#endif

#if LDAP_SET_REBIND_PROC_ARGS == 3
  ldap_set_rebind_proc (session->ld, _rebind_proc, (void *) session);
#elif LDAP_SET_REBIND_PROC_ARGS == 2
  ldap_set_rebind_proc (session->ld, _rebind_proc);
#endif

#if defined(HAVE_LDAP_SET_OPTION) && defined(LDAP_OPT_DEREF)
  ldap_set_option (session->ld, LDAP_OPT_DEREF, &session->conf->deref);
#else
  session->ld->ld_deref = session->conf->deref;
#endif

#if defined(HAVE_LDAP_SET_OPTION) && defined(LDAP_OPT_TIMELIMIT)
  ldap_set_option (session->ld, LDAP_OPT_TIMELIMIT, &session->conf->timelimit);
#else
  session->ld->ld_timelimit = session->conf->timelimit;
#endif


#if defined(HAVE_LDAP_SET_OPTION) && defined(LDAP_X_OPT_CONNECT_TIMEOUT)
  /*
   * This is a new option in the Netscape SDK which sets
   * the TCP connect timeout. For want of a better value,
   * we use the bind_timelimit to control this.
   */
  timeout = session->conf->bind_timelimit * 1000;
  ldap_set_option (session->ld, LDAP_X_OPT_CONNECT_TIMEOUT, &timeout);
#endif

#if defined(HAVE_LDAP_SET_OPTION) && defined(LDAP_OPT_NETWORK_TIMEOUT)
  tv.tv_sec = session->conf->bind_timelimit;
  tv.tv_usec = 0;
  ldap_set_option (session->ld, LDAP_OPT_NETWORK_TIMEOUT, &tv);
#endif

#if defined(HAVE_LDAP_SET_OPTION) && defined(LDAP_OPT_REFERRALS)
  ldap_set_option (session->ld, LDAP_OPT_REFERRALS,
		   session->
		   conf->referrals ? LDAP_OPT_ON : LDAP_OPT_OFF);
#endif

#if defined(HAVE_LDAP_SET_OPTION) && defined(LDAP_OPT_RESTART)
  ldap_set_option (session->ld, LDAP_OPT_RESTART,
		   session->
		   conf->restart ? LDAP_OPT_ON : LDAP_OPT_OFF);
#endif

#ifdef HAVE_LDAP_START_TLS_S
  if (session->conf->ssl_on == SSL_START_TLS)
    {
      int version, rc;

      if (ldap_get_option (session->ld, LDAP_OPT_PROTOCOL_VERSION, &version)
	  == LDAP_SUCCESS)
	{
	  if (version < LDAP_VERSION3)
	    {
	      version = LDAP_VERSION3;
	      ldap_set_option (session->ld, LDAP_OPT_PROTOCOL_VERSION,
			       &version);
	    }

	  /* set up SSL context */
	  _set_ssl_options (session);

	  rc = ldap_start_tls_s (session->ld, NULL, NULL);
	  if (rc != LDAP_SUCCESS)
	    {
	      if (isatty (fileno (stderr)))
		fprintf (stderr, "ldap_starttls_s: %s",
			 ldap_err2string (rc));
	      else
		syslog (LOG_ERR, "ldap_starttls_s: %s",
			ldap_err2string (rc));
	      return rc;
	    }
	}
    }
#endif /* HAVE_LDAP_START_TLS_S */
  return 0;
}

#if defined HAVE_LDAP_START_TLS_S || (defined(HAVE_LDAP_SET_OPTION) && defined(LDAP_OPT_X_TLS))
/* Some global TLS-specific options need to be set before we create our
 * session context, so we set them here. */
static int
_set_ssl_default_options (ldap_session_t *session)
{
  int rc;

  /* ca cert file */
  if (session->conf->tls_cacertfile != NULL)
    {
      rc = ldap_set_option (NULL, LDAP_OPT_X_TLS_CACERTFILE,
			    session->conf->tls_cacertfile);
      if (rc != LDAP_SUCCESS)
	{
	  if (isatty (fileno (stderr)))
	    fprintf (stderr, "ldap_set_option(LDAP_OPT_X_TLS_CACERTFILE): %s",
		     ldap_err2string (rc));
	  else
	    syslog (LOG_ERR,
		    "ldap_set_option(LDAP_OPT_X_TLS_CACERTFILE): %s",
		    ldap_err2string (rc));
	  return LDAP_OPERATIONS_ERROR;
	}
    }

  if (session->conf->tls_cacertdir != NULL)
    {
      /* ca cert directory */
      rc = ldap_set_option (NULL, LDAP_OPT_X_TLS_CACERTDIR,
			    session->conf->tls_cacertdir);
      if (rc != LDAP_SUCCESS)
	{
	  if (isatty (fileno (stderr)))
	    fprintf (stderr, "ldap_set_option(LDAP_OPT_X_TLS_CACERTDIR): %s",
		     ldap_err2string (rc));
	  else
	    syslog (LOG_ERR,
		    "ldap_set_option(LDAP_OPT_X_TLS_CACERTDIR): %s",
		    ldap_err2string (rc));
	  return LDAP_OPERATIONS_ERROR;
	}
    }

  /* require cert? */
  rc = ldap_set_option (NULL, LDAP_OPT_X_TLS_REQUIRE_CERT,
			&session->conf->tls_checkpeer);
  if (rc != LDAP_SUCCESS)
    {
      if (isatty (fileno (stderr)))
	fprintf (stderr, "ldap_set_option(LDAP_OPT_X_TLS_REQUIRE_CERT): %s",
		 ldap_err2string (rc));
      else
	syslog (LOG_ERR,
		"ldap_set_option(LDAP_OPT_X_TLS_REQUIRE_CERT): %s",
		ldap_err2string (rc));
      return LDAP_OPERATIONS_ERROR;
    }

  if (session->conf->tls_ciphers != NULL)
    {
      /* set cipher suite, certificate and private key: */
      rc = ldap_set_option (NULL, LDAP_OPT_X_TLS_CIPHER_SUITE,
			    session->conf->tls_ciphers);
      if (rc != LDAP_SUCCESS)
	{
	  if (isatty (fileno (stderr)))
	    fprintf (stderr,
		     "ldap_set_option(LDAP_OPT_X_TLS_CIPHER_SUITE): %s",
		     ldap_err2string (rc));
	  else
	    syslog (LOG_ERR,
		    "ldap_set_option(LDAP_OPT_X_TLS_CIPHER_SUITE): %s",
		    ldap_err2string (rc));
	  return LDAP_OPERATIONS_ERROR;
	}
    }

  if (session->conf->tls_cert != NULL)
    {
      rc = ldap_set_option (NULL, LDAP_OPT_X_TLS_CERTFILE,
			    session->conf->tls_cert);
      if (rc != LDAP_SUCCESS)
	{
	  if (isatty (fileno (stderr)))
	    fprintf (stderr, "ldap_set_option(LDAP_OPT_X_TLS_CERTFILE): %s",
		     ldap_err2string (rc));
	  else
	    syslog (LOG_ERR,
		    "ldap_set_option(LDAP_OPT_X_TLS_CERTFILE): %s",
		    ldap_err2string (rc));
	  return LDAP_OPERATIONS_ERROR;
	}
    }

  if (session->conf->tls_key != NULL)
    {
      rc = ldap_set_option (NULL, LDAP_OPT_X_TLS_KEYFILE,
			    session->conf->tls_key);
      if (rc != LDAP_SUCCESS)
	{
	  if (isatty (fileno (stderr)))
	    fprintf (stderr,
		     "ldap_set_option(LDAP_OPT_X_TLS_KEYFILE): %s",
		     ldap_err2string (rc));
	  else
	    syslog (LOG_ERR,
		    "ldap_set_option(LDAP_OPT_X_TLS_KEYFILE): %s",
		    ldap_err2string (rc));
	  return LDAP_OPERATIONS_ERROR;
	}
    }

  return LDAP_SUCCESS;
}

/* Now we can set the per-context TLS-specific options. */
static int
_set_ssl_options (ldap_session_t *session __UNUSED__)
{
  return LDAP_SUCCESS;
}
#endif

static int
_connect_anonymously (ldap_session_t *session)
{
  int rc;
  int msgid;
  struct timeval timeout;
  LDAPMessage *result;

  if (session->ld == NULL)
    {
      rc = open_ldap_session (session);
      if (rc != 0)
	return rc;
    }

  if (session->conf->rootbinddn && geteuid () == 0)
    {
      msgid = ldap_simple_bind (session->ld,
				session->conf->rootbinddn,
				session->conf->rootbindpw);
    }
  else
    {
      msgid = ldap_simple_bind (session->ld,
				session->conf->binddn, session->conf->bindpw);
    }

  if (msgid == -1)
    {
      if (isatty (fileno (stderr)))
	fprintf (stderr, "ldap_simple_bind %s",
		 ldap_err2string (ldap_get_lderrno (session->ld, 0, 0)));
      else
	syslog (LOG_ERR, "ldap_simple_bind %s",
		ldap_err2string (ldap_get_lderrno (session->ld, 0, 0)));
      return ldap_get_lderrno (session->ld, 0, 0);
    }

  timeout.tv_sec = session->conf->bind_timelimit;	/* default 10 */
  timeout.tv_usec = 0;
  rc = ldap_result (session->ld, msgid, FALSE, &timeout, &result);
  if (rc == -1 || rc == 0)
    {
      if (isatty (fileno (stderr)))
	fprintf (stderr, "ldap_result %s\n",
		 ldap_err2string (ldap_get_lderrno (session->ld, 0, 0)));
      else
	syslog (LOG_ERR, "ldap_result %s",
		ldap_err2string (ldap_get_lderrno (session->ld, 0, 0)));
      return ldap_get_lderrno (session->ld, 0, 0);
    }

#ifdef HAVE_LDAP_PARSE_RESULT
  ldap_parse_result (session->ld, result, &rc, 0, 0, 0, 0, TRUE);
#else
  rc = ldap_result2error (session->ld, result, TRUE);
#endif

  if (rc != LDAP_SUCCESS)
    {
      if (isatty (fileno (stderr)))
	fprintf (stderr, "error trying to bind (%s)\n",
		 ldap_err2string (rc));
      else
	syslog (LOG_ERR, "error trying to bind (%s)",
		ldap_err2string (rc));
      return rc;
    }

  if (session->info != NULL)
    {
      session->info->bound_as_user = 0;
    }

  return LDAP_SUCCESS;
}

#if defined(LDAP_API_FEATURE_X_OPENLDAP) && (LDAP_API_VERSION > 2000)
#if LDAP_SET_REBIND_PROC_ARGS == 3
static int
_rebind_proc (LDAP * ld, LDAP_CONST char *url __attribute__ ((unused)),
	      ber_tag_t request __attribute__ ((unused)),
	      ber_int_t msgid __attribute__ ((unused)), void *arg)
#else
static int
_rebind_proc (LDAP * ld, LDAP_CONST char *url, int request, ber_int_t msgid)
#endif
{
#if LDAP_SET_REBIND_PROC_ARGS == 3
  ldap_session_t *session = (ldap_session_t *) arg;
#else
  /* ugly hack */
  ldap_session_t *session = global_session;
#endif
  char *who, *cred;

  if (session->info != NULL && session->info->bound_as_user == 1)
    {
      who = session->info->userdn;
      cred = session->info->userpw;
    }
  else
    {
      if (session->conf->rootbinddn != NULL && geteuid () == 0)
	{
	  who = session->conf->rootbinddn;
	  cred = session->conf->rootbindpw;
	}
      else
	{
	  who = session->conf->binddn;
	  cred = session->conf->bindpw;
	}
    }

  return ldap_simple_bind_s (ld, who, cred);
}
#else
#if LDAP_SET_REBIND_PROC_ARGS == 3
static int
_rebind_proc (LDAP * ld,
	      char **whop, char **credp, int *methodp, int freeit, void *arg)
#else
static int
_rebind_proc (LDAP * ld, char **whop, char **credp, int *methodp, int freeit)
#endif
{
#if LDAP_SET_REBIND_PROC_ARGS == 3
  ldap_session_t *session = (ldap_session_t *) arg;
#else
  /* ugly hack */
  ldap_session_t *session = global_session;
#endif

  if (freeit)
    {
      _pam_drop (*whop);
      _pam_overwrite (*credp);
      _pam_drop (*credp);
      return LDAP_SUCCESS;
    }

  if (session->info != NULL && session->info->bound_as_user == 1)
    {
      /*
       * We're authenticating as a user.
       */
      *whop = strdup (session->info->userdn);
      *credp = strdup (session->info->userpw);
    }
  else
    {
      if (session->conf->rootbinddn != NULL && geteuid () == 0)
	{
	  *whop = strdup (session->conf->rootbinddn);
	  *credp = session->conf->rootbindpw != NULL ?
	    strdup (session->conf->rootbindpw) : NULL;
	}
      else
	{
	  *whop = session->conf->binddn != NULL ?
	    strdup (session->conf->binddn) : NULL;
	  *credp = session->conf->bindpw != NULL ?
	    strdup (session->conf->bindpw) : NULL;
	}
    }

  *methodp = LDAP_AUTH_SIMPLE;

  return LDAP_SUCCESS;
}
#endif


static int
_connect_as_user (ldap_session_t *session, const char *password)
{
  int rc, msgid;
#if defined(HAVE_LDAP_PARSE_RESULT) && defined(HAVE_LDAP_CONTROLS_FREE)
  int parserc;
#endif
  struct timeval timeout;
  LDAPMessage *result;
#ifdef HAVE_LDAP_CONTROLS_FREE
  LDAPControl **controls;
#endif

  /* avoid binding anonymously with a DN but no password */
  if (password == NULL || password[0] == '\0')
    return 1;

  /* this shouldn't ever happen */
  if (session->info == NULL)
    return 1;

  /* if we already bound as the user don't bother retrying */
  if (session->info->bound_as_user)
    return 1;

  if (session->ld == NULL)
    {
      rc = open_ldap_session (session);
      if (rc != LDAP_SUCCESS)
	return rc;
    }

  /*
   * We copy the password temporarily so that when referrals are
   * chased, the correct credentials are set by the rebind
   * procedure.
   */
  if (session->info->userpw != NULL)
    {
      _pam_overwrite (session->info->userpw);
      _pam_drop (session->info->userpw);
    }

  session->info->userpw = strdup (password);
  if (session->info->userpw == NULL)
    return 1;

  msgid = ldap_simple_bind (session->ld, session->info->userdn,
			    session->info->userpw);
  if (msgid == -1)
    {
      if (isatty (fileno (stderr)))
	fprintf (stderr, "ldap_simple_bind %s\n",
		 ldap_err2string (ldap_get_lderrno (session->ld, 0, 0)));
      else
	syslog (LOG_ERR, "ldap_simple_bind %s",
		ldap_err2string (ldap_get_lderrno (session->ld, 0, 0)));
      _pam_overwrite (session->info->userpw);
      _pam_drop (session->info->userpw);
      return ldap_get_lderrno (session->ld, 0, 0);
    }

  timeout.tv_sec = 10;
  timeout.tv_usec = 0;
  rc = ldap_result (session->ld, msgid, FALSE, &timeout, &result);
  if (rc == -1 || rc == 0)
    {
      if (isatty (fileno (stderr)))
	fprintf (stderr, "ldap_result %s\n",
		 ldap_err2string (ldap_get_lderrno (session->ld, 0, 0)));
      else
	syslog (LOG_ERR, "ldap_result %s",
		ldap_err2string (ldap_get_lderrno (session->ld, 0, 0)));
      _pam_overwrite (session->info->userpw);
      _pam_drop (session->info->userpw);
      return 1;
    }

#if defined(HAVE_LDAP_PARSE_RESULT) && defined(HAVE_LDAP_CONTROLS_FREE)
  controls = 0;
  parserc =
    ldap_parse_result (session->ld, result, &rc, 0, 0, 0, &controls, TRUE);
  if (parserc != LDAP_SUCCESS)
    {
      if (isatty (fileno (stderr)))
	fprintf (stderr, "ldap_parse_result %s\n",
		 ldap_err2string (parserc));
      else
	syslog (LOG_ERR, "ldap_parse_result %s",
		ldap_err2string (parserc));
      _pam_overwrite (session->info->userpw);
      _pam_drop (session->info->userpw);
      return parserc;
    }

  if (controls != NULL)
    {
      LDAPControl **ctlp;
      for (ctlp = controls; *ctlp != NULL; ctlp++)
	{
	  if (!strcmp ((*ctlp)->ldctl_oid, LDAP_CONTROL_PWEXPIRING))
	    {
	      char seconds[32];
	      snprintf (seconds, sizeof seconds, "%.*s",
			(int) (*ctlp)->ldctl_value.bv_len,
			(*ctlp)->ldctl_value.bv_val);
	      session->info->password_expiration_time = atol (seconds);
	    }
	  else if (!strcmp ((*ctlp)->ldctl_oid, LDAP_CONTROL_PWEXPIRED))
	    {
	      session->info->password_expired = 1;
	      _pam_overwrite (session->info->userpw);
	      _pam_drop (session->info->userpw);
	      rc = LDAP_SUCCESS;
	      /* That may be a lie, but we need to get to the acct_mgmt
	       * step and force the change...
	       */
	    }
	}
      ldap_controls_free (controls);
    }
#else
  rc = ldap_result2error (session->ld, result, TRUE);
#endif

  if (rc != LDAP_SUCCESS)
    {
      if (isatty (fileno (stderr)))
	fprintf (stderr, "error trying to bind as user \"%s\" (%s)\n",
		 session->info->userdn, ldap_err2string (rc));
      else
	syslog (LOG_ERR, "error trying to bind as user \"%s\" (%s)",
		session->info->userdn, ldap_err2string (rc));
      _pam_overwrite (session->info->userpw);
      _pam_drop (session->info->userpw);
      return rc;
    }

  session->info->bound_as_user = 1;
  /* userpw is now set. Be sure to clobber it later. */

  return 0;
}

static int
_get_integer_value (LDAP *ld, LDAPMessage *e, const char *attr, int *ptr)
{
  char **vals;

  vals = ldap_get_values (ld, e, attr);
  if (vals == NULL)
      return 1;
  *ptr = atoi (vals[0]);
  ldap_value_free (vals);

  return 0;
}

static int
_get_long_integer_value (LDAP *ld, LDAPMessage *e, const char *attr,
			 long int *ptr)
{
  char **vals;

  vals = ldap_get_values (ld, e, attr);
  if (vals == NULL)
    return 1;
  *ptr = atol (vals[0]);
  ldap_value_free (vals);

  return 0;
}

static int
_get_string_value (LDAP * ld, LDAPMessage * e, const char *attr, char **ptr)
{
  char **vals;
  int rc;

  vals = ldap_get_values (ld, e, attr);
  if (vals == NULL)
    return 0;

  *ptr = strdup (vals[0]);
  if (*ptr == NULL)
    rc = 1;
  else
    rc = 0;

  ldap_value_free (vals);

  return rc;
}

static int
_get_string_values (LDAP * ld, LDAPMessage * e, const char *attr, char ***ptr)
{
  char **vals;

  vals = ldap_get_values (ld, e, attr);
  if (vals == NULL)
    {
      return 1;
    }
  *ptr = vals;

  return 0;
}

static int
_escape_string (const char *str, char *buf, size_t buflen)
{
  int ret = 1;
  char *p = buf;
  char *limit = p + buflen - 3;
  const char *s = str;

  while (p < limit && *s)
    {
      switch (*s)
	{
	case '*':
	  strcpy (p, "\\2a");
	  p += 3;
	  break;
	case '(':
	  strcpy (p, "\\28");
	  p += 3;
	  break;
	case ')':
	  strcpy (p, "\\29");
	  p += 3;
	  break;
	case '\\':
	  strcpy (p, "\\5c");
	  p += 3;
	  break;
	default:
	  *p++ = *s;
	  break;
	}
      s++;
    }

  if (*s == '\0')
    {
      /* got to end */
      *p = '\0';
      ret = 0;
    }

  return ret;
}

int
get_ldapuser_info (ldap_session_t *session, const char *user)
{
  /* XXX make this strings variable */
  char filter[1024], escapedUser[strlen (user) * 3 + 3];
  int rc;
  LDAPMessage *res, *msg;

  rc = _connect_anonymously (session);
  if (rc != 0)
    return rc;

#if defined(HAVE_LDAP_SET_OPTION) && defined(LDAP_OPT_SIZELIMIT)
  rc = 1;
  ldap_set_option (session->ld, LDAP_OPT_SIZELIMIT, &rc);
#else
  session->ld->ld_sizelimit = 1;
#endif

  rc = _escape_string (user, escapedUser, sizeof (escapedUser));
  if (rc != 0)
    return rc;

  if (session->conf->filter != NULL)
    snprintf (filter, sizeof filter, "(&(%s)(%s=%s))",
	      session->conf->filter, session->conf->userattr, escapedUser);
  else
    snprintf (filter, sizeof filter, "(%s=%s)",
	      session->conf->userattr, escapedUser);

  rc = ldap_search_s (session->ld, session->conf->base,
		      session->conf->scope, filter, NULL, 0, &res);

  if (rc != LDAP_SUCCESS &&
      rc != LDAP_TIMELIMIT_EXCEEDED && rc != LDAP_SIZELIMIT_EXCEEDED)
    {
      if (isatty (fileno (stderr)))
	fprintf (stderr, "ldap_search_s %s", ldap_err2string (rc));
      else
	syslog (LOG_ERR, "ldap_search_s %s", ldap_err2string (rc));
      return 1;
    }

  msg = ldap_first_entry (session->ld, res);
  if (msg == NULL)
    {
      ldap_msgfree (res);
      return 1;
    }

  if (session->info != NULL)
    _release_user_info (&session->info);

  session->info =
    (ldap_user_info_t *) calloc (1, sizeof (ldap_user_info_t));
  if (session->info == NULL)
    {
      ldap_msgfree (res);
      return 1;
    }

  session->info->username = strdup (user);
  if (session->info->username == NULL)
    {
      ldap_msgfree (res);
      _release_user_info (&session->info);
      return 1;
    }

  session->info->userdn = ldap_get_dn (session->ld, msg);
  if (session->info->userdn == NULL)
    {
      ldap_msgfree (res);
      _release_user_info (&session->info);
      return 1;
    }

  session->info->bound_as_user = 0;

  /*
   * it might be better to do a compare later, that way we can
   * avoid fetching any attributes at all
   */
  _get_string_values (session->ld, msg, "host", &session->info->hosts_allow);

  /* get UID */
#ifdef UID_NOBODY
  session->info->uid = UID_NOBODY;
#else
  session->info->uid = (uid_t) - 2;
#endif /* UID_NOBODY */
  _get_integer_value (session->ld, msg, "uidNumber",
		      (uid_t *) & session->info->uid);

  /*
   * get mapped user; some PAM host applications let PAM_USER be reset
   * by the user (such as some of those provided with FreeBSD).
   */
  session->info->tmpluser = NULL;
  if (session->conf->tmplattr != NULL)
    {
      if (_get_string_value (session->ld, msg,
			     session->conf->tmplattr,
			     &session->info->tmpluser) != 0)
	{
	  /* set to default template user */
	  session->info->tmpluser =
	    session->conf->tmpluser ? strdup (session->conf->tmpluser) : NULL;
	}
    }

  /* Assume shadow controls.  Allocate shadow structure and link to session. */
  session->info->shadow.lstchg = -1;
  session->info->shadow.min = 0;
  session->info->shadow.max = 0;
  session->info->shadow.warn = 0;
  session->info->shadow.inact = 0;
  session->info->shadow.expire = 0;
  session->info->shadow.flag = 0;

  _get_long_integer_value (session->ld, msg, "shadowLastChange",
			   &session->info->shadow.lstchg);
  _get_long_integer_value (session->ld, msg, "shadowMin",
			   &session->info->shadow.min);
  _get_long_integer_value (session->ld, msg, "shadowMax",
			   &session->info->shadow.max);
  _get_long_integer_value (session->ld, msg, "shadowWarning",
			   &session->info->shadow.warn);
  _get_long_integer_value (session->ld, msg, "shadowInactive",
			   &session->info->shadow.inact);
  _get_long_integer_value (session->ld, msg, "shadowExpire",
			   &session->info->shadow.expire);
  _get_long_integer_value (session->ld, msg, "shadowFlag",
			   &session->info->shadow.flag);

  ldap_msgfree (res);

  return 0;
}

ldap_session_t *
create_ldap_session (const char *configFile)
{
  ldap_session_t *session;

  session = (ldap_session_t *) calloc (1, sizeof (*session));
#if LDAP_SET_REBIND_PROC_ARGS < 3
  global_session = session;
#endif
  if (session == NULL)
    return NULL;

  session->ld = NULL;
  session->conf = NULL;
  session->info = NULL;

  session->conf = read_ldap_config (configFile);
  if (session->conf == NULL)
    {
      free (session);
      return NULL;
    }

  return session;
}

static int
_reopen (ldap_session_t *session)
{
  /* FYI: V3 lets us avoid five unneeded binds in a password change */
  if (session->conf->version == LDAP_VERSION2)
    {
      if (session->ld != NULL)
	{
	  ldap_unbind (session->ld);
	  session->ld = NULL;
	}
      if (session->info != NULL)
	{
	  session->info->bound_as_user = 0;
	}
      return open_ldap_session (session);
    }
  return 0;
}

int
ldap_authentication (ldap_session_t *session,
		    const char *user, const char *password)
{
  int rc = 0;

  if (session->info == NULL)
    {
      rc = get_ldapuser_info (session, user);
      if (rc != 0)
	return rc;
    }

  rc = _reopen (session);
  if (rc != LDAP_SUCCESS)
    return rc;

  rc = _connect_as_user (session, password);
  _reopen (session);
  _connect_anonymously (session);
  return rc;
}

int
ldap_update_shell (ldap_session_t *session, const char *user,
		   const char *old_password, const char *new_shell)
{
  return ldap_update_field (session, user, NULL, old_password,
			    "loginShell", new_shell);
}

int
ldap_update_gecos (ldap_session_t *session, const char *user,
		   const char *old_password, const char *new_gecos)
{
  return ldap_update_field (session, user, NULL, old_password,
			    "gecos", new_gecos);
}

int
ldap_update_field (ldap_session_t *session, const char *user,
		   const char *binddn, const char *bindpw,
		   const char *field, const char *new_value)
{
  LDAPMod *mods[2], mod;
  char *strvals[2];
  int rc = 0;
  char *cp_binddn = NULL;
  char *cp_bindpw = NULL;

  if (binddn != NULL)
    {
      cp_binddn = session->conf->binddn;
      cp_bindpw = session->conf->bindpw;
      session->conf->binddn = strdup(binddn);
      session->conf->bindpw = strdup(bindpw);
    }

  if (session->info == NULL && user != NULL)
    {
      rc = get_ldapuser_info (session, user);
      if (rc != 0)
	{
	  if (binddn != NULL)
	    {
	      free (session->conf->binddn);
	      free (session->conf->bindpw);
	      session->conf->binddn = cp_binddn;
	      session->conf->bindpw = cp_bindpw;
	    }
	  return rc;
	}
    }

  if ((!session->conf->rootbinddn || (geteuid () != 0))
      && (binddn == NULL))
    {
      /* We're not root or don't have a rootbinddn so
	 let's try binding as the user. Exept we have a
	 binddn for this case.  */
      rc = _reopen (session);
      if (rc != 0)
	return rc;

      rc = _connect_as_user (session, bindpw);
      if (rc != 0)
	return rc;
    }

  /* update field */
  strvals[0] = strdupa (new_value);
  strvals[1] = NULL;

  mod.mod_values = strvals;
  mod.mod_type = strdupa (field);
  mod.mod_op = LDAP_MOD_REPLACE;

  mods[0] = &mod;
  mods[1] = NULL;

  rc = ldap_modify_s (session->ld, session->info->userdn, mods);
  _reopen (session);
  _connect_anonymously (session);

  if (binddn != NULL)
    {
      free (session->conf->binddn);
      free (session->conf->bindpw);
      session->conf->binddn = cp_binddn;
      session->conf->bindpw = cp_bindpw;
    }

  return rc;
}

#endif /* USE_LDAP */
