"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SecurityClient = void 0;

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class SecurityClient {
  constructor(esClient) {
    this.esClient = esClient;
  }

  async authenticate(request, credentials) {
    const authHeader = Buffer.from(`${credentials.username}:${credentials.password}`).toString('base64');

    try {
      const esResponse = await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.authinfo', {
        headers: {
          authorization: `Basic ${authHeader}`
        }
      });
      return {
        username: credentials.username,
        roles: esResponse.roles,
        backendRoles: esResponse.backend_roles,
        tenants: esResponse.tenants,
        selectedTenant: esResponse.user_requested_tenant,
        credentials,
        proxyCredentials: credentials,
        tenancy_configs: esResponse.tenancy_configs
      };
    } catch (error) {
      throw new Error(error.message);
    }
  }

  async authenticateWithHeader(request, headerName, headerValue, whitelistedHeadersAndValues = {}, additionalAuthHeaders = {}) {
    try {
      const credentials = {
        headerName,
        headerValue
      };
      const headers = {};

      if (headerValue) {
        headers[headerName] = headerValue;
      } // cannot get config elasticsearch.requestHeadersWhitelist from kibana.yml file in new platfrom
      // meanwhile, do we really need to save all headers in cookie?


      const esResponse = await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.authinfo', {
        headers
      });
      return {
        username: esResponse.user_name,
        roles: esResponse.roles,
        backendRoles: esResponse.backend_roles,
        tenants: esResponse.teanats,
        selectedTenant: esResponse.user_requested_tenant,
        credentials
      };
    } catch (error) {
      throw new Error(error.message);
    }
  }

  async authenticateWithHeaders(request, additionalAuthHeaders = {}) {
    try {
      const esResponse = await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.authinfo', {
        headers: additionalAuthHeaders
      });
      return {
        username: esResponse.user_name,
        roles: esResponse.roles,
        backendRoles: esResponse.backend_roles,
        tenants: esResponse.tenants,
        selectedTenant: esResponse.user_requested_tenant
      };
    } catch (error) {
      throw new Error(error.message);
    }
  }

  async authinfo(request, headers = {}) {
    try {
      return await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.authinfo', {
        headers
      });
    } catch (error) {
      throw new Error(error.message);
    }
  }

  async dashboardsinfo(request, headers = {}) {
    try {
      return await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.dashboardsinfo', {
        headers
      });
    } catch (error) {
      throw new Error(error.message);
    }
  } // Multi-tenancy APIs


  async getMultitenancyInfo(request) {
    try {
      return await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.multitenancyinfo');
    } catch (error) {
      throw new Error(error.message);
    }
  }

  async putMultitenancyConfigurations(request, tenancyConfigSettings) {
    const body = {
      multitenancy_enabled: tenancyConfigSettings.multitenancy_enabled,
      private_tenant_enabled: tenancyConfigSettings.private_tenant_enabled,
      default_tenant: tenancyConfigSettings.default_tenant
    };

    try {
      return await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.tenancy_configs', {
        body
      });
    } catch (error) {
      throw new Error(error.message);
    }
  }

  async getTenantInfoWithInternalUser() {
    try {
      return this.esClient.callAsInternalUser('opensearch_security.tenantinfo');
    } catch (error) {
      throw new Error(error.message);
    }
  }

  async getTenantInfo(request) {
    try {
      return await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.tenantinfo');
    } catch (error) {
      throw new Error(error.message);
    }
  }

  async getSamlHeader(request) {
    try {
      // response is expected to be an error
      await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.authinfo');
    } catch (error) {
      // the error looks like
      // wwwAuthenticateDirective:
      //   '
      //     X-Security-IdP realm="Open Distro Security"
      //     location="https://<your-auth-domain.com>/api/saml2/v1/sso?SAMLRequest=<some-encoded-string>"
      //     requestId="<request_id>"
      //   '
      if (!error.wwwAuthenticateDirective) {
        throw error;
      }

      try {
        const locationRegExp = /location="(.*?)"/;
        const requestIdRegExp = /requestId="(.*?)"/;
        const locationExecArray = locationRegExp.exec(error.wwwAuthenticateDirective);
        const requestExecArray = requestIdRegExp.exec(error.wwwAuthenticateDirective);

        if (locationExecArray && requestExecArray) {
          return {
            location: locationExecArray[1],
            requestId: requestExecArray[1]
          };
        }

        throw Error('failed parsing SAML config');
      } catch (parsingError) {
        console.log(parsingError);
        throw new Error(parsingError);
      }
    }

    throw new Error(`Invalid SAML configuration.`);
  }

  async authToken(requestId, samlResponse, acsEndpoint = undefined) {
    const body = {
      RequestId: requestId,
      SAMLResponse: samlResponse,
      acsEndpoint
    };

    try {
      return await this.esClient.asScoped().callAsCurrentUser('opensearch_security.authtoken', {
        body
      });
    } catch (error) {
      console.log(error);
      throw new Error('failed to get token');
    }
  }

}

exports.SecurityClient = SecurityClient;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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