/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.index.autoforcemerge;

import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.action.admin.indices.forcemerge.ForceMergeRequest;
import org.opensearch.action.admin.indices.stats.CommonStats;
import org.opensearch.action.admin.indices.stats.CommonStatsFlags;
import org.opensearch.cluster.ClusterState;
import org.opensearch.cluster.node.DiscoveryNode;
import org.opensearch.cluster.service.ClusterService;
import org.opensearch.common.lifecycle.AbstractLifecycleComponent;
import org.opensearch.common.unit.TimeValue;
import org.opensearch.common.util.concurrent.AbstractAsyncTask;
import org.opensearch.common.util.concurrent.OpenSearchExecutors;
import org.opensearch.gateway.remote.RemoteClusterStateService;
import org.opensearch.index.IndexService;
import org.opensearch.index.IndexSettings;
import org.opensearch.index.autoforcemerge.ForceMergeManagerSettings;
import org.opensearch.index.engine.SegmentsStats;
import org.opensearch.index.shard.IndexShard;
import org.opensearch.index.shard.IndexShardState;
import org.opensearch.index.translog.TranslogStats;
import org.opensearch.indices.IndicesService;
import org.opensearch.monitor.MonitorService;
import org.opensearch.monitor.fs.FsService;
import org.opensearch.monitor.jvm.JvmService;
import org.opensearch.monitor.os.OsService;
import org.opensearch.threadpool.ThreadPool;
import org.opensearch.threadpool.ThreadPoolStats;

public class AutoForceMergeManager
extends AbstractLifecycleComponent {
    private final ThreadPool threadPool;
    private final OsService osService;
    private final FsService fsService;
    private final JvmService jvmService;
    private final IndicesService indicesService;
    private final ClusterService clusterService;
    private final AsyncForceMergeTask task;
    private ConfigurationValidator configurationValidator;
    private NodeValidator nodeValidator;
    private ShardValidator shardValidator;
    private final ForceMergeManagerSettings forceMergeManagerSettings;
    private final CommonStatsFlags flags = new CommonStatsFlags(CommonStatsFlags.Flag.Segments, CommonStatsFlags.Flag.Translog);
    private final Set<Integer> mergingShards;
    private Integer allocatedProcessors;
    private static final Logger logger = LogManager.getLogger(AutoForceMergeManager.class);

    public AutoForceMergeManager(ThreadPool threadPool, MonitorService monitorService, IndicesService indicesService, ClusterService clusterService) {
        this.threadPool = threadPool;
        this.osService = monitorService.osService();
        this.fsService = monitorService.fsService();
        this.jvmService = monitorService.jvmService();
        this.clusterService = clusterService;
        this.indicesService = indicesService;
        this.forceMergeManagerSettings = new ForceMergeManagerSettings(clusterService, this::modifySchedulerInterval);
        this.task = new AsyncForceMergeTask();
        this.mergingShards = new HashSet<Integer>();
    }

    protected void doStart() {
        this.configurationValidator = new ConfigurationValidator();
        this.nodeValidator = new NodeValidator();
        this.shardValidator = new ShardValidator();
        this.allocatedProcessors = OpenSearchExecutors.allocatedProcessors(this.clusterService.getSettings());
    }

    protected void doStop() {
        if (this.task != null) {
            this.task.close();
        }
    }

    protected void doClose() {
        if (this.task != null) {
            this.task.close();
        }
    }

    private void modifySchedulerInterval(TimeValue schedulerInterval) {
        this.task.setInterval(schedulerInterval);
    }

    private void triggerForceMerge() {
        if (!this.configurationValidator.hasWarmNodes()) {
            logger.debug("No warm nodes found. Skipping Auto Force merge.");
            return;
        }
        if (!this.nodeValidator.validate().isAllowed()) {
            logger.debug("Node capacity constraints are not allowing to trigger auto ForceMerge");
            return;
        }
        int iteration = this.nodeValidator.getMaxConcurrentForceMerges();
        for (IndexShard shard : this.getShardsBasedOnSorting(this.indicesService)) {
            if (iteration == 0) break;
            if (!this.nodeValidator.validate().isAllowed()) {
                logger.debug("Node conditions no longer suitable for force merge.");
                break;
            }
            --iteration;
            CompletableFuture.runAsync(() -> {
                try {
                    this.mergingShards.add(shard.shardId().getId());
                    shard.forceMerge(new ForceMergeRequest(new String[0]).maxNumSegments(this.forceMergeManagerSettings.getSegmentCount()));
                    logger.debug("Merging is completed successfully for the shard {}", (Object)shard.shardId());
                }
                catch (Exception e) {
                    logger.error("Error during force merge for shard {}\nException: {}", (Object)shard.shardId(), (Object)e);
                }
                finally {
                    this.mergingShards.remove(shard.shardId().getId());
                }
            }, this.threadPool.executor("force_merge"));
            logger.info("Successfully triggered force merge for shard {}", (Object)shard.shardId());
            try {
                Thread.sleep(this.forceMergeManagerSettings.getForcemergeDelay().getMillis());
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                logger.error("Timer was interrupted while waiting between shards", (Throwable)e);
                break;
            }
        }
    }

    private List<IndexShard> getShardsBasedOnSorting(Iterable<IndexService> indicesService) {
        HashMap<IndexShard, Long> shardAgeCache = new HashMap<IndexShard, Long>();
        return StreamSupport.stream(indicesService.spliterator(), false).flatMap(indexService -> StreamSupport.stream(indexService.spliterator(), false)).filter(shard -> !shard.shardId().getIndexName().startsWith(".")).filter(shard -> shard.routingEntry().primary()).filter(shard -> !this.mergingShards.contains(shard.shardId().getId())).filter(shard -> this.shardValidator.validate((IndexShard)shard).isAllowed()).peek(shard -> shardAgeCache.computeIfAbsent((IndexShard)shard, this::getEarliestLastModifiedAge)).sorted(new ShardAgeComparator(shardAgeCache)).limit(this.getNodeValidator().getMaxConcurrentForceMerges().intValue()).collect(Collectors.toList());
    }

    private long getEarliestLastModifiedAge(IndexShard shard) {
        CommonStats stats = new CommonStats(this.indicesService.getIndicesQueryCache(), shard, this.flags);
        return stats.getTranslog() != null ? stats.getTranslog().getEarliestLastModifiedAge() : 0L;
    }

    protected AsyncForceMergeTask getTask() {
        return this.task;
    }

    protected ConfigurationValidator getConfigurationValidator() {
        return this.configurationValidator;
    }

    protected NodeValidator getNodeValidator() {
        return this.nodeValidator;
    }

    protected ShardValidator getShardValidator() {
        return this.shardValidator;
    }

    protected final class AsyncForceMergeTask
    extends AbstractAsyncTask {
        public AsyncForceMergeTask() {
            super(logger, AutoForceMergeManager.this.threadPool, AutoForceMergeManager.this.forceMergeManagerSettings.getSchedulerInterval(), true);
            this.rescheduleIfNecessary();
        }

        @Override
        protected boolean mustReschedule() {
            return true;
        }

        @Override
        protected void runInternal() {
            if (!AutoForceMergeManager.this.configurationValidator.validate().isAllowed()) {
                return;
            }
            AutoForceMergeManager.this.triggerForceMerge();
        }

        @Override
        protected String getThreadPool() {
            return "generic";
        }
    }

    protected class ConfigurationValidator
    implements ValidationStrategy {
        private final boolean isOnlyDataNode;
        private boolean isRemoteStoreEnabled = false;
        private boolean hasWarmNodes = false;

        ConfigurationValidator() {
            DiscoveryNode localNode = AutoForceMergeManager.this.clusterService.localNode();
            this.isOnlyDataNode = localNode.isDataNode() && !localNode.isWarmNode();
            this.isRemoteStoreEnabled = this.isRemoteStorageEnabled();
        }

        @Override
        public ValidationResult validate() {
            if (!AutoForceMergeManager.this.forceMergeManagerSettings.isAutoForceMergeFeatureEnabled().booleanValue()) {
                logger.debug("Cluster configuration shows auto force merge feature is disabled. Closing task.");
                return new ValidationResult(false);
            }
            if (!this.isRemoteStoreEnabled) {
                logger.debug("Cluster configuration is not meeting the criteria. Closing task.");
                AutoForceMergeManager.this.task.close();
                return new ValidationResult(false);
            }
            if (!this.isOnlyDataNode) {
                logger.debug("Node configuration doesn't meet requirements. Closing task.");
                AutoForceMergeManager.this.task.close();
                return new ValidationResult(false);
            }
            return new ValidationResult(true);
        }

        private boolean isRemoteStorageEnabled() {
            return AutoForceMergeManager.this.clusterService.getSettings().getAsBoolean(RemoteClusterStateService.REMOTE_CLUSTER_STATE_ENABLED_SETTING.getKey(), false);
        }

        private boolean hasWarmNodes() {
            if (this.hasWarmNodes) {
                return true;
            }
            ClusterState clusterState = AutoForceMergeManager.this.clusterService.state();
            this.hasWarmNodes = clusterState.getNodes().getNodes().values().stream().anyMatch(DiscoveryNode::isWarmNode);
            return this.hasWarmNodes;
        }
    }

    protected class NodeValidator
    implements ValidationStrategy {
        protected NodeValidator() {
        }

        @Override
        public ValidationResult validate() {
            if (this.isCpuUsageOverThreshold()) {
                return new ValidationResult(false);
            }
            if (this.isDiskUsageOverThreshold()) {
                return new ValidationResult(false);
            }
            double jvmUsedPercent = AutoForceMergeManager.this.jvmService.stats().getMem().getHeapUsedPercent();
            if (jvmUsedPercent >= AutoForceMergeManager.this.forceMergeManagerSettings.getJvmThreshold()) {
                logger.debug("JVM memory: {}% breached the threshold: {}", (Object)jvmUsedPercent, (Object)AutoForceMergeManager.this.forceMergeManagerSettings.getJvmThreshold());
                return new ValidationResult(false);
            }
            if (!this.areForceMergeThreadsAvailable()) {
                logger.debug("No force merge threads available");
                return new ValidationResult(false);
            }
            return new ValidationResult(true);
        }

        private boolean areForceMergeThreadsAvailable() {
            for (ThreadPoolStats.Stats stats : AutoForceMergeManager.this.threadPool.stats()) {
                if (!stats.getName().equals("force_merge")) continue;
                return stats.getQueue() == 0;
            }
            return false;
        }

        private boolean isCpuUsageOverThreshold() {
            double[] loadAverage = AutoForceMergeManager.this.osService.stats().getCpu().getLoadAverage();
            double loadAverage5m = loadAverage[1] / (double)AutoForceMergeManager.this.allocatedProcessors.intValue() * 100.0;
            if (loadAverage5m >= AutoForceMergeManager.this.forceMergeManagerSettings.getCpuThreshold()) {
                logger.debug("Load Average: 5m({}%) breached the threshold: {}", (Object)loadAverage5m, (Object)AutoForceMergeManager.this.forceMergeManagerSettings.getCpuThreshold());
                return true;
            }
            double loadAverage1m = loadAverage[0] / (double)AutoForceMergeManager.this.allocatedProcessors.intValue() * 100.0;
            if (loadAverage1m >= AutoForceMergeManager.this.forceMergeManagerSettings.getCpuThreshold()) {
                logger.debug("Load Average: 1m({}%) breached the threshold: {}", (Object)loadAverage1m, (Object)AutoForceMergeManager.this.forceMergeManagerSettings.getCpuThreshold());
                return true;
            }
            double cpuPercent = AutoForceMergeManager.this.osService.stats().getCpu().getPercent();
            if (cpuPercent >= AutoForceMergeManager.this.forceMergeManagerSettings.getCpuThreshold()) {
                logger.debug("CPU usage: {} breached the threshold: {}", (Object)cpuPercent, (Object)AutoForceMergeManager.this.forceMergeManagerSettings.getCpuThreshold());
                return true;
            }
            return false;
        }

        private boolean isDiskUsageOverThreshold() {
            long available;
            long total = AutoForceMergeManager.this.fsService.stats().getTotal().getTotal().getBytes();
            double diskPercent = (double)(total - (available = AutoForceMergeManager.this.fsService.stats().getTotal().getAvailable().getBytes())) / (double)total * 100.0;
            if (diskPercent >= AutoForceMergeManager.this.forceMergeManagerSettings.getDiskThreshold()) {
                logger.debug("Disk usage: {}% breached the threshold: {}", (Object)diskPercent, (Object)AutoForceMergeManager.this.forceMergeManagerSettings.getDiskThreshold());
                return true;
            }
            return false;
        }

        public Integer getMaxConcurrentForceMerges() {
            return Math.max(1, AutoForceMergeManager.this.allocatedProcessors / 8) * AutoForceMergeManager.this.forceMergeManagerSettings.getConcurrencyMultiplier();
        }
    }

    protected class ShardValidator
    implements ValidationStrategy {
        protected ShardValidator() {
        }

        @Override
        public ValidationResult validate(IndexShard shard) {
            if (shard.state() != IndexShardState.STARTED) {
                logger.debug("Shard({}) skipped: Shard is not in started state.", (Object)shard.shardId());
                return new ValidationResult(false);
            }
            if (!this.isIndexAutoForceMergeEnabled(shard)) {
                logger.debug("Shard({}) skipped: Shard doesn't belong to a warm candidate index", (Object)shard.shardId());
                return new ValidationResult(false);
            }
            CommonStats stats = new CommonStats(AutoForceMergeManager.this.indicesService.getIndicesQueryCache(), shard, AutoForceMergeManager.this.flags);
            TranslogStats translogStats = stats.getTranslog();
            if (translogStats != null && translogStats.getEarliestLastModifiedAge() < AutoForceMergeManager.this.forceMergeManagerSettings.getTranslogAge().getMillis()) {
                logger.debug("Shard({}) skipped: Translog is too recent. Age({}ms)", (Object)shard.shardId(), (Object)translogStats.getEarliestLastModifiedAge());
                return new ValidationResult(false);
            }
            SegmentsStats segmentsStats = stats.getSegments();
            if (segmentsStats != null && segmentsStats.getCount() <= (long)AutoForceMergeManager.this.forceMergeManagerSettings.getSegmentCount().intValue()) {
                logger.debug("Shard({}) skipped: Shard has {} segments, not exceeding threshold of {}", (Object)shard.shardId(), (Object)segmentsStats.getCount(), (Object)AutoForceMergeManager.this.forceMergeManagerSettings.getSegmentCount());
                return new ValidationResult(false);
            }
            return new ValidationResult(true);
        }

        private boolean isIndexAutoForceMergeEnabled(IndexShard shard) {
            IndexSettings indexSettings = shard.indexSettings();
            return indexSettings.isAutoForcemergeEnabled();
        }
    }

    public static final class ValidationResult {
        private final boolean allowed;

        public ValidationResult(boolean allowed) {
            this.allowed = allowed;
        }

        public boolean isAllowed() {
            return this.allowed;
        }
    }

    private static class ShardAgeComparator
    implements Comparator<IndexShard> {
        private final Map<IndexShard, Long> shardAgeCache;

        public ShardAgeComparator(Map<IndexShard, Long> shardAgeCache) {
            this.shardAgeCache = shardAgeCache;
        }

        @Override
        public int compare(IndexShard s1, IndexShard s2) {
            long age1 = this.shardAgeCache.get(s1);
            long age2 = this.shardAgeCache.get(s2);
            return Long.compare(age1, age2);
        }
    }

    public static interface ValidationStrategy {
        default public ValidationResult validate() {
            return new ValidationResult(false);
        }

        default public ValidationResult validate(IndexShard shard) {
            return new ValidationResult(false);
        }
    }
}

