﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/databrew/GlueDataBrewRequest.h>
#include <aws/databrew/GlueDataBrew_EXPORTS.h>
#include <aws/databrew/model/EncryptionMode.h>
#include <aws/databrew/model/JobSample.h>
#include <aws/databrew/model/LogSubscription.h>
#include <aws/databrew/model/ProfileConfiguration.h>
#include <aws/databrew/model/S3Location.h>
#include <aws/databrew/model/ValidationConfiguration.h>

#include <utility>

namespace Aws {
namespace GlueDataBrew {
namespace Model {

/**
 */
class CreateProfileJobRequest : public GlueDataBrewRequest {
 public:
  AWS_GLUEDATABREW_API CreateProfileJobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateProfileJob"; }

  AWS_GLUEDATABREW_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the dataset that this job is to act upon.</p>
   */
  inline const Aws::String& GetDatasetName() const { return m_datasetName; }
  inline bool DatasetNameHasBeenSet() const { return m_datasetNameHasBeenSet; }
  template <typename DatasetNameT = Aws::String>
  void SetDatasetName(DatasetNameT&& value) {
    m_datasetNameHasBeenSet = true;
    m_datasetName = std::forward<DatasetNameT>(value);
  }
  template <typename DatasetNameT = Aws::String>
  CreateProfileJobRequest& WithDatasetName(DatasetNameT&& value) {
    SetDatasetName(std::forward<DatasetNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of an encryption key that is used to protect
   * the job.</p>
   */
  inline const Aws::String& GetEncryptionKeyArn() const { return m_encryptionKeyArn; }
  inline bool EncryptionKeyArnHasBeenSet() const { return m_encryptionKeyArnHasBeenSet; }
  template <typename EncryptionKeyArnT = Aws::String>
  void SetEncryptionKeyArn(EncryptionKeyArnT&& value) {
    m_encryptionKeyArnHasBeenSet = true;
    m_encryptionKeyArn = std::forward<EncryptionKeyArnT>(value);
  }
  template <typename EncryptionKeyArnT = Aws::String>
  CreateProfileJobRequest& WithEncryptionKeyArn(EncryptionKeyArnT&& value) {
    SetEncryptionKeyArn(std::forward<EncryptionKeyArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The encryption mode for the job, which can be one of the following:</p> <ul>
   * <li> <p> <code>SSE-KMS</code> - <code>SSE-KMS</code> - Server-side encryption
   * with KMS-managed keys.</p> </li> <li> <p> <code>SSE-S3</code> - Server-side
   * encryption with keys managed by Amazon S3.</p> </li> </ul>
   */
  inline EncryptionMode GetEncryptionMode() const { return m_encryptionMode; }
  inline bool EncryptionModeHasBeenSet() const { return m_encryptionModeHasBeenSet; }
  inline void SetEncryptionMode(EncryptionMode value) {
    m_encryptionModeHasBeenSet = true;
    m_encryptionMode = value;
  }
  inline CreateProfileJobRequest& WithEncryptionMode(EncryptionMode value) {
    SetEncryptionMode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the job to be created. Valid characters are alphanumeric (A-Z,
   * a-z, 0-9), hyphen (-), period (.), and space.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateProfileJobRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Enables or disables Amazon CloudWatch logging for the job. If logging is
   * enabled, CloudWatch writes one log stream for each job run.</p>
   */
  inline LogSubscription GetLogSubscription() const { return m_logSubscription; }
  inline bool LogSubscriptionHasBeenSet() const { return m_logSubscriptionHasBeenSet; }
  inline void SetLogSubscription(LogSubscription value) {
    m_logSubscriptionHasBeenSet = true;
    m_logSubscription = value;
  }
  inline CreateProfileJobRequest& WithLogSubscription(LogSubscription value) {
    SetLogSubscription(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of nodes that DataBrew can use when the job processes
   * data.</p>
   */
  inline int GetMaxCapacity() const { return m_maxCapacity; }
  inline bool MaxCapacityHasBeenSet() const { return m_maxCapacityHasBeenSet; }
  inline void SetMaxCapacity(int value) {
    m_maxCapacityHasBeenSet = true;
    m_maxCapacity = value;
  }
  inline CreateProfileJobRequest& WithMaxCapacity(int value) {
    SetMaxCapacity(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of times to retry the job after a job run fails.</p>
   */
  inline int GetMaxRetries() const { return m_maxRetries; }
  inline bool MaxRetriesHasBeenSet() const { return m_maxRetriesHasBeenSet; }
  inline void SetMaxRetries(int value) {
    m_maxRetriesHasBeenSet = true;
    m_maxRetries = value;
  }
  inline CreateProfileJobRequest& WithMaxRetries(int value) {
    SetMaxRetries(value);
    return *this;
  }
  ///@}

  ///@{

  inline const S3Location& GetOutputLocation() const { return m_outputLocation; }
  inline bool OutputLocationHasBeenSet() const { return m_outputLocationHasBeenSet; }
  template <typename OutputLocationT = S3Location>
  void SetOutputLocation(OutputLocationT&& value) {
    m_outputLocationHasBeenSet = true;
    m_outputLocation = std::forward<OutputLocationT>(value);
  }
  template <typename OutputLocationT = S3Location>
  CreateProfileJobRequest& WithOutputLocation(OutputLocationT&& value) {
    SetOutputLocation(std::forward<OutputLocationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Configuration for profile jobs. Used to select columns, do evaluations, and
   * override default parameters of evaluations. When configuration is null, the
   * profile job will run with default settings.</p>
   */
  inline const ProfileConfiguration& GetConfiguration() const { return m_configuration; }
  inline bool ConfigurationHasBeenSet() const { return m_configurationHasBeenSet; }
  template <typename ConfigurationT = ProfileConfiguration>
  void SetConfiguration(ConfigurationT&& value) {
    m_configurationHasBeenSet = true;
    m_configuration = std::forward<ConfigurationT>(value);
  }
  template <typename ConfigurationT = ProfileConfiguration>
  CreateProfileJobRequest& WithConfiguration(ConfigurationT&& value) {
    SetConfiguration(std::forward<ConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>List of validation configurations that are applied to the profile job.</p>
   */
  inline const Aws::Vector<ValidationConfiguration>& GetValidationConfigurations() const { return m_validationConfigurations; }
  inline bool ValidationConfigurationsHasBeenSet() const { return m_validationConfigurationsHasBeenSet; }
  template <typename ValidationConfigurationsT = Aws::Vector<ValidationConfiguration>>
  void SetValidationConfigurations(ValidationConfigurationsT&& value) {
    m_validationConfigurationsHasBeenSet = true;
    m_validationConfigurations = std::forward<ValidationConfigurationsT>(value);
  }
  template <typename ValidationConfigurationsT = Aws::Vector<ValidationConfiguration>>
  CreateProfileJobRequest& WithValidationConfigurations(ValidationConfigurationsT&& value) {
    SetValidationConfigurations(std::forward<ValidationConfigurationsT>(value));
    return *this;
  }
  template <typename ValidationConfigurationsT = ValidationConfiguration>
  CreateProfileJobRequest& AddValidationConfigurations(ValidationConfigurationsT&& value) {
    m_validationConfigurationsHasBeenSet = true;
    m_validationConfigurations.emplace_back(std::forward<ValidationConfigurationsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the Identity and Access Management (IAM)
   * role to be assumed when DataBrew runs the job.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  CreateProfileJobRequest& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Metadata tags to apply to this job.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateProfileJobRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateProfileJobRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The job's timeout in minutes. A job that attempts to run longer than this
   * timeout period ends with a status of <code>TIMEOUT</code>.</p>
   */
  inline int GetTimeout() const { return m_timeout; }
  inline bool TimeoutHasBeenSet() const { return m_timeoutHasBeenSet; }
  inline void SetTimeout(int value) {
    m_timeoutHasBeenSet = true;
    m_timeout = value;
  }
  inline CreateProfileJobRequest& WithTimeout(int value) {
    SetTimeout(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Sample configuration for profile jobs only. Determines the number of rows on
   * which the profile job will be executed. If a JobSample value is not provided,
   * the default value will be used. The default value is CUSTOM_ROWS for the mode
   * parameter and 20000 for the size parameter.</p>
   */
  inline const JobSample& GetJobSample() const { return m_jobSample; }
  inline bool JobSampleHasBeenSet() const { return m_jobSampleHasBeenSet; }
  template <typename JobSampleT = JobSample>
  void SetJobSample(JobSampleT&& value) {
    m_jobSampleHasBeenSet = true;
    m_jobSample = std::forward<JobSampleT>(value);
  }
  template <typename JobSampleT = JobSample>
  CreateProfileJobRequest& WithJobSample(JobSampleT&& value) {
    SetJobSample(std::forward<JobSampleT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_datasetName;

  Aws::String m_encryptionKeyArn;

  EncryptionMode m_encryptionMode{EncryptionMode::NOT_SET};

  Aws::String m_name;

  LogSubscription m_logSubscription{LogSubscription::NOT_SET};

  int m_maxCapacity{0};

  int m_maxRetries{0};

  S3Location m_outputLocation;

  ProfileConfiguration m_configuration;

  Aws::Vector<ValidationConfiguration> m_validationConfigurations;

  Aws::String m_roleArn;

  Aws::Map<Aws::String, Aws::String> m_tags;

  int m_timeout{0};

  JobSample m_jobSample;
  bool m_datasetNameHasBeenSet = false;
  bool m_encryptionKeyArnHasBeenSet = false;
  bool m_encryptionModeHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_logSubscriptionHasBeenSet = false;
  bool m_maxCapacityHasBeenSet = false;
  bool m_maxRetriesHasBeenSet = false;
  bool m_outputLocationHasBeenSet = false;
  bool m_configurationHasBeenSet = false;
  bool m_validationConfigurationsHasBeenSet = false;
  bool m_roleArnHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_timeoutHasBeenSet = false;
  bool m_jobSampleHasBeenSet = false;
};

}  // namespace Model
}  // namespace GlueDataBrew
}  // namespace Aws
