﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/backup/BackupRequest.h>
#include <aws/backup/Backup_EXPORTS.h>
#include <aws/backup/model/Index.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Backup {
namespace Model {

/**
 */
class UpdateRecoveryPointIndexSettingsRequest : public BackupRequest {
 public:
  AWS_BACKUP_API UpdateRecoveryPointIndexSettingsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateRecoveryPointIndexSettings"; }

  AWS_BACKUP_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of a logical container where backups are stored. Backup vaults are
   * identified by names that are unique to the account used to create them and the
   * Region where they are created.</p> <p>Accepted characters include lowercase
   * letters, numbers, and hyphens.</p>
   */
  inline const Aws::String& GetBackupVaultName() const { return m_backupVaultName; }
  inline bool BackupVaultNameHasBeenSet() const { return m_backupVaultNameHasBeenSet; }
  template <typename BackupVaultNameT = Aws::String>
  void SetBackupVaultName(BackupVaultNameT&& value) {
    m_backupVaultNameHasBeenSet = true;
    m_backupVaultName = std::forward<BackupVaultNameT>(value);
  }
  template <typename BackupVaultNameT = Aws::String>
  UpdateRecoveryPointIndexSettingsRequest& WithBackupVaultName(BackupVaultNameT&& value) {
    SetBackupVaultName(std::forward<BackupVaultNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An ARN that uniquely identifies a recovery point; for example,
   * <code>arn:aws:backup:us-east-1:123456789012:recovery-point:1EB3B5E7-9EB0-435A-A80B-108B488B0D45</code>.</p>
   */
  inline const Aws::String& GetRecoveryPointArn() const { return m_recoveryPointArn; }
  inline bool RecoveryPointArnHasBeenSet() const { return m_recoveryPointArnHasBeenSet; }
  template <typename RecoveryPointArnT = Aws::String>
  void SetRecoveryPointArn(RecoveryPointArnT&& value) {
    m_recoveryPointArnHasBeenSet = true;
    m_recoveryPointArn = std::forward<RecoveryPointArnT>(value);
  }
  template <typename RecoveryPointArnT = Aws::String>
  UpdateRecoveryPointIndexSettingsRequest& WithRecoveryPointArn(RecoveryPointArnT&& value) {
    SetRecoveryPointArn(std::forward<RecoveryPointArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>This specifies the IAM role ARN used for this operation.</p> <p>For example,
   * arn:aws:iam::123456789012:role/S3Access</p>
   */
  inline const Aws::String& GetIamRoleArn() const { return m_iamRoleArn; }
  inline bool IamRoleArnHasBeenSet() const { return m_iamRoleArnHasBeenSet; }
  template <typename IamRoleArnT = Aws::String>
  void SetIamRoleArn(IamRoleArnT&& value) {
    m_iamRoleArnHasBeenSet = true;
    m_iamRoleArn = std::forward<IamRoleArnT>(value);
  }
  template <typename IamRoleArnT = Aws::String>
  UpdateRecoveryPointIndexSettingsRequest& WithIamRoleArn(IamRoleArnT&& value) {
    SetIamRoleArn(std::forward<IamRoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Index can have 1 of 2 possible values, either <code>ENABLED</code> or
   * <code>DISABLED</code>.</p> <p>To create a backup index for an eligible
   * <code>ACTIVE</code> recovery point that does not yet have a backup index, set
   * value to <code>ENABLED</code>.</p> <p>To delete a backup index, set value to
   * <code>DISABLED</code>.</p>
   */
  inline Index GetIndex() const { return m_index; }
  inline bool IndexHasBeenSet() const { return m_indexHasBeenSet; }
  inline void SetIndex(Index value) {
    m_indexHasBeenSet = true;
    m_index = value;
  }
  inline UpdateRecoveryPointIndexSettingsRequest& WithIndex(Index value) {
    SetIndex(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_backupVaultName;

  Aws::String m_recoveryPointArn;

  Aws::String m_iamRoleArn;

  Index m_index{Index::NOT_SET};
  bool m_backupVaultNameHasBeenSet = false;
  bool m_recoveryPointArnHasBeenSet = false;
  bool m_iamRoleArnHasBeenSet = false;
  bool m_indexHasBeenSet = false;
};

}  // namespace Model
}  // namespace Backup
}  // namespace Aws
