# frozen_string_literal: true

module GitlabQuality
  module TestTooling
    module GitlabClient
      class WorkItemsClient < GitlabGraphqlClient
        def work_item(workitem_iid:, widgets: [:notes, :linked_items, :labels, :hierarchy])
          query = <<~GQL
            query {
              namespace(fullPath: "#{group}") {
                workItem(iid: "#{workitem_iid}") {
                  #{work_item_fields}
                  #{work_item_widgets(widgets)}
                }
              }
            }
          GQL
          post(query)[:workItem]
        end

        def group_work_items(labels: [], cursor: '', state: 'opened', created_after: nil, extras: [:work_item_fields])
          query = <<~GQL
            query {
              group(fullPath: "#{group}") {
                workItems(after: "#{cursor}",#{created_after ? " createdAfter: \"#{created_after}\"," : ''} labelName: [#{labels.map { |label| "\"#{label}\"" }.join(', ')}], state: #{state}) {
                  nodes {
                    #{construct_extras(extras)}
                  }
                  pageInfo {
                    hasNextPage
                    endCursor
                  }
                }
              }
            }
          GQL
          begin
            post(query)[:workItems]
          rescue StandardError => e
            puts "Error: #{e}"
          end
        end

        def create_discussion(id:, note:)
          post(
            <<~GQL
              mutation CreateDiscussion {
                  createDiscussion(input: {noteableId: "#{id}", body: "#{note}"}) {
                      clientMutationId
                      errors
                      note {
                          #{note_fields}
                      }
                  }
              }
            GQL
          )
        end

        def create_discussion_note(work_item_id:, discussion_id:, text:)
          query = <<~GQL
            mutation CreateNote {
                createNote(input: { discussionId: "#{discussion_id}", noteableId: "#{work_item_id}", body: "#{text}" }) {
                    clientMutationId
                    errors
                    note {
                        #{note_fields}
                    }
                }
            }
          GQL
          post(query)
        end

        def update_note(note_id:, body:)
          query = <<~GQL
            mutation UpdateNote {
                updateNote(input: { body: "#{body}", id: "#{note_id}" }) {
                    clientMutationId
                    errors
                }
            }
          GQL
          post(query)
        end

        def create_linked_items(work_item_id:, item_ids:, link_type:)
          query = <<~GQL
            mutation WorkItemAddLinkedItems {
                workItemAddLinkedItems(
                    input: { id: "#{work_item_id}", workItemsIds: [#{item_ids.map { |id| "\"#{id}\"" }.join(', ')}], linkType: #{link_type} }
                ) {
                    clientMutationId
                }
            }
          GQL
          post(query)
        end

        def add_labels(work_item_id:, label_ids:)
          query =
            <<~GQL
              mutation WorkItemUpdate {
                  workItemUpdate(input: { id: "#{work_item_id}", labelsWidget: { addLabelIds: [#{label_ids.map { |id| "\"#{id}\"" }.join(', ')}] } }) {
                      clientMutationId
                      errors
                  }
              }
            GQL
          post(query)
        end

        def paginated_call(method_name, args)
          results = []

          # Check if the method exists
          raise ArgumentError, "Unknown method: #{method_name}" unless respond_to?(method_name, true)

          method_obj = method(method_name)

          loop do
            # Call the method directly using the method object
            response = method_obj.call(**args)

            break unless response

            results += response[:nodes]
            break unless response[:pageInfo][:hasNextPage]

            args[:cursor] = response[:pageInfo][:endCursor]
          end

          results
        end

        private

        attr_reader :group

        def construct_extras(extras)
          extras_string = ""
          extras.each do |extra|
            next unless respond_to?(extra, true)

            method_obj = method(extra)
            extras_string += method_obj.call
            extras_string += "\n"
          end
          extras_string
        end

        # https://docs.gitlab.com/api/graphql/reference/#workitem
        def work_item_fields
          <<~GQL
            author {
              username
            }
            createdAt
            iid
            id
            project {
              fullPath
              id
            }
            state
            title
            webUrl
            workItemType {
              name
            }
          GQL
        end

        def work_item_widget_notes
          <<~GQL
            ... on WorkItemWidgetNotes {
              discussions(filter: ONLY_COMMENTS) {
                nodes {
                        notes {
                          nodes {
                          #{note_fields}
                          }
                        }
                      }
              }
            }
          GQL
        end

        def work_item_widget_linked_items
          <<~GQL
            ... on WorkItemWidgetLinkedItems {
                linkedItems {
                    nodes {
                        linkType
                        workItem {
                            #{work_item_fields}
                        }
                    }
                }
            }
          GQL
        end

        def work_item_widget_labels
          <<~GQL
            ... on WorkItemWidgetLabels{
              labels{
                 nodes{
                     title
                 }
              }
            }
          GQL
        end

        def work_item_widget_hierarchy
          <<~GQL
            ... on WorkItemWidgetHierarchy {
              children {
                  nodes{
                   #{work_item_fields}
                  }
              }
            }
          GQL
        end

        def work_item_widgets(widgets = [])
          <<~GQL
            widgets(onlyTypes: [#{types_for_widgets(widgets)}]) {
              #{work_item_widget_notes if widgets.include?(:notes)}
              #{work_item_widget_linked_items if widgets.include?(:linked_items)}
              #{work_item_widget_labels if widgets.include?(:labels)}
              #{work_item_widget_hierarchy if widgets.include?(:hierarchy)}
            }
          GQL
        end

        def types_for_widgets(widgets = [])
          widgets.map(&:upcase).join(', ')
        end

        # https://docs.gitlab.com/api/graphql/reference/#note
        def note_fields
          <<~GQL
            author {
              username
            }
            awardEmoji {
              nodes {
              #{emoji_fields}
              }
            }
            body
            id
            url
            discussion {
              id
            }
          GQL
        end

        # https://docs.gitlab.com/api/graphql/reference/#awardemoji
        def emoji_fields
          <<~GQL
            emoji
            name
            user {
                username
            }
          GQL
        end
      end
    end
  end
end
