/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.bmc.objectstorage.transfer.internal.download;

import com.oracle.bmc.model.Range;
import com.oracle.bmc.objectstorage.requests.GetObjectRequest;
import com.oracle.bmc.objectstorage.transfer.DownloadManager;
import com.oracle.bmc.objectstorage.transfer.internal.download.DownloadThread;
import com.oracle.bmc.util.StreamUtils;
import jakarta.annotation.Nullable;
import java.beans.ConstructorProperties;
import java.io.IOException;
import java.io.InputStream;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MultithreadStream
extends InputStream {
    private static final Logger LOG = LoggerFactory.getLogger(MultithreadStream.class);
    private final DownloadManager downloadManager;
    private final GetObjectRequest baseRequest;
    private final long objectSize;
    private final InputStream firstPart;
    private final int numThreads;
    private final int partSize;
    private final ExecutorService executorService;
    private final boolean shutdownExecutorService;
    private final AsyncRead[] asyncReads;
    private int asyncReadIndex;
    private long nextReadOffset;
    private long bytesReadSoFar;
    private boolean isClosed;

    public MultithreadStream(DownloadManager downloadManager, GetObjectRequest baseRequest, long objectSize, InputStream firstPart, int numThreads, ExecutorService executorService, int partSize) {
        assert (partSize > 0);
        assert (numThreads > 0);
        assert (objectSize >= (long)partSize);
        this.downloadManager = downloadManager;
        this.baseRequest = baseRequest;
        this.objectSize = objectSize;
        this.numThreads = numThreads;
        this.partSize = partSize;
        this.firstPart = firstPart;
        this.bytesReadSoFar = 0L;
        this.asyncReads = new AsyncRead[this.numThreads];
        this.asyncReadIndex = 0;
        this.nextReadOffset = this.partSize;
        if (executorService == null) {
            this.executorService = Executors.newFixedThreadPool(this.numThreads);
            this.shutdownExecutorService = true;
        } else {
            this.executorService = executorService;
            this.shutdownExecutorService = false;
        }
        int numParts = Math.toIntExact((objectSize + (long)partSize - 1L) / (long)partSize);
        int readsToStart = Math.min(numThreads, numParts - 1);
        for (int i = 0; i < readsToStart; ++i) {
            this.asyncReads[i] = this.startAsyncRead(null);
        }
    }

    @Override
    public synchronized void close() throws IOException {
        for (int i = 0; i < this.asyncReads.length; ++i) {
            if (this.asyncReads[i] == null) continue;
            this.asyncReads[i].thread.requestCancel();
        }
        StreamUtils.closeQuietly((InputStream)this.firstPart);
        long deadline = 30000L;
        long stopMillis = System.currentTimeMillis() + 30000L;
        for (int i = 0; i < this.asyncReads.length; ++i) {
            if (this.asyncReads[i] == null) continue;
            long millisToWait = Math.max(0L, stopMillis - System.currentTimeMillis());
            try {
                this.asyncReads[i].future.get(millisToWait, TimeUnit.MILLISECONDS);
                continue;
            }
            catch (InterruptedException | ExecutionException | TimeoutException e) {
                LOG.warn("Ignoring exception from async read", (Throwable)e);
            }
        }
        if (this.shutdownExecutorService) {
            try {
                this.executorService.shutdownNow();
                this.executorService.awaitTermination(0L, TimeUnit.MILLISECONDS);
            }
            catch (InterruptedException e) {
                LOG.warn("Ignoring exception from executor service termination", (Throwable)e);
            }
        }
        this.isClosed = true;
    }

    @Override
    public int read() throws IOException {
        byte[] b = new byte[1];
        int r = this.read(b, 0, 1);
        if (r < 0) {
            return -1;
        }
        return Byte.toUnsignedInt(b[0]);
    }

    @Override
    public synchronized int read(byte[] b, int off, int len) throws IOException {
        if (len == 0) {
            return 0;
        }
        if (this.allDataRead()) {
            return -1;
        }
        if (this.isClosed) {
            throw new IOException("Stream has been closed");
        }
        if (this.bytesReadSoFar < (long)this.partSize) {
            int maxBytesToRead = Math.min(len, Math.toIntExact((long)this.partSize - this.bytesReadSoFar));
            int bytesRead = this.firstPart.read(b, off, maxBytesToRead);
            if (bytesRead < 0) {
                LOG.error("Truncated download. Got {} from read (expected {} bytes remaining)", (Object)bytesRead, (Object)maxBytesToRead);
                throw new IOException("Truncated read");
            }
            assert (bytesRead > 0);
            this.bytesReadSoFar += (long)bytesRead;
            assert (this.bytesReadSoFar <= (long)this.partSize);
            if (this.bytesReadSoFar >= (long)this.partSize) {
                MultithreadStream.closeQuietly(this.firstPart);
            }
            LOG.trace("Read {} bytes from first part ({} total, object is {} bytes)", new Object[]{bytesRead, this.asyncReadIndex, this.bytesReadSoFar, this.objectSize});
            return bytesRead;
        }
        LOG.trace("Reading from thread {}", (Object)this.asyncReadIndex);
        AsyncRead asyncRead = this.asyncReads[this.asyncReadIndex];
        int bytesRead = asyncRead.thread.read(b, off, len);
        if (bytesRead < 0) {
            LOG.error("Truncated download. Got {} from read", (Object)bytesRead);
            throw new IOException("Truncated read");
        }
        this.bytesReadSoFar += (long)bytesRead;
        LOG.trace("Read {} bytes from thread {} ({} total, object is {} bytes)", new Object[]{bytesRead, this.asyncReadIndex, this.bytesReadSoFar, this.objectSize});
        if (asyncRead.thread.allDataRead()) {
            this.asyncReads[this.asyncReadIndex] = null;
            try {
                byte[] buffer = this.joinAsyncRead(asyncRead);
                assert (buffer != null);
                if (this.allDataRead()) {
                    this.shutdownExecutorService();
                } else {
                    if (!this.allReadsStarted()) {
                        this.asyncReads[this.asyncReadIndex] = this.startAsyncRead(buffer);
                        assert (this.asyncReads[this.asyncReadIndex] != null);
                    }
                    this.advanceAsyncReadIndex();
                    assert (this.asyncReads[this.asyncReadIndex] != null);
                }
            }
            catch (InterruptedException | ExecutionException e) {
                throw new IOException("Unable to start AsyncRead", e);
            }
        }
        return bytesRead;
    }

    private byte[] joinAsyncRead(AsyncRead asyncRead) throws ExecutionException, InterruptedException {
        assert (asyncRead.thread.allDataRead());
        return (byte[])asyncRead.future.get();
    }

    private boolean allReadsStarted() {
        return this.nextReadOffset >= this.objectSize;
    }

    private boolean allDataRead() {
        return this.bytesReadSoFar >= this.objectSize;
    }

    private void advanceAsyncReadIndex() {
        assert (this.asyncReadIndex >= 0);
        assert (this.asyncReadIndex < this.asyncReads.length);
        ++this.asyncReadIndex;
        if (this.asyncReadIndex >= this.asyncReads.length) {
            this.asyncReadIndex = 0;
        }
        assert (this.asyncReadIndex >= 0);
        assert (this.asyncReadIndex < this.asyncReads.length);
    }

    private void shutdownExecutorService() throws InterruptedException {
        for (AsyncRead asyncRead : this.asyncReads) {
            assert (asyncRead == null);
        }
        if (this.shutdownExecutorService) {
            this.executorService.shutdownNow();
            this.executorService.awaitTermination(1L, TimeUnit.SECONDS);
        }
    }

    private AsyncRead startAsyncRead(@Nullable byte[] buffer) {
        boolean isEndOnlyRange = this.baseRequest.getRange() != null && this.baseRequest.getRange().getStartByte() == null && this.baseRequest.getRange().getEndByte() != null;
        RangeWrapper rangeWrapper = isEndOnlyRange ? this.endOnlyRange() : this.notEndOnlyRange();
        int rangeSize = Math.toIntExact(rangeWrapper.getRangeSize());
        Range range = rangeWrapper.getRange();
        GetObjectRequest getObjectRequest = GetObjectRequest.builder().copy(this.baseRequest).range(range).build();
        if (buffer == null) {
            buffer = new byte[Math.toIntExact(rangeSize)];
        } else assert (buffer.length >= Math.toIntExact(rangeSize));
        LOG.debug("Starting async read of {}/{}/{} from {}-{} ({})", new Object[]{getObjectRequest.getNamespaceName(), getObjectRequest.getBucketName(), getObjectRequest.getObjectName(), getObjectRequest.getRange().getStartByte(), getObjectRequest.getRange().getEndByte(), this.asyncReadIndex});
        DownloadThread thread = new DownloadThread(this.downloadManager, getObjectRequest, buffer, rangeSize);
        Future<byte[]> future = this.executorService.submit(() -> thread.run());
        AsyncRead asyncRead = new AsyncRead(future, thread);
        this.nextReadOffset += (long)rangeSize;
        return asyncRead;
    }

    private static void closeQuietly(InputStream s) {
        try {
            s.close();
        }
        catch (Throwable t) {
            LOG.trace("Ignoring error from InputStream.close", t);
        }
    }

    public RangeWrapper notEndOnlyRange() {
        long baseRequestRangeStart = this.baseRequest.getRange() != null && this.baseRequest.getRange().getStartByte() != null ? this.baseRequest.getRange().getStartByte() : 0L;
        long baseRequestRangeEnd = this.baseRequest.getRange() != null && this.baseRequest.getRange().getEndByte() != null ? this.baseRequest.getRange().getEndByte() : Long.MAX_VALUE;
        assert (baseRequestRangeStart < baseRequestRangeEnd);
        long rangeStart = baseRequestRangeStart + this.nextReadOffset;
        long rangeEnd = rangeStart + (long)this.partSize - 1L;
        rangeEnd = Math.min(rangeEnd, baseRequestRangeEnd);
        rangeEnd = Math.min(rangeEnd, baseRequestRangeStart + this.objectSize - 1L);
        assert (rangeEnd >= rangeStart);
        long rangeSize = rangeEnd - rangeStart + 1L;
        assert (rangeSize > 0L);
        Range range = new Range(Long.valueOf(rangeStart), Long.valueOf(rangeEnd));
        return new RangeWrapper(rangeSize, range);
    }

    public RangeWrapper endOnlyRange() {
        long baseRequestRangeEnd = this.baseRequest.getRange().getEndByte();
        if (baseRequestRangeEnd > this.objectSize) {
            baseRequestRangeEnd = this.objectSize;
        }
        long rangeEnd = baseRequestRangeEnd - this.nextReadOffset;
        rangeEnd = Math.max(rangeEnd, 0L);
        long rangeSize = Math.min(rangeEnd, (long)this.partSize);
        assert (rangeSize > 0L);
        Range range = new Range(null, Long.valueOf(rangeEnd));
        return new RangeWrapper(rangeSize, range);
    }

    private final class RangeWrapper {
        private final long rangeSize;
        private final Range range;

        @ConstructorProperties(value={"rangeSize", "range"})
        public RangeWrapper(long rangeSize, Range range) {
            this.rangeSize = rangeSize;
            this.range = range;
        }

        public long getRangeSize() {
            return this.rangeSize;
        }

        public Range getRange() {
            return this.range;
        }

        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (!(o instanceof RangeWrapper)) {
                return false;
            }
            RangeWrapper other = (RangeWrapper)o;
            if (this.getRangeSize() != other.getRangeSize()) {
                return false;
            }
            Range this$range = this.getRange();
            Range other$range = other.getRange();
            return !(this$range == null ? other$range != null : !this$range.equals(other$range));
        }

        public int hashCode() {
            int PRIME = 59;
            int result = 1;
            long $rangeSize = this.getRangeSize();
            result = result * 59 + (int)($rangeSize >>> 32 ^ $rangeSize);
            Range $range = this.getRange();
            result = result * 59 + ($range == null ? 43 : $range.hashCode());
            return result;
        }

        public String toString() {
            return "MultithreadStream.RangeWrapper(rangeSize=" + this.getRangeSize() + ", range=" + this.getRange() + ")";
        }
    }

    private static final class AsyncRead {
        private final Future<byte[]> future;
        private final DownloadThread thread;

        public AsyncRead(Future<byte[]> future, DownloadThread thread) {
            this.future = future;
            this.thread = thread;
        }
    }
}

