﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/storagegateway/StorageGatewayRequest.h>
#include <aws/storagegateway/StorageGateway_EXPORTS.h>
#include <aws/storagegateway/model/CacheAttributes.h>
#include <aws/storagegateway/model/EncryptionType.h>
#include <aws/storagegateway/model/NFSFileShareDefaults.h>
#include <aws/storagegateway/model/ObjectACL.h>

#include <utility>

namespace Aws {
namespace StorageGateway {
namespace Model {

/**
 * <p>UpdateNFSFileShareInput</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/storagegateway-2013-06-30/UpdateNFSFileShareInput">AWS
 * API Reference</a></p>
 */
class UpdateNFSFileShareRequest : public StorageGatewayRequest {
 public:
  AWS_STORAGEGATEWAY_API UpdateNFSFileShareRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateNFSFileShare"; }

  AWS_STORAGEGATEWAY_API Aws::String SerializePayload() const override;

  AWS_STORAGEGATEWAY_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the file share to be updated.</p>
   */
  inline const Aws::String& GetFileShareARN() const { return m_fileShareARN; }
  inline bool FileShareARNHasBeenSet() const { return m_fileShareARNHasBeenSet; }
  template <typename FileShareARNT = Aws::String>
  void SetFileShareARN(FileShareARNT&& value) {
    m_fileShareARNHasBeenSet = true;
    m_fileShareARN = std::forward<FileShareARNT>(value);
  }
  template <typename FileShareARNT = Aws::String>
  UpdateNFSFileShareRequest& WithFileShareARN(FileShareARNT&& value) {
    SetFileShareARN(std::forward<FileShareARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A value that specifies the type of server-side encryption that the file share
   * will use for the data that it stores in Amazon S3.</p>  <p>We recommend
   * using <code>EncryptionType</code> instead of <code>KMSEncrypted</code> to set
   * the file share encryption method. You do not need to provide values for both
   * parameters.</p> <p>If values for both parameters exist in the same request, then
   * the specified encryption methods must not conflict. For example, if
   * <code>EncryptionType</code> is <code>SseS3</code>, then
   * <code>KMSEncrypted</code> must be <code>false</code>. If
   * <code>EncryptionType</code> is <code>SseKms</code> or <code>DsseKms</code>, then
   * <code>KMSEncrypted</code> must be <code>true</code>.</p>
   */
  inline EncryptionType GetEncryptionType() const { return m_encryptionType; }
  inline bool EncryptionTypeHasBeenSet() const { return m_encryptionTypeHasBeenSet; }
  inline void SetEncryptionType(EncryptionType value) {
    m_encryptionTypeHasBeenSet = true;
    m_encryptionType = value;
  }
  inline UpdateNFSFileShareRequest& WithEncryptionType(EncryptionType value) {
    SetEncryptionType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Optional. The Amazon Resource Name (ARN) of a symmetric customer master key
   * (CMK) used for Amazon S3 server-side encryption. Storage Gateway does not
   * support asymmetric CMKs. This value must be set if <code>KMSEncrypted</code> is
   * <code>true</code>, or if <code>EncryptionType</code> is <code>SseKms</code> or
   * <code>DsseKms</code>.</p>
   */
  inline const Aws::String& GetKMSKey() const { return m_kMSKey; }
  inline bool KMSKeyHasBeenSet() const { return m_kMSKeyHasBeenSet; }
  template <typename KMSKeyT = Aws::String>
  void SetKMSKey(KMSKeyT&& value) {
    m_kMSKeyHasBeenSet = true;
    m_kMSKey = std::forward<KMSKeyT>(value);
  }
  template <typename KMSKeyT = Aws::String>
  UpdateNFSFileShareRequest& WithKMSKey(KMSKeyT&& value) {
    SetKMSKey(std::forward<KMSKeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The default values for the file share. Optional.</p>
   */
  inline const NFSFileShareDefaults& GetNFSFileShareDefaults() const { return m_nFSFileShareDefaults; }
  inline bool NFSFileShareDefaultsHasBeenSet() const { return m_nFSFileShareDefaultsHasBeenSet; }
  template <typename NFSFileShareDefaultsT = NFSFileShareDefaults>
  void SetNFSFileShareDefaults(NFSFileShareDefaultsT&& value) {
    m_nFSFileShareDefaultsHasBeenSet = true;
    m_nFSFileShareDefaults = std::forward<NFSFileShareDefaultsT>(value);
  }
  template <typename NFSFileShareDefaultsT = NFSFileShareDefaults>
  UpdateNFSFileShareRequest& WithNFSFileShareDefaults(NFSFileShareDefaultsT&& value) {
    SetNFSFileShareDefaults(std::forward<NFSFileShareDefaultsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The default storage class for objects put into an Amazon S3 bucket by the S3
   * File Gateway. The default value is <code>S3_STANDARD</code>. Optional.</p>
   * <p>Valid Values: <code>S3_STANDARD</code> | <code>S3_INTELLIGENT_TIERING</code>
   * | <code>S3_STANDARD_IA</code> | <code>S3_ONEZONE_IA</code> </p>
   */
  inline const Aws::String& GetDefaultStorageClass() const { return m_defaultStorageClass; }
  inline bool DefaultStorageClassHasBeenSet() const { return m_defaultStorageClassHasBeenSet; }
  template <typename DefaultStorageClassT = Aws::String>
  void SetDefaultStorageClass(DefaultStorageClassT&& value) {
    m_defaultStorageClassHasBeenSet = true;
    m_defaultStorageClass = std::forward<DefaultStorageClassT>(value);
  }
  template <typename DefaultStorageClassT = Aws::String>
  UpdateNFSFileShareRequest& WithDefaultStorageClass(DefaultStorageClassT&& value) {
    SetDefaultStorageClass(std::forward<DefaultStorageClassT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A value that sets the access control list (ACL) permission for objects in the
   * S3 bucket that a S3 File Gateway puts objects into. The default value is
   * <code>private</code>.</p>
   */
  inline ObjectACL GetObjectACL() const { return m_objectACL; }
  inline bool ObjectACLHasBeenSet() const { return m_objectACLHasBeenSet; }
  inline void SetObjectACL(ObjectACL value) {
    m_objectACLHasBeenSet = true;
    m_objectACL = value;
  }
  inline UpdateNFSFileShareRequest& WithObjectACL(ObjectACL value) {
    SetObjectACL(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The list of clients that are allowed to access the S3 File Gateway. The list
   * must contain either valid IPv4/IPv6 addresses or valid CIDR blocks.</p>
   */
  inline const Aws::Vector<Aws::String>& GetClientList() const { return m_clientList; }
  inline bool ClientListHasBeenSet() const { return m_clientListHasBeenSet; }
  template <typename ClientListT = Aws::Vector<Aws::String>>
  void SetClientList(ClientListT&& value) {
    m_clientListHasBeenSet = true;
    m_clientList = std::forward<ClientListT>(value);
  }
  template <typename ClientListT = Aws::Vector<Aws::String>>
  UpdateNFSFileShareRequest& WithClientList(ClientListT&& value) {
    SetClientList(std::forward<ClientListT>(value));
    return *this;
  }
  template <typename ClientListT = Aws::String>
  UpdateNFSFileShareRequest& AddClientList(ClientListT&& value) {
    m_clientListHasBeenSet = true;
    m_clientList.emplace_back(std::forward<ClientListT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The user mapped to anonymous user.</p> <p>Valid values are the following:</p>
   * <ul> <li> <p> <code>RootSquash</code>: Only root is mapped to anonymous
   * user.</p> </li> <li> <p> <code>NoSquash</code>: No one is mapped to anonymous
   * user.</p> </li> <li> <p> <code>AllSquash</code>: Everyone is mapped to anonymous
   * user.</p> </li> </ul>
   */
  inline const Aws::String& GetSquash() const { return m_squash; }
  inline bool SquashHasBeenSet() const { return m_squashHasBeenSet; }
  template <typename SquashT = Aws::String>
  void SetSquash(SquashT&& value) {
    m_squashHasBeenSet = true;
    m_squash = std::forward<SquashT>(value);
  }
  template <typename SquashT = Aws::String>
  UpdateNFSFileShareRequest& WithSquash(SquashT&& value) {
    SetSquash(std::forward<SquashT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A value that sets the write status of a file share. Set this value to
   * <code>true</code> to set the write status to read-only, otherwise set to
   * <code>false</code>.</p> <p>Valid Values: <code>true</code> | <code>false</code>
   * </p>
   */
  inline bool GetReadOnly() const { return m_readOnly; }
  inline bool ReadOnlyHasBeenSet() const { return m_readOnlyHasBeenSet; }
  inline void SetReadOnly(bool value) {
    m_readOnlyHasBeenSet = true;
    m_readOnly = value;
  }
  inline UpdateNFSFileShareRequest& WithReadOnly(bool value) {
    SetReadOnly(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A value that enables guessing of the MIME type for uploaded objects based on
   * file extensions. Set this value to <code>true</code> to enable MIME type
   * guessing, otherwise set to <code>false</code>. The default value is
   * <code>true</code>.</p> <p>Valid Values: <code>true</code> | <code>false</code>
   * </p>
   */
  inline bool GetGuessMIMETypeEnabled() const { return m_guessMIMETypeEnabled; }
  inline bool GuessMIMETypeEnabledHasBeenSet() const { return m_guessMIMETypeEnabledHasBeenSet; }
  inline void SetGuessMIMETypeEnabled(bool value) {
    m_guessMIMETypeEnabledHasBeenSet = true;
    m_guessMIMETypeEnabled = value;
  }
  inline UpdateNFSFileShareRequest& WithGuessMIMETypeEnabled(bool value) {
    SetGuessMIMETypeEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A value that sets who pays the cost of the request and the cost associated
   * with data download from the S3 bucket. If this value is set to
   * <code>true</code>, the requester pays the costs; otherwise, the S3 bucket owner
   * pays. However, the S3 bucket owner always pays the cost of storing data.</p>
   *  <p> <code>RequesterPays</code> is a configuration for the S3 bucket that
   * backs the file share, so make sure that the configuration on the file share is
   * the same as the S3 bucket configuration.</p>  <p>Valid Values:
   * <code>true</code> | <code>false</code> </p>
   */
  inline bool GetRequesterPays() const { return m_requesterPays; }
  inline bool RequesterPaysHasBeenSet() const { return m_requesterPaysHasBeenSet; }
  inline void SetRequesterPays(bool value) {
    m_requesterPaysHasBeenSet = true;
    m_requesterPays = value;
  }
  inline UpdateNFSFileShareRequest& WithRequesterPays(bool value) {
    SetRequesterPays(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the file share. Optional.</p>  <p>
   * <code>FileShareName</code> must be set if an S3 prefix name is set in
   * <code>LocationARN</code>, or if an access point or access point alias is
   * used.</p> <p>A valid NFS file share name can only contain the following
   * characters: <code>a</code>-<code>z</code>, <code>A</code>-<code>Z</code>,
   * <code>0</code>-<code>9</code>, <code>-</code>, <code>.</code>, and
   * <code>_</code>.</p>
   */
  inline const Aws::String& GetFileShareName() const { return m_fileShareName; }
  inline bool FileShareNameHasBeenSet() const { return m_fileShareNameHasBeenSet; }
  template <typename FileShareNameT = Aws::String>
  void SetFileShareName(FileShareNameT&& value) {
    m_fileShareNameHasBeenSet = true;
    m_fileShareName = std::forward<FileShareNameT>(value);
  }
  template <typename FileShareNameT = Aws::String>
  UpdateNFSFileShareRequest& WithFileShareName(FileShareNameT&& value) {
    SetFileShareName(std::forward<FileShareNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies refresh cache information for the file share.</p>
   */
  inline const CacheAttributes& GetCacheAttributes() const { return m_cacheAttributes; }
  inline bool CacheAttributesHasBeenSet() const { return m_cacheAttributesHasBeenSet; }
  template <typename CacheAttributesT = CacheAttributes>
  void SetCacheAttributes(CacheAttributesT&& value) {
    m_cacheAttributesHasBeenSet = true;
    m_cacheAttributes = std::forward<CacheAttributesT>(value);
  }
  template <typename CacheAttributesT = CacheAttributes>
  UpdateNFSFileShareRequest& WithCacheAttributes(CacheAttributesT&& value) {
    SetCacheAttributes(std::forward<CacheAttributesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The notification policy of the file share. <code>SettlingTimeInSeconds</code>
   * controls the number of seconds to wait after the last point in time a client
   * wrote to a file before generating an <code>ObjectUploaded</code> notification.
   * Because clients can make many small writes to files, it's best to set this
   * parameter for as long as possible to avoid generating multiple notifications for
   * the same file in a small time period.</p>  <p>
   * <code>SettlingTimeInSeconds</code> has no effect on the timing of the object
   * uploading to Amazon S3, only the timing of the notification.</p> <p>This setting
   * is not meant to specify an exact time at which the notification will be sent. In
   * some cases, the gateway might require more than the specified delay time to
   * generate and send notifications.</p>  <p>The following example sets
   * <code>NotificationPolicy</code> on with <code>SettlingTimeInSeconds</code> set
   * to 60.</p> <p> <code>{\"Upload\": {\"SettlingTimeInSeconds\": 60}}</code> </p>
   * <p>The following example sets <code>NotificationPolicy</code> off.</p> <p>
   * <code>{}</code> </p>
   */
  inline const Aws::String& GetNotificationPolicy() const { return m_notificationPolicy; }
  inline bool NotificationPolicyHasBeenSet() const { return m_notificationPolicyHasBeenSet; }
  template <typename NotificationPolicyT = Aws::String>
  void SetNotificationPolicy(NotificationPolicyT&& value) {
    m_notificationPolicyHasBeenSet = true;
    m_notificationPolicy = std::forward<NotificationPolicyT>(value);
  }
  template <typename NotificationPolicyT = Aws::String>
  UpdateNFSFileShareRequest& WithNotificationPolicy(NotificationPolicyT&& value) {
    SetNotificationPolicy(std::forward<NotificationPolicyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the storage used for audit logs.</p>
   */
  inline const Aws::String& GetAuditDestinationARN() const { return m_auditDestinationARN; }
  inline bool AuditDestinationARNHasBeenSet() const { return m_auditDestinationARNHasBeenSet; }
  template <typename AuditDestinationARNT = Aws::String>
  void SetAuditDestinationARN(AuditDestinationARNT&& value) {
    m_auditDestinationARNHasBeenSet = true;
    m_auditDestinationARN = std::forward<AuditDestinationARNT>(value);
  }
  template <typename AuditDestinationARNT = Aws::String>
  UpdateNFSFileShareRequest& WithAuditDestinationARN(AuditDestinationARNT&& value) {
    SetAuditDestinationARN(std::forward<AuditDestinationARNT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_fileShareARN;

  EncryptionType m_encryptionType{EncryptionType::NOT_SET};

  Aws::String m_kMSKey;

  NFSFileShareDefaults m_nFSFileShareDefaults;

  Aws::String m_defaultStorageClass;

  ObjectACL m_objectACL{ObjectACL::NOT_SET};

  Aws::Vector<Aws::String> m_clientList;

  Aws::String m_squash;

  bool m_readOnly{false};

  bool m_guessMIMETypeEnabled{false};

  bool m_requesterPays{false};

  Aws::String m_fileShareName;

  CacheAttributes m_cacheAttributes;

  Aws::String m_notificationPolicy;

  Aws::String m_auditDestinationARN;
  bool m_fileShareARNHasBeenSet = false;
  bool m_encryptionTypeHasBeenSet = false;
  bool m_kMSKeyHasBeenSet = false;
  bool m_nFSFileShareDefaultsHasBeenSet = false;
  bool m_defaultStorageClassHasBeenSet = false;
  bool m_objectACLHasBeenSet = false;
  bool m_clientListHasBeenSet = false;
  bool m_squashHasBeenSet = false;
  bool m_readOnlyHasBeenSet = false;
  bool m_guessMIMETypeEnabledHasBeenSet = false;
  bool m_requesterPaysHasBeenSet = false;
  bool m_fileShareNameHasBeenSet = false;
  bool m_cacheAttributesHasBeenSet = false;
  bool m_notificationPolicyHasBeenSet = false;
  bool m_auditDestinationARNHasBeenSet = false;
};

}  // namespace Model
}  // namespace StorageGateway
}  // namespace Aws
