﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/acm-pca/ACMPCARequest.h>
#include <aws/acm-pca/ACMPCA_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace ACMPCA {
namespace Model {

/**
 */
class DeletePermissionRequest : public ACMPCARequest {
 public:
  AWS_ACMPCA_API DeletePermissionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeletePermission"; }

  AWS_ACMPCA_API Aws::String SerializePayload() const override;

  AWS_ACMPCA_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Number (ARN) of the private CA that issued the
   * permissions. You can find the CA's ARN by calling the <a
   * href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_ListCertificateAuthorities.html">ListCertificateAuthorities</a>
   * action. This must have the following form: </p> <p>
   * <code>arn:aws:acm-pca:<i>region</i>:<i>account</i>:certificate-authority/<i>12345678-1234-1234-1234-123456789012</i>
   * </code>. </p>
   */
  inline const Aws::String& GetCertificateAuthorityArn() const { return m_certificateAuthorityArn; }
  inline bool CertificateAuthorityArnHasBeenSet() const { return m_certificateAuthorityArnHasBeenSet; }
  template <typename CertificateAuthorityArnT = Aws::String>
  void SetCertificateAuthorityArn(CertificateAuthorityArnT&& value) {
    m_certificateAuthorityArnHasBeenSet = true;
    m_certificateAuthorityArn = std::forward<CertificateAuthorityArnT>(value);
  }
  template <typename CertificateAuthorityArnT = Aws::String>
  DeletePermissionRequest& WithCertificateAuthorityArn(CertificateAuthorityArnT&& value) {
    SetCertificateAuthorityArn(std::forward<CertificateAuthorityArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Web Services service or identity that will have its CA permissions
   * revoked. At this time, the only valid service principal is
   * <code>acm.amazonaws.com</code> </p>
   */
  inline const Aws::String& GetPrincipal() const { return m_principal; }
  inline bool PrincipalHasBeenSet() const { return m_principalHasBeenSet; }
  template <typename PrincipalT = Aws::String>
  void SetPrincipal(PrincipalT&& value) {
    m_principalHasBeenSet = true;
    m_principal = std::forward<PrincipalT>(value);
  }
  template <typename PrincipalT = Aws::String>
  DeletePermissionRequest& WithPrincipal(PrincipalT&& value) {
    SetPrincipal(std::forward<PrincipalT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Web Services account that calls this action.</p>
   */
  inline const Aws::String& GetSourceAccount() const { return m_sourceAccount; }
  inline bool SourceAccountHasBeenSet() const { return m_sourceAccountHasBeenSet; }
  template <typename SourceAccountT = Aws::String>
  void SetSourceAccount(SourceAccountT&& value) {
    m_sourceAccountHasBeenSet = true;
    m_sourceAccount = std::forward<SourceAccountT>(value);
  }
  template <typename SourceAccountT = Aws::String>
  DeletePermissionRequest& WithSourceAccount(SourceAccountT&& value) {
    SetSourceAccount(std::forward<SourceAccountT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_certificateAuthorityArn;

  Aws::String m_principal;

  Aws::String m_sourceAccount;
  bool m_certificateAuthorityArnHasBeenSet = false;
  bool m_principalHasBeenSet = false;
  bool m_sourceAccountHasBeenSet = false;
};

}  // namespace Model
}  // namespace ACMPCA
}  // namespace Aws
