/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ghidra.program.model.pcode;

import static ghidra.program.model.pcode.AttributeId.*;
import static ghidra.program.model.pcode.ElementId.*;

import java.io.IOException;

import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressSpace;
import ghidra.program.model.data.MutabilitySettingsDefinition;
import ghidra.program.model.listing.Program;
import ghidra.program.model.listing.VariableStorage;
import ghidra.util.exception.AssertException;
import ghidra.util.exception.InvalidInputException;

/**
 * A HighSymbol mapping based on local hashing of the symbol's Varnode within a
 * function's syntax tree.  The storage address of a temporary Varnode (a Varnode in
 * the "unique" address space) is too ephemeral to use as a permanent way to identify it.
 * This symbol stores a hash (generated by DynamicHash) that is better suited to
 * identifying the Varnode.
 */
public class DynamicEntry extends SymbolEntry {
	private long hash;			// Hash encoding the specific Varnode

	/**
	 * Constructor for use with restoreXML
	 * @param sym is the owning HighSymbol
	 */
	public DynamicEntry(HighSymbol sym) {
		super(sym);
	}

	/**
	 * Construct given the underlying symbol, defining Address of the Varnode, and the hash value
	 * @param sym is the given symbol
	 * @param addr is the defining Address
	 * @param h is the hash value
	 */
	public DynamicEntry(HighSymbol sym, Address addr, long h) {
		super(sym);
		pcaddr = addr;
		hash = h;
	}

	/**
	 * @return the hash value
	 */
	public long getHash() {
		return hash;
	}

	@Override
	public void decode(Decoder decoder) throws DecoderException {
		int addrel = decoder.openElement(ELEM_HASH);
		hash = decoder.readUnsignedInteger(ATTRIB_VAL);
		decoder.closeElement(addrel);
		decodeRangeList(decoder);
	}

	@Override
	public void encode(Encoder encoder) throws IOException {
		encoder.openElement(ELEM_HASH);
		encoder.writeUnsignedInteger(ATTRIB_VAL, hash);
		encoder.closeElement(ELEM_HASH);
		encodeRangelist(encoder);
	}

	@Override
	public VariableStorage getStorage() {
		Program program = symbol.getProgram();
		try {
			return new VariableStorage(program, AddressSpace.HASH_SPACE.getAddress(getHash()),
				getSize());
		}
		catch (InvalidInputException e) {
			throw new AssertException("Unexpected exception", e);
		}
	}

	/**
	 * Build a new DynamicEntry, given the underlying temporary
	 * Varnode attached to a symbol.  The hash is created from local information in the
	 * syntax tree near the Varnode.
	 * @param vn is the underlying Varnode
	 * @return the new DynamicEntry
	 */
	public static DynamicEntry build(Varnode vn) {
		HighVariable highVariable = vn.getHigh();
		HighSymbol highSymbol = highVariable.getSymbol();
		HighFunction highFunction = highSymbol.getHighFunction();
		DynamicHash dynamicHash = new DynamicHash(vn, highFunction);
		DynamicEntry entry =
			new DynamicEntry(highSymbol, dynamicHash.getAddress(), dynamicHash.getHash());
		return entry;
	}

	@Override
	public int getSize() {
		return symbol.type.getLength();
	}

	@Override
	public int getMutability() {
		return MutabilitySettingsDefinition.NORMAL;
	}

}
