/* -*- mode: C++; c-basic-offset: 4; indent-tabs-mode: nil -*- */
// vim: ft=cpp:expandtab:ts=8:sw=4:softtabstop=4:
#ident "$Id$"
/*======
This file is part of TokuDB


Copyright (c) 2006, 2015, Percona and/or its affiliates. All rights reserved.

    TokuDBis is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License, version 2,
    as published by the Free Software Foundation.

    TokuDB is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with TokuDB.  If not, see <http://www.gnu.org/licenses/>.

======= */

#ident "Copyright (c) 2006, 2015, Percona and/or its affiliates. All rights reserved."

#ifndef _HATOKU_HTON_H
#define _HATOKU_HTON_H

#include "db.h"

extern handlerton *tokudb_hton;

extern DB_ENV *db_env;

enum srv_row_format_enum {
    SRV_ROW_FORMAT_UNCOMPRESSED = 0,
    SRV_ROW_FORMAT_ZLIB = 1,
    SRV_ROW_FORMAT_SNAPPY = 2,
    SRV_ROW_FORMAT_QUICKLZ = 3,
    SRV_ROW_FORMAT_LZMA = 4,
    SRV_ROW_FORMAT_FAST = 5,
    SRV_ROW_FORMAT_SMALL = 6,
    SRV_ROW_FORMAT_DEFAULT = 7
};
typedef enum srv_row_format_enum srv_row_format_t;

static inline srv_row_format_t toku_compression_method_to_row_format(toku_compression_method method) {
    switch (method) {
    case TOKU_NO_COMPRESSION:
        return SRV_ROW_FORMAT_UNCOMPRESSED;        
    case TOKU_ZLIB_WITHOUT_CHECKSUM_METHOD:
    case TOKU_ZLIB_METHOD:
        return SRV_ROW_FORMAT_ZLIB;
    case TOKU_SNAPPY_METHOD:
        return SRV_ROW_FORMAT_SNAPPY;
    case TOKU_QUICKLZ_METHOD:
        return SRV_ROW_FORMAT_QUICKLZ;
    case TOKU_LZMA_METHOD:
        return SRV_ROW_FORMAT_LZMA;
    case TOKU_DEFAULT_COMPRESSION_METHOD:
        return SRV_ROW_FORMAT_DEFAULT;
    case TOKU_FAST_COMPRESSION_METHOD:
        return SRV_ROW_FORMAT_FAST;
    case TOKU_SMALL_COMPRESSION_METHOD:
        return SRV_ROW_FORMAT_SMALL;
    default:
        assert(0);
    }
}

static inline toku_compression_method row_format_to_toku_compression_method(srv_row_format_t row_format) {
    switch (row_format) {
    case SRV_ROW_FORMAT_UNCOMPRESSED:
        return TOKU_NO_COMPRESSION;
    case SRV_ROW_FORMAT_QUICKLZ:
    case SRV_ROW_FORMAT_FAST:
        return TOKU_QUICKLZ_METHOD;
    case SRV_ROW_FORMAT_SNAPPY:
        return TOKU_SNAPPY_METHOD;
    case SRV_ROW_FORMAT_ZLIB:
    case SRV_ROW_FORMAT_DEFAULT:
        return TOKU_ZLIB_WITHOUT_CHECKSUM_METHOD;
    case SRV_ROW_FORMAT_LZMA:
    case SRV_ROW_FORMAT_SMALL:
        return TOKU_LZMA_METHOD;
    default:
        assert(0);
    }
}

static inline enum row_type row_format_to_row_type(srv_row_format_t row_format) {
#if TOKU_INCLUDE_ROW_TYPE_COMPRESSION
    switch (row_format) {
    case SRV_ROW_FORMAT_UNCOMPRESSED:
        return ROW_TYPE_TOKU_UNCOMPRESSED;
    case SRV_ROW_FORMAT_ZLIB:
        return ROW_TYPE_TOKU_ZLIB;
    case SRV_ROW_FORMAT_SNAPPY:
        return ROW_TYPE_TOKU_SNAPPY;
    case SRV_ROW_FORMAT_QUICKLZ:
        return ROW_TYPE_TOKU_QUICKLZ;
    case SRV_ROW_FORMAT_LZMA:
        return ROW_TYPE_TOKU_LZMA;
    case SRV_ROW_FORMAT_SMALL:
        return ROW_TYPE_TOKU_SMALL;
    case SRV_ROW_FORMAT_FAST:
        return ROW_TYPE_TOKU_FAST;
    case SRV_ROW_FORMAT_DEFAULT:
        return ROW_TYPE_DEFAULT;
    }
#endif
    return ROW_TYPE_DEFAULT;
}

static inline srv_row_format_t row_type_to_row_format(enum row_type type) {
#if TOKU_INCLUDE_ROW_TYPE_COMPRESSION
    switch (type) {
    case ROW_TYPE_TOKU_UNCOMPRESSED:
        return SRV_ROW_FORMAT_UNCOMPRESSED;
    case ROW_TYPE_TOKU_ZLIB:
        return SRV_ROW_FORMAT_ZLIB;
    case ROW_TYPE_TOKU_SNAPPY:
        return SRV_ROW_FORMAT_SNAPPY;
    case ROW_TYPE_TOKU_QUICKLZ:
        return SRV_ROW_FORMAT_QUICKLZ;
    case ROW_TYPE_TOKU_LZMA:
        return SRV_ROW_FORMAT_LZMA;
    case ROW_TYPE_TOKU_SMALL:
        return SRV_ROW_FORMAT_SMALL;
    case ROW_TYPE_TOKU_FAST:
        return SRV_ROW_FORMAT_FAST;
    case ROW_TYPE_DEFAULT:
        return SRV_ROW_FORMAT_DEFAULT;
    default:
        return SRV_ROW_FORMAT_DEFAULT;
    }
#endif
    return SRV_ROW_FORMAT_DEFAULT;
}

static inline enum row_type toku_compression_method_to_row_type(toku_compression_method method) {
    return row_format_to_row_type(toku_compression_method_to_row_format(method));
}

static inline toku_compression_method row_type_to_toku_compression_method(enum row_type type) {
    return row_format_to_toku_compression_method(row_type_to_row_format(type));
}

// thread variables

static MYSQL_THDVAR_BOOL(commit_sync, 
    PLUGIN_VAR_THDLOCAL, 
    "sync on txn commit",
    /* check */ NULL, 
    /* update */ NULL,
    /* default*/ true
);

static MYSQL_THDVAR_UINT(pk_insert_mode,
    0,
    "set the primary key insert mode",
    NULL, 
    NULL, 
    1, // default
    0, // min?
    2, // max
    1 // blocksize
);

static uint get_pk_insert_mode(THD* thd) {
    return THDVAR(thd, pk_insert_mode);
}

static MYSQL_THDVAR_BOOL(load_save_space,
    0,
    "compress intermediate bulk loader files to save space",
    NULL, 
    NULL, 
    true
);

static bool get_load_save_space(THD* thd) {
    return (THDVAR(thd, load_save_space) != 0);
}

static MYSQL_THDVAR_BOOL(disable_slow_alter,
    0,
    "if on, alter tables that require copy are disabled",
    NULL, 
    NULL, 
    false
);

static bool get_disable_slow_alter(THD* thd) {
    return (THDVAR(thd, disable_slow_alter) != 0);
}

static MYSQL_THDVAR_BOOL(disable_hot_alter,
    0,
    "if on, hot alter table is disabled",
    NULL, 
    NULL, 
    false
);

static bool get_disable_hot_alter(THD* thd) {
    return THDVAR(thd, disable_hot_alter) != 0;
}

static MYSQL_THDVAR_BOOL(create_index_online,
    0,
    "if on, create index done online",
    NULL, 
    NULL, 
    true
);

static bool get_create_index_online(THD* thd) {
    return (THDVAR(thd, create_index_online) != 0);
}

static MYSQL_THDVAR_BOOL(alter_print_error,
    0,
    "Print errors for alter table operations",
    NULL, 
    NULL,
    false
);

static MYSQL_THDVAR_BOOL(disable_prefetching,
    0,
    "if on, prefetching disabled",
    NULL, 
    NULL, 
   false
);

static bool get_disable_prefetching(THD* thd) {
    return (THDVAR(thd, disable_prefetching) != 0);
}

static MYSQL_THDVAR_BOOL(prelock_empty,
    0,
    "Tokudb Prelock Empty Table",
    NULL, 
    NULL, 
    true
);

static bool get_prelock_empty(THD* thd) {
    return (THDVAR(thd, prelock_empty) != 0);
}

static MYSQL_THDVAR_UINT(block_size,
    0,
    "fractal tree block size",
    NULL, 
    NULL, 
    4<<20, // default
    4096,  // min
    ~0U,   // max
    1      // blocksize???
);

static uint get_tokudb_block_size(THD* thd) {
    return THDVAR(thd, block_size);
}

static MYSQL_THDVAR_UINT(read_block_size,
    0,
    "fractal tree read block size",
    NULL, 
    NULL, 
    64*1024, // default
    4096,  // min
    ~0U,   // max
    1      // blocksize???
);

static uint get_tokudb_read_block_size(THD* thd) {
    return THDVAR(thd, read_block_size);
}

static MYSQL_THDVAR_UINT(read_buf_size,
    0,
    "fractal tree read block size", //TODO: Is this a typo?
    NULL, 
    NULL, 
    128*1024, // default
    0,  // min
    1*1024*1024,   // max
    1      // blocksize???
);

static uint get_tokudb_read_buf_size(THD* thd) {
    return THDVAR(thd, read_buf_size);
}

#if TOKU_INCLUDE_UPSERT
static MYSQL_THDVAR_BOOL(disable_slow_update, 
    PLUGIN_VAR_THDLOCAL, 
    "disable slow update",
    NULL, // check
    NULL, // update
    false // default
);

static MYSQL_THDVAR_BOOL(disable_slow_upsert, 
    PLUGIN_VAR_THDLOCAL, 
    "disable slow upsert",
    NULL, // check
    NULL, // update
    false // default
);
#endif

static MYSQL_THDVAR_UINT(fanout,
    0,
    "fractal tree fanout",
    NULL, 
    NULL, 
    16,    // default
    2,     // min
    16*1024,  // max
    1      // blocksize???
);

static uint get_tokudb_fanout(THD* thd) {
    return THDVAR(thd, fanout);
}

static MYSQL_THDVAR_UINT(analyze_time, 0, "analyze time (seconds)", NULL /*check*/, NULL /*update*/, 5 /*default*/, 0 /*min*/, ~0U /*max*/, 1 /*blocksize*/);

static MYSQL_THDVAR_DOUBLE(analyze_delete_fraction, 0, "fraction of rows allowed to be deleted", NULL /*check*/, NULL /*update*/, 1.0 /*def*/, 0 /*min*/, 1.0 /*max*/, 1);

static void tokudb_checkpoint_lock(THD * thd);
static void tokudb_checkpoint_unlock(THD * thd);

static void tokudb_checkpoint_lock_update(
    THD* thd,
    struct st_mysql_sys_var* var,
    void* var_ptr,
    const void* save) 
{
    my_bool* val = (my_bool *) var_ptr;
    *val= *(my_bool *) save ? true : false;
    if (*val) {
        tokudb_checkpoint_lock(thd);
    }
    else {
        tokudb_checkpoint_unlock(thd);
    }
}
  
static MYSQL_THDVAR_BOOL(checkpoint_lock,
    0,
    "Tokudb Checkpoint Lock",
    NULL, 
    tokudb_checkpoint_lock_update, 
    false
);

static const char *tokudb_row_format_names[] = {
    "tokudb_uncompressed",
    "tokudb_zlib",
    "tokudb_snappy",
    "tokudb_quicklz",
    "tokudb_lzma",
    "tokudb_fast",
    "tokudb_small",
    "tokudb_default",
    NullS
};

static TYPELIB tokudb_row_format_typelib = {
    array_elements(tokudb_row_format_names) - 1,
    "tokudb_row_format_typelib",
    tokudb_row_format_names,
    NULL
};

static MYSQL_THDVAR_ENUM(row_format, PLUGIN_VAR_OPCMDARG,
                         "Specifies the compression method for a table during this session. "
                         "Possible values are TOKUDB_UNCOMPRESSED, TOKUDB_ZLIB, TOKUDB_SNAPPY, "
                         "TOKUDB_QUICKLZ, TOKUDB_LZMA, TOKUDB_FAST, TOKUDB_SMALL and TOKUDB_DEFAULT",
                         NULL, NULL, SRV_ROW_FORMAT_ZLIB, &tokudb_row_format_typelib);

static inline srv_row_format_t get_row_format(THD *thd) {
    return (srv_row_format_t) THDVAR(thd, row_format);
}

static MYSQL_THDVAR_UINT(lock_timeout_debug, 0, "TokuDB lock timeout debug", NULL /*check*/, NULL /*update*/, 1 /*default*/, 0 /*min*/, ~0U /*max*/, 1);

static MYSQL_THDVAR_STR(last_lock_timeout, PLUGIN_VAR_MEMALLOC, "last TokuDB lock timeout", NULL /*check*/, NULL /*update*/, NULL /*default*/);

static MYSQL_THDVAR_BOOL(hide_default_row_format, 0, "hide the default row format", NULL /*check*/, NULL /*update*/, true);

static const uint64_t DEFAULT_TOKUDB_LOCK_TIMEOUT = 4000; /*milliseconds*/

static MYSQL_THDVAR_ULONGLONG(lock_timeout, 0, "TokuDB lock timeout", NULL, NULL, DEFAULT_TOKUDB_LOCK_TIMEOUT, 0 /*min*/, ~0ULL /*max*/, 1 /*blocksize*/);

static uint64_t tokudb_get_lock_wait_time_callback(uint64_t default_wait_time) {
    THD *thd = current_thd;
    uint64_t wait_time = THDVAR(thd, lock_timeout);
    return wait_time;
}

static MYSQL_THDVAR_ULONGLONG(loader_memory_size,
    0,
    "TokuDB loader memory size",
    NULL, 
    NULL, 
    100*1000*1000, /*default*/
    0, /*min*/
    ~0ULL, /*max*/
    1 /*blocksize*/
);

static uint64_t tokudb_get_loader_memory_size_callback(void) {
    THD *thd = current_thd;
    uint64_t memory_size = THDVAR(thd, loader_memory_size);
    return memory_size;
}

static const uint64_t DEFAULT_TOKUDB_KILLED_TIME = 4000; 

static MYSQL_THDVAR_ULONGLONG(killed_time, 0, "TokuDB killed time", NULL, NULL, DEFAULT_TOKUDB_KILLED_TIME, 0 /*min*/, ~0ULL /*max*/, 1 /*blocksize*/);

static uint64_t tokudb_get_killed_time_callback(uint64_t default_killed_time) {
    THD *thd = current_thd;
    uint64_t killed_time = THDVAR(thd, killed_time);
    return killed_time;
}

static int tokudb_killed_callback(void) {
    THD *thd = current_thd;
    return thd_killed(thd);
}

static bool tokudb_killed_thd_callback(void *extra, uint64_t deleted_rows) {
    THD *thd = static_cast<THD *>(extra);
    return thd_killed(thd) != 0;
}

enum {
    TOKUDB_EMPTY_SCAN_DISABLED = 0,
    TOKUDB_EMPTY_SCAN_LR = 1,
    TOKUDB_EMPTY_SCAN_RL = 2,
};

static const char *tokudb_empty_scan_names[] = {
    "disabled",
    "lr",
    "rl",
    NullS
};

static TYPELIB tokudb_empty_scan_typelib = {
    array_elements(tokudb_empty_scan_names) - 1,
    "tokudb_empty_scan_typelib",
    tokudb_empty_scan_names,
    NULL
};

static MYSQL_THDVAR_ENUM(empty_scan, PLUGIN_VAR_OPCMDARG,
    "TokuDB algorithm to check if the table is empty when opened. ",
    NULL, NULL, TOKUDB_EMPTY_SCAN_RL, &tokudb_empty_scan_typelib
);

#if TOKUDB_CHECK_JEMALLOC
static uint tokudb_check_jemalloc;
static MYSQL_SYSVAR_UINT(check_jemalloc, tokudb_check_jemalloc, 0, "Check if jemalloc is linked",
                         NULL, NULL, 1, 0, 1, 0);
#endif

static MYSQL_THDVAR_BOOL(bulk_fetch, PLUGIN_VAR_THDLOCAL, "enable bulk fetch",
                         NULL /*check*/, NULL /*update*/, true /*default*/);

#if TOKU_INCLUDE_XA
static MYSQL_THDVAR_BOOL(support_xa,
    PLUGIN_VAR_OPCMDARG,
    "Enable TokuDB support for the XA two-phase commit",
    NULL, // check
    NULL, // update
    true  // default
);
#endif

static MYSQL_THDVAR_BOOL(rpl_unique_checks, PLUGIN_VAR_THDLOCAL, "enable unique checks on replication slave",
                         NULL /*check*/, NULL /*update*/, true /*default*/);

static MYSQL_THDVAR_ULONGLONG(rpl_unique_checks_delay, PLUGIN_VAR_THDLOCAL, "time in milliseconds to add to unique checks test on replication slave",
                              NULL, NULL, 0 /*default*/, 0 /*min*/, ~0ULL /*max*/, 1 /*blocksize*/);

static MYSQL_THDVAR_BOOL(rpl_lookup_rows, PLUGIN_VAR_THDLOCAL, "lookup a row on rpl slave",
                         NULL /*check*/, NULL /*update*/, true /*default*/);

static MYSQL_THDVAR_ULONGLONG(rpl_lookup_rows_delay, PLUGIN_VAR_THDLOCAL, "time in milliseconds to add to lookups on replication slave",
                              NULL, NULL, 0 /*default*/, 0 /*min*/, ~0ULL /*max*/, 1 /*blocksize*/);

static MYSQL_THDVAR_BOOL(rpl_check_readonly, PLUGIN_VAR_THDLOCAL, "check if the slave is read only",
                         NULL /*check*/, NULL /*update*/, true /*default*/);

static MYSQL_THDVAR_STR(optimize_index_name, PLUGIN_VAR_THDLOCAL + PLUGIN_VAR_MEMALLOC, "optimize index name (default all indexes)", NULL /*check*/, NULL /*update*/, NULL /*default*/);

static MYSQL_THDVAR_DOUBLE(optimize_index_fraction, 0, "optimize index fraction (default 1.0 all)", NULL /*check*/, NULL /*update*/, 1.0 /*def*/, 0 /*min*/, 1.0 /*max*/, 1);

static MYSQL_THDVAR_ULONGLONG(optimize_throttle, 0, "optimize throttle (default no throttle)", NULL /*check*/, NULL /*update*/, 0 /*def*/, 0 /*min*/, ~0ULL /*max*/, 1);

extern HASH tokudb_open_tables;
extern pthread_mutex_t tokudb_mutex;
extern uint32_t tokudb_write_status_frequency;
extern uint32_t tokudb_read_status_frequency;

void toku_hton_update_primary_key_bytes_inserted(uint64_t row_size);

#endif //#ifdef _HATOKU_HTON
