
/**
 *    Copyright (C) 2018-present MongoDB, Inc.
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the Server Side Public License, version 1,
 *    as published by MongoDB, Inc.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    Server Side Public License for more details.
 *
 *    You should have received a copy of the Server Side Public License
 *    along with this program. If not, see
 *    <http://www.mongodb.com/licensing/server-side-public-license>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the Server Side Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */

#define MONGO_LOG_DEFAULT_COMPONENT ::mongo::logger::LogComponent::kAccessControl

#include "mongo/platform/basic.h"

#include "mongo/db/auth/security_key.h"

#include <string>
#include <sys/stat.h>

#include "mongo/base/status_with.h"
#include "mongo/db/server_options.h"
#include "mongo/util/mongoutils/str.h"

namespace mongo {

using std::string;

StatusWith<std::string> readSecurityFile(const std::string& filename) {
    struct stat stats;

    // check obvious file errors
    if (stat(filename.c_str(), &stats) == -1) {
        return StatusWith<std::string>(ErrorCodes::InvalidPath,
                                       str::stream() << "Error reading file " << filename << ": "
                                                     << strerror(errno));
    }

#if !defined(_WIN32)
    if (serverGlobalParams.relaxPermChecks && stats.st_uid == 0) {
        /* In case the owner is root then permission of the key file
         * can be a bit more open than for the non root users. The
         * group read is also permissible values for the file permission.
         * -rwx--r---- 740 owner read, write and execute and group read
         * bit.
         * These remaining bit should not be set for key file.
         * S_IWGRP -- Group Write.
         * S_IXGRP -- Group Execute.
         * S_IRWXO -- Read, Write and Execute for others.
         * ref: https://www.gnu.org/software/libc/manual/html_node/Permission-Bits.html
         */
        if ((stats.st_mode & (S_IWGRP | S_IXGRP | S_IRWXO)) != 0) {
            return StatusWith<std::string>(ErrorCodes::InvalidPath,
                                           str::stream() << "permissions on " << filename
                                                         << " are too open");
        }
    } else {
        // Check permissions: must be X00, where X is >= 4 in case of non root owner.
        if ((stats.st_mode & (S_IRWXG | S_IRWXO)) != 0) {
            return StatusWith<std::string>(ErrorCodes::InvalidPath,
                                           str::stream() << "permissions on " << filename
                                                         << " are too open");
        }
    }
#endif

    FILE* file = fopen(filename.c_str(), "rb");
    if (!file) {
        return StatusWith<std::string>(ErrorCodes::InvalidPath,
                                       str::stream() << "error opening file: " << filename << ": "
                                                     << strerror(errno));
    }

    string str = "";

    // strip key file
    const unsigned long long fileLength = stats.st_size;
    unsigned long long read = 0;
    while (read < fileLength) {
        char buf;
        int readLength = fread(&buf, 1, 1, file);
        if (readLength < 1) {
            fclose(file);
            return StatusWith<std::string>(ErrorCodes::UnsupportedFormat,
                                           str::stream() << "error reading file: " << filename);
        }
        read++;

        // check for whitespace
        if ((buf >= '\x09' && buf <= '\x0D') || buf == ' ') {
            continue;
        }

        // check valid base64
        if ((buf < 'A' || buf > 'Z') && (buf < 'a' || buf > 'z') && (buf < '0' || buf > '9') &&
            buf != '+' && buf != '/' && buf != '=') {
            fclose(file);
            return StatusWith<std::string>(
                ErrorCodes::UnsupportedFormat,
                str::stream() << "invalid char in key file " << filename << ": " << buf);
        }

        str += buf;
    }

    fclose(file);
    return StatusWith<std::string>(str);
}

}  // namespace mongo
