/**
 *    Copyright (C) 2014 MongoDB Inc.
 *
 *    This program is free software: you can redistribute it and/or  modify
 *    it under the terms of the GNU Affero General Public License, version 3,
 *    as published by the Free Software Foundation.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU Affero General Public License for more details.
 *
 *    You should have received a copy of the GNU Affero General Public License
 *    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the GNU Affero General Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */

#pragma once

#include <string>
#include <rocksdb/status.h>

#include "mongo/util/assert_util.h"

namespace mongo {

    inline std::string rocksGetNextPrefix(const rocksdb::Slice& prefix) {
        // next prefix lexicographically, assume same length
        std::string nextPrefix(prefix.data(), prefix.size());
        for (int i = static_cast<int>(nextPrefix.size()) - 1; i >= 0; --i) {
            nextPrefix[i]++;
            // if it's == 0, that means we've overflowed, so need to keep adding
            if (nextPrefix[i] != 0) {
                break;
            }
        }
        return nextPrefix;
    }

    std::string encodePrefix(uint32_t prefix);
    bool extractPrefix(const rocksdb::Slice& slice, uint32_t* prefix);
    int get_internal_delete_skipped_count();

    Status rocksToMongoStatus_slow(const rocksdb::Status& status, const char* prefix);

    /**
     * converts rocksdb status to mongodb status
     */
    inline Status rocksToMongoStatus(const rocksdb::Status& status, const char* prefix = NULL) {
        if (MONGO_likely(status.ok())) {
            return Status::OK();
        }
        return rocksToMongoStatus_slow(status, prefix);
    }

#define invariantRocksOK(expression) do { \
        auto _invariantRocksOK_status = expression; \
        if (MONGO_unlikely(!_invariantRocksOK_status.ok())) { \
            invariantOKFailed(#expression, rocksToMongoStatus(_invariantRocksOK_status), \
                              __FILE__, __LINE__); \
        } \
    } while (false)

}  // namespace mongo
