﻿/*
* Copyright 2010-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
*
* Licensed under the Apache License, Version 2.0 (the "License").
* You may not use this file except in compliance with the License.
* A copy of the License is located at
*
*  http://aws.amazon.com/apache2.0
*
* or in the "license" file accompanying this file. This file is distributed
* on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
* express or implied. See the License for the specific language governing
* permissions and limitations under the License.
*/

#include <aws/eks/model/ErrorCode.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>

using namespace Aws::Utils;


namespace Aws
{
  namespace EKS
  {
    namespace Model
    {
      namespace ErrorCodeMapper
      {

        static const int SubnetNotFound_HASH = HashingUtils::HashString("SubnetNotFound");
        static const int SecurityGroupNotFound_HASH = HashingUtils::HashString("SecurityGroupNotFound");
        static const int EniLimitReached_HASH = HashingUtils::HashString("EniLimitReached");
        static const int IpNotAvailable_HASH = HashingUtils::HashString("IpNotAvailable");
        static const int AccessDenied_HASH = HashingUtils::HashString("AccessDenied");
        static const int OperationNotPermitted_HASH = HashingUtils::HashString("OperationNotPermitted");
        static const int VpcIdNotFound_HASH = HashingUtils::HashString("VpcIdNotFound");
        static const int Unknown_HASH = HashingUtils::HashString("Unknown");


        ErrorCode GetErrorCodeForName(const Aws::String& name)
        {
          int hashCode = HashingUtils::HashString(name.c_str());
          if (hashCode == SubnetNotFound_HASH)
          {
            return ErrorCode::SubnetNotFound;
          }
          else if (hashCode == SecurityGroupNotFound_HASH)
          {
            return ErrorCode::SecurityGroupNotFound;
          }
          else if (hashCode == EniLimitReached_HASH)
          {
            return ErrorCode::EniLimitReached;
          }
          else if (hashCode == IpNotAvailable_HASH)
          {
            return ErrorCode::IpNotAvailable;
          }
          else if (hashCode == AccessDenied_HASH)
          {
            return ErrorCode::AccessDenied;
          }
          else if (hashCode == OperationNotPermitted_HASH)
          {
            return ErrorCode::OperationNotPermitted;
          }
          else if (hashCode == VpcIdNotFound_HASH)
          {
            return ErrorCode::VpcIdNotFound;
          }
          else if (hashCode == Unknown_HASH)
          {
            return ErrorCode::Unknown;
          }
          EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
          if(overflowContainer)
          {
            overflowContainer->StoreOverflow(hashCode, name);
            return static_cast<ErrorCode>(hashCode);
          }

          return ErrorCode::NOT_SET;
        }

        Aws::String GetNameForErrorCode(ErrorCode enumValue)
        {
          switch(enumValue)
          {
          case ErrorCode::SubnetNotFound:
            return "SubnetNotFound";
          case ErrorCode::SecurityGroupNotFound:
            return "SecurityGroupNotFound";
          case ErrorCode::EniLimitReached:
            return "EniLimitReached";
          case ErrorCode::IpNotAvailable:
            return "IpNotAvailable";
          case ErrorCode::AccessDenied:
            return "AccessDenied";
          case ErrorCode::OperationNotPermitted:
            return "OperationNotPermitted";
          case ErrorCode::VpcIdNotFound:
            return "VpcIdNotFound";
          case ErrorCode::Unknown:
            return "Unknown";
          default:
            EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
            if(overflowContainer)
            {
              return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
            }

            return {};
          }
        }

      } // namespace ErrorCodeMapper
    } // namespace Model
  } // namespace EKS
} // namespace Aws
