/**
 *    Copyright (C) 2014 MongoDB Inc.
 *
 *    This program is free software: you can redistribute it and/or  modify
 *    it under the terms of the GNU Affero General Public License, version 3,
 *    as published by the Free Software Foundation.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU Affero General Public License for more details.
 *
 *    You should have received a copy of the GNU Affero General Public License
 *    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the GNU Affero General Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */

#define MONGO_LOG_DEFAULT_COMPONENT ::mongo::logger::LogComponent::kStorage

#include "mongo/platform/basic.h"

#include "rocks_snapshot_manager.h"
#include "rocks_recovery_unit.h"

#include <rocksdb/db.h>

#include "mongo/base/checked_cast.h"
#include "mongo/util/log.h"

namespace mongo {
    // This only checks invariants
    Status RocksSnapshotManager::prepareForCreateSnapshot(OperationContext* opCtx) {
        RocksRecoveryUnit::getRocksRecoveryUnit(opCtx)->prepareForCreateSnapshot(opCtx);
        return Status::OK();
    }

    void RocksSnapshotManager::setCommittedSnapshot(const Timestamp& timestamp) {
        stdx::lock_guard<stdx::mutex> lock(_mutex);

        uint64_t nameU64 = timestamp.asULL();
        _updatedCommittedSnapshot = !_committedSnapshot || *_committedSnapshot < nameU64;
        invariant(_updatedCommittedSnapshot || *_committedSnapshot == nameU64);
        if (!_updatedCommittedSnapshot)
            return;
        _committedSnapshot = nameU64;
        auto insResult = _snapshotMap.insert(SnapshotMap::value_type(nameU64, {}));
        if (insResult.second) {
            insResult.first->second = createSnapshot();
        }
        _committedSnapshotIter = insResult.first;
    }

    void RocksSnapshotManager::cleanupUnneededSnapshots() {
        stdx::lock_guard<stdx::mutex> lock(_mutex);
        if (!_updatedCommittedSnapshot) {
            return;
        }

        // erasing snapshots with timestamps less than *_committedSnapshot
        _snapshotMap.erase(_snapshotMap.begin(), _committedSnapshotIter);
        _updatedCommittedSnapshot = false;
    }

    void RocksSnapshotManager::dropAllSnapshots() {
        stdx::lock_guard<stdx::mutex> lock(_mutex);
        _committedSnapshot = boost::none;
        _updatedCommittedSnapshot = false;
        _committedSnapshotIter = SnapshotMap::iterator{};
        _maxSeenSnapshot = 0;
        _snapshotMap.clear();
    }

    bool RocksSnapshotManager::haveCommittedSnapshot() const {
        stdx::lock_guard<stdx::mutex> lock(_mutex);
        return bool(_committedSnapshot);
    }

    uint64_t RocksSnapshotManager::getCommittedSnapshotName() const {
        stdx::lock_guard<stdx::mutex> lock(_mutex);

        assertCommittedSnapshot_inlock();
        return *_committedSnapshot;
    }

    RocksSnapshotManager::SnapshotHolder RocksSnapshotManager::getCommittedSnapshot() const {
        stdx::lock_guard<stdx::mutex> lock(_mutex);

        assertCommittedSnapshot_inlock();
        return _committedSnapshotIter->second;
    }

    void RocksSnapshotManager::insertSnapshot(const Timestamp timestamp) {
        uint64_t nameU64 = timestamp.asULL();
        auto holder = createSnapshot();
        SnapshotMap::iterator it;

        stdx::lock_guard<stdx::mutex> lock(_mutex);
        std::tie(it, std::ignore) = _snapshotMap.insert(SnapshotMap::value_type(nameU64, {}));

        if (_snapshotMap.size() > 1 && nameU64 < _maxSeenSnapshot) {
            holder.inaccurate = true;
            // Timestamps came out of order, so use the freshest one
            for (auto it_end = _snapshotMap.end(); it != it_end; ++it) {
                it->second = holder;
            }
            // The most recent timestamp is accurate
            _snapshotMap.rbegin()->second.inaccurate = false;
        } else {
            it->second = std::move(holder);
            _maxSeenSnapshot = nameU64;
        }
    }

    void RocksSnapshotManager::assertCommittedSnapshot_inlock() const {
        uassert(ErrorCodes::ReadConcernMajorityNotAvailableYet,
                "Committed view disappeared while running operation", _committedSnapshot);
    }

    RocksSnapshotManager::SnapshotHolder RocksSnapshotManager::createSnapshot() {
        auto db = this->_db;
        return {SnapshotHolder::Snapshot(db->GetSnapshot(),
                                         [db](const rocksdb::Snapshot* snapshot) {
                                             if (snapshot != nullptr) {
                                                 invariant(db != nullptr);
                                                 db->ReleaseSnapshot(snapshot);
                                             }
                                         }),
                false};
    }

}  // namespace mongo
