// sorted_data_interface_test_bulkbuilder.cpp

/**
 *    Copyright (C) 2014 MongoDB Inc.
 *
 *    This program is free software: you can redistribute it and/or  modify
 *    it under the terms of the GNU Affero General Public License, version 3,
 *    as published by the Free Software Foundation.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU Affero General Public License for more details.
 *
 *    You should have received a copy of the GNU Affero General Public License
 *    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the GNU Affero General Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */

#include "mongo/db/storage/sorted_data_interface_test_harness.h"

#include <boost/scoped_ptr.hpp>

#include "mongo/db/storage/sorted_data_interface.h"
#include "mongo/unittest/unittest.h"

namespace mongo {

    using boost::scoped_ptr;

    // Add a key using a bulk builder.
    TEST( SortedDataInterface, BuilderAddKey ) {
        scoped_ptr<HarnessHelper> harnessHelper( newHarnessHelper() );
        scoped_ptr<SortedDataInterface> sorted( harnessHelper->newSortedDataInterface( false ) );

        {
            scoped_ptr<OperationContext> opCtx( harnessHelper->newOperationContext() );
            ASSERT( sorted->isEmpty( opCtx.get() ) );
        }

        {
            scoped_ptr<OperationContext> opCtx( harnessHelper->newOperationContext() );
            scoped_ptr<SortedDataBuilderInterface> builder(
                    sorted->getBulkBuilder( opCtx.get(), true ) );

            ASSERT_OK( builder->addKey( key1, loc1 ) );
            builder->commit( false );
        }

        {
            scoped_ptr<OperationContext> opCtx( harnessHelper->newOperationContext() );
            ASSERT_EQUALS( 1, sorted->numEntries( opCtx.get() ) );
        }
    }

    // Add a compound key using a bulk builder.
    TEST( SortedDataInterface, BuilderAddCompoundKey ) {
        scoped_ptr<HarnessHelper> harnessHelper( newHarnessHelper() );
        scoped_ptr<SortedDataInterface> sorted( harnessHelper->newSortedDataInterface( false ) );

        {
            scoped_ptr<OperationContext> opCtx( harnessHelper->newOperationContext() );
            ASSERT( sorted->isEmpty( opCtx.get() ) );
        }

        {
            scoped_ptr<OperationContext> opCtx( harnessHelper->newOperationContext() );
            scoped_ptr<SortedDataBuilderInterface> builder(
                    sorted->getBulkBuilder( opCtx.get(), true ) );

            ASSERT_OK( builder->addKey( compoundKey1a, loc1 ) );
            builder->commit( false );
        }

        {
            scoped_ptr<OperationContext> opCtx( harnessHelper->newOperationContext() );
            ASSERT_EQUALS( 1, sorted->numEntries( opCtx.get() ) );
        }
    }

    // Add the same key multiple times using a bulk builder and verify that
    // the returned status is ErrorCodes::DuplicateKey when duplicates are
    // not allowed.
    TEST( SortedDataInterface, BuilderAddSameKey ) {
        scoped_ptr<HarnessHelper> harnessHelper( newHarnessHelper() );
        scoped_ptr<SortedDataInterface> sorted( harnessHelper->newSortedDataInterface( true ) );

        {
            scoped_ptr<OperationContext> opCtx( harnessHelper->newOperationContext() );
            ASSERT( sorted->isEmpty( opCtx.get() ) );
        }

        {
            scoped_ptr<OperationContext> opCtx( harnessHelper->newOperationContext() );
            scoped_ptr<SortedDataBuilderInterface> builder(
                    sorted->getBulkBuilder( opCtx.get(), false ) );

            ASSERT_OK( builder->addKey( key1, loc1 ) );
            ASSERT_EQUALS( ErrorCodes::DuplicateKey, builder->addKey( key1, loc2 ) );
            builder->commit( false );
        }

        {
            scoped_ptr<OperationContext> opCtx( harnessHelper->newOperationContext() );
            ASSERT_EQUALS( 1, sorted->numEntries( opCtx.get() ) );
        }
    }

    // Add the same key multiple times using a bulk builder and verify that
    // the returned status is OK when duplicates are allowed.
    TEST( SortedDataInterface, BuilderAddSameKeyWithDupsAllowed ) {
        scoped_ptr<HarnessHelper> harnessHelper( newHarnessHelper() );
        scoped_ptr<SortedDataInterface> sorted( harnessHelper->newSortedDataInterface( false ) );

        {
            scoped_ptr<OperationContext> opCtx( harnessHelper->newOperationContext() );
            ASSERT( sorted->isEmpty( opCtx.get() ) );
        }

        {
            scoped_ptr<OperationContext> opCtx( harnessHelper->newOperationContext() );
            scoped_ptr<SortedDataBuilderInterface> builder(
                    sorted->getBulkBuilder( opCtx.get(), true /* allow duplicates */ ) );

            ASSERT_OK( builder->addKey( key1, loc1 ) );
            ASSERT_OK( builder->addKey( key1, loc2 ) );
            builder->commit( false );
        }

        {
            scoped_ptr<OperationContext> opCtx( harnessHelper->newOperationContext() );
            ASSERT_EQUALS( 2, sorted->numEntries( opCtx.get() ) );
        }
    }

    // Add multiple keys using a bulk builder.
    TEST( SortedDataInterface, BuilderAddMultipleKeys ) {
        scoped_ptr<HarnessHelper> harnessHelper( newHarnessHelper() );
        scoped_ptr<SortedDataInterface> sorted( harnessHelper->newSortedDataInterface( false ) );

        {
            scoped_ptr<OperationContext> opCtx( harnessHelper->newOperationContext() );
            ASSERT( sorted->isEmpty( opCtx.get() ) );
        }

        {
            scoped_ptr<OperationContext> opCtx( harnessHelper->newOperationContext() );
            scoped_ptr<SortedDataBuilderInterface> builder(
                    sorted->getBulkBuilder( opCtx.get(), true ) );

            ASSERT_OK( builder->addKey( key1, loc1 ) );
            ASSERT_OK( builder->addKey( key2, loc2 ) );
            ASSERT_OK( builder->addKey( key3, loc3 ) );
            builder->commit( false );
        }

        {
            scoped_ptr<OperationContext> opCtx( harnessHelper->newOperationContext() );
            ASSERT_EQUALS( 3, sorted->numEntries( opCtx.get() ) );
        }
    }

    // Add multiple compound keys using a bulk builder.
    TEST( SortedDataInterface, BuilderAddMultipleCompoundKeys ) {
        scoped_ptr<HarnessHelper> harnessHelper( newHarnessHelper() );
        scoped_ptr<SortedDataInterface> sorted( harnessHelper->newSortedDataInterface( false ) );

        {
            scoped_ptr<OperationContext> opCtx( harnessHelper->newOperationContext() );
            ASSERT( sorted->isEmpty( opCtx.get() ) );
        }

        {
            scoped_ptr<OperationContext> opCtx( harnessHelper->newOperationContext() );
            scoped_ptr<SortedDataBuilderInterface> builder(
                    sorted->getBulkBuilder( opCtx.get(), true ) );

            ASSERT_OK( builder->addKey( compoundKey1a, loc1 ) );
            ASSERT_OK( builder->addKey( compoundKey1b, loc2 ) );
            ASSERT_OK( builder->addKey( compoundKey1c, loc4 ) );
            ASSERT_OK( builder->addKey( compoundKey2b, loc3 ) );
            ASSERT_OK( builder->addKey( compoundKey3a, loc5 ) );
            builder->commit( false );
        }

        {
            scoped_ptr<OperationContext> opCtx( harnessHelper->newOperationContext() );
            ASSERT_EQUALS( 5, sorted->numEntries( opCtx.get() ) );
        }
    }

} // namespace mongo
