/*
 * bltGrElem.c --
 *
 *	This module implements a elements in the graph widget
 *	for the Tk toolkit.
 *
 * Copyright 1991-1993 by AT&T Bell Laboratories.
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that the copyright notice and warranty
 * disclaimer appear in supporting documentation, and that the
 * names of AT&T Bell Laboratories any of their entities not be used
 * in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * AT&T disclaims all warranties with regard to this software, including
 * all implied warranties of merchantability and fitness.  In no event
 * shall AT&T be liable for any special, indirect or consequential
 * damages or any damages whatsoever resulting from loss of use, data
 * or profits, whether in an action of contract, negligence or other
 * tortuous action, arising out of or in connection with the use or
 * performance of this software.
 *
 */

#include <ctype.h>
#include <errno.h>

#include <tk.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>

#include "bltGraph.h"


#ifdef hpux
#define BLACK		"black"
#define WHITE		"white"
#define LIGHTGREY       "lightgrey"
#define BISQUE1		"bisque1"
#else
#define BLACK		"#000000"
#define WHITE		"#ffffff"
#define LIGHTGREY       "#d3d3d3"
#define BISQUE1		"#ffe4c4"
#endif

enum DataModes {
    DATA_OVERWRITE, DATA_APPEND
};

enum DataTypes {
    DATA_SINGLE = 1, DATA_PAIRS = 2
};

/*
 * Sun's bundled and unbundled C compilers can't grok static
 * function typedefs (it can handle extern) like
 *
 * 	static Tk_OptionParseProc parseProc;
 *  	static Tk_OptionPrintProc printProc;
 *
 * Provide forward declarations here:
 */
static int ParseSymbolType _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp, Tk_Window tkwin, char *value, char *widgRec,
	int offset));
static char *PrintSymbolType _ANSI_ARGS_((ClientData clientData,
	Tk_Window tkwin, char *widgRec, int offset,
	Tcl_FreeProc **freeProcPtr));
static int ParseVector _ANSI_ARGS_((ClientData clientData, Tcl_Interp *interp,
	Tk_Window tkwin, char *value, char *widgRec, int offset));
static char *PrintVector _ANSI_ARGS_((ClientData clientData, Tk_Window tkwin,
	char *widgRec, int offset, Tcl_FreeProc **freeProcPtr));
static int ParseCoordPairs _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp, Tk_Window tkwin, char *value, char *widgRec,
	int offset));
static char *PrintCoordPairs _ANSI_ARGS_((ClientData clientData,
	Tk_Window tkwin, char *widgRec, int offset,
	Tcl_FreeProc **freeProcPtr));

#include "bltGrElem.h"

/*
 * Element attributes: either symbol types or line styles
 */
static char *symbolNames[] =
{
    "Line", "Square", "Circle", "Diamond", "Plus", "Cross",
};
static int numSymbols = sizeof(symbolNames) / sizeof(char *);

typedef struct {
    Tcl_Interp *interp;		/* Interpreter of the graph widget. This is
				 * only needed for Tcl_PrintDouble calls in
				 * the custom configure print routines. */
    ElementClassType type;	/* Type of element is LINE_ELEMENT */
    unsigned int flags;
    Tk_Uid id;			/* Identifier to refer the element. Used in
				 * the "insert", "delete", or "show",
				 * commands. */
    int mapped;			/* If non-zero, element is currently visible.*/
    Tk_ConfigSpec *configSpecs;	/* Configuration specifications */
    char *label;		/* Label displayed in legend */
    SymbolType symbol;		/* Element symbol type */
    double symbolScale;		/* Scale factor when computing symbol size */
    int symbolSize;		/* Computed size of symbol in pixels. */
    Vector x, y;		/* Contains array of numeric values */

    ElemConfigProc *configProc;
    ElemDestroyProc *destroyProc;
    ElemDisplayProc *displayProc;
    ElemLimitsProc *limitsProc;
    ElemLayoutProc *layoutProc;
    ElemPrintProc *printProc;
    ElemDrawSymbolsProc *drawSymbolsProc;
    ElemPrintSymbolsProc *printSymbolsProc;
    /*
     * Line specific configurable attributes
     */
    XColor *fgColorPtr;		/* Line color */
    XColor *bgColorPtr;		/* Background color (for dashes lines)*/
    int lineWidth;		/* Line width */
    int dashes;			/* Dash on-off list value */
    int noretrace;		/* If non-zero, break connected line segments
				 * where x-coordinate values are not
				 * monotonically increasing. */
    GC outlineGC;		/* Symbol outline graphics context */
    GC fillGC;			/* Symbol fill graphics context */
    GC lineGC;			/* Line graphics context */
    /*
     * Drawing related structures
     */
    XPoint *pointArr;		/* Array of window coordinates representing
				 * the points of the line (malloc'ed) */
    int numPoints;		/* Number of points in point array */

} LineElement;

static Tk_CustomOption SymbolOption =
{
    ParseSymbolType, PrintSymbolType, NULL
};

static Tk_CustomOption XVectorOption =
{
    ParseVector, PrintVector, (ClientData)X_AXIS_TYPE
};
static Tk_CustomOption YVectorOption =
{
    ParseVector, PrintVector, (ClientData)Y_AXIS_TYPE
};

static Tk_CustomOption TwinOption =
{
    ParseCoordPairs, PrintCoordPairs, NULL
};

#define DEF_ELEM_BG_COLOR	 WHITE
#define DEF_ELEM_FG_COLOR	 BLACK
#define DEF_ELEM_BORDERWIDTH	 "2"

static Tk_ConfigSpec lineConfigSpecs[] =
{
    {TK_CONFIG_COLOR, "-background", "lineElemBackground", "ElemBackground",
	DEF_ELEM_BG_COLOR, Tk_Offset(LineElement, bgColorPtr),
	TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_COLOR, "-background", "lineElemBackground", "ElemBackground",
	DEF_ELEM_BG_COLOR, Tk_Offset(LineElement, bgColorPtr),
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_SYNONYM, "-bg", "lineElemBackground", (char *)NULL,
	(char *)NULL, 0, 0},
    {TK_CONFIG_INT, "-dashes", "lineElemDashes", "ElemDashes",
	(char *)NULL, Tk_Offset(LineElement, dashes), 0},
    {TK_CONFIG_CUSTOM, "-data", "lineElemData", "ElemData",
	(char *)NULL, 0, 0, &TwinOption},
    {TK_CONFIG_SYNONYM, "-fg", "lineElemForeground", (char *)NULL,
	(char *)NULL, 0, TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_SYNONYM, "-fg", "lineElemForeground", (char *)NULL,
	(char *)NULL, 0, 0},
    {TK_CONFIG_COLOR, "-foreground", "lineElemForeground", "ElemForeground",
	DEF_ELEM_FG_COLOR, Tk_Offset(LineElement, fgColorPtr),
	TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_COLOR, "-foreground", "lineElemForeground", "ElemForeground",
	DEF_ELEM_FG_COLOR, Tk_Offset(LineElement, fgColorPtr),
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_PIXELS, "-linewidth", "lineElemLinewidth", "ElemLinewidth",
	(char *)NULL, Tk_Offset(LineElement, lineWidth), 0},
    {TK_CONFIG_DOUBLE, "-scale", "lineElemScale", "ElemScale",
	(char *)NULL, Tk_Offset(LineElement, symbolScale), 0},
    {TK_CONFIG_CUSTOM, "-symbol", "lineElemSymbol", "ElemSymbol",
	(char *)NULL, Tk_Offset(LineElement, symbol), 0, &SymbolOption},
    {TK_CONFIG_CUSTOM, "-xdata", "lineElemXdata", "ElemXdata",
	(char *)NULL, 0, 0, &XVectorOption},
    {TK_CONFIG_CUSTOM, "-ydata", "lineElemYdata", "ElemYdata",
	(char *)NULL, 0, 0, &YVectorOption},
    {TK_CONFIG_STRING, "-label", "lineElemLabel", "ElemLabel",
	(char *)NULL, Tk_Offset(LineElement, label), TK_CONFIG_NULL_OK},
    {TK_CONFIG_BOOLEAN, "-noretrace", "lineElemNoretrace", "ElemNoretrace",
	(char *)NULL, Tk_Offset(LineElement, noretrace), 0},
    {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

typedef struct {
    Tcl_Interp *interp;		/* Interpreter of the graph widget */
    ElementClassType type;	/* Type of element is BAR_ELEMENT */
    unsigned int flags;
    Tk_Uid id;			/* Identifier to refer the element. Used in
				 * the "insert", "delete", or "show",
				 * commands. */
    int mapped;			/* If non-zero, element is currently visible.*/
    Tk_ConfigSpec *configSpecs;	/* Configuration specifications */
    char *label;		/* Label displayed in legend */
    SymbolType symbol;		/* Element symbol type */
    double symbolScale;		/* Size of symbol as a percentage of the
				 * drawing area. */
    int symbolSize;		/* Computed size of symbol in pixels. */
    Vector x, y;		/* Contains array of numeric values */

    ElemConfigProc *configProc;
    ElemDestroyProc *destroyProc;
    ElemDisplayProc *displayProc;
    ElemLimitsProc *limitsProc;
    ElemLayoutProc *layoutProc;
    ElemPrintProc *printProc;
    ElemDrawSymbolsProc *drawSymbolsProc;
    ElemPrintSymbolsProc *printSymbolsProc;
    /*
     * Bar specific attributes
     */
    XColor *fgColorPtr;		/* Bar color */
    Tk_3DBorder border;		/* 3D border and background color */
    GC gc;			/* Possibly shared graphics context */
    int borderWidth;		/* 3D border width of bar */
    int relief;			/* Relief of bar */
    int stacked;		/* If non-zero, the set of bar y-values
				 * represent a stacked bar segments */
    double beforeSpace;		/* Spacing before the line */
    Pixmap stipple;		/* Bitmap representing stippled pattern of
				 * bar. If None, indicates a solid fill. */
    int padX;			/* Spacing on either side of bar */
    XRectangle *rectArr;	/* Array of rectangle coordinates composing
				 * the set of a bars possible */
    int numRects;		/* Number of bars in the set */

} BarElement;

static Tk_ConfigSpec barConfigSpecs[] =
{
    {TK_CONFIG_BORDER, "-background", "barElemBackground", "ElemBackground",
	DEF_ELEM_BG_COLOR, Tk_Offset(BarElement, border),
	TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_BORDER, "-background", "barElemBackground", "ElemBackground",
	DEF_ELEM_BG_COLOR, Tk_Offset(BarElement, border),
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_SYNONYM, "-bd", "barElemBorderwidth", (char *)NULL,
	(char *)NULL, 0, 0},
    {TK_CONFIG_SYNONYM, "-bg", "barElemBackground", (char *)NULL,
	(char *)NULL, 0, 0},
    {TK_CONFIG_PIXELS, "-borderwidth", "barElemBorderwidth",
	"ElemBorderwidth",
	DEF_ELEM_BORDERWIDTH, Tk_Offset(BarElement, borderWidth), 0},
    {TK_CONFIG_SYNONYM, "-fg", "barElemForeground", (char *)NULL,
	(char *)NULL, 0, 0},
    {TK_CONFIG_CUSTOM, "-data", "barElemData", "ElemData",
	(char *)NULL, 0, 0, &TwinOption},
    {TK_CONFIG_COLOR, "-foreground", "barElemForeground", "ElemForeground",
	DEF_ELEM_FG_COLOR, Tk_Offset(BarElement, fgColorPtr),
	TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_COLOR, "-foreground", "barElemForeground", "ElemForeground",
	DEF_ELEM_FG_COLOR, Tk_Offset(BarElement, fgColorPtr),
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_STRING, "-label", "barElemLabel", "ElemLabel",
	(char *)NULL, Tk_Offset(BarElement, label), TK_CONFIG_NULL_OK},
    {TK_CONFIG_RELIEF, "-relief", "barElemRelief", "ElemRelief",
	(char *)NULL, Tk_Offset(BarElement, relief), 0},
    {TK_CONFIG_BOOLEAN, "-stacked", "barElemStacked", "ElemStacked",
	(char *)NULL, Tk_Offset(BarElement, stacked), 0},
    {TK_CONFIG_BITMAP, "-stipple", "barElemStipple", "ElemStipple",
	(char *)NULL, Tk_Offset(BarElement, stipple), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-xdata", "barElemXdata", "ElemXdata",
	(char *)NULL, 0, 0, &XVectorOption},
    {TK_CONFIG_CUSTOM, "-ydata", "barElemYdata", "ElemYdata",
	(char *)NULL, 0, 0, &YVectorOption},
    {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};


extern char *sys_errlist[];

extern int strncasecmp _ANSI_ARGS_((CONST char *s1, CONST char *s2, size_t n));
extern int toupper _ANSI_ARGS_((int c));
extern int Blt_FindCmd _ANSI_ARGS_((Tcl_Interp *, char *, ClientData *));


/* ----------------------------------------------------------------------
 * Custom option parse and print procedures
 * ----------------------------------------------------------------------
 */

/*
 *----------------------------------------------------------------------
 *
 * ParseSymbol --
 *
 *	Convert the string representation of a line style or symbol name
 *	into its numeric form.
 *
 * Results:
 *	The return value is a standard Tcl result.  The symbol type is
 *	written into the widget record.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ParseSymbolType(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;	/* not used */
    Tcl_Interp *interp;		/* Interpreter to send results back to */
    Tk_Window tkwin;		/* not used */
    char *value;		/* String representing symbol type */
    char *widgRec;		/* Element information record */
    int offset;			/* Offset of symbol type field in record */
{
    register char c;
    int *symbolPtr = (int *)(widgRec + offset);
    register int i;
    int length;

    length = strlen(value);
    c = toupper(*value);
    for (i = 0; i < numSymbols; i++) {
	if ((c == *symbolNames[i]) &&
	    (strncasecmp(value, symbolNames[i], length) == 0)) {
	    *symbolPtr = i;
	    return TCL_OK;
	}
    }
    Tcl_AppendResult(interp, "bad symbol name \"", value, "\"", (char *)NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * PrintSymbolType --
 *
 *	Convert the symbol value into a string.
 *
 * Results:
 *	The string representing the symbol type or line style is returned.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static char *
PrintSymbolType(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;	/* not used */
    Tk_Window tkwin;		/* not used */
    char *widgRec;		/* Element information record */
    int offset;			/* Offset of symbol type field in record */
    Tcl_FreeProc **freeProcPtr;	/* not used */
{
    int symbol = *(int *)(widgRec + offset);

    if (symbol >= 0 || symbol < numSymbols) {
	return (symbolNames[symbol]);
    }
    return NULL;
}


static int
CnvtExprArr(interp, exprArr, numExpr, dataArr, start, step)
    Tcl_Interp *interp;		/* Interpreter to report error back to */
    char *exprArr[];		/* Array of expression strings */
    int numExpr;		/* Number of expressions in array */
    double dataArr[];
    int start, step;		/* Starting point and step */
{
    register int i, count;
    double x;

    /*
     * Parse and evaluate the list of numeric expressions into an array of
     * floating point numbers.
     */
    for (count = 0, i = start; i < numExpr; i += step, count++) {
	if (Tcl_ExprDouble(interp, exprArr[i], &x) != TCL_OK) {
	    return TCL_ERROR;
	}
	dataArr[count] = x;
    }
    return TCL_OK;
}


static void
GetDataLimits(vecPtr)
    register Vector *vecPtr;
{
    register int i;

    /*
     * Find the minimum, positive minimum, and maximum values in the array.
     */
    vecPtr->min = vecPtr->max = vecPtr->data[0];
    vecPtr->logMin = Blt_posInfinity;
    for (i = 0; i < vecPtr->length; i++) {
	if ((vecPtr->data[i] > 0.0) && (vecPtr->data[i] < vecPtr->logMin)) {
	    vecPtr->logMin = vecPtr->data[i];
	}
	if (vecPtr->data[i] < vecPtr->min) {
	    vecPtr->min = vecPtr->data[i];
	} else if (vecPtr->data[i] > vecPtr->max) {
	    vecPtr->max = vecPtr->data[i];
	}
    }
}

static int
AppendVector(interp, vecPtr, exprArr, numExpr, start, step, mode)
    Tcl_Interp *interp;		/* Interpreter to report error back to */
    Vector *vecPtr;		/* Vector to be filled */
    char *exprArr[];		/* Array of expression strings */
    int numExpr;		/* Number of expressions in array */
    int start, step;		/* Starting point and step */
    enum DataModes mode;	/* Append or overwrite current data */
{
    unsigned int arraySize;
    double *dataArr;
    unsigned int offset;

    if (numExpr < 1) {
	if (vecPtr->data != NULL) {
	    free((char *)vecPtr->data);
	}
	vecPtr->data = NULL;
	vecPtr->length = 0;
	return TCL_OK;
    }
    offset = (mode == DATA_APPEND) ? vecPtr->length : 0;
    arraySize = (numExpr / step) + offset;
    dataArr = (double *)calloc(arraySize, sizeof(double));

    if (dataArr == NULL) {
	Tcl_AppendResult(interp, "Can't allocate vector: ",
	    Tcl_PosixError(interp), (char *)NULL);
	return TCL_ERROR;
    }
    if (offset > 0) {
	/* Copy previous contents */
	memcpy((char *)dataArr, (char *)vecPtr->data, offset * sizeof(double));
    }
    if (CnvtExprArr(interp, exprArr, numExpr,
	    &(dataArr[offset]), start, step) != TCL_OK) {
	free((char *)dataArr);
	return TCL_ERROR;
    }
    if (vecPtr->data != NULL) {
	free((char *)vecPtr->data);
    }
    vecPtr->data = dataArr;
    vecPtr->length = arraySize;
    GetDataLimits(vecPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ParseVector --
 *
 *	Given a Tcl list of numeric expression representing the element
 *	values, convert into an array of double precision values. In
 *	addition, the minimum and maximum values are saved.  Since
 *	elastic values are allow (values which translate to the
 *	min/max of the graph), we must try to get the non-elastic
 *	(not +-Infinity) minimum and maximum.
 *
 * Results:
 *	The return value is a standard Tcl result.  The vector is passed
 *	back via the vecPtr.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ParseVector(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;	/* Type of axis vector to fill */
    Tcl_Interp *interp;		/* Interpreter to send results back to */
    Tk_Window tkwin;		/* not used */
    char *value;		/* Tcl list of expressions */
    char *widgRec;		/* Element record */
    int offset;			/* Offset of vector in Element record */
{
    Element *elemPtr = (Element *)(widgRec + offset);
    register Vector *vecPtr;
    enum AxisTypes axisType = (enum AxisTypes)clientData;
    int numExprs;
    char **exprArr = NULL;

    vecPtr = (axisType == X_AXIS_TYPE) ? &(elemPtr->x) : &(elemPtr->y);

    /* Split the list of expressions and check the values */
    if (Tcl_SplitList(interp, value, &numExprs, &exprArr) != TCL_OK) {
	return TCL_ERROR;
    }
    if (numExprs >= 65535) {
	interp->result = "Vector is too large";	/* XDrawLines limit */
	goto error;
    }
    if (AppendVector(interp, vecPtr, exprArr, numExprs, 0, DATA_SINGLE,
	    DATA_OVERWRITE) != TCL_OK) {
	goto error;
    }
    free((char *)exprArr);
    return TCL_OK;

  error:
    /* Clean up, release allocated storage */
    if (exprArr)
	free((char *)exprArr);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * PrintVector --
 *
 *	Convert the vector of floating point values into a Tcl list.
 *
 * Results:
 *	The string representation of the vector is returned.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static char *
PrintVector(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;	/* Type of axis vector to print */
    Tk_Window tkwin;		/* not used */
    char *widgRec;		/* Element record */
    int offset;			/* Offset of vector in Element record */
    Tcl_FreeProc **freeProcPtr;	/* Memory deallocation scheme to use */
{
    Element *elemPtr = (Element *)(widgRec + offset);
    register Vector *vecPtr;
    enum AxisTypes axisType = (enum AxisTypes)clientData;
    register int i;
    char **stringArr;
    char *result;
    char string[TCL_DOUBLE_SPACE];

    vecPtr = (axisType == X_AXIS_TYPE) ? &(elemPtr->x) : &(elemPtr->y);

    result = "";
    if (vecPtr->length == 0) {
	return result;
    }
    stringArr = (char **)calloc(vecPtr->length, sizeof(char *));

    if (stringArr == NULL) {
	return NULL;
    }
    for (i = 0; i < vecPtr->length; i++) {
	Tcl_PrintDouble(elemPtr->interp, vecPtr->data[i], string);
	stringArr[i] = strdup(string);
	if (stringArr[i] == NULL) {
	    goto error;
	}
    }
    result = Tcl_Merge(vecPtr->length, stringArr);
    *freeProcPtr = TCL_DYNAMIC;

  error:
    for (i = 0; i < vecPtr->length; i++) {
	if (stringArr[i] == NULL) {
	    break;
	}
	free(stringArr[i]);
    }
    free((char *)stringArr);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * ParseCoordPairs --
 *
 *	This procedure is like ParseVector except that it
 *	interprets the list of numeric expressions as X Y coordinate
 *	pairs.  The minimum and maximum for both the X and Y vectors are
 *	determined.
 *
 * Results:
 *	The return value is a standard Tcl result.  The vectors are passed
 *	back via the widget record (elemPtr).
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ParseCoordPairs(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;	/* not used */
    Tcl_Interp *interp;		/* Interpreter to send results back to */
    Tk_Window tkwin;		/* not used */
    char *value;		/* Tcl list of numeric expressions */
    char *widgRec;		/* Element record */
    int offset;			/* not used */
{
    Element *elemPtr = (Element *)widgRec;
    int numExprs;
    char **exprArr = NULL;

    /* Split the list of numbers and check the values */
    if (Tcl_SplitList(interp, value, &numExprs, &exprArr) != TCL_OK) {
	return TCL_ERROR;
    }
    if (numExprs >= 131070) {
	interp->result = "Vector is too large";	/* XDrawLines limit */
	goto error;
    } else if (numExprs & 1) {
	interp->result = "Odd number of values in -xydata option";
	goto error;
    }
    if (AppendVector(interp, &(elemPtr->x), exprArr, numExprs, 0, DATA_PAIRS,
	    DATA_OVERWRITE) != TCL_OK) {
	goto error;

    }
    if (AppendVector(interp, &(elemPtr->y), exprArr, numExprs, 1, DATA_PAIRS,
	    DATA_OVERWRITE) != TCL_OK) {
	goto error;
    }
    free((char *)exprArr);
    return TCL_OK;
  error:
    /* Clean up, release allocated storage */
    if (exprArr)
	free((char *)exprArr);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * PrintCoordPairs --
 *
 *	Convert pairs of floating point values in the X and Y arrays
 *	into a Tcl list.
 *
 * Results:
 *	The return value is a string (Tcl list).
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static char *
PrintCoordPairs(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;	/* not used */
    Tk_Window tkwin;		/* not used */
    char *widgRec;		/* Element information record */
    int offset;			/* not used */
    Tcl_FreeProc **freeProcPtr;	/* Memory deallocation scheme to use */
{
    register Element *elemPtr = (Element *)widgRec;
    register int i, count;
    char **stringArr;
    unsigned int arraySize;
    char *result;
    char string[TCL_DOUBLE_SPACE];

    result = "";
    if (elemPtr->x.length == 0 || elemPtr->y.length == 0) {
	return result;
    }
    arraySize = elemPtr->x.length + elemPtr->y.length;
    stringArr = (char **)calloc(arraySize, sizeof(char *));

    if (stringArr == NULL) {
	return NULL;
    }
    count = 0;
    for (i = 0; i < elemPtr->x.length; i++) {
	Tcl_PrintDouble(elemPtr->interp, elemPtr->x.data[i], string);
	stringArr[count] = strdup(string);
	if (stringArr[count] == NULL) {
	    goto error;
	}
	count++;
	Tcl_PrintDouble(elemPtr->interp, elemPtr->y.data[i], string);
	stringArr[count] = strdup(string);
	if (stringArr[count] == NULL) {
	    goto error;
	}
	count++;
    }
    result = Tcl_Merge(arraySize, stringArr);
    *freeProcPtr = TCL_DYNAMIC;

  error:
    for (i = 0; i < arraySize; i++) {
	if (stringArr[i] == NULL) {
	    break;
	}
	free(stringArr[i]);
    }
    free((char *)stringArr);
    return (result);
}

/*
 *----------------------------------------------------------------------
 *
 * ConfigureLine --
 *
 *	Sets up the appropriate configuration parameters in the GC.
 *      It is assumed the parameters have been previously set by
 *	a call to Tk_ConfigureWidget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information such as line width, line style, color
 *	etc. get set in a new GC.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ConfigureLine(graphPtr, elemPtr)
    Graph *graphPtr;
    Element *elemPtr;
{
    LineElement *lineElemPtr = (LineElement *)elemPtr;
    unsigned long gcMask;
    GC newGC;
    XGCValues gcValues;

    /* 
     * Outlines of symbols: need foreground only 
     */
    gcValues.foreground = lineElemPtr->fgColorPtr->pixel;
    newGC = Tk_GetGC(graphPtr->tkwin, GCForeground, &gcValues);
    if (lineElemPtr->outlineGC != NULL) {
	Tk_FreeGC(graphPtr->display, lineElemPtr->outlineGC);
    }
    lineElemPtr->outlineGC = newGC;

    /* 
     * Fills for symbols: need reversed foreground and background
     */
    gcMask = (GCBackground|GCForeground);
    gcValues.background = lineElemPtr->fgColorPtr->pixel;
    gcValues.foreground = lineElemPtr->bgColorPtr->pixel;
    newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
    if (lineElemPtr->fillGC != NULL) {
	Tk_FreeGC(graphPtr->display, lineElemPtr->fillGC);
    }
    lineElemPtr->fillGC = newGC;
    /*
     * Lines 
     */
    gcMask |= (GCLineWidth | GCLineStyle | GCCapStyle | GCJoinStyle);
    gcValues.background = lineElemPtr->bgColorPtr->pixel;
    gcValues.foreground = lineElemPtr->fgColorPtr->pixel;
    gcValues.cap_style = CapButt;
    gcValues.join_style = JoinRound;
    gcValues.line_style = LineSolid;
    gcValues.line_width = lineElemPtr->lineWidth;
    if (lineElemPtr->dashes > 0) {
	gcValues.dash_offset = 0;
	gcValues.line_style = LineDoubleDash;
	gcValues.dashes = lineElemPtr->dashes;
	gcMask |= (GCDashList | GCDashOffset);
    }
    newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
    if (lineElemPtr->lineGC != NULL) {
	Tk_FreeGC(graphPtr->display, lineElemPtr->lineGC);
    }
    lineElemPtr->lineGC = newGC;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * GetLineLimits --
 *
 *	Returns the limits of the line element
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static int
GetLineLimits(elemPtr, axisType, axisScale, minPtr, maxPtr)
    Element *elemPtr;
    enum AxisTypes axisType;	/* Indicates X or Y axis */
    int axisScale;		/* If non-zero, axis is scaled logarithmically.
				 * Need the minimum positive value, instead of
				 * the minimum value (which may be the same) */
    double *minPtr, *maxPtr;
{
    LineElement *lineElemPtr = (LineElement *)elemPtr;
    int numPoints;

    *minPtr = Blt_posInfinity;
    *maxPtr = Blt_negInfinity;
    numPoints = MIN(lineElemPtr->x.length, lineElemPtr->y.length);
    if (numPoints > 0) {
	if (axisType == X_AXIS_TYPE) {
	    *minPtr = (axisScale) ? lineElemPtr->x.logMin : lineElemPtr->x.min;
	    *maxPtr = lineElemPtr->x.max;
	} else if (axisType == Y_AXIS_TYPE) {
	    *minPtr = (axisScale) ? lineElemPtr->y.logMin : lineElemPtr->y.min;
	    *maxPtr = lineElemPtr->y.max;
	}
    }
    return (numPoints);
}

/*
 *----------------------------------------------------------------------
 *
 * LayoutLine --
 *
 *	Calculates the actual window coordinates of the line element.
 *	The window coordinates are saved in an allocated point array.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Memory is (re)allocated for the point array.
 *
 *----------------------------------------------------------------------
 */
static void
LayoutLine(graphPtr, elemPtr)
    Graph *graphPtr;		/* Graph widget record */
    Element *elemPtr;		/* Element component record */
{
    LineElement *lineElemPtr = (LineElement *)elemPtr;
    register int i, n;
    XPoint *pointArr;
    unsigned int numPoints;

    numPoints = MIN(lineElemPtr->x.length, lineElemPtr->y.length);
    if (numPoints < 1) {
	return;			/* No points */
    }
    pointArr = (XPoint *)malloc(numPoints * sizeof(XPoint));

    if (pointArr == NULL) {
	return;			/* Can't allocate point array */
    }
    n = 0;
    for (i = 0; i < numPoints; i++) {
	pointArr[n++] = Blt_WinPt(graphPtr, lineElemPtr->x.data[i],
	    lineElemPtr->y.data[i]);
    }

    /*
     * Free storage of previously allocated point array
     */

    if (lineElemPtr->pointArr != NULL) {
	free((char *)lineElemPtr->pointArr);
    }
    lineElemPtr->symbolSize =
	(int)rint(graphPtr->avgSymSize * lineElemPtr->symbolScale);

    /*
     * Make the symbol size odd so that its center is a single pixel.
     */

    lineElemPtr->symbolSize |= 0x01;
    lineElemPtr->pointArr = pointArr;
    lineElemPtr->numPoints = n;
}

/*
 * -----------------------------------------------------------------
 *
 * DrawLineSymbols --
 *
 * 	Draw a symbol centered at each point given in the array of
 *	window coordinates (coordArr) based upon the element symbol
 *	type and size.
 *
 * Results:
 *	None.
 *
 * Problems:
 *	Most notable is the round-off errors generated when
 *	calculating the centered position of the symbol.
 * -----------------------------------------------------------------
 */
static void
DrawLineSymbols(graphPtr, elemPtr, size, coordArr, numCoords)
    Graph *graphPtr;		/* Pixmap to draw on */
    Element *elemPtr;		/* Element record */
    int size;			/* Size of element */
    XPoint *coordArr;		/* Center x,y coordinate of symbol */
    int numCoords;
{
    LineElement *lineElemPtr = (LineElement *)elemPtr;
    register int i, n;
    XPoint offset[13], points[13];
    register int x, y;
    int radius;

    radius = (size / 2);

    if ((lineElemPtr->symbol == CROSS_SYMBOL) ||
	(lineElemPtr->symbol == PLUS_SYMBOL)) {
	int delta;

	delta = (size / 6);

	/*
	 *
	 *          2   3	The plus/cross symbol is a closed polygon
	 *			of 12 points. The diagram to the left
	 *    0,12  1   4    5	represents the positions of the points
	 *           x,y	which are computed below. The extra
	 *     11  10   7    6	(thirteenth) point connects the first and
	 *			last points.
	 *          9   8
	 */

	offset[0].x = offset[11].x = offset[12].x = -radius;
	offset[2].x = offset[1].x = offset[10].x = offset[9].x =
	    -delta;
	offset[3].x = offset[4].x = offset[7].x = offset[8].x =
	    delta;
	offset[5].x = offset[6].x = radius;
	offset[2].y = offset[3].y = -radius;
	offset[0].y = offset[1].y = offset[4].y = offset[5].y =
	    offset[12].y = -delta;
	offset[11].y = offset[10].y = offset[7].y = offset[6].y =
	    delta;
	offset[9].y = offset[8].y = radius;
	if (lineElemPtr->symbol == CROSS_SYMBOL) {
	    register int i;
	    register double dx, dy;

	    /*
	     * For the cross symbol, rotate the points by 45 degrees.
	     */
	    for (i = 0; i < 12; i++) {
		dx = (double)(offset[i].x) * M_SQRT1_2;
		dy = (double)(offset[i].y) * M_SQRT1_2;
		offset[i].x = (int)rint(dx - dy);
		offset[i].y = (int)rint(dx + dy);
	    }
	    offset[12] = offset[0];
	}
    } else if (lineElemPtr->symbol == DIAMOND_SYMBOL) {

	/*
	 *
	 *             		The plus symbol is a closed polygon
	 *	      1		of 4 points. The diagram to the left
	 *                  	represents the positions of the points
	 *       0,4 x,y  2	which are computed below. The extra
	 *     			(fifth) point connects the first and
	 *	      3		last points.
	 *
	 */

	offset[0].y = offset[4].y = offset[2].y = offset[1].x =
	    offset[3].x = 0;
	offset[1].y = offset[4].x = offset[0].x = -radius;
	offset[3].y = offset[2].x = radius;
    } else if (lineElemPtr->symbol == SQUARE_SYMBOL) {
	size = (int)ceil(size * M_SQRT1_2);
	radius = size / 2;
    } else if (lineElemPtr->symbol == CIRCLE_SYMBOL) {
	size--;
    }
#ifdef notdef
    fprintf(stderr, "%s: %d %d\n", symbolNames[lineElemPtr->symbol], size,
	radius);
#endif
    for (i = 0; i < numCoords; i++) {
	x = coordArr[i].x, y = coordArr[i].y;
	switch (lineElemPtr->symbol) {
	    case LINE_SYMBOL:
		/*
	         * Draw an extra line offset by one pixel from the previous
		 * to give a thicker appearance
		 */
		XDrawLine(graphPtr->display, graphPtr->canvas,
		    lineElemPtr->lineGC, x - radius, y, x + radius, y);
		XDrawLine(graphPtr->display, graphPtr->canvas,
		    lineElemPtr->lineGC, x - radius,
		    y + 1, x + radius, y + 1);
		break;
	    case PLUS_SYMBOL:
	    case CROSS_SYMBOL:
		for (n = 0; n < 13; n++) {
		    points[n].x = offset[n].x + x;
		    points[n].y = offset[n].y + y;
		}
		XFillPolygon(graphPtr->display, graphPtr->canvas,
		    lineElemPtr->fillGC, points, 13, Complex, CoordModeOrigin);
		XDrawLines(graphPtr->display, graphPtr->canvas,
		    lineElemPtr->outlineGC, points, 13, CoordModeOrigin);
		break;
	    case SQUARE_SYMBOL:
		x -= radius, y -= radius;
		XFillRectangle(graphPtr->display, graphPtr->canvas,
		    lineElemPtr->fillGC, x, y, size, size);
		XDrawRectangle(graphPtr->display, graphPtr->canvas,
		    lineElemPtr->outlineGC, x, y, size, size);
		break;
	    case CIRCLE_SYMBOL:
		x -= radius, y -= radius;
		XFillArc(graphPtr->display, graphPtr->canvas,
		    lineElemPtr->fillGC, x, y, size, size, 0, 23040);
		XDrawArc(graphPtr->display, graphPtr->canvas,
		    lineElemPtr->outlineGC, x, y, size, size, 0, 23040);
		break;
	    case DIAMOND_SYMBOL:
		for (n = 0; n < 5; n++) {
		    points[n].x = offset[n].x + x;
		    points[n].y = offset[n].y + y;
		}
		XFillPolygon(graphPtr->display, graphPtr->canvas,
		    lineElemPtr->fillGC, points, 5, Convex, CoordModeOrigin);
		XDrawLines(graphPtr->display, graphPtr->canvas,
		    lineElemPtr->outlineGC, points, 5, CoordModeOrigin);
		break;
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * DisplayLine --
 *
 *	Draws the connected line(s) representing the element.
 *	If the line is made up of non-line symbols and the line
 *	width parameter has been set (linewidth > 0), the element
 *	will also be drawn as a line (with the linewidth requested).
 *	The line may consist of separate line segments depending if a)
 *	the preceding x coordinate value is less than the current
 *	and b) the line -noretrace option is turned on.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	X drawing commands are output.
 *
 *----------------------------------------------------------------------
 */
static void
DisplayLine(graphPtr, elemPtr)
    Graph *graphPtr;
    Element *elemPtr;
{
    LineElement *lineElemPtr = (LineElement *)elemPtr;
    register int n;

    if (lineElemPtr->numPoints < 1) {
	return;
    }

    /* 
     * Draw lines if symbol is "line" or linewidth is greater than 0 
     */
    if ((lineElemPtr->lineWidth > 0) || (lineElemPtr->symbol == LINE_SYMBOL)) {
	int lastX, curX;
	int numPoints = 0;
	int start;

	lastX = 0;
	for (n = 0; n < lineElemPtr->numPoints; n++) {
	    curX = lineElemPtr->pointArr[n].x;
	    if ((lineElemPtr->noretrace) && (numPoints > 0) &&
		(curX < lastX)) {
		start = n - numPoints;
		XDrawLines(graphPtr->display, graphPtr->canvas,
		    lineElemPtr->lineGC, &(lineElemPtr->pointArr[start]),
		    numPoints, CoordModeOrigin);
		numPoints = 0;
	    }
	    numPoints++;
	    lastX = curX;
	}
	if (numPoints > 0) {
	    start = n - numPoints;
	    XDrawLines(graphPtr->display, graphPtr->canvas,
		lineElemPtr->lineGC, &(lineElemPtr->pointArr[start]),
		numPoints, CoordModeOrigin);
	}
    }
    /* Draw non-line symbols */

    if (lineElemPtr->symbol != LINE_SYMBOL) {
	(*lineElemPtr->drawSymbolsProc) (graphPtr, (Element *)lineElemPtr,
	    lineElemPtr->symbolSize, lineElemPtr->pointArr,
	    lineElemPtr->numPoints);
    }
}

/*
 * -----------------------------------------------------------------
 *
 * PrintLineSymbols --
 *
 * 	Draw a symbol centered at the given x,y window coordinate
 *	based upon the element symbol type and size.
 *
 * Results:
 *	None.
 *
 * Problems:
 *	Most notable is the round-off errors generated when
 *	calculating the centered position of the symbol.
 * -----------------------------------------------------------------
 */
static void
PrintLineSymbols(graphPtr, elemPtr, size, pointArr, numPoints)
    Graph *graphPtr;
    Element *elemPtr;
    int size;
    XPoint *pointArr;
    int numPoints;
{
    LineElement *lineElemPtr = (LineElement *)elemPtr;
    double symbolSize;
    register int i;
    int x, y;

    symbolSize = (double)size;
    if (elemPtr->symbol == LINE_SYMBOL) {
	Blt_LineWidthToPostScript(graphPtr, lineElemPtr->lineWidth + 2);
	Blt_LineDashesToPostScript(graphPtr, lineElemPtr->dashes);
    } else {
	if ((elemPtr->symbol == SQUARE_SYMBOL) ||
	    (elemPtr->symbol == DIAMOND_SYMBOL)) {
	    /* Adjust square and diamond symbol sizes */
	    symbolSize = ((double)size * M_SQRT1_2) - 1;
	}
	Blt_LineWidthToPostScript(graphPtr, lineElemPtr->lineWidth);
    }
    Tcl_AppendResult(graphPtr->interp, "/BgColorProc {\n   ", (char *)NULL);
    Blt_BackgroundToPostScript(graphPtr, lineElemPtr->bgColorPtr);
    Tcl_AppendResult(graphPtr->interp, "} def\n", (char *)NULL);
    Blt_ForegroundToPostScript(graphPtr, lineElemPtr->fgColorPtr);
    Blt_LineDashesToPostScript(graphPtr, 0);
    for (i = 0; i < numPoints; i++) {
	x = pointArr[i].x, y = pointArr[i].y;
	sprintf(graphPtr->scratchPtr, "%d %d %g %.2s\n", x, y, symbolSize,
	    symbolNames[(int)lineElemPtr->symbol]);
	Tcl_AppendResult(graphPtr->interp, graphPtr->scratchPtr, (char *)NULL);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * PrintLine --
 *
 *	Similar to the DrawLine procedure, prints PostScript related
 *	commands to form the connected line(s) representing the element.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	PostScript pen width, dashes, and color settings are changed.
 *
 *----------------------------------------------------------------------
 */
static void
PrintLine(graphPtr, elemPtr)
    Graph *graphPtr;
    Element *elemPtr;
{
    LineElement *lineElemPtr = (LineElement *)elemPtr;
    register int n;

    if (lineElemPtr->numPoints < 1) {
	return;
    }
    Blt_LineWidthToPostScript(graphPtr, lineElemPtr->lineWidth);
    Blt_ForegroundToPostScript(graphPtr, lineElemPtr->fgColorPtr);
    if (lineElemPtr->dashes > 0) {
	Blt_LineDashesToPostScript(graphPtr, lineElemPtr->dashes);

	Tcl_AppendResult(graphPtr->interp, "/DashesProc {\ngsave\n",
	    (char *)NULL);
	Blt_BackgroundToPostScript(graphPtr, lineElemPtr->bgColorPtr);
	Blt_LineDashesToPostScript(graphPtr, 0);
	Tcl_AppendResult(graphPtr->interp, "stroke grestore\n} def\n",
	    (char *)NULL);
    } else {
	Tcl_AppendResult(graphPtr->interp, "/DashesProc {} def\n",
	    (char *)NULL);
    }
    if ((lineElemPtr->lineWidth > 0) || (lineElemPtr->symbol == LINE_SYMBOL)) {
	int lastX, curX;
	int numPoints = 0;

	sprintf(graphPtr->scratchPtr, "/curX %d def\n/curY %d def\n",
	    lineElemPtr->pointArr[0].x, lineElemPtr->pointArr[0].y);
	Tcl_AppendResult(graphPtr->interp, graphPtr->scratchPtr, (char *)NULL);

	lastX = lineElemPtr->pointArr[0].x;
	for (n = 1; n < lineElemPtr->numPoints; n++) {
	    curX = lineElemPtr->pointArr[n].x;
	    if (numPoints > 0) {
		if ((!lineElemPtr->noretrace) || (curX >= lastX)) {
		    sprintf(graphPtr->scratchPtr, "%d %d DrawLine\n",
			lineElemPtr->pointArr[n].x,
			lineElemPtr->pointArr[n].y);
		    Tcl_AppendResult(graphPtr->interp, graphPtr->scratchPtr,
			(char *)NULL);
		} else {
		    numPoints = 0;
		    sprintf(graphPtr->scratchPtr, 
			"/curX %d def\n/curY %d def\n", 
			lineElemPtr->pointArr[n].x, lineElemPtr->pointArr[n].y);
		    Tcl_AppendResult(graphPtr->interp, graphPtr->scratchPtr,
			(char *)NULL);
		}
	    }
	    numPoints++;
	    lastX = curX;
	}
    }
    /* Draw non-line symbols */

    if (lineElemPtr->symbol != LINE_SYMBOL) {
	(*lineElemPtr->printSymbolsProc) (graphPtr, (Element *)lineElemPtr,
	    lineElemPtr->symbolSize, lineElemPtr->pointArr,
	    lineElemPtr->numPoints);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * DestroyLine --
 *
 *	Release memory and resources allocated for the line element.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the line element is freed up.
 *
 *----------------------------------------------------------------------
 */
static void
DestroyLine(graphPtr, elemPtr)
    Graph *graphPtr;
    Element *elemPtr;
{
    LineElement *lineElemPtr = (LineElement *)elemPtr;

    Tk_FreeOptions(lineElemPtr->configSpecs, (char *)lineElemPtr,
	graphPtr->display, 0);

    if (lineElemPtr->outlineGC != NULL) {
	Tk_FreeGC(graphPtr->display, lineElemPtr->outlineGC);
    }
    if (lineElemPtr->fillGC != NULL) {
	Tk_FreeGC(graphPtr->display, lineElemPtr->fillGC);
    }
    if (lineElemPtr->lineGC != NULL) {
	Tk_FreeGC(graphPtr->display, lineElemPtr->lineGC);
    }
    if (lineElemPtr->pointArr != NULL) {
	free((char *)lineElemPtr->pointArr);
    }
    if (lineElemPtr->x.data != NULL) {
	free((char *)lineElemPtr->x.data);
    }
    if (lineElemPtr->y.data != NULL) {
	free((char *)lineElemPtr->y.data);
    }
    free((char *)lineElemPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * CreateLine --
 *
 *	Allocate memory and initialize methods for the new line element.
 *
 * Results:
 *	The pointer to the newly allocated element structure is returned.
 *
 * Side effects:
 *	Memory is allocated for the line element structure.
 *
 *----------------------------------------------------------------------
 */
static Element *
CreateLine()
{
    register LineElement *lineElemPtr;

    lineElemPtr = (LineElement *)calloc(1, sizeof(LineElement));

    if (lineElemPtr == NULL) {
	return NULL;
    }
    lineElemPtr->configSpecs = lineConfigSpecs;
    lineElemPtr->configProc = ConfigureLine;
    lineElemPtr->destroyProc = DestroyLine;
    lineElemPtr->displayProc = DisplayLine;
    lineElemPtr->limitsProc = GetLineLimits;
    lineElemPtr->layoutProc = LayoutLine;
    lineElemPtr->printProc = PrintLine;
    lineElemPtr->drawSymbolsProc = DrawLineSymbols;
    lineElemPtr->printSymbolsProc = PrintLineSymbols;
    lineElemPtr->type = LINE_ELEM_TYPE;
    lineElemPtr->symbolScale = 1.0;
    lineElemPtr->symbol = LINE_SYMBOL;
    lineElemPtr->noretrace = 0;
    return ((Element *)lineElemPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * ConfigureBar --
 *
 *	Sets up the appropriate configuration parameters in the GC.
 *      It is assumed the parameters have been previously set by
 *	a call to Tk_ConfigureWidget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information such as bar foreground/background
 *	color and stipple etc. get set in a new GC.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ConfigureBar(graphPtr, elemPtr)
    Graph *graphPtr;
    register Element *elemPtr;
{
    GC newGC;
    XGCValues gcValues;
    unsigned long gcMask;
    BarElement *barElemPtr = (BarElement *)elemPtr;

    gcValues.foreground = barElemPtr->fgColorPtr->pixel;
    gcValues.background = (Tk_3DBorderColor(barElemPtr->border))->pixel;
    gcMask = GCForeground | GCBackground;;
    if (barElemPtr->stipple != None) {
	gcValues.stipple = barElemPtr->stipple;
	gcValues.fill_style = FillOpaqueStippled;
	gcMask |= (GCStipple | GCFillStyle);
    }
    barElemPtr->symbol = SQUARE_SYMBOL;	/* Use square for bar charts */
    newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
    if (barElemPtr->gc != NULL) {
	Tk_FreeGC(graphPtr->display, barElemPtr->gc);
    }
    barElemPtr->gc = newGC;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * GetBarLimits --
 *
 *	Returns the limits of the bar data
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static int
GetBarLimits(elemPtr, axisType, axisScale, minPtr, maxPtr)
    Element *elemPtr;
    enum AxisTypes axisType;	/* Indicates X or Y axis */
    int axisScale;		/* If non-zero, axis is scaled logarithmically.
				 * Need the minimum positive value, instead of
				 * the minimum value (which may be the same) */
    double *minPtr, *maxPtr;
{
    BarElement *barElemPtr = (BarElement *)elemPtr;
    int numPoints;

    *minPtr = Blt_posInfinity, *maxPtr = Blt_negInfinity;

    if ((barElemPtr->x.length < 1) || (barElemPtr->y.length < 1)) {
	return 0;
    }
    numPoints = barElemPtr->y.length;
    if (axisType == X_AXIS_TYPE) {
	*minPtr = (axisScale) ? barElemPtr->x.logMin : barElemPtr->x.min;
	*minPtr -= 0.5;
	*maxPtr = barElemPtr->x.max + 0.5;
    } else if (axisType == Y_AXIS_TYPE) {
	*minPtr = (axisScale) ? barElemPtr->y.logMin : barElemPtr->y.min;
	if (barElemPtr->stacked) {
	    register int i;
	    double sum = 0.0;

	    for (i = 0; i < barElemPtr->y.length; i++) {
		sum += barElemPtr->y.data[i];
	    }
	    *maxPtr = sum;
	} else {
	    *maxPtr = barElemPtr->y.max;
	}
    }
    return (numPoints);
}

/*
 *----------------------------------------------------------------------
 *
 * LayoutBar --
 *
 *	Calculates the actual window coordinates of the bar element.
 *	The window coordinates are saved in the bar element structure.
 *
 * Results:
 *	None.
 *
 * Notes:
 *	A bar can have multiple y values.  In this case, the bar can be
 * 	represented as either a set of contiguous bars (no spacing) or a
 *      single multi-segmented (stacked) bar.
 *
 *	The X axis layout for a barchart may be presented in one of two ways.
 *	If x values are used, the bars are placed at those coordinates.
 *	Otherwise, the range will represent the number of values.
 *
 *----------------------------------------------------------------------
 */
static void
LayoutBar(graphPtr, elemPtr)
    Graph *graphPtr;
    Element *elemPtr;
{
    BarElement *barElemPtr = (BarElement *)elemPtr;
    double sum;			/* Sum of bar y values */
    int curX, curY;		/* Current and last Y positions */
    int lastY;			/* Last y-coordinate used */
    XRectangle *rectArr;
    int avgWidth;		/* Width of each bar in set */
    register int i;

    if ((barElemPtr->x.length < 1) || (barElemPtr->y.length < 1)) {
	return;			/* No bars */
    }
    barElemPtr->numRects = barElemPtr->y.length;
    rectArr = (XRectangle *)malloc(barElemPtr->numRects * sizeof(XRectangle));

    if (rectArr == NULL) {
	barElemPtr->numRects = 0;
	return;
    }
    curY = lastY = Blt_WinY(graphPtr, 0.0);
    /* 
     * Watch out for limiting of values 
     */
    avgWidth = Blt_WinXAbs(graphPtr, graphPtr->barWidth) - 
	Blt_WinXAbs(graphPtr, 0.0);
    if (avgWidth < 1) {
	avgWidth = 1;
    }
    /* Use only the first data value in the X array for positioning */
    curX = Blt_WinX(graphPtr, barElemPtr->x.data[0]) - avgWidth / 2;
    sum = 0.0;
    if (!barElemPtr->stacked) {
	avgWidth /= barElemPtr->y.length;
    }
    for (i = 0; i < barElemPtr->y.length; i++) {
	rectArr[i].width = avgWidth - (barElemPtr->padX * 2);
	rectArr[i].x = curX;
	if (barElemPtr->stacked) {
	    /* Next point is the current sum the y values */
	    sum += barElemPtr->y.data[i];
	    lastY = curY;
	    curY = Blt_WinY(graphPtr, sum);
	} else {
	    /* Adjust the x-coordinate by the next bar width */
	    curY = Blt_WinY(graphPtr, barElemPtr->y.data[i]);
	    curX += avgWidth;
	}
	if (barElemPtr->y.max < 0.0) {
	    /* Rectangle y-origin is last y-coordinate */
	    rectArr[i].y = lastY;
	    rectArr[i].height = (curY - lastY) + 1;
	} else {
	    /* Rectangle y-origin is current y-coordinate */
	    rectArr[i].y = curY;
	    rectArr[i].height = (lastY - curY) + 1;
	}
    }
    barElemPtr->symbolSize =
	(int)rint(graphPtr->avgSymSize * barElemPtr->symbolScale);
    if (barElemPtr->rectArr != NULL) {
	free((char *)barElemPtr->rectArr);
    }
    barElemPtr->rectArr = rectArr;
}

/*
 * -----------------------------------------------------------------
 *
 * DrawBarSymbols --
 *
 * 	Draw a symbol centered at the given x,y window coordinate
 *	based upon the element symbol type and size.
 *
 * Results:
 *	None.
 *
 * Problems:
 *	Most notable is the round-off errors generated when
 *	calculating the centered position of the symbol.
 * -----------------------------------------------------------------
 */
static void
DrawBarSymbols(graphPtr, elemPtr, size, pointArr, numPoints)
    Graph *graphPtr;
    Element *elemPtr;
    int size;
    XPoint *pointArr;
    int numPoints;
{
    BarElement *barElemPtr = (BarElement *)elemPtr;
    register int i;
    int x, y, radius;

    radius = (size / 2);
    size--;
    for (i = 0; i < numPoints; i++) {
	x = pointArr[i].x - radius;
	y = pointArr[i].y - radius;
	XFillRectangle(graphPtr->display, graphPtr->canvas, barElemPtr->gc,
	    x, y, size, size);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * DrawBar --
 *
 *	Draws the rectangle representing the bar element.
 *	If the relief option is set to "raised" or "sunken" and the
 *	bar borderwidth is set (borderwidth > 0), a 3D border is
 *	drawn around the bar.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	X drawing commands are output.
 *
 *----------------------------------------------------------------------
 */
static void
DisplayBar(graphPtr, elemPtr)
    Graph *graphPtr;
    Element *elemPtr;
{
    BarElement *barElemPtr = (BarElement *)elemPtr;

    if (barElemPtr->numRects > 0) {
	XFillRectangles(graphPtr->display, graphPtr->canvas, barElemPtr->gc,
	    barElemPtr->rectArr, barElemPtr->numRects);

	/* Add 3D border to each bar segment */

	if ((barElemPtr->borderWidth > 0) &&
	    (barElemPtr->relief != TK_RELIEF_FLAT)) {
	    register int i;

	    for (i = 0; i < barElemPtr->numRects; i++) {
		Tk_Draw3DRectangle(graphPtr->display, graphPtr->canvas,
		    barElemPtr->border, barElemPtr->rectArr[i].x,
		    barElemPtr->rectArr[i].y, barElemPtr->rectArr[i].width,
		    barElemPtr->rectArr[i].height, barElemPtr->borderWidth,
		    barElemPtr->relief);
	    }
	}
    }
}

/*
 * -----------------------------------------------------------------
 *
 * PrintBarSymbols --
 *
 * 	Draw a symbol centered at the given x,y window coordinate
 *	based upon the element symbol type and size.
 *
 * Results:
 *	None.
 *
 * Problems:
 *	Most notable is the round-off errors generated when
 *	calculating the centered position of the symbol.
 * -----------------------------------------------------------------
 */
static void
PrintBarSymbols(graphPtr, elemPtr, size, pointArr, numPoints)
    Graph *graphPtr;
    Element *elemPtr;
    int size;
    XPoint *pointArr;
    int numPoints;
{
    BarElement *barElemPtr = (BarElement *)elemPtr;
    register int i;
    int x, y;

    Tcl_AppendResult(graphPtr->interp, "/BgColorProc {\n", (char *)NULL);
    Blt_BackgroundToPostScript(graphPtr, Tk_3DBorderColor(barElemPtr->border));
    Tcl_AppendResult(graphPtr->interp, "} def\n", (char *)NULL);

    if (barElemPtr->stipple != None) {
	unsigned int width, height;

	Tcl_AppendResult(graphPtr->interp, "/StippleProc {\n    gsave\n    ",
	    (char *)NULL);
	Tk_SizeOfBitmap(graphPtr->display, barElemPtr->stipple, &width,
	    &height);
	Blt_StippleToPostScript(graphPtr, barElemPtr->stipple, width, height, 1);
	Tcl_AppendResult(graphPtr->interp, "} def\n", (char *)NULL);
    }
    Blt_ForegroundToPostScript(graphPtr, barElemPtr->fgColorPtr);
    for (i = 0; i < numPoints; i++) {
	x = pointArr[i].x, y = pointArr[i].y;
	sprintf(graphPtr->scratchPtr, "%d %d %d %.2s\n", x, y, size,
	    symbolNames[(int)elemPtr->symbol]);
	Tcl_AppendResult(graphPtr->interp, graphPtr->scratchPtr, (char *)NULL);
	if (barElemPtr->stipple != None) {
	    Tcl_AppendResult(graphPtr->interp, "/StippleProc cvx exec\n",
		(char *)NULL);
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * PrintBar --
 *
 *	Similar to the DrawBar procedure, prints PostScript related
 *	commands to form rectangle representing the bar element.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	PostScript pen width, dashes, and color settings are changed.
 *
 *----------------------------------------------------------------------
 */
static void
PrintBar(graphPtr, elemPtr)
    Graph *graphPtr;
    Element *elemPtr;
{
    BarElement *barElemPtr = (BarElement *)elemPtr;
    register int i;

    for (i = 0; i < barElemPtr->numRects; i++) {
	if (barElemPtr->stipple != None) {
	    unsigned int width, height;

	    Blt_BackgroundToPostScript(graphPtr,
		Tk_3DBorderColor(barElemPtr->border));
	    Blt_RectangleToPostScript(graphPtr, barElemPtr->rectArr[i].x,
		barElemPtr->rectArr[i].y, barElemPtr->rectArr[i].width,
		barElemPtr->rectArr[i].height);
	    Tk_SizeOfBitmap(graphPtr->display, barElemPtr->stipple,
		&width, &height);
	    Blt_ForegroundToPostScript(graphPtr, barElemPtr->fgColorPtr);
	    Blt_StippleToPostScript(graphPtr, barElemPtr->stipple, width,
		height, True);
	} else {
	    Blt_ForegroundToPostScript(graphPtr, barElemPtr->fgColorPtr);
	    Blt_RectangleToPostScript(graphPtr, barElemPtr->rectArr[i].x,
		barElemPtr->rectArr[i].y, barElemPtr->rectArr[i].width,
		barElemPtr->rectArr[i].height);
	}
	if ((barElemPtr->borderWidth > 0) &&
	    (barElemPtr->relief != TK_RELIEF_FLAT)) {
	    Blt_Print3DRectangle(graphPtr, barElemPtr->border,
		barElemPtr->rectArr[i].x, barElemPtr->rectArr[i].y,
		barElemPtr->rectArr[i].width,
		barElemPtr->rectArr[i].height, barElemPtr->borderWidth,
		barElemPtr->relief);
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * DestroyBar --
 *
 *	Release memory and resources allocated for the bar element.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the bar element is freed up.
 *
 *----------------------------------------------------------------------
 */
static void
DestroyBar(graphPtr, elemPtr)
    Graph *graphPtr;
    Element *elemPtr;
{
    BarElement *barElemPtr = (BarElement *)elemPtr;

    Tk_FreeOptions(barElemPtr->configSpecs, (char *)barElemPtr,
	graphPtr->display, 0);

    if (barElemPtr->gc != NULL) {
	Tk_FreeGC(graphPtr->display, barElemPtr->gc);
    }
    if (barElemPtr->rectArr != NULL) {
	free((char *)barElemPtr->rectArr);
    }
    if (barElemPtr->x.data != NULL) {
	free((char *)barElemPtr->x.data);
    }
    if (barElemPtr->y.data != NULL) {
	free((char *)barElemPtr->y.data);
    }
    free((char *)barElemPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * CreateBar --
 *
 *	Allocate memory and initialize methods for the new bar element.
 *
 * Results:
 *	The pointer to the newly allocated element structure is returned.
 *
 * Side effects:
 *	Memory is allocated for the bar element structure.
 *
 *----------------------------------------------------------------------
 */
static Element *
CreateBar()
{
    register BarElement *barElemPtr;

    barElemPtr = (BarElement *)calloc(1, sizeof(BarElement));

    if (barElemPtr == NULL) {
	return NULL;
    }
    barElemPtr->configSpecs = barConfigSpecs;
    barElemPtr->configProc = ConfigureBar;
    barElemPtr->destroyProc = DestroyBar;
    barElemPtr->displayProc = DisplayBar;
    barElemPtr->limitsProc = GetBarLimits;
    barElemPtr->layoutProc = LayoutBar;
    barElemPtr->printProc = PrintBar;
    barElemPtr->drawSymbolsProc = DrawBarSymbols;
    barElemPtr->printSymbolsProc = PrintBarSymbols;
    barElemPtr->type = BAR_ELEM_TYPE;
    barElemPtr->relief = TK_RELIEF_FLAT;
    barElemPtr->symbolScale = 1.0;
    barElemPtr->stacked = 0;
    barElemPtr->stipple = None;
    return ((Element *)barElemPtr);
}

/*
 * Generic element routines:
 */
/*
 *----------------------------------------------------------------------
 *
 * CreateElement --
 *
 *	Add a new element to the graph.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
static int
CreateElement(graphPtr, argc, argv)
    Graph *graphPtr;
    int argc;
    char **argv;
{
    register Element *elemPtr;
    Tcl_HashEntry *entryPtr;
    Blt_ListEntry *listPtr;
    int dummy;

    if (argc < 4) {
	Tcl_AppendResult(graphPtr->interp, "wrong # args: should be \"",
	    argv[0], " element create name ?options?\"", (char *)NULL);
	return TCL_ERROR;
    }
    entryPtr = Tcl_FindHashEntry(&(graphPtr->elemTable), argv[3]);
    if (entryPtr != NULL) {
	Tcl_AppendResult(graphPtr->interp, "element \"", argv[3],
	    "\" already exists in \"", argv[0], "\"", (char *)NULL);
	return TCL_ERROR;
    }
    elemPtr = NULL;
    if (graphPtr->type == BAR_ELEM_TYPE) {
	elemPtr = CreateBar();
    } else if (graphPtr->type == LINE_ELEM_TYPE) {
	elemPtr = CreateLine();
    } 
    if (elemPtr == NULL) {
	Tcl_AppendResult(graphPtr->interp, "can't create element \"",
	    argv[3], "\"", (char *)NULL);
	return TCL_ERROR;
    }
    elemPtr->interp = graphPtr->interp;
    elemPtr->id = Tk_GetUid(argv[3]);
    elemPtr->label = (char *)strdup(argv[3]);
    if (Tk_ConfigureWidget(graphPtr->interp, graphPtr->tkwin,
	    elemPtr->configSpecs, argc - 4, argv + 4, 
			   (char *)elemPtr, 0) != TCL_OK) {
	(*elemPtr->destroyProc) (graphPtr, elemPtr);
	return TCL_ERROR;
    }
    entryPtr = Tcl_CreateHashEntry(&(graphPtr->elemTable),
	(char *)elemPtr->id, &dummy);
    Tcl_SetHashValue(entryPtr, (ClientData)elemPtr);
    (*elemPtr->configProc) (graphPtr, elemPtr);
    listPtr = Blt_CreateListEntry(elemPtr->id);
    Blt_SetListValue(listPtr, (ClientData)elemPtr);
    Blt_LinkListAfter(&(graphPtr->elemList), listPtr,
	(Blt_ListEntry *)NULL);
    elemPtr->mapped = 1;
    elemPtr->flags |= LAYOUT_PENDING;
    Blt_ComputeAxes(graphPtr);
    Blt_EventuallyRedraw(graphPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ConfigureElement --
 *
 *	Sets the element specifications by the given the command line
 *	arguments and calls the element specification configuration
 *	routine. If zero or one command line options are given, only
 *	information about the option(s) is returned in interp->result.
 *      If the element configuration has changed and the element is
 *	currently displayed, the axis limits are updated and recomputed.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 * Side Effects:
 *	Graph will be redrawn to reflect the new display list.
 *
 *----------------------------------------------------------------------
 */
static int
ConfigureElement(graphPtr, argc, argv)
    Graph *graphPtr;
    int argc;
    char *argv[];
{
    Tcl_HashEntry *entryPtr;
    Element *elemPtr;
    int flags = TK_CONFIG_ARGV_ONLY;
    int result;

    if (argc < 4) {
	Tcl_AppendResult(graphPtr->interp, "wrong # args: should be \"",
	    argv[0], " element configure name ?options?\"", (char *)NULL);
	return TCL_ERROR;
    }
    entryPtr = Tcl_FindHashEntry(&(graphPtr->elemTable), argv[3]);
    if (entryPtr == NULL) {
	Tcl_AppendResult(graphPtr->interp, "can't find element \"", argv[3],
	    "\" in \"", argv[0], "\"", (char *)NULL);
	return TCL_ERROR;
    }
    elemPtr = (Element *)Tcl_GetHashValue(entryPtr);
    if (argc == 4) {
	return (Tk_ConfigureInfo(graphPtr->interp, graphPtr->tkwin,
		elemPtr->configSpecs, (char *)elemPtr, (char *)NULL, flags));
    } else if (argc == 5) {
	return (Tk_ConfigureInfo(graphPtr->interp, graphPtr->tkwin,
		elemPtr->configSpecs, (char *)elemPtr, argv[4], flags));
    }
    result = Tk_ConfigureWidget(graphPtr->interp, graphPtr->tkwin,
	elemPtr->configSpecs, argc - 4, argv + 4, (char *)elemPtr, flags);
    if ((*elemPtr->configProc) (graphPtr, elemPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    if (result != TCL_OK) {
	return TCL_ERROR;
    }
    elemPtr->flags |= LAYOUT_PENDING;
    graphPtr->flags |= (LAYOUT_PENDING | REDRAW_ALL);
    Blt_ComputeAxes(graphPtr);
    Blt_EventuallyRedraw(graphPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * AppendElement --
 *
 *	Given a Tcl list of numeric expression representing the element
 *	values, convert into an array of double precision values. In
 *	addition, the minimum and maximum values are saved.  Since
 *	elastic values are allow (values which translate to the
 *	min/max of the graph), we must try to get the non-elastic
 *	(not +-Infinity) minimum and maximum.
 *
 * Results:
 *	The return value is a standard Tcl result.  The vector is passed
 *	back via the vecPtr.
 *
 *----------------------------------------------------------------------
 */
static int
AppendElement(graphPtr, argc, argv)
    Graph *graphPtr;
    int argc;
    char **argv;
{
    Tcl_Interp *interp = graphPtr->interp;
    register Vector *vecPtr;
    int numExprs;
    char **exprArr = NULL;
    int result = TCL_ERROR;
    unsigned int arraySize;
    register Element *elemPtr;
    Tcl_HashEntry *entryPtr;

    if (argc != 5) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
	    " element append name coords\"", NULL);
	return TCL_ERROR;
    }
    entryPtr = Tcl_FindHashEntry(&(graphPtr->elemTable), argv[3]);
    if (entryPtr == NULL) {
	Tcl_AppendResult(interp, "can't find element \"", argv[3],
	    "\" in \"", argv[0], "\"", (char *)NULL);
	return TCL_ERROR;
    }
    elemPtr = (Element *)Tcl_GetHashValue(entryPtr);

    /* Split the list of expressions and check the values */

    if (Tcl_SplitList(interp, argv[4], &numExprs, &exprArr) != TCL_OK) {
	return TCL_ERROR;
    }
    if (numExprs < 2) {
	interp->result = "too few numeric expressions in coordinate list";
	goto error;
    }
    vecPtr = &(elemPtr->x);
    arraySize = (numExprs / 2) + vecPtr->length;
    if (arraySize >= 65535) {
	interp->result = "X vector is too large";	/* XDrawLines limit */
	goto error;
    }
    if (AppendVector(interp, vecPtr, exprArr, numExprs, 0, DATA_PAIRS,
	    DATA_APPEND) != TCL_OK) {
	goto error;
    }
    vecPtr = &(elemPtr->y);
    arraySize = (numExprs / 2) + vecPtr->length;
    if (arraySize >= 65535) {
	interp->result = "Y vector is too large";	/* XDrawLines limit */
	goto error;
    }
    if (AppendVector(interp, vecPtr, exprArr, numExprs, 1, DATA_PAIRS,
	    DATA_APPEND) != TCL_OK) {
	goto error;
    }
    result = TCL_OK;
    Blt_ComputeAxes(graphPtr);
    Blt_EventuallyRedraw(graphPtr);
  error:
    if (exprArr != NULL) {
	free((char *)exprArr);
    }
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * DeleteElements --
 *
 *	Delete the named elements from the graph.
 *
 * Results:
 *	TCL_ERROR is returned if any of the named elements can not be found.
 *	Otherwise TCL_OK is returned;
 *
 * Side Effects:
 *	If the element is currently mapped, the plotting area of the
 *	graph is redrawn. Memory and resources allocated by the elements
 *	are released.
 *
 *----------------------------------------------------------------------
 */
static int
DeleteElements(graphPtr, argc, argv)
    Graph *graphPtr;		/* Graph widget */
    int argc;			/* Number of element names */
    char **argv;		/* List of element names */
{
    register Element *elemPtr;
    Tcl_HashEntry *entryPtr;
    Blt_ListEntry *listPtr;
    register int i;
    int count;

    if (argc < 4) {
	Tcl_AppendResult(graphPtr->interp, "wrong # args: should be \"",
	    argv[0], " element delete name ?name...?\"", (char *)NULL);
	return TCL_ERROR;
    }
    count = 0;
    for (i = 3; i < argc; i++) {
	entryPtr = Tcl_FindHashEntry(&(graphPtr->elemTable), argv[i]);
	if (entryPtr == NULL) {
	    Tcl_AppendResult(graphPtr->interp, "can't find element \"",
		argv[i], "\" in \"", argv[0], (char *)NULL);
	    return TCL_ERROR;
	}
	elemPtr = (Element *)Tcl_GetHashValue(entryPtr);
	Tcl_DeleteHashEntry(entryPtr);

	listPtr = Blt_FindListEntry(&(graphPtr->elemList), argv[i]);
	if (listPtr != NULL) {
	    count++;
	    Blt_DeleteListEntry(&(graphPtr->elemList), listPtr);
	}
	/*
	 * Call the element's own destructor to release the memory and
	 * resources allocated for it.
	 */
	(*elemPtr->destroyProc) (graphPtr, elemPtr);
    }
    if (count > 0) {
	Blt_ComputeAxes(graphPtr);
	Blt_EventuallyRedraw(graphPtr);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ElementNames --
 *
 *	Runs through the given list of element entries and builds a
 *	Tcl list of element names.  This procedure is used in the
 *	"names" and "show" commands.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *	interp->result contains the list of element names.
 *
 *----------------------------------------------------------------------
 */
static int
ElementNames(graphPtr, argc, argv)
    Graph *graphPtr;
    int argc;
    char **argv;
{
    register Element *elemPtr;
    register Tcl_HashEntry *entryPtr;
    Tcl_HashSearch cursor;

    if ((argc < 3) || (argc > 4)) {
	Tcl_AppendResult(graphPtr->interp, "wrong # args: should be \"",
	    argv[0], " element names ?pattern?", (char *)NULL);
	return TCL_ERROR;
    }
    for (entryPtr = Tcl_FirstHashEntry(&(graphPtr->elemTable), &cursor);
	entryPtr != NULL; entryPtr = Tcl_NextHashEntry(&cursor)) {
	elemPtr = (Element *)Tcl_GetHashValue(entryPtr);
	if ((argc == 3) || Tcl_StringMatch(elemPtr->id, argv[3])) {
	    Tcl_AppendElement(graphPtr->interp, elemPtr->id);
	}
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ResetDisplayList --
 *
 *	Given a Tcl list of element names, this procedure rebuilds the
 *	display list, ignoring invalid element names. This list describes
 *	not only only which elements to draw, but in what order.  This is
 *	only important for bar and pie charts.
 *
 * Results:
 *	The return value is a standard Tcl result.  Only if the Tcl list
 *	cannot be split, a TCL_ERROR is returned and interp->result contains
 *	an error message.
 *
 * Side effects:
 *	The graph is eventually redrawn using the new display list.
 *
 *----------------------------------------------------------------------
 */
static int
ResetDisplayList(graphPtr, newList)
    Graph *graphPtr;		/* Graph widget record */
    char *newList;		/* Tcl list of element names */
{
    int numNames;		/* Number of names found in Tcl name list */
    char **nameArr;		/* Broken out array of element names */
    Blt_ListEntry *listPtr;
    Tcl_HashEntry *hashPtr;
    Tcl_HashSearch cursor;
    register int count;
    Element *elemPtr;		/* Element information record */

    if (Tcl_SplitList(graphPtr->interp, newList, &numNames,
	    &nameArr) != TCL_OK) {
	Tcl_AppendResult(graphPtr->interp, "can't split name list \"", newList,
	    "\"", (char *)NULL);
	return TCL_ERROR;
    }
    /*
     * Clear the display list and mark all elements unmapped.
     */
    Blt_ClearList(&(graphPtr->elemList));
    for (hashPtr = Tcl_FirstHashEntry(&(graphPtr->elemTable), &cursor);
	hashPtr != NULL; hashPtr = Tcl_NextHashEntry(&cursor)) {
	elemPtr = (Element *)Tcl_GetHashValue(hashPtr);
	elemPtr->mapped = 0;
    }

    /*
     * Rebuild the display list, checking that each name it exists
     * (we're currently ignoring invalid element names).
     */
    for (count = 0; count < numNames; count++) {
	hashPtr = Tcl_FindHashEntry(&(graphPtr->elemTable), nameArr[count]);
	if (hashPtr != NULL) {
	    elemPtr = (Element *)Tcl_GetHashValue(hashPtr);
	    elemPtr->mapped = 1;
	    listPtr = Blt_CreateListEntry(elemPtr->id);
	    if (listPtr == NULL) {
		free((char *)nameArr);
		return TCL_ERROR;
	    }
	    Blt_SetListValue(listPtr, (ClientData)elemPtr);
	    Blt_LinkListAfter(&(graphPtr->elemList), listPtr,
		(Blt_ListEntry *)NULL);
	}
    }
    free((char *)nameArr);
    graphPtr->flags |= (LAYOUT_PENDING | LAYOUT_ALL);
    Blt_ComputeAxes(graphPtr);
    Blt_EventuallyRedraw(graphPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ShowElements --
 *
 *	Displays or rebuilds the element display list.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *	interp->result contains the list of element names.
 *
 *----------------------------------------------------------------------
 */
static int
ShowElements(graphPtr, argc, argv)
    Graph *graphPtr;
    int argc;
    char **argv;
{
    register Element *elemPtr;
    register Blt_ListEntry *entryPtr;

    if ((argc < 3) || (argc > 4)) {
	Tcl_AppendResult(graphPtr->interp, "wrong # args: should be \"",
	    argv[0], " element show ?nameList?\"", (char *)NULL);
	return TCL_ERROR;
    }
    if (argc == 4) {
	ResetDisplayList(graphPtr, argv[3]);
    }
    for (entryPtr = Blt_FirstListEntry(&(graphPtr->elemList));
	entryPtr != NULL; entryPtr = Blt_NextListEntry(entryPtr)) {
	elemPtr = (Element *)Blt_GetListValue(entryPtr);
	Tcl_AppendElement(graphPtr->interp, (char *)elemPtr->id);
    }
    return TCL_OK;
}

/*
 * Global routines:
 */

/*
 *--------------------------------------------------------------
 *
 * Blt_ElementCmd --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a widget managed by this module.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */
/* ARGSUSED */
int
Blt_ElementCmd(graphPtr, argc, argv)
    Graph *graphPtr;		/* Graph widget record */
    int argc;			/* # arguments */
    char **argv;		/* Argument list */
{
    int result = TCL_ERROR;
    char c;
    int length;

    if (argc < 3) {
	Tcl_AppendResult(graphPtr->interp, "wrong # args: should be \"",
	    argv[0], " element option ?args?\"", NULL);
	return TCL_ERROR;
    }
    c = argv[2][0];
    length = strlen(argv[2]);
    if ((c == 'a') && (strncmp(argv[2], "append", length) == 0)) {
	result = AppendElement(graphPtr, argc, argv);
    } else if ((c == 'c') && (length > 1) &&
	(strncmp(argv[2], "create", length) == 0)) {
	result = CreateElement(graphPtr, argc, argv);
    } else if ((c == 'c') && (length > 1) &&
	(strncmp(argv[2], "configure", length) == 0)) {
	result = ConfigureElement(graphPtr, argc, argv);
    } else if ((c == 'd') && (strncmp(argv[2], "delete", length) == 0)) {
	result = DeleteElements(graphPtr, argc, argv);
    } else if ((c == 'n') && (strncmp(argv[2], "names", length) == 0)) {
	result = ElementNames(graphPtr, argc, argv);
    } else if ((c == 's') && (strncmp(argv[2], "show", length) == 0)) {
	result = ShowElements(graphPtr, argc, argv);
    } else {
	Tcl_AppendResult(graphPtr->interp, "bad element option \"", argv[2],
	    "\": should be append, configure, create, delete, names, or show",
	    (char *)NULL);
	return TCL_ERROR;
    }
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * UpdateVector --
 *
 *	User routine
 *
 * Results:
 *	The return value is a standard Tcl result.  The vector is passed
 *	back via the vecPtr.
 *
 *----------------------------------------------------------------------
 */
static int
UpdateVector(graphPtr, vecPtr, start, numValues, valueArr)
    Graph *graphPtr;		/* Graph widget */
    Vector *vecPtr;		/* Element vector */
    int start;			/* Starting value */
    int numValues;		/* Number of elements in array */
    double *valueArr;		/* Array of floating point values */
{
    unsigned int arraySize;
    double *newArr;
    register int offset;
    register int i;

    offset = vecPtr->length;
    arraySize = (numValues / 2) + offset;
    newArr = (double *)malloc(arraySize * sizeof(double));
    if (newArr == NULL) {
	Tcl_AppendResult(graphPtr->interp, "can't allocate data vector: ",
			 Tcl_PosixError(graphPtr->interp), (char *)NULL);
	return TCL_ERROR;
    }
    if (offset > 0) {
	memcpy((char *)newArr, (char *)vecPtr->data, offset*sizeof(double));
    }
    for (i = start; i < numValues; i += 2) {
	newArr[offset++] = valueArr[i];
    }
    if (vecPtr->data != NULL) {
	free((char *)vecPtr->data);
    }
    vecPtr->data = newArr;
    vecPtr->length = offset;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_GraphElement --
 *
 *	User convenience routine to set data of an individual graph
 *	element.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 * Side Effects:
 *      Graph is redrawn with new data values. Axes are recalculated.
 *
 *----------------------------------------------------------------------
 */
int
Blt_GraphElement(interp, pathName, elemName, numValues, valueArr)
    Tcl_Interp *interp;		/* Interpreter to send results back to */
    char *pathName;		/* Path name of graph widget */
    char *elemName;		/* Name of element to set */
    int numValues;		/* Number of coordinates in array */
    double *valueArr;		/* Array of XY coordinate values */
{
    Tcl_HashEntry *entryPtr;
    Graph *graphPtr;
    Element *elemPtr;
    Tk_Window mainWin, tkwin;
    Tk_Uid classUid;
    ClientData clientData;

    mainWin = Tk_MainWindow(interp);
    if (mainWin == NULL) {
	return TCL_ERROR;
    }
    tkwin = Tk_NameToWindow(interp, pathName, mainWin);
    if (tkwin == NULL) {
	return TCL_ERROR;
    }
    classUid = Tk_Class(tkwin);
    if (classUid != Tk_GetUid("Blt_graph")) { 
	Tcl_AppendResult(interp, "wrong class \"", classUid, "\"", 
			 " for \"", pathName, "\"", (char *)NULL);
	return TCL_ERROR;
    }
    if (Blt_FindCmd(interp, pathName, &clientData) != TCL_OK) {
	Tcl_AppendResult(interp, "can't find command \"", pathName,
	    "\"", (char *)NULL);
	return TCL_ERROR;
    }
    if (numValues <= 2) {
	Tcl_AppendResult(interp, "too few values in array", (char *)NULL);
	return TCL_ERROR;
    }
    graphPtr = (Graph *)clientData;
    entryPtr = Tcl_FindHashEntry(&(graphPtr->elemTable), elemName);
    if (entryPtr == NULL) {
	Tcl_AppendResult(interp, "can't find element \"", elemName, "\" in \"",
	    pathName, "\"", (char *)NULL);
	return TCL_ERROR;
    }
    elemPtr = (Element *)Tcl_GetHashValue(entryPtr);
    UpdateVector(graphPtr, &(elemPtr->x), 0, numValues, valueArr);
    UpdateVector(graphPtr, &(elemPtr->y), 1, numValues, valueArr);
    Blt_ComputeAxes(graphPtr);
    Blt_EventuallyRedraw(graphPtr);
    return TCL_OK;
}

