#include <stdlib.h>

#include "mfsk.h"
#include "trx.h"
#include "fft.h"
#include "sfft.h"
#include "filter.h"
#include "interleave.h"
#include "viterbi.h"

#define	K	7
#define	POLY1	0x6d
#define	POLY2	0x4f

static void mfsk_txinit(struct trx *trx)
{
	struct mfsk *m = (struct mfsk *) trx->modem;

	m->bitstate = 0;
	m->txstate = STATE_PREAMBLE;
}

static void mfsk_rxinit(struct trx *trx)
{
	struct mfsk *m = (struct mfsk *) trx->modem;

	m->symcounter = 0;
	m->metric1 = 0.0;
	m->metric2 = 0.0;
}

static void mfsk_free(struct mfsk *s)
{
	if (s) {
		fft_free(s->fft);
		sfft_free(s->sfft);
		filter_free(s->hilbert);

		free(s->pipe);

		encoder_free(s->enc);
		viterbi_free(s->dec1);
		viterbi_free(s->dec2);

		free(s);
	}
}

static void mfsk_destructor(struct trx *trx)
{
	struct mfsk *s = (struct mfsk *) trx->modem;

	mfsk_free(s);

	trx->modem = NULL;

	trx->txinit = NULL;
	trx->rxinit = NULL;
	trx->txprocess = NULL;
	trx->rxprocess = NULL;
	trx->destructor = NULL;
}

void mfsk_init(struct trx *trx)
{
	struct mfsk *s;

	if ((s = calloc(1, sizeof(struct mfsk))) == NULL)
		return;

	switch (trx->mode) {
	case MODE_MFSK16:
		s->symlen = 512;
		s->symbits = 4;
		s->basetone = 64;		/* 1000 Hz */
                break;

	case MODE_MFSK8:
		s->symlen = 1024;
		s->symbits = 5;
		s->basetone = 128;		/* 1000 Hz */
                break;

	default:
		mfsk_free(s);
		return;
	}

	s->numtones = 1 << s->symbits;
	s->tonespacing = (double) SampleRate / s->symlen;

	if (!(s->fft = fft_init(s->symlen, FFT_FWD))) {
		g_warning("mfsk_init: init_fft failed\n");
		mfsk_free(s);
		return;
	}
	if (!(s->sfft = sfft_init(s->symlen, s->basetone, s->basetone + s->numtones))) {
		g_warning("mfsk_init: init_sfft failed\n");
		mfsk_free(s);
		return;
	}
	if (!(s->hilbert = filter_init_hilbert(37))) {
		g_warning("mfsk_init: init_hilbert failed\n");
		mfsk_free(s);
		return;
	}

	if (!(s->pipe = calloc(2 * s->symlen, sizeof(struct rxpipe)))) {
		g_warning("mfsk_init: allocating pipe failed\n");
		mfsk_free(s);
		return;
	}

	if (!(s->enc = encoder_init(K, POLY1, POLY2))) {
		g_warning("mfsk_init: encoder_init failed\n");
		mfsk_free(s);
		return;
	}
	if (!(s->dec1 = viterbi_init(K, POLY1, POLY2))) {
		g_warning("mfsk_init: viterbi_init failed\n");
		mfsk_free(s);
		return;
	}
	if (!(s->dec2 = viterbi_init(K, POLY1, POLY2))) {
		g_warning("mfsk_init: viterbi_init failed\n");
		mfsk_free(s);
		return;
	}

	if (!(s->txinlv = interleave_init(s->symbits, INTERLEAVE_FWD))) {
		g_warning("mfsk_init: interleave_init failed\n");
		mfsk_free(s);
		return;
	}
	if (!(s->rxinlv = interleave_init(s->symbits, INTERLEAVE_REV))) {
		g_warning("mfsk_init: interleave_init failed\n");
		mfsk_free(s);
		return;
	}

	trx->modem = s;

	trx->txinit = mfsk_txinit;
	trx->rxinit = mfsk_rxinit;

	trx->txprocess = mfsk_txprocess;
	trx->rxprocess = mfsk_rxprocess;

	trx->destructor = mfsk_destructor;

	trx->samplerate = SampleRate;
	trx->fragmentsize = s->symlen;
	trx->bandwidth = (s->numtones - 1) * s->tonespacing;
}
