#include <ctype.h>

#include "psk31.h"
#include "varicode.h"
#include "snd.h"

static void send_symbol(struct trx *trx, int sym)
{
	struct psk31 *s = (struct psk31 *) trx->modem;
	float buf[SymbolLen], delta;
	complex symbol;
	int i;

	if (s->qpsk && trx->reverse)
		sym = (4 - sym) & 3;

	/* differential QPSK modulation - top bit flipped */
	switch (sym) {
	case 0:
		symbol.re = -1.0;	/* 180 degrees */
		symbol.im =  0.0;
		break;
	case 1:
		symbol.re =  0.0;	/* 270 degrees */
		symbol.im = -1.0;
		break;
	case 2:
		symbol.re =  1.0;	/*   0 degrees */
		symbol.im =  0.0;
		break;
	case 3:
		symbol.re =  0.0;	/*  90 degrees */
		symbol.im =  1.0;
		break;
	}
	symbol = cmul(s->prevsymbol, symbol);

	delta = 2.0 * M_PI * (trx->frequency + trx->txoffset) / SampleRate;

	for (i = 0; i < SymbolLen; i++) {
		float ival, qval;

		ival = s->txshape[i] * s->prevsymbol.re +
			(1 - s->txshape[i]) * symbol.re;

		qval = s->txshape[i] * s->prevsymbol.im + 
			(1 - s->txshape[i]) * symbol.im;

		buf[i] = ival * cos(s->phaseacc) + qval * sin(s->phaseacc);

		s->phaseacc += delta;

		if (s->phaseacc > M_PI)
			s->phaseacc -= 2.0 * M_PI;
	}

	sound_write(buf, SymbolLen);

	/* save the current symbol */
	s->prevsymbol = symbol;
}

static void send_bit(struct trx *trx, int bit)
{
	struct psk31 *s = (struct psk31 *) trx->modem;
	unsigned int sym;

	if (s->qpsk)
		sym = encoder_encode(s->enc, bit);
	else
		sym = bit << 1;

	send_symbol(trx, sym);
}

static void send_char(struct trx *trx, unsigned char c)
{
	char *code;

	code = psk_varicode_encode(c);

	while (*code) {
		send_bit(trx, (*code - '0'));
		code++;
	}

	send_bit(trx, 0);
	send_bit(trx, 0);
}

static void flush_tx(struct trx *trx)
{
	struct psk31 *s = (struct psk31 *) trx->modem;
	int i;

	/* flush the encoder (QPSK only) */
	if (s->qpsk) {
		send_bit(trx, 0);
		send_bit(trx, 0);
		send_bit(trx, 0);
		send_bit(trx, 0);
		send_bit(trx, 0);
	}

	/* DCD off sequence (unmodulated carrier) */
	for (i = 0; i < 32; i++)
		send_symbol(trx, 2);
}

int psk31_txprocess(struct trx *trx)
{
	struct psk31 *s = (struct psk31 *) trx->modem;
	int c;

	if (trx->tune) {
		trx->tune = 0;
		send_symbol(trx, 0);	/* send phase reversals */
		return 0;
	}

	if (s->preamble > 0) {
		s->preamble--;
		send_symbol(trx, 0);	/* send phase reversals */
		return 0;
	}

	c = trx_get_tx_char();

	/* TX buffer empty */
	if (c == -1) {
		/* stop if requested to... */
		if (trx->stopflag) {
			flush_tx(trx);
			return -1;	/* we're done */
		}

		send_symbol(trx, 0);	/* send phase reversals */
		send_symbol(trx, 0);
		return 0;
	}

	send_char(trx, c);

	trx_put_echo_char(c);

	return 0;
}
