/*
 * Copyright (C) 2002-2005 the xine-project
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Id: desktop_integration.c,v 1.8.2.2 2005/07/14 23:46:17 dsalt Exp $
 *
 * helper functions to register gxine with kde and gnome
 * (and possibly cde in the future)
 *
 * most code stolen from openoffice
 *
 * oo_1.0.1_src/sysui/oounix/office/gnome
 */

#include "globals.h"

#include <stdlib.h>
#include <stdio.h>
#include <dlfcn.h>
#include <alloca.h>
#include <string.h>

#include "utils.h"

/* stolen from gnome-vfs-mime-info.h */
typedef enum {
  GNOME_VFS_OK,
  GNOME_VFS_ERROR_NOT_FOUND,
  GNOME_VFS_ERROR_GENERIC,
  GNOME_VFS_ERROR_INTERNAL,
  GNOME_VFS_ERROR_BAD_PARAMETERS,
  GNOME_VFS_ERROR_NOT_SUPPORTED,
  GNOME_VFS_ERROR_IO,
  GNOME_VFS_ERROR_CORRUPTED_DATA,
  GNOME_VFS_ERROR_WRONG_FORMAT,
  GNOME_VFS_ERROR_BAD_FILE,
  GNOME_VFS_ERROR_TOO_BIG,
  GNOME_VFS_ERROR_NO_SPACE,
  GNOME_VFS_ERROR_READ_ONLY,
  GNOME_VFS_ERROR_INVALID_URI,
  GNOME_VFS_ERROR_NOT_OPEN,
  GNOME_VFS_ERROR_INVALID_OPEN_MODE,
  GNOME_VFS_ERROR_ACCESS_DENIED,
  GNOME_VFS_ERROR_TOO_MANY_OPEN_FILES,
  GNOME_VFS_ERROR_EOF,
  GNOME_VFS_ERROR_NOT_A_DIRECTORY,
  GNOME_VFS_ERROR_IN_PROGRESS,
  GNOME_VFS_ERROR_INTERRUPTED,
  GNOME_VFS_ERROR_FILE_EXISTS,
  GNOME_VFS_ERROR_LOOP,
  GNOME_VFS_ERROR_NOT_PERMITTED,
  GNOME_VFS_ERROR_IS_DIRECTORY,
  GNOME_VFS_ERROR_NO_MEMORY,
  GNOME_VFS_ERROR_HOST_NOT_FOUND,
  GNOME_VFS_ERROR_INVALID_HOST_NAME,
  GNOME_VFS_ERROR_HOST_HAS_NO_ADDRESS,
  GNOME_VFS_ERROR_LOGIN_FAILED,
  GNOME_VFS_ERROR_CANCELLED,
  GNOME_VFS_ERROR_DIRECTORY_BUSY,
  GNOME_VFS_ERROR_DIRECTORY_NOT_EMPTY,
  GNOME_VFS_ERROR_TOO_MANY_LINKS,
  GNOME_VFS_ERROR_READ_ONLY_FILE_SYSTEM,
  GNOME_VFS_ERROR_NOT_SAME_FILE_SYSTEM,
  GNOME_VFS_ERROR_NAME_TOO_LONG,
  GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE,
  GNOME_VFS_ERROR_SERVICE_OBSOLETE,
  GNOME_VFS_ERROR_PROTOCOL_ERROR,
  GNOME_VFS_NUM_ERRORS
} GnomeVFSResult;

/*
 * the bunch of gnome vfs functions needed
 */
 
struct gnome_vfs_functions 
{
  void  (* application_registry_remove_application) ( const char *app_id );
  void  (* application_registry_add_mime_type)      ( const char *app_id, const char *mime_type );
  void  (* application_registry_set_value)          ( const char *app_id, const char *key, const char *value);
  void  (* application_registry_set_bool_value)     ( const char *app_id, const char *key, int value);
  void  (* application_registry_remove_mime_type)   ( const char *app_id, const char *mime_type );

  void  (* mime_freeze)                      ( void );
  void  (* mime_thaw)                        ( void );
  void  (* mime_registered_mime_type_delete) ( const char * mime_type );
    
  GnomeVFSResult (* mime_set_default_application)           ( const char *mime_type, const char *application_id );
  GnomeVFSResult (* mime_set_registered_type_key)           ( const char *mime_type, const char *key, const char *data );
  GnomeVFSResult (* mime_set_value)                         ( const char *mime_type, const char *key, const char *value );
};

/*
 * the mime info as data structur
 */

struct mime_info
{
  const char * mime_type;
  const char * description;
  const char * extensions;
  const char * icon_filename;
};
    
/*
 * make the code read as normal gnome app
 */
 
#define gnome_vfs_application_registry_remove_application gnome_vfs->application_registry_remove_application
#define gnome_vfs_application_registry_add_mime_type gnome_vfs->application_registry_add_mime_type
#define gnome_vfs_application_registry_remove_mime_type gnome_vfs->application_registry_remove_mime_type
#define gnome_vfs_application_registry_set_value gnome_vfs->application_registry_set_value
#define gnome_vfs_application_registry_set_bool_value gnome_vfs->application_registry_set_bool_value
 
#define gnome_vfs_mime_thaw gnome_vfs->mime_thaw
#define gnome_vfs_mime_freeze gnome_vfs->mime_freeze
#define gnome_vfs_mime_registered_mime_type_delete gnome_vfs->mime_registered_mime_type_delete
#define gnome_vfs_mime_set_registered_type_key gnome_vfs->mime_set_registered_type_key
#define gnome_vfs_mime_set_value gnome_vfs->mime_set_value
#define gnome_vfs_mime_set_default_application gnome_vfs->mime_set_default_application

static void * gnome_vfs_handle = NULL;

static struct gnome_vfs_functions * gnome_vfs = NULL;


/*
 * load the gnome-vfs library and initialize the function pointers
 */
 
int gnome_vfs_init() {

  void (* gnome_vfs_init_func) (void) = NULL;
  void (* gnome_vfs_mime_get_value) ( const char *, const char * ) = NULL;
    
  /* load the gnomevfs library */
  gnome_vfs_handle = dlopen("libgnomevfs-2.so.0", RTLD_LAZY);

  if( NULL == gnome_vfs_handle )  {
    fprintf( stderr, "%s.\n", dlerror() );
    return 0;
  }
    
  gnome_vfs = ( struct gnome_vfs_functions * ) calloc( 1, sizeof( struct gnome_vfs_functions ) );     
  
  /* this call causes gnome-vfs to become initialized */
  gnome_vfs_init_func = (void(*)(void)) dlsym( gnome_vfs_handle, "gnome_vfs_init" );

  if( NULL == gnome_vfs_init_func )  {
    fprintf( stderr, "%s.\n", dlerror() );
    return 0;
  }

  gnome_vfs_init_func ();

  gnome_vfs_mime_get_value = (void(*)(const char*,const char*)) dlsym( gnome_vfs_handle, "gnome_vfs_mime_get_value" );
  
  if( NULL == gnome_vfs_mime_get_value )  {
    fprintf( stderr, "%s.\n", dlerror() );
    return 0;
  }
    
  gnome_vfs_mime_get_value( "text/plain", "description" );
  
  /* extract needed symbols from the library */
  gnome_vfs->mime_set_registered_type_key = (GnomeVFSResult(*)(const char*,const char*,const char*)) dlsym( gnome_vfs_handle, "gnome_vfs_mime_set_registered_type_key" );    

  if( NULL == gnome_vfs->mime_set_registered_type_key ) {
    fprintf( stderr, "%s.\n", dlerror() );
    return 0;
  }

  gnome_vfs->mime_freeze = (void(*)()) dlsym( gnome_vfs_handle, "gnome_vfs_mime_freeze" );    

  if( NULL == gnome_vfs->mime_freeze ) {
    fprintf( stderr, "%s.\n", dlerror() );
    return 0;
  }

  gnome_vfs->mime_thaw = (void(*)()) dlsym( gnome_vfs_handle, "gnome_vfs_mime_thaw" );    

  if( NULL == gnome_vfs->mime_thaw )  {
    fprintf( stderr, "%s.\n", dlerror() );
    return 0;
  }
    
  gnome_vfs->mime_set_value = (GnomeVFSResult(*)(const char*,const char*,const char*)) dlsym( gnome_vfs_handle, "gnome_vfs_mime_set_value" );    

  if( NULL == gnome_vfs->mime_set_value )  {
    fprintf( stderr, "%s.\n", dlerror() );
    return 0;
  }

  gnome_vfs->mime_registered_mime_type_delete = (void(*)(const char*)) dlsym( gnome_vfs_handle, "gnome_vfs_mime_registered_mime_type_delete" );    

  if( NULL == gnome_vfs->mime_registered_mime_type_delete )  {
    fprintf( stderr, "%s.\n", dlerror() );
    return 0;
  }

  gnome_vfs->mime_set_default_application = (GnomeVFSResult(*)(const char*,const char*)) dlsym( gnome_vfs_handle, "gnome_vfs_mime_set_default_application" );
     
  if( NULL == gnome_vfs->mime_set_default_application )  {
    fprintf( stderr, "%s.\n", dlerror() );
    return 0;
  }
    
  gnome_vfs->application_registry_remove_application  = (void(*)(const char*)) dlsym( gnome_vfs_handle, "gnome_vfs_application_registry_remove_application" );
    
  if( NULL == gnome_vfs->application_registry_remove_application )  {
    fprintf( stderr, "%s.\n", dlerror() );
    return 0;
  }
    
  gnome_vfs->application_registry_add_mime_type = (void(*)(const char*,const char*)) dlsym( gnome_vfs_handle, "gnome_vfs_application_registry_add_mime_type" );
    
  if( NULL == gnome_vfs->application_registry_add_mime_type )  {
    fprintf( stderr, "%s.\n", dlerror() );
    return 0;
  }
    

  gnome_vfs->application_registry_set_value = (void(*)(const char*,const char*,const char*)) dlsym( gnome_vfs_handle, "gnome_vfs_application_registry_set_value" );
    
  if( NULL == gnome_vfs->application_registry_set_value )  {
    fprintf( stderr, "%s.\n", dlerror() );
    return 0;
  }
    

  gnome_vfs->application_registry_set_bool_value = (void(*)(const char*,const char*,int)) dlsym( gnome_vfs_handle, "gnome_vfs_application_registry_set_bool_value" );
    
  if( NULL == gnome_vfs->application_registry_set_bool_value )  {
    fprintf( stderr, "%s.\n", dlerror() );
    return 0;
  }
    

  gnome_vfs->application_registry_remove_mime_type = (void(*)(const char*,const char*)) dlsym( gnome_vfs_handle, "gnome_vfs_application_registry_remove_mime_type" );
    
  if( NULL == gnome_vfs->application_registry_remove_mime_type )  {
    fprintf( stderr, "%s.\n", dlerror() );
    return 0;
  }

  return 1;
}

/*
 * release the gnome-vfs library and free the function pointer struct
 */
 
void gnome_vfs_shutdown() {
  /* release the gnomevfs library */
  if ( gnome_vfs_handle )  {
    void (* gnome_vfs_mime_info_shutdown) ( void ) = NULL;
    void (* gnome_vfs_application_registry_sync) ( void ) = NULL;
    
    /* extract and call shutdown functions */
    gnome_vfs_mime_info_shutdown = (void(*)()) dlsym( gnome_vfs_handle, "gnome_vfs_mime_shutdown" );

    if( NULL == gnome_vfs_mime_info_shutdown )  {
      fprintf( stderr, "%s.\n", dlerror() );
      return;
    }

    gnome_vfs_application_registry_sync = (void(*)()) dlsym( gnome_vfs_handle, "gnome_vfs_application_registry_sync" );
    
    if( NULL == gnome_vfs_application_registry_sync )  {
      fprintf( stderr, "%s.\n", dlerror() );
      return;
    }

    gnome_vfs_mime_info_shutdown();
    gnome_vfs_application_registry_sync();
    dlclose( gnome_vfs_handle );
  }
  
  if( gnome_vfs )
    free( gnome_vfs );
}

/*
 *  add the installed application to the gnome application info registry
 */
static GnomeVFSResult register_application( const char * application_id, 
					    const char * product_name, 
					    const char * product_version,
					    const char * command, 
					    const char * install_path ) {

  char * command_full = (char *) alloca( strlen( install_path ) + strlen( command ) + 10 );
  char * product = (char *) alloca( strlen( product_name ) + strlen( product_version ) + 2 );
  
  sprintf( command_full, "%s/%s", install_path, command );
  sprintf( product, "%s %s", product_name, product_version );
  
  /* entry will be created automatically */
  gnome_vfs_application_registry_set_value( application_id, "name", product );
  gnome_vfs_application_registry_set_value( application_id, "command", command_full );
  gnome_vfs_application_registry_set_bool_value( application_id, "can_open_multiple_files", 1 );
  gnome_vfs_application_registry_set_bool_value( application_id, "requires_terminal", 0 );
  
  return GNOME_VFS_OK;
}



void di_register_gnome (void) {

  char *mime_types;

  if (!gnome_vfs_init()) {
    printf (_("desktop_integration: initialization failed.\n"));
    return ;
  }

  /* freeze the lists to enable multiple updates */
  gnome_vfs_mime_freeze();

  register_application ("gxine", "gxine", VERSION, "gxine", bindir);
  /*
   * iterate over all mime types xine supports
   * and register it as a handler for each of them
   */

  mime_types = xine_get_mime_types (xine);
  while (mime_types) {

    char *next = mime_types;

    if ( (next=strchr (mime_types, ';')) ) {
      char *end;

      *next=0;

      end = strchr (mime_types, ':');
      if (end) {
	*end=0;

	printf ("desktop_integration: registering xine for mime type '%s'\n",
		mime_types);
	gnome_vfs_application_registry_add_mime_type ("gxine", mime_types );  
	gnome_vfs_mime_set_default_application (mime_types, "gxine" ); 
      }
    }

    if (next)
      mime_types = next+1;
    else
      mime_types = NULL;
  }

  /* force the user.mime and user.keys to be written  */
  gnome_vfs_mime_thaw();
  
  gnome_vfs_shutdown();
}

void di_register_kde (void) {

  gchar *fname;
  gchar *kdehome;
  FILE  *f;
  char  *mime_types;
  
  kdehome = getenv ("KDEHOME");

  if (kdehome) 
    fname = g_strconcat (kdehome, "/share/applnk/Multimedia", 
			 NULL);
  else
    fname = g_strconcat (g_get_home_dir(), "/.kde/share/applnk/Multimedia", 
			 NULL);
  ensure_path_exists (fname);
  g_free (fname);

  /*
   * produce gxine.desktop
   */

  if (kdehome) 
    fname = g_strconcat (kdehome, "/share/applnk/Multimedia/gxine.desktop", 
			 NULL);
  else
    fname = g_strconcat (g_get_home_dir(), "/.kde/share/applnk/Multimedia/gxine.desktop", 
			 NULL);
  
  
  f = fopen (fname, "w");
  if (!f) {
    printf (_("desktop_integration: error while creating %s\n"),
	    fname);

    g_free (fname);
  }

  fprintf (f, "[Desktop Entry]\n"
	   "Version=1.0\n"
	   "Encoding=UTF-8\n"
	   "Name=gxine\n"
	   "GenericName=Video Player\n"
 	   "Comment=Play films and songs, or watch digital TV\n"
	   "MultipleArgs=true\n"
	   "Terminal=0\n"
	   "Icon=%s/gxine-logo.png\n"
	   "Exec=%s/gxine\n"
	   "Type=Application\n"
 	   "StartupNotify=true\n"
 	   "Categories=GTK;Application;AudioVideo;\n"
	   "MimeType=",
	   icondir,
	   bindir);

  /*
   * iterate over all mime types xine supports
   * and register it as a handler for each of them
   */

  mime_types = xine_get_mime_types (xine);
  while (mime_types) {

    char *next = mime_types;

    if ( (next=strchr (mime_types, ';')) ) {
      char *end;

      *next=0;

      end = strchr (mime_types, ':');
      if (end) {
	*end=0;

	printf (_("desktop_integration: KDE registering xine for mime type '%s'\n"),
		mime_types);

	fprintf (f, "%s;", mime_types);

      }
    }

    if (next)
      mime_types = next+1;
    else
      mime_types = NULL;
  }
  
  fprintf (f, "\n");
  fclose (f);
  g_free (fname);
}

void di_register_mailcap (void) {

  char  *mc;
  gchar *fname;
  int    n;
  FILE  *f;
  char  *mime_types;

  fname = g_strconcat (g_get_home_dir(), "/.mailcap", NULL);

  mc = read_entire_file (fname, &n);

  f = fopen (fname, "w");
  if (!f) {
    printf (_("desktop_integration: cannot open %s for writing\n"),
	    fname);
    g_free (fname);
    return;
  }

  if (mc)
    fwrite (mc, n, 1, f);

  /*
   * iterate over all mime types xine supports
   * and register it as a handler for each of them
   */

  mime_types = xine_get_mime_types (xine);
  while (mime_types) {

    char *next = mime_types;

    if ( (next=strchr (mime_types, ';')) ) {
      char *end;

      *next=0;

      end = strchr (mime_types, ':');
      if (end) {
	*end=0;

	if (!mc || !strstr (mc, mime_types)) {

	  printf (_("desktop_integration: mailcap: registering xine for mime type '%s'\n"),
		  mime_types);
	  
	  fprintf (f, "%s;%s/gxine %%s\n", mime_types,
		   bindir);
	} else {
	  printf (_("desktop_integration: %s has already a handler\n"),
		  mime_types);
	}
      }
    }

    if (next)
      mime_types = next+1;
    else
      mime_types = NULL;
  }
  
  fprintf (f, "\n");

  fclose (f);
}

