/*
 * Copyright (C) 2002-2004 the xine-project
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Id: infobar.c,v 1.22 2005/01/12 18:44:14 dsalt Exp $
 *
 * a nice black information display area
 */

#include "globals.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <xine.h>

#include "infobar.h"
#include "utils.h"
#include "playlist.h"
#include "player.h"

#define NUMLINES 3

struct infobar_s {

  GtkWidget            *w;

  PangoLayout          *pl[NUMLINES];
  PangoFontDescription *pfd, *pfd_big;

  GdkGC                *black_gc, *blue_gc, *white_gc;

  gchar                *line[NUMLINES];
};

static void paint_bar (infobar_t *bar)
{
  int i;
  gdk_draw_rectangle (bar->w->window, bar->black_gc, TRUE,
		      0, 0, 
		      bar->w->allocation.width, bar->w->allocation.height+1);

  pango_layout_set_text (bar->pl[0], bar->line[0] ? : "", -1);
  gdk_draw_layout (bar->w->window, bar->blue_gc, 7, 2, bar->pl[0]);

  for (i = 1; i < NUMLINES; ++i)
  {
    pango_layout_set_text (bar->pl[i], bar->line[i] ? : "", -1);
    gdk_draw_layout (bar->w->window, bar->blue_gc, 7, 9 + 14 * i, bar->pl[i]);
  }
}

static gboolean expose_event_cb (GtkWidget *widget, 
				 GdkEventExpose *event, 
				 gpointer data) {

  infobar_t *bar = (infobar_t *) data;

  gdk_window_clear_area (widget->window,
                         event->area.x, event->area.y,
                         event->area.width, event->area.height+1);

  gdk_gc_set_clip_rectangle (widget->style->fg_gc[widget->state],
                             &event->area);

  paint_bar (bar);

  gdk_gc_set_clip_rectangle (widget->style->fg_gc[widget->state],
                             NULL);
  return TRUE;
}

void infobar_clear (infobar_t *bar)
{
  int i;
  if (!bar) return;
  for (i = 0; i < NUMLINES; ++i)
  {
    free (bar->line[i]);
    bar->line[i] = NULL;
  }
  paint_bar (bar);
}

void infobar_line (infobar_t *bar, unsigned int line, const char *format, ...)
{
  if (!bar || line >= NUMLINES)
    return;

  free (bar->line[line]);
  bar->line[line] = NULL;

  if (format)
  {
    va_list argp;
    va_start (argp, format);
    bar->line[line] = g_strdup_vprintf (format, argp);
    va_end (argp);
  }

  paint_bar (bar);
}

static void alloc_color(GtkWidget *window, GdkGC *gc, gint r, gint g, gint b) {

  GdkColor *color=malloc(sizeof(GdkColor));

  if (gc) {
    color->red = r; color->green = g; color->blue = b;
    if (!gdk_colormap_alloc_color (gtk_widget_get_colormap (window), color, FALSE, TRUE)) {
      printf ("failed to alloc color (%d/%d/%d)\n",color->red,color->green,color->blue);
      exit(1);
    }
    gdk_gc_set_foreground (gc, color);
  }
}

void infobar_show_metadata (void)
{
  char	     *strbuf = NULL;
  const char *t;
  const char *str;
  int         n, w, h;

  if (!bar) return;

  if ( (t=xine_get_meta_info (stream, XINE_META_INFO_TITLE)) ) {
    const char *a;

    if ( (a=xine_get_meta_info (stream, XINE_META_INFO_ARTIST)) ) {
      strbuf = g_strdup_printf ("%s (%s) - gxine %s", t, a, VERSION);
      infobar_line (bar, 0, "%s (%s)", t, a);
    } else {
      strbuf = g_strdup_printf ("%s - gxine %s", t, VERSION);
      infobar_line (bar, 0, "%s", t);
    }
  } else {
    const char *str = NULL;
    if (!playlist_showing_logo ())
    {
      str = player_get_cur_title ();
      if (!str || !str[0])
      {
        const char *mrl = player_get_cur_mrl ();
        str = (strrchr (mrl, '/') ? : mrl - 1) + 1; /* leaf */
      }
    }
    if (str && str[0])
    {
      strbuf = g_strdup_printf ("%s - gxine %s", str, VERSION);
      infobar_line (bar, 0, "%s", str);
    }
    else
    {
      strbuf = g_strdup_printf ("gxine %s", VERSION);
      infobar_line (bar, 0, "%s", strbuf);
    }
  }
  gtk_window_set_title (GTK_WINDOW (app), strbuf);

  /*
   * display some metainfo
   */

  strbuf[0]=0;

  str = xine_get_meta_info (stream, XINE_META_INFO_SYSTEMLAYER);
  if (str) asreprintf (&strbuf, "%s%s ", strbuf, str);

  w = xine_get_stream_info (stream, XINE_STREAM_INFO_VIDEO_WIDTH);
  h = xine_get_stream_info (stream, XINE_STREAM_INFO_VIDEO_HEIGHT);
  if (w && h) asreprintf (&strbuf, "%s%dx%d ", strbuf, w, h); 

  str = xine_get_meta_info (stream, XINE_META_INFO_VIDEOCODEC);
  if (str) asreprintf (&strbuf, "%s%s ", strbuf, str);

  n = xine_get_stream_info (stream, XINE_STREAM_INFO_VIDEO_BITRATE)/1000;
  if (n > 10 && n < 20000) asreprintf (&strbuf, "%s%d kBit/s ", strbuf, n); 

  infobar_line (bar, 1, strbuf);

  strbuf[0] = 0;

  n = xine_get_stream_info (stream, XINE_STREAM_INFO_AUDIO_SAMPLERATE)/1000;
  if ((n>0) && (n<256)) asreprintf (&strbuf, "%s%d kHz ", strbuf, n);

  n = xine_get_stream_info (stream, XINE_STREAM_INFO_AUDIO_BITRATE)/1000;
  if ((n>0) && (n<1024)) asreprintf (&strbuf, "%s%d kBit/s ", strbuf, n);

  str = xine_get_meta_info (stream, XINE_META_INFO_AUDIOCODEC);
  if (str) asreprintf (&strbuf, "%s%s ", strbuf, str);

  infobar_line (bar, 2, strbuf);

  free (strbuf);
}


infobar_t *create_infobar (GtkWidget **w) {

  infobar_t *bar;
  int i;

  bar = (infobar_t *) malloc (sizeof (infobar_t));

  bar->w = gtk_drawing_area_new ();
  gtk_drawing_area_size (GTK_DRAWING_AREA(bar->w), 300, 12 + NUMLINES * 14);

  for (i = 0; i < NUMLINES; ++i)
    bar->line[i] = NULL;

  *w = bar->w;

  return bar;
}

void postinit_infobar (infobar_t *bar)
{
  int i;
  GtkWidget *window;

  if (!bar) return;

  window = gtk_widget_get_toplevel (bar->w);

  g_signal_connect (bar->w, "expose-event", 
		    G_CALLBACK (expose_event_cb), bar);

  bar->black_gc = gdk_gc_new (window->window);
  alloc_color (window, bar->black_gc, 0, 0, 0);

  bar->white_gc = gdk_gc_new (window->window);
  alloc_color (window, bar->white_gc, 65535, 65535, 65535);

  bar->blue_gc = gdk_gc_new (window->window);
  alloc_color (window, bar->blue_gc, 20000, 20000, 65535);

  /* 
   * pango stuff (antialiased font rendering
   */

  bar->pfd_big = pango_font_description_from_string ("Sans Bold 12");
  bar->pl[0] = gtk_widget_create_pango_layout (bar->w, NULL);
  pango_layout_set_font_description (bar->pl[0], bar->pfd_big);

  bar->pfd = pango_font_description_from_string ("Sans Bold 8");
  for (i = 1; i < NUMLINES; ++i)
  {
    bar->pl[i] = gtk_widget_create_pango_layout (bar->w, NULL);
    pango_layout_set_font_description (bar->pl[i], bar->pfd);
  }

}
