/* AbiSource Program Utilities
 * Copyright (C) 2002 Dom Lachowicz <cinamod@hotmail.com>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA.
 */

#include <gsf/gsf-output-stdio.h>
#include <gsf/gsf-outfile.h>
#include <gsf/gsf-outfile-zip.h>

#include <locale.h>

#include "ie_impexp_OpenWriter.h"
#include "ie_exp_OpenWriter.h"

#include "ut_debugmsg.h"

/*****************************************************************************/
/*****************************************************************************/

IE_Exp_OpenWriter_Sniffer::IE_Exp_OpenWriter_Sniffer()
  : IE_ExpSniffer ("OpenWriter::SXW")
{
}

IE_Exp_OpenWriter_Sniffer::~IE_Exp_OpenWriter_Sniffer()
{
}

/*!
 * Recognize this suffix
 */
bool IE_Exp_OpenWriter_Sniffer::recognizeSuffix(const char * szSuffix)
{
  return (!UT_stricmp(szSuffix,".sxw"));
}

/*!
 * Construct an importer for us
 */
UT_Error IE_Exp_OpenWriter_Sniffer::constructExporter(PD_Document * pDocument,
						      IE_Exp ** ppie)
{
  *ppie = new IE_Exp_OpenWriter (pDocument);
  return UT_OK;
}

/*!
 * Get the dialog labels
 */
bool IE_Exp_OpenWriter_Sniffer::getDlgLabels(const char ** pszDesc,
					     const char ** pszSuffixList,
					     IEFileType * ft)
{
  *pszDesc = "OpenOffice Writer (.sxw)";
  *pszSuffixList = "*.sxw";
  *ft = getFileType();
  return true;
}

/*****************************************************************************/
/*****************************************************************************/

/*!
 * Write out a message to the stream. Message is an array of content
 */
static void writeToStream (GsfOutput * stream, const char * const message [], size_t nElements)
{
    for(UT_uint32 k = 0; k < nElements; k++)
      gsf_output_write(stream, strlen(message[k]), reinterpret_cast<const guint8 *>(message[k]));
}

static void writeString (GsfOutput * output, const UT_String & str)
{
   gsf_output_write (output, str.length(), reinterpret_cast<const guint8*>(str.c_str()));
}

static void writeUTF8String (GsfOutput * output, const UT_UTF8String & str)
{
  gsf_output_write (output, str.byteLength(), reinterpret_cast<const guint8*>(str.utf8_str()));
}

static void outputCharData (GsfOutput * output, const UT_UCSChar * data, UT_uint32 length)
{
   UT_UTF8String sBuf;
   const UT_UCSChar * pData;
   
   UT_ASSERT(sizeof(UT_Byte) == sizeof(char));
   
   for (pData=data; (pData<data+length); /**/)
   {
      switch (*pData)
      {
      case '<':
	 sBuf += "&lt;";
	 pData++;
	 break;
	 
      case '>':
	 sBuf += "&gt;";
	 pData++;
	 break;
	 
      case '&':
	 sBuf += "&amp;";
	 pData++;
	 break;
	 
      case UCS_TAB:
	 sBuf += "\t";
	 pData++;
	 break;
	 
      default:
	 if (*pData < 0x20)         // Silently eat these characters.
	    pData++;
	 else
	 {
	    sBuf.appendUCS4 (pData, 1);
	    pData++;
	 }
      }
   }
   
   writeUTF8String (output, sBuf);
}


/*****************************************************************************/
/*****************************************************************************/

OO_WriterImpl::OO_WriterImpl(GsfOutfile *pOutfile, OO_StylesContainer *pStylesContainer) : 
   OO_ListenerImpl(), m_pStylesContainer(pStylesContainer)
{ 
        m_pContentStream = gsf_outfile_new_child (pOutfile, "content.xml", FALSE);

	static const char * const preamble [] = 
	   {
	      "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n",
	      "<!DOCTYPE office:document-content PUBLIC \"-//OpenOffice.org//DTD OfficeDocument 1.0//EN\" \"office.dtd\">\n",
	      "<office:document-content xmlns:office=\"http://openoffice.org/2000/office\" xmlns:style=\"http://openoffice.org/2000/style\" xmlns:text=\"http://openoffice.org/2000/text\" xmlns:table=\"http://openoffice.org/2000/table\" xmlns:draw=\"http://openoffice.org/2000/drawing\" xmlns:fo=\"http://www.w3.org/1999/XSL/Format\" xmlns:xlink=\"http://www.w3.org/1999/xlink\" xmlns:number=\"http://openoffice.org/2000/datastyle\" xmlns:svg=\"http://www.w3.org/2000/svg\" xmlns:chart=\"http://openoffice.org/2000/chart\" xmlns:dr3d=\"http://openoffice.org/2000/dr3d\" xmlns:math=\"http://www.w3.org/1998/Math/MathML\" xmlns:form=\"http://openoffice.org/2000/form\" xmlns:script=\"http://openoffice.org/2000/script\" office:class=\"text\" office:version=\"1.0\">\n",
	      "<office:script/>\n"
	      "<office:automatic-styles>\n",
	   };

	writeToStream(m_pContentStream, preamble, NrElements(preamble));
	UT_Vector *tempStylesValuesList = m_pStylesContainer->enumerateSpanStyles();
	UT_Vector *tempStylesKeysList = m_pStylesContainer->getSpanStylesKeys();
	
	for (UT_uint32 i=0; i<tempStylesValuesList->size(); i++) 
	{
	   int *styleNum = static_cast<int *>(tempStylesValuesList->getNthItem(i));
	   UT_String *styleProps = static_cast<UT_String *>(tempStylesKeysList->getNthItem(i));
	   UT_String styleString = UT_String_sprintf("<style:style style:name=\"S%i\" style:family=\"%s\"><style:properties %s/></style:style>\n", *styleNum, "text", styleProps->c_str());
	   writeString(m_pContentStream, styleString);
	   UT_DEBUGMSG(("%s", styleString.c_str()));
	   UT_DEBUGMSG(("\"%s\"\n", styleProps->c_str()));
	}
	delete(tempStylesKeysList);
	delete(tempStylesValuesList);
    
    //m_acc.writeStylePreamble(m_contentStream);
    
    static const char * const midsection [] = 
       {
	  "</office:automatic-styles>\n",
	  "<office:body>\n",
	  "<text:sequence-decls>\n",
	  "<text:sequence-decl text:display-outline-level=\"0\" text:name=\"Illustration\"/>\n",
	  "<text:sequence-decl text:display-outline-level=\"0\" text:name=\"Table\"/>\n",
	  "<text:sequence-decl text:display-outline-level=\"0\" text:name=\"Text\"/>\n",
	  "<text:sequence-decl text:display-outline-level=\"0\" text:name=\"Drawing\"/>\n",
	  "</text:sequence-decls>\n"
       };

    writeToStream(m_pContentStream, midsection, NrElements(midsection));
}

OO_WriterImpl::~OO_WriterImpl()
{
    static const char * const postamble [] = 
      {
	  "</office:body>\n",
	  "</office:document-content>\n"
      };
    writeToStream(m_pContentStream, postamble, NrElements(postamble));

    gsf_output_close(m_pContentStream);
    g_object_unref(G_OBJECT(m_pContentStream));
}

void OO_WriterImpl::insertText(const UT_UCSChar * data, UT_uint32 length)
{
   outputCharData(m_pContentStream, data, length);
}

void OO_WriterImpl::openBlock()
{
   UT_UTF8String begP ("<text:p text:style-name=\"Standard\">");
   writeUTF8String (m_pContentStream, begP);
}

void OO_WriterImpl::closeBlock()
{
   UT_UTF8String endP ("</text:p>\n");
   writeUTF8String (m_pContentStream, endP);
}

void OO_WriterImpl::openSpan(UT_String &props)
{
   UT_UTF8String spanString = UT_UTF8String_sprintf("<text:span text:style-name=\"S%i\">",  
						    m_pStylesContainer->getStyleNum(props));

   writeUTF8String(m_pContentStream, spanString);

}

void OO_WriterImpl::closeSpan()
{
   UT_UTF8String closeSpan = "</text:span>";
   writeUTF8String(m_pContentStream, closeSpan);
}

void OO_StylesContainer::addSpanStyle(UT_String &key)
{
   const void *temp;
   //if (!m_spanStylesHash.contains(key.utf8_str(), temp)) 
   if (!m_spanStylesHash.pick(key.c_str())) 
   {
      UT_DEBUGMSG(("OO_AccumulatorImpl: props of this type: %s not yet found, adding style at num: %i\n", key.c_str(), (m_spanStylesHash.size()+1)));
      int *val = new int;
      char *keyCopy = new char[strlen(key.c_str())+1];
      keyCopy = strcpy(keyCopy, key.c_str());
      *val = (int)m_spanStylesHash.size()+1;
      m_spanStylesHash.insert(keyCopy, val);
   }
   else 
   {
      UT_DEBUGMSG(("OO_AccumulatorImpl: props of this type: %s already there, forget it\n", key.c_str()));
   }
}

const int OO_StylesContainer::getStyleNum(UT_String &key) const
{
   if (int *val = reinterpret_cast<int *>(const_cast<void *>(m_spanStylesHash.pick(key.c_str())))) {
      return *val;
   }
   else
      return 0;
}

UT_Vector * OO_StylesContainer::enumerateSpanStyles() const
{
   return m_spanStylesHash.enumerate();
}


UT_Vector * OO_StylesContainer::getSpanStylesKeys() const
{
   return m_spanStylesHash.keys();
}

void OO_AccumulatorImpl::openSpan(UT_String &props)
{
   m_pStylesContainer->addSpanStyle(props);

}

OO_Listener::OO_Listener (PD_Document * pDocument, IE_Exp_OpenWriter * pie, OO_ListenerImpl *pListenerImpl)
   : PL_Listener (), m_pDocument(pDocument), m_pie(pie), m_pListenerImpl(pListenerImpl), m_bInBlock(false), m_bInSpan(false)
{
}

bool OO_Listener::populate(PL_StruxFmtHandle sfh,
			   const PX_ChangeRecord * pcr)
{
	switch (pcr->getType())
	{
		case PX_ChangeRecord::PXT_InsertSpan:
		{
			const PX_ChangeRecord_Span * pcrs = static_cast<const PX_ChangeRecord_Span *>(pcr);
			PT_BufIndex bi = pcrs->getBufIndex();
			PT_AttrPropIndex api = pcr->getIndexAP();
			if (api)
			{
			   _openSpan(api);
			}   
			m_pListenerImpl->insertText(m_pDocument->getPointer(bi), pcrs->getLength());
			if (api)
			{
			   _closeSpan();
			}
			return true;
		}
	default:
	   break;
	}
}
   
bool OO_Listener::populateStrux(PL_StruxDocHandle sdh,
				const PX_ChangeRecord * pcr,
				PL_StruxFmtHandle * psfh)
{
   const PX_ChangeRecord_Strux * pcrx = static_cast<const PX_ChangeRecord_Strux *> (pcr);
   *psfh = 0;							// we don't need it.
    
   switch (pcrx->getStruxType())
   {
   case PTX_Block:
   {
      _closeSpan ();
      _openBlock(pcr->getIndexAP());
   }
   }
   
   return true;
}

bool OO_Listener::change(PL_StruxFmtHandle sfh,
			 const PX_ChangeRecord * pcr)
{
   UT_ASSERT_NOT_REACHED();
   return true;
}
   
bool OO_Listener::insertStrux(PL_StruxFmtHandle sfh,
				      const PX_ChangeRecord * pcr,
				      PL_StruxDocHandle sdh,
				      PL_ListenerId lid,
				      void (* pfnBindHandles)(PL_StruxDocHandle sdhNew,
							      PL_ListenerId lid,
							      PL_StruxFmtHandle sfhNew))
{
   UT_ASSERT_NOT_REACHED();
   return true;
}
   
bool OO_Listener::signal(UT_uint32 iSignal)
{
   UT_ASSERT_NOT_REACHED();
   return true;
}

void OO_Listener::endDocument()
{
   _closeBlock();
}

void OO_Listener::_openBlock (PT_AttrPropIndex apiSpan)
{
   if (m_bInBlock)
      _closeBlock();
   
   m_pListenerImpl->openBlock();
   m_bInBlock = true;
}

void OO_Listener::_closeBlock ()
{
   if (!m_bInBlock)
      return;
   
   m_pListenerImpl->closeBlock();
   m_bInBlock = false;
}

// define some attribute bits
#define IMP_SUPERSCRIPT_BIT 0
#define IMP_SUBSCRIPT_BIT 2
#define IMP_ITALICS_BIT 4
#define IMP_BOLD_BIT 8
#define IMP_STRIKEOUT_BIT 16
#define IMP_UNDERLINE_BIT 32

void OO_Listener::_openSpan(PT_AttrPropIndex api)
{
   if (!m_bInBlock)
   {
      return;
   }
   const PP_AttrProp * pAP = NULL;
   bool bHaveProp = m_pDocument->getAttrProp(api,&pAP);
   
   UT_String props;

   if (bHaveProp && pAP)
   {
      const XML_Char * szValue;
      
      if ((pAP->getProperty("text-position", szValue))
	  && !UT_strcmp(szValue, "superscript"))
      {
	 props += "style:text-position=\"super 58%\" ";
      }
      
      if ((pAP->getProperty("text-position", szValue))
	  && !UT_strcmp(szValue, "subscript"))
      
      {
	 props += "style:text-position=\"sub 58%\" ";
      }
      
      if ((pAP->getProperty("font-style", szValue))
	  && !UT_strcmp(szValue, "italic"))
      {
	 props += "fo:font-style=\"italic\" ";
      }
      
      if ((pAP->getProperty("font-weight", szValue))
	  && !UT_strcmp(szValue, "bold")
	 )
      {
	 props += "fo:font-weight=\"bold\" ";
      }

      if ((pAP->getProperty("text-decoration", szValue)))
      {
	 const XML_Char* pszDecor = szValue;
	 
	 XML_Char* p;
	 if (!UT_cloneString(static_cast<char *&>(p), pszDecor))
	 {
	    // TODO outofmem
	 }
	 
	 UT_return_if_fail(p || !pszDecor);
	 XML_Char*	q = strtok(p, " ");

	 while (q)
	 {
	    if (0 == UT_strcmp(q, "line-through"))
	    {
	       props += "style:text-crossing-out=\"single-line\" ";
	    }
	    
	    q = strtok(NULL, " ");
	 }
	 
	 free(p);
      }
      
      if (
	 (pAP->getProperty("text-decoration", szValue))
	 )
      {
	 const XML_Char* pszDecor = szValue;
	 
	 XML_Char* p;
	 if (!UT_cloneString(static_cast<char *&>(p), pszDecor))
	 {
	    // TODO outofmem
	 }
	 
	 UT_return_if_fail(p || !pszDecor);
	 XML_Char*	q = strtok(p, " ");
	 
	 while (q)
			{
			   if (0 == UT_strcmp(q, "underline"))
			   {
			      props += "style:text-underline=\"single\" ";
			   }
			   
			   q = strtok(NULL, " ");
			}
	 
	 free(p);
      }
      
      if (
	 (pAP->getProperty("color", szValue))
	 || (pAP->getProperty("font-size", szValue))
	 || (pAP->getProperty("font-family", szValue))
	 || (pAP->getProperty("bgcolor", szValue))
	 )
      {
	 const XML_Char* pszColor = NULL;
	 const XML_Char* pszBgColor = NULL;
	 const XML_Char* pszFontSize = NULL;
	 const XML_Char* pszFontFamily = NULL;
	 
	 pAP->getProperty("color", pszColor);
	 pAP->getProperty("font-size", pszFontSize);
	 pAP->getProperty("font-family", pszFontFamily);
	 pAP->getProperty("bgcolor", pszBgColor);
	 
	 if (pszColor)
	 {
	    // ...
	 }
	 
	 if (pszFontSize)
	 {
	    setlocale (LC_NUMERIC, "C");

	    UT_String fontSizeProps;
	    fontSizeProps = UT_String_sprintf(fontSizeProps, "fo:font-size=\"%gpt\" ", UT_convertToPoints(pszFontSize));
	    props += fontSizeProps;
	       
	    setlocale (LC_NUMERIC, "");
	 }
	 
	 if (pszFontFamily)
	 {
	    // ...
	 }
	 
	 if (pszBgColor)
	 {
	    // ...
	 }
      }
      //m_pAP_Span = pAP;
   }
   
   m_pListenerImpl->openSpan(props);
   m_bInSpan = true;
}

void OO_Listener::_closeSpan()
{
   if (m_bInSpan)
      m_pListenerImpl->closeSpan();
   m_bInSpan = false;
}


/*****************************************************************************/
/*****************************************************************************/

/*!
 * Class holding 1 static member. Its sole duty is to write
 * out a OOo meta.xml file based on Abi's metadata.
 */
class OO_MetaDataWriter
{
public:
  
  static bool writeMetaData(PD_Document * pDoc, GsfOutfile * oo)
  {
    GsfOutput * meta = gsf_outfile_new_child (oo, "meta.xml", FALSE);

    static const char * const preamble [] = 
      {
	"<?xml version='1.0' encoding='UTF-8'?>\n",
	"<!DOCTYPE office:document-meta PUBLIC '-//OpenOffice.org//DTD OfficeDocument 1.0//EN' 'office.dtd'>\n",
	"<office:document-meta xmlns:office='http://openoffice.org/2000/office' xmlns:xlink='http://www.w3.org/1999/xlink' xmlns:dc='http://purl.org/dc/elements/1.1/' xmlns:meta='http://openoffice.org/2000/meta' office:version='1.0'>\n",
	"<office:meta>\n",
	"<meta:generator>AbiWord</meta:generator>\n"
      };

    static const char * const postamble [] =
      {
	"</office:meta>\n",
	"</office:document-meta>\n"
      };

    writeToStream(meta, preamble, NrElements(preamble));

    UT_String meta_val, val;

    if (pDoc->getMetaDataProp(PD_META_KEY_DATE, meta_val) && meta_val.size()) {
      val = UT_String_sprintf("<dc:date>%s</dc:date>\n", meta_val.c_str());
      gsf_output_write(meta, val.size(), reinterpret_cast<const guint8*>(val.c_str()));
    }
    if (pDoc->getMetaDataProp(PD_META_KEY_LANGUAGE, meta_val) && meta_val.size()) {
      val = UT_String_sprintf("<dc:language>%s</dc:language>\n", meta_val.c_str());
      gsf_output_write(meta, val.size(), reinterpret_cast<const guint8*>(val.c_str()));
    }

    writeToStream(meta, postamble, NrElements(postamble));

    gsf_output_close(meta);
    g_object_unref(G_OBJECT(meta));
    return true;
  }

private:
  OO_MetaDataWriter ();
};

/*****************************************************************************/
/*****************************************************************************/

/*!
 * Class holding 1 static member. Its sole duty is to write
 * out a OOo settings file. Probably will just dump "standard"
 * info to the settings file, since Abi pretty much ignores
 * OOo's settings file on import.
 */
class OO_SettingsWriter
{
public:

  static bool writeSettings(PD_Document * pDoc, GsfOutfile * oo)
  {
    GsfOutput * settings = gsf_outfile_new_child (oo, "settings.xml", FALSE);

    static const char * const preamble [] = 
      {
	"<?xml version='1.0' encoding='UTF-8'?>\n",
	"<!DOCTYPE office:document-settings PUBLIC '-//OpenOffice.org//DTD OfficeDocument 1.0//EN' 'office.dtd'>\n",
	"<office:document-settings xmlns:office='http://openoffice.org/2000/office' xmlns:xlink='http://www.w3.org/1999/xlink' xmlns:config='http://openoffice.org/2001/config' office:version='1.0'>\n",
	"<office:settings>\n",
	"</office:settings>\n",
	"</office:document-settings>"
      };

    writeToStream (settings, preamble, NrElements(preamble));

    gsf_output_close(settings);
    g_object_unref(G_OBJECT(settings));

    return true;
  }

private:
  OO_SettingsWriter ();
};

/*****************************************************************************/
/*****************************************************************************/

/*!
 * Class holding 1 static member. Its sole duty is to write
 * out any pictures from inside the Abi document to the
 * OOo pictures directory
 */
class OO_PicturesWriter
{
public:
  
  static bool writePictures(PD_Document * pDoc, GsfOutfile * oo)
  {
    const char * szName;
    const char * szMimeType;
    const UT_ByteBuf * pByteBuf;

    // create Pictures directory
    GsfOutput * pictures = gsf_outfile_new_child(oo, "Pictures", TRUE);
    
    for (UT_uint32 k=0; (pDoc->enumDataItems(k,NULL,&szName,&pByteBuf,reinterpret_cast<const void **>(const_cast<const char **>(&szMimeType)))); k++)
      {
	// create individual pictures
	UT_String name = UT_String_sprintf("IMG-%d.png", k);
	GsfOutput * img = gsf_outfile_new_child(GSF_OUTFILE(pictures), name.c_str(), FALSE);
	gsf_output_write(img, pByteBuf->getLength(),  pByteBuf->getPointer(0));
	gsf_output_close(img);
	g_object_unref(G_OBJECT(img));
      }

    gsf_output_close(pictures);
    g_object_unref(G_OBJECT(pictures));

    return true;
  }

private:

  OO_PicturesWriter ();
};

/*****************************************************************************/
/*****************************************************************************/

/*!
 * Class holding 1 static member. Its sole duty is to create 
 * the OOo manifest file
 */
class OO_ManifestWriter
{
public:

  static bool writeManifest(PD_Document * pDoc, GsfOutfile * oo)
  {
    // create Pictures directory
    GsfOutput * meta_inf = gsf_outfile_new_child(oo, "META-INF", TRUE);
    GsfOutput * manifest = gsf_outfile_new_child(GSF_OUTFILE(meta_inf), "manifest.xml", FALSE);

    UT_String name;

    static const char * const preamble [] = 
      {
	"<?xml version='1.0' encoding='UTF-8'?>\n",
	"<!DOCTYPE manifest:manifest PUBLIC '-//OpenOffice.org//DTD Manifest 1.0//EN' 'Manifest.dtd'>\n",
	"<manifest:manifest xmlns:manifest='http://openoffice.org/2001/manifest'>\n",
	"<manifest:file-entry manifest:media-type='application/vnd.sun.xml.writer' manifest:full-path='/'/>\n",
	"<manifest:file-entry manifest:media-type='text/xml' manifest:full-path='content.xml'/>\n",
	"<manifest:file-entry manifest:media-type='text/xml' manifest:full-path='styles.xml'/>\n",
	"<manifest:file-entry manifest:media-type='text/xml' manifest:full-path='meta.xml'/>\n",
	"<manifest:file-entry manifest:media-type='text/xml' manifest:full-path='settings.xml'/>\n"
      };

    static const char * const postamble [] =
      {
	"</manifest:manifest>\n"
      };


    writeToStream (manifest, preamble, NrElements(preamble));

    const char * szName;
    const char * szMimeType;
    const UT_ByteBuf * pByteBuf;
    for (UT_uint32 k = 0; (pDoc->enumDataItems(k,NULL,&szName,&pByteBuf,reinterpret_cast<const void **>(const_cast<const char **>(&szMimeType)))); k++)
      {
	if (k == 0) {
	  name = "<manifest:file-entry manifest:media-type='' manifest:full-path='Pictures/'/>\n";
	  gsf_output_write(manifest, name.size(), reinterpret_cast<const guint8 *>(name.c_str()));
	}
	  
	name = UT_String_sprintf("<manifest:file-entry manifest:media-type='%s' manifest:full-path='Pictures/IMG-%d.png'/>\n", k, szMimeType);
	gsf_output_write (manifest, name.size(), reinterpret_cast<const guint8 *>(name.c_str()));
      }

    writeToStream (manifest, postamble, NrElements(postamble));

    gsf_output_close(manifest);
    g_object_unref(G_OBJECT(manifest));
    gsf_output_close(meta_inf);
    g_object_unref(G_OBJECT(meta_inf));

    return true;
  }

private:
  OO_ManifestWriter ();
};

/*****************************************************************************/
/*****************************************************************************/

/*!
 * Class holding 1 static member. Its sole duty is to write
 * out a OOo styles file. For now, we just dump "standard"
 * info to the settings file, putting the actual styles definitions
 * in the <automatic-styles> section of the content.
 */
class OO_StylesWriter
{
public:

  static bool writeStyles(PD_Document * pDoc, GsfOutfile * oo)
  {
     GsfOutput * styleStream = gsf_outfile_new_child (oo, "styles.xml", FALSE);
     
     static const char * const preamble [] = 
	{
	   // TODO!
	   
	   "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n",
	   "<!DOCTYPE office:document-styles PUBLIC \"-//OpenOffice.org//DTD OfficeDocument 1.0//EN\" \"office.dtd\">\n",
	   "<office:document-styles xmlns:office=\"http://openoffice.org/2000/office\" xmlns:style=\"http://openoffice.org/2000/style\" xmlns:text=\"http://openoffice.org/2000/text\" xmlns:table=\"http://openoffice.org/2000/table\" xmlns:draw=\"http://openoffice.org/2000/drawing\" xmlns:fo=\"http://www.w3.org/1999/XSL/Format\" xmlns:xlink=\"http://www.w3.org/1999/xlink\" xmlns:number=\"http://openoffice.org/2000/datastyle\" xmlns:svg=\"http://www.w3.org/2000/svg\" xmlns:chart=\"http://openoffice.org/2000/chart\" xmlns:dr3d=\"http://openoffice.org/2000/dr3d\" xmlns:math=\"http://www.w3.org/1998/Math/MathML\" xmlns:form=\"http://openoffice.org/2000/form\" xmlns:script=\"http://openoffice.org/2000/script\" office:version=\"1.0\">\n",
	   "<office:font-decls>\n",
	   "<style:font-decl style:name=\"Arial Unicode MS\" fo:font-family=\"'Arial Unicode MS'\" style:font-pitch=\"variable\"/>\n",
	   "<style:font-decl style:name=\"HG Mincho Light J\" fo:font-family=\"'HG Mincho Light J'\" style:font-pitch=\"variable\"/>\n",
	   "<style:font-decl style:name=\"Nimbus Roman No9 L\" fo:font-family=\"'Nimbus Roman No9 L'\" style:font-family-generic=\"roman\" style:font-pitch=\"variable\"/>\n",
	   "</office:font-decls>\n",
	   "<office:styles>\n",
	   "<style:default-style style:family=\"graphics\">\n",
	   "<style:properties draw:start-line-spacing-horizontal=\"0.283cm\" draw:start-line-spacing-vertical=\"0.283cm\" draw:end-line-spacing-horizontal=\"0.283cm\" draw:end-line-spacing-vertical=\"0.283cm\" fo:color=\"#000000\" style:font-name=\"Nimbus Roman No9 L\" fo:font-size=\"12pt\" fo:language=\"en\" fo:country=\"US\" style:font-name-asian=\"HG Mincho Light J\" style:font-size-asian=\"12pt\" style:language-asian=\"none\" style:country-asian=\"none\" style:font-name-complex=\"Arial Unicode MS\" style:font-size-complex=\"12pt\" style:language-complex=\"none\" style:country-complex=\"none\" style:text-autospace=\"ideograph-alpha\" style:punctuation-wrap=\"simple\" style:line-break=\"strict\">\n",
	   "<style:tab-stops/>\n",
	   "</style:properties>\n",
	   "</style:default-style>\n",
	   "<style:default-style style:family=\"paragraph\">\n",
	   "<style:properties fo:color=\"#000000\" style:font-name=\"Nimbus Roman No9 L\" fo:font-size=\"12pt\" fo:language=\"en\" fo:country=\"US\" style:font-name-asian=\"HG Mincho Light J\" style:font-size-asian=\"12pt\" style:language-asian=\"none\" style:country-asian=\"none\" style:font-name-complex=\"Arial Unicode MS\" style:font-size-complex=\"12pt\" style:language-complex=\"none\" style:country-complex=\"none\" fo:hyphenate=\"false\" fo:hyphenation-remain-char-count=\"2\" fo:hyphenation-push-char-count=\"2\" fo:hyphenation-ladder-count=\"no-limit\" style:text-autospace=\"ideograph-alpha\" style:punctuation-wrap=\"hanging\" style:line-break=\"strict\" style:tab-stop-distance=\"2.205cm\"/>\n",
	   "</style:default-style>\n",
	   "<style:style style:name=\"Standard\" style:family=\"paragraph\" style:class=\"text\"/>\n",
	   "<text:outline-style>\n",
	   "<text:outline-level-style text:level=\"1\" style:num-format=\"\"/>\n",
	   "<text:outline-level-style text:level=\"2\" style:num-format=\"\"/>\n",
	   "<text:outline-level-style text:level=\"3\" style:num-format=\"\"/>\n",
	   "<text:outline-level-style text:level=\"4\" style:num-format=\"\"/>\n",
	   "<text:outline-level-style text:level=\"5\" style:num-format=\"\"/>\n",
	   "<text:outline-level-style text:level=\"6\" style:num-format=\"\"/>\n",
	   "<text:outline-level-style text:level=\"7\" style:num-format=\"\"/>\n",
	   "<text:outline-level-style text:level=\"8\" style:num-format=\"\"/>\n",
	   "<text:outline-level-style text:level=\"9\" style:num-format=\"\"/>\n",
	   "<text:outline-level-style text:level=\"10\" style:num-format=\"\"/>\n",
	   "</text:outline-style>\n",
	   "<text:footnotes-configuration style:num-format=\"1\" text:start-value=\"0\" text:footnotes-position=\"page\" text:start-numbering-at=\"document\"/>\n",
	   "<text:endnotes-configuration style:num-format=\"i\" text:start-value=\"0\"/>\n",
	   "<text:linenumbering-configuration text:number-lines=\"false\" text:offset=\"0.499cm\" style:num-format=\"1\" text:number-position=\"left\" text:increment=\"5\"/>\n",
	   "</office:styles>\n",
	   "<office:automatic-styles>\n",
	   "<style:page-master style:name=\"pm1\">\n",
	   "<style:properties fo:page-width=\"21.59cm\" fo:page-height=\"27.94cm\" style:num-format=\"1\" style:print-orientation=\"portrait\" fo:margin-top=\"2.54cm\" fo:margin-bottom=\"2.54cm\" fo:margin-left=\"3.175cm\" fo:margin-right=\"3.175cm\" style:footnote-max-height=\"0cm\">\n",
	   "<style:footnote-sep style:width=\"0.018cm\" style:distance-before-sep=\"0.101cm\" style:distance-after-sep=\"0.101cm\" style:adjustment=\"left\" style:rel-width=\"25%\" style:color=\"#000000\"/>\n",
	   "</style:properties>\n",
	   "<style:header-style/>\n",
	   "<style:footer-style/>\n",
	   "</style:page-master>\n",
	   "</office:automatic-styles>\n",
	   "<office:master-styles>\n",
	   "<style:master-page style:name=\"Standard\" style:page-master-name=\"pm1\"/>\n",
	   "</office:master-styles>\n",
	   "</office:document-styles>\n"
      };
     
     writeToStream(styleStream, preamble, NrElements(preamble));
     
     
     gsf_output_close(styleStream);
     g_object_unref(G_OBJECT(styleStream));
     
     return true;
  }

private:
  OO_StylesWriter ();
};

/*****************************************************************************/
/*****************************************************************************/


IE_Exp_OpenWriter::IE_Exp_OpenWriter (PD_Document * pDoc)
  : IE_Exp (pDoc), m_oo(0)
{
}

IE_Exp_OpenWriter::~IE_Exp_OpenWriter()
{
}

/*!
 * This writes out our AbiWord file as an OpenOffice
 * compound document
 */
UT_Error IE_Exp_OpenWriter::_writeDocument(void)
{
  UT_return_val_if_fail(m_oo, UT_ERROR);

  if (!OO_MetaDataWriter::writeMetaData(getDoc(), m_oo))
    return UT_ERROR;

  if (!OO_SettingsWriter::writeSettings(getDoc(), m_oo))
    return UT_ERROR;

  if (!OO_PicturesWriter::writePictures(getDoc(), m_oo))
    return UT_ERROR;

  if (!OO_ManifestWriter::writeManifest(getDoc(), m_oo))
    return UT_ERROR;

  if (!OO_StylesWriter::writeStyles(getDoc(), m_oo))
    return UT_ERROR;


  OO_StylesContainer stylesContainer;
  OO_AccumulatorImpl accumulatorImpl(&stylesContainer);
  OO_Listener listener1(getDoc(), this, &accumulatorImpl);
  if (!getDoc()->tellListener(static_cast<PL_Listener *>(&listener1)))
    return UT_ERROR;

  OO_WriterImpl writerImpl(m_oo, &stylesContainer);
  OO_Listener listener2(getDoc(), this, &writerImpl);
  if (!getDoc()->tellListener(static_cast<PL_Listener *>(&listener2)))
    return UT_ERROR;
  listener2.endDocument();

  return UT_OK;
}

/*!
 * Open the underlying ZIP file for writing
 */
bool IE_Exp_OpenWriter::_openFile(const char * szFilename)
{
  UT_return_val_if_fail(!m_oo, false);
  
  GsfOutput * sink = GSF_OUTPUT (gsf_output_stdio_new (szFilename, NULL));
  if (!sink)
    return false;

  m_oo = GSF_OUTFILE (gsf_outfile_zip_new (sink, NULL));
  g_object_unref(G_OBJECT(sink));

  return (m_oo != 0);
}

/*!
 * Close our underlying ZIP file
 */
bool IE_Exp_OpenWriter::_closeFile(void)
{
  if(m_oo) {
    gsf_output_close(GSF_OUTPUT(m_oo));
    g_object_unref(G_OBJECT(m_oo));
    m_oo = 0;
  }

  return true;
}

/*****************************************************************************/
/*****************************************************************************/
