/*
 *  Copyright (C) 2000, 2001, 2002 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include "galeon-embed-persist.h"
#include "galeon-debug.h"

#if !defined (ENABLE_MOZILLA_EMBED) && !defined (ENABLE_GTKHTML_EMBED)
#error "At least one embed needs to be enabled"
#endif

#ifdef ENABLE_MOZILLA_EMBED
#include "mozilla-embed.h"
#include "mozilla-embed-persist.h"
#endif

#ifdef ENABLE_GTKHTML_EMBED
#include "gtkhtml-embed.h"
#include "gtkhtml-embed-persist.h"
#endif

#include <glib/gi18n.h>

#define GALEON_EMBED_PERSIST_GET_PRIVATE(object) (G_TYPE_INSTANCE_GET_PRIVATE ((object), \
						  GALEON_EMBED_PERSIST_TYPE, GaleonEmbedPersistPrivate))
enum
{
        COMPLETED,
        CANCELLED,
        LAST_SIGNAL
};

enum
{
	PROP_0,
	PROP_EMBED,
	PROP_SOURCE,
	PROP_DEST,
	PROP_MAX_SIZE,
	PROP_FLAGS,
	PROP_HANDLER,
	PROP_FC_PARENT,
	PROP_FC_TITLE
};

struct GaleonEmbedPersistPrivate
{
	char *dir;
	char *src;
	PersistHandlerInfo *handler;
	GaleonEmbed *embed;
	int max_size;
	EmbedPersistFlags flags;
	GtkWidget *fc_parent;
	char *fc_title;
};


static GObjectClass *parent_class = NULL;
static guint galeon_embed_persist_signals[LAST_SIGNAL] = { 0 };


gresult 
galeon_embed_persist_set_source (GaleonEmbedPersist *persist,
				 const char *url)
{
	persist->priv->src = g_strdup(url);

	g_object_notify (G_OBJECT(persist), "source");

	return G_OK;
}

gresult 
galeon_embed_persist_get_source (GaleonEmbedPersist *persist,
				 const char **url)
{
	*url = persist->priv->src;
	return G_OK;
}

gresult 
galeon_embed_persist_set_dest (GaleonEmbedPersist *persist,
			       const char *dir)
{
	persist->priv->dir = g_strdup (dir);

	g_object_notify (G_OBJECT(persist), "dest");

	return G_OK;
}

gresult 
galeon_embed_persist_get_dest (GaleonEmbedPersist *persist,
			       const char **dir)
{
	*dir = persist->priv->dir;
	return G_OK;
}
						     
gresult 
galeon_embed_persist_set_max_size (GaleonEmbedPersist *persist,
			           int kb_size)
{
	persist->priv->max_size = kb_size;

	g_object_notify (G_OBJECT(persist), "max_size");
	
	return G_OK;
}

gresult
galeon_embed_persist_set_embed (GaleonEmbedPersist *persist,
		                GaleonEmbed *embed)
{
	persist->priv->embed = embed;

	g_object_notify (G_OBJECT(persist), "embed");

	return G_OK;
}

gresult
galeon_embed_persist_get_embed (GaleonEmbedPersist *persist,
		                GaleonEmbed **embed)
{
	*embed = persist->priv->embed;
	return G_OK;
}

gresult
galeon_embed_persist_set_flags (GaleonEmbedPersist *persist,
		                EmbedPersistFlags flags)
{
	persist->priv->flags = flags;

	g_object_notify (G_OBJECT(persist), "flags");

	return G_OK;
}

gresult
galeon_embed_persist_get_flags (GaleonEmbedPersist *persist,
		                EmbedPersistFlags *flags)
{
	*flags = persist->priv->flags;
	return G_OK;
}

gresult 
galeon_embed_persist_set_handler (GaleonEmbedPersist *persist,
				  const char *command,
				  gboolean need_terminal)
{
	persist->priv->handler = g_new0 (PersistHandlerInfo, 1);
	persist->priv->handler->command = g_strdup (command);
	persist->priv->handler->need_terminal = need_terminal;

	g_object_notify (G_OBJECT(persist), "handler");

	return G_OK;
}

gresult
galeon_embed_persist_get_fc_title (GaleonEmbedPersist *persist,
				   const char **title)
{
	*title = persist->priv->fc_title;
	return G_OK;
}

gresult
galeon_embed_persist_set_fc_title (GaleonEmbedPersist *persist,
				   const char *title)
{
	g_free (persist->priv->fc_title);
	persist->priv->fc_title = g_strdup (title);
	return G_OK;
}

gresult
galeon_embed_persist_get_fc_parent (GaleonEmbedPersist *persist,
				    GtkWidget **parent)
{
	*parent = persist->priv->fc_parent;
	return G_OK;
}

gresult
galeon_embed_persist_set_fc_parent (GaleonEmbedPersist *persist,
				    GtkWidget *parent)
{
	persist->priv->fc_parent = parent;
	return G_OK;
}

gresult 
galeon_embed_persist_save (GaleonEmbedPersist *persist)
{
	GaleonEmbedPersistClass *klass = GALEON_EMBED_PERSIST_GET_CLASS (persist);
	return klass->save (persist);
}


static void 
galeon_embed_persist_set_property (GObject *object,
                                   guint prop_id,
                                   const GValue *value,
                                   GParamSpec *pspec)
{
	GaleonEmbedPersist *persist;
	PersistHandlerInfo *h;
	
	persist = GALEON_EMBED_PERSIST (object);
  
	switch (prop_id)
	{
		case PROP_EMBED:
			galeon_embed_persist_set_embed (persist,
							g_value_get_object (value));
			break;
		case PROP_SOURCE:
			galeon_embed_persist_set_source (persist,
							 g_value_get_string (value));
			break;
		case PROP_DEST:
			galeon_embed_persist_set_dest  (persist,
							 g_value_get_string (value));
			break;
		case PROP_MAX_SIZE:
			galeon_embed_persist_set_max_size  (persist,
							    g_value_get_int (value));
			break;
		case PROP_FLAGS:
			galeon_embed_persist_set_flags 
				(persist,
				(EmbedPersistFlags)g_value_get_int (value));
			break;
		case PROP_HANDLER:
			h = (PersistHandlerInfo *)g_value_get_pointer (value);
			galeon_embed_persist_set_handler
				(persist, h->command, h->need_terminal);

			break;
		case PROP_FC_TITLE:
			galeon_embed_persist_set_fc_title (persist,
							   g_value_get_string (value));
			break;

		case PROP_FC_PARENT:
			galeon_embed_persist_set_fc_parent (persist,
						  GTK_WIDGET (g_value_get_object (value)));
			break;
	}
}

static void 
galeon_embed_persist_get_property (GObject *object,
                                   guint prop_id,
                                   GValue *value,
                                   GParamSpec *pspec)
{
	GaleonEmbedPersist *persist;

	persist = GALEON_EMBED_PERSIST (object);
  
	switch (prop_id)
	{
		case PROP_EMBED:
			g_value_set_object (value, persist->priv->embed);
			break;
		case PROP_SOURCE:
			g_value_set_string (value, persist->priv->src);
			break;
		case PROP_DEST:
			g_value_set_string (value, persist->priv->dir);
			break;
		case PROP_MAX_SIZE:
			g_value_set_int (value, persist->priv->max_size);
			break;
		case PROP_FLAGS:
			g_value_set_int (value, (int)persist->priv->flags);
			break;
		case PROP_HANDLER:
			g_value_set_pointer (value, persist->priv->handler);
			break;
	        case PROP_FC_TITLE:
			g_value_set_string (value, persist->priv->fc_title);
			break;
		case PROP_FC_PARENT:
			g_value_set_object (value, persist->priv->fc_parent);
			break;
	}
}

static void
galeon_embed_persist_init (GaleonEmbedPersist *persist)
{
        persist->priv = GALEON_EMBED_PERSIST_GET_PRIVATE (persist);
	LOG ("GaleonEmbedPersist ctor (%p)", persist);
}

static void
galeon_embed_persist_finalize (GObject *object)
{
        GaleonEmbedPersist *persist = GALEON_EMBED_PERSIST (object);

	g_free (persist->priv->dir);
	g_free (persist->priv->src);
	g_free (persist->priv->fc_title);
	
	if (persist->priv->handler)
	{
		g_free (persist->priv->handler->command);
		g_free (persist->priv->handler);
	}
	
	LOG ("GaleonEmbedPersist dtor (%p)", persist);
        G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
galeon_embed_persist_class_init (GaleonEmbedPersistClass *klass)
{
        GObjectClass *object_class = G_OBJECT_CLASS (klass);

        parent_class = g_type_class_peek_parent (klass);
	
        object_class->finalize = galeon_embed_persist_finalize;
        object_class->set_property = galeon_embed_persist_set_property;
	object_class->get_property = galeon_embed_persist_get_property;
	
	/* init signals */
        galeon_embed_persist_signals[COMPLETED] =
                g_signal_new ("completed",
                              G_OBJECT_CLASS_TYPE (object_class),
                              G_SIGNAL_RUN_LAST,
                              G_STRUCT_OFFSET (GaleonEmbedPersistClass, completed),
                              NULL, NULL,
                              g_cclosure_marshal_VOID__VOID,
                              G_TYPE_NONE,
                              0);

        galeon_embed_persist_signals[CANCELLED] =
                g_signal_new ("cancelled",
                              G_OBJECT_CLASS_TYPE (object_class),
                              G_SIGNAL_RUN_LAST,
                              G_STRUCT_OFFSET (GaleonEmbedPersistClass, cancelled),
                              NULL, NULL,
                              g_cclosure_marshal_VOID__VOID,
                              G_TYPE_NONE,
                              0);

	g_object_class_install_property (object_class,
                                         PROP_EMBED,
                                         g_param_spec_object ("embed",
                                                              "Embed",
                                                              "The embed containing the document",
                                                              G_TYPE_OBJECT,
                                                              G_PARAM_READWRITE));
	g_object_class_install_property (object_class,
                                         PROP_SOURCE,
                                         g_param_spec_string  ("source",
                                                               "Source",
                                                               "Url of the document to save",
                                                               NULL,
                                                               G_PARAM_READWRITE));
	
	g_object_class_install_property (object_class,
                                         PROP_DEST,
                                         g_param_spec_string ("dest",
                                                              "Destination",
                                                              "Destination directory",
                                                              NULL,
                                                              G_PARAM_READWRITE));

	g_object_class_install_property (object_class,
                                         PROP_MAX_SIZE,
                                         g_param_spec_int    ("max_size",
                                                              "Maxsize",
                                                              "Maximum size of the file",
                                                              0,
							      G_MAXINT,
							      0,
                                                              G_PARAM_READWRITE));

	g_object_class_install_property (object_class,
                                         PROP_FLAGS,
                                         g_param_spec_int    ("flags",
                                                              "Flags",
                                                              "Flags",
                                                              0,
							      G_MAXINT,
							      0,
                                                              G_PARAM_READWRITE));

	g_object_class_install_property (object_class,
                                         PROP_HANDLER,
                                         g_param_spec_pointer ("handler",
                                                              "Handler",
                                                              "Handler",
                                                              G_PARAM_READWRITE));


	g_object_class_install_property (object_class,
                                         PROP_FC_PARENT,
                                         g_param_spec_object ("fc_parent",
                                                              "File choose parent window",
                                                              "File choose parent window",
							      GTK_TYPE_WIDGET,
                                                              G_PARAM_READWRITE));

	g_object_class_install_property (object_class,
                                         PROP_FC_TITLE,
                                         g_param_spec_string ("fc_title",
                                                              "File chooser title",
                                                              "File chooser title",
                                                              NULL,
                                                              G_PARAM_READWRITE));
	g_type_class_add_private (object_class, sizeof (GaleonEmbedPersistPrivate));
}


G_DEFINE_TYPE (GaleonEmbedPersist, galeon_embed_persist, G_TYPE_OBJECT);


GaleonEmbedPersist *
galeon_embed_persist_new (GaleonEmbed *embed)
{
	GaleonEmbedPersist *persist;
	GType type = 0;

#ifdef ENABLE_MOZILLA_EMBED
	if (IS_MOZILLA_EMBED (embed))
	{
		type = MOZILLA_EMBED_PERSIST_TYPE;
	}
#endif

#ifdef ENABLE_GTKHTML_EMBED
	if (IS_GTKHTML_EMBED (embed))
	{
		type = GTKHTML_EMBED_PERSIST_TYPE;
	}
#endif
	if (type == 0)
	{
#ifdef ENABLE_GTKHTML_EMBED
		type = GTKHTML_EMBED_PERSIST_TYPE;
#elif ENABLE_MOZILLA_EMBED
		type = MOZILLA_EMBED_PERSIST_TYPE;
#endif
	}

	g_assert (type != 0);
	
        persist = GALEON_EMBED_PERSIST (g_object_new (type, 
						      "embed", embed,
						      NULL));

        return persist;
}
