/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999-2000 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>
                2001-2003 by Stanislav Visnovsky <visnovsky@kde.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

**************************************************************************** */
#include "prefwidgets.h"
#include "resources.h"
#include "cmdedit.h"

#include <kcombobox.h>
#include <klocale.h>
#include <kdialog.h>
#include <kfiledialog.h>
#include <kmessagebox.h>
#include <klineedit.h> 
#include <kurlcompletion.h>
#include <kfontdialog.h>
#include <kcolorbutton.h>
#include <ksconfig.h>
#include <kurldrag.h>
#include <kurlrequester.h>

#include <qlayout.h>
#include <qobjectlist.h>
#include <qlabel.h>
#include <qvbox.h>
#include <qlineedit.h>
#include <qcheckbox.h>
#include <qgroupbox.h>
#include <qhbuttongroup.h>
#include <qpushbutton.h>
#include <qwhatsthis.h>


static QSize sizeHintForWidget(const QWidget* widget)
{
  //
  // The size is computed by adding the sizeHint().height() of all
  // widget children and taking the width of the widest child and adding
  // layout()->margin() and layout()->spacing()
  //

  QSize size;

  int numChild = 0;
  QObjectList *l = (QObjectList*)(widget->children());

  for( uint i=0; i < l->count(); i++ )
  {
    QObject *o = l->at(i);
    if( o->isWidgetType() )
    {
      numChild += 1;
      QWidget *w=((QWidget*)o);

      QSize s = w->sizeHint();
      if( s.isEmpty() == true )
      {
          s = QSize( 50, 100 ); // Default size
      }
      size.setHeight( size.height() + s.height() );
      if( s.width() > size.width() ) { size.setWidth( s.width() ); }
    }
  }

  if( numChild > 0 )
  {
    size.setHeight( size.height() + widget->layout()->spacing()*(numChild-1) );
    size += QSize( widget->layout()->margin()*2, widget->layout()->margin()*2 + 1 );
  }
  else
  {
    size = QSize( 1, 1 );
  }

  return( size );
}


GeneralPreferences::GeneralPreferences(QWidget* parent)
         : QWidget(parent)
{
    QWidget* page = this;

    QVBoxLayout* layout=new QVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());

    QGroupBox* box=new QGroupBox(1,Qt::Horizontal,page);
    box->setMargin(KDialog::marginHint());
    layout->addWidget(box);

    QLabel* label=new QLabel(i18n("&Base folder of PO files:"),box);
    QHBox* hbox = new QHBox(box);
    hbox->setSpacing(KDialog::spacingHint());

    KFile::Mode mode = static_cast<KFile::Mode>( KFile::Directory | KFile::ExistingOnly | KFile::LocalOnly );
	
												
    _poDirEdit = new KURLRequester(hbox);
    _poDirEdit->setMode( mode );
    _poDirEdit->setMinimumSize(250,_poDirEdit->sizeHint().height());
    label->setBuddy(_poDirEdit);


    label=new QLabel(i18n("Ba&se folder of POT files:"),box);
    hbox = new QHBox(box);
    hbox->setSpacing(KDialog::spacingHint());

    _potDirEdit = new KURLRequester(hbox);
    _potDirEdit->setMode( mode );
    _potDirEdit->setMinimumSize(250,_potDirEdit->sizeHint().height());
    label->setBuddy(_potDirEdit);



    QWhatsThis::add(box,i18n("<qt><p><b>Base folders</b></p>\n"
     "<p>Type in the folders which contain all your PO and POT files.\n"
     "The files and the folders in these folders will then be merged into one\n"
     "tree.</p></qt>"));


    box=new QGroupBox(1,Qt::Horizontal,page);
    box->setMargin(KDialog::marginHint());
    layout->addWidget(box);

    _openWindowButton = new QCheckBox(i18n("O&pen files in new window"),box);


    QWhatsThis::add(_openWindowButton,i18n("<qt><p><b>Open files in new window</b></p>\n"
"<p>If this is activated all files that are opened from the Catalog Manager are opened\n"
"in a new window.</p></qt>"));

    _killButton = new QCheckBox( i18n("&Kill processes on exit") , box );

    QWhatsThis::add( _killButton , i18n("<qt><p><b>Kill processes on exit</b></p>\n"
"<p>If you check this, KBabel tries to kill the processes, that have not exited already when KBabel exits,\n"
"by sending a kill signal to them.</p>\n"
"<p>NOTE: It's not guaranteed that the processes will be killed.</p></qt>") );


    _indexButton = new QCheckBox( i18n("Create inde&x for file contents"), box );
    
    QWhatsThis::add( _indexButton , i18n("<qt><p><b>Create index for file contents</b></p>\n"
"<p>If you check this, KBabel will create an index for each PO file to speed up the find/replace functions.</p>\n"
"<p>NOTE: This will slow down updating the file information considerably.</p></qt>") );


    layout->addStretch(1);

    page->setMinimumSize(sizeHintForWidget(page));

    defaults();

    setMinimumSize(sizeHint());
}


GeneralPreferences::~GeneralPreferences()
{
}


void GeneralPreferences::setSettings(const CatManSettings& settings)
{
   _poDirEdit->setURL(settings.poBaseDir);
   _potDirEdit->setURL(settings.potBaseDir);

   _openWindowButton->setChecked(settings.openWindow);

   _killButton->setChecked( settings.killCmdOnExit );
   _indexButton->setChecked( settings.indexWords );
}


void GeneralPreferences::mergeSettings(CatManSettings& settings) const
{
    settings.poBaseDir=_poDirEdit->url();
    settings.potBaseDir=_potDirEdit->url();
    settings.openWindow=_openWindowButton->isChecked();

    settings.killCmdOnExit = _killButton->isChecked() ;
    settings.indexWords = _indexButton->isChecked() ;
}

void GeneralPreferences::defaults()
{
   _poDirEdit->setURL(Defaults::CatalogManager::poBaseDir());
   _potDirEdit->setURL(Defaults::CatalogManager::potBaseDir());

   _openWindowButton->setChecked(Defaults::CatalogManager::openWindow);

   _killButton->setChecked( Defaults::CatalogManager::killCmdOnExit );
   _indexButton->setChecked( Defaults::CatalogManager::indexWords );
}

DirCommandsPreferences::DirCommandsPreferences(QWidget* parent)
         : QWidget(parent)
{
    QWidget* page = this;

    QVBoxLayout* layout=new QVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());

    QGroupBox* box = new QGroupBox( 1 , Qt::Horizontal , i18n("Commands for Folders") , page );
    box->setMargin( KDialog::marginHint() );
    layout->addWidget( box );

    _dirCmdEdit = new CmdEdit( box );
    new QLabel( i18n("Replaceables:\n@PACKAGE@, @PODIR@, @POTDIR@\n"
	"@POFILES@, @MARKEDPOFILES@"), box);

    QWhatsThis::add( box , i18n("<qt><p><b>Commands for folders</b></p>"
"<p>Insert here the commands you want to execute in folders from "
"the Catalog Manager. The commands are then shown in the submenu "
"<b>Commands</b> in the Catalog Manager's context menu.</p>"
"<p>The following strings will be replaced in a command:<ul>"
"<li>@PACKAGE@: The name of the folder without path</li>"
"<li>@PODIR@: The name of the PO-folder with path</li>"
"<li>@POTDIR@: The name of the template folder with path</li>"
"<li>@POFILES@: The names of the PO files with path</li>"
"<li>@MARKEDPOFILES@: The names of the marked PO files with path</li>"
"</ul></p>"
"</qt>") );



    layout->addStretch(1);
    page->setMinimumSize(sizeHintForWidget(page));

    defaults();

    setMinimumSize(sizeHint());
}


DirCommandsPreferences::~DirCommandsPreferences()
{
}


void DirCommandsPreferences::setSettings(const CatManSettings& settings)
{
   _dirCmdEdit->setCommands( settings.dirCommands , settings.dirCommandNames );
}


void DirCommandsPreferences::mergeSettings(CatManSettings& settings) const
{
    _dirCmdEdit->commands( settings.dirCommands , settings.dirCommandNames );
}

void DirCommandsPreferences::defaults()
{
   _dirCmdEdit->setCommands( Defaults::CatalogManager::dirCommands()
                            , Defaults::CatalogManager::dirCommandNames() );
}


FileCommandsPreferences::FileCommandsPreferences(QWidget* parent)
         : QWidget(parent)
{
    QWidget* page = this;

    QVBoxLayout* layout=new QVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());

    QGroupBox* box=new QGroupBox( 1 , Qt::Horizontal , i18n("Commands for Files") , page );
    box->setMargin( KDialog::marginHint() );
    layout->addWidget( box );

    _fileCmdEdit = new CmdEdit( box );
    new QLabel( i18n("Replaceables:\n"
"@PACKAGE@, @POFILE@,@POTFILE@,\n@PODIR@, @POTDIR@"), box);



    QWhatsThis::add( box , i18n("<qt><p><b>Commands for files</b></p>"
"<p>Insert here the commands you want to execute on files from "
"the Catalog Manager. The commands are then shown in the submenu "
"<b>Commands</b> in the Catalog Manager's context menu.</p>"
"<p>The following strings will be replaced in a command:<ul>"
"<li>@PACKAGE@: The name of the file without path and extension</li>"
"<li>@POFILE@: The name of the PO-file with path and extension</li>"
"<li>@POTFILE@: The name of the corresponding template file with path "
"and extension</li>"
"<li>@POEMAIL@: The name and email address of the last translator</li>"
"<li>@PODIR@: The name of the folder the PO-file is in, with path</li>"
"<li>@POTDIR@: The name of the folder the template file is in, with "
"path</li></ul></p></qt>") );



    layout->addStretch(1);
    page->setMinimumSize(sizeHintForWidget(page));

    defaults();

    setMinimumSize(sizeHint());
}


FileCommandsPreferences::~FileCommandsPreferences()
{
}


void FileCommandsPreferences::setSettings(const CatManSettings& settings)
{
   _fileCmdEdit->setCommands( settings.fileCommands , settings.fileCommandNames );
}


void FileCommandsPreferences::mergeSettings(CatManSettings& settings) const
{
    _fileCmdEdit->commands( settings.fileCommands , settings.fileCommandNames );
}

void FileCommandsPreferences::defaults()
{
   _fileCmdEdit->setCommands( Defaults::CatalogManager::fileCommands()
                            , Defaults::CatalogManager::fileCommandNames() );
}

ViewPreferences::ViewPreferences(QWidget* parent)
         : QWidget(parent)
{
    QWidget* page = this;

    QVBoxLayout* layout=new QVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());

    QGroupBox* box=new QGroupBox(2, Qt::Horizontal,i18n("Shown Columns"),page);
    box->setMargin(KDialog::marginHint());
    layout->addWidget(box);

    _flagColumnCheckbox = new QCheckBox( i18n("Fla&g"), box );
    _fuzzyColumnCheckbox = new QCheckBox( i18n("&Fuzzy"), box );
    _untranslatedColumnCheckbox = new QCheckBox( i18n("&Untranslated"), box );
    _totalColumnCheckbox = new QCheckBox( i18n("&Total"), box );
    _cvsColumnCheckbox = new QCheckBox( i18n("&CVS status"), box );
    _revisionColumnCheckbox = new QCheckBox( i18n("Last &revision"), box );
    _translatorColumnCheckbox = new QCheckBox( i18n("Last t&ranslator"), box );

    QWhatsThis::add(box,i18n("<qt><p><b>Shown columns</b></p>\n"
     "<p></p></qt>"));
     
    layout->addStretch(1);

    page->setMinimumSize(sizeHintForWidget(page));

    defaults();

    setMinimumSize(sizeHint());
}


ViewPreferences::~ViewPreferences()
{
}


void ViewPreferences::setSettings(const CatManSettings& settings)
{
    _flagColumnCheckbox->setChecked(settings.flagColumn);
    _fuzzyColumnCheckbox->setChecked(settings.fuzzyColumn);
    _untranslatedColumnCheckbox->setChecked(settings.untranslatedColumn);
    _totalColumnCheckbox->setChecked(settings.totalColumn);
    _cvsColumnCheckbox->setChecked(settings.cvsColumn);
    _revisionColumnCheckbox->setChecked(settings.revisionColumn);
    _translatorColumnCheckbox->setChecked(settings.translatorColumn);
}


void ViewPreferences::mergeSettings(CatManSettings& settings) const
{
    settings.flagColumn = _flagColumnCheckbox->isChecked();
    settings.fuzzyColumn = _fuzzyColumnCheckbox->isChecked();
    settings.untranslatedColumn = _untranslatedColumnCheckbox->isChecked();
    settings.totalColumn = _totalColumnCheckbox->isChecked();
    settings.cvsColumn = _cvsColumnCheckbox->isChecked();
    settings.revisionColumn = _revisionColumnCheckbox->isChecked();
    settings.translatorColumn = _translatorColumnCheckbox->isChecked();
}

void ViewPreferences::defaults()
{
    _flagColumnCheckbox->setChecked(Defaults::CatalogManager::flagColumn);
    _fuzzyColumnCheckbox->setChecked(Defaults::CatalogManager::fuzzyColumn);
    _untranslatedColumnCheckbox->setChecked(Defaults::CatalogManager::untranslatedColumn);
    _totalColumnCheckbox->setChecked(Defaults::CatalogManager::totalColumn);
    _cvsColumnCheckbox->setChecked(Defaults::CatalogManager::cvsColumn);
    _revisionColumnCheckbox->setChecked(Defaults::CatalogManager::revisionColumn);
    _translatorColumnCheckbox->setChecked(Defaults::CatalogManager::translatorColumn);
}

#include "prefwidgets.moc"
