/*
 *  xcall - Packet radio program for GTK+
 *  Copyright (C) 2001 Joop Stakenborg <pa4tu@amsat.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* callbacks.c: functions that connect to a widget */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <sys/types.h>
#include <signal.h>
#include <stdlib.h>

#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include "packet.h"
#include "history.h"
#include "utils.h"
#include "init.h"
#include "record.h"
#include "proc.h"
#include "types.h"

extern GtkWidget *window1;
extern gboolean connected, connecting, recording;
extern gint connectpid, fprec, combotype;;
extern preferencestype preferences;
GtkWidget *preferencesdialog;
extern GList *axportnames, *recordhistory, *connecthistory;
gint buttonnumber;
extern gchar *remotecallsign;

/****************************** menu ****************************************/
/*************************** PROGRAM ****************************************/
void on_exit1_activate (GtkMenuItem *menuitem, gpointer user_data) {
  gtk_signal_emit_by_name(GTK_OBJECT(window1), "delete_event");
}


void on_record1_activate (GtkMenuItem *menuitem, gpointer user_data) {
  GString *recinfo = g_string_new("");

  if (!recording) {
    g_string_sprintf(recinfo, _("Select or enter a filename for recording"));
    combotype = RECORDON;
    showcombodialog(recinfo, _("%s - recording"));  
  }
  else {
    g_string_sprintf(recinfo, _("Turn recording off?"));
    combotype = RECORDOFF;
    showcombodialog(recinfo, _("%s - recording"));
  }
  g_string_free(recinfo, TRUE);
}

/****************************** HOST ****************************************/
void on_connect1_activate (GtkMenuItem *menuitem, gpointer user_data) {
  GString *cinfo = g_string_new("");

  if (!connected && !connecting) {
    g_string_sprintf(cinfo, _("Select or enter port and callsign for connection"));
    combotype = CONNECT;
    showcombodialog(cinfo, _("%s - connection"));
  } else if(!connected && connecting) {
    g_string_sprintf(cinfo, _("Kill connection to %s?"), remotecallsign);
    combotype = KILL;
    showcombodialog(cinfo, _("%s - connection"));
  } else if(connected) {
    g_string_sprintf(cinfo, _("Disconnect from %s?"), remotecallsign);
    combotype = DISCONNECT;
    showcombodialog(cinfo, _("%s - connection"));
  }
}


/****************************** SETTINGS ************************************/
void on_preferences1_activate (GtkMenuItem *menuitem, gpointer user_data) {
  GtkWidget *fontentry, *combo, *dlentry, *timeoutentry, *menucheck,
    *scrollcheck, *poscheck, *sizecheck, *dxcheck, *frame;
  gchar *timeout;

  preferencesdialog = create_dialog2();
  set_title(preferencesdialog, _("%s - preferences"));
  fontentry = lookup_widget(preferencesdialog, "entry2");
  combo = lookup_widget(preferencesdialog, "combo1");
  dlentry = lookup_widget(preferencesdialog, "entry5");
  timeoutentry = lookup_widget(preferencesdialog, "entry4");
  menucheck = lookup_widget(preferencesdialog, "checkbutton2");
  scrollcheck = lookup_widget(preferencesdialog, "checkbutton1");
  sizecheck = lookup_widget(preferencesdialog, "checkbutton4");
  poscheck = lookup_widget(preferencesdialog, "checkbutton5");
  dxcheck = lookup_widget(preferencesdialog, "dxcheckbutton");
  frame = lookup_widget(preferencesdialog, "frame8");

  /* set state of the check buttons */
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(menucheck), preferences.menu);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(scrollcheck), preferences.scrollbar);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(sizecheck), preferences.size);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(poscheck), preferences.origin);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dxcheck), preferences.dxcluster);

  /* dx cluster frame */
  gtk_widget_set_sensitive(frame, preferences.dxcluster);

  /* fill in the entries */
  gtk_entry_set_text(GTK_ENTRY(fontentry), preferences.font->str);
  gtk_entry_set_text(GTK_ENTRY(dlentry), preferences.downloaddir->str);
  timeout = g_strdup_printf("%d", preferences.timeout);
  gtk_entry_set_text(GTK_ENTRY(timeoutentry), timeout);

  /* fill in the popdown list */
  if (axportnames) gtk_combo_set_popdown_strings(GTK_COMBO(combo), axportnames);

  /* set color of the buttons */
  setbuttoncolor("button5", preferences.statusfgcolor);
  setbuttoncolor("button6", preferences.errorfgcolor);
  setbuttoncolor("button7", preferences.rxfgcolor);
  setbuttoncolor("button8", preferences.txfgcolor);
  setbuttoncolor("button9", preferences.backgroundcolor);
  setbuttoncolor("dxbgbutton", preferences.dxbackgroundcolor);
  setbuttoncolor("dx1_8button", preferences.dx1_8color);
  setbuttoncolor("dx3_5button", preferences.dx3_5color);
  setbuttoncolor("dx7button", preferences.dx7color);
  setbuttoncolor("dx10button", preferences.dx10color);
  setbuttoncolor("dx14button", preferences.dx14color);
  setbuttoncolor("dx18button", preferences.dx18color);
  setbuttoncolor("dx21button", preferences.dx21color);
  setbuttoncolor("dx24button", preferences.dx24color);
  setbuttoncolor("dx28button", preferences.dx28color);
  setbuttoncolor("dxotherbutton", preferences.dxothercolor);

  gtk_widget_set_sensitive(window1, 0);
  gtk_widget_show(preferencesdialog);
}

/******************************** ABOUT *************************************/
void on_about1_activate (GtkMenuItem *menuitem, gpointer user_data) {
  GString *info = g_string_new("");

  g_string_sprintf(info, _("%s version %s\n\
Packet radio program for AX.25, NET/ROM and Rose\n\n\
This program is distributed under the\nGNU General Public License"), PACKAGE, VERSION);
  showdialog(info, _("%s - about"));
  g_string_free(info, TRUE);
}


/********************************* main window ******************************/
void on_window1_show (GtkWidget *widget, gpointer user_data) {
  GtkWidget *maintext, *scrollwindow, *menu;
  GtkStyle *maintextstyle;

  /* check for available ports here, so we have a nice pop-window when things go wrong */
  portsinit();

  maintext = lookup_widget(window1, "text1");
  scrollwindow = lookup_widget(window1, "scrolledwindow1");
  menu = lookup_widget(window1, "menubar1");

  if (!preferences.menu) widget_visible("menu", 0);
  if (!preferences.scrollbar) 
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwindow), GTK_POLICY_NEVER, GTK_POLICY_NEVER);

  maintextstyle = gtk_style_new();
  maintextstyle->base[GTK_STATE_NORMAL] = preferences.backgroundcolor;
  maintextstyle->base[GTK_STATE_INSENSITIVE] = preferences.backgroundcolor;
  gtk_widget_set_style(maintext, maintextstyle);

  writemessage(_("\t%s version "), PACKAGE, STATUSMESSAGE);
  writemessage("%s\n", VERSION, STATUSMESSAGE);
  writemessage("Copyright (C) 2001, Joop Stakenborg <pa4tu@amsat.org>\n", NULL, STATUSMESSAGE);

  /* set state of some menus */
  connected = FALSE;
  connecting = FALSE;
  recording = FALSE;
}

void on_entry1_activate (GtkEditable *editable, gpointer user_data) {
  GString *entry;
  GtkWidget *mainentry;

  mainentry = lookup_widget(window1, "entry1");
  entry = g_string_new(gtk_editable_get_chars(GTK_EDITABLE(mainentry), 0, -1));
  gtk_entry_select_region(GTK_ENTRY(mainentry), 0, GTK_ENTRY(mainentry)->text_length);
  gtk_entry_set_text(GTK_ENTRY(mainentry), "");

  if (entry->len > 0) tx_save(entry->str);
  if (connected) tx(entry);
  
  g_string_free(entry, TRUE);
}

gboolean on_window1_key_press_event (GtkWidget *widget, GdkEventKey *event, gpointer user_data) {
  GtkWidget *maintext;
  GtkAdjustment *adj;

  maintext = lookup_widget(window1, "text1");
  adj = GTK_ADJUSTMENT(GTK_TEXT(maintext)->vadj);
  switch (event->keyval)
  {
    case GDK_Up:
      gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
      tx_previous();
    break;
    case GDK_Down:
      gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
      tx_next();
    break;
    case GDK_Page_Up:
      gtk_adjustment_set_value(adj,adj->value-adj->page_size);
    break;
    case GDK_Page_Down:
      if(adj->value < adj->upper - adj->page_size)
        gtk_adjustment_set_value(adj, adj->value+adj->page_size);
    break;
    default:
    break;
  }
  return FALSE;
}

gboolean
on_window1_delete_event                (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
  if (connecting) kill(connectpid, SIGHUP);
  if (recording) stoprecord();

  gdk_window_get_size(window1->window, &preferences.width, &preferences.height);
  gdk_window_get_root_origin(window1->window, &preferences.x, &preferences.y);

  savepreferences();
  savehistory();
  gtk_exit(0);

  return FALSE;
}


/*************************** preferences dialog *****************************/

/* OK button is clicked */
void on_button2_clicked (GtkButton *button, gpointer user_data) {
  GtkWidget *fontentry, *text, *scrollcheck, *menucheck, *scrollwindow,
    *timeoutentry, *dlentry, *sizecheck, *poscheck;
  GtkStyle *textstyle;
  gchar *timeout, *dldir, *timecopy, *fill;
  gboolean noerror = TRUE;
  GString *info = g_string_new("");
  gint len;

  fontentry = lookup_widget(preferencesdialog, "entry2");
  menucheck = lookup_widget(preferencesdialog, "checkbutton2");
  scrollcheck = lookup_widget(preferencesdialog, "checkbutton1");
  sizecheck = lookup_widget(preferencesdialog, "checkbutton4");
  poscheck = lookup_widget(preferencesdialog, "checkbutton5");
  scrollwindow = lookup_widget(window1, "scrolledwindow1");
  timeoutentry = lookup_widget(preferencesdialog, "entry4");
  dlentry = lookup_widget(preferencesdialog, "entry5");

  /* get the font name */
  preferences.font = g_string_new(gtk_editable_get_chars(GTK_EDITABLE(fontentry), 0, -1));

  /* check if the entered path for downloading is valid */
  dldir = g_strdup((gtk_editable_get_chars(GTK_EDITABLE(dlentry), 0, -1)));
  g_strstrip(dldir);
  if (strlen(dldir) == 0) {
    dldir = g_strdup_printf("%s/.%s", g_get_home_dir(), PACKAGE);
    preferences.downloaddir = g_string_new(dldir);
  } else if (g_strncasecmp(dldir, "/", 1) != 0) {
    noerror = FALSE;
    g_string_sprintf(info, _("%s is not a valid pathname,\n\
you must enter the full path, starting with a '/'"), dldir);
    showdialog(info, _("%s - error"));
  } else preferences.downloaddir = g_string_new(dldir);

  /* check if timeout is a number */
  timeout = g_strdup((gtk_editable_get_chars(GTK_EDITABLE(timeoutentry), 0, -1)));
  g_strstrip(timeout);
  if ((len = strlen(timeout)) == 0) preferences.timeout = 90;
  else {
    timecopy = g_strdup(timeout);
    g_strdelimit(timecopy, "0123456789", '#');
    fill = g_strnfill(len, '#');
    if (g_strncasecmp(timecopy, fill, len) != 0) {
      noerror = FALSE;
      g_string_sprintf(info, _("%s is not a valid number,\n\
digits must be between 0 and 9"), timeout);
      showdialog(info, _("%s - error"));
    } else preferences.timeout = atoi(timeout);
  }

  /* set state of some widgets */
  preferences.menu = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(menucheck));
  widget_visible("menubar1", preferences.menu);
  preferences.scrollbar = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(scrollcheck));
  if (preferences.scrollbar)
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwindow), GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
  else
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwindow), GTK_POLICY_NEVER, GTK_POLICY_NEVER);

  /* window position and size  */
  preferences.size = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(sizecheck));
  preferences.origin = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(poscheck));

  /* set the background */
  text = lookup_widget(window1, "text1");
  textstyle = gtk_style_new();
  textstyle->base[GTK_STATE_NORMAL] = preferences.backgroundcolor;
  textstyle->base[GTK_STATE_INSENSITIVE] = preferences.backgroundcolor;
  gtk_widget_set_style(text, textstyle);

  if (noerror) {
    g_string_free(info, TRUE);
    gtk_widget_destroy(preferencesdialog);
  }
}

/* Cancel button clicked */
void on_button3_clicked (GtkButton *button, gpointer user_data) {
  gtk_widget_destroy(preferencesdialog);
}

/* font button clicked */
void on_button4_clicked (GtkButton *button, gpointer user_data) {
  GtkWidget *fontdialog;
  
  fontdialog = create_fontselectiondialog1();
  gtk_font_selection_dialog_set_preview_text(GTK_FONT_SELECTION_DIALOG(fontdialog), _("How about this font?"));
  gtk_font_selection_dialog_set_font_name(GTK_FONT_SELECTION_DIALOG(fontdialog), preferences.font->str);
  set_title(fontdialog, _("%s - select font"));
  gtk_widget_set_sensitive(preferencesdialog, 0);
  gtk_widget_show(fontdialog);
}

/* one of the color buttons is clicked */
void on_color_clicked (GtkButton *button, gpointer user_data) {
  GtkWidget *colordialog;
  gdouble rgb[3];
  GdkColor buttoncolor;
  GtkStyle *buttonstyle;

  /* save the pressed button */
  buttonnumber = atoi(user_data);
  
  colordialog = create_colorselectiondialog1();
  gtk_widget_destroy(GTK_COLOR_SELECTION_DIALOG(colordialog)->help_button);
  set_title(colordialog, _("%s - select color"));
  gtk_widget_set_sensitive(preferencesdialog, 0);

  /* get the color of the button and apply to the color selection dialog */
  buttonstyle = gtk_widget_get_style(GTK_WIDGET(button));
  buttoncolor = buttonstyle->bg[GTK_STATE_NORMAL];
  rgb[0] = buttoncolor.red / 65535.0;
  rgb[1] = buttoncolor.green / 65535.0;
  rgb[2] = buttoncolor.blue / 65535.0;
  gtk_color_selection_set_color(GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(colordialog)->colorsel), rgb);
  
  gtk_widget_show(colordialog);
}


/* another entry in the combobox is selected */
void on_combo_entry1_changed (GtkEditable *editable, gpointer user_data) {
  GtkWidget *combo;
  gchar *portentry;
  
  combo = lookup_widget(preferencesdialog, "combo1");
  portentry = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(combo)->entry));
  /* read settings from proc filesystem and fill in the labels */
  axupdatelabels(portentry);
}

/* dx cluster support */
void on_dxcheckbutton_clicked (GtkButton *button, gpointer user_data) {
  GtkWidget *frame;

  frame = lookup_widget(preferencesdialog, "frame8");
  preferences.dxcluster = gtk_toggle_button_get_active
    (GTK_TOGGLE_BUTTON(button));
  gtk_widget_set_sensitive(frame, preferences.dxcluster);
}

void on_dialog2_destroy (GtkObject *object, gpointer user_data) {
  GtkWidget *mainentry;
  
  mainentry = lookup_widget(window1, "entry1");
  gtk_widget_grab_focus(mainentry);
  gtk_widget_set_sensitive(window1, 1);
}

/******************************* about dialog *******************************/
void on_button1_clicked (GtkButton *button, gpointer user_data) {

  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}


/******************************** font dialog *******************************/
void on_ok_button1_clicked (GtkButton *button, gpointer user_data) {
  GtkWidget *fontdialog, *fontentry;
  GString *font;
  
  fontdialog = gtk_widget_get_ancestor(GTK_WIDGET(button), gtk_font_selection_dialog_get_type());
  fontentry = lookup_widget(preferencesdialog, "entry2");
  font = g_string_new(gtk_font_selection_dialog_get_font_name(GTK_FONT_SELECTION_DIALOG(fontdialog)));
  gtk_entry_set_text(GTK_ENTRY(fontentry), font->str);
  g_string_free(font, TRUE);
  gtk_widget_destroy(fontdialog);
}


void on_cancel_button1_clicked (GtkButton *button, gpointer user_data) {
  GtkWidget *fontdialog;
  
  fontdialog = gtk_widget_get_ancestor(GTK_WIDGET(button), gtk_font_selection_dialog_get_type());
  gtk_widget_destroy(fontdialog);
}


void on_fontselectiondialog1_destroy (GtkObject *object, gpointer user_data) {
  gtk_widget_set_sensitive(preferencesdialog, 1);
}


/******************************* color dialog *******************************/
void on_ok_button2_clicked (GtkButton *button, gpointer user_data) {
  GtkWidget *colordialog;
  GdkColor newcolor;
  gdouble rgb[3];
  
  colordialog = gtk_widget_get_ancestor(GTK_WIDGET(button), gtk_color_selection_dialog_get_type());
  gtk_color_selection_get_color(GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(colordialog)->colorsel), rgb);
  newcolor.red = rgb[0] * 65535.0;
  newcolor.green = rgb[1] * 65535.0;
  newcolor.blue = rgb[2] * 65535.0;
  gtk_widget_set_sensitive(preferencesdialog, 1);
  updatebuttoncolor(newcolor);
  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}


void on_cancel_button2_clicked (GtkButton *button, gpointer user_data) {
  GtkWidget *colordialog;

  colordialog = gtk_widget_get_ancestor(GTK_WIDGET(button), gtk_color_selection_dialog_get_type());
  gtk_widget_destroy(colordialog);
}


void on_colorselectiondialog1_destroy (GtkObject *object, gpointer user_data) {
  gtk_widget_set_sensitive(preferencesdialog, 1);
}


/** dialog with a combobox and 2 buttons used for connection and recording **/

void on_dialog3_destroy (GtkObject *object, gpointer user_data) {
  GtkWidget *txentry;

  txentry = lookup_widget(window1, "entry1");
  gtk_widget_set_sensitive(window1, 1);
  gtk_widget_grab_focus(txentry);
}

/* OK */
void on_button10_clicked (GtkButton *button, gpointer user_data) {
  GtkWidget *combodialog, *combo;
  gchar *entry, *ptr;
  GList *node;

  combodialog = gtk_widget_get_toplevel(GTK_WIDGET(button));
  combo =  lookup_widget(combodialog, "combo2");

  /* if the dialog turns recording on, we have to store the entered filename */
  if (combotype == RECORDON) {
    entry = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(combo)->entry));

    /* check if it is already in the list, if yes remove it */
    /* this will assure that an already present item will appear as the first item the next time */
    node = g_list_find_custom(recordhistory, entry, (GCompareFunc)g_strcasecmp);
    if (node) {
      ptr = node->data;
      recordhistory = g_list_remove(recordhistory, ptr);
      free(ptr);
    }
    recordhistory = g_list_prepend(recordhistory, g_strdup(entry));

    /* if the list gets larger than a certain length, throw away the last element */
    if (g_list_length(recordhistory) > HISTORY)
    recordhistory = g_list_remove(recordhistory, g_list_last(recordhistory)->data);

    startrecord(entry);
  }

  else if (combotype == RECORDOFF) {
    stoprecord();
  }

  else if (combotype == CONNECT) {
    entry = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(combo)->entry));
    node = g_list_find_custom(connecthistory, entry, (GCompareFunc)g_strcasecmp);
    if (node) {
      ptr = node->data;
      connecthistory = g_list_remove(connecthistory, ptr);
      free(ptr);
    }
    connecthistory = g_list_prepend(connecthistory, g_strdup(entry));
    if (g_list_length(connecthistory) > HISTORY)
    connecthistory = g_list_remove(connecthistory, g_list_last(connecthistory)->data);
    connectto(entry);
  }

  else if (combotype == DISCONNECT) disconn();
  else if (combotype == KILL) killconn();

  gtk_widget_destroy(combodialog);
}

/* cancel */
void on_button11_clicked (GtkButton *button, gpointer user_data) {
  GtkWidget *combodialog;

  combodialog = gtk_widget_get_toplevel(GTK_WIDGET(button));
  gtk_widget_destroy(combodialog);
}
