/*								-*-WEB-*-
**  File: parse.y
**  Date: 28 Apr 1992
**  Description: YACC grammar for Widget Builder
**  Author: Bert Bos <bert@let.rug.nl>
*/

%{
static char rcsid[] = "$Header: parse.y,v 1.17 92/12/07 19:32:32 bert Exp $";
#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include "types.c"
#include "debug.e"
#include "util.e"

#define NIL ((STRING) NULL)
#define new(ptr) ptr = calloc(1, sizeof(*(ptr))) /* set to zeros */
#define yerr(m) if (! YYRECOVERING()) yyerror(m)

#define alloca malloc

extern int lineno;
extern char *yytext;
extern int yyleng;
char *filename;

#if defined(__STDC__) || defined(__cplusplus)
# define P_(s) s
#else
# define P_(s) ()
#endif

extern int yylex P_((void));
extern void err P_((Boolean fatal, char *format,...));
static void yyerror P_((const char *s));
static void add_section P_((Section *psect, Section s));
static Section new_section P_((STRING, Decl, Section));
static char * no_quotes P_((char *s));

#undef P_
%}

%union {
  STRING string;
  Option option;
  Class class;
  Section section;
  Decl decl;
  int i;
}

%token <i> CLASS
%token <i> CLASSVARS
%token <i> PUBLIC
%token <i> PRIVATE
%token <i> METHODS
%token <i> ACTIONS
%token <i> TRANSLATIONS
%token <i> IMPORTS
%token <i> EXPORTS
%token <i> UTILITIES
%token <i> VAR
%token <i> DEF
%token <i> TYPE
%token <i> PROC
%token <i> TRANS
%token <i> INCL
%token <string> TEXT
%token <string> BRACKETED
%token ILL_CHAR
%token UNKNOWN
%token ILL_BODY
%token NOCODE
%token NODOC
%token GUARD
%token GUARDP
%token FILE_OPT
%token <string> IDENT
%token LPAR
%token RPAR
%token STAR
%token TYPE
%token INCL
%token <string> NUMBER
%token LBRACK
%token RBRACK
%token SEMI
%token EQUALS
%token PLUS
%token COMMA
%token COLON
%token TILDE
%token EXCLAM
%token <string> CTOK
%token <string> CSTRING
%token <string> BODY
%token <string> COMMENT

%type <class> class
%type <class> sections
%type <string> superclass
%type <option> options
%type <option> option
%type <section> publicvars
%type <section> pubvarsections
%type <decl> pubvardef
%type <string> symbol
%type <section> actions
%type <section> actionsections
%type <decl> actiondef
%type <section> translations
%type <section> transsections
%type <decl> transline
%type <string> stars
%type <string> modifiers
%type <string> mod_syms
%type <string> event
%type <string> more_events
%type <string> count
%type <string> detail
%type <string> actioncalls
%type <string> actioncall
%type <string> arguments
%type <string> more_arguments
%type <section> classvars
%type <section> methods
%type <section> imports
%type <section> exports
%type <section> privatevars
%type <section> utilities
%type <string> C_stuff
%type <string> macro_stuff
%type <decl> methoddef
%type <section> methodsections
%type <decl> type_and_name
%type <section> exportsections
%type <decl> exportdef
%type <decl> params
%type <string> C_thing
%type <decl> paramlist
%type <decl> more_params
%type <section> importsections
%type <decl> importdecl
%type <section> utilsections
%type <decl> utildef
%type <decl> classvardecl
%type <section> classvarsections
%type <decl> privatedecl
%type <section> privatesections

%%

program
 : class program		{add_class($1);}
 | myerror program		{yerr("error in class");}
 | /* empty */
 ;
class
 : CLASS IDENT superclass
   options TEXT sections	{$$ = $6; $$->name = $2; $$->superclass = $3;
				$$->options = $4; $$->lineno = $1; $$->text =
				$5; $$->filename = hash(filename);}
 | CLASS IDENT superclass
   options sections		{$$ = $5; $$->name = $2; $$->superclass = $3;
				$$->options = $4; $$->lineno = $1;
				$$->filename = hash(filename);}
 ;
superclass
 : LPAR IDENT RPAR		{$$ = $2;}
 | LPAR myerror RPAR		{yerr("name of superclass expected");
				$$ = NULL;}
 | /* empty */			{$$ = NULL;}
 ;
options
 : options option		{$$ = $2; $$->next = $1;}
 | /* empty */			{$$ = NULL;}
 ;
option
 : NOCODE			{new($$); $$->opt = hash("nocode");}
 | NODOC			{new($$); $$->opt = hash("nodoc");
				$$->val = NULL;}
 | GUARD EQUALS IDENT		{new($$); $$->opt = hash("guard");
				$$->val = $3;}
 | GUARDP EQUALS IDENT		{new($$); $$->opt = hash("guardp");
				$$->val = $3;}
 | FILE_OPT EQUALS IDENT	{new($$); $$->opt = hash("file");
				$$->val = $3;}
 /* and more... */
 ;
sections
 : sections classvars		{$$ = $1; add_section(&$$->classvars, $2);}
 | sections publicvars		{$$ = $1; add_section(&$$->publicvars, $2);}
 | sections privatevars		{$$ = $1; add_section(&$$->privatevars, $2);}
 | sections methods		{$$ = $1; add_section(&$$->methods, $2);}
 | sections actions		{$$ = $1; add_section(&$$->actions, $2);}
 | sections translations	{$$ = $1; add_section(&$$->translations, $2);}
 | sections imports		{$$ = $1; add_section(&$$->imports, $2);}
 | sections exports		{$$ = $1; add_section(&$$->exports, $2);}
 | sections utilities		{$$ = $1; add_section(&$$->utilities, $2);}   
 | /* empty */			{new($$);}
 ;

publicvars
 : PUBLIC pubvarsections	{$$ = $2;}
 ;
pubvarsections
 : TEXT pubvardef
   pubvarsections		{$$ = new_section($1, $2, $3);}
 | pubvardef pubvarsections	{$$ = new_section(NIL, $1, $2);}
 | TEXT pubvarsections		{$$ = new_section($1, (Decl) NULL, $2);}
 | /* empty */			{$$ = NULL;}
 ;
pubvardef
 : VAR symbol type_and_name
   symbol EQUALS symbol
   C_stuff opt_semi		{$$ = $3; $$->typesym = $2; $$->valuesym = $6;
				$$->value = $7; $$->namesym = $4; $$->lineno =
				$1; $$->tp = Var;}
 | DEF IDENT params EQUALS
   macro_stuff	    		{new($$); $$->name = $2; $$->tp = Def;
				$$->body = $5; $$->params = $3; $$->lineno =
				$1;}
 | DEF myerror macro_stuff	{$$ = NULL;yerr("incorrect macro definition");}
 ;
opt_semi
 : SEMI
 | /* empty */
 ;
symbol
 : BRACKETED			{$$ = $1;}
 | /* empty */			{$$ = NULL;}
 ;


actions
 : ACTIONS actionsections	{$$ = $2;}
 ;
actionsections
 : TEXT actiondef
   actionsections		{$$ = new_section($1, $2, $3);}
 | actiondef actionsections	{$$ = new_section(NIL, $1, $2);}
 | TEXT actionsections		{$$ = new_section($1, (Decl) NULL, $2);}
 | /* empty */			{$$ = NULL;}
 ;
actiondef
 : PROC IDENT BODY		{new($$); $$->name = $2; $$->body = $3;
				$$->lineno = $1; $$->tp = Proc;}
 | PROC myerror BODY		{yerr("error in action name"); $$ = NULL;}
 | DEF IDENT params EQUALS
   macro_stuff	    		{new($$); $$->name = $2; $$->tp = Def;
				$$->body = $5; $$->params = $3; $$->lineno =
				$1;}
 | DEF myerror macro_stuff	{$$ = NULL;yerr("incorrect macro definition");}
 ;


translations
 : TRANSLATIONS transsections	{$$ = $2;}
 ;
transsections
 : TEXT transline transsections	{$$ = new_section($1, $2, $3);}
 | transline transsections	{$$ = new_section(NIL, $1, $2);}
 | TEXT transsections		{$$ = new_section($1, (Decl) NULL, $2);}
 | /* empty */			{$$ = NULL;}
 ;
transline
 : TRANS mod_syms modifiers
   event more_events count
   COLON actioncalls opt_semi	{new($$); $$->tp = Trans; $$->type =
				catstr(5, get($2), get($3), get($4), get($5),
				get($6)); $$->value = $8; $$->lineno = $1;
				delete($2); delete($3); delete($4);
				delete($5); delete($6);}
 | TRANS myerror COLON
   actioncalls opt_semi		{yerr("error before ':' in translation"); $$ =
				NULL;}
 ;
mod_syms
 : COLON			{$$ = hash(":");}
 | EXCLAM			{$$ = hash("!");}
 | TILDE			{$$ = hash("~");}
 | /* empty */			{$$ = NIL;}
 ;
modifiers
 : IDENT modifiers		{$$ = catstr(2, get($1), get($2)); delete($1);
				delete($2);}
 | /* empty */			{$$ = NIL;}
 ;
event
 : BRACKETED detail		{$$ = catstr(4, "<", get($1), ">", get($2));
				delete($1); delete($2);}
 | CSTRING			{$$ = $1;}
 ;
more_events
 : COMMA event more_events	{$$ = catstr(3, ",", get($2), get($3));
				delete($2); delete($3);}
 | /* empty */			{$$ = NIL;}
 ;
count
 : LPAR NUMBER RPAR		{$$ = catstr(3, "(", get($2), ")");
				delete($2);}
 | LPAR NUMBER PLUS RPAR	{$$ = catstr(3, "(", get($2), "+)");
				delete($2);}
 | LPAR myerror RPAR		{yerr("count expected after '('");
				$$ = NIL;}
 | /* empty */			{$$ = NIL;}
 ;
detail
 : IDENT			{$$ = $1;}
 | /* empty */			{$$ = NIL;}
 ;
actioncalls
 : actioncall actioncalls	{$$ = catstr(2, get($1), get($2)); delete($1);
				delete($2);}
 | actioncall			{$$ = $1;}
 ;
actioncall
 : IDENT LPAR arguments RPAR	{$$ = catstr(4, get($1), "(", get($3), ") ");
				delete($1); delete($3);}
 | IDENT LPAR myerror RPAR	{yerr("error in argument");
				$$ = NULL;}
 ;
arguments
 : CSTRING more_arguments	{char *s; s = no_quotes(get($1)); delete($1);
				$$ = catstr(2, s, get($2)); free(s);
				delete($2);}
 | /* empty */			{$$ = NULL;}
 ;
more_arguments
 : COMMA CSTRING more_arguments	{char *s; s = no_quotes(get($2)); delete($2);
				$$ = catstr(3, ",", s, get($3)); free(s);
				delete($3);}
				/* The arguments shouldn't be quoted... */
 | /* empty */			{$$ = NULL;}
 ;


methods
 : METHODS methodsections	{$$ = $2;}
 ;
methodsections
 : TEXT methoddef
   methodsections		{$$ = new_section($1, $2, $3);}
 | methoddef methodsections	{$$ = new_section(NIL, $1, $2);}
 | TEXT methodsections		{$$ = new_section($1, (Decl) NULL, $2);}
 | /* empty */			{$$ = NULL;}
 ;
methoddef
 : PROC type_and_name
   params BODY			{$$ = $2; $$->params = $3; $$->body = $4;
				$$->lineno = $1; $$->tp = Proc;}
 | PROC myerror BODY		{$$ = NULL; yerr("error in method heading");}
 | DEF IDENT params EQUALS
   macro_stuff	    		{new($$); $$->name = $2; $$->tp = Def;
				$$->body = $5; $$->params = $3; $$->lineno =
				$1;}
 | DEF myerror macro_stuff	{$$ = NULL;yerr("incorrect macro definition");}
 ;
type_and_name
 : type_and_name stars IDENT	{STRING h; $$ = $1;
				if ($$->type != NIL) {
				  h = catstr(4, get($$->type), " ",
				  get($$->name), get($2)); delete($$->type);
				  $$->type = h; delete($$->name);
				} else {
				  $$->type = catstr(2, get($$->name), get($2));
				  delete($$->name);
				}
				$$->name = $3;}
 | type_and_name LBRACK RBRACK	{STRING h; $$ = $1; h = catstr(2,
				get($$->name), "[]"); delete($$->name);
				$$->name = h;}
 | type_and_name
   LBRACK IDENT RBRACK		{STRING h; $$ = $1; h = catstr(4,
				get($$->name), "[", get($3), "]");
				delete($$->name); delete($3); $$->name = h;}
 | type_and_name
   LBRACK NUMBER RBRACK		{STRING h; $$ = $1; h = catstr(4,
				get($$->name), "[", get($3), "]");
				delete($$->name); delete($3); $$->name = h;}
 | IDENT			{new($$); $$->name = $1;}
 | myerror IDENT		{yerr("error in type expression"); new($$);}
 ;

stars
 : stars STAR			{$$ = catstr(2, get($1), "*");}
 | /* empty */			{$$ = hash(" ");}
 ;

exports
 : EXPORTS exportsections	{$$ = $2;}
 ;
exportsections
 : TEXT exportdef
   exportsections		{$$ = new_section($1, $2, $3);}
 | exportdef exportsections	{$$ = new_section(NIL, $1, $2);}
 | TEXT exportsections		{$$ = new_section($1, (Decl) NULL, $2);}
 | /* empty */			{$$ = NULL;}
 ;
exportdef
 : INCL BRACKETED		{new($$); $$->lineno = $1; $$->name =
				catstr(3, "<", get($2), ">"); $$->tp = Incl;
				delete($2);}
 | INCL CSTRING			{new($$); $$->lineno = $1; $$->name = $2;
				$$->tp = Incl;}
 | TYPE IDENT EQUALS C_stuff
   opt_semi	     		{new($$); $$->lineno = $1; $$->type = $4;
				$$->name = $2; $$->tp = Type;}
 | TYPE myerror EQUALS C_stuff	{$4 = NULL;
				yerr("should be one identifier before '='");}
 | VAR type_and_name opt_semi	{$$ = $2; $$->lineno = $1; $$->tp = Var;}
 | VAR type_and_name EQUALS
   C_stuff opt_semi		{$$ = $2; $$->lineno = $1; $$->tp = Var;
				$$->value = $4;}
 | PROC type_and_name
   params BODY			{$$ = $2; $$->params = $3; $$->body = $4;
				$$->lineno = $1; $$->tp = Proc;}
 | PROC myerror BODY		{$$ = NULL; yerr("error in function heading");}
 | DEF IDENT params EQUALS
   macro_stuff	    		{new($$); $$->name = $2; $$->tp = Def;
				$$->body = $5; $$->params = $3; $$->lineno =
				$1;}
 | DEF myerror macro_stuff	{$$ = NULL;yerr("incorrect macro definition");}
 ;
macro_stuff
 : LBRACK macro_stuff		{$$ = catstr(2, "[", get($2)); delete($2);}
 | LPAR macro_stuff		{$$ = catstr(2, "(", get($2)); delete($2);}
 | C_stuff			{$$ = $1;}
 | /* empty */			{$$ = NIL;}
 ;
C_stuff
 : C_stuff C_thing		{$$ = catstr(2, get($1), get($2)); delete($1);
				delete($2);}
 | C_thing			{$$ = $1;}
 ;
C_thing
 : C_thing LPAR			{$$ = catstr(2, get($1), "("); delete($1);}
 | IDENT LBRACK			{$$ = catstr(2, get($1), "["); delete($1);}
 | IDENT			{$$ = catstr(2, get($1), " "); delete($1);}
 | STAR				{$$ = hash("*");}
 | TILDE			{$$ = hash("~");}
 | EXCLAM			{$$ = hash("!");}
 | COLON			{$$ = hash(":");}
 | RBRACK			{$$ = hash("]");}
 | BODY				{$$ = $1;}
 | RPAR				{$$ = hash(")");}
 | RPAR	LBRACK			{$$ = hash(")[");}
 | COMMA			{$$ = hash(",");}
 | NUMBER			{$$ = catstr(2, get($1), " "); delete($1);}
 | CSTRING			{$$ = $1;}
 | EQUALS			{$$ = hash("=");}
 | PLUS				{$$ = hash("+");}
 | CTOK				{$$ = $1;}
 | COMMENT			{$$ = $1;}
 ;
params
 : LPAR paramlist RPAR		{$$ = $2;}
 | LPAR myerror RPAR		{yerr("error in parameter list");$$ = NULL;}
 | /* empty */			{$$ = NULL;}
 ;
paramlist
 : type_and_name more_params	{$$ = $1; $$->next = $2;}
 | /* empty */			{$$ = NULL;}
 ;
more_params
 : COMMA type_and_name
   more_params			{$$ = $2; $$->next = $3;}
 | /* empty */			{$$ = NULL;}
 ;


imports
 : IMPORTS importsections	{$$ = $2;}
 ;
importsections
 : TEXT importdecl
   importsections		{$$ = new_section($1, $2, $3);}
 | importdecl importsections	{$$ = new_section(NIL, $1, $2);}
 | TEXT importsections		{$$ = new_section($1, (Decl) NULL, $2);}
 | /* empty */			{$$ = NULL;}
 ;
importdecl
 : INCL BRACKETED		{new($$); $$->lineno = $1; $$->name =
				catstr(3, "<", get($2), ">"); $$->tp = Incl;
				delete($2);}
 | INCL CSTRING			{new($$); $$->lineno = $1; $$->name = $2;
				$$->tp = Incl;}
 | VAR type_and_name opt_semi	{$$ = $2; $$->lineno = $1; $$->tp = Var;}
 | PROC type_and_name
   params opt_semi		{$$ = $2; $$->params = $3; $$->lineno = $1;
				$$->tp = Proc;}
 ;


utilities
 : UTILITIES utilsections	{$$ = $2;}
 ;
utilsections
 : TEXT utildef utilsections	{$$ = new_section($1, $2, $3);}
 | utildef utilsections		{$$ = new_section(NIL, $1, $2);}
 | TEXT utilsections		{$$ = new_section($1, (Decl) NULL, $2);}
 | /* empty */			{$$ = NULL;}
 ;
utildef
 : TYPE IDENT EQUALS C_stuff
   opt_semi	     		{new($$); $$->lineno = $1; $$->type = $4;
				$$->tp = Type; $$->name = $2;}
 | TYPE myerror EQUALS C_stuff	{$4 = NULL;
				yerr("should be one identifier before '='");}
 | VAR type_and_name opt_semi	{$$ = $2; $$->lineno = $1; $$->tp = Var;}
 | VAR type_and_name EQUALS
   C_stuff opt_semi		{$$ = $2; $$->lineno = $1; $$->tp = Var;
				$$->value = $4;}
 | PROC type_and_name
   params BODY			{$$ = $2; $$->params = $3; $$->body = $4;
				$$->lineno = $1; $$->tp = Proc;}
 | PROC myerror BODY		{$$ = NULL; yerr("error in function heading");}
 | DEF IDENT params EQUALS
   macro_stuff	    		{new($$); $$->name = $2; $$->tp = Def;
				$$->body = $5; $$->params = $3; $$->lineno =
				$1;}
 | DEF myerror macro_stuff	{$$ = NULL;yerr("incorrect macro definition");}
 ;


classvars
 : CLASSVARS classvarsections	{$$ = $2;}
 ;
classvarsections
 : TEXT classvardecl
   classvarsections		{$$ = new_section($1, $2, $3);}
 | classvardecl
   classvarsections		{$$ = new_section(NIL, $1, $2);}
 | TEXT classvarsections	{$$ = new_section($1, (Decl) NULL, $2);}
 | /* empty */			{$$ = NULL;}
 ;
classvardecl
 : VAR type_and_name EQUALS
   C_stuff opt_semi		{$$ = $2; $$->value = $4; $$->lineno = $1;
				$$->tp = Var;}
 | VAR myerror EQUALS C_stuff
   opt_semi	    		{$$ = NULL; yerr("error in class variable");}
 | DEF IDENT params EQUALS
   macro_stuff	    		{new($$); $$->name = $2; $$->tp = Def;
				$$->body = $5; $$->params = $3; $$->lineno =
				$1;}
 | DEF myerror macro_stuff	{$$ = NULL;yerr("incorrect macro definition");}
 ;


privatevars
 : PRIVATE privatesections	{$$ = $2;}
 ;
privatesections
 : TEXT privatedecl
   privatesections		{$$ = new_section($1, $2, $3);}
 | privatedecl privatesections	{$$ = new_section(NIL, $1, $2);}
 | TEXT privatesections		{$$ = new_section($1, (Decl) NULL, $2);}
 | /* empty */			{$$ = NULL;}
 ;
privatedecl
 : VAR type_and_name opt_semi	{$$ = $2; $$->lineno = $1; $$->tp = Var;}
 | DEF IDENT params EQUALS
   macro_stuff	    		{new($$); $$->name = $2; $$->tp = Def;
				$$->body = $5; $$->params = $3; $$->lineno =
				$1;}
 | DEF myerror macro_stuff	{$$ = NULL;yerr("incorrect macro definition");}
 ;

myerror
 : error ILL_CHAR		{yerr("illegal character");}
 | error UNKNOWN		{yerr("unknown keyword");}
 | error ILL_BODY		{yerr("missing closing brace");}
 | error			{yerr("syntax error");}
 ;

%%

/*
 * new_section -- allocate a new section and initialize it
 */
static Section new_section(text, decl, next)
  STRING text;
  Decl decl;
  Section next;
{
  Section h;

  new(h);
  h->text = text;
  h->decl = decl;
  h->next = next;
  return h;
}
  
/*
 * yyerror -- write error message to screen
 * for this to work, the parser must have been generated with option -t
 */
#ifdef __STDC__
static void yyerror(const char *s)
#else
static void yyerror(s)
  char *s;
#endif
{
# if defined(YYBISON) && defined(YYDEBUG)
  char temp[20];
  (void) strncpy(temp, yytext, 15);
  if (yyleng > 15) (void) strcpy(temp + 15, "...");
  err(False, "%s at `%s' (token = %s)\n",
    s, temp, yytname[YYTRANSLATE(yychar)]);
# else /* YYBISON && YYDEBUG*/
  err(False, "%s\n", s);
# endif /* YYBISON && YYDEBUG */
  rcsid[0] = 'x'; /* make lint happy */
}

/*
 * add_section -- add a section to a class, append if already set
 */
static void add_section(psect, s)
  Section *psect;
  Section s;
{
  Section p;

  if (*psect == NULL) {
    *psect = s;
  } else {
    for (p = *psect; p->next; p = p->next) ; /* skip */
    p->next = s;
  }
}

/*
 * no_quotes -- make a copy of a string, but remove the quotes
 */
static char * no_quotes(s)
    char *s;
{
    char *t;

    t = strdup(s + 1);
    t[strlen(t)-1] = '\0';
    return t;
}
