------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             E X P _ U T I L                              --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.22 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

--  Package containing utility procedures used throughout the expander

with Rtsfind;
with Types;   use Types;

package Exp_Util is

   Expander_Active : Boolean := False;
   --  A flag that indicates if expansion is active (True) or deactivated
   --  (False). When expansion is deactivated all calls to expander routines
   --  have no effect. Note that the initial setting of False is merely to
   --  prevent saving of an undefined value for an initial call to the
   --  Expander_Mode_Save_And_Set procedure.

   function Build_Runtime_Call
     (Loc  : Source_Ptr;
      RE   : Rtsfind.RE_Id)
      return Node_Id;
   --  Build an N_Procedure_Call_Statement calling the given runtime entity.
   --  The call has no parameters. The first argument provides the location
   --  information for the tree and for error messages. The call node is not
   --  analyzed on return, the caller is responsible for analyzing it.

   procedure Expander_Mode_Save_And_Set (Status : Boolean);
   --  Saves the current setting of the Expander_Active flag on an internal
   --  stack and then sets the flag to the given value.

   procedure Expander_Mode_Restore;
   --  Restores the setting of the Expander_Active flag using the top entry
   --  pushed onto the stack by Expander_Mode_Save_And_Reset, popping the
   --  stack, except that if any errors have been detected, then the state
   --  of the flag is left set to False.

   function Expand_Class_Wide_Subtype
     (Loc  : Source_Ptr;
      Typ  : Entity_Id;
      Exp  : Node_Id)
      return List_Id;
   --  When a Class_Wide_Subtype value is encountered, that is to say in such
   --  cases as:
   --
   --    X: T'Class := Exp
   --    new T'Class'(Exp)
   --
   --  This function generates the list of declarations defining a record
   --  of the following form :
   --
   --    type anon is record
   --       _parent : Root_Type_Of (Typ) constrained with Exp discriminants;
   --       Extension : String (1 .. expr to match size of Exp);
   --    end record;
   --
   --  This record is compatible with any value of T'Class thanks to the
   --  first field and has the same size as Exp thanks to the second field.
   --  This record is attached to the subclass by its Equivalent_Type field.
   --  The declarations are not analyzed, the caller is responsible for
   --  analyzing them after they have been inserted into the tree.

   procedure Expand_Subtype_From_Expr (N : Node_Id; T : Entity_Id);
   --  Build subtype from initial value in object declarations
   --  where the object definition is an unconstrained type.

   procedure Insert_List_Before_And_Analyze (N : Node_Id; L : List_Id);
   --  Inserts list L before node N using Atree.Insert_List_Before, and then,
   --  after this insertion is complete, analyzes all the nodes in the list.
   --  The list must be non-empty for this call.

   function Is_Unconstrained (N : Node_Id) return Boolean;
   --  Return true if N is an expression of an unconstrained type.

   function Make_Tagged_Copy (
     Loc  : Source_Ptr;
     Lhs  : Node_Id;
     Rhs  : Node_Id;
     Typ  : Entity_Id)
     return Node_Id;
   --  expand the code for copying the value of Rhs to Lhs. The value of a
   --  tagged type excludes its tag. If the type is controlled it also excludes
   --  the finalization pointers

   procedure Protect_Statements (N : Node_Id; E : Entity_Id);
   --  This function protects the handled statement sequence of node N by
   --  adding a cleanup that is a call to the procedure referenced by the
   --  entity E. If necessary (if the handled statement sequence already has
   --  an exception handler, or a cleanup), an extra block is wrapped around.

   procedure Traceback_Store (N : Node_Id; Anal : Boolean := True);
   --  Constructs call node to the Store_Tb traceback store function and
   --  inserts the call before the node N. This construction and insertion
   --  happens only if Debug_Flag_B is set, otherwise the call has no effect.
   --  Also the insertion is suppressed if it duplicates an adjacent call,
   --  or if the unit containing the node N is not the main unit. Normally
   --  the call is analyzed before it is inserted. If Anal is set to False
   --  the call to Analyze is suppressed in which case the caller must
   --  ensure that a subsequent call to Analyze is made for the node.

   procedure Wrap_Cleanup_Procedure (N : Node_Id);
   --  Given an N_Subprogram_Body node, this procedure adds an Abort_Defer
   --  call at the start of the statement sequence, and an Abort_Undefer call
   --  at the end of the statement sequence. All cleanup routines (i.e. those
   --  that are called from "at end" handlers) must defer abort on entry and
   --  undefer abort on exit. Note that it is assumed that the code for the
   --  procedure does not contain any return statements which would allow the
   --  flow of control to escape doing the undefer call.

end Exp_Util;
