------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                               T B U I L D                                --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.60 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;   use Atree;
with Einfo;   use Einfo;
with Errout;  use Errout;
with Namet;   use Namet;
with Nlists;  use Nlists;
with Nmake;   use Nmake;
with Sinfo;   use Sinfo;
with Stand;   use Stand;

package body Tbuild is

   Serial_Values : array (Character range 'A' .. 'Z') of Nat;
   --  Serial numbers used by New_Internal_Name to make sure created names
   --  are unique, the x'th entry is used for names starting with letter x.
   --  These counters are initialized to zero by Initialize, and then
   --  incremented as internal names are created.

   -----------------------
   -- Local Subprograms --
   -----------------------

   procedure Add_Nat_To_Name_Buffer (V : Nat);
   --  Add decimal representation of given value to the end of the string
   --  currently stored in Name_Buffer, incrementing Name_Len as required.

   ----------------------------
   -- Add_Nat_To_Name_Buffer --
   ----------------------------

   procedure Add_Nat_To_Name_Buffer (V : Nat) is
   begin
      if V >= 10 then
         Add_Nat_To_Name_Buffer (V / 10);
      end if;

      Name_Len := Name_Len + 1;
      Name_Buffer (Name_Len) := Character'Val (Character'Pos ('0') + V rem 10);
   end Add_Nat_To_Name_Buffer;

   -----------------------------
   -- Create_Raise_Expression --
   -----------------------------

   procedure Create_Raise_Expression (N : Node_Id; Excep_Id : Entity_Id) is
      Original_Node  : Node_Id;
      Raise_Node     : Node_Id;
      Ident_Node     : Node_Id;
      Statement_List : List_Id;

   begin
      Original_Node := New_Copy (N);
      Change_Node (N, N_Expression_Actions);
      Set_Etype (N, Etype (Original_Node));
      Set_Expression (N, Original_Node);
      Ident_Node := New_Node (N_Identifier, Sloc (N));
      Set_Chars (Ident_Node, Chars (Excep_Id));
      Set_Entity (Ident_Node, Excep_Id);
      Raise_Node := New_Node (N_Raise_Statement, Sloc (N));
      Set_Name (Raise_Node, Ident_Node);
      Statement_List := New_List (Raise_Node);
      Set_Actions (N, Statement_List);
   end Create_Raise_Expression;

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize is
   begin
      Serial_Values := (others => 0);
   end Initialize;

   -----------------------
   -- Make_DT_Component --
   -----------------------

   function Make_DT_Component
     (Loc  : Source_Ptr;
      Typ  : Entity_Id;
      I    : Positive)
      return Node_Id
   is
      X : Node_Id;
      Full_Type : Entity_Id := Typ;

   begin
      if Ekind (Typ) in Private_Kind then
         Full_Type := Full_Declaration (Typ);
      end if;

      X := First_Component (
             Designated_Type (Etype (Access_Disp_Table (Full_Type))));

      for J in 2 .. I loop
         X := Next_Component (X);
      end loop;

      return New_Reference_To (X, Loc);
   end Make_DT_Component;

   --------------------
   -- Make_DT_Access --
   --------------------

   function Make_DT_Access
     (Loc  : Source_Ptr;
      Rec  : Node_Id;
      Typ  : Entity_Id)
      return Node_Id
   is
      Full_Type : Entity_Id := Typ;

   begin
      if Ekind (Typ) in Private_Kind then
         Full_Type := Full_Declaration (Typ);
      end if;

      return
        Make_Unchecked_Type_Conversion (Loc,
          Subtype_Mark =>
            New_Occurrence_Of (Etype (Access_Disp_Table (Full_Type)), Loc),
          Expression =>
            Make_Selected_Component (Loc,
              Prefix => New_Copy (Rec),
              Selector_Name =>
                New_Reference_To (Tag_Component (Full_Type), Loc)));
   end Make_DT_Access;

   --------------------------
   -- New_Constraint_Error --
   --------------------------

   function New_Constraint_Error (Loc : Source_Ptr) return Node_Id is
      Ident_Node : Node_Id;
      Raise_Node : Node_Id;

   begin
      Ident_Node := New_Node (N_Identifier, Loc);
      Set_Chars (Ident_Node, Chars (Standard_Entity (S_Constraint_Error)));
      Set_Entity (Ident_Node, Standard_Entity (S_Constraint_Error));
      Raise_Node := New_Node (N_Raise_Statement, Loc);
      Set_Name (Raise_Node, Ident_Node);
      return Raise_Node;
   end New_Constraint_Error;

   -----------------------
   -- New_External_Name --
   -----------------------

   function New_External_Name
     (Related_Id   : Name_Id;
      Suffix       : Character;
      Suffix_Index : Nat := 0;
      Prefix       : Character := ' ')
      return         Name_Id
   is
   begin
      pragma Assert (Is_OK_Internal_Letter (Suffix));
      Get_Name_String (Related_Id);

      if Prefix /= ' ' then
         pragma Assert (Is_OK_Internal_Letter (Prefix));

         for J in reverse 1 .. Name_Len loop
            Name_Buffer (J + 1) := Name_Buffer (J);
         end loop;

         Name_Len := Name_Len + 1;
         Name_Buffer (1) := Prefix;
      end if;

      Name_Len := Name_Len + 1;
      Name_Buffer (Name_Len) := Suffix;

      if Suffix_Index /= 0 then
         Add_Nat_To_Name_Buffer (Suffix_Index);
      end if;

      return Name_Find;
   end New_External_Name;

   function New_External_Name
     (Suffix       : Character;
      Suffix_Index : Nat)
      return         Name_Id
   is
   begin
      Name_Buffer (1) := Suffix;
      Name_Len := 1;
      Add_Nat_To_Name_Buffer (Suffix_Index);
      return Name_Find;
   end New_External_Name;

   -----------------------
   -- New_Internal_Name --
   -----------------------

   function New_Internal_Name (Id_Char : Character) return Name_Id is
   begin
      pragma Assert (Is_OK_Internal_Letter (Id_Char));
      Name_Buffer (1) := Id_Char;
      Name_Len := 1;
      Serial_Values (Id_Char) := Serial_Values (Id_Char) + 1;
      Add_Nat_To_Name_Buffer (Serial_Values (Id_Char));
      return Name_Enter;
   end New_Internal_Name;

   -----------------------
   -- New_Occurrence_Of --
   -----------------------

   function New_Occurrence_Of
     (Def_Id : Entity_Id;
      Loc    : Source_Ptr)
      return   Node_Id
   is
      Occurrence : Node_Id;

   begin
      Occurrence := New_Node (N_Identifier, Loc);
      Set_Chars (Occurrence, Chars (Def_Id));
      Set_Entity (Occurrence, Def_Id);
      Set_Etype (Occurrence, Etype (Def_Id));
      return Occurrence;
   end New_Occurrence_Of;

   ----------------------
   -- New_Reference_To --
   ----------------------

   function New_Reference_To
     (Def_Id : Entity_Id;
      Loc    : Source_Ptr)
      return   Node_Id
   is
      Occurrence : Node_Id;

   begin
      Occurrence := New_Node (N_Identifier, Loc);
      Set_Chars (Occurrence, Chars (Def_Id));
      Set_Entity (Occurrence, Def_Id);
      return Occurrence;
   end New_Reference_To;

   -------------------
   -- Raise_Warning --
   -------------------

   procedure Raise_Warning
     (N        : Node_Id;
      Excep_Id : Entity_Id;
      Reason   : String)
   is
   begin
      Error_Msg_N (Reason, N);
      Error_Msg_NE ("& will be raised at runtime?!", N, Excep_Id);
   end Raise_Warning;

end Tbuild;
