/* font-open.c: find font filenames.  This bears no relation (but the
   interface) to the original font_open.c, so I renamed it.  */

#include "config.h"

#include <kpathsea/filefmt.h>
#include <kpathsea/pathsearch.h>
#include <kpathsea/tex-glyph.h>
#include <kpathsea/tex-make.h>

static string vf_path;


/* Initialize fallback paths.  */

void
init_font_open ()
{
  vf_path = KPSE_VF_PATH ();
  
  kpse_init_fallback_resolutions ("XDVISIZES");
  
  /* Might have a program-specific path.  */
  kpse_override_path = getenv ("XDVIFONTS");
  
  {
    /* Might have a program-specific path for MakeTeXPK, too.  */
    string makepk = getenv ("XDVIMAKEPK");

    if (makepk)
      KPSE_MAKE_SPEC_PROGRAM (kpse_make_specs[kpse_any_glyph_format]) = makepk;
  }
  
  /* The base DPI won't change during the run.  */
  xputenv ("MAKETEX_BASE_DPI", itoa ((unsigned) (pixels_per_inch + .5)));

  KPSE_MAKE_SPEC_ENABLED (kpse_make_specs[kpse_any_glyph_format])
    = use_maketexpk;
}

/* Return the full filename found (in which case set *FONT_RET to NULL),
   or NULL.  */

char *
try_vf (font, font_ret)
    char *font;
    char **font_ret;
{
  string vf_name = concat (font, ".vf");
  string ret = kpse_path_search (vf_path, vf_name);
  
  if (ret)
    *font_ret = NULL;
  free (vf_name);
  
  return ret;
}

/* We try for a VF first because that's what dvips does.  Also, it's
   easier to avoid running MakeTeXPK if we have a VF this way.  I doubt it
   makes any practical difference.  */

FILE *
font_open (font, font_ret, dpi, mag_ret, filename_ret)
    _Xconst char *font;
    char **font_ret;
    float dpi;
    int *mag_ret;
    char **filename_ret;
{
  FILE *ret;
  char *name = try_vf (font, font_ret);
  
  if (name)
    /* VF fonts don't have a resolution, but loadfont will complain if
       we don't return what it asked for.  */
    *mag_ret = dpi;
  else
    {
      kpse_font_file_type file_ret;
      name = kpse_find_glyph_format (font, (unsigned) (dpi + .5),
                                     kpse_any_glyph_format, &file_ret);
      if (name)
        {
          /* If we got it normally, from an alias, or from MakeTeXPK,
             don't fill in FONT_RET.  That tells loadfont to complain.  */
          *font_ret
             = KPSE_FONT_FILE_SOURCE (file_ret) == kpse_source_fallback
               ? KPSE_FONT_FILE_NAME (file_ret)
               : NULL; /* tell loadfont we found something good */
          
          *mag_ret = KPSE_FONT_FILE_DPI (file_ret);
        }
    }
  
  /* loadfont wants the magnification, not the resolution.  */
  *mag_ret *= 5;

  /* If we found a name, return the stream.  */
  ret = name ? xfopen (name) : NULL;
  *filename_ret = name;

  return ret;
}
