/*
 *      Copyright (C) 1993 Bas Laarhoven.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 $Source: /usr/src/distr/ftape-0.9.9d/RCS/ftape-rw.c,v $
 $Author: bas $
 *
 $Revision: 1.21 $
 $Date: 1994/01/16 15:44:54 $
 $State: ALPHA $
 *
 *      This file contains the reading and writing code
 *      for the QIC-40/80 floppy-tape driver for Linux.
 */

static char RCSid[] =
"$Id: ftape-rw.c,v 1.21 1994/01/16 15:44:54 bas ALPHA $";


#include <linux/string.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <asm/dma.h>
#include <asm/segment.h>
#include <asm/system.h>

#include "kernel-interface.h"
#include "qic117.h"
#include "fdc-io.h"
#include "ftape-io.h"
#include "ftape-rw.h"
#include "ecc.h"

/*      Global vars.
 */
int tracing = 4;                /* for debugging: higher -> more verbose */
unsigned char trace_id = 0;

fdc_mode_enum fdc_mode = fdc_idle;
volatile enum runner_status_enum runner_status = idle;

unsigned int bad_sector_table[ 4200]; /* max size for QIC-80, 307.5 ft tape */
volatile int head;
volatile int tail;              /* not volatile but need same type as head */
int fdc_setup_error;
buffer_struct buffer[ NR_FTAPE_BUFFERS];
volatile unsigned int bad_sector_map; /* bad sector map for one segment */
volatile unsigned int next_segment; /* next segment for read ahead */
volatile unsigned int sector_count; /* nr of sectors to read from segment */
volatile unsigned int remaining; /* nr of sector remaining in segment */
volatile unsigned long int buffer_ptr; /* where to put next sector(s) */
volatile unsigned char sector_offset; /* offset for first sector to read */

struct wait_queue *wait_intr = NULL;
unsigned char deblock_buffer[ 29 * SECTOR_SIZE];
int location_known = 0;
int current_segment = -1;
volatile ftape_fast_start_struct ftape_fast_start = { 0, 0, 0 };

/*      Local vars.
 */
static int buf_pos_rd = 0;
static int buf_pos_wr = 0;
static int buf_len_rd = 0;
static int last_write_failed = 0;
static int need_flush = 0;
unsigned char _fdc_head;
unsigned char _fdc_cyl;
unsigned char _fdc_sect;
static struct {
  int id;
  int size;
  int free;
} ftape_last_segment;


/*      Increment cyclic buffer nr.
 */
inline void
next_buffer( volatile int* x)
{
  if (++*x >= NR_ITEMS( buffer)) {
    *x = 0;
  }
}

/*      Count nr of 1's in pattern.
 */
inline int
count_bits( unsigned int mask)
{
  int bits;
  for (bits = 0; mask != 0; mask >>= 1) {
    if (mask & 1) {
      ++bits;
    }
  }
  return bits;
}

/*      Calculate Floppy Disk Controller and DMA parameters for a new segment.
 */
int
setup_new_segment( unsigned int head, unsigned int segment_id,
                  unsigned int offset, unsigned int count)
{
  buffer[ head].segment_id = segment_id;
  buffer[ head].crc_error_map = 0; /* bad bits will get merged in */
  next_segment= segment_id + 1; /* for read ahead */
  buffer_ptr = (long) buffer[ head].address;
  bad_sector_map = bad_sector_table[ segment_id];
  if (bad_sector_map != 0) {
    TRACEx2( 4, "setup_new_segment", "segment: %d, bad sector map: %08lx",
            segment_id, bad_sector_map);
  } else {
    TRACEx1( 4, "setup_new_segment", "segment: %d", segment_id);
  }
  if (offset != 0 || count != sectors_per_segment) {
    TRACEx2( 4, "setup_new_segment", "offset: %d, count: %d",
            offset, count);
  }
  /*
   *    Segments with 3 or less sectors are not written with
   *    valid data because there is no space for the ecc.
   *    The data written is whatever happens to be in the buffer.
   *    Reading such a segment will return a zero byte-count.
   *    To allow us to read/write segments with all bad sectors
   *    we fake one readable sector in the segment. This prevents
   *    having to handle these segments in a very special way.
   *    It is not important if the reading of this bad sector
   *    fails or not (the data is ignored). It is only read to
   *    keep the driver running.
   *    The QIC-40/80 spec. has no information on how to handle
   *    this case, so this is my interpretation.
   */
  if (bad_sector_map == 0xffffffff) {
    TRACE( 4, "setup_new_segment", "empty segment, using fake sector");
    bad_sector_map = EMPTY_SEGMENT; /* fake one readable/writeable sector */
  }
  remaining = count;
  sector_offset = offset;
  _fdc_head = segment_id / segments_per_head;
  _fdc_cyl = (segment_id % segments_per_head) / segments_per_cylinder;
  _fdc_sect = (segment_id % segments_per_cylinder) * sectors_per_segment + 1;
  fdc_setup_error = 0;
  return 0;
}

/*      Determine size of next cluster of good sectors.
 */
int
calc_next_cluster( void)
{
  /* Skip bad sectors.
   */
  while (remaining > 0 && (bad_sector_map & 1) != 0) {
    bad_sector_map >>= 1;
    ++sector_offset;
    --remaining;
  }
  
  /* Find next cluster of good sectors
   */
  if (bad_sector_map == 0) {  /* speed up */
    sector_count = remaining;
  } else {
    sector_count = 0;
    while (sector_count < remaining && (bad_sector_map & 1) == 0) {
      ++sector_count;
      bad_sector_map >>= 1;
    }
  }
  return sector_count;
}

/*      Setup Floppy Disk Controller and DMA to read or write the next cluster
 *      of good sectors from or to the current segment.
 */
int
setup_fdc_and_dma( unsigned char operation)
{
  unsigned long flags;
  unsigned char out[ 9];
  int result;
  int dma_mode;

  if (operation == FDC_READ || operation == FDC_READ_DELETED) {
    dma_mode = DMA_MODE_READ;
  } else if (operation == FDC_WRITE || operation == FDC_WRITE_DELETED) {
    dma_mode = DMA_MODE_WRITE;
  } else {
    TRACE( -1, "setup_fdc_and_dma", "bug: illegal operation parameter");
    return -EIO;
  }
  TRACEx2( 5, "setup_fdc_and_dma", "xfer %d sectors to/from 0x%08lx",
          sector_count, (unsigned long) buffer_ptr);
  /* Program the DMA controller.
   */
  save_flags( flags);
  cli();                      /* could be called from ISR ! */
  disable_dma( FLOPPY_DMA);
  clear_dma_ff( FLOPPY_DMA);
  set_dma_mode( FLOPPY_DMA, dma_mode);
  set_dma_addr( FLOPPY_DMA, buffer_ptr);
  set_dma_count( FLOPPY_DMA, SECTOR_SIZE * sector_count);
  enable_dma( FLOPPY_DMA);
  
  /* Issue FDC command to start reading/writing.
   */
  out[0] = operation;
  out[1] = FTAPE_UNIT;
  out[2] = _fdc_cyl;
  out[3] = _fdc_head;
  out[4] = _fdc_sect + sector_offset;
  out[5] = 3;                   /* Sector size of 1K. */
  out[6] = _fdc_sect + sector_offset + sector_count - 1; /* last sector */
  out[7] = 109;                 /* Gap length. */
  out[8] = 0xff;		/* No limit to transfer size. */
  restore_flags( flags);

  TRACEx4( 6, "setup_fdc_and_dma", "c: 0x%02x, h: 0x%02x, s: 0x%02x, "
          "cnt: 0x%02x", out[ 2], out[ 3], out[ 4], out[ 6]- out[ 4]+ 1);

  /* After all sectors are done an interrupt will follow.
   */
  result = fdc_command( out, 9);
  if (result != 0) {
    fdc_mode = fdc_idle;
    TRACE( 1, "setup_fdc_and_dma", "fdc_command failed");
  }
  fdc_setup_error = result;
  return result;
}

/*      Read Id of first sector passing tape head.
 */
int
ftape_read_id( int *location)
{
  int result;
  unsigned char out[2];
  int retry;

  for (retry = 0; retry < 2; ++retry) {

    out[ 0] = FDC_READID;
    out[1] = FTAPE_UNIT;
    result = fdc_command( out, 2);
    if (result < 0) {
      TRACE( 1, "ftape_read_id", "fdc_command failed");
      *location = -1;
      break;
    }
    result = fdc_interrupt_wait( 5 * SECOND);
    if (result < 0) {
      TRACE( 1, "ftape_read_id", "fdc_interrupt_wait failed");
      *location = -1;
      break;
    }
    /*    result set by isr, will be -1 on error.
     */
    if (fdc_sect == 0) {
      *location = -1;
      result = -EIO;            /* if too many retries */
      TRACE( 5, "ftape_read_id", "current location unknown");
    } else {
      *location = (segments_per_head * fdc_head
                   + segments_per_cylinder * fdc_cyl
                   + (fdc_sect - 1) / sectors_per_segment);
      result = 0;
      TRACEi( 5, "ftape_read_id", "current location =", *location);
      break;
    }
  }
  return result;
}


int
ftape_smart_stop( int* location_known, int* location)
{
  int result;
  int status;

  /* Make sure fdc is ready. There's no information on how long
   * this can take, but 100 usecs seems very long to me.
   * If it still fails, the read_id that follows will fail and
   * tell us so !
   * If the tape is at it's end (drive not ready), the index cue
   * pulses will make sure read_id returns with an error.
   */
  fdc_ready_wait( 100 /* usec */);
  *location_known = (ftape_read_id( location) == 0);
  if (*location < 0) {
    *location_known = 0;
  }
  result = ftape_command_wait( QIC_STOP_TAPE, 5, &status);
  if (result < 0) {
    TRACE( 1, "ftape_smart_stop", "qic_stop_tape command_wait failed");
    *location_known = 0;
  }
  if (*location_known) {
    TRACEi( 4, "ftape_smart_stop", "tape stopped passing segment:", *location);
  } else {
    TRACE( 4, "ftape_smart_stop", "tape stopped at undetermined location");
  }
  return result;
}


/*      Wait until runner has finished tail buffer.
 */
int
wait_segment( buffer_state_enum state)
{
  int result;

  while (buffer[ tail].status == state) {
    /*  First buffer still being worked on, wait up to 5 seconds.
     */
    result = fdc_interrupt_wait( 5 * SECOND);
    if (result < 0) {
      if (result == -EINTR) {
        return result;
      }
      TRACE( 1, "wait_segment", "fdc_interrupt_wait failed");
      return -ETIME;
    }
    if (fdc_setup_error) {
      TRACE( 1, "wait_segment", "setup error");
      /* recover... */
      return -EIO;
    }
  }
  return 0;
}


int
correct_and_copy( unsigned int tail, unsigned char* destination)
{
  struct memory_segment mseg;
  int result;
  BAD_SECTOR read_bad;

  mseg.read_bad = buffer[ tail].crc_error_map;
  mseg.marked_bad = 0;          /* not used... */
  mseg.blocks = buffer[ tail].bytes / SECTOR_SIZE;
  mseg.data = buffer[ tail].address;

  /*    If there are no data sectors we can skip this segment.
   */
  if (mseg.blocks <= 3) {
    TRACE( 4, "correct_and_copy", "empty segment");
    return 0;
  }
  read_bad = mseg.read_bad;

  history.crc_errors += count_bits( read_bad);
  result = ecc_correct_data( &mseg);

  if (read_bad != 0 || mseg.corrected != 0) {
    TRACElx( 4, "correct_and_copy", "bad sectormap:", read_bad);
    TRACElx( 4, "correct_and_copy", "corrected map:", mseg.corrected);
    history.corrected += count_bits( mseg.corrected);
  }

  if (result == ECC_CORRECTED || result == ECC_OK) {
    if (result == ECC_CORRECTED) {
      TRACEi( 3, "correct_and_copy", "ecc corrected segment:",
             buffer[ tail].segment_id);
    }
    memcpy( destination, mseg.data, (mseg.blocks - 3) * SECTOR_SIZE);
    if ((read_bad ^ mseg.corrected) & mseg.corrected) {
      /* sectors corrected without crc errors set */
      history.crc_failures++;
    }
    return (mseg.blocks - 3) * SECTOR_SIZE;
  } else {
    TRACEi( 1, "correct_and_copy", "ecc failure on segment",
           buffer[ tail].segment_id);
    history.ecc_failures++;
    return -EAGAIN;             /* should retry */
  }
}

inline unsigned int
get1( unsigned char* address, unsigned int offset)
{
  return address[ offset];
}

inline unsigned int
get2( unsigned char* address, unsigned int offset)
{
  return address[ offset] + 256 * address[ offset + 1];
}

inline unsigned int
get4( unsigned char* address, unsigned int offset)
{
  return get2( address, offset) + 256 * 256 * get2( address, offset + 2);
}

/* forward */ int
read_segment( unsigned int segment_id, unsigned char* address);

int
read_header_segment( unsigned char* address)
{
  int i;
  int result;
  int header_segment = -1;
  unsigned int tape_segments_per_track;
  unsigned int tape_tracks_per_cartridge;
  unsigned int max_floppy_side;
  unsigned int max_floppy_track;
  unsigned int max_floppy_sector;
  unsigned int failed_sector_count;

  /*    Find first header segment on tape.
   *      THIS IS NOT TESTED !!! Can only be tested with
   *      a tape that has bad sectors in first segment(s).
   *    Get start of data area and verify tape parameters.
   */
  ftape_last_segment.id = 5;    /* will allow us to read the header ! */
  TRACE( 5, "read_header_segment", "reading...");
  do {
    ++header_segment;
    bad_sector_table[ header_segment] = 0x00000000;
    result = read_segment( header_segment, address);
    if (header_segment > 5) {
      /* probably something else wrong, don't keep stuck
       * in this loop !
       */
      TRACE( 1, "read_header_segment", "header segment not found");
      return -EIO;
    }
  } while (result < 0);
  result = ftape_abort_operation( &current_segment);
  if (result < 0) {
    TRACE( 1, "read_header_segment", "ftape_abort_operation failed");
    return -EIO;
  }
  if (get4( address, 0) != 0xaa55aa55) {
    TRACE( 1, "read_header_segment", "wrong signature");
    return -EIO;
  }
  if (get2( address, 6) != header_segment && /* first header */
      get2( address, 8) != header_segment) { /* duplicate */
    TRACE( 1, "read_header_segment", "wrong segment nr");
    return -EIO;
  }

  /*    Verify tape parameters...
   *    QIC-40/80 spec:                 tape_parameters:
   *
   *    segments-per-track              segments_per_track
   *    tracks-per-cartridge            tracks_per_tape
   *    max-floppy-side                 (segments_per_track *
   *                                    tracks_per_tape - 1) /
   *                                    segments_per_head
   *    max-floppy-track                segments_per_head /
   *                                    segments_per_cylinder - 1
   *    max-floppy-sector               segments_per_cylinder *
   *                                    sectors_per_segment
   */

  tape_segments_per_track = get2( address, 24);
  tape_tracks_per_cartridge = get1( address, 26);
  max_floppy_side = get1( address, 27);
  max_floppy_track = get1( address, 28);
  max_floppy_sector = get1( address, 29);

  TRACEi( 4, "read_header_segment", "segments-per-track:  ",
         tape_segments_per_track);
  TRACEi( 4, "read_header_segment", "tracks-per-cartridge:",
         tape_tracks_per_cartridge);
  TRACEi( 4, "read_header_segment", "max-floppy-side:     ",
         max_floppy_side);
  TRACEi( 4, "read_header_segment", "max-floppy-track:    ",
         max_floppy_track);
  TRACEi( 4, "read_header_segment", "max-floppy-sector:   ",
         max_floppy_sector);

  if (tape_segments_per_track == 0 && tape_tracks_per_cartridge == 0 &&
      max_floppy_side == 0 && max_floppy_track == 0 &&
      max_floppy_sector == 0) {
    /*
     *  QIC-40 Rev E and earlier
     */
    tape_segments_per_track = 68; 
    tape_tracks_per_cartridge = 20;
    max_floppy_side = 1;
    max_floppy_track = 169;
    max_floppy_sector = 128;
  }

#ifdef CONNER_BUG
  /*  If included, this test will compensate for the wrong parameter
   *  on tapes formatted by Conner's DOS software.
   *  Maybe there are other bad combinations, but this is the only
   *  one I've heard of up to now - 03/01/94 Bas.
   */
  if (tape_segments_per_track == 150 &&
      tape_tracks_per_cartridge == 28 &&
      max_floppy_side == 7 &&
      max_floppy_track == 149 &&
      max_floppy_sector == 128) {
    max_floppy_side = 6;
  }
#endif
  /*
   *    in case report_drive_configuration command failed,
   *    we set the parameters according to the tape data.
   */
  if (unknown_drive_config) {
    segments_per_track = tape_segments_per_track;
    tracks_per_tape = tape_tracks_per_cartridge;
    segments_per_head = ( (tracks_per_tape + max_floppy_side) /
                         (max_floppy_side + 1) ) * segments_per_track;
  }
  /*
   *    Verify drive_configuration with tape parameters
   */
  if ( (segments_per_track != tape_segments_per_track) ||
      (tracks_per_tape != tape_tracks_per_cartridge) ||
      ((segments_per_track * tracks_per_tape - 1) / segments_per_head
       != max_floppy_side) ||
      (segments_per_head / segments_per_cylinder - 1 != max_floppy_track) ||
      (segments_per_cylinder * sectors_per_segment != max_floppy_sector) ) {
    TRACE( 1, "read_header_segment", "Tape parameters error");
    return -EIO;
  }
  first_data_segment = get2( address, 10); /* first data segment */
  TRACEi( 4, "read_header_segment", "first data segment:", first_data_segment);

  /* First copy the bad sector map from sector 2 and up into our own table.
   * Then merge the failed sector log into this map.
   */
  memcpy( bad_sector_table,
         address + 2 * SECTOR_SIZE, sizeof( bad_sector_table));
  failed_sector_count = get2( address, 144);
  if (failed_sector_count > 0) {
    TRACE( -1, "read_header_segment",
          "Gosh, somebody is using the failed sector list !");
    TRACE( -1, "read_header_segment",
          "So it makes sense to implement it, sometime...");
  }
#if 0
  bad_sector_table[ segments_per_track * tracks_per_tape - 3] = 0x000003e0;
  bad_sector_table[ segments_per_track * tracks_per_tape - 2] = 0xff3fffff;
  bad_sector_table[ segments_per_track * tracks_per_tape - 1] = 0xffffe000;
#endif
  /*  Find the highest segment id that allows still one full
   *  deblock_buffer to be written to tape.
   */
  ftape_last_segment.size = 0;
  for (i = segments_per_track * tracks_per_tape - 1; i >= 0; --i) {
    int space = sectors_per_segment - 3 - count_bits( bad_sector_table[ i]);
    if (space > 0) {
      ftape_last_segment.size += space; /* sectors free */
      ftape_last_segment.free = (ftape_last_segment.size -
                                 sizeof( deblock_buffer) / SECTOR_SIZE);
      if (ftape_last_segment.free >= 0) {
        ftape_last_segment.id = i;
        TRACEx2( 4, "read_header_segment", "`last' segment is %d, %d Kb",
                ftape_last_segment.id, ftape_last_segment.size);
        break;
      }
    }
  }
#if 0
  /*  Enable to test error recovery on Verbatime tape
   *  with bad area.
   */
  bad_sector_table[ 16] = 0xffffff80;
  bad_sector_table[ 17] = 0xffffffff;
  bad_sector_table[ 18] = 0x0000ffff;
#endif
#if 0
  /*  Enable to test bad sector handling
   */
  bad_sector_table[ 30] = 0xfffffffe;
  bad_sector_table[ 32] = 0x7fffffff;
  bad_sector_table[ 34] = 0xfffeffff;
  bad_sector_table[ 36] = 0x55555555;
  bad_sector_table[ 38] = 0xffffffff;
  bad_sector_table[ 50] = 0xffff0000;
  bad_sector_table[ 51] = 0xffffffff;
  bad_sector_table[ 52] = 0xffffffff;
  bad_sector_table[ 53] = 0x0000ffff;
#endif
#if 0
  /*  Enable when testing multiple volume tar dumps.
   */
  for (i = first_data_segment; i <= ftape_last_segment.id - 7; ++i) {
    bad_sector_table[ i] = 0xffffffff;
  }
#endif
    
  if (tracing > 2) {    
    unsigned int map;
    int good_sectors = 0;
    int bad_sectors;
    unsigned int total_bad = 0;

    for (i = first_data_segment;
         i < segments_per_track * tracks_per_tape; ++i) {
      map = bad_sector_table[ i];
      bad_sectors = count_bits( map);
      if (bad_sectors > 0) {
        TRACEx2( 8, "read_header_segment",
                "bsm for segment %4d: 0x%08x", i, map);
        if (bad_sectors > sectors_per_segment - 3) {
          bad_sectors = sectors_per_segment - 3;
        }
        total_bad += bad_sectors;
      }
      good_sectors += sectors_per_segment - 3 - bad_sectors;
    }
    TRACEx1( 3, "read_header_segment",
           "%d Kb usable on this tape",
            good_sectors - ftape_last_segment.free);
    if (total_bad == 0) {
      TRACE( 1, "read_header_segment",
            "WARNING: this tape has no bad blocks registered !");
    }
  }
  ftape_reset_position();
  return 0;
}


/*      Read given segment into buffer at address.
 */
int
read_segment( unsigned int segment_id, unsigned char* address)
{
  int read_done= 0;
  int result;
  int bytes_read = 0;
  int retry = 0;
  
  TRACEi( 5, "read_segment", "entered, segment_id =", segment_id);

  if (ftape_state != reading) {
    if (ftape_state == writing) {
      ftape_flush_buffers();    /* flush write buffer */
      TRACE( 5, "read_segment", "calling ftape_abort_operation");
      result = ftape_abort_operation( &current_segment);
      if (result < 0) {
        TRACE( 1, "read_segment", "ftape_abort_operation failed");
        return -EIO;
      }
    } else {
      /* clear remaining read buffers */
      ftape_zap_buffers();
    }
    ftape_state = reading;
  }
  if (segment_id >= segments_per_track * tracks_per_tape) {
    TRACE( 5, "read_segment", "reading past end of tape");
    return -ENOSPC;
  }
  do {
    /*    Search all full buffers for the first matching the wanted segment.
     *    Clear other buffers on the fly.
     */
    while (!read_done && buffer[ tail].status == full) {
      if (buffer[ tail].segment_id == segment_id) {
        /*        If out buffer is yet full, return its contents.
         */
        TRACEi( 5, "read_segment", "found segment in cache :", segment_id);
        bytes_read = correct_and_copy( tail, address);
        TRACEi( 5, "read_segment", "segment contains (bytes) :", bytes_read);
        if (bytes_read < 0) {
          if (bytes_read == -EAGAIN) {
            TRACE( 1, "read_segment", "ecc failed, retry");
            /* keep read_done == 0, will trigger ftape_abort_operation
               because reading wrong segment.
             */
            ++retry;
          } else {
            return bytes_read;
          }
        } else {
          read_done = 1;
        }
      } else {
        TRACEi( 5, "read_segment", "zapping segment in cache :",
               buffer[ tail].segment_id);
      }
      buffer[ tail].status = empty;
      next_buffer( &tail);
    }
    if (!read_done && buffer[ tail].status == reading) {
      if (buffer[ tail].segment_id == segment_id) {
        int result = wait_segment( reading);
        if (result == -EINTR) {
          return result;
        }
        if (result < 0) {
          TRACE( 1, "read_segment", "wait_segment failed while reading");
          return -EIO;
        }
      } else {
        /*        We're reading the wrong segment, stop runner.
         */
        ftape_abort_operation( &current_segment);
      }
    }
    /*    if just passed the last segment on a track, wait for BOT or EOT mark.
     */
    if (runner_status == logical_eot) {
      int status;
      result = ftape_ready_wait( 10 * SECOND, &status);
      if (result < 0) {
        TRACE( 1, "read_segment",
              "ftape_ready_wait waiting for eot/bot failed");
      }
      if (status & (QIC_STATUS_AT_BOT | QIC_STATUS_AT_EOT) == 0) {
        TRACE( 1, "read_segment", "eot/bot not reached");
      }
      runner_status = end_of_tape;
    }
    /*    should runner stop ?
     */
    if (runner_status == aborting || runner_status == buffer_overrun ||
        runner_status == end_of_tape) {
      if (runner_status != end_of_tape) {
        ftape_smart_stop( &location_known, &current_segment);
      }
      if (runner_status == aborting) {
        expect_stray_interrupt = 0;
        if (buffer[ head].status == reading || buffer[ head].status == error) {
          buffer[ head].status = empty;
          ++retry;
        }
      }
      runner_status = idle;       /* aborted ? */
    }
    /*    If segment to read is empty, do not start runner for it,
     *    but wait for next read call.
     */
    if (bad_sector_table[ segment_id] == 0xffffffff) {
      bytes_read = 0;
      read_done = 1;
    }
    if (read_done) {
      break;
    }
    
    /*    Now at least one buffer is empty !
     *    Restart runner & tape if needed.
     */
    TRACEx3( 8, "read_segment", "head: %d, tail: %d, runner_status: %d",
            head, tail, runner_status);
    TRACEx2( 8, "read_segment", "buffer[].status, [head]: %d, [tail]: %d",
            buffer[ head].status, buffer[ tail].status);
    
    if (runner_status == idle && buffer[ tail].status != full) {
      result= ftape_start_tape( segment_id);
      if (result < 0) {
        TRACEx1( 1, "read_segment", "Error: segment %d unreachable",
                segment_id);
        return result;
      }
      runner_status = running;
      
      /*  Start runner reading.
       */
      setup_new_segment( head, segment_id, 0, sectors_per_segment);
      buffer[ head].status = reading;
      if (calc_next_cluster() == 0) {
        TRACE( -1, "read_segment", "bug: empty segment");
        return -EIO;
      }
      if (setup_fdc_and_dma( FDC_READ)) {
        buffer[ head].status = error;
        TRACE( 1, "read_segment", "setup error while starting");
        /* recover ...? */
        return -EIO;
      }
    }
    
    /*    If we got a segment: quit, or else retry up to limit.
     *    Allow at least three attempts, First causing the error,
     *    Second for possibly failing (fast) ftape_start_tape, Third
     *    for recovery of ftape_start_tape.
     */
  } while (!read_done && retry < (3 + RETRIES_ON_SOFT_ERROR));

  if (read_done) {
    return bytes_read;
  } else {
    TRACE( 1, "read_segment", "too many retries");
    return -EIO;
  }
}

int
seek_reverse( int count)
{
  int result;
  int status;

  result = ftape_ready_wait( 5, &status);  /* timeout for stop-tape command */
  if (result < 0) {
    TRACE( 1, "seek_reverse", "drive not ready");
    return -EIO;
  }

  /* Issue this tape command first. */
  result = ftape_command( QIC_SKIP_REVERSE);
  if (result == 0) {
    /* Issue the low nibble of the command. */
    result = ftape_parameter( 2 + (count & 0x0f));
    if (result == 0) {
      /* Issue the high nibble and wait for the command to complete. */
      result = ftape_parameter( 2 + ((count >> 4) & 0x0f));
      if (result == 0) {
        result = ftape_ready_wait( 85, &status);
      }
    }
  }
  return result;
}

/*      Get the tape running and position it just before the
 *      requested segment.
 *      Seek tape-track and reposition as needed.
 *      Use information from smart_stop and ftape_track.
 */

int
ftape_start_tape( int segment_id)
{
  int track = segment_id / segments_per_track;
  int next_segment = -1;
  int result;
  int status;
  int at_end_of_track;
  int tape_running = 0;
  int rewind_gaps;
  static int reverse_margin;
  static int last_segment = -1;
  int ids_read;

  if (ftape_fast_start.error) {
    ++ftape_fast_start.offset;
    ++reverse_margin;
    ftape_fast_start.error = 0;
    TRACE( 5, "ftape_start_tape", "fast start disabled");
  }
  if (last_segment == segment_id) { /* might be a retry */
    TRACEi( 5, "ftape_start_tape", "retry seek to segment", segment_id);
  } else {
    reverse_margin = ftape_fast_start.offset;
    TRACEi( 5, "ftape_start_tape", "seek segment", segment_id);
  }
  last_segment = segment_id;

  do {
    if (!tape_running) {
      result = ftape_ready_wait( 15, &status);
      if (result < 0) {
        TRACEi( 1, "ftape_start_tape",
               "wait for ready failed with code", result);
        return result;
      }
    }
    result = ftape_report_drive_status( &status);
    if (result < 0) {
      TRACEi( 1, "ftape_start_tape",
             "ftape_report_drive_status failed with code", result);
      return result;
    }
    if (ftape_track != track) {
      /* current unknown or not equal to destination
       */
      TRACEi( 5, "ftape_start_tape", "seeking head to track", track);
      ftape_seek_head_to_track( track);
      location_known = 0;
    }
    at_end_of_track = 0;
    if (location_known) {
      TRACE( 5, "ftape_start_tape", "position known");
      next_segment = 1 + current_segment;
    } else while (!location_known) {
      if (status & (QIC_STATUS_AT_BOT | QIC_STATUS_AT_EOT)) {
        if (((ftape_track & 1) == 0 && (status & QIC_STATUS_AT_EOT)) ||
            ((ftape_track & 1) != 0 && (status & QIC_STATUS_AT_BOT))) {
          /* tape at end of track, will stop or is already stopped */
          tape_running = 0;
          TRACE( 5, "ftape_start_tape", "positioned at end of track");
          next_segment = (ftape_track + 1) * segments_per_track;
          at_end_of_track = 1;
        } else {
          TRACE( 5, "ftape_start_tape", "positioned at begin of track");
          next_segment = ftape_track * segments_per_track;
        }
        location_known = 1;
      } else {
        if (!tape_running) {
          ftape_command( QIC_LOGICAL_FORWARD);
          TRACE( 5,"ftape_start_tape", "tape started at unknown position");
          tape_running = 1;
        }
        result = ftape_read_id( &current_segment);
        if (result == 0 && current_segment >= 0) {
          next_segment = 1 + current_segment;
          location_known = 1;
        } else {
          TRACE( 5,"ftape_start_tape", "ftape_read_id failed");
          result = ftape_report_drive_status( &status);
          if (result < 0) {
            TRACEi( 1, "ftape_start_tape",
                   "ftape_report_drive_status failed with code", result);
            return result;
          }
        }
      }
    } /* while (!location_known) */
    
    /*
     * We're on the right track now and our position is also known
     */
    rewind_gaps = next_segment - segment_id;
    if (at_end_of_track) {
      /* According to the qic-40/80 spec. one should be sufficient,
       * in practice more may be needed.
       * If set too low, we'll need one or more extra retries !
       */
      rewind_gaps += 4;
    }
    if (rewind_gaps > 0) {        /* in or past target segment */
      if (tape_running) {
        result = ftape_command_wait( QIC_STOP_TAPE, 5, &status);
        if (result < 0) {
          TRACEi( 1, "ftape_start_tape",
                 "stop tape failed failed with code", result);
          return result;
        }
        TRACE( 5,"ftape_start_tape", "tape stopped");
        tape_running = 0;
      }
      TRACE( 5, "ftape_start_tape", "repositioning");
      seek_reverse( rewind_gaps + reverse_margin - 1);
      ++history.rewinds;
      next_segment = segment_id - reverse_margin;
      result = ftape_ready_wait( 15, &status);
      if (result < 0) {
        TRACEi( 1, "ftape_start_tape",
               "wait for ready failed with code", result);
        return result;
      }
      result = ftape_report_drive_status( &status);
      if (result < 0) {
        TRACEi( 1, "ftape_start_tape",
               "ftape_report_drive_status failed with code", result);
        return result;
      }
      if (status & (QIC_STATUS_AT_BOT | QIC_STATUS_AT_EOT)) {
        /* can only be at the beginning of a track */
        next_segment = ftape_track * segments_per_track;
        tape_running = 0;       /* probably stopped */
      }
    }
    /*
     * We're on track, before the right segment now
     */
    if (!tape_running) {
      ftape_command( QIC_LOGICAL_FORWARD);
      TRACEi( 5,"ftape_start_tape",
             "starting tape before segment", next_segment);
      tape_running = 1;
    }
    ids_read = 0;
    if (ftape_fast_start.offset == 0 && 
        !(status & (QIC_STATUS_AT_BOT | QIC_STATUS_AT_EOT))) {
      ftape_fast_start.active = 1;
    }
    while (next_segment < segment_id) {
      result = ftape_read_id( &current_segment);
      if (result == 0 && current_segment >= 0) {
        next_segment = 1 + current_segment;
        TRACEi( 5, "ftape_start_tape", "skipping segment", current_segment);
      } else {
        if (ids_read++ > 25) {  /* must be enough */
          TRACE( 1, "ftape_start_tape", "read_id failed completely");
          return -EIO;
        }
      }
    }
    if (ftape_fast_start.active) {
      TRACE( 5, "ftape_start_tape", "fast start active");
    }
    if (next_segment == segment_id) {
      return 0;                   /* on target */
    }
    if (reverse_margin == 0) {
      reverse_margin = 1;
    } else {
      reverse_margin *= 2;        /* for retry */
    }
  } while (reverse_margin <= 8);

  TRACE( 1, "ftape_start_tape", "failed to reposition");
  return -EIO;
}

int
_ftape_read( char* buff, int req_len)
{
  int result;
  int cnt;
  int to_do = req_len;
  static int remaining;
  int bytes_read = 0;

  history.used |= 1;
  if (first_data_segment == -1) {
    result = read_header_segment( deblock_buffer);
    if (result < 0) {
      return result;
    }
  }
  /*  As GNU tar doesn't accept partial read counts when the multiple
   *  volume flag is set, we make sure to return the requested amount
   *  of data. Except, of course, at the end of the tape.
   */
  while (to_do > 0) {           /* don't return with a partial count ! */ 
    /*  If we're reading the `last' segment(s) on tape, make sure we don't
     *  get more than 29 Kb from it (As it only contains this much).
     *  This works only for sequential access, so random access should
     *  stay away from this `last' segment.
     *  Note: ftape_seg_pos points to the next segment what will be
     *        read, so it's one too hight here!
     */
    if (ftape_seg_pos - 1 >= ftape_last_segment.id) {
      TRACEi( 5, "_ftape_read", "remaining of last segment:", remaining);
      if (to_do > remaining) {
        to_do = remaining;      /* fake a smaller request */
        TRACE( 5, "_ftape_read", "clipped request to remaining");
      }
    }
    while (buf_len_rd == 0) {
      /*  When starting to read the `last' segment, set remaining
       */
      if (ftape_seg_pos == ftape_last_segment.id) {
        remaining = sizeof( deblock_buffer);
        TRACEi( 5, "_ftape_read", "remaining set to:", remaining);
      }
      result = read_segment( ftape_seg_pos, deblock_buffer);
      if (result < 0) {
        TRACEx1( 4, "_ftape_read", "read_segment result: %d", result);
        return result;
      }
      /*  Allow escape from this loop on signal !
       */
      if (current->signal & ~_BLOCKABLE) {
        TRACE( 2, "_ftape_read", "interrupted by signal");
        return -EINTR;
      }
      buf_pos_rd = 0;
      buf_len_rd = result;
      ++ftape_seg_pos;
    }
    /*  Take as much as we can use
     */
    cnt = (buf_len_rd < to_do) ? buf_len_rd : to_do;
    TRACEi( 7, "_ftape_read", "nr bytes just read:", cnt);
    result = verify_area( VERIFY_WRITE, buff, cnt);
    if (result) {
      TRACE( 1, "_ftape_read", "verify_area failed");
      return -EIO;
    }
    memcpy_tofs( buff, deblock_buffer + buf_pos_rd, cnt);
    buff += cnt;
    to_do -= cnt;               /* what's left from req_len */
    remaining -= cnt;           /* what remains on this tape */
    bytes_read += cnt;          /* what we got so far */
    buf_pos_rd += cnt;          /* index in buffer */
    buf_len_rd -= cnt;          /* remaining bytes in buffer */
    ftape_chr_pos += cnt;       /* tape position */
  }
  return bytes_read;
}

int
copy_and_gen_ecc( char* destination, unsigned char* source,
                 unsigned int bad_sector_map)
{
  int result;
  struct memory_segment mseg;
  int bads = count_bits( bad_sector_map);

  if (bads > 0) {
    TRACEi( 4, "copy_and_gen_ecc", "bad sectors in map:", bads);
  }
  if (bads + 3 >= sectors_per_segment) {
    TRACE( 4, "copy_and_gen_ecc", "empty segment");
    mseg.blocks = 0;            /* skip entire segment */
    return 0;                   /* nothing written */
  } else {
    mseg.blocks = sectors_per_segment - bads;
  }
  mseg.data = destination;
  memcpy( mseg.data, source, (mseg.blocks - 3) * SECTOR_SIZE);
  result = ecc_set_segment_parity( &mseg);
  if (result < 0) {
    TRACE( 1, "copy_and_gen_ecc", "ecc_set_segment_parity failed");
    return result;
  }

  return (mseg.blocks - 3) * SECTOR_SIZE;
}

/*      Write given segment from buffer at address onto tape.
 */
int
write_segment( unsigned int segment_id, unsigned char* address, int flushing)
{
  int result;
  int bytes_written = 0;
  
  TRACEi( 5, "write_segment", "entered, segment_id =", segment_id);

  if (ftape_state != writing) {
    if (ftape_state == reading) {
      TRACE( 5, "write_segment", "calling ftape_abort_operation");
      result = ftape_abort_operation( &current_segment);
      if (result < 0) {
        TRACE( 1, "write_segment", "ftape_abort_operation failed");
        return -EIO;
      }
    }
    /* clear remaining read buffers */
    ftape_zap_buffers();
    ftape_state = writing;
  }
 
  /*    if all buffers full we'll have to wait...
   */
  wait_segment( writing);
  if (buffer[ tail].status == error) {
    /*  setup for a retry
     */
    buffer[ tail].status = full;
    bytes_written = -EAGAIN;    /* force retry */
  } else if (buffer[ tail].status != empty) {
    TRACE( 1, "write_segment", "wait for empty segment failed");
    return -EIO;
  }

  /*    If just passed last segment on tape: wait for BOT or EOT mark.
   */
  if (runner_status == logical_eot) {
    int status;
    result = ftape_ready_wait( 10 * SECOND, &status);
    if (result < 0) {
      TRACE( 1,"write_segment", "ftape_ready_wait waiting for eot/bot failed");
    }
    if (status & (QIC_STATUS_AT_BOT | QIC_STATUS_AT_EOT) == 0) {
      TRACE( 1, "write_segment", "eot/bot not reached");
    }
    runner_status = end_of_tape;
  }
  /*    should runner stop ?
   */
  if (runner_status == aborting || runner_status == buffer_underrun ||
      runner_status == end_of_tape) {
    if (runner_status != end_of_tape) {
      ftape_smart_stop( &location_known, &current_segment);
    }
    if (runner_status == aborting) {
      expect_stray_interrupt = 0;
      if (buffer[ head].status == writing) {
        buffer[ head].status = empty; /* ????? */
      }
    }
    runner_status = idle;       /* aborted ? */
  }

#ifdef TESTING
  /*  This allows us to skip part of the tape by setting all bits in the
   *  bad sector table.
   *  Include it in the std code if it doesn't interfere with normal operation.
   */
  if (runner_status == idle && bad_sector_table[ segment_id] == 0xffffffff) {
    return bytes_written;       /* will be 0 */
  }
#endif

  if (buffer[ tail].status == empty) {

    /*    now at least one buffer is empty, fill it with our data.
     *    skip bad sectors and generate ecc.
     *    copy_and_gen_ecc return nr of bytes written,
     *    range 0..29 Kb inclusive !
     */
    result = copy_and_gen_ecc( buffer[ tail].address, address,
                              bad_sector_table[ segment_id]);
    if (result < 0) {
      TRACE( 1, "write_segment", "copy_and_gen_ecc failed");
      return result;
    }
    bytes_written = result;
    buffer[ tail].segment_id = segment_id;
    buffer[ tail].status = full;
    next_buffer( &tail);
  }

  if (runner_status == running) {
    /* isr will keep things running */
    return bytes_written;
  }
  /*    Start tape only if all buffers full or flush mode.
   *    This will give higher probability of streaming.
   */
  if (buffer[ tail].status == full || flushing) {

    int segment_id = buffer[ head].segment_id;

    if (!flushing && head != tail) {
      TRACE( 1, "write_segment", "head != tail");
      return -EIO;
    }
    setup_new_segment( head, segment_id, 0, sectors_per_segment);
    if (buffer[ head].status != full) {
      TRACE( 1, "write_segment", "unexpected head status");
      return -EIO;
    }
    buffer[ head].status = writing;

    if (runner_status == idle) {
      TRACEi( 5, "write_segment",
             "starting runner for segment", segment_id);
      result= ftape_start_tape( segment_id);
      if (result < 0) {
        TRACEi( 1, "write_segment",
               "Error - couldn't reach segment", segment_id);
        return result;
      }
      runner_status = running;
    }
    /*  Start runner writing.
     *  if there are 3 or less usable sectors in the segment
     *  they are not filled with valid data, and it must be
     *  ignored when reading.
     */
    if (calc_next_cluster() == 0) {
      /* could happen ! must skip this segment... TODO */
      TRACE( 4, "write_segment", "empty segment");
      return 0;
    }
    if (setup_fdc_and_dma( FDC_WRITE)) {
      buffer[ head].status = error;
      TRACE( 1, "write_segment", "setup error while starting");
      /* recover ... */
      return -EIO;
    }
  }
  return bytes_written;
}

int
_write_segment( unsigned int segment_id, unsigned char* buffer, int flush)
{
  static int retry = 0;
  int result;

  if (segment_id > ftape_last_segment.id && !flush) {
    return -ENOSPC;             /* tape full */
  }
  result = write_segment( segment_id, buffer, flush);

  history.used |= 2;
  if (result >= 0) {            /* success */
    retry = 0;                  /* clear after success */
  }
  if (result == 0) {            /* empty segment */
    TRACE( 4, "_write_segment", "empty segment, skipping");
  }
  if (result < 0) {
    if (result == -EAGAIN) {
      if (++retry > 3) {        /* only this often */
        TRACE( 1, "_write_segment", "write error, all retries failed");
        result = -EIO;
      } else {
        TRACE( 2, "_write_segment", "write error, retrying");
      }
    } else {
      TRACEi( 1, "_write_segment", "write_segment failed, error:", -result);
    }
  }
  return result;
}

void
ftape_zap_buffers( void)
{
  int i;

  for (i = 0; i < NR_ITEMS( buffer); ++i) {
    buffer[ i].address = ((char*)tape_buffer)+ i* BUFF_SIZE;
    buffer[ i].status = empty;
    buffer[ i].bytes = 0;
  }
  buf_len_rd = 0;
  buf_pos_rd = 0;

  ftape_state = idle;
}

int
ftape_flush_buffers( void)
{
  int result;
  int cnt;
  int buf_data_len;
  int _buf_pos_wr;

  TRACEi( 5, "ftape_flush_buffers", "entered, ftape_state =", ftape_state);

  if (ftape_state != writing && !need_flush) {
    return 0;
  }

  /*  Make copy of buf_pos_wr and clear original so if write
   *  fails once, further write attempts will be prevented.
   */
  _buf_pos_wr = buf_pos_wr;
  buf_pos_wr = 0;

  TRACE( 5, "ftape_flush_buffers", "flushing write buffers");

  if (last_write_failed) {
    ftape_zap_buffers();
    return write_protected ? -EROFS : -EIO;
  }

  buf_data_len = _buf_pos_wr;
  /*
   *    If there is any data not written to tape yet, append zero's
   *    to the buffer to fill the remainder of the segment.
   *    Then write the segment(s) to tape.
   */
  TRACEi( 7, "ftape_flush_buffers",
         "flush, remainder in buffer:", _buf_pos_wr);

  do {
    cnt = sizeof( deblock_buffer) - _buf_pos_wr; /* remaining */
    TRACEi( 7, "ftape_flush_buffers", "flush, padding count:", cnt);
    /* pad buffer with 0's */
    memset( deblock_buffer + _buf_pos_wr, 0, cnt);
    result = _write_segment( ftape_seg_pos, deblock_buffer, 1);
    if (result < 0) {
      if (result == -EAGAIN) {
        continue;
      }
      if (result != -ENOSPC) {
        last_write_failed = 1;
      }
      return result;
    }
    TRACEi( 7, "ftape_flush_buffers", "flush, moved out buffer:", result);
    if (result < buf_data_len) {
      /* Partial write, we'll need another segment
       */
      _buf_pos_wr = sizeof( deblock_buffer) - result;
      if (result > 0) {
        memmove( deblock_buffer, deblock_buffer + result, _buf_pos_wr);
      }
    } else {
      _buf_pos_wr = 0;
    }
    buf_data_len -= result;
    ++ftape_seg_pos;
  } while (buf_data_len > 0);
  TRACEi( 7, "ftape_flush_buffers",
         "flush buffer netto pad-count:", -buf_data_len);
  /*
   *  Wait until all data is actually written to tape.
   */
  while (head != tail || buffer[ head].status != empty) {
    int i;
    
    TRACEx2( 7, "ftape_flush_buffers", "tail: %d, head: %d", tail, head);
    for (i = 0; i < NR_ITEMS( buffer); ++i) {
      TRACEx3( 8, "ftape_flush_buffers",
              "buffer[ %d] segment_id: %d, status: %d",
              i, buffer[ i].segment_id, buffer[ i].status);
    }
    result = fdc_interrupt_wait( 5 * SECOND);
    if (result < 0) {
      TRACE( 1, "wait_segment", "fdc_interrupt_wait failed");
      last_write_failed = 1;
      return result;
    }
    TRACE( 5, "ftape_flush_buffers", "looping until writes done");
  }
  ftape_state = idle;
  last_write_failed = 0;
  need_flush = 0;
  return 0;
}

int
_ftape_write( char* buff, int req_len)
{
  int result;
  int cnt;
  int written = 0;

  /*
   *    If we haven't read the header segment yet, do it now.
   *    This will verify the configuration and get the
   *    bad sector table. We'll use the buffer for scratch.
   */
  if (first_data_segment == -1) {
    result = read_header_segment( deblock_buffer);
    if (result < 0) {
      return result;
    }
  }
  if (write_protected) {
    last_write_failed = 1;
    TRACE( 1, "_ftape_write", "error: cartridge write protected");
    return -EROFS;
  }
  if (ftape_seg_pos > ftape_last_segment.id) {
    return -ENOSPC;             /* full is full */
  }
  /*
   *    This part writes data blocks to tape until the
   *    requested amount is written.
   *    The data will go in a buffer until it's enough
   *    for a segment without bad sectors. Then we'll write
   *    that segment to tape.
   *    The bytes written will be removed from the buffer
   *    and the process is repeated until there is less
   *    than one segment to write left in the buffer.
   */
  while (req_len > 0) {
    int space_left = sizeof( deblock_buffer) - buf_pos_wr;

    TRACEi( 7, "_ftape_write", "remaining req_len:", req_len);
    TRACEi( 7, "_ftape_write", "          buf_pos:", buf_pos_wr);

    cnt = (req_len < space_left) ? req_len : space_left;
    result = verify_area( VERIFY_READ, buff, cnt);
    if (result) {
      TRACE( 1, "_ftape_write", "verify_area failed");
      last_write_failed = 1;
      return result;
    }
    memcpy_fromfs( deblock_buffer + buf_pos_wr, buff, cnt);
    buff += cnt;
    req_len -= cnt;
    buf_pos_wr += cnt;
    TRACEi( 7, "_ftape_write", "moved into blocking buffer:", cnt);

    while (buf_pos_wr >= sizeof( deblock_buffer)) {
      /*  If this is the last buffer to be written, let flush handle it.
       */
      if (ftape_seg_pos >= ftape_last_segment.id) {
        TRACEi( 7, "_ftape_write",
               "remaining in blocking buffer:", buf_pos_wr);
        TRACEi( 7, "_ftape_write", "just written bytes:", written + cnt);
        return written + cnt;
      }
      /* Got one full buffer, write it to disk
       */
      result = _write_segment( ftape_seg_pos, deblock_buffer, 0);
      TRACEi( 5, "_ftape_write", "_write_segment result =", result);
      if (result < 0) {
        if (result == -EAGAIN) {
          TRACE( 5, "_ftape_write", "retry...");
          continue;             /* failed, retry same segment */
        }
        last_write_failed = 1;
        return result;
      }
      if (result > 0 && result < buf_pos_wr) {
        /* Partial write: move remainder in lower part of buffer
         */
        memmove( deblock_buffer, deblock_buffer + result, buf_pos_wr - result);
      }
      TRACEi( 7, "_ftape_write", "moved out of blocking buffer:", result);
      buf_pos_wr -= result;     /* remainder */
      ++ftape_seg_pos;
      /* Allow us to escape from this loop with a signal !
       */
      if (current->signal & ~_BLOCKABLE) {
        TRACE( 1, "_ftape_write", "interrupted by signal");
        last_write_failed = 1;
        return -EINTR;          /* is this the right return value ? */
      }
    }
    ftape_chr_pos += cnt;       /* tape position */
    written += cnt;
  }
  TRACEi( 7, "_ftape_write", "remaining in blocking buffer:", buf_pos_wr);
  TRACEi( 7, "_ftape_write", "just written bytes:", written);

  last_write_failed = 0;
  if (written > 0) {
    need_flush = 1;
  }
  return written;               /* bytes written */
}
