/* Directory routines
   Copyright (C) 1994 Miguel de Icaza.
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#ifdef __svr4__
#define d_namelen d_reclen
#endif
#include "global.h"
#include "dir.h"
#include "util.h"

static char rcsid [] = "$Header: /usr/users/miguel/c/CVS/nc/dir.c,v 1.8 1994/09/08 19:27:55 miguel Exp $";

/* If true show files starting with a dot */
int show_dot_files = 1;

/* If true show files ending in ~ */
int show_backups = 0;

/* If false then directories are shown separately from files */
int mix_all_files = 0;

/* Reverse flag */
static int reverse = 1;

int unsorted (const void *a, const void *b)
{
    return 0;
}

int sort_name (const file_entry *a, const file_entry *b)
{
    int ad = S_ISDIR (a->buf.st_mode) ? 1 : 0;
    int bd = S_ISDIR (b->buf.st_mode) ? 1 : 0;

    if (ad == bd || mix_all_files)
	return strcmp (a->fname, b->fname) * reverse;
    else
	return (bd-ad) * reverse;
}

int sort_ext (const file_entry *a, const file_entry *b)
{
    char *exta, *extb;
    int r;
    int ad = S_ISDIR (a->buf.st_mode) ? 1 : 0;
    int bd = S_ISDIR (b->buf.st_mode) ? 1 : 0;

    if (ad == bd || mix_all_files){
	exta = extension (a->fname);
	extb = extension (b->fname);
	r = strcmp (exta, extb);
	if (r)
	    return r * reverse;
	else
	    return sort_name (a, b);
    } else
	return (bd-ad) * reverse;
}

int sort_time (const file_entry *a, const file_entry *b)
{
    int ad = S_ISDIR (a->buf.st_mode) ? 1 : 0;
    int bd = S_ISDIR (b->buf.st_mode) ? 1 : 0;

    if (ad == bd || mix_all_files)
	return (a->buf.st_mtime - b->buf.st_mtime) * reverse;
    else
	return (bd-ad) * reverse;
}

int sort_size (const file_entry *a, const file_entry *b)
{
    int ad = S_ISDIR (a->buf.st_mode) ? 1 : 0;
    int bd = S_ISDIR (b->buf.st_mode) ? 1 : 0;

    if (ad == bd || mix_all_files)
	return (b->buf.st_size - a->buf.st_size) * reverse;
    else
	return (bd-ad) * reverse;
}

void do_sort (dir_list *list, sortfn *sort, int top, int reverse_f)
{
    reverse = reverse_f ? -1 : 1;
    qsort (&(list->list) [1], top, sizeof (file_entry), sort);
}

void clean_dir (dir_list *list, int count)
{
    int i;
    
    for (i = 0; i < count; i++)
	if ((list->list) [i].fname)
	    free ((list->list) [i].fname);
}

/* Used to set up a directory list when there is no access to a directory */
int set_zero_dir (dir_list *list)
{
    (list->list) [0].fname = "/";
    (list->list) [0].f.marked = 0;
    lstat ("/", &((list->list) [0].buf));
	
    return 1;
}

int do_load_dir(dir_list *list, sortfn *sort, int reverse)
{
    DIR           *dirp;
    struct dirent *dp;
    file_entry    tmp_entry;
    int           next_free = 0;

    dirp = opendir (".");
    if (!dirp){
	return set_zero_dir (list);
    }
    for (dp = readdir (dirp); dp; dp = readdir (dirp)){
	if (!show_dot_files){
	    if (dp->d_name [0] == '.'){
		if (!(dp->d_name [1] == 0))
		    if (!(dp->d_name [1] == '.' && dp->d_namlen == 2))
			continue;
	    }
	}
	if (!show_backups && dp->d_name [dp->d_namlen-1] == '~')
	    continue;

	/* Need to grow the *list? */
	if (next_free == list->size){
	    list->list = realloc (list->list, sizeof (file_entry) *
				  (list->size + RESIZE_STEPS));
	    if (!list->list)
		return next_free;
	    list->size += RESIZE_STEPS;
	}
	list->list [next_free].fnamelen = dp->d_namlen;
	list->list [next_free].fname = strdup (dp->d_name);
	list->list [next_free].f.marked = 0;
	lstat (dp->d_name, &((list->list) [next_free].buf));
	next_free++;
    }
    /* swap the . and .. entries */
    tmp_entry = (list->list) [1];
    list->list [1] = (list->list) [0];
    list->list [0] = tmp_entry;
    if (next_free)
	do_sort (list, sort, next_free-1, reverse);
    closedir (dirp);
    return next_free;
}

int link_isdir (file_entry *file)
{
    struct stat b;
    
    if (S_ISLNK (file->buf.st_mode)){
	stat (file->fname, &b);
	if (S_ISDIR (b.st_mode))
	    return 1;
    }
    return 0;
}
 
int if_link_is_exe (file_entry *file)
{
    struct stat b;

    if (S_ISLNK (file->buf.st_mode)){
	stat (file->fname, &b);
	return is_exe (b.st_mode);
    }
    return 1;
}

static dir_list dir_copy = { 0, 0 };

static void alloc_dir_copy (int size)
{
    if (dir_copy.size < size)
	if (dir_copy.size)
	    free (dir_copy.list);

    dir_copy.list = xmalloc (sizeof (file_entry) * size, "alloc_dir_copy");
    dir_copy.size = size;
}

int do_reload_dir (dir_list *list, sortfn *sort, int count, int rev)
{
    DIR           *dirp;
    struct dirent *dp;
    file_entry    tmp_entry;
    int           next_free = 0;
    int           i, found;

    dirp = opendir (".");
    if (!dirp)
	return set_zero_dir (list);

    alloc_dir_copy (list->size);
    for (i = 0; i < count; i++){
	dir_copy.list [i].fnamelen = list->list [i].fnamelen;
	dir_copy.list [i].fname =    list->list [i].fname;
	dir_copy.list [i].f.marked = list->list [i].f.marked;
    }

    for (dp = readdir (dirp); dp; dp = readdir (dirp)){
	if (!show_dot_files){
	    if (dp->d_name [0] == '.')
		if (!(dp->d_name [1] == 0))
		    if (!(dp->d_name [1] == '.' && dp->d_namlen == 2))
			continue;
	}
	if (!show_backups && dp->d_name [dp->d_namlen-1] == '~')
	    continue;

	/* Need to grow the *list? */
	if (next_free == list->size){
	    list->list = realloc (list->list, sizeof (file_entry) *
				  (list->size + RESIZE_STEPS));
	    if (!list->list)
		return next_free;
	    list->size += RESIZE_STEPS;
	}
	
	for (found = i = 0; i < count; i++)
	    if (dp->d_namlen == dir_copy.list [i].fnamelen
		&& !strcmp (dp->d_name, dir_copy.list [i].fname)){
		list->list [next_free].f.marked = dir_copy.list [i].f.marked;
		found = 1;
	    }
	
	if (!found)
	    list->list [next_free].f.marked = 0;
	
	list->list [next_free].fnamelen = dp->d_namlen;
	list->list [next_free].fname = strdup (dp->d_name);
	lstat (dp->d_name, &(list->list [next_free].buf));
	next_free++;
    }
    closedir (dirp);
    /* swap the . and .. entries */
    tmp_entry = list->list [1];
    list->list [1] = list->list [0];
    list->list [0] = tmp_entry;
    do_sort (list, sort, next_free-1, rev);
    clean_dir (&dir_copy, count);
    return next_free;
}
