/* View file module for the Midnight Commander
   Copyright (C) 1994 Miguel de Icaza
   Copyright (C) 1994 Janne Kukonlehto
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */
#ifdef HAVE_UNISTD_H
#include <sys/types.h>
#include <unistd.h>
#endif
#include <string.h>
#include <stdio.h>
#include <ncurses.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <sys/time.h>		/* select: timeout */
#include <unistd.h>
#include <ctype.h>		/* toupper */
#include <malloc.h>
#include <errno.h>
#include "win.h"
#include "color.h"
#include "util.h"
#include "input.h"
#include "dialog.h"
#include "file.h"
#include "mouse.h"
#include "global.h"
#include "zip.h"

#ifndef MAP_FILE
#define MAP_FILE 0
#endif

int max_dirt_limit = 10;

/* Build-in file viewer */

static WINDOW *status;
static WINDOW *fkeys;
static WINDOW *view_win;

#define get_byte(from) (data[from])
static char *data;		/* Memory area for the file to be viewed */
static int  file;		/* File descriptor */
static long last;		/* Last byte showed */
static long last_byte;		/* Last byte of file */
static long first;		/* First byte in file */
static long start_display;	/* First char displayed */
static struct stat s;		/* stat for file */
static int  start_col = 0;	/* First displayed column */
static int dirty = 0;		/* Number of skipped updates */
static int hex_mode = 0;	/* Hexadecimal mode flag */
static int bytes_per_line;	/* Number of bytes per line in hex mode */
static int quit = 0;		/* Quit flag */
static int wrap_mode = 1;	/* Wrap mode */
static int mmaping = 0;		/* Did we use mmap on the file? */
static long int gziped = 0;	/* Size of gziped file or 0 */
static int do_pop_refresh;	/* Should pop the refresh function? */
static char *search_exp;	/* The search expression */

/* Pointer to the last search command */
static void (*last_search)(char *) = 0;

static char hex_char[] = "0123456789ABCDEF";
static char rcsid [] = "$Id: view.c,v 1.15 1994/10/13 04:28:06 miguel Exp $";

#if defined(_AIX) || defined(__aix__)
#define IFAIX(x) case (x):
#else
#define IFAIX(x)
#endif

/* Case insensitive search of text in data */
static int icase_search_p (char *text, char *data, int nothing)
{
    int p = icase_search (text, data) != 0; 
    return p;
}

static void view_refresh ()
{
    touchwin (fkeys);
    wrefresh (fkeys);
    touchwin (view_win);
    wrefresh (view_win);
    touchwin (status);
    wrefresh (status);
}

static void view_status ()
{
    wmove (status, 0, 26);
    wprintw (status, "Col %d", start_col);
    wmove (status, 0, COLS-5);
    wprintw (status, "%3d%%", last_byte == last ? 100 :
	     (start_display-first)*100 / s.st_size);
    wrefresh (status);
}

/* Shows the file pointed to by *start_display on view_win */
static long display ()
{
    int col = 0;
    int row = 0;
    long from;
    int c;
    
    from = start_display;
    wclr (view_win);
    if (hex_mode){
        char hex_buff[10];   /* A temporary buffer for sprintf and mvwaddstr */
        int bytes;	     /* Number of bytes already printed on the line */
        int text_start = COLS - bytes_per_line - 1;  /* Start of text column */
	
        for (;row < LINES-2 && from < last_byte; row++){
            /* Print the hex offset */
            sprintf (hex_buff, "%05X", from - first);
            mvwaddstr (view_win, row, 0, hex_buff);
	    
            /* Hex dump starts from column seven */
            col = 7;
	    
            /* Each hex number is two digits */
            hex_buff[2] = 0;
            for (bytes = 0; bytes < bytes_per_line && from < last_byte; bytes++, from++){
                c = (unsigned char) get_byte (from);
		
                /* Print a hex number (sprintf is too slow) */
                hex_buff [0] = hex_char [(c >> 4)];
                hex_buff [1] = hex_char [c & 15];
                mvwaddstr (view_win, row, col, hex_buff);
                col += 3;
                if ((bytes & 3) == 3 && bytes + 1 < bytes_per_line){
		    
                    /* Hex numbers are printed in the groups of four */
                    /* Groups are separated by a vline */
		    
                    waddch (view_win, ' ');
                    wvline (view_win, 0, 1);
                    col += 2;
                }
                /* Print the corresponding ascii character */
                if (c > 31 && c < 128)
                    mvwaddch (view_win, row, text_start + bytes, c);
                else
                    mvwaddch (view_win, row, text_start + bytes, '.');
            }
        }
    } else {
    	for (; row < LINES-2 && from < last_byte; from++){
	    c = get_byte (from);
    	    if ((c == '\n') || (col == COLS && wrap_mode)){
       	        col = 0;
       	        row++;
		if (c == '\n')
		    continue;
       	    }
	    if (c == '\r')
		continue;
       	    if (c == '\t'){
       	        col = ((col)/8)*8 + 8;
       	        continue;
       	    }
       	    if (!(col >= COLS)){
       		if (c > 31)
       		    mvwaddch (view_win, row, col, c);
       		else
       		    mvwaddch (view_win, row, col, '.');
       	        col++;
       	    } 
        }
    }
    wrefresh (view_win);
    last = from;
    view_status ();
    return from;
}

static void toggle_hex_mode ()
{
    hex_mode = 1 - hex_mode;
    set_label_text (fkeys, 2, hex_mode ? "" : wrap_mode ? "Unwrap" : "Wrap");
    set_label_text (fkeys, 4, hex_mode ? "Ascii" : "Hex");
    set_label_text (fkeys, 6, hex_mode ? "" : "RxSrch");
    wrefresh (fkeys);
    bytes_per_line = 2 * (COLS - 7) / 9;
    bytes_per_line &= 0xfffc;
    display ();
}

static void toggle_wrap_mode ()
{
    if (hex_mode) return;
    wrap_mode = !wrap_mode;
    set_label_text (fkeys, 2, wrap_mode ? "Unwrap" : "Wrap");
    wrefresh (fkeys);
    display ();
}

static void search (char *text, int (*search)(char *, char *, int))
{
    char *s;
    long p;

    if (verbose)
	message (D_INSERT, " Search ", "Searching %s", text);
    for (p = start_display; p < last_byte; ){
	for (; p < last_byte; p++)
	    if (get_byte (p) == '\n'){
		p++;
		break;
	    }
	s = extract_line (&get_byte (p), &get_byte (last_byte));
	if ((*search) (text, s, match_normal)){
	    start_display = p;
	    break;
	}
    }
    if (verbose)
	destroy_dialog ();
    if (p >= last_byte)
	message (0, " Search ", " Search string not found ");
}

/* Search buffer (it's size is len) in the complete buffer */
/* returns the position where the block was found */
static long block_search (char *buffer, int len)
{
    char *d = buffer;
    long e = start_display;

    if (!len)
	return 0;
    
    for (; e < last_byte; e++){
	if (*d == get_byte (e))
	    d++;
	else
	    d = buffer;
	if (d - buffer == len)
	    return e - len;
    }
    return 0;
}

/* States of our funny recognizer */
enum {normal, inside_quotes, zero, hex1, hex2, oct1};

/* This routine doesn't report all the user mistakes, it just ignores them */
static void hex_search (char *text)
{
    char buffer [120];		/* Where we hold the information */
    int  i, block_len;
    int  v = 0;
    long pos;			/* Where did we found the string */
    char *p;			/* Temporary */
    int  state = normal;	/* Initial state of the micro-scanner */
    
    /* First convert the string to a stream of bytes */
    for (i = block_len = 0; text [i] && block_len < sizeof (buffer); i++){
	switch (state){
	case inside_quotes:
	    if (text [i] == '"')
		state = normal;
	    else
		buffer [block_len++] = text [i];
	    break;

	case normal:
	    if (text [i] == '"'){
		state = inside_quotes;
		break;
	    }
	    if (text [i] == '0'){
		state = zero;
		break;
	    }
	    if (text [i] == 'x'){
		state = hex1;
		break;
	    }
	    break;

	case zero:
	    if (text [i] == 'x')
		state = hex1;
	    break;

	case hex1:
	    v = 0;
	    text [i] = toupper (text [i]);
	    if ((p = strchr (hex_char, text [i])) != 0){
		v = (p - hex_char) << 4;
		state = hex2;
	    }
	    break;

	case hex2:
	    if ((p = strchr (hex_char, text [i])) != 0){
		v |= (p - hex_char);
		state = normal;
	    }
	    buffer [block_len++] = v;
	    break;
	}
    }
    /* Then start the search */
    pos = block_search (buffer, block_len);
    if (!pos){
	message (0, " Search ", " Search string not found ");
	return;
    }
    
    /* Adjust the file offset */
    start_display = (pos & (~(bytes_per_line-1)));
}

static void do_regexp_search (char *regexp)
{
    search_exp = regexp;
    search (regexp, regexp_match);
    touchwin (view_win);
    display ();
}

static void regexp_search ()
{
    char *regexp = "";
    static char *old = 0;

    if (hex_mode) return;
    regexp = old ? old : regexp;
    regexp = input_dialog (" Search ", "Enter regexp:", regexp);
    if ((!regexp) || (!*regexp)){
	regexp = "";
	return;
    }
    if (old) free (old);
    old = strdup (regexp);
    do_regexp_search (regexp);
    last_search = do_regexp_search;
}

static void do_normal_search (char *text)
{
    search_exp = text;
    if (hex_mode)
	hex_search (text);
    else 
	search (text, icase_search_p);
    touchwin (view_win);
    display ();
}

static void normal_search ()
{
    static char *old;
    char *exp = "";
    
    exp = old ? old : exp;
    exp = input_dialog (" Search ", "Enter search string:", exp);
    if ((!exp) || (!*exp)){
	exp = "";
	return;
    }
    if (old) free (old);
    old = strdup (exp);
    do_normal_search (exp);
    last_search = do_normal_search;
}

static void help_cmd ()
{
    interactive_display (LIBDIR "mc.hlp", "[Internal file viewer]");
    view_refresh ();
}

inline static int quit_cmd ()
{
    return quit = 1;
}
    
/* returns the new current pointer */
static long move_forward2 (long current, int lines)
{
    long p;
    int  line;
    int  col = 0;

    if (last == last_byte)
	return current;
    
    if (hex_mode){
        int i;
        for (i = 0, p = current; i < lines * bytes_per_line; p++, i++)
            if (p >= last_byte)
		return current;
        return p;
    } else {
        for (line = col = 0, p = current; p < last_byte; p++){
	    if (line >= lines)
	        return p;
	    if (wrap_mode){
		col++;
		if (col == COLS){
		    col = 0;
		    line++;
		}
	    }
	    if (get_byte (p) == '\n'){
	        line++;
		col = 0;
	    }
        }
    }
    return current;
}

/* returns the new current pointer */
static long move_backward2 (long current, int lines)
{
    long p;
    int line, col;

    if (current == first)
	return current;
    if (hex_mode){
        int i;
        for (i=0,p=current; p > first && i < lines * bytes_per_line; p--, i++)
	    ;
    } else {
        for (p = current-1; p > first; p--)
	    if (get_byte (p) == '\n'){
	        p--;
	        break;
	    }
        /* p points to the char before the new line */
        if (p <= first)
	    return current;
    
        for (line = col = 0; p > first; p--){
	    if (get_byte (p) == '\n')
	        line++;
	    if (line == lines)
	        return p+1;
        }
    }
    return p;
}

static void move_backward (int i)
{
    start_display = move_backward2 (start_display, i);
    dirty++;
}

static void move_forward (int i)
{
    start_display = move_forward2 (start_display, i);
    dirty++;
}

static void one_line_up ()
{
    move_backward (1);
}

static void one_line_down ()
{
    move_forward (1);
}

static void move_to_top (int dummy)
{
    start_display = first;
    display ();
    dirty = 0;
}

static void move_to_bottom (int dummy)
{
    start_display = move_backward2 (last_byte, LINES-3);
    display ();
    dirty = 0;
}

static void init_view ()
{
#ifdef BUGGY_CURSES
    /* See key.c: mi_getch /BUGGY_CURSES/ */
    touchwin (stdscr);
#endif
    status = newwin (1, COLS, 0, 0);
    view_win = newwin (LINES-2, COLS, 1, 0);
    wattron (view_win, NORMAL_COLOR);
    wattron (status, SELECTED_COLOR);
    wclr (status);

    fkeys  = push_fkey ();

    define_label (fkeys, 1, "Help", help_cmd);
    define_label (fkeys, 2, "Unwrap", toggle_wrap_mode);
    define_label (fkeys, 3, "", 0);
    define_label (fkeys, 4, "Hex", toggle_hex_mode);
    define_label (fkeys, 5, "", 0);
    define_label (fkeys, 6, "RxSrch", regexp_search);
    define_label (fkeys, 7, "Search", normal_search);
#if defined(_AIX) || defined(__aix__)
    define_label (fkeys, 8, "Top", move_to_top);
    define_label (fkeys, 9, "Bottom", move_to_bottom);
#else
    define_label (fkeys, 8, "", 0);
    define_label (fkeys, 9, "", 0);
#endif
    define_label_quit (fkeys, 10, "Quit", quit_cmd);
    
    /* workaround ncurses 1.8.5 bug */
    wmove (fkeys, 0, 0);
    waddch (fkeys, '1');
    clearok (view_win, TRUE);
    wrefresh (fkeys);

    /* At least, we can set the update function */
    push_refresh (view_refresh);
    do_pop_refresh = 1;
}

/* Load filename into core */
/* returns 1 if used mmap, 0 if used malloc */
static int load_view_file (char *filename)
{
    char *gzip_cmd = "gzip -dc ";
    char *cmd;
    FILE *gzip;
    int  n;
#ifndef HAVE_MMAP
    int count;
#endif
    
    if ((file = open (filename, O_RDONLY)) < 0){
	message (1, " Error ", " Can't open file \"%s\" \n %s ",
		 filename, unix_error_string (errno));
	return -1;
    }
    if (fstat (file, &s) < 0){
	message (1, " Error ", " Can't stat file \n %s ",
		 unix_error_string (errno));
	close (file);
	return -1;
    }
    if (S_ISDIR (s.st_mode) || S_ISSOCK (s.st_mode) || S_ISFIFO (s.st_mode)){
	message (1, " Error ", " Can't view: not a regular file ");
	close (file);
	return -1;
    }
    if (s.st_size < 1) {
	message(1, " Error ", " Can't view empty file ");
	close(file);
	return -1;
    }

    /* First, try to open a compressed file */
    if ((gziped = is_gunzipable (file)) != 0){
	close (file);
	s.st_size = gziped;

	cmd = copy_strings (gzip_cmd, filename, 0);

	if ((data = (char *) malloc (gziped)) == NULL){
	    free (cmd);
	    message (1, " Error ", " Not enought memory for unziped file ");
	    return -1;
	}
	first = 0;

	open_error_pipe ();
	if ((gzip = popen (cmd, "r")) == NULL){
	    gc_free (cmd);
	    free (data);

	    close_error_pipe (1, " Can't spawn child gzip program ");
	    return -1;
	}
	gc_free (cmd);

	/* First, check is gzip produced any output */
	n = getc (gzip);
	if (n == EOF){
	    free (data);
	    pclose (gzip);
	    close_error_pipe (1, " Empty output from child gzip ");
	    return -1;
	}

	/* Yes, load the rest of the file */
	get_byte (first) = n;

	/* Now, read the rest of the file */
	n = fread (&get_byte (first+1), 1, gziped-1, gzip);
	pclose (gzip);
	close_error_pipe (0, NULL);
	if (n < gziped-1 && n >= 0){
	    /* The uncompressed size was shorter than the guess */
	    /* Correct the situation */
	    s.st_size = n + 1;
	    data = realloc (data, n + 1);
	}
	return 0;
    } else
	gziped = 0;
    
#ifdef HAVE_MMAP
    data = mmap (0, s.st_size, PROT_READ, MAP_FILE | MAP_SHARED, file, 0);
    if (data == (caddr_t) -1){
	close (file);
        message (1, " Error ", " Can't mmap file \n %s ",
		 unix_error_string (errno));
	return -1;
    }
    first = 0;
    return 1;
#else
    if ((data = malloc (s.st_size)) == NULL){
	message (1, " Error ", " Machine without mmap: no free memory to load file in RAM ");
	return -1;
    }
    first = 0;
    lseek (file, 0, SEEK_SET);
    count = read (file, &get_byte (first), s.st_size);
    if (count != s.st_size){
	free (first);
	close (file);
	message (1, " Error ", " Can't read file \n %s ",
		 unix_error_string (errno));
	return -1;
    }
    close (file);
    return 0;
#endif
}

void free_file ()
{
#ifdef HAVE_MMAP
    if (mmaping){
	munmap (data, s.st_size);
	close (file);
    } else
#endif /* HAVE_MMAP */
	free (data);
}

static void do_view (char *filename)
{
    int c;
    int dirt_limit = 1;	/* Force update after dirt_limit skipped ones */
    
    quit = 0;
    hex_mode = 0;
    dirty = 0;
    if ((mmaping = load_view_file (filename)) == -1)
	return;
    
    push_frame (0, 0, 0);
    push_event (1, 1, COLS, LINES/2, click_may_break_loop, one_line_up);
    push_event (1, LINES/2, COLS, LINES-1, click_may_break_loop, one_line_down);
    init_view ();
    
    wprintw (status, "File: %-20s", name_trunc (filename, 20));
    wmove   (status, 0, 42);
    wprintw (status, "%s bytes", size_trunc (s.st_size));
	     
    start_display = first;
    last_byte = first + s.st_size;
    
    last_search = 0;		/* Start a new search */
    view_refresh ();
    display ();
    do {
        if (dirty > dirt_limit){
            /* Too many updates skipped -> force a update */
            display ();
            dirty = 0;
            /* Raise the update skipping limit */
            dirt_limit++;
            if (dirt_limit > max_dirt_limit) dirt_limit = max_dirt_limit;
        }
        if (dirty){
            /* Check for incoming key presses       *
             * If there are any we skip this update */
	    
            struct fd_set select_set;
            struct timeval timeout;
            FD_ZERO (&select_set);
            FD_SET (0, &select_set);
            timeout.tv_sec = 0;
            timeout.tv_usec = 0;
            select (FD_SETSIZE, &select_set, 0, 0, &timeout);
            if (FD_ISSET (0, &select_set)){
                /* Incoming key press -> skipping full update,
		   only the status line is updated */
                view_status ();
            } else {
                /* No incoming key presses -> we have time to
		   update the screen properly */
                display ();
                dirty = 0;
                dirt_limit = 1;
            }
        }
	c = mi_getch ();
	if (check_fkeys (c))
	    /* Nothing */;
	else if (check_movement_keys (c, 1, LINES-2, move_backward, move_forward,
				      move_to_top, move_to_bottom))
	    /* Nothing */;
        else switch (c){

	case '/':
	    if (dirty){
	        display ();
	        dirty = 0;
	    }
	    regexp_search ();
	    break;

				/* Continue search */
	case XCTRL('s'):
	case 'n':
	    if (last_search){
		(*last_search)(search_exp);
		break;
	    }
	    /* if not... then ask for an expression */

	case XCTRL('l'):
	    if (dirty){
	        display ();
	        dirty = 0;
	    }
	    view_refresh ();
	    break;
	    
	case ESC_CHAR:
	    quit = 1;
	}
    } while (!quit);
    free_file ();
    pop_fkey (fkeys);
    pop_frame ();
}

void view (char *filename)
{
    do_pop_refresh = 0;
    do_view (filename);
    if (do_pop_refresh)
	pop_refresh ();
}
