/*
 * manpath.c
 *
 * Copyright (c) 1990, 1991, John W. Eaton.
 *
 * You may distribute under the terms of the GNU General Public
 * License as specified in the file COPYING that comes with the man
 * distribution.  
 *
 * John W. Eaton
 * jwe@che.utexas.edu
 * Department of Chemical Engineering
 * The University of Texas at Austin
 * Austin, Texas  78712
 *
 * Changed PATH->manpath algorithm
 * Added: an empty string in MANPATH denotes the system path
 * Added: use LANG to search in /usr/man/<locale>
 * Lots of other minor things, including spoiling the indentation.
 * aeb - 940315
 */

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "gripes.h"
#include "man.h"

#ifdef STDC_HEADERS
#include <string.h>
#include <stdlib.h>
#else
extern int strcmp ();
extern int strncmp ();
extern char *memcpy ();
extern char *getenv();
extern void free ();
#endif

extern char *index (), *rindex ();	/* not always in <string.h> */
extern int is_directory ();
extern void gripe ();
extern void add_to_list ();
extern char *getval ();
extern char *my_malloc ();
extern char *get_expander ();
extern int debug, fsstnd, do_compress;
extern int alt_system;
extern char *alt_system_name;

extern struct dirs cfdirlist;
extern struct dir srchlist;	/* linked list, 1st entry unused */

/*
 * Add a directory to the manpath list if it isn't already there.
 */
void
add_to_srchlist (dir, lang, perrs)
     char *dir;
     char *lang;
     int perrs;
{
    if (alt_system) {
	add_to_list(dir, alt_system_name, perrs);
    } else {
	add_to_list(dir, lang, perrs); /* first try with $LANG */
	add_to_list(dir, (char *) 0, perrs); /* then without */
    }
}

void
add_to_list (dir, lang, perrs)
     char *dir;
     char *lang;
     int perrs;
{
  int status;
  struct dir *dp;

  if (!lang)
    lang = "";

  /* only add absolute paths */
  if (*dir != '/') {
      char cwd[BUFSIZ];

      if (!getcwd(cwd, sizeof(cwd)))
	return; /* cwd not readable, or pathname very long */
      if (strlen(dir) + strlen(lang) + strlen(cwd) + 3 > sizeof(cwd))
	return;
      if (!strncmp (dir, "./", 2))
	dir += 2;
      if (!strncmp (dir, "../", 3)) {
	  dir += 3;
	  *rindex (cwd, '/') = 0;
      }
      strcat (cwd, "/");
      strcat (cwd, dir);
      if (*lang) {
	  strcat (cwd, "/");
	  strcat (cwd, lang);
      }
      dir = cwd;
  } else if (*lang) {
      char cwd[BUFSIZ];

      strcpy (cwd, dir);
      strcat (cwd, "/");
      strcat (cwd, lang);
      dir = cwd;
  }

  dp = &srchlist;
  while (dp->nxt) {
      dp = dp->nxt;
      if (!strcmp (dp->dir, dir)) {
	  if (debug)
	    gripe (ALREADY_IN_MANPATH, dir);
	  return;
      }
  }

  /*
   * Not found -- add it.
   */
  status = is_directory(dir);

  if (status < 0 && perrs) {
      gripe (CANNOT_STAT, dir);
  }
  else if (status == 0 && perrs) {
      gripe (IS_NO_DIR, dir);
  }
  else if (status == 1) {
      if (debug)
	gripe (ADDING_TO_MANPATH, dir);

      dp->nxt = (struct dir *) my_malloc(sizeof(struct dir));
      dp = dp->nxt;
      dp->nxt = 0;
      dp->dir = strdup (dir);
  }
}

void
to_srchlist(s, perrs)
     char *s;
     int perrs;
{
  void get_manpath ();
  char *lang, *path;

  lang = getenv("LANG");
  if (*s) {
      add_to_srchlist (s, lang, perrs);
      return;
  }

  /* empty string: insert default path */
  path = getenv ("PATH");
  get_manpath (path, lang, perrs);
}

/*
 * Input: a non-NULL string, with : as separator
 * For each entry in the string, call dirfn.
 */
void
split (pathstring, dirfn, perrs)
     char *pathstring;
     void (*dirfn)();
     int perrs;
{
    register char *p, *q, *r;

    p = strdup(pathstring);
    for (q = p; ; ) {
	r = index(q, ':');
	if (r) {
	    *r = 0;
	    dirfn (q, perrs);
	    q = r+1;
	} else {
	    dirfn (q, perrs);
	    break;
	}
    }
    free (p);
}
	
extern char *opt_manpath;

void
manpath ()
{
  char *manp;
  static int done = 0;

  if (done)
    return;

  if ((manp = opt_manpath) == NULL && (manp = getenv ("MANPATH")) == NULL)
    manp = "";

  split (manp, to_srchlist, 0);
  done = 1;
}

void
prmanpath ()
{
  register struct dir *dp, *dp0;
    
  dp0 = dp = srchlist.nxt;
  while (dp) {
      if (dp != dp0)
	printf(":");
      printf("%s", dp->dir);
      dp = dp->nxt;
  }
  printf("\n");
}

/*
 * For each directory in the user's path, see if it is one of the
 * directories listed in the man.config file.  If so, and it is
 * not already in the manpath, add it.  If the directory is not listed
 * in the manpath.config file, see if there is a subdirectory `man' or
 * `MAN'.  If so, and it is not already in the manpath, add it.
 *
 * Example:  user has <dir>/bin in his path and the directory
 * <dir>/bin/man exists -- the directory <dir>/bin/man will be added
 * to the manpath.
 * Try also <dir>/man ?and <dir>?, and, if LANG is set, <dir>/$LANG/man.
 * aeb - 940320
 */
void
get_manpath (path, lang, perrs)
     char *path, *lang;
     int perrs;
{
  register char *t;
  register char *p;
  register char *end;
  register struct dirs *dlp;
  void add_to_srchlist ();
  char *find_man_subdir ();

  if (path) {
    for (p = strdup(path); ; p = end+1) {
      if ((end = strchr(p, ':')) != NULL)
	*end = '\0';
	  
      if (debug)
	gripe (PATH_DIR, p);
	  
      /*
       * The directory we're working on is in the config file.
       * If we haven't added it to the list yet, do.
       */
      if (*p)
	for (dlp = cfdirlist.nxt; dlp; dlp = dlp->nxt)
	  if (!strcmp (p, dlp->bindir)) {
	      if (debug)
		gripe (IS_IN_CONFIG);
		  
	      add_to_srchlist (dlp->mandir, lang, perrs);
	      goto found;
	  }
	  
      /*
       * The directory we're working on isn't in the config file.  See if it
       * has man or MAN subdirectories.  If so, and this subdirectory hasn't
       * been added to the list, do. (Try also a few other places nearby.)
       */
      if (debug)
	gripe (IS_NOT_IN_CONFIG);
	  
      t = find_man_subdir (p);
      if (t != NULL) {
	  if (debug)
	    gripe (MAN_NEARBY);
	      
	  add_to_srchlist (t, lang, perrs);
	  free (t);
      } else {
	  if (debug)
	    gripe (NO_MAN_NEARBY);
      }
	  
    found:
	  
      if (!end)
	break;
    }
  }

  if (debug)
    gripe (ADDING_MANDIRS);

  for (dlp = cfdirlist.nxt; dlp; dlp = dlp->nxt)
    if (dlp->mandatory)
	add_to_srchlist (dlp->mandir, lang, perrs);

}

/*
 * Check to see if the current directory has man or MAN
 * or ../man or ../man1 or ../man8 subdirectories. 
 */
char *
find_man_subdir (p)
     register char *p;
{
  int len;
  register char *t, *sp;

  len = strlen (p);

  t = my_malloc ((unsigned) len + 20);

  memcpy (t, p, len);
  strcpy (t + len, "/man");
  
  if (is_directory (t) == 1)
    return t;

  strcpy (t + len, "/MAN");
  
  if (is_directory (t) == 1)
    return t;

  /* find parent directory */
  t[len] = 0;
  if ((sp = rindex (t, '/')) != NULL) {
      *sp = 0;
      len = sp - t;
  } else {
      strcpy (t + len, "/..");
      len += 3;
  }

  /* look for the situation with packagedir/bin and packagedir/man */
  strcpy (t + len, "/man");

  if (is_directory (t) == 1)
    return t;

  /* look for the situation with pkg/bin and pkg/man1 or pkg/man8 */
  /* (looking for all man[1-9] would probably be a waste of stats) */
  strcpy (t + len, "/man1");

  if (is_directory (t) == 1) {
      t[len] = 0;
      return t;
  }

  strcpy (t + len, "/man8");

  if (is_directory (t) == 1) {
      t[len] = 0;
      return t;
  }

  free (t);
  return NULL;
}

/*
 * Change a name of the form .../man1/name.1[.Z] to .../cat1/name.1[.gz]
 * or (FSSTND) change /usr/.../man/.../man1/name.1 into
 *                     /var/catman/.../cat1/name.1
 *
 * Returns 0 on failure.
 */
char *
convert_to_cat (name0,ext,cat)
     char *name0;
     int  ext, cat;
{
  char *name, *cat_name;
  register char *t0, *t1, *t2, *t3, *t4;
  register struct dirs *dlp;
  int len;

  name = strdup (name0);

  t0 = rindex (name, '.');
  if (t0 && get_expander(t0)) /* this is a compressee extension - remove it */
    *t0 = 0;

  t1 = rindex (name, '/');
  if (t1 == NULL)
    return 0;
  *t1 = 0;			/* remove page.1 part */

  t2 = rindex (name, '/');
  if (t2 == NULL)
    return 0;
  *t1 = '/';
  *t2 = 0;			/* remove man1 part */

  if (strncmp(t2+1, "man", 3) != 0)
    return 0;
  t2[1] = 'c';
  t2[3] = 't';

  len = (do_compress ? strlen (getval("COMPRESS_EXT")) : 0);

  /* Explicitly given cat file? */
  for (dlp = cfdirlist.nxt; dlp; dlp = dlp->nxt)
    if (!strcmp (name, dlp->mandir)) {
	if (!dlp->catdir[0])
	  break;
	*t2 = '/';
	len += strlen (dlp->catdir) + strlen (t2) + 1;
	cat_name = (char *) my_malloc (len);
	strcpy (cat_name, dlp->catdir);
	strcat (cat_name, t2);
	goto gotit;
    }

  if (fsstnd && !strncmp(name, "/usr/", 5)) {
      /* search, starting at the end, for a part `man' to delete */
      t3 = t2;
      while ((t4 = rindex(name, '/')) != NULL && strcmp(t4, "/man")) {
	  *t3 = '/';
	  t3 = t4;
	  *t3 = 0;
      }
      *t3 = '/';
      if (t4) {
	  *t4 = 0;
	  len += strlen("/var/catman") + strlen (name+4) + strlen (t3) + 1;
	  cat_name = (char *) my_malloc (len);
	  strcpy (cat_name, "/var/catman");
	  strcat (cat_name, name+4);
	  strcat (cat_name, t3);
	  goto gotit;
      }
  } else
    *t2 = '/';

  cat = CAT;

  if (do_compress) {
      len += strlen (name) + 1;
      cat_name = (char *) my_malloc (len);
      strcpy (cat_name, name);
  } else
    cat_name = name;

gotit:

  if (cat == SCAT)
    return NULL;

  if (ext && do_compress)
    strcat (cat_name, getval("COMPRESS_EXT"));

  if (debug)
    gripe (CATNAME_IS, cat_name);

  if (name != cat_name)
    free (name);

  return cat_name;
}
