#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <linux/module.h>
#include <linux/unistd.h>

/*
 * Original author: Jon Tombs <jon@gtex02.us.es>,
 * and extended by Bjorn Ekwall <bj0rn@blox.se> in 1994 (C).
 * See the file COPYING for your rights (GNU GPL)
 */
/*
 * This is here as syscall.h and sys/syscall.h redefine the defines in
 * unistd.h why doesn't unistd #include them?
 */

#define WANT_TO_REMOVE 1
#define CAN_REMOVE 2

struct ref {
	int modnum;
	struct ref *next;
};

struct mod {
	int status; /* or of: WANT_TO_REMOVE, CAN_REMOVE */
	char name[MOD_MAX_NAME];
	struct ref *ref;
};

struct mod *loaded;
int current = -1;

extern int syscall(int, ...);

static int
delete_module(const char *name) {
  return syscall( __NR_delete_module, name);
}

void *
ckalloc(size_t nbytes)
{
	void *p;

	if ((p = malloc(nbytes)) == NULL) {
		fputs("insmod:  malloc failed\n", stderr);
		exit(2);
	}
	return p;
}

/* build the references as shown in /proc/ksyms */
void
get_stacks()
{
	FILE *fp;
	struct ref *rp;
	int i;
	char line[200]; /* or whatever... */
	char *p;
	char *r;

	if ((fp = fopen("/proc/modules", "r")) == (FILE *)0) {
		perror("/proc/modules");
		exit(1);
	}

	while (fgets(line, 200, fp)) {
		if (loaded) {
			++current;
			loaded = (struct mod *)realloc(loaded,
				(current + 1) * sizeof(struct mod));
		} else {
			current = 0;
			loaded = (struct mod *)ckalloc(sizeof(struct mod));
		}

		loaded[current].status = 0;
		loaded[current].ref = (struct ref *)0;

		/* save the module name */
		p = strchr(line, ' ');
		*p = '\0';
		strcpy(loaded[current].name, line);

		/* any references? */
		if ((p = strchr(p + 1, '['))) {
			*(strrchr(p + 1, ']')) = '\0';
			do {
				r = p + 1;
				if ((p = strchr(r, ' ')))
					*p = '\0';
				for (i = 0; i < current; ++i) {
					if (strcmp(loaded[i].name, r) == 0)
						break;
				}

				if (i == current) { /* not found! */
					fprintf(stderr,
					"Strange reference in "
					"/proc/modules: '%s' used by '%s'?\n",
					loaded[current].name, r);
					exit(1);
				}

				rp = (struct ref *)ckalloc(sizeof(struct ref));
				rp->modnum = i;
				rp->next = loaded[current].ref;
				loaded[current].ref = rp;
			} while (p);
		}
	}

	fclose(fp);
}

int
main(int argc, char **argv)
{
	struct ref *rp;
	int i, j;
	int count;
	char **list;

	if (argc == 1) {
		fprintf(stderr, "usage: rmmod [-r] module ...\n");
		exit(1);
	}
	/* else */

	if (strcmp(argv[1], "-r") != 0) {
		while (argc > 1) {
			if (delete_module(argv[1]) < 0) {
				perror(argv[1]);
			}
			--argc;
			++argv;
		}
		exit(0);
	}
	/* else recursive removal */

	count = argc - 2;
	list = &(argv[2]);
	if (count <= 0) {
		fprintf(stderr, "usage: rmmod [-r] module ...\n");
		exit(1);
	}

	get_stacks();

	for (i = 0; i < count; ++i) {
		for (j = 0; j <= current; ++j) {
			if (strcmp(loaded[j].name, list[i]) == 0) {
				loaded[j].status = WANT_TO_REMOVE;
				break;
			}
		}
		if (j > current)
			fprintf(stderr, "module '%s' not loaded\n", list[i]);
	}

	for (i = 0; i <= current; ++i) {
		if (loaded[i].ref || (loaded[i].status == WANT_TO_REMOVE))
			loaded[i].status |= CAN_REMOVE;

		for (rp = loaded[i].ref; rp; rp = rp->next) {
			switch (loaded[rp->modnum].status) {
			case CAN_REMOVE:
			case WANT_TO_REMOVE | CAN_REMOVE:
				break;

			case WANT_TO_REMOVE:
				if (loaded[rp->modnum].ref == (struct ref *)0)
					break;
				/* else fallthtough */
			default:
				loaded[i].status &= ~CAN_REMOVE;
				break;
			}
		}

		switch (loaded[i].status) {
		case CAN_REMOVE:
		case WANT_TO_REMOVE | CAN_REMOVE:
			if (delete_module(loaded[i].name) < 0) {
				perror(loaded[i].name);
			}
			break;

		case WANT_TO_REMOVE:
			fprintf(stderr, "module '%s' is in use!\n",
				loaded[i].name);
			break;
		}
	}

	return 0;
}
