/* Widgets for the Midnight Commander
   Copyright (C) 1994 Radek Doulik, Miguel de Icaza

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */
static char rcsid [] = "$Id: widget.c,v 1.2 1995/01/27 02:36:51 miguel Exp $";
#include <string.h>
#include <stdio.h>
#include <malloc.h>
#include <ncurses.h>
#include "mad.h"
#include "global.h"
#include "win.h"
#include "color.h"
#include "util.h"
#include "input.h"
#include "mouse.h"
#include "dlg.h"
#include "widget.h"

static int button_event (Gpm_Event *event, WButton *b);

static int button_callback (Dlg_head *h, int Msg, int Par)
{
    WButton *b = (WButton *) h->current->widget;
    int stop = 0;

    switch (Msg) {
    case WIDGET_INIT:
	push_event (b->widget.x, b->widget.y,
		    b->widget.x + strlen (b->text), b->widget.y,
		    (mouse_h) button_event, h->current->widget,
		    event_use_frame);
	break;
	
    case WIDGET_HOTKEY:
        if (b->hotkey == Par ||
	    (b->hotkey >= 'a' && b->hotkey <= 'z' && b->hotkey-32 == Par)){
	    button_callback (h, WIDGET_KEY, ' ');        /* to make action */
	    return 1;
	} else
	    return 0;
	
    case WIDGET_KEY:
	if (Par != ' ' && Par != '\n')
	    break;

	if (b->callback)
	    stop = (*b->callback)(b->action, b->callback_data);
	if (!b->callback || stop){
	    h->running = 0;
	    h->ret_value = b->action;
	}
	return 1;

    case WIDGET_UNFOCUS:
    case WIDGET_FOCUS:
    case WIDGET_DRAW:
	wattrset (h->window, (Msg==WIDGET_FOCUS) ? FOCUSC : NORMALC);
	mvwprintw (h->window, b->widget.y, b->widget.x, b->text);

	if (b->hotpos >= 0){
	    wattrset (h->window, (Msg ==  WIDGET_FOCUS) ? HOT_FOCUSC : HOT_NORMALC);
	    mvwaddch (h->window, b->widget.y, b->widget.x + b->hotpos,
		      b->text [b->hotpos]);
	    wmove (h->window, b->widget.y, b->widget.x + b->hotpos);
        }
	return 1;	                                      
    }	
    return default_proc (h, Msg, Par);
}

static int button_event (Gpm_Event *event, WButton *b)
{
    if (event->type & (GPM_DOWN|GPM_UP)){
    	Dlg_head *h=b->widget.parent;
	dlg_select_widget (h, b);
	if (event->type & GPM_UP){
	    button_callback (h, WIDGET_KEY, ' ');
	    (*h->callback) (h, ' ', DLG_POST_KEY);
	    wrefresh (h->window);
	    return MOU_ENDLOOP;
	}
    }
    return MOU_NORMAL;
}

static void button_destroy (WButton *b)
{
    free (b->text);
}

WButton *button_new (int y, int x, int action, char *text, int hkey, int hpos,
		     int (*callback)(int, void *), void *callback_data)
{
    WButton *b = xmalloc (sizeof (WButton), "new_button");

    if (hkey>='A' && hkey<='Z')
	hkey |= 0x20;

    init_widget (&b->widget, y, x, button_callback,(destroy_fn)button_destroy);
    b->action = action;
    b->selected = 0;
    b->text   = strdup (text);
    b->hotkey = hkey;
    b->hotpos = hpos;
    b->callback = callback;
    b->callback_data = callback_data;

    return b;
}

void button_set_text (WButton *b, char *text)
{
    free (b->text);
    b->text = strdup (text);
    button_callback (b->widget.parent, WIDGET_DRAW, 0);
#ifdef BUGGY_CURSES
    touchwin (stdscr);
#endif
    wrefresh (b->widget.parent->window);
}

/* Radio button widget */
static int radio_event (Gpm_Event *event, WRadio *r);

static int radio_callback (Dlg_head *h, int Msg, int Par)
{
    int i, max, m;
    /* static int selected_color; Never used */
    WRadio *r = (WRadio *) h->current->widget;

    switch (Msg) {
    case WIDGET_INIT:

	/* Compute the longest string */
	max = 0;
	for (i = 0; i < r->count; i++){
	    m = strlen (r->texts [i]);
	    if (m > max)
		max = m;
	}

	push_event (r->widget.x, r->widget.y,
		    r->widget.x + max, r->widget.y+r->count-1,
		    (mouse_h) radio_event, h->current->widget,
		    event_use_frame);
	break;
	
    case WIDGET_HOTKEY:
	if (r->upper_letter_is_hotkey){
	    int  i;
	    char *s;
	    
	    for (i = 0; i < r->count; i++){
		for (s = r->texts [i]; *s; s++){
		    if (!(*s >= 'A' && *s <= 'Z' && (0x20|*s) == (0x20|Par)))
			continue;
		    r->pos = i;
		    radio_callback (h, WIDGET_KEY, ' '); /* Take action */
		    return 1;
		}
	    }
	}
	return 0;
	
    case WIDGET_KEY:
	switch (Par){
	case ' ':
	    r->sel = r->pos;
    	    (*h->callback) (h, h->current->dlg_id, DLG_ACTION);
	    return 1;

	case KEY_UP:
	case KEY_LEFT:
	    if (r->pos > 0){
		r->pos--;
		return 1;
	    }
	    return 0;
	    
	case KEY_DOWN:
	case KEY_RIGHT:
	    if (r->count - 1 > r->pos) {
		r->pos++;
		return 1;
	    }
	}
	return 0;
	
    case WIDGET_UNFOCUS:
    case WIDGET_FOCUS:
    case WIDGET_DRAW:
	for (i = 0; i < r->count; i++){
	    wattrset (h->window, (i==r->pos && Msg==WIDGET_FOCUS) ? FOCUSC :NORMALC);
	    mvwprintw (h->window, r->widget.y + i, r->widget.x, "(%c) %s",
		       (r->sel == i) ? '*' : ' ', r->texts[i]);
	}
	wmove (h->window, r->widget.y + r->pos, r->widget.x+1);
	return 1;
	break;
    }
    return default_proc (h, Msg, Par);
}

static int radio_event (Gpm_Event *event, WRadio *r)
{
    if (event->type & (GPM_DOWN|GPM_UP)){
    	Dlg_head *h=r->widget.parent;
	r->pos = event->y;
	dlg_select_widget (h, r);
	if (event->type & GPM_UP){
	    radio_callback (h, WIDGET_KEY, ' ');
	    radio_callback (h, WIDGET_FOCUS, 0);
	    (*h->callback) (h, ' ', DLG_POST_KEY);
	    wrefresh (h->window);
	    return MOU_ENDLOOP;
	}
    }
    return MOU_NORMAL;
}

WRadio *radio_new (int y, int x, int count, char **texts, int use_hotkey)
{
    WRadio *r = xmalloc (sizeof (WRadio), "radio_new");

    init_widget (&r->widget, y, x, radio_callback, 0);
    r->state = 1;
    r->pos = 0;
    r->sel = 0;
    r->count = count;
    r->texts = texts;
    r->upper_letter_is_hotkey = use_hotkey;
    
    return r;
}


/* Checkbutton widget */

static int check_event (Gpm_Event *event, WCheck *b);

static int check_callback (Dlg_head *h, int Msg, int Par)
{
    WCheck *c = (WCheck *) h->current->widget;
    
    switch (Msg) {
    case WIDGET_INIT:
	push_event (c->widget.x, c->widget.y,
		    c->widget.x + strlen (c->text), c->widget.y,
		    (mouse_h) check_event, h->current->widget,
		    event_use_frame);
	break;

    case WIDGET_HOTKEY:
        if (c->hotkey==Par ||
	    (c->hotkey>='a' && c->hotkey<='z' && c->hotkey-32==Par)){
	    check_callback (h, WIDGET_KEY, ' ');        /* make action */
	    return 1;
	} else
	    return 0;
    case WIDGET_KEY:
	if (Par != ' ')
	    break;
	c->state ^= C_BOOL;
	c->state ^= C_CHANGE;
        (*h->callback) (h, h->current->dlg_id, DLG_ACTION);
	
	return 1;

    case WIDGET_FOCUS:
    case WIDGET_UNFOCUS:	
    case WIDGET_DRAW:
	wattrset (h->window, (Msg == WIDGET_FOCUS) ? FOCUSC : NORMALC);
	mvwprintw (h->window, c->widget.y, c->widget.x, "[%c] %s",
		   (c->state & C_BOOL) ? 'x' : ' ', c->text);

	if (c->hotpos >= 0){
	    wattrset (h->window, (Msg == WIDGET_FOCUS) ? HOT_FOCUSC : HOT_NORMALC);
	    mvwaddch (h->window, c->widget.y, c->widget.x + c->hotpos+4,
		      c->text [c->hotpos]);
	}

	wmove (h->window, c->widget.y, c->widget.x + 1);
	
	return 1;
    }
    return default_proc (h, Msg, Par);
}

static int check_event (Gpm_Event *event, WCheck *c)
{
    if (event->type & (GPM_DOWN|GPM_UP)){
    	Dlg_head *h=c->widget.parent;
	dlg_select_widget (h, c);
	if (event->type & GPM_UP){
	    check_callback (h, WIDGET_KEY, ' ');
	    check_callback (h, WIDGET_FOCUS, 0);
	    (*h->callback) (h, ' ', DLG_POST_KEY);
	    wrefresh (h->window);
	    return MOU_NORMAL;
	}
    }
    return MOU_NORMAL;
}

WCheck *check_new (int y, int x, int state, char *text, int hkey, int hpos)
{
    WCheck *c =  xmalloc (sizeof (WRadio), "check_new");

    if (hkey >= 'A' && hkey <= 'Z')
	hkey |= 0x20;

    init_widget (&c->widget, y, x, check_callback, 0);
    c->state = state ? C_BOOL : 0;
    c->text = text;
    c->hotkey = hkey;
    c->hotpos = hpos;

    return c;
}


/* Input widget */

static void input_destroy (WInput *i)
{
    destroy_input (i->in, IN_NORMAL);
}

static int input_event (Gpm_Event *event, WInput *b);

static int input_callback (Dlg_head *h, int Msg, int Par)
{
    WInput *i = (WInput *) h->current->widget;
    int    t;

    switch (Msg){
    case WIDGET_INIT:
	push_event (i->widget.x, i->widget.y,
		    i->widget.x + i->in->field_len, i->widget.y,
		    (mouse_h) input_event, h->current->widget,
		    event_use_frame);
	break;
	
    case WIDGET_KEY:
	if (Par == KEY_UP || Par == KEY_DOWN ||
	    Par == ESC_CHAR || Par == KEY_F(10) ||
	    Par == XCTRL('g'))
	    return 0;		/* We don't handle up/down */

	if (Par == '\n'){
	    dlg_one_down (h);
	    return 1;
	}
	handle_char (i->in, Par);
	i->updated = 1;
	return 1;

    case WIDGET_CHECK_HOTKEY:
	return 0;		/* Don't want to have hotkeys on */
	      
    case WIDGET_FOCUS:
    case WIDGET_UNFOCUS:	
    case WIDGET_DRAW:
	/* Very ugly hack */
	t = i->in->first;
	update_input (i->in);
	i->in->first = t;
	break;
    }
    return default_proc (h, Msg, Par);
}

static int input_event (Gpm_Event *event, WInput *i)
{
    if (event->type & (GPM_DOWN|GPM_DRAG)){
	dlg_select_widget (i->widget.parent, i);
	
	i->in->point = strlen (i->in->buffer);
	if (event->x - i->in->first_shown - 1 < i->in->point)
	    i->in->point = event->x - i->in->first_shown - 1;
	if (i->in->point < 0)
	    i->in->point = 0;
	
	update_input (i->in);
    }
    return MOU_NORMAL;
}

WInput *input_new (WINDOW *w, int y, int x, int color, int len, char *text)
{
    WInput *i = xmalloc (sizeof (WInput), "input_new");

    init_widget (&i->widget, y, x, input_callback, (destroy_fn)input_destroy);
    i->in = create_input (x, y, w, color, len, text);
    i->updated = 0;

    return i;
}



/* Listbox widget */

void listbox_draw (WINDOW *w, WListbox *l, Dlg_head *h, int focused)
{
    WLEntry *e;
    int i;
    int sel_line;
    int normalc, selc;
    char *text; 

    if (focused){
	normalc = NORMALC;
	selc    = FOCUSC;
    } else {
	normalc = NORMALC;
	selc    = HOT_FOCUSC;
    }
    sel_line = -1;

    for (e = l->top, i = 0; (i < l->height); i++){
	
	/* Display the entry */
	if (e == l->current && sel_line == -1){
	    sel_line = l->widget.y+i;
	    wattrset (w, selc);
	} else
	    wattrset (w, normalc);
	
	wmove   (w, l->widget.y + i, l->widget.x);

	if ((i > 0 && e == l->list) || !l->list)
	    text = "";
	else {
	    text = e->text;
	    e = e->next;
	}
	wprintw (w, " %-*s ", l->width-2, name_trunc (text, l->width-2));
    }
    wmove (w, sel_line, l->widget.x);
}

/* Returns the number of items between s and e,
   must be on the same linked list */
static int listbox_cdiff (WLEntry *s, WLEntry *e)
{
    int count;

    for (count = 0; s != e; count++)
	s = s->next;
    return count;
}

static WLEntry *listbox_check_hotkey (WListbox *l, int key)
{
    int i;
    WLEntry *e;
    
    i = 0;
    e = l->list;
    if (!e)
	return 0;
    
    while (1){

	/* If we didn't find anything, return */
	if (i && e == l->list)
	    return 0;

	if (e->hotkey == key)
	    return e;
	
	i++;
	e = e->next;
    }
}

/* Used only for display updating, for avoiding line at a time scroll */
void listbox_select_last (WListbox *l, int set_top)
{
    if (l->list){
	l->current = l->list->prev;
	l->pos = l->count - 1;
	if (set_top)
	    l->top = l->list->prev;
    }
}

/* Makes *e the selected entry (sets current and pos) */
void listbox_select_entry (WListbox *l, WLEntry *dest)
{
    WLEntry *e;
    int pos;
    int top_seen;
    
    top_seen = 0;
    
    /* Special case */
    for (pos = 0, e = l->list; pos < l->count; e = e->next, pos++){

	if (e == l->top)
	    top_seen = 1;
	
	if (e == dest){
	    l->current = e;
	    if (top_seen){
		while (listbox_cdiff (l->top, l->current) >= l->height)
		    l->top = l->top->next;
	    } else {
		l->top = l->current;
	    }
	    l->pos = pos;
	    return;
	}
    }
    /* If we are unable to find it, set decent values */
    l->current = l->top = l->list;
    l->pos = 0;
}

/* Selects from base the pos element */
static WLEntry *listbox_select_pos (WListbox *l, WLEntry *base, int pos)
{
    WLEntry *last = l->list->prev;

    if (base == last)
    	return last;
    while (pos--){
	base = base->next;
	if (base == last)
	    break;
    }
    return base;
}

static inline int listbox_back (WListbox *l)
{
    if (l->pos){
	listbox_select_entry (l, listbox_select_pos (l, l->list, l->pos-1));
	return 1;
    }
    return 0;
}

static inline int listbox_fwd (WListbox *l)
{
    if (l->current != l->list->prev){
	listbox_select_entry (l, listbox_select_pos (l, l->list, l->pos+1));
	return 1;
    }
    return 0;
}

/* Returns 1 if we want a redraw */
static int listbox_key (WListbox *l, int key)
{
    int i;
    int j = 0;
    
    switch (key){
    case KEY_HOME:
    case KEY_A1:
	l->current = l->top = l->list;
	l->pos = 0;
	return 1;
	
    case KEY_END:
    case KEY_C1:
	l->current = l->top = l->list->prev;
	l->pos = l->count - 1;
	return 1;
	
    case XCTRL('p'):
    case KEY_UP:
	listbox_back (l);
	return 1;
	
    case XCTRL('n'):
    case KEY_DOWN:
	listbox_fwd (l);
	return 1;

    case KEY_NPAGE:
    case XCTRL('v'):
	for (i = 0; i < l->height-1; i++)
	    j |= listbox_fwd (l);
	return j > 0;
	
    case KEY_PPAGE:
    case ALT('v'):
	for (i = 0; i < l->height-1; i++)
	    j |= listbox_back (l);
	return j > 0;
    }
    return 0;
}

static int listbox_event (Gpm_Event *event, WListbox *l);
static int listbox_callback (Dlg_head *h, int msg, int par)
{
    WListbox *l = (WListbox *) h->current->widget;
    WLEntry  *e;
    /* int selected_color; Never used */
    int ret_code;
    
    switch (msg){
    case WIDGET_INIT:
	push_event (l->widget.x, l->widget.y - 1,
		    l->widget.x + l->width, l->widget.y + l->height,
		    (mouse_h) listbox_event, h->current->widget,
		    event_use_frame);
	break;

    case WIDGET_HOTKEY:
	if ((e = listbox_check_hotkey (l, par)) != NULL){
	    listbox_select_entry (l, e);

	    /* Take the appropriate action */
	    if (l->action == listbox_finish){
		l->widget.parent->running   = 0;
		l->widget.parent->ret_value = B_ENTER;
	    } else if (l->action == listbox_cback){
		if ((*l->cback)(l) == listbox_finish){
		    l->widget.parent->running = 0;
		    l->widget.parent->ret_value = B_ENTER;
		}
	    }
	    return 1;
	} else
	    return 0;
	
    case WIDGET_KEY:
	if ((ret_code = listbox_key (l, par)))
	    listbox_draw (h->window, l, h, 1);
	return ret_code;

    case WIDGET_FOCUS:
    case WIDGET_UNFOCUS:
    case WIDGET_DRAW:
	listbox_draw (h->window, l, h, msg != WIDGET_UNFOCUS);
	return 1;
    }
    return default_proc (h, msg, par);
}

static int listbox_event (Gpm_Event *event, WListbox *l)
{
    int i;
    
    Dlg_head *h = l->widget.parent;
    
    /* Single click */
    if (event->type & GPM_DOWN)
	dlg_select_widget (l->widget.parent, l);
    if (event->type & (GPM_DOWN|GPM_DRAG)){
	if (event->x < 0 || event->x >= l->width)
	    return MOU_REPEAT;
	if (event->y < 1)
	    for (i = -event->y; i >= 0; i--)
		listbox_back (l);
	else if (event->y > l->height)
	    for (i = event->y - l->height; i > 0; i--)
		listbox_fwd (l);
	else
	    listbox_select_entry (l, listbox_select_pos (l, l->top,
							 event->y - 1));
	
	/* We need to refresh ourselves since the dialog manager doesn't */
	/* know about this event */
	listbox_callback (h, WIDGET_DRAW, 0);
	wrefresh (l->widget.parent->window);
	return MOU_REPEAT;
    }

    /* Double click */
    if ((event->type & (GPM_DOUBLE|GPM_UP)) == (GPM_UP|GPM_DOUBLE)){
     	if (event->x < 0 || event->x >= l->width)
     	    return MOU_NORMAL;
     	if (event->y < 1 || event->y > l->height)
     	    return MOU_NORMAL;
	
	dlg_select_widget (l->widget.parent, l);
	listbox_select_entry (l, listbox_select_pos (l, l->top, event->y - 1)); 

	switch (l->action){
	case listbox_nothing:
	    break;

	case listbox_finish:
	    h->running   = 0;
	    h->ret_value = B_ENTER;
	    return MOU_ENDLOOP;

	case listbox_cback:
	    if ((*l->cback)(l) == listbox_finish)
		return MOU_ENDLOOP;
	}
    }
    return MOU_NORMAL;
}

static void listbox_destroy (WListbox *l)
{
    WLEntry *n, *p = l->list;
    int i;

    for (i = 0; i < l->count; i++){
	n = p->next;
	free (p->text);
	free (p);
	p = n;
    }
}

WListbox *listbox_new (int y, int x, int width, int height,
		       int action, lcback callback)
{
    WListbox *l = xmalloc (sizeof (WListbox), "listbox_new");

    init_widget (&l->widget, y, x, listbox_callback, (destroy_fn)listbox_destroy);

    l->list   = l->top = l->current = 0;
    l->pos    = 0;
    l->width  = width;
    l->height = height;
    l->count  = 0;
    l->top    = 0;
    l->current= 0;
    l->cback  = callback;
    l->action = action;
    return l;
}

/* Listbox item adding function.  They still lack a lot of functionality */
/* any takers? */
static void listbox_append_item (WListbox *l, WLEntry *e)
{
    if (!l->list){
	l->list = e;
	l->top = e;
	l->current = e;
	e->next = l->list;
	e->prev = l->list;
    } else {
	e->next = l->list;
	e->prev = l->list->prev;
	l->list->prev->next = e;
	l->list->prev = e;
    }
    l->count++;
}

char *listbox_add_item (WListbox *l, int pos, int hotkey, char *text,
			void *data)
{
    WLEntry *entry;

    if (!l)
	return 0;

    entry = xmalloc (sizeof (WLEntry), "listbox_add_item");
    entry->text = strdup (text);
    entry->data = data;
    entry->hotkey = hotkey;

    listbox_append_item (l, entry);
    
    return entry->text;
}

/* Selects the nth entry in the listbox */
void listbox_select_by_number (WListbox *l, int n)
{
    listbox_select_entry (l, listbox_select_pos (l, l->list, n));
}

WLEntry *listbox_search_text (WListbox *l, char *text)
{
    WLEntry *e;

    e = l->list;
    do {
	if(!strcmp (e->text, text))
	    return e;
	e = e->next;
    } while (e!=l->list);

    return NULL;
}

/* Returns the current string text as well as the associated extra data */
void listbox_get_current (WListbox *l, char **string, char **extra)
{
    if (!l->current){
	*string = 0;
	*extra  = 0;
    }
    if (string && l->current)
	*string = l->current->text;
    if (*extra && l->current)
	*extra = l->current->data;
}
