------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             S E M _ E V A L                              --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.23 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

--  This package contains various subprograms involved in compile time
--  evaluation of expressions and checks for staticness of expressions
--  and types. It also contains the circuitry for checking for violations
--  of pure and preelaborated conditions (this naturally goes here, since
--  these rules involve consideration of staticness).

with Types; use Types;

package Sem_Eval is

   ------------------------------------
   -- Handling of Static Expressions --
   ------------------------------------

   --  This package contains a set of routine that process individual
   --  subexpression nodes with the objective of folding (precomputing)
   --  the value of static expressions that are known at compile time and
   --  properly computing the setting of two flags that appear in every
   --  subexpression node:

   --    Is_Potentially_Static

   --      This flag is set on any expression that is potentially static
   --      according to the rules in RM section 4.9 (3-14).

   --    Raises_Constraint_Error

   --      This flag can only be set for potentially static nodes. It is set
   --      on subexpressions whose evaluation causes constraint error to be
   --      raised. A node with the Raises_Constraint_Error flag is set is
   --      not static, although it is still considered potentially static.

   --  The Is_Potentially_Static flag must be set exactly accurately according
   --  to the precise rules in the RM. This is required for proper semantic
   --  error detection in situations where static expressions are required.

   --  If a potentially static expression will raise constraint error at
   --  runtime if it is evaluated, then the Raises_Constraint_Error flag
   --  must be set accurately.

   --  If a potentially static expression does not raise constraint error,
   --  then the Raises_Constraint_Error flag is off, and the expression
   --  must be computed at compile time, which means that it has the form
   --  of either a literal, or a constant that is itself (recursively)
   --  either a literal or a constant.

   --  The above three rules must be followed exactly in order for legality
   --  checks to be accurate. For subexpressions that are not potentially
   --  static according to the RM definition, it might be possible to fold
   --  them anyway, but we make no attempt to do so. For one thing it is a
   --  bit tricky to follow the rule (RM 4.9(36)) that requires that any
   --  static expression appearing in a non static constant must be in
   --  range if we start folding non-static expressions.

   --  A static expression is one whose Is_Potentially_Static flag is set
   --  and whose Raises_Constraint_Error flag is clear (RM 4.9(2)), and the
   --  requirements stated above mean that all static expressions are either
   --  literals, or names of constant entities whose value is static.

   -----------------
   -- Subprograms --
   -----------------

   procedure Check_Non_Static_Context (N : Node_Id);
   --  Deals with the special checks required for a static expression
   --  that appears in a non-static context (see RM 4.9(36)). Called
   --  at the outer level on all expressions that appear in contexts
   --  that do not require a static expression.

   procedure Check_Static_Expression (N : Node_Id);
   --  This procedure must be called at the outer level for any expression
   --  that appears in a context that requires a static expression. It checks
   --  that the expression is static, and gives an error message if it is not.

   function Expr_Value (N : Node_Id) return Uint;
   --  Returns the folded value of the expression. This function is called
   --  in instances where it has already been determined that the expression
   --  is static. This version is used for integer values, and enumeration or
   --  character literals. In the latter two cases, the value returned is the
   --  Pos value in the relevant enumeration type.

   function Expr_Value (N : Node_Id) return Ureal;
   --  Returns the folded value of the expression. This function is called
   --  in instances where it has already been determined that the expression
   --  is static. This version is used for real values.

   function Expr_Value (N : Node_Id) return Entity_Id;
   --  Returns the folded value of the expression. This function is called
   --  in instances where it has already been determined that the expression
   --  is static. This version is used for enumeration types and returns the
   --  corresponding enumeration literal.

   procedure Eval_Actual                 (N : Node_Id);
   procedure Eval_Aggregate              (N : Node_Id);
   procedure Eval_Allocator              (N : Node_Id);
   procedure Eval_Arithmetic_Op          (N : Node_Id);
   procedure Eval_Character_Literal      (N : Node_Id);
   procedure Eval_Concatenation          (N : Node_Id);
   procedure Eval_Conditional_Expression (N : Node_Id);
   procedure Eval_Entity_Name            (N : Node_Id);
   procedure Eval_Indexed_Component      (N : Node_Id);
   procedure Eval_Integer_Literal        (N : Node_Id);
   procedure Eval_Logical_Op             (N : Node_Id);
   procedure Eval_Membership_Op          (N : Node_Id);
   procedure Eval_Op_Expon               (N : Node_Id);
   procedure Eval_Op_Not                 (N : Node_Id);
   procedure Eval_Real_Literal           (N : Node_Id);
   procedure Eval_Relational_Op          (N : Node_Id);
   procedure Eval_Shift                  (N : Node_Id);
   procedure Eval_Short_Circuit          (N : Node_Id);
   procedure Eval_Slice                  (N : Node_Id);
   procedure Eval_String_Literal         (N : Node_Id);
   procedure Eval_Qualified_Expression   (N : Node_Id);
   procedure Eval_Type_Conversion        (N : Node_Id);
   procedure Eval_Unary_Op               (N : Node_Id);
   procedure Eval_Unchecked_Conversion   (N : Node_Id);

   procedure Fold_Str (N : Node_Id; Val : String_Id);
   --  Rewrite N with a new N_String_Literal node as the result of the
   --  compile time evaluation of the node N. Val is the resulting string
   --  value from the folding operation. The Potentially_Static flag is
   --  set in the result node.

   procedure Fold_Uint (N : Node_Id; Val : Uint);
   --  Rewrite N with a (N_Integer_Literal, N_Identifier, N_Character_Literal)
   --  node as the result of the compile time evaluation of the node N. Val
   --  is the result in the integer case and is the position of the literal
   --  in the literals list for the enumeration case. Potentially_Static
   --  is set True in the result node.

   procedure Fold_Ureal (N : Node_Id; Val : Ureal);
   --  Rewrite N with a new N_Real_Literal node as the result of the compile
   --  time evaluation of the node N. Val is the resulting real value from
   --  the folding operation. The Potentially_Static flag is set in the
   --  result node.

   function Is_In_Range (N : Node_Id; Typ : Entity_Id) return Boolean;
   --  Check if the subexpression N is in range of the given type.
   --  The node passed is static, and the type is a static subtype.
   --  If the value is not in range, False is returned.

   function Is_Static_Expression (N : Node_Id) return Boolean;
   --  This procedure checks if the expression given by N is a legal static
   --  expression. It can only be applied to a fully resolved expression.

   function Is_Static_Subtype (Typ : Entity_Id) return Boolean;
   --  Determines whether a subtype fits the definition of an Ada static
   --  subtype as given in LRM 4.9.

private

   pragma Inline (Eval_Actual);
   pragma Inline (Eval_Allocator);
   pragma Inline (Eval_Character_Literal);
   pragma Inline (Eval_Conditional_Expression);
   pragma Inline (Eval_Indexed_Component);
   pragma Inline (Eval_Integer_Literal);
   pragma Inline (Eval_Real_Literal);
   pragma Inline (Eval_Shift);
   pragma Inline (Eval_Slice);
   pragma Inline (Eval_String_Literal);
   pragma Inline (Eval_Unchecked_Conversion);
   pragma Inline (Is_Static_Expression);

end Sem_Eval;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.21
--  date: Wed Jul 20 16:25:46 1994;  author: banner
--  (Is_In_Range): moved here from the body of the package so that is can be
--   shared by other modules.
--  ----------------------------
--  revision 1.22
--  date: Tue Jul 26 16:57:13 1994;  author: dewar
--  (Fold_Str, Fold_Uint, Fold_Ureal): Moved here from body
--  (Expression_Is_Not_Foldable): Moved here from body
--  ----------------------------
--  revision 1.23
--  date: Fri Aug  5 15:08:10 1994;  author: dewar
--  (Expression_Is_Not_Foldable): Move back to body
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
