------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                               S I N P U T                                --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.38 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Alloc;  use Alloc;
with Debug;  use Debug;
with Namet;  use Namet;
with Osint;  use Osint;
with Output; use Output;

package body Sinput is

   Source_Cache : Source_Buffer_Ptr := new Source_Buffer (1 .. 0);
   --  This is used to cache the most recently referenced source buffer to
   --  save lookups in the source file table. It must contain a valid source
   --  buffer pointer, but from the point of view of correctness it doesn't
   --  matter which one. It is initialized to the value in Source when a new
   --  Source file is read to ensure that it always has a valid value. The
   --  junk initial default value assures that a range test can be made (and
   --  failed) before the entry is initialized.

   Source_Cache_Index : Source_File_Index := No_Source_File;
   --  Contains the index of the entry corresponding to Source_Cache

   type Source_File_Record is record

      File_Name : File_Name_Type;
      --  Corresponding file name (simple name with no directory info)

      Full_File_Name : Name_Id;
      --  Full file name (full name with directory info), used for
      --  generation of error messages (and not for any other purpose)

      Source_Text : Source_Buffer_Ptr;
      --  Text of source file. Note that every source file has a distinct set
      --  of non-overlapping bounds, so it is possible to determine which file
      --  is referenced from a given subscript (Source_Ptr) value.

      Time_Stamp : Time_Stamp_Type;
      --  Time stamp of the source file

      Lines_Table : Lines_Table_Ptr;
      --  Pointer to lines table for this source

      Last_Line : Line_Number_Type;
      --  Subscript of last entry in Lines_Table (may be different from 'Last
      --  value because of the use of expandable tables). On completion of
      --  compilation of a unit (status = loaded), this is the number of
      --  source lines in the file.

      Keyword_Casing : Casing_Type;
      --  Casing style used in file for keyword casing. This is initialized
      --  to Unknown, and then set from the first occurrence of a keyword. This
      --  value is used only for formatting of error messages.

      Identifier_Casing : Casing_Type;
      --  Casing style used in file for identifier casing. This is initialized
      --  to Unknown, and then set from an identifier in the program as soon as
      --  one is found whose casing is sufficiently clear to make a decision.
      --  This value is used for formatting of error messages, and also is used
      --  in the detection of keywords misused as identifiers.

   end record;

   package Source_File is new Table (
     Table_Component_Type => Source_File_Record,
     Table_Index_Type     => Source_File_Index,
     Table_Low_Bound      => 1,
     Table_Initial        => 50,
     Table_Increment      => 100,
     Table_Name           => "Source_File");

   -----------------------
   -- Local Subprograms --
   -----------------------

   function Get_Source_Buffer_Ptr (P : Source_Ptr) return Source_Buffer_Ptr;
   --  Returns the pointer to the source buffer containing the given source
   --  location. Fatal error if given value not within a valid source file.

   -----------------
   -- Backup_Line --
   -----------------

   procedure Backup_Line (P : in out Source_Ptr) is
      Src : constant Source_Buffer_Ptr := Get_Source_Buffer_Ptr (P);

   begin
      P := P - 1;
      if P = Src'First then return; end if;

      if Src (P) = CR then
         if Src (P - 1) = LF then
            P := P - 1;
         end if;

      else -- Src (P) = LF
         if Src (P - 1) = CR then
            P := P - 1;
         end if;
      end if;

      --  Now find first character of the previous line

      while P > Src'First
        and then Src (P - 1) /= LF
        and then Src (P - 1) /= CR
      loop
         P := P - 1;
      end loop;
   end Backup_Line;

   -----------------------
   -- Get_Column_Number --
   -----------------------

   function Get_Column_Number (P : Source_Ptr) return Column_Number_Type is
      S     : Source_Ptr;
      C     : Column_Number_Type;
      Src : constant Source_Buffer_Ptr := Get_Source_Buffer_Ptr (P);

   begin
      S := Line_Start (P);
      C := 1;

      while S < P loop
         if Src (S) = HT then
            C := (C - 1) / 8 * 8 + (8 + 1);
         else
            C := C + 1;
         end if;

         S := S + 1;
      end loop;

      return C;
   end Get_Column_Number;

   ---------------------
   -- Get_Line_Number --
   ---------------------

   function Get_Line_Number (P : Source_Ptr) return Line_Number_Type is
      Sfile : Source_File_Index;
      Table : Lines_Table_Ptr;
      Lo    : Line_Number_Type;
      Hi    : Line_Number_Type;
      Mid   : Line_Number_Type;

   begin
      --  If the input source pointer is not a meaningful value then return
      --  at once with line number 1. This can happen for a file not found
      --  condition for a file loaded indirectly by RTE, and also perhaps on
      --  some unknown internal error conditions. In either case we certainly
      --  don't want to blow up.

      if P < 1 then
         return 1;

      --  Otherwise we can do the binary search

      else
         Sfile := Get_Source_File_Index (P);
         Table := Source_File.Table (Sfile).Lines_Table;
         Lo    := 1;
         Hi    := Source_File.Table (Sfile).Last_Line;

         loop
            Mid := (Lo + Hi) / 2;

            if P < Table (Mid) then
               Hi := Mid - 1;

            else -- P >= Table (Mid)

               if Mid = Hi or else
                  P < Table (Mid + 1)
               then
                  return Mid;
               else
                  Lo := Mid + 1;
               end if;

            end if;

         end loop;
      end if;
   end Get_Line_Number;

   ---------------------------
   -- Get_Source_Buffer_Ptr --
   ---------------------------

   function Get_Source_Buffer_Ptr (P : Source_Ptr) return Source_Buffer_Ptr is
   begin
      if P not in Source_Cache'range then
         Source_Cache_Index := Get_Source_File_Index (P);
         Source_Cache := Source_File.Table (Source_Cache_Index).Source_Text;
      end if;

      return Source_Cache;
   end Get_Source_Buffer_Ptr;

   ---------------------------
   -- Get_Source_File_Index --
   ---------------------------

   function Get_Source_File_Index
     (S    : Source_Ptr)
      return Source_File_Index
   is
   begin
      if S in Source_Cache'range then
         return Source_Cache_Index;

      else
         for J in 1 .. Source_File.Last loop
            if S in Source_File.Table (J).Source_Text'range then
               Source_Cache_Index := J;
               Source_Cache :=
                 Source_File.Table (Source_Cache_Index).Source_Text;
               return Source_Cache_Index;
            end if;
         end loop;
      end if;

      pragma Assert (False);
   end Get_Source_File_Index;

   ----------------
   -- Line_Start --
   ----------------

   function Line_Start (P : Source_Ptr) return Source_Ptr is
      S   : Source_Ptr := P;
      Src : constant Source_Buffer_Ptr := Get_Source_Buffer_Ptr (P);

   begin
      while S > Src'First
        and then Src (S - 1) /= CR
        and then Src (S - 1) /= LF
      loop
         S := S - 1;
      end loop;

      return S;
   end Line_Start;

   ----------------------
   -- Load_Source_File --
   ----------------------

   function Load_Source_File
     (N    : File_Name_Type)
      return Source_File_Index
   is
      Src  : Source_Buffer_Ptr;
      Lptr : Lines_Table_Ptr;
      X    : Source_File_Index;

   begin
      for J in 1 .. Source_File.Last loop
         if Source_File.Table (J).File_Name = N then
            Source_Cache_Index := J;
            Source_Cache := Source_File.Table (J).Source_Text;
            return J;
         end if;
      end loop;

      --  Here we must build a new entry in the file table

      Src := Read_Source_File (N);

      if Src = null then
         return No_Source_File;

      else
         Source_File.Increment_Last;
         X := Source_File.Last;

         if Debug_Flag_L then
            Write_Str ("*** Build source file table entry, Index = ");
            Write_Int (Int (X));
            Write_Str (", file name = ");
            Write_Name (N);
            Write_Eol;
         end if;

         Lptr := new Lines_Table_Type (1 .. Alloc_Lines_Initial);
         Lptr (1) := Src'First;

         Source_File.Table (X).File_Name         := N;
         Source_File.Table (X).Full_File_Name    := Full_Source_Name;
         Source_File.Table (X).Source_Text       := Src;
         Source_File.Table (X).Time_Stamp        := Current_Source_File_Stamp;
         Source_File.Table (X).Lines_Table       := Lptr;
         Source_File.Table (X).Last_Line         := 1;
         Source_File.Table (X).Keyword_Casing    := Unknown;
         Source_File.Table (X).Identifier_Casing := Unknown;

         Source_Cache := Src;
         Source_Cache_Index := X;
         return X;
      end if;
   end Load_Source_File;

   ---------------------------
   -- Skip_Line_Terminators --
   ---------------------------

   --  There are two distinct concepts of line terminator in GNAT

   --    A logical line terminator is what corresponds to the "end of a line"
   --    as described in RM 2.2 (13). Any of the characters FF, LF, CR or VT
   --    acts as an end of logical line in this sense, and it is essentially
   --    irrelevant whether one or more appears in sequence (since if a
   --    sequence of such characters is regarded as separate ends of line,
   --    then the intervening logical lines are null in any case).

   --    A physical line terminator is a sequence of format effectors that
   --    is treated as ending a physical line. Physical lines have no Ada
   --    semantic significance, but they are significant for error reporting
   --    purposes, since errors are identified by line and column location.

   --  In GNAT, a physical line is ended by any of the sequences LF, CR/LF,
   --  CR or LF/CR. LF is used in typical Unix systems, CR/LF in DOS systems,
   --  and CR alone in System 7. We don't know of any system using LF/CR, but
   --  it seems reasonable to include this case for consistency. In addition,
   --  we recognize any of these sequences in any of the operating systems,
   --  for better behavior in treating foreign files (e.g. a Unix file with
   --  LF terminators transferred to a DOS system).

   procedure Skip_Line_Terminators
     (P        : in out Source_Ptr;
      Physical : out Boolean)
   is
   begin
      pragma Assert (Source (P) in Line_Terminator);

      if Source (P) = CR then
         if Source (P + 1) = LF then
            P := P + 2;
         else
            P := P + 1;
         end if;

      elsif Source (P) = LF then
         if Source (P + 1) = CR then
            P := P + 2;
         else
            P := P + 1;
         end if;

      else -- Source (P) = FF or else Source (P) = VT
         P := P + 1;
         Physical := False;
         return;
      end if;

      --  Fall through in the physical line terminator case. First deal with
      --  making a possible entry into the lines table if one is needed.

      declare
         Lines_Table : Lines_Table_Ptr :=
           Source_File.Table (Current_Source_File).Lines_Table;

         Last_Line : Line_Number_Type :=
           Source_File.Table (Current_Source_File).Last_Line;

      begin
         Physical := True;

         --  Make entry in lines table if not already made (in some scan backup
         --  cases, we will be rescanning previously scanned source, so the
         --  entry may have already been made on the previous forward scan).

         if Source (P) /= EOF
           and then P > Lines_Table (Last_Line)
         then

            --  Reallocate the lines table if it has got too large. Note that
            --  we don't use the normal Table package mechanism because we
            --  have several of these tables, one for each source file.

            if Last_Line = Lines_Table'Last then

               declare
                  New_Lines_Table : Lines_Table_Ptr :=
                     new Lines_Table_Type
                       (1 .. Last_Line * (100 + Alloc_Lines_Increment) / 100);
               begin
                  if Debug_Flag_D then
                     Write_Str ("--> Allocating new lines table, size = ");
                     Write_Int (Int (New_Lines_Table'Last));
                     Write_Eol;
                  end if;

                  New_Lines_Table (1 .. Lines_Table'Last) :=
                    Lines_Table (1 .. Lines_Table'Last);
                  Free_Lines (Lines_Table);
                  Lines_Table := New_Lines_Table;
                  Source_File.Table (Current_Source_File).Lines_Table :=
                    Lines_Table;
               end;
            end if;

            Last_Line := Last_Line + 1;
            Lines_Table (Last_Line) := P;
            Source_File.Table (Current_Source_File).Last_Line :=
              Last_Line;
         end if;
      end;
   end Skip_Line_Terminators;

   --------------------
   -- Write_Location --
   --------------------

   procedure Write_Location (P : Source_Ptr) is
   begin
      if P = No_Location then
         Write_Str ("<no location>");

      elsif P = Standard_Location then
         Write_Str ("<standard location>");

      else
         Write_Char ('"');
         Write_Name_Decoded
           (Source_File.Table (Get_Source_File_Index (P)).File_Name);
         Write_Str (""", line ");
         Write_Int (Int (Get_Line_Number (P)));
         Write_Char ('(');
         Write_Int (Int (Get_Column_Number (P)));
         Write_Char (')');
      end if;
   end Write_Location;

   ----------------------
   -- Write_Time_Stamp --
   ----------------------

   procedure Write_Time_Stamp (S : Source_File_Index) is
      T : constant Time_Stamp_Type := Time_Stamp (S);

   begin
      Write_Char (T (1));
      Write_Char (T (2));
      Write_Char ('-');

      Write_Char (T (3));
      Write_Char (T (4));
      Write_Char ('-');

      Write_Char (T (5));
      Write_Char (T (6));
      Write_Char (' ');

      Write_Char (T (7));
      Write_Char (T (8));
      Write_Char (':');

      Write_Char (T (9));
      Write_Char (T (10));
      Write_Char ('.');

      Write_Char (T (11));
      Write_Char (T (12));
   end Write_Time_Stamp;

   ----------------------------------------------
   -- Access Subprograms for Source File Table --
   ----------------------------------------------

   function File_Name (S : Source_File_Index) return File_Name_Type is
   begin
      return Source_File.Table (S).File_Name;
   end File_Name;

   function Full_File_Name (S : Source_File_Index) return File_Name_Type is
   begin
      return Source_File.Table (S).Full_File_Name;
   end Full_File_Name;

   function Source_Text (S : Source_File_Index) return Source_Buffer_Ptr is
   begin
      return Source_File.Table (S).Source_Text;
   end Source_Text;

   function Time_Stamp (S : Source_File_Index) return Time_Stamp_Type is
   begin
      return Source_File.Table (S).Time_Stamp;
   end Time_Stamp;

   function Lines_Table (S : Source_File_Index) return Lines_Table_Ptr is
   begin
      return Source_File.Table (S).Lines_Table;
   end Lines_Table;

   function Last_Line (S : Source_File_Index) return Line_Number_Type is
   begin
      return Source_File.Table (S).Last_Line;
   end Last_Line;

   function Keyword_Casing (S : Source_File_Index) return Casing_Type is
   begin
      return Source_File.Table (S).Keyword_Casing;
   end Keyword_Casing;

   function Identifier_Casing (S : Source_File_Index) return Casing_Type is
   begin
      return Source_File.Table (S).Identifier_Casing;
   end Identifier_Casing;

   procedure Set_Keyword_Casing (S : Source_File_Index; C : Casing_Type) is
   begin
      Source_File.Table (S).Keyword_Casing := C;
   end Set_Keyword_Casing;

   procedure Set_Identifier_Casing (S : Source_File_Index; C : Casing_Type) is
   begin
      Source_File.Table (S).Identifier_Casing := C;
   end Set_Identifier_Casing;

   function Last_Source_File return Source_File_Index is
   begin
      return Source_File.Last;
   end Last_Source_File;

   function Num_Source_Files return Nat is
   begin
      return Int (Source_File.Last) - Int (Source_File.First) + 1;
   end Num_Source_Files;

end Sinput;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.36
--  date: Mon May 30 17:03:52 1994;  author: dewar
--  Minor reformatting
--  ----------------------------
--  revision 1.37
--  date: Fri Jul  8 10:17:49 1994;  author: dewar
--  (Write_Location): Remove obsolete use of Scol
--  ----------------------------
--  revision 1.38
--  date: Fri Jul 22 11:44:08 1994;  author: dewar
--  Change name Get_Col_Number to Get_Column_Number
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
