------------------------------------------------------------------------------
--                                                                          --
--                         GNAT RUNTIME COMPONENTS                          --
--                                                                          --
--             A D A . T E X T _ I O . W I D E _ T E X T _ I O              --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.3 $                              --
--                                                                          --
-- This specification is adapted from the Ada Reference Manual for use with --
-- GNAT.  In accordance with the copyright of that document, you can freely --
-- copy and modify this specification,  provided that if you redistribute a --
-- modified version,  any changes that you have made are clearly indicated. --
--                                                                          --
------------------------------------------------------------------------------

--  In GNAT, package Wide_Text_IO is a private child of Ada.Text_IO. This
--  allows Wide_Text_IO to share the declarations in the private part of
--  Text_IO (as well as sharing its routines). The Ada.Wide_Text_IO unit
--  in the standard library is simply a renaming of his private child.

--  Note: the generic subpackages of Wide_Text_IO (Integer_IO, Float_IO,
--  Fixed_IO, Modular_IO, Decimal_IO and Enumeration_IO) appear as private
--  children in GNAT. These children are with'ed automatically if they are
--  referenced, so this rearrangement is invisible to user programs, but has
--  the advantage that only the needed parts of Wide_Text_IO are processed
--  and loaded.

with System.WCh_Con;

private package Ada.Text_IO.Wide_Text_IO is

   type File_Type is limited private;
   type File_Mode is (In_File, Out_File, Append_File);

   type Count is new Ada.Text_IO.Count;

   subtype Positive_Count is Count range 1 .. Count'Last;

   Unbounded : constant Count := 0;
   --  Line and page length

   subtype Field is Ada.Text_IO.Field;

   subtype Number_Base is Integer range 2 .. 16;

   type Type_Set is (Lower_Case, Upper_Case);

   ---------------------
   -- File Management --
   ---------------------

   procedure Create
     (File : in out File_Type;
      Mode : in File_Mode := Out_File;
      Name : in String := "";
      Form : in String := "");

   procedure Open
     (File : in out File_Type;
      Mode : in File_Mode;
      Name : in String;
      Form : in String := "");

   procedure Close  (File : in out File_Type);
   procedure Delete (File : in out File_Type);
   procedure Reset  (File : in out File_Type; Mode : in File_Mode);
   procedure Reset  (File : in out File_Type);

   function Mode (File : in File_Type) return File_Mode;
   function Name (File : in File_Type) return String;
   function Form (File : in File_Type) return String;

   function Is_Open (File : in File_Type) return Boolean;

   ------------------------------------------------------
   -- Control of default input, output and error files --
   ------------------------------------------------------

   procedure Set_Input  (File : in File_Type);
   procedure Set_Output (File : in File_Type);
   procedure Set_Error  (File : in File_Type);

   function Standard_Input  return File_Type;
   function Standard_Output return File_Type;
   function Standard_Error  return File_Type;

   function Current_Input  return File_Type;
   function Current_Output return File_Type;
   function Current_Error  return File_Type;

   type File_Access is access constant File_Type;

   function Standard_Input  return File_Access;
   function Standard_Output return File_Access;
   function Standard_Error  return File_Access;

   function Current_Input  return File_Access;
   function Current_Output return File_Access;
   function Current_Error  return File_Access;

   --------------------
   -- Buffer control --
   --------------------

   procedure Flush (File : in out File_Type);
   procedure Flush;

   --------------------------------------------
   -- Specification of line and page lengths --
   --------------------------------------------

   procedure Set_Line_Length (File : in File_Type; To : in Count);
   procedure Set_Line_Length (To : in Count);

   procedure Set_Page_Length (File : in File_Type; To : in Count);
   procedure Set_Page_Length (To : in Count);

   function Line_Length (File : in File_Type) return Count;
   function Line_Length return Count;

   function Page_Length (File : in File_Type) return Count;
   function Page_Length return Count;

   ------------------------------------
   -- Column, Line, and Page Control --
   ------------------------------------

   procedure New_Line (File : in File_Type; Spacing : in Positive_Count := 1);
   procedure New_Line (Spacing : in Positive_Count := 1);

   procedure Skip_Line (File : in File_Type; Spacing : in Positive_Count := 1);
   procedure Skip_Line (Spacing : in Positive_Count := 1);

   function End_Of_Line (File : in File_Type) return Boolean;
   function End_Of_Line return Boolean;

   procedure New_Page (File : in File_Type);
   procedure New_Page;

   procedure Skip_Page (File : in File_Type);
   procedure Skip_Page;

   function End_Of_Page (File : in File_Type) return Boolean;
   function End_Of_Page return Boolean;

   function End_Of_File (File : in File_Type) return Boolean;
   function End_Of_File return Boolean;

   procedure Set_Col (File : in File_Type;  To : in Positive_Count);
   procedure Set_Col (To : in Positive_Count);

   procedure Set_Line (File : in File_Type; To : in Positive_Count);
   procedure Set_Line (To : in Positive_Count);

   function Col (File : in File_Type) return Positive_Count;
   function Col return Positive_Count;

   function Line (File : in File_Type) return Positive_Count;
   function Line return Positive_Count;

   function Page (File : in File_Type) return Positive_Count;
   function Page return Positive_Count;

   -----------------------------
   -- Characters Input-Output --
   -----------------------------

   procedure Get (File : in File_Type; Item : out Wide_Character);
   procedure Get (Item : out Wide_Character);
   procedure Put (File : in File_Type; Item : in Wide_Character);
   procedure Put (Item : in Wide_Character);

   procedure Look_Ahead
     (File        : in File_Type;
      Item        : out Wide_Character;
      End_Of_Line : out Boolean);

   procedure Look_Ahead
     (Item        : out Wide_Character;
      End_Of_Line : out Boolean);

   procedure Get_Immediate
     (File : in File_Type;
      Item : out Wide_Character);

   procedure Get_Immediate
     (Item : out Wide_Character);

   procedure Get_Immediate
     (File      : in File_Type;
      Item      : out Wide_Character;
      Available : out Boolean);

   procedure Get_Immediate
     (Item      : out Wide_Character;
      Available : out Boolean);

   --------------------------
   -- Strings Input-Output --
   --------------------------

   procedure Get (File : in File_Type; Item : out Wide_String);
   procedure Get (Item : out Wide_String);
   procedure Put (File : in File_Type; Item : in Wide_String);
   procedure Put (Item : in Wide_String);

   procedure Get_Line
     (File : in File_Type;
      Item : out Wide_String;
      Last : out Natural);

   procedure Get_Line
     (Item : out Wide_String;
      Last : out Natural);

   procedure Put_Line
     (File : in File_Type;
      Item : in Wide_String);

   procedure Put_Line
     (Item : in Wide_String);

   --  Exceptions

   Status_Error : exception renames IO_Exceptions.Status_Error;
   Mode_Error   : exception renames IO_Exceptions.Mode_Error;
   Name_Error   : exception renames IO_Exceptions.Name_Error;
   Use_Error    : exception renames IO_Exceptions.Use_Error;
   Device_Error : exception renames IO_Exceptions.Device_Error;
   End_Error    : exception renames IO_Exceptions.End_Error;
   Data_Error   : exception renames IO_Exceptions.Data_Error;
   Layout_Error : exception renames IO_Exceptions.Layout_Error;

private
   -------------------------------------
   -- Wide_Text_IO File Control Block --
   -------------------------------------

   package FCB renames System.File_Control_Block;

   Default_Wide_Encoding_Method :
     System.Wch_Con.WC_Encoding_Method := System.Wch_Con.WCEM_Hex;

   type Wide_Text_AFCB is new Ada.Text_IO.Text_AFCB with record

      WC_Method : System.WCh_Con.WC_Encoding_Method :=
                    Default_Wide_Encoding_Method;
      --  Encoding method to be used for this file

      Saved_Wide_Character : Wide_Character range
                               Wide_Character'Val (16#0080#) ..
                               Wide_Character'Val (16#FFFF#);
      --  This field is valid only if Before_Wide_Character is set. It
      --  contains a wide character read by Look_Ahead. If Look_Ahead
      --  reads a character in the range 16#0000# to 16#007F#, then it
      --  can use ungetc to put it back, but ungetc cannot be called
      --  more than once, so for characters above this range, we don't
      --  try to back up the file. Instead we save the character in this
      --  field and set the flag Before_Wide_Character to indicate that
      --  we are logically positioned before this character even though
      --  the stream is physically positioned after it.

   end record;

   type File_Type is access all Wide_Text_AFCB;

   function AFCB_Allocate
     (Control_Block : Wide_Text_AFCB)
      return          FCB.AFCB_Ptr;

   procedure AFCB_Close (File : access Wide_Text_AFCB);
   procedure AFCB_Free  (File : access Wide_Text_AFCB);

   --  Note: The Read and Write operations are inherited from Text_IO

   --  The following representation clause allows the use of unchecked
   --  conversion for rapid translation between the File_Mode type
   --  used in this package and System.File_IO.

   for File_Mode use
     (In_File     => 0,  -- System.File_IO.File_Mode'Pos (In_File)
      Out_File    => 2,  -- System.File_IO.File_Mode'Pos (Out_File)
      Append_File => 3); -- System.File_IO.File_Mode'Pos (Append_File)

   ----------------------
   -- Utility Routines --
   ----------------------

   procedure Setup (File : File_Type);
   --  Setup fields of file control block following Open or Create. This is
   --  in the spec so it can be accessed from Ada.Wide_Text_IO.C_Streams.
   --  The special task required is to set the wide text encoding method
   --  from a possible specification in the form file.

   --------------------
   -- Standard Files --
   --------------------

   Standard_Err_AFCB : aliased Wide_Text_AFCB;
   Standard_In_AFCB  : aliased Wide_Text_AFCB;
   Standard_Out_AFCB : aliased Wide_Text_AFCB;

   Standard_Err : aliased File_Type := Standard_Err_AFCB'Access;
   Standard_In  : aliased File_Type := Standard_In_AFCB'Access;
   Standard_Out : aliased File_Type := Standard_Out_AFCB'Access;
   --  Standard files

   Current_In   : aliased File_Type := Standard_In;
   Current_Out  : aliased File_Type := Standard_Out;
   Current_Err  : aliased File_Type := Standard_Err;
   --  Current files

end Ada.Text_IO.Wide_Text_IO;
