------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--                     S Y S T E M . T A S K _ T I M E R                    --
--                                                                          --
--                                  S p e c                                 --
--                                                                          --
--                             $Revision: 1.3 $                             --
--                                                                          --
--       Copyright (c) 1991,1992,1993,1994, FSU, All Rights Reserved        --
--                                                                          --
-- GNARL is free software; you can redistribute it  and/or modify it  under --
-- terms  of  the  GNU  Library General Public License  as published by the --
-- Free Software  Foundation;  either version 2, or (at  your  option)  any --
-- later  version.  GNARL is distributed  in the hope that  it will be use- --
-- ful, but but WITHOUT ANY WARRANTY;  without even the implied warranty of --
-- MERCHANTABILITY  or  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Gen- --
-- eral Library Public License  for more details.  You should have received --
-- a  copy of the GNU Library General Public License along with GNARL;  see --
-- file COPYING.LIB.  If not,  write to the  Free Software Foundation,  675 --
-- Mass Ave, Cambridge, MA 02139, USA.                                      --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Finalization;

with Ada.Calendar;
--  Used for, Calendar.Time

with Ada.Real_Time;
--  Used for, Real_Time.Time
--            Real_Time.Time_Span

with System.Task_Clock;
--  Used for, Stimespec

package System.Task_Timer is

   --  The contents of this package, with the exception of Delay_Block,
   --  are internal to GNARL; Delay_Block is part of the compiler interface.
   --  Unfortunately, they must be visible so that they can be accessed
   --  from the body of Ada.Calendar.Delays and Ada.Real_Time.Delays,
   --  and they must be in the same package as
   --  Delay_Block so that they can be used to implement its finalization.

   type Delay_Block is limited private;
   --  An object of this type is passed by the compiler to the Wait
   --  entry of each delay object
   --  (e.g. Ada.Calendar.Delays.Delay_Object.Wait).  This is used by
   --  the delay object implementation to queue the delay request and
   --  suspend the task.

   protected Timer is
      pragma Priority (Priority'Last);

      --  The following Enqueue entries enqueue elements in wake-up time
      --  order using a single timer queue (time in System.Real_Time.Time).

      entry Enqueue_Time_Span
        (T : in Ada.Real_Time.Time_Span;
         D : access Delay_Block);
      entry Enqueue_Duration
         (T : in Duration;
         D : access Delay_Block);
      entry Enqueue_Real_Time
        (T : in Ada.Real_Time.Time;
         D : access Delay_Block);
      entry Enqueue_Calendar_Time
        (T : in Ada.Calendar.Time;
         D : access Delay_Block);
      procedure Dequeue (D : access Delay_Block);
      procedure Service (T : out System.Task_Clock.Stimespec);
      function Empty return Boolean;
--  private
--  Private protected operations are not currently supported by the
--  compiler.
      procedure Time_Enqueue
        (T : in System.Task_Clock.Stimespec;
         D : access Delay_Block);
   end Timer;

private

   --  Signal_Object provides a simple suspension capability.

   protected type Signal_Object is
      pragma Priority (Priority'Last);
      entry Wait;
      procedure Signal;
   private
      Open   : Boolean := False;
   end Signal_Object;

   type Q_Link is access all Delay_Block;

   type Delay_Block is new Ada.Finalization.Limited_Controlled with record
      S_O      : Signal_Object;
      T        : System.Task_Clock.Stimespec;    --  wake up time
      Next     : Q_Link;
      Previous : Q_Link;
   end record;

   procedure Finalize (Object : in out Delay_Block);

end System.Task_Timer;
