/* Panel managing.
   Copyright (C) 1994 Miguel de Icaza.
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <ncurses.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <string.h>
#include <stdlib.h>	/* For malloc() and free() */
#ifdef HAVE_UNISTD_H
    #include <unistd.h>	/* For chdir(), readlink() and getwd()/getcwd() */
#endif
#include "mem.h"
#include "mad.h"
#include "global.h"
#include "dir.h"
#include "panel.h"
#include "util.h"
#include "color.h"
#include "tree.h"
#include "win.h"
#include "main.h"

#define ELEMENTS(arr) ( sizeof(arr) / sizeof((arr)[0]) )

static char rcsid [] = "$Id: screen.c,v 1.18 1995/01/27 02:36:16 miguel Exp $";

int show_mini_info = 1;		/* Show miniinfo */

int extra_info = show_perm;

/* If true, then use stat() on the cwd to determine directory changes */
int fast_reload = 0;

/* If we have an info panel, this points to it */
Panel *the_info_panel = 0;

void set_colors (Panel *panel)
{
    wstandend (panel->win_file);
    if (hascolors){
	wattron (panel->win_file, NORMAL_COLOR);
    }
}

void panel_refresh (Panel *panel)
{
    wrefresh (panel->win_file);
}

void set_attr (Panel *panel, int hilight, int marked)
{
    int color;

    color = (marked * 2 + hilight);
    wstandend (panel->win_file);
    
    /* Does this work on B&W terminals? */
    if (hascolors)
	wattron (panel->win_file,sel_mark_color[color]|(marked ? A_BOLD : 0));
    else {
	if (hilight)
	    wattron (panel->win_file, A_REVERSE | (marked ? A_BOLD : 0));
	else
	    if (marked)
		wattron (panel->win_file, A_BOLD);
    }
}

/* This functions return a string representation of a file entry */
static inline char *string_file_type (file_entry *fe)
{
    static char buffer [2];

    if (S_ISDIR (fe->buf.st_mode))
	buffer [0] = '/';
    else if (S_ISLNK (fe->buf.st_mode))
	buffer [0] = '@';
    else if (S_ISSOCK (fe->buf.st_mode))
	buffer [0] = '=';
    else if (S_ISCHR (fe->buf.st_mode))
	buffer [0] = '-';
    else if (S_ISBLK (fe->buf.st_mode))
	buffer [0] = '+';
    else if (S_ISFIFO (fe->buf.st_mode))
	buffer [0] = '|';
    else if (is_exe (fe->buf.st_mode))
	buffer [0] = '*';
    else
	buffer [0] = ' ';
    return buffer;
}

static inline char *string_file_permission (file_entry *fe)
{
    return string_perm (fe->buf.st_mode);
}

static inline char *string_file_nlinks (file_entry *fe)
{
    static char buffer [16];

    sprintf (buffer, "%16d", fe->buf.st_nlink);
    return buffer;
}

static inline char *string_file_owner (file_entry *fe)
{
    return get_owner (fe->buf.st_uid);
}

static inline char *string_file_group (file_entry *fe)
{
    return get_group (fe->buf.st_gid);
}

static inline char *string_file_size (file_entry *fe)
{
    static char buffer [16];
	
    sprintf (buffer, "%d", (int) fe->buf.st_size);
    return buffer;
}

static inline char *string_file_mtime (file_entry *fe)
{
    return file_date (fe->buf.st_mtime);
}

static inline char *string_file_atime (file_entry *fe)
{
    return file_date (fe->buf.st_atime);
}

static inline char *string_file_ctime (file_entry *fe)
{
    return file_date (fe->buf.st_ctime);
}

static inline char *string_file_name (file_entry *fe)
{
    return fe->fname;
}

static inline char *string_space (file_entry *fe)
{
    return " ";
}

static inline char *string_marked (file_entry *fe)
{
    return fe->f.marked ? "*" : " ";
}

static inline char *string_file_perm_octal (file_entry *fe)
{
    static char buffer [9];

    sprintf (buffer, "0%06o", fe->buf.st_mode);
    return buffer;
}

static inline char *string_inode (file_entry *fe)
{
    static char buffer [9];

    sprintf (buffer, "%ld", fe->buf.st_ino);
    return buffer;
}

static struct {
    char *id;
    int  min_size;
    int  expands;
    int  default_just;
    char *title;
    char *(*string_fn)(file_entry *);
} formats [] = {
{ "name",  12, 1, J_LEFT,  "Name",       string_file_name },
{ "size",  7,  1, J_RIGHT, "Size",       string_file_size },
{ "type",  1,  0, J_LEFT,  "",           string_file_type },
{ "mtime", 12, 0, J_RIGHT, "MTime",      string_file_mtime },
{ "perm",  10, 0, J_RIGHT, "Permission", string_file_permission },
{ "mode",  7,  0, J_RIGHT, "OctM",       string_file_perm_octal },
{ "|",     1,  0, J_RIGHT, "|",          0 },
{ "nlink", 2,  0, J_RIGHT, "Nl",        string_file_nlinks },
{ "owner", 8,  0, J_LEFT,  "Owner",      string_file_owner },
{ "group", 8,  0, J_LEFT,  "Group",      string_file_group },
{ "atime", 12, 0, J_RIGHT, "ATime",      string_file_atime },
{ "ctime", 12, 0, J_RIGHT, "CTime",      string_file_ctime },
{ "space", 1,  0, J_RIGHT, " ",          string_space },
{ "mark",  1,  0, J_RIGHT, " ",          string_marked },
{ "inode", 5,  0, J_RIGHT, "Inode",      string_inode },
};

void repaint_file (Panel *panel, int file_index, int move)
{
    int    i, d;
    char   *txt;
    int    top = panel->fmt_count;
    WINDOW *win = panel->win_file;
    int    second_column = 0;
    char   *format;
    int    just_mode;
    int    empty_line = file_index >= panel->count;
    int	   length = 0, width;

    if (panel->split){
	second_column = (file_index - panel->top_file) / panel->lines;
	if (second_column)
	    width = panel->cols - panel->cols/2 - 1;
	else
	    width = panel->cols/2 - 1;
    } else
        width = panel->cols;

    if (move){
	if (panel->split)
	    wmove (win, (file_index - panel->top_file) % panel->lines + 2,
		   (second_column * (panel->cols/2) + 1));
	else
	    wmove (win, file_index - panel->top_file + 2, 1);
    }
    
    for (i = 0; i < top; i++){
	file_entry *fe = &panel->dir.list [file_index];

	just_mode = panel->format [i].just_mode;
	if (panel->format [i].string_fn){
	    if (empty_line)
		txt = " ";
	    else {
		if (panel->format [i].string_fn == string_file_name)
		    txt = name_trunc ((*panel->format [i].string_fn)(fe),
				      panel->format [i].field_len);
		else {
		    int txt_len;
		    
		    txt = (*panel->format [i].string_fn)(fe);
		    txt_len = strlen (txt);
		    d = txt_len - panel->format [i].field_len;
		    if (d >0){
			if (just_mode == J_LEFT)
			    txt [panel->format [i].field_len] = 0;
			else
			    txt += d;
		    }
		}
	    }
	    format = just_mode == J_LEFT ? "%-*s" : "%*s";
	    if (strlen (txt) > panel->format [i].field_len){
		fprintf (stderr, "\n\n\n\n\nFatal\n\n\n");
	    }
	    wprintw (win, format, panel->format [i].field_len, txt);
	    length += panel->format [i].field_len;
	} else {
	    waddch (win, ACS_VLINE);
	    length++;
	}
    }

    if (length < width)
    	wprintw (win, "%*s", width - length, "");
    
    if (panel->split){
	if (second_column)
	    waddch (win, ' ');
	else {
	    wattrset (win, NORMAL_COLOR);
	    waddch (win, ACS_VLINE);
	}
    }
/*    wrefresh (win); */
}

/* FIXME: This function uses hard coded constants */
/* panel->cols - 22 is not a wise idea */
void mini_info_brief (Panel *panel)
{
    int   isdir = 0;
    char  *sdir_txt = "";
    
    #define entry (panel->dir.list [panel->selected].buf)
    
    if (S_ISDIR (entry.st_mode)){
	isdir = 1;
	sdir_txt = strcmp (panel->dir.list [panel->selected].fname, "..")
	    ? ">SUB-DIR<" : ">UP--DIR<";
    }
    
    wprintw (panel->win_file, 
	     "%s%-*s", string_file_type (&panel->dir.list [panel->selected]),
	     panel->cols-22, split_extension
	     (name_trunc (panel->dir.list [panel->selected].fname,
			  panel->cols-22), panel->cols-22));
    
    waddch (panel->win_file, ACS_VLINE);
    wprintw (panel->win_file, "%*s",
	     extra_info == show_perm ? 9 : 7,
	     isdir ? sdir_txt : size_trunc (entry.st_size));
    waddch (panel->win_file, ACS_VLINE);

    wprintw (panel->win_file, "%10s", string_perm (entry.st_mode));
    wrefresh (panel->win_file);
}

void display_mini_info (Panel *panel)
{
    if (panel->view_type == view_info
	|| panel->view_type == view_quick)
	return;
    
    if (!show_mini_info)
	return;
    
    /* This is used to clear the mini_info area */
    wmove (panel->win_file, panel->lines+3, 1);
    
    if (searching){
	wattrset (panel->win_file, INPUT_COLOR);
	wprintw (panel->win_file, "/%-*s", panel->cols-1, search_buffer);
	wattrset (panel->win_file, NORMAL_COLOR);
	return;
    }
    
    if (panel->view_type == view_tree){
	wattrset (panel->win_file, NORMAL_COLOR);
	wprintw (panel->win_file, "%-*s", panel->cols,
		 name_trunc (panel->cwd, panel->cols));
	return;
    }

    if (panel->marked){
	char buffer [100];
	char *p;
	
	wattrset (panel->win_file, MARKED_COLOR | A_BOLD);
	wprintw (panel->win_file, "%*s", panel->cols, " ");
	wmove (panel->win_file, panel->lines+3, 1);
	sprintf (buffer, "  %s bytes in %d file%s",
		 size_trunc_sep (panel->total), panel->marked,
		 panel->marked == 1 ? "" : "s");
	p = buffer;
	if (strlen (buffer) > cpanel->cols-2){
	    buffer [cpanel->cols-2] = 0;
	    p += 2;
	}
	wprintw (panel->win_file, "%-*s", cpanel->cols, p);
	return;
    }

    set_colors (panel);
    
    if (S_ISLNK (panel->dir.list [panel->selected].buf.st_mode)){
	char *link, link_target [MAXPATHLEN];
	int  len;

	link = copy_strings (panel->cwd, "/",
			     panel->dir.list [panel->selected].fname, 0);
	len = readlink (link, link_target, MAXPATHLEN);
	free (link);
	if (len > 0){
	    link_target[len] = 0;
	    wprintw (panel->win_file, "-> %-*s", panel->cols - 3,
		     name_trunc (link_target, panel->cols - 3));
	} else 
	    wprintw (panel->win_file, "<readlink failed>");
	return;
    }

    if (panel->user_mini_status){
	int display;
	char *err;

	if (panel->format)
	    free (panel->format);
	panel->format = parse_display_format (panel, panel->mini_status_format,
					      &display, &err, 1);
	if (err){
	    beep ();	/* For debugging */
	    free (panel->mini_status_format);
	    panel->mini_status_format = strdup (DEFAULT_USER_FORMAT);
	    panel->format =
		parse_display_format (panel, panel->mini_status_format,
				      &display, &err, 1);
	}
	repaint_file (panel, panel->selected, 0);
	
	/* This clears the second half of the mini status line, 'cause
	   repaint_file writes only the first half */
	if (panel->split){
	    wmove (panel->win_file, panel->lines+3, panel->cols/2);
	    wprintw(panel->win_file, "%*s", panel->cols - panel->cols/2 + 1, "");
	}
	
	if (panel->format)
	    free (panel->format);
	panel->format = parse_display_format (panel, panel_format (panel),
					      &display, &err, 0);
	return;
    }

    if (panel->view_type == view_brief){
	mini_info_brief (panel);
	return;
    }
    
    wprintw(panel->win_file,"%-*s", panel->cols,
	    name_trunc(panel->dir.list [panel->selected].fname,
		       panel->cols));
}

void paint_dir (Panel *panel)
{
    int i;
    int hilight;		/* Color used for hilighting */
    int marked;			/* Color used for marked file */
    int items;			/* Number of items */

    if (panel->view_type == view_tree ||
	panel->view_type == view_info ||
	panel->view_type == view_quick)
	return;
    
    items = panel->split ? panel->lines * 2 : panel->lines;
    
    for (i = 0; i < items; i++){
	if (i+panel->top_file >= panel->count)
	    set_attr (panel, 0, 0);
	else {
	    hilight = panel->selected==i+panel->top_file && panel->active;
	    marked  = panel->dir.list [i+panel->top_file].f.marked;
	    set_attr (panel, hilight, marked);
	}
	repaint_file (panel, i+panel->top_file, 1);
    }
    wstandend (panel->win_file);
}

void show_dir (Panel *panel)
{
    char tmp [200];

    set_colors (panel);
    box (panel->win_file, ACS_VLINE, ACS_HLINE);
    
    if (panel->active)
	wattron (panel->win_file, REVERSE_COLOR);
    
    if (panel->view_type == view_tree)
	mvwaddstr (panel->win_file, 0, 1, "Tree view");
    else {
trim (strip_home (panel->cwd), tmp, panel->cols-3);
	mvwaddstr (panel->win_file, 0, 1, tmp);
    }
    
    if (panel->active)
	wstandend (panel->win_file);
}

/* To be used only by long_frame and full_frame to adjust top_file */
static void adjust_top_file (Panel *panel)
{
    if (panel->selected - panel->top_file > panel->lines)
	panel->top_file = panel->selected;
}

extern void paint_info_panel (Panel *panel);

void info_frame (Panel *panel)
{
    panel->win_file = panel->small_frame;
    set_colors (panel);
    werase (panel->win_file);
    wclr (panel->win_file);
    box (panel->win_file, ACS_VLINE, ACS_HLINE);
}

void paint_panel (Panel *panel)
{
    switch (panel->view_type){

    case view_info:
	paint_info_panel (panel);
	break;

    case view_tree:
	show_dir (panel);
	show_tree (panel);
	display_mini_info (panel);
	break;

    case view_quick:
	paint_quick_view_panel (panel);
	break;

    default:
	show_dir (panel);
	paint_dir (panel);
	display_mini_info (panel);
	if (show_mini_info){
	    wstandend (panel->win_file);
	    wmove (panel->win_file, panel->lines+2, 1);
	    whline (panel->win_file, ACS_HLINE|NORMAL_COLOR, panel->cols);
	}
    }
    panel_refresh (panel);
}

void Xtry_to_select (Panel *panel, char *name)
{
    int i;
    char *subdir;
    
    if (!name){
	panel->selected = 0;
	panel->top_file = 0;
	return;
    }

    /* We only want the last component of the directory */
    for (subdir = name + strlen (name) - 1; subdir >= name; subdir--){
	if (*subdir == '/'){
	    subdir++;
	    break;
	}
    }
    if (subdir < name)
	subdir = name;
    
    /* Search that subdirectory, if found select it */
    for (i = 0; i < panel->count; i++){
	if (strcmp (subdir, panel->dir.list [i].fname))
	    continue;

	if (i != panel->selected){
	    panel->selected = i;
	    panel->top_file = panel->selected - panel->lines/2;
	    if (panel->top_file < 0)
		panel->top_file = 0;
	}
	return;
    }

    /* Try to select a file near the file that is missing */
    if (panel->selected >= panel->count){
	panel->selected = panel->count-1;
	panel->top_file = panel->selected - panel->lines/2;
	if (panel->top_file < 0)
	    panel->top_file = 0;
    } else
	return;
}

/* Sets the frame to use depending on the parameter */
void panel_set_frame_to (Panel *panel, int display)
{
    if (display == frame_full){
	panel->win_file = panel->big_frame;
	panel->cols     = panel->full_cols;
    } else if (display == frame_half){
	panel->win_file = panel->small_frame;
	panel->cols     = panel->half_cols;
    }
}

/* Contents initialization. Can be called only once. */
/* Assumes that panel comes with sort_type and view_type already set */
void do_init_panel (Panel *panel)
{
    char *err;
    int  display;		/* Full/half */

    get_current_wd (panel->cwd, sizeof (panel->cwd)-2);
    strcpy (panel->lwd, ".");

    panel->dir.list  = (file_entry *) malloc (MIN_FILES * sizeof (file_entry));
    panel->dir.size  = MIN_FILES;
    panel->filter    = 0;
    panel->split     = 0;
    panel->top_file  = 0;
    panel->selected  = 0;
    panel->is_status = 0;
    panel->marked    = 0;
    panel->total     = 0;
    panel->reverse   = 0;
    panel->format    = 0;
    panel->dirs_marked = 0;
    panel->dont_reload = 0;

    if (!panel->user_format)
	panel->user_format = strdup (DEFAULT_USER_FORMAT);
    if (!panel->mini_status_format)
	panel->mini_status_format = strdup (DEFAULT_USER_FORMAT);
    
    /* Load the default format */
    panel->format = parse_display_format (panel, panel_format (panel),
					      &display, &err, 0);
    if (err) free (err);
    panel_set_frame_to (panel, display);

    if (panel->view_type == view_info){
	info_frame (panel);
	the_info_panel = panel;
    } else if (panel->view_type == view_quick){
	info_frame (panel);
    } else
	paint_frame (panel);
}

/* Curses window initialization. Can be called many times. */
void init_panel (Panel *panel, int x1, int y1, int x2, int y2)
{
    panel->win_file = newwin (y2 - y1, x2 - x1, y1, x1);
    panel->small_frame = panel->win_file;
    panel->big_frame   = newwin (y2 - y1, COLS, y1, 0);

    set_colors (panel);
    wclr (panel->small_frame);

    panel->lines     = y2-y1-3-(show_mini_info*2);
    panel->half_cols = x2-x1-2;
    panel->full_cols = COLS-2;
    panel->cols      = panel->half_cols;

    if (panel->view_type == view_info){
	info_frame (panel);
	the_info_panel = panel;
    } else if (panel->view_type == view_quick){
	info_frame (panel);
    } else
	paint_frame (panel);

    /* Configure repaint mode for the panel */
    leaveok (panel->win_file, TRUE);
    leaveok (panel->big_frame, TRUE);
}

void panel_reload (Panel *panel)
{
    int i;
    struct stat current_stat;
    
    if (!PANEL_ISVIEW (panel))
	return;
    
    if (panel->view_type == view_tree){
	tree_rescan_cmd ();
	paint_panel (panel);
	return;
    }
    
    if (fast_reload
	&& !stat (panel->cwd, &current_stat)
	&& current_stat.st_ctime == panel->dir_stat.st_ctime
	&& current_stat.st_mtime == panel->dir_stat.st_mtime)
	return;

    while (chdir (panel->cwd) == -1){
	char *last_slash;

	if (panel->cwd [0] == '/' && panel->cwd [1] == 0){
	    panel->count = set_zero_dir (&panel->dir);
	    return;
	}
	last_slash = strrchr (panel->cwd, '/');
	if (!last_slash || last_slash == panel->cwd)
	    strcpy (panel->cwd, "/");
	else
	    *last_slash = 0;
        bzero (&(panel->dir_stat), sizeof (panel->dir_stat));
	show_dir (panel);
    }
    
    panel->count = do_reload_dir (&panel->dir, panel->sort_type, panel->count,
				  panel->reverse, panel->filter);
    panel->marked = 0;
    panel->total  = 0;
    
    for (i = 0; i < panel->count; i++)
	if (panel->dir.list [i].f.marked){
	    panel->marked++;
	    panel->total += panel->dir.list [i].buf.st_size;
	}
    
    display_mini_info (panel);
}

void paint_frame (Panel *panel)
{
    int  header_len;
    int  spaces, extra;
    int  i, top, side, width;
    char *txt, buffer[30]; /*Hope that this is enough ;-) */
    
    adjust_top_file (panel);
    
    werase (panel->win_file);
    show_dir (panel);

    wmove (panel->win_file, 1, 1);

    for (side = 0; side <= panel->split; side++){
	if (side){
	    set_attr (panel, 0, 0);
	    waddch (panel->win_file, ACS_VLINE);
	    width = panel->cols - panel->cols/2;
	} else if (panel->split)
	    width = panel->cols/2 - 1;
	else
	    width = panel->cols;
	
	for (i = 0, top = panel->fmt_count; i < top; i++){

	    if (panel->format [i].string_fn){
		txt = panel->format [i].title;
		header_len = strlen (txt);
		if (header_len > panel->format [i].field_len){
		    strcpy (buffer, txt);
		    txt = buffer;
		    txt [panel->format [i].field_len] = 0;
		    header_len = strlen (txt);
		}
		
		set_attr (panel, 0, 1);
		spaces = (panel->format [i].field_len - header_len) / 2;
		extra  = (panel->format [i].field_len - header_len) % 2;
		wprintw (panel->win_file, "%*s%-s%*s", spaces, "",
			 panel->format [i].title, spaces+extra, "");
		width -= 2 * spaces + extra + header_len;
	    } else {
		set_attr (panel, 0, 0);
		waddch  (panel->win_file, ACS_VLINE);
		width --;
		continue;
	    }
	}
	if (width > 0)
	    wprintw (panel->win_file, "%*s", width, "");
    }
}

static char *parse_panel_size (Panel *panel, char *format, int *frame_size)
{
    format = skip_separators (format);
    
    *frame_size = frame_half;
    panel->split = 0;
    
    if (strncmp (format, "full", 4) == 0){
	*frame_size = frame_full;
	panel->cols = panel->full_cols;
	format += 4;
    } else if (strncmp (format, "half", 4) == 0){
	format += 4;
	panel->cols = panel->half_cols;
    }

    /* Now, the optional column specifier */
    format = skip_separators (format);
    
    if (*format == '1' || *format == '2'){
	panel->split = *format == '2';
	format++;
    }

    return skip_separators (format);
}

/* Format is:

   all              := panel_format? format
   panel_format     := [full|half] [1|2]
   format           := one_format_e
                     | format , one_format_e

   one_format_e := format.id opt_size
   opt_size         := : size
*/

format_e *parse_display_format (Panel *panel, char *format, 
				int *frame_size, char **error, int isstatus)
{
    #define MAX_EXPAND 4
    
    format_e *darr;		/* The formats we return */
    int  fields;		/* Number of fields */
    int  expand_list [MAX_EXPAND]; /* Expand at most 4 fields. */
    int  expand_top = 0;	/* Max used element in expand */
    int  field;
    int  usable_columns;	/* Usable columns in the panel */
    int  total_cols = 0;	/* Used columns by the format */
    char *s;
    int  i, j;
    int  set_justify;		/* flag: set justification mode? */
    int  justify = 0;		/* Which mode. */
    int  cols_save;

    *error = 0;
    
    if (!format)
	format = "half type,name,|,size,|,perm";

    /* Determine the panel size */
    cols_save = panel->cols;
    format = parse_panel_size (panel, format, frame_size);
    /* This makes sure that the panel and mini status full/half mode
       setting is equal */
    if (isstatus){
    	panel->cols = cols_save;
    }

    usable_columns = (panel->cols / (panel->split+1)) - panel->split;
    
    /* Count the number of fields */
    fields = 1;
    for (s = format; *s; s++)
	if (*s == ',')
	    fields++;

    panel->fmt_count = fields;
    
    /* Allocate the space */
    darr = xmalloc (sizeof (format_e) * fields, "parse_display_format");
    
    for (field = 0; field < fields; field++){
	int found = 0;

	format = skip_separators (format);

	if (*format == '<' || *format == '>'){
	    set_justify = 1;
	    justify = *format == '<' ? J_LEFT : J_RIGHT;
	    format = skip_separators (format+1);
	} else
	    set_justify = 0;
	
	for (i = 0; i < ELEMENTS(formats); i++){
	    int klen = strlen (formats [i].id);

	    if (strncmp (format, formats [i].id, klen) != 0)
		continue;

	    format += klen;

	    darr [field].field_len = formats [i].min_size;
	    darr [field].string_fn = formats [i].string_fn;
	    darr [field].title     = formats [i].title;

	    if (set_justify)
		darr [field].just_mode = justify;
	    else
		darr [field].just_mode = formats [i].default_just;

	    found = 1;

	    format = skip_separators (format);

	    /* If we have a size specifier */
	    if (*format == ':'){
		int req_length;
		
		format++;
		req_length = atoi (format);
#if 0		
		if (req_length > darr [field].field_len)
#endif
		    darr [field].field_len = req_length;

		format = skip_numbers (format);
	    } else {
		/* If the size was specified, we don't want auto-expansion */
		/* Take note if it's expandable */
		if (formats [i].expands && expand_top < MAX_EXPAND)
		    expand_list [expand_top++] = field;
	    }
	    break;
	}
	if (!found){
	    char old_char;
	    
	    int pos = min (8, strlen (format));
	    free (darr);
	    old_char = format [pos];
	    format [pos] = 0;
	    *error = copy_strings("Unknow tag on display format: ", format, 0);
	    format [pos] = old_char;
	    return 0;
	}
	total_cols += darr [field].field_len;

    }
    
    /* If we used more columns than the available columns, adjust that */
    if (total_cols > usable_columns){
	int dif   = total_cols - usable_columns;
	
	while (dif)
	    for (j = 0; j < field; j++){
		if (darr [j].field_len-1){
		    darr [j].field_len--;
		    if (dif)
			dif--;
		}
	    }
	total_cols = usable_columns;
    }

    /* Expand the available space */
    if ((usable_columns > total_cols) && expand_top){
	int spaces = (usable_columns - total_cols) / expand_top;
	int extra  = (usable_columns - total_cols) % expand_top;
	
	for (i = 0; i < expand_top; i++)
	    darr [expand_list [i]].field_len += spaces;

	/* The modulo goes to the first one */
	darr [expand_list [0]].field_len += extra;
    }
    
    return darr;
}

/* Switches the panel to the mode specified in the format */
char *set_panel_format (Panel *p, char *format)
{
    int  display;
    char *err;

    if (p->format)
	free (p->format);
    
    p->format = parse_display_format (p, format, &display, &err, 0);
    if (err)
	return err;

    panel_set_frame_to (p, display);
    paint_frame (p);
    paint_panel (p);
    
    return 0;
}

/* Given the panel->view_type returns the format string to be parsed */
char *panel_format (Panel *panel)
{
    switch (panel->view_type){

    case view_long:
	return "full perm,space,nlink,space,owner,space,group,space,size,space,mtime,space,name";

    case view_brief:
	return "half 2,type,name";


    case view_user:
	return panel->user_format;

    default:
    case view_full:
	return "half type,name,|,size,|,mtime";
    }
}
