/* Directory tree browser for the Midnight Commander
   Copyright (C) 1994 Janne Kukonlehto
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */
#include <sys/types.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <fcntl.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <errno.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>	/* For free() and atoi() */
#include <string.h>
#include <ncurses.h>
#include "mad.h"
#include "global.h"
#include "util.h"
#include "color.h"
#include "dialog.h"
#include "dir.h"
#include "win.h"
#include "input.h"
#include "panel.h"
#include "mouse.h"
#include "main.h"
#include "file.h"	/* For copy_dir_dir(), move_dir_dir(), erase_dir() */
#include "help.h"
#include "key.h"	/* For mi_getch() */
#include "tree.h"
void set_navig_label (WINDOW *fkeys);

static char rcsid [] = "$Id: tree.c,v 1.2 1995/01/27 02:36:29 miguel Exp $";

static tree_entry *tree_first = NULL;	/* First entry in the list */
static tree_entry *tree_last = NULL;	/* Last entry in the list */
int tree_lines;				/* Height of tree window */
/* static int tree_count=0; (currently unused) Number of dirs in the list */
static int topdiff;    			/* The difference between the topmost shown and the selected */
static tree_entry *selected_ptr;	/* The selected directory */
static WINDOW *wtree;  			/* The tree window */
static WINDOW *fkeys;			/* The fkey window */
static char search_buffer_tree [256];	/* Current search string */
static int done;			/* Flag: exit tree */
int tree_navigation_flag = 1;		/* Flag: 0 = old, 1 = new navigation */
static char check_name [MAXPATHLEN];	/* Directory which is been checked */
static tree_entry *check_start;		/* Start of checked subdirectories */
static int check_sublevel;     		/* Sublevel of check_name */
static tree_entry **tree_shown = NULL;	/* Entries currently on screen */

/* Returns number of common characters */
static inline int str_common (char *s1, char *s2)
{
    int result = 0;

    while (*s1++ == *s2++)
	result++;
    return result;
}

static tree_entry *back_ptr (tree_entry *ptr, int *count)
{
    int i = 0;

    while (ptr && ptr->prev && i < *count){
	ptr = ptr->prev;
	i ++;
    }
    *count = i;
    return ptr;
}

static tree_entry *forw_ptr (tree_entry *ptr, int *count)
{
    int i = 0;

    while (ptr && ptr->next && i < *count){
	ptr = ptr->next;
	i ++;
    }
    *count = i;
    return ptr;
}

/* Searches for specified directory */
static tree_entry *whereis (char *name)
{
    tree_entry *current = tree_first;
    int flag = -1;

#if 0
    if (tree_last){
	flag = strcmp (tree_last->name, name);
	if (flag <= 0){
	    current = tree_last;
	} else if (selected_ptr){
	    flag = strcmp (selected_ptr->name, name);
	    if (flag <= 0){
		current = selected_ptr;
	    }
	}
    }
#endif
    while (current && flag < 0
	   && (flag = strcmp (current->name, name)) < 0)
	current = current->next;

    if (flag == 0)
	return current;
    else
	return NULL;
}

/* Add a directory to the list of directories */
tree_entry *tree_add_entry (char *name)
{
    int flag = -1;
    tree_entry *current = tree_first;
    tree_entry *old = NULL;
    tree_entry *new;
    int i, len;
    int submask = 0;

    if (tree_last && tree_last->next)
	abort ();
#if 0
    if (tree_last){
	flag = strcmp (tree_last->name, name);
	if (flag <= 0){
	    current = tree_last;
	    old = current->prev;
	} else if (selected_ptr){
	    flag = strcmp (selected_ptr->name, name);
	    if (flag <= 0){
		current = selected_ptr;
		old = current->prev;
	    }
	}
    }
#endif
    /* Search for the correct place */
    while (current && flag < 0
	   && (flag = strcmp (current->name, name)) < 0){
	old = current;
	current = current->next;
    }
    
    if (flag == 0)
	return current; /* Already in the list */

    /* Not in the list -> add it */
    new = xmalloc (sizeof (tree_entry), "tree, tree_entry");
    if (!current){
	/* Append to the end of the list */
	if (!tree_first){
	    /* Empty list */
	    tree_first = new;
	    new->prev = NULL;
	} else {
	    old->next = new;
	    new->prev = old;
	}
	new->next = NULL;
	tree_last = new;
    } else {
	/* Insert in to the middle of the list */
	new->prev = old;
	if (old){
	    /* Yes, in the middle */
	    new->next = old->next;
	    old->next = new;
	} else {
	    /* Nope, in the beginning of the list */
	    new->next = tree_first;
	    tree_first = new;
	}
	new->next->prev = new;
    }
    /* tree_count++; */

    /* Calculate attributes */
    new->name = strdup (name);
    len = strlen (new->name);
    new->sublevel = 0;
    for (i = 0; i < len; i++)
	if (new->name [i] == '/'){
	    new->sublevel++;
	    new->subname = new->name + i + 1;
	}
    if (new->next)
	submask = new->next->submask;
    else
	submask = 0;
    submask |= 1 << new->sublevel;
    submask &= (2 << new->sublevel) - 1;
    new->submask = submask;
    new->mark = 0;

    /* Correct the submasks of the previous entries */
    current = new->prev;
    while (current && current->sublevel > new->sublevel){
	current->submask |= 1 << new->sublevel;
	current = current->prev;
    }

    /* The entry has now been added */

    if (new->sublevel > 1){
	/* Let's check if the parent directory is in the tree */
	char *parent = strdup (new->name);
	int i;

	for (i = strlen (parent) - 1; i > 1; i--){
	    if (parent [i] == '/'){
		parent [i] = 0;
		tree_add_entry (parent);
		break;
	    }
	}
	free (parent);
    }

    return new;
}

/* Append a directory to the list of directories */
static tree_entry *tree_append_entry (char *name)
{
    tree_entry *current, *new;
    int i, len;
    int submask = 0;

    /* We assume the directory is not yet in the list */

    new = xmalloc (sizeof (tree_entry), "tree, tree_entry");
    if (!tree_first){
        /* Empty list */
        tree_first = new;
        new->prev = NULL;
    } else {
        tree_last->next = new;
        new->prev = tree_last;
    }
    new->next = NULL;
    tree_last = new;

    /* tree_count++; */

    /* Calculate attributes */
    new->name = strdup (name);
    len = strlen (new->name);
    new->sublevel = 0;
    for (i = 0; i < len; i++)
	if (new->name [i] == '/'){
	    new->sublevel++;
	    new->subname = new->name + i + 1;
	}
    submask = 1 << new->sublevel;
    submask &= (2 << new->sublevel) - 1;
    new->submask = submask;
    new->mark = 0;

    /* Correct the submasks of the previous entries */
    current = new->prev;
    while (current && current->sublevel > new->sublevel){
	current->submask |= 1 << new->sublevel;
	current = current->prev;
    }

    /* The entry has now been appended */
    return new;
}

static void remove_entry (tree_entry *entry)
{
    tree_entry *current = entry->prev;
    long submask = 0;

    if (selected_ptr == entry){
	if (selected_ptr->next)
	    selected_ptr = selected_ptr->next;
	else
	    selected_ptr = selected_ptr->prev;
    }

    /* Correct the submasks of the previous entries */
    if (entry->next)
	submask = entry->next->submask;
    while (current && current->sublevel > entry->sublevel){
	submask |= 1 << current->sublevel;
	submask &= (2 << current->sublevel) - 1;
	current->submask = submask;
	current = current->prev;
    }

    /* Unlink the entry from the list */
    if (entry->prev)
	entry->prev->next = entry->next;
    else
	tree_first = entry->next;
    if (entry->next)
	entry->next->prev = entry->prev;
    else
	tree_last = entry->prev;
    /* tree_count--; */

    /* Free the memory used by the entry */
    free (entry->name);
    free (entry);
}

void tree_remove_entry (char *name)
{
    tree_entry *current, *base, *old;
    int len, base_sublevel;

    /* Miguel Ugly hack */
    if (name [0] == '/' && name [1] == 0)
	return;
    /* Miguel Ugly hack end */
    
    base = whereis (name);
    if (!base)
	return;	/* Doesn't exist */
    if (check_name [0] == '/' && check_name [1] == 0)
	base_sublevel = base->sublevel;
    else
	base_sublevel = base->sublevel + 1;
    len = strlen (base->name);
    current = base->next;
    while (current
	   && strncmp (current->name, base->name, len) == 0
	   && current->sublevel >= base_sublevel){
	old = current;
	current = current->next;
	remove_entry (old);
    }
    remove_entry (base);
}

void destroy_tree (void)
{
    tree_entry *current, *old;
    
    current = tree_first;
    while (current){
	old = current;
	current = current->next;
	free (old->name);
	free (old);
    }

    selected_ptr = tree_first = tree_last = NULL;
}

/* Mark the subdirectories of the current directory for delete */
void start_tree_check (void)
{
    tree_entry *current;
    int len;

    /* Search for the start of subdirectories */
    get_current_wd (check_name, MAXPATHLEN);
    check_start = NULL;
    current = whereis (check_name);
    if (!current){
	/* Cwd doesn't exist -> add it */
	current = tree_add_entry (check_name);
	return;
    }

    /* Mark old subdirectories for delete */
    check_start = current->next;
    len = strlen (check_name);
    if (check_name [0] == '/' && check_name [1] == 0)
	check_sublevel = current->sublevel;
    else
	check_sublevel = current->sublevel + 1;

    current = check_start;
    while (current
	   && strncmp (current->name, check_name, len) == 0
	   && current->sublevel >= check_sublevel){
	current->mark = 1;
	current = current->next;
    }
}

/* This subdirectory exists -> clear deletion mark */
void do_tree_check (const char *subname)
{
    char *name;
    tree_entry *current, *base;
    int flag = 0, len;

    /* Calculate the full name of the subdirectory */
    if (subname [0] == '.' &&
	(subname [1] == 0 || (subname [1] == '.' && subname [2] == 0)))
	return;
    if (check_name [0] == '/' && check_name [1] == 0)
	name = copy_strings ("/", subname, 0);
    else
	name = copy_strings (check_name, "/", subname, 0);

    /* Search for the subdirectory */
    current = check_start;
    while (current && (flag = strcmp (current->name, name)) < 0)
	current = current->next;
    
    if (flag != 0)
	/* Doesn't exist -> add it */
	current = tree_add_entry (name);
    free (name);

    /* Clear the deletion mark from the subdirectory and its children */
    base = current;
    if (base){
	len = strlen (base->name);
	base->mark = 0;
	current = base->next;
	while (current
	       && strncmp (current->name, base->name, len) == 0
	       && current->sublevel >= check_sublevel){
	    current->mark = 0;
	    current = current->next;
	}
    }
}

/* Delete subdirectories which still have the deletion mark */
void end_tree_check (void)
{
    tree_entry *current, *old;
    int len;

    /* Check delete marks and delete if found */
    len = strlen (check_name);

    current = check_start;
    while (current
	   && strncmp (current->name, check_name, len) == 0
	   && current->sublevel >= check_sublevel){
	old = current;
	current = current->next;
	if (old->mark)
	    remove_entry (old);
    }
}

/* Loads the .mc.tree file */
void load_tree (void)
{
    char *filename;
    FILE *file;
    char name [MAXPATHLEN], oldname[MAXPATHLEN];
    char *different;
    int len, common;

    filename = copy_strings (home_dir, "/.mc.tree", 0);
    file = fopen (filename, "r");
    free (filename);
    if (!file){
	/* No new tree file -> let's try the old file */
	filename = copy_strings (home_dir, "/.mc.tree.", 0);
	file = fopen (filename, "r");
	free (filename);
    }
    
    if (file){
	/* File open -> read contents */
	oldname [0] = 0;
	while (fgets (name, MAXPATHLEN, file)){
	    len = strlen (name);
	    if (name [len - 1] == '\n'){
		name [--len] = 0;
	    }
	    if (name [0] != '/'){
		/* Clear-text decompression */
		common = atoi (strtok (name, " "));
		different = strtok (NULL, "");
		strcpy (oldname + common, different);
		tree_append_entry (oldname);
	    } else {
		tree_append_entry (name);
		strcpy (oldname, name);
	    }
	}
	fclose (file);
    }
    if (!tree_first){
	/* Nothing loaded -> let's add some standard directories */
	tree_add_entry ("/");
	selected_ptr = tree_first;
	tree_rescan_cmd ();
	tree_add_entry (home_dir);
	tree_chdir (home_dir);
	tree_rescan_cmd ();
    }
}

/* Save the .mc.tree file */
void save_tree (void)
{
    tree_entry *current;
    char *filename;
    FILE *file;
    int i, common;

    filename = copy_strings (home_dir, "/.mc.tree", 0);
    file = fopen (filename, "w");
    free (filename);
    if (!file){
	fprintf (stderr, "Can't open the .mc.tree file for writing:\n%s\n",
		 unix_error_string (errno));
	return;
    }

    current = tree_first;
    while (current){
	if (current->prev && (common = str_common (current->prev->name, current->name)) > 2)
	    /* Clear-text compression */
	    i = fprintf (file, "%d %s\n", common, current->name + common);
	else
	    i = fprintf (file, "%s\n", current->name);
	if (i == EOF){
	    fprintf (stderr, "Can't write to the .mc.tree file:\n%s\n",
		 unix_error_string (errno));
	    break;
	}
	current = current->next;
    }
    fclose (file);
}

void show_tree (Panel *panel)
{
    tree_entry *current;
    int i, j, topsublevel;
    int x = 1, y = 1;

    /* Initialize */
    if (!panel){
	wclrn (wtree, 2);
    } else {
	wtree = panel->win_file;
        wattrset (wtree, NORMAL_COLOR);
	tree_lines = panel->lines;
	wmove (wtree, 1, 1);
	whline (wtree, ' ', panel->cols);
    }
    if (tree_shown)
	free (tree_shown);
    tree_shown = (tree_entry**)xmalloc (sizeof (tree_entry*) * tree_lines,
					"tree, show_tree");
    for (i = 0; i < tree_lines; i++)
	tree_shown [i] = NULL;
    if (tree_first)
	topsublevel = tree_first->sublevel;
    else
	topsublevel = 0;
    if (!selected_ptr){
	selected_ptr = tree_first;
	topdiff = 0;
    }
    current = selected_ptr;
    /* Calculate the directory which is to be shown on the topmost line */
    if (tree_navigation_flag){
	i = 0;
	while (current->prev && i < topdiff){
	    current = current->prev;
	    if (current->sublevel < selected_ptr->sublevel){
		if (strncmp (current->name, selected_ptr->name,
			     strlen (current->name)) == 0)
		    i++;
	    } else if (current->sublevel == selected_ptr->sublevel){
		for (j = strlen (current->name) - 1; current->name [j] != '/'; j--);
		if (strncmp (current->name, selected_ptr->name, j) == 0)
		    i++;
	    } else if (current->sublevel == selected_ptr->sublevel + 1
		       && strlen (selected_ptr->name) > 1){
		if (strncmp (current->name, selected_ptr->name,
			     strlen (selected_ptr->name)) == 0)
		    i++;
	    }
	}
	topdiff = i;
    } else
	current = back_ptr (current, &topdiff);

    /* Loop for every line */
    for (i = 0; i < tree_lines && current; i++){
	tree_shown [i] = current;
	/* Move to the beginning of the line */
	if (panel){
	    wmove (wtree, i + 2, 1);
	    waddch (wtree, ' ');
	} else
	    wmove (wtree, i+2, 3);
	if (current->sublevel == topsublevel){
	    /* Top level directory */
	    if ((!panel || panel->active) && current == selected_ptr){
		/* Selected directory -> change color */
		if (panel)
		    wattrset (wtree, SELECTED_COLOR);
		else
		    wattrset (wtree, MARKED_COLOR | A_BOLD);
	    }
	    /* Show full name */
	    if (panel)
		waddstr (wtree,
			 name_trunc (current->name, panel->cols - 4));
	    else
		waddstr (wtree,
			 name_trunc(current->name, 56));
	} else{
	    /* Sub level directory */

	    /* Output branch parts */
	    for (j = 0; j < current->sublevel - topsublevel - 1; j++){
		if (panel){
		    if (panel->cols - 6 - 3 * j < 9)
			break;
		} else {
		    if (56 - 3 * j < 9)
			break;
		}
		waddch (wtree, ' ');
		if (current->submask & (1 << (j + topsublevel + 1)))
		    waddch (wtree, ACS_VLINE);
		else
		    waddch (wtree, ' ');
		waddch (wtree, ' ');
	    }
	    waddch (wtree, ' '); j++;
	    if (!current->next || !(current->next->submask & (1 << current->sublevel)))
		waddch (wtree, ACS_LLCORNER);
	    else
		waddch (wtree, ACS_LTEE);
	    waddch (wtree, ACS_HLINE);

	    if ((!panel || panel->active) && current == selected_ptr){
		/* Selected directory -> change color */
		if (panel)
		    wattrset (wtree, SELECTED_COLOR);
		else
		    wattrset (wtree, MARKED_COLOR | A_BOLD);
	    }
	    /* Show sub-name */
	    if (panel){
		waddch (wtree, ' ');
		waddstr (wtree,
			 name_trunc (current->subname,
				     panel->cols - 4 - 3 * j));
	    } else {
		waddch (wtree, ' ');
		waddstr (wtree,
			 name_trunc (current->subname,
				     56 - 3 * j));
	    }
	}
	waddch (wtree, ' ');
	/* Return to normal color */
	if (panel){
	    wattrset (wtree, NORMAL_COLOR);
	    getyx (wtree, y, x);
	    whline (wtree, ' ', panel->cols - x + 1);
	}
	else
	    wattrset (wtree, REVERSE_COLOR);

	/* Calculate the next value for current */
	if (tree_navigation_flag){
	    current = current->next;
	    while (current){
		if (current->sublevel < selected_ptr->sublevel){
		    if (strncmp (current->name, selected_ptr->name,
				 strlen (current->name)) == 0)
			break;
		} else if (current->sublevel == selected_ptr->sublevel){
		    for (j = strlen (current->name) - 1; current->name [j] != '/'; j--);
		    if (strncmp (current->name, selected_ptr->name, j) == 0)
			break;
		} else if (current->sublevel == selected_ptr->sublevel + 1
			   && strlen (selected_ptr->name) > 1){
		    if (strncmp (current->name, selected_ptr->name,
				 strlen (selected_ptr->name)) == 0)
			break;
		}
		current = current->next;
	    }
	} else
	    current = current->next;
    }
    if (!panel){
	/* Show mini info */
	wmove (wtree, tree_lines+2, 2);
	whline (wtree, 0, 60);
	wmove (wtree, tree_lines + 3, 3);
	if (search_buffer_tree [0]){
	    /* Show search string */
	    wattrset (wtree, INPUT_COLOR);
	    waddch (wtree, ' ');
	    waddstr (wtree, search_buffer_tree);
	    waddch (wtree, ' ');
	    wattrset (wtree, REVERSE_COLOR);
	} else {
	    /* Show full name of selected directory */
	    waddstr (wtree, name_trunc (selected_ptr->name, 58));
	}
    } else {
	/* Clear possible extra lines */
	for (j = y + 1; j < tree_lines + 2; j++){
	    wmove (wtree, j, 1);
	    whline (wtree, ' ', panel->cols);
	}
	/* Horizontal line */
	wmove (wtree, tree_lines + 2, 1);
	whline (wtree, 0, panel->cols);

	if (selected_ptr){
	    strcpy (panel->cwd, selected_ptr->name);
	    if (panel->active)
		chdir (panel->cwd);
	}
    }
    wrefresh (wtree);
}

static void check_focus (void)
{
    if (topdiff < 3)
	topdiff = 3;
    else if (topdiff >= tree_lines - 3)
	topdiff = tree_lines - 3 - 1;
}

void tree_move_backward (int i)
{
    tree_entry *current;
    int j = 0;
    
    if (tree_navigation_flag){
	current = selected_ptr;
	while (j < i && current->prev
	       && current->prev->sublevel >= selected_ptr->sublevel){
	    current = current->prev;
	    if (current->sublevel == selected_ptr->sublevel){
		selected_ptr = current;
		j ++;
	    }
	}
	i = j;
    } else
	selected_ptr = back_ptr (selected_ptr, &i);
    topdiff -= i;
    check_focus ();
}

void tree_move_forward (int i)
{
    tree_entry *current;
    int j = 0;

    if (tree_navigation_flag){
	current = selected_ptr;
	while (j < i && current->next
	       && current->next->sublevel >= selected_ptr->sublevel){
	    current = current->next;
	    if (current->sublevel == selected_ptr->sublevel){
		selected_ptr = current;
		j ++;
	    }
	}
	i = j;
    } else
	selected_ptr = forw_ptr (selected_ptr, &i);
    topdiff += i;
    check_focus ();
}

void tree_move_to_child (void)
{
    tree_entry *current;

    /* Do we have a starting point? */
    if (!selected_ptr)
	return;
    /* Take the next entry */
    current = selected_ptr->next;
    /* Is it the child of the selected entry */
    if (current && current->sublevel > selected_ptr->sublevel){
	/* Yes -> select this entry */
	selected_ptr = current;
	topdiff++;
	check_focus ();
    } else {
	/* No -> rescan and try again */
	tree_rescan_cmd ();
	current = selected_ptr->next;
	if (current && current->sublevel > selected_ptr->sublevel){
	    selected_ptr = current;
	    topdiff++;
	    check_focus ();
	}
    }
}

void tree_move_to_parent (void)
{
    tree_entry *current;

    if (!selected_ptr)
	return;
    current = selected_ptr->prev;
    while (current && current->sublevel >= selected_ptr->sublevel){
	current = current->prev;
	topdiff--;
    }
    if (!current)
	current = tree_first;
    selected_ptr = current;
    check_focus ();
}

void tree_move_to_top (int dummy)
{
    selected_ptr = tree_first;
    topdiff = 0;
}

void tree_move_to_bottom (int dummy)
{
    selected_ptr = tree_last;
    topdiff = tree_lines - 3 - 1;
}

void tree_chdir (char *dir)
{
    tree_entry *current;

    current = whereis (dir);
    if (current){
	selected_ptr = current;
	check_focus ();
    }
}

int tree_init (char *current_dir, int lines)
{
    tree_lines = lines;
    topdiff = lines / 2;
    selected_ptr = tree_first;
    tree_chdir (current_dir);
    return 0;
}

/* Handle mouse click */
void tree_event (int y)
{
    if (tree_shown [y]){
	selected_ptr = tree_shown [y];
	topdiff = y;
    }
}

/* Mouse callback */
static int event_callback (Gpm_Event *event)
{
    if (!(event->type & GPM_UP))
	return MOU_ENDLOOP;
    event->y --;

    if (event->y < 0)
	tree_move_backward (tree_lines - 1);
    else if (event->y >= tree_lines)
	tree_move_forward (tree_lines - 1);
    else{
	tree_event (event->y);
	if ((event->type & (GPM_UP|GPM_DOUBLE)) == (GPM_UP|GPM_DOUBLE)){
	    done = 1;
	}
    }
    return MOU_ENDLOOP;
}

/* Search tree for text */
int search_tree (char *text)
{
    tree_entry *current;
    int len;
    int wrapped = 0;
    int found = 0;

    len = strlen (text);
    current = selected_ptr;
    found = 0;
    while (!wrapped || current != selected_ptr){
	if (strncmp (current->subname, text, len) == 0){
	    selected_ptr = current;
	    found = 1;
	    break;
	}
	current = current->next;
	if (!current){
	    current = tree_first;
	    wrapped = 1;
	}
	topdiff++;
    }
    check_focus ();
    return found;
}

static void win_init (void);
static void win_done (void);

static int help_cmd (void)
{
    win_done ();
    interactive_display (LIBDIR "mc.hlp", "[Directory Tree]");
    win_init ();
    return 1;
}

int tree_rescan_cmd (void)
{
    DIR *dirp;
    struct dirent *dp;
    struct stat buf;
    char old_dir [MAXPATHLEN];

    if (!selected_ptr)
	return 0;
    if (!get_current_wd (old_dir, MAXPATHLEN))
	return 0;
    if (chdir (selected_ptr->name))
	return 0;
    start_tree_check ();
    dirp = opendir (".");
    if (dirp){
	for (dp = readdir (dirp); dp; dp = readdir (dirp)){
	    lstat (dp->d_name, &buf);
	    if (S_ISDIR (buf.st_mode))
		do_tree_check (dp->d_name);
	}
	closedir (dirp);
    }
    end_tree_check ();
    chdir (old_dir);
    return 1;
}

int tree_forget_cmd (void)
{
    if (selected_ptr)
	tree_remove_entry (selected_ptr->name);
    return 1;
}

static int toggle_nav_mode (void)
{
    tree_navigation_flag = 1 - tree_navigation_flag;
    set_navig_label (fkeys);
    return 1;
}

void tree_copy (char *default_dest)
{
    char *dest;

    if (!selected_ptr)
	return;
    sprintf (cmd_buf, "Copy \"%s\" directory to:", name_trunc (selected_ptr->name, 60));
    dest = input_expand_dialog (" Copy ", cmd_buf, default_dest);
    if (!dest || !*dest){
	return;
    }
    create_op_win (OP_COPY);
    copy_dir_dir (selected_ptr->name, dest, 1);
    destroy_op_win ();
    free (dest);
}

static int copy_cmd (void)
{
    win_done ();
    tree_copy ("");
    win_init ();
    return 1;
}

void tree_move (char *default_dest)
{
    char *dest;
    struct stat buf;

    if (!selected_ptr)
	return;
    sprintf (cmd_buf, "Move \"%s\" directory to:", name_trunc (selected_ptr->name, 60));
    dest = input_expand_dialog (" Move ", cmd_buf, default_dest);
    if (!dest || !*dest){
	return;
    }
    if (stat (dest, &buf)){
	message (1, " Error ", " Can't stat the destination \n %s ",
		 unix_error_string (errno));
	free (dest);
	return;
    }
    if (!S_ISDIR (buf.st_mode)){
	message (1, " Error ", " The destination isn't a directory ");
	free (dest);
	return;
    }
    create_op_win (OP_MOVE);
    move_dir_dir (selected_ptr->name, dest);
    destroy_op_win ();
    free (dest);
}

static int move_cmd (void)
{
    win_done ();
    tree_move ("");
    win_init ();
    return 1;
}

static int tree_mkdir_cmd (void)
{
    char old_dir [MAXPATHLEN];

    if (!selected_ptr)
	return 0;
    if (!get_current_wd (old_dir, MAXPATHLEN))
	return 0;
    if (chdir (selected_ptr->name))
	return 0;
    win_done ();
    mkdir_cmd ();
    win_init ();
    tree_rescan_cmd ();
    chdir (old_dir);
    return 1;
}

static int rmdir_cmd (void)
{
    char old_dir [MAXPATHLEN];

    if (selected_ptr){
	if (!get_current_wd (old_dir, MAXPATHLEN))
	    return 0;
	if (chdir ("/"))
	    return 0;
	win_done ();
	if (confirm_delete){
	    char *cmd_buf;
	    int result;

	    cmd_buf = xmalloc (strlen (selected_ptr->name) + 20,
			       "tree, rmdir_cmd");
	    sprintf (cmd_buf, "  Delete %s?  ", selected_ptr->name);
	    result = query_dialog (" Delete ", cmd_buf, 3, 2, " Yes ", " No ");
	    free (cmd_buf);
	    if (result != 0){
		win_init ();
		return 0;
	    }
	}
	create_op_win (OP_DELETE);
	if (erase_dir (selected_ptr->name) == FILE_CONT)
	    tree_forget_cmd ();
	destroy_op_win ();
	win_init ();
	chdir (old_dir);
	return 1;
    } else
	return 0;
}

static int quit_cmd (void)
{
    return done = 1;
}

void set_navig_label (WINDOW *fkeys)
{
    set_label_text (fkeys, 4, tree_navigation_flag ? "Static" : "Dynamc");
}

static void win_init (void)
{
    create_dialog (60, tree_lines + 2, " Directory tree ", "", 0);
    wtree = get_top_text ();
    push_event (3, 2, 60 + 2, tree_lines + 3, (mouse_h) event_callback, 0,
		event_use_frame);

    fkeys = push_fkey (1);
    define_label_quit (fkeys, 1, "Help", help_cmd);
    define_label_quit (fkeys, 2, "Rescan", tree_rescan_cmd);
    define_label_quit (fkeys, 3, "Forget", tree_forget_cmd);
    define_label_quit (fkeys, 4, "Navig", toggle_nav_mode);
    set_navig_label (fkeys);
    define_label_quit (fkeys, 5, "Copy", copy_cmd);
    define_label_quit (fkeys, 6, "RenMov", move_cmd);
    define_label_quit (fkeys, 7, "Mkdir", tree_mkdir_cmd);
    define_label_quit (fkeys, 8, "Rmdir", rmdir_cmd);
    define_label (fkeys, 9, "", 0);
    define_label_quit (fkeys, 10, "Quit", quit_cmd);
}

static void win_done (void)
{
    pop_fkey (fkeys);
    pop_event ();
    destroy_dialog ();
}

/* Show directory tree dialog and return the selected directory */
char *tree (char *current_dir)
{
    int i, c;
    char *result = 0;

    /* Initialize */
    tree_lines = 16;
    if (tree_lines > LINES - 6)
	tree_lines = LINES - 6;
    if (tree_init (current_dir, tree_lines))
	return NULL;
    win_init ();
    search_buffer_tree [0] = 0;
    done = 0;

    /* Main loop */
    while (!done){
	show_tree (NULL);
	c = mi_getch ();
	check_fkeys (c);
	if (check_movement_keys (c, 0, tree_lines, tree_move_backward,
				 tree_move_forward, tree_move_to_top,
				 tree_move_to_bottom))
	    /* Nothing */;
	else switch (c){
	case KEY_LEFT:
	case XCTRL('F'):
	    tree_move_to_parent ();
	    break;
	case KEY_RIGHT:
	case XCTRL('B'):
	    tree_move_to_child ();
	    break;
	case XCTRL('R'):
	    /* Rescan directory */
	    tree_rescan_cmd ();
	    break;

	case '\n':
	    /* Return this directory */
	    result = selected_ptr->name;
	    done = 1;
	    break;
	case ESC_CHAR:
	case XCTRL('g'):
	case XCTRL('c'):
	case KEY_F(10):
	    /* Cancel */
	    result = NULL;
	    done = 1;
	    break;
	case XCTRL('h'):
	case KEY_BACKSPACE:
	case 0177:
	    i = strlen (search_buffer_tree);
	    if (i)
		search_buffer_tree [i - 1] = 0;
	    break;
	case ALT ('\n'):
	case XCTRL ('s'):
	case ALT ('s'):
	    /* Search next occurence */
	    tree_move_forward (1);
	    show_tree (NULL);
	    search_tree (search_buffer_tree);
	    break;
	case -1:
	    /* Mouse click */
	    result = selected_ptr->name;
	    break;
	default:
	    if (is_printable (c)){
		i = strlen (search_buffer_tree);
		search_buffer_tree [i] = c;
		search_buffer_tree [i+1] = 0;
		search_tree (search_buffer_tree);
	    }
	    break;
	}
    }

    /* Clean up */
    win_done ();
    if (result)
	result = strdup (result);
    do_refresh ();
    return result;
}
